/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module contains options that can be set by both malaga and mallex. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "input.h"
#include "commands.h"
#include "rule_type.h"
#include "rules.h"
#include "value_parser.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL
#include "options.h"

/* commands =================================================================*/

LOCAL void do_sort_records_option (string_t arguments)
/* Select the order in which attributes in a record are printed. */
{
  if (*arguments == EOS)
    printf ("sort-records: %s\n", 
	    attribute_order == INTERNAL_ORDER ? "internal" :
	    attribute_order == ALPHABETIC_ORDER ? "alphabetic" :
	    attribute_order == DEFINITION_ORDER ? "definition" : "");
  else
  {
    string_t argument;

    argument = parse_word (&arguments);
    if (strcmp_no_case (argument, "internal") == 0)
      attribute_order = INTERNAL_ORDER;
    else if (strcmp_no_case (argument, "alphabetic") == 0)
      attribute_order = ALPHABETIC_ORDER;
    else if (strcmp_no_case (argument, "definition") == 0)
      attribute_order = DEFINITION_ORDER;
    else 
      error ("\"internal\", \"alphabetic\" or \"definition\" expected, "
	     "not \"%s\"\n", argument);
    
    free_mem (&argument);
  }

  parse_end (arguments);
}

GLOBAL command_t sort_records_option =
{
  "sort-records", do_sort_records_option,
  "Usage:\n"
  "  set sort-records internal -- print attributes in internal order\n"
  "  set sort-records alphabetic -- print attributes in alphabetic order\n"
  "  set sort-records definition -- print attributes in symbol-table order\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_hidden_option (string_t arguments)
/* Hide specified attributes when printing values. */
{
  int_t i;
  
  if (*arguments == EOS)
  {
    symbol_t *attribs = get_hidden_attributes ();

    printf ("hidden:");
    if (attribs[0] == SYMBOL_MAX)
      printf (" (none)");
    else
    {
      for (i = 0; attribs[i] != SYMBOL_MAX; i++)
	printf (" %s", get_symbol_name (attribs[i]));
    }
    printf ("\n");
  }
  else
  {
    while (*arguments != EOS)
    {
      string_t argument = parse_word (&arguments);
      
      if (strcmp_no_case (argument, "none") == 0)
	clear_hidden_attributes ();
      else if (*argument == '+')
	add_hidden_attribute (find_symbol (argument + 1));
      else if (*argument == '-')
	remove_hidden_attribute (find_symbol (argument + 1));
      else
	error ("\"+...\", \"-...\" or \"none\" expected, not \"%s\"", 
	       argument);
      
      free_mem (&argument);
    }
  }
}

GLOBAL command_t hidden_option =
{
  "hidden", do_hidden_option,
  "Set hidden attributes in categories.\n"
  "Usage: set hidden <argument> {<argument>}, where <argument> may be:\n"
  "  none -- don't hide any attributes\n"
  "  +<attribute> -- also hide <attribute>\n"
  "  -<attribute> -- don't hide <attribute> any longer\n"
};

/*---------------------------------------------------------------------------*/

#ifdef HANGUL

LOCAL void do_hangul_option (string_t arguments)
/* Switch to hangul output. */
{
  if (*arguments == EOS)
    printf ("hangul: %s\n", convert_to_ksc ? "yes" : "no");
  else
    convert_to_ksc = parse_yes_no (&arguments);

  parse_end (arguments);
}

GLOBAL command_t hangul_option =
{
  "hangul", do_hangul_option,
  "Usage:\n"
  "  set hangul yes -- display output in KSC5601 Hangul characters\n"
  "  set hangul no -- display output in romanised Hangul\n"
};

#endif

/*---------------------------------------------------------------------------*/

LOCAL void do_output_option (string_t arguments)
/* Determine if output is printed after analysis. */
{
  if (*arguments == EOS)
    printf ("output: %s\n", show_output ? "yes" : "no");
  else
    show_output = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t output_option =
{
  "output", do_output_option,
  "Usage:\n"
  "  set output yes -- show output after analysis\n"
  "  set output no -- don't show output after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_result_option (string_t arguments)
/* Determine if result is shown after analysis. */
{
  if (*arguments == EOS)
    printf ("result: %s\n", show_result ? "yes" : "no");
  else
    show_result = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t result_option =
{
  "result", do_result_option,
  "Usage:\n"
  "  set result yes -- show result after analysis\n"
  "  set result no -- don't show result after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_variables_option (string_t arguments)
/* Determine if variables are displayed in debug mode. */
{
  if (*arguments == EOS)
    printf ("variables: %s\n", show_variables ? "yes" : "no");
  else
    show_variables = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t variables_option =
{
  "variables", do_variables_option,
  "Usage:\n"
  "  set variables yes -- show variables automatically in debug mode\n"
  "  set variables no -- don't show variables automatically\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_switch_option (string_t arguments)
/* Set switches that can be read from Malaga rules. */
{
  if (*arguments == EOS)
  {
    if (! start_switches ())
      printf ("switch: no switches defined\n");
    else
    {
      symbol_t key;
      value_t value;
      string_t value_string;

      while (TRUE)
      {
	value = get_next_switch (&key);
	if (value == NULL)
	  break;

	value_string = value_to_readable (value, FALSE);
	printf ("switch \"%s\": %s\n", get_symbol_name (key), value_string);
	free_mem (&value_string);
      }
    }
  }
  else
  {
    string_t name;

    name = parse_word (&arguments);
    parse_value_string (arguments);
    set_switch (find_symbol (name), value_stack[--top]);
    free_mem (&name);
  }
}

GLOBAL command_t switch_option =
{
  "switch", do_switch_option, 
  "Set a switch that can be read from Malaga rule files.\n"
  "Usage: set switch <switch> <value>\n"
};

/* end of file ==============================================================*/
