// Maria boolean negation operator class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "NotExpression.h"
#include "Type.h"
#include "LeafValue.h"
#include "Constant.h"
#include "Property.h"
#include "Printer.h"

/** @file NotExpression.C
 * Boolean negation operator
 */

/* Copyright  1998-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

NotExpression::NotExpression (class Expression& expr) :
  Expression (),
  myExpr (&expr)
{
  setType (*myExpr->getType ());
  assert (getType ()->getKind () == Type::tBool);
}

NotExpression::~NotExpression ()
{
  myExpr->destroy ();
}

class Expression*
NotExpression::construct (class Expression& expr)
{
  if (expr.getKind () == Expression::eNot) {
    class Expression* e = const_cast<class Expression*>
      (static_cast<class NotExpression&>(expr).myExpr)->copy ();
    expr.destroy ();
    return e;
  }
  return (new class NotExpression (expr))->cse ();
}

class Value*
NotExpression::do_eval (const class Valuation& valuation) const
{
  if (class Value* v = myExpr->eval (valuation)) {
    assert (v->getKind () == Value::vLeaf);
    static_cast<class LeafValue*>(v)->
      setValue (!bool (static_cast<class LeafValue&>(*v)));
    return constrain (valuation, v);
  }
  else
    return NULL;
}

class Expression*
NotExpression::ground (const class Valuation& valuation,
		       class Transition* transition,
		       bool declare)
{
  class Expression* e = myExpr->ground (valuation, transition, declare);
  if (!e)
    return NULL;

  assert (valuation.isOK ());

  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return construct (*e)->ground (valuation);
}

class Expression*
NotExpression::substitute (class Substitution& substitution)
{
  class Expression* e = myExpr->substitute (substitution);
  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return construct (*e);
}

bool
NotExpression::depends (const class VariableSet& vars,
			bool complement) const
{
  return myExpr->depends (vars, complement);
}

bool
NotExpression::forVariables (bool (*operation)
			     (const class Expression&,void*),
			     void* data) const
{
  return myExpr->forVariables (operation, data);
}

class Expression*
NotExpression::disqualify (const class Transition& transition)
{
  if (class Expression* expr = myExpr->disqualify (transition)) {
    if (expr == myExpr) {
      expr->destroy ();
      return copy ();
    }
    if (expr->getKind () == Expression::eConstant) {
      const class Value& v = static_cast<class Constant*>(expr)->getValue ();
      assert (v.getKind () == Value::vLeaf);
      return (new class Constant
	      (*new class LeafValue
	       (*getType (),
		!bool (static_cast<const class LeafValue&>(v)))))->cse ();
    }
    else
      return construct (*expr);
  }
  return 0;
}

class Ltl*
NotExpression::toFormula (class Property& property)
{
  return property.addUnop (Property::opNot, *myExpr);
}

#ifdef EXPR_COMPILE
# include "CExpression.h"

void
NotExpression::compile (class CExpression& cexpr,
			unsigned indent,
			const char* lvalue,
			const class VariableSet* vars) const
{
  myExpr->compile (cexpr, indent, lvalue, vars);
  class StringBuffer& out = cexpr.getOut ();
  out.indent (indent);
  out.append (lvalue);
  out.append ("=!");
  out.append (lvalue);
  out.append (";\n");
  compileConstraint (cexpr, indent, lvalue);
}

#endif // EXPR_COMPILE

/** Determine whether an expression needs to be enclosed in parentheses
 * @param kind	kind of the expression
 * @return	whether parentheses are necessary
 */
inline static bool
needParentheses (enum Expression::Kind kind)
{
  switch (kind) {
  case Expression::eVariable:
  case Expression::eConstant:
  case Expression::eUndefined:
  case Expression::eStructComponent:
  case Expression::eUnionComponent:
  case Expression::eVectorIndex:
  case Expression::eNot:
  case Expression::eTypecast:
    return false;
  case Expression::eStruct:
  case Expression::eUnion:
  case Expression::eVector:
  case Expression::eUnop:
  case Expression::eBinop:
  case Expression::eBuffer:
  case Expression::eBufferRemove:
  case Expression::eBufferUnop:
  case Expression::eBufferWrite:
  case Expression::eBufferIndex:
  case Expression::eCardinality:
  case Expression::eMarking:
  case Expression::ePlaceContents:
  case Expression::eSubmarking:
  case Expression::eMapping:
  case Expression::eEmptySet:
  case Expression::eStructAssign:
  case Expression::eVectorAssign:
  case Expression::eVectorShift:
    assert (false);
  case Expression::eBooleanBinop:
  case Expression::eUnionType:
  case Expression::eIfThenElse:
  case Expression::eSet:
  case Expression::eRelop:
  case Expression::eTemporalBinop:
  case Expression::eTemporalUnop:
  case Expression::eTransitionQualifier:
    break;
  }

  return true;
}

void
NotExpression::display (const class Printer& printer) const
{
  printer.delimiter ('!');
  if (::needParentheses (myExpr->getKind ())) {
    printer.delimiter ('(')++;
    myExpr->display (printer);
    --printer.delimiter (')');
  }
  else
    myExpr->display (printer);
}
