// Maria temporal unary operator class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "TemporalUnop.h"
#include "BoolType.h"
#include "Net.h"
#include "Property.h"
#include "Printer.h"

/** @file TemporalUnop.C
 * Temporal unary operators
 */

/* Copyright  1998-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

TemporalUnop::TemporalUnop (enum Op op,
			    class Expression& expr) :
  Expression (),
  myOp (op), myExpr (&expr)
{
  assert (myExpr->getType ()->getKind () == Type::tBool);
  setType (Net::getBoolType ());
}

TemporalUnop::~TemporalUnop ()
{
  myExpr->destroy ();
}

class Expression*
TemporalUnop::ground (const class Valuation& valuation,
		      class Transition* transition,
		      bool declare)
{
  class Expression* expr = myExpr->ground (valuation, transition, declare);

  if (!expr)
    return NULL;

  assert (valuation.isOK ());

  if (expr == myExpr) {
    expr->destroy ();
    return copy ();
  }
  else
    return (new class TemporalUnop (myOp, *expr))->cse ();
}

class Expression*
TemporalUnop::substitute (class Substitution& substitution)
{
  class Expression* expr = myExpr->substitute (substitution);

  if (!expr)
    return NULL;

  if (expr == myExpr) {
    expr->destroy ();
    return copy ();
  }
  else
    return (new class TemporalUnop (myOp, *expr))->cse ();
}

bool
TemporalUnop::depends (const class VariableSet& vars,
		       bool complement) const
{
  return myExpr->depends (vars, complement);
}

bool
TemporalUnop::forVariables (bool (*operation)
			    (const class Expression&,void*),
			    void* data) const
{
  return myExpr->forVariables (operation, data);
}

class Ltl*
TemporalUnop::toFormula (class Property& property)
{
  switch (myOp) {
  case Finally:
    return property.addUnop (Property::opFinally, *myExpr);
  case Globally:
    return property.addUnop (Property::opGlobally, *myExpr);
  case Next:
    return property.addUnop (Property::opNext, *myExpr);
  }
  assert (false);
  return 0;
}

/** Convert an operator to a string
 * @param op	the operator to convert
 * @return	a string corresponding to the operator
 */
static const char*
getOpString (enum TemporalUnop::Op op)
{
  switch (op) {
  case TemporalUnop::Finally:
    return "<>";
  case TemporalUnop::Globally:
    return "[]";
  case TemporalUnop::Next:
    return "()";
  }

  return "???";
}

void
TemporalUnop::display (const class Printer& printer) const
{
  printer.printRaw (::getOpString (myOp));
  printer.delimiter ('(')++;
  myExpr->display (printer);
  --printer.delimiter (')');
}
