/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QDialog>
#include <QFile>
#include <QDebug>
#include <QScrollArea>


/////////////////////// Local includes
#include "application.hpp"
#include "calculatorChemPadDlg.hpp"
#include "chemPadButton.hpp"
#include "calculatorWnd.hpp"


namespace massXpert
{

  CalculatorChemPadDlg::CalculatorChemPadDlg(QWidget *parent)
    : QDialog(parent)
  {
    m_ui.setupUi(this);
    setSizeGripEnabled(true);

    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);

    restoreGeometry(settings.value("calculator_chem_pad_dlg/geometry")
		     .toByteArray());

    connect(this,
	     SIGNAL(rejected()),
	     this,
	     SLOT(close()));
  }


  CalculatorChemPadDlg::~CalculatorChemPadDlg()
  {
  
  }


  bool
  CalculatorChemPadDlg::setup(QString &filePath)
  {  
    CalculatorWnd *wnd = static_cast<CalculatorWnd *>(parentWidget());
  
    int iter = 0;

    QFile file(filePath);
    if (! file.open(QFile::ReadOnly)) 
      {
	qDebug() << "Chemical pad's configuration file could not be opened.";
      
	return false;
      }
  
    QStringList stringList;
    int columnCount;
  
    if (!parse(filePath, columnCount, stringList))
      {
	qDebug() << "Failed to parse the configuration "
	  "file for the chemical pad";
      
	return false;
      }
  
#if 0
    iter = 0;
  
    while(iter < stringList.size())
      {
	QString string = stringList.at(iter);
	string += '/' ;
	string += stringList.at(++iter);
	string += '/';
	string += stringList.at(++iter);

	++iter;
      
	qDebug() << string;
      }
#endif

    // At this point, stringList contains a series of button item
    // strings in pacakges of three strings. For example for two
    // buttons, we'd have:
    //
    // protonate
    // +H1
    // adds a proton
    // hydrate
    // +H2O1
    // adds a water molecule

    // What about the number of columns ?
    if (columnCount <= 0) 
      columnCount = 3;
  
    int buttonCount =  stringList.size() / 3;
    //  qDebug() << __FILE__ << __LINE__ 
    // << "button count is " << buttonCount;

    if (buttonCount == 0)
      return true;
  
    int row = 0;
    int column = 0;
  
    while(iter < stringList.size())
      {
	ChemPadButton *button = 
	  new ChemPadButton(stringList.at(iter), 
			     stringList.at(++iter),
			     wnd, this);
	button->setToolTip(stringList.at(++iter));

	m_ui.gridLayout->addWidget(button, row, column);

	++iter;
	++column;
      
	// Remember, columns are numbered like indexes: starting from 0
	//(thus the '=' sign along with '>' below).
	if(column >= columnCount)
	  {
	    column = 0;
	    ++row;
	  }
      }

    return true;
  }


  bool
  CalculatorChemPadDlg::parse(QString &filePath, 
			       int &columnCount,
			       QStringList &stringList)
  {
    qint64 lineLength;
    QString line;
    char buffer [1024];
  
    QFile file(filePath);
    if (! file.open(QFile::ReadOnly)) 
      return false;

    // The file we have to parse is like this:
    //
    // chempad_columns%3
  
    // chempadkey%protonate%+H1%adds a proton
    // chempadkey%hydrate%+H2O1%adds a water molecule
    //
    // Note that any line starting with something else than 'chempad' is
    // ignored.

    lineLength = file.readLine(buffer, sizeof(buffer));
  
    while(lineLength != -1)
      {
	// The line is now in buffer, and we want to convert
	// it to Unicode by setting it in a QString.
	line = buffer;
      
	if(!line.startsWith("chempad", Qt::CaseSensitive))
	  {
	    lineLength = file.readLine(buffer, sizeof(buffer));
	    continue;
	  }
      
	if(line.startsWith("chempad_columns%", Qt::CaseSensitive))
	  {
	    int percentSign = line.lastIndexOf('%');
	  
	    QString columnsString = line.mid(percentSign, line.length());
	  
	    bool ok;
	  
	    columnCount = columnsString.toInt(&ok);
	  
	    // If the line contains an error, the columnCount value is 3.
	    if (columnCount == 0 && !ok)
	      columnCount = 3;

	    lineLength = file.readLine(buffer, sizeof(buffer));
	    continue;
	  }
            
	QStringList list = line.split("%", QString::SkipEmptyParts);

	// The first item of the list is "chempadkey", which we have to
	// remove.
	if(list.first() != "chempadkey")
	  {
	    qDebug() << "Error with chemical pad line" 
		      << line.toAscii();
	  
	    lineLength = file.readLine(buffer, sizeof(buffer));
	    continue;
	  }
      
	list.removeFirst();
      
	// Now the number of items in the list should be 3.
	if(list.size() != 3)
	  {
	    qDebug() << "Error with chemical pad line" 
		      << line.toAscii();
	  
	    lineLength = file.readLine(buffer, sizeof(buffer));
	    continue;
	  }
	 
	stringList = stringList << list;
      
	lineLength = file.readLine(buffer, sizeof(buffer));
      }
  
    // At this time, according to the example above, stringList should 
    // contain six elements in this order:
    //
    // protonate
    // +H1
    // adds a proton
    // hydrate
    // +H2O1
    // adds a water molecule

    return true;
  }



  ////////////////////////////// SLOTS ///////////////////////////////
  void 
  CalculatorChemPadDlg::close()
  {
    // Before closing set the parent checkbox to proper state.
    CalculatorWnd *wnd = static_cast<CalculatorWnd *>(parentWidget());
  
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);

    settings.setValue("calculator_chem_pad_dlg/geometry", saveGeometry());

    wnd->chemPadDlgClosed();
  }

} // namespace massXpert
