/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QMessageBox>
#include <QFileDialog>


/////////////////////// Local includes
#include "cleavageDlg.hpp"
#include "application.hpp"


namespace massXpert
{

  CleavageDlg::CleavageDlg(QWidget *parent,
			    Polymer *polymer,
			    const PolChemDef *polChemDef,
			    const CalcOptions *calcOptions,
			    const IonizeRule *ionizeRule)
    : QDialog(parent),
      mp_polymer(polymer), 
      mp_polChemDef(polChemDef), 
      mp_calcOptions(calcOptions),
      mp_ionizeRule(ionizeRule)
  {
    Q_ASSERT(parent);
    Q_ASSERT(!mp_polymer.isNull() && mp_polChemDef && 
	      mp_calcOptions && mp_ionizeRule);

    m_ui.setupUi(this);
  
    mp_editorWnd = static_cast<SequenceEditorWnd *>(parent);

    populateCleaveSpecComboBox();

    setupTreeView();
    
    m_ui.leftCapPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));
    m_ui.rightCapPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    m_ui.leftModifPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));
    m_ui.rightModifPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    m_ui.modifsPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    m_ui.crossLinksPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    mpa_cleaver = 0;
  
    // The tolerance when filtering mono/avg masses...
    QStringList stringList;
  
    stringList << tr("AMU") << tr("PCT") << tr("PPM");
  
    m_ui.toleranceComboBox->insertItems(0, stringList);
  
    m_ui.toleranceComboBox->setToolTip(tr("AMU: atom mass unit \n"
					    "PCT: percent \n"
					    "PPM: part per million"));

    filterAct = new QAction(tr("Toggle Filtering"), this);
    filterAct->setShortcut(QKeySequence(Qt::CTRL+Qt::Key_F));
    this->addAction(filterAct);
    connect(filterAct, 
	     SIGNAL(triggered()), 
	     this, 
	     SLOT(filterOptionsToggled()));

    m_ui.filteringOptionsGroupBox->addAction(filterAct);
    // When the dialog box is created it is created with the groupbox
    // unchecked.
    m_ui.filteringOptionsFrame->setVisible(false);
  
    // When the filtering group box will be opened, the focus will be on the 
    // first widget of the groupbox:
    mp_focusWidget = m_ui.filterPartialLineEdit;

    // The results-exporting menus. ////////////////////////////////

    QStringList comboBoxItemList;

    comboBoxItemList 
      << tr("To Clipboard") 
      << tr("To File")
      << tr("Select File");
  
    m_ui.exportResultsComboBox->addItems(comboBoxItemList);
  
    connect(m_ui.exportResultsComboBox,
	     SIGNAL(activated(int)),
	     this,
	     SLOT(exportResults(int)));

    mpa_resultsString = new QString();
  
    //////////////////////////////////// The results-exporting menus.
  

    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
  
    settings.beginGroup("cleavage_dlg");

    restoreGeometry(settings.value("geometry").toByteArray());

    m_ui.oligomersSplitter->
      restoreState(settings.value("oligomersSplitter").toByteArray());
  
    m_ui.oligoDetailsSplitter->
      restoreState(settings.value("oligoDetailsSplitter").toByteArray());
  
    settings.endGroup();


    connect(m_ui.cleavePushButton,
	     SIGNAL(clicked()),
	     this,
	     SLOT(cleave()));

    connect(m_ui.filterPartialLineEdit,
	     SIGNAL(returnPressed()),
	     this,
	     SLOT(filterPartial()));
  
    connect(m_ui.filterMonoMassLineEdit,
	     SIGNAL(returnPressed()),
	     this,
	     SLOT(filterMonoMass()));
  
    connect(m_ui.filterAvgMassLineEdit,
	     SIGNAL(returnPressed()),
	     this,
	     SLOT(filterAvgMass()));
  
    connect(m_ui.filterChargeLineEdit,
	     SIGNAL(returnPressed()),
	     this,
	     SLOT(filterCharge()));

    connect(m_ui.filteringOptionsGroupBox,
	     SIGNAL(clicked(bool)),
	     this,
	     SLOT(filterOptions(bool)));
  }


  CleavageDlg::~CleavageDlg()
  {
    delete mpa_resultsString;
  
    delete mpa_oligomerTreeViewModel;

    delete mpa_proxyModel;
  
    if (mpa_cleaver)
      delete mpa_cleaver;
  }


  void
  CleavageDlg::populateCleaveSpecComboBox()
  {
    PolChemDef *polChemDef = mp_editorWnd->polChemDef();
    Q_ASSERT(polChemDef);
  
    for (int iter = 0; iter < polChemDef->cleaveSpecList().size(); ++iter)
      {
	CleaveSpec *cleaveSpec = polChemDef->cleaveSpecList().at(iter);
	Q_ASSERT(cleaveSpec);
      
	m_ui.cleavingAgentsComboBox->addItem(cleaveSpec->name());
      }
  
    return;
  }


  void 
  CleavageDlg::setupTreeView()
  {
     // Model stuff all thought for sorting.
    mpa_oligomerTreeViewModel = 
      new CleaveOligomerTreeViewModel(&m_oligomerList, this);

    mpa_proxyModel = new CleaveOligomerTreeViewSortProxyModel(this);
    mpa_proxyModel->setSourceModel(mpa_oligomerTreeViewModel);
    mpa_proxyModel->setFilterKeyColumn(-1);
  
    m_ui.oligomerTreeView->setModel(mpa_proxyModel);
    m_ui.oligomerTreeView->setParentDlg(this);
    mpa_oligomerTreeViewModel->setTreeView(m_ui.oligomerTreeView);
  }
  

  SequenceEditorWnd *
  CleavageDlg::editorWnd()
  {
    return mp_editorWnd;
  }


  void 
  CleavageDlg::cleave()
  {
    // What's the cleavage agent ?
    int value = m_ui.cleavingAgentsComboBox->currentIndex();

    CleaveSpec *cleaveSpec = mp_polChemDef->cleaveSpecList().at(value);
    
    value = m_ui.partialCleavageSpinBox->value();
  
    CleaveOptions cleaveOptions(*cleaveSpec, value, false);
  
    // Set the ionization levels.
    int valueStart = m_ui.ionizeLevelStartSpinBox->value();
    int valueEnd = m_ui.ionizeLevelEndSpinBox->value();

    cleaveOptions.setIonizeLevels(valueStart, valueEnd);
    
    OligomerList oligomerList;
    
    if (mpa_cleaver)
      delete mpa_cleaver;
    
    mpa_cleaver= new Cleaver(mp_polymer, mp_polChemDef, cleaveOptions,
			      *mp_calcOptions, *mp_ionizeRule);

    mpa_cleaver->setOligomerList(&oligomerList);
    
    if (!mpa_cleaver->cleave())
      {
	QMessageBox::critical(this, 
			       tr("massXpert - Polymer Cleavage"),
			       tr("Failed to perform cleavage."),
			       QMessageBox::Ok);
	return;
      }
        
    // We now have new oligomers in oligomerList. We either stack
    // these on top of the previous ones, or we replace the previous
    // ones with the new ones.
    // Are we stacking new oligomer on top of the old ones?

    if (!m_ui.stackOligomersCheckBox->isChecked())
      mpa_oligomerTreeViewModel->removeOligomers();
    
    
    // At this point we can set up the data to the treeview model.
    int initialOligomers = oligomerList.size();
    int addedOligomers = mpa_oligomerTreeViewModel->addOligomers(&oligomerList);
    
    if (initialOligomers != addedOligomers)
      qFatal("Fatal error at %s@%d. Program aborted.",
	      __FILE__, __LINE__);

    if (oligomerList.size())
      qFatal("Fatal error at %s@%d. Program aborted.",
	      __FILE__, __LINE__);
        
    // Logically, the oligomerList should be empty by now.
    Q_ASSERT(oligomerList.isEmpty());
    
    updateCleavageDetails();

    // Set focus to the treeView.
    m_ui.oligomerTreeView->setFocus();
    
    QString title;
    
    int oligomerCount = mpa_oligomerTreeViewModel->rowCount();
    
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    if (!oligomerCount)
      title = tr("Oligomers(empty list)");
    else if (oligomerCount == 1)
      title = tr("Oligomers(one item)");
    else
      title = tr("Oligomers(%1 items)")
	.arg(locale.toString(oligomerCount));
	
    m_ui.oligomerGroupBox->setTitle(title);
  }


  void
  CleavageDlg::updateCleavageDetails()
  {
    if (mp_calcOptions->capping() & MXT_CAP_LEFT)
      m_ui.leftCapPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.leftCapPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    if (mp_calcOptions->capping() & MXT_CAP_RIGHT)
      m_ui.rightCapPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.rightCapPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));


    if (mp_calcOptions->monomerEntities() & MXT_MONOMER_CHEMENT_MODIF)
      m_ui.modifsPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.modifsPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    if (mp_calcOptions->monomerEntities() & MXT_MONOMER_CHEMENT_CROSS_LINK)
      m_ui.crossLinksPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.crossLinksPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));


    if (mp_calcOptions->polymerEntities() & MXT_POLYMER_CHEMENT_LEFT_END_MODIF)
      m_ui.leftModifPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.leftModifPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));

    if (mp_calcOptions->polymerEntities() & MXT_POLYMER_CHEMENT_RIGHT_END_MODIF)
      m_ui.rightModifPixmapLabel->setPixmap(QPixmap(":/images/greenled.png"));
    else
      m_ui.rightModifPixmapLabel->setPixmap(QPixmap(":/images/redled.png"));
  }


  void
  CleavageDlg::updateOligomerSequence(QString *text)
  {
    Q_ASSERT(text);
  
    m_ui.oligomerSequenceTextEdit->clear();
    m_ui.oligomerSequenceTextEdit->append(*text);
  }


  void 
  CleavageDlg::closeEvent(QCloseEvent *event)
  {
    if (event)
      printf("%s", "");
  
    QSettings settings 
     (static_cast<Application *>(qApp)->configSettingsFilePath(), 
       QSettings::IniFormat);
  
    settings.beginGroup("cleavage_dlg");

    settings.setValue("geometry", saveGeometry());

    settings.setValue("oligomersSplitter", 
		       m_ui.oligomersSplitter->saveState());

    settings.setValue("oligoDetailsSplitter", 
		       m_ui.oligoDetailsSplitter->saveState());

    settings.endGroup();
  }



  bool
  CleavageDlg::calculateTolerance(double mass)
  {
    // Get the tolerance that is in its lineEdit.
  
    QString text = m_ui.toleranceLineEdit->text();
    double tolerance = 0;
    bool ok = false;
  
    if (!text.isEmpty())
      {
	// Convert the string to a double.
	Application *application = static_cast<Application *>(qApp);
	QLocale locale = application->locale();

	ok = false;
	tolerance = locale.toDouble(text, &ok);
      
	if(!tolerance && !ok)
	  return false;
      }
    else
      {
	m_tolerance = 0;
      }
    
    // What's the item currently selected in the comboBox?
    int index = m_ui.toleranceComboBox->currentIndex();
  
    if (index == 0)
      {
	// MXT_MASS_TOLERANCE_AMU
	m_tolerance = tolerance;
      }
    else if (index == 1)
      {
	// MXT_MASS_TOLERANCE_PCT
	m_tolerance =(tolerance / 100) * mass;
      }
    else if (index == 2)
      {
	// MXT_MASS_TOLERANCE_PPM
	m_tolerance =(tolerance / 1000000) * mass;
      }
    else
      Q_ASSERT(0);
  
    return true;
  }


  void 
  CleavageDlg::filterOptions(bool checked)
  {
    if (!checked)
      {
	mpa_proxyModel->setFilterKeyColumn(-1);
      
	mpa_proxyModel->applyNewFilter();

	m_ui.filteringOptionsFrame->setVisible(false);
      }
    else
      {
	m_ui.filteringOptionsFrame->setVisible(true);

	// In this case, set focus to the last focused widget in the
	// groupbox or the first widget in the groubox if this is the
	// first time the filtering is used.
	mp_focusWidget->setFocus();
      }
  }


  void 
  CleavageDlg::filterOptionsToggled()
  {
    bool isChecked = m_ui.filteringOptionsGroupBox->isChecked();
  
    m_ui.filteringOptionsGroupBox->setChecked(!isChecked);
    filterOptions(!isChecked);
  }


  void 
  CleavageDlg::filterPartial()
  {
    // First off, we have to get the partial that is in the lineEdit.
  
    QString text = m_ui.filterPartialLineEdit->text();

    if (text.isEmpty())
      return;
  
    // Convert the string to a int.
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    bool ok = false;
    int partial = locale.toInt(text, &ok);
  
    if (!partial && !ok)
      return;

    mpa_proxyModel->setPartialFilter(partial);
    mpa_proxyModel->setFilterKeyColumn(0);
    mpa_proxyModel->applyNewFilter();

    mp_focusWidget = m_ui.filterPartialLineEdit;
  }




  void 
  CleavageDlg::filterMonoMass()
  {
    // First off, we have to get the mass that is in the lineEdit.

    QString text = m_ui.filterMonoMassLineEdit->text();
  
    if (text.isEmpty())
      return;
  
    // Convert the string to a double.
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    bool ok = false;
    double mass = locale.toDouble(text, &ok);
  
    if (!mass && !ok)
      return;

    // At this point, depending on the item that is currently selected
    // in the comboBox, we'll have to actually compute the tolerance.

    if (!calculateTolerance(mass))
      return ;

    mpa_proxyModel->setMonoFilter(mass);
    mpa_proxyModel->setTolerance(m_tolerance);
  
    mpa_proxyModel->setFilterKeyColumn(3);
    mpa_proxyModel->applyNewFilter();

    mp_focusWidget = m_ui.filterMonoMassLineEdit;
  }


  void 
  CleavageDlg::filterAvgMass()
  {
    // First off, we have to get the mass that is in the lineEdit.

    QString text = m_ui.filterAvgMassLineEdit->text();
  
    if (text.isEmpty())
      return;
  
    // Convert the string to a double.
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    bool ok = false;
    double mass = locale.toDouble(text, &ok);
  
    if (!mass && !ok)
      return;

    // At this point, depending on the item that is currently selected
    // in the comboBox, we'll have to actually compute the tolerance.

    if (!calculateTolerance(mass))
      return ;

    mpa_proxyModel->setAvgFilter(mass);
    mpa_proxyModel->setTolerance(m_tolerance);
  
    mpa_proxyModel->setFilterKeyColumn(4);
    mpa_proxyModel->applyNewFilter();

    mp_focusWidget = m_ui.filterAvgMassLineEdit;
  }


  void 
  CleavageDlg::filterCharge()
  {
    // First off, we have to get the charge that is in the lineEdit.

    QString text = m_ui.filterChargeLineEdit->text();

    if (text.isEmpty())
      return;
  
    // Convert the string to a int.
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    bool ok = false;
    int charge = locale.toInt(text, &ok);

    if (!charge && !ok)
      return;

    mpa_proxyModel->setChargeFilter(charge);
    mpa_proxyModel->setFilterKeyColumn(5);
    mpa_proxyModel->applyNewFilter();

    mp_focusWidget = m_ui.filterChargeLineEdit;
  }



  // The results-exporting functions. ////////////////////////////////
  // The results-exporting functions. ////////////////////////////////
  // The results-exporting functions. ////////////////////////////////
  void
  CleavageDlg::exportResults(int index)
  {
    // Remember that we had set up the combobox with the following strings:
    // << tr("To Clipboard") 
    // << tr("To File")
    // << tr("Select File");

    if (index == 0)
      {
	exportResultsClipboard();
      }
    else if (index == 1)
      {
	exportResultsFile();
      }
    else if (index == 2)
      {
	selectResultsFile();
      }
    else 
      Q_ASSERT(0);
  
  }


  void
  CleavageDlg::prepareResultsTxtString()
  {
    mpa_resultsString->clear();
  
    *mpa_resultsString += QObject::tr("\n---------------------------\n"
				       "Polymer sequence cleavage: \n"
				       "---------------------------\n");
  
    bool withSequence = m_ui.withSequenceCheckBox->isChecked();
    
    QString *text = 
      m_ui.oligomerTreeView->selectedOligomersAsPlainText(withSequence);

    *mpa_resultsString += *text;
    
    delete text;
  }


  bool 
  CleavageDlg::exportResultsClipboard()
  {
    prepareResultsTxtString();
  
    QClipboard *clipboard = QApplication::clipboard();

    clipboard->setText(*mpa_resultsString, QClipboard::Clipboard);
  
    return true;
  }


  bool 
  CleavageDlg::exportResultsFile()
  {
    if (m_resultsFilePath.isEmpty())
      {
	if(!selectResultsFile())
	  return false;
      }
  
    QFile file(m_resultsFilePath);
  
    if (!file.open(QIODevice::WriteOnly | QIODevice::Append))
      {
	QMessageBox::information(this, 
				  tr("massXpert - Export Data"),
				  tr("Failed to open file in append mode."),
				  QMessageBox::Ok);
	return false;
      }
  
    QTextStream stream(&file);
    stream.setCodec("UTF-8");

    prepareResultsTxtString();
  
    stream << *mpa_resultsString;
  
    file.close();

    return true;
  }


  bool 
  CleavageDlg::selectResultsFile()
  {
    m_resultsFilePath = 
      QFileDialog::getSaveFileName(this, tr("Select file to export data to"),
				    QDir::homePath(),
				    tr("Data files(*.dat *.DAT)"));
  
    if (m_resultsFilePath.isEmpty())
      return false;

    return true;
  }
  //////////////////////////////////// The results-exporting functions.
  //////////////////////////////////// The results-exporting functions.
  //////////////////////////////////// The results-exporting functions.

} // namespace massXpert
