/*
  mbdesktop - a desktop for handhelds etc. 

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include "mbdesktop.h"
#include "mbdesktop_view.h"
#include "mbdesktop_module.h"

#include <dlfcn.h>

#ifdef USE_PNG
#define FOLDER_IMG "mbfolder.png"
#else
#define FOLDER_IMG "mbfolder.xpm"
#endif

static void
modules_init (MBDesktop *mb);


#ifdef USE_XSETTINGS
static void 
mbdesktop_xsettings_notify_cb (const char       *name,
			       XSettingsAction   action,
			       XSettingsSetting *setting,
			       void             *data);
#endif

static Bool
mbdesktop_get_theme_via_root_prop(MBDesktop *mb);


#ifdef USE_DNOTIFY
MBDesktop *_Gmb;

static void 
sig_reload_handler(int sig, siginfo_t *si, void *data)
{
  MBDesktop *mb = _Gmb; 	/* Reloads the entire desktop */

  mbdesktop_item_folder_contents_free(_Gmb, _Gmb->top_head_item);

  mb->current_folder_item = mb->top_head_item;

  modules_init(mb);

  mb->kbd_focus_item = mb->current_head_item 
    = mb->scroll_offset_item = mb->top_head_item->item_child;

  mbdesktop_view_paint(_Gmb, False);
}

#endif

#ifdef USE_XSETTINGS

#define XSET_UNKNOWN    0
#define XSET_THEME      1
#define XSET_BG         2
#define XSET_FONT       3
#define XSET_FONT_COL   6
#define XSET_TITLE_FONT 4
#define XSET_ICON       5


static Bool
mbdesktop_xsettings_process (MBDesktop        *mb, 
			     const char       *name, 
			     XSettingsSetting *setting
			     )
{
  int i = 0;
  int key = XSET_UNKNOWN;
  
  struct _mb_xsettings { char *name; int value; } mb_xsettings[] = {
    { "Net/ThemeName",               XSET_THEME      },
    { "MATCHBOX/Background",         XSET_BG         },
    { "MATCHBOX/Desktop/Font",       XSET_FONT       },
    { "MATCHBOX/Desktop/TitleFont",  XSET_TITLE_FONT },
    { "MATCHBOX/Desktop/FontColor",  XSET_FONT_COL   },
    { "MATCHBOX/Desktop/IconSize",   XSET_ICON       },
    { NULL,       -1 } 
  };
  
  while(  mb_xsettings[i].name != NULL )
    {
      if (!strcmp(name, mb_xsettings[i].name)
	  && setting != NULL 	/* XXX set to NULL when action deleted */
	  && ( setting->type == XSETTINGS_TYPE_STRING
	       || setting->type == XSETTINGS_TYPE_INT ) )
	{
	  key = mb_xsettings[i].value;
	  break;
	}
      i++;
    }
    
  if (key == XSET_UNKNOWN) {
    /* xsettings_setting_free(setting); cause seg ? */
    return False;
  }

  switch (key)
    {
    case XSET_THEME:
      mbdesktop_switch_theme (mb, setting->data.v_string);
      break;
    case XSET_BG:
      mb->bg_def = strdup(setting->data.v_string);
      if (mb->bg_img != NULL) 	/* Background not yet created */
	{
	  mbdesktop_bg_parse_spec(mb, mb->bg_def);
	  mbdesktop_view_init_bg(mb);
	  mbdesktop_view_paint(mb, False);
	}
      break;
    case XSET_FONT:
      mbdesktop_set_font(mb, setting->data.v_string);
      if (mb->top_head_item != NULL)
	  mbdesktop_view_paint(mb, False);
      break;
    case XSET_TITLE_FONT:
      mbdesktop_set_title_font(mb, setting->data.v_string);
      if (mb->top_head_item != NULL)
	  mbdesktop_view_paint(mb, False);
      break;
    case XSET_ICON:
      mb->icon_size = setting->data.v_int;
      if (mb->top_head_item != NULL)
	mbdesktop_view_paint(mb, False);
      break;
    case XSET_FONT_COL:
      mbdesktop_set_font_color(mb, setting->data.v_string);
      if (mb->top_head_item != NULL)
	  mbdesktop_view_paint(mb, False);
      break;
    }
  
  /* xsettings_setting_free(setting); */
  return True;
}


static void 
mbdesktop_xsettings_notify_cb (const char       *name,
			       XSettingsAction   action,
			       XSettingsSetting *setting,
			       void             *data)
{
  MBDesktop *mb = (MBDesktop *)data;

  switch (action)
    {
    case XSETTINGS_ACTION_NEW:
    case XSETTINGS_ACTION_CHANGED:
      mbdesktop_xsettings_process (mb, name, setting); 
      break;
    case XSETTINGS_ACTION_DELETED:
        break;
    }

}

#endif

static Bool
mbdesktop_get_theme_via_root_prop(MBDesktop *mb)
{
#ifndef USE_XSETTINGS
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  char * value;
  
  status = XGetWindowProperty(mb->dpy, mb->root,
			      mb->atom_mb_theme, 
			      0L, 512L, False,
			      AnyPropertyType, &realType,
			      &format, &n, &extra,
			      (unsigned char **) &value);
  
  if (status != Success || value == 0 || *value == 0 || n == 0)
    {
      if (value) XFree(value);
      return False;
    }
  
  if (mb->theme_name == NULL || strcmp(mb->theme_name, value))
    {
      if (mb->theme_name) free(mb->theme_name);
      mb->theme_name = strdup(value);
      
      XFree(value);
      return True;
    }
  if (value) XFree(value);
#endif // USE_XSETTINGS
  return False;
}


void
mbdesktop_switch_theme (MBDesktop *mb, char *theme_name )
{
  if (theme_name != NULL)
    {
      if (mb->theme_name) free(mb->theme_name);
      mb->theme_name = strdup(theme_name);
    }

  mbdesktop_switch_icon_theme(mb, mb->top_head_item); 
  mbdesktop_set_scroll_buttons(mb);

  if (mb->bg_img)
    mbdesktop_view_paint(mb, False);

  /* XXX switch desktop bg theme too */
}

void
mbdesktop_switch_icon_theme (MBDesktop     *mb, 
			     MBDesktopItem *item )
{
  MBDesktopItem *item_cur = item;

  while (item_cur != NULL)
    {
      if (item_cur->item_child)
	mbdesktop_switch_icon_theme (mb, item_cur->item_child );

      if (item_cur->icon_name)
	mbdesktop_item_set_icon_from_theme (mb, item_cur);
      item_cur = item_cur->item_next_sibling;
    }
}

Bool
mbdesktop_get_workarea(MBDesktop *mb, int *x, int *y, int *w, int *h)
{
  Atom real_type; int real_format;
  unsigned long items_read, items_left;
  long *coords;

  Atom workarea_atom =
    XInternAtom(mb->dpy, "_NET_WORKAREA",False);

  if (XGetWindowProperty(mb->dpy, mb->root,
			 workarea_atom, 0L, 4L, False,
			 XA_CARDINAL, &real_type, &real_format,
			 &items_read, &items_left,
			 (unsigned char **) &coords) == Success
      && items_read) {
    *x = coords[0];
    *y = coords[1];
    *w = coords[2];
    *h = coords[3];
    XFree(coords);
    return True;
  }
  return False;
}

void
mbdesktop_bg_free_config(MBDesktop *mb)
{
  if (mb->bg->type == BG_TILED_PXM || mb->bg->type == BG_STRETCHED_PXM)
    free(mb->bg->data.filename);
  free(mb->bg);
  mb->bg = NULL;
}

Bool
mbdesktop_bg_parse_spec(MBDesktop *mb, char *spec)
{
  /*
  img-stretched:filename>
  img-tiled:<filename>
  col-solid:<color definition>
  col-gradient-vertical:<start color>,<end color>
  col-gradient-horizontal:<start color>,<end color>
  */

  XColor tmpxcol;
  int i, mapping_cnt, spec_offset = 0, type = 0;
  char *bg_def = NULL, *p = NULL;

  struct conf_mapping_t {
    char *name;
    int   id;
  } conf_mapping[] = {
    { "img-stretched:",           BG_STRETCHED_PXM  },
    { "img-tiled:",               BG_TILED_PXM      },
    { "img-centered:",            BG_CENTERED_PXM   },
    { "col-solid:",               BG_SOLID          },
    { "col-gradient-vertical:",   BG_GRADIENT_VERT  },
    { "col-gradient-horizontal:", BG_GRADIENT_HORIZ },
  };

  mapping_cnt = (sizeof(conf_mapping)/sizeof(struct conf_mapping_t));

  if (mb->bg != NULL) mbdesktop_bg_free_config(mb);

  mb->bg = malloc(sizeof(MBDesktopBG));
  memset(mb->bg, 0, sizeof(mb->bg));

  if (spec == NULL)
    {
      /* XXX we should probably check theme.desktop too for a bg_def */
      bg_def = "#8395ac"; 	/* Defualt col - red on error */
      type   = BG_SOLID;
    }
  else
    {
      for (i=0; i<mapping_cnt; i++)
	{
	  spec_offset = strlen(conf_mapping[i].name);
	  if (spec_offset < strlen(spec)
	      && !strncmp(conf_mapping[i].name, spec, spec_offset))
	    {
	      type = conf_mapping[i].id;
	      break;
	    }
	}

      if (!type)
	{
	  /* Assume we've just been passed an image filename */
	  mb->bg->type = BG_STRETCHED_PXM;
	  mb->bg->data.filename = strdup(spec);
	  return True;
	} else bg_def = spec + spec_offset;
    }


  mb->bg->type = type;

  switch(type)
    {
    case BG_SOLID:
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  bg_def, &tmpxcol);
      mb->bg->data.cols[0] = tmpxcol.red   >> 8;
      mb->bg->data.cols[1] = tmpxcol.green >> 8;
      mb->bg->data.cols[2] = tmpxcol.blue  >> 8;
      break;
    case BG_TILED_PXM:
    case BG_STRETCHED_PXM:
    case BG_CENTERED_PXM:
      mb->bg->data.filename = strdup(bg_def);
      break;
    case BG_GRADIENT_HORIZ:
    case BG_GRADIENT_VERT:
      p = bg_def;
      while(*p != ',' && *p != '\0') p++;
      if (*p == '\0')
	{
	  mbdesktop_bg_free_config(mb);
	  return False; 	/* XXX need to reset on fail */
	}
      *p = '\0';
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  bg_def, &tmpxcol);
      mb->bg->data.gcols[0] = (tmpxcol.red   >> 8);
      mb->bg->data.gcols[2] = (tmpxcol.green >> 8);
      mb->bg->data.gcols[4] = (tmpxcol.blue  >> 8);
      p++;
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  p, &tmpxcol);
      mb->bg->data.gcols[1] = (tmpxcol.red   >> 8);
      mb->bg->data.gcols[3] = (tmpxcol.green >> 8);
      mb->bg->data.gcols[5] = (tmpxcol.blue  >> 8);
      break;
    }    

  return True;
}

void
mbdesktop_set_font(MBDesktop *mb, char *spec)
{
#ifdef USE_XFT
  /* xft text */

  if ((mb->font = XftFontOpenName(mb->dpy, mb->scr, spec)) == NULL)
    { fprintf(stderr, "mbdesktop: Cant open font %s\n", spec); exit(1); }
  
#else
  if ((mb->font = XLoadQueryFont(mb->dpy, spec)) == NULL)
    { fprintf(stderr, "mbdesktop: Cant open font %s\n", spec); exit(1); }

  XSetFont(mb->dpy, mb->gc, mb->font->fid);

#endif
}

void
mbdesktop_set_title_font(MBDesktop *mb, char *spec)
{
#ifdef USE_XFT
  /* xft text */

  if ((mb->titlefont = XftFontOpenName(mb->dpy, mb->scr, spec)) == NULL)
    { 
      fprintf(stderr, "mbdesktop: Cant open font %s\n", spec); 
      exit(1); 
    }

#else
  if ((mb->titlefont = XLoadQueryFont(mb->dpy, spec)) == NULL)
    { 
      fprintf(stderr, "mbdesktop: Cant open font %s\n", spec); 
      exit(1); 
    }
#endif
}

void
mbdesktop_set_font_color(MBDesktop *mb, char *spec)
{
  XColor tmpxcol;
#ifdef USE_XFT
  XRenderColor  colortmp;  
#endif

  if (!XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		   spec, &tmpxcol))
    {
      if (!strcmp(spec, "white"))
	{
	  fprintf(stderr, "mbdesktop: Failed to parse font color\n"); 
	  exit(1);
	}
      mbdesktop_set_font_color(mb, "white");
    }

#ifdef USE_XFT
   colortmp.red   = tmpxcol.red;
   colortmp.green = tmpxcol.green;
   colortmp.blue  = tmpxcol.blue;
   colortmp.alpha =  0xffff; 
   XftColorAllocValue(mb->dpy, mb->pixbuf->vis, mb->pixbuf->root_cmap,
		      &colortmp, &mb->xftcol);

   colortmp.red   = 0x0;
   colortmp.green = 0x0;
   colortmp.blue  = 0x0;
   colortmp.alpha =  0xffff; 
   XftColorAllocValue(mb->dpy, mb->pixbuf->vis, mb->pixbuf->root_cmap,
		      &colortmp, &mb->xftbgcol);

#endif

}

void
mbdesktop_calculate_item_dimentions(MBDesktop *mb)
{
  mb->title_offset = mb->titlefont->ascent + mb->titlefont->descent + 4;
  mb->item_height  = mb->icon_size + ( 2.5 * (mb->font->ascent 
					      + mb->font->descent) );
  mb->item_width   = mb->icon_size + mb->icon_padding;

}

void
mbdesktop_set_scroll_buttons(MBDesktop *mb)
{
  /* XXX free existing */
  MBPixbufImage *img_tmp = NULL;

#ifdef USE_PNG
#define UP_IMG   "mbup.png"
#define DOWN_IMG "mbdown.png"
#else
#define UP_IMG   "mbup.xpm"
#define DOWN_IMG "mbdown.xpm"
#endif

  if (mb->img_scroll_up) mb_pixbuf_img_free(mb->pixbuf, mb->img_scroll_up);
  if (mb->img_scroll_down) mb_pixbuf_img_free(mb->pixbuf, mb->img_scroll_down);

  /* scroll buttons */
  if ((mb->img_scroll_up = mb_pixbuf_img_new_from_file(mb->pixbuf,
						       mb_dot_desktop_icon_get_full_path (mb->theme_name, 16, UP_IMG)))
      == NULL)
    fprintf(stderr, "mbdesktop: warning failed to load %s\n", UP_IMG);

  if ((mb->img_scroll_down = mb_pixbuf_img_new_from_file(mb->pixbuf, 
							 mb_dot_desktop_icon_get_full_path (mb->theme_name, 16, DOWN_IMG)))
      == NULL)
    fprintf(stderr, "mbdesktop: warning failed to load %s\n", DOWN_IMG);

  if (mb_pixbuf_img_get_width(mb->img_scroll_up) != 16
      || mb_pixbuf_img_get_height(mb->img_scroll_up) != 16)
    {
      img_tmp = mb_pixbuf_img_scale (mb->pixbuf, mb->img_scroll_up, 16, 16);
      mb_pixbuf_img_free(mb->pixbuf, mb->img_scroll_up);
      mb->img_scroll_up = img_tmp;
    }

  if (mb_pixbuf_img_get_width(mb->img_scroll_down) != 16
      || mb_pixbuf_img_get_height(mb->img_scroll_down) != 16)
    {
      img_tmp = mb_pixbuf_img_scale (mb->pixbuf, mb->img_scroll_down, 16, 16);
      mb_pixbuf_img_free(mb->pixbuf, mb->img_scroll_down);
      mb->img_scroll_down = img_tmp;
    }

}

void
mbdesktop_set_icon(MBDesktop *mb)
{
#ifdef USE_PNG
#define MBDESKTOP_ICON "mbdesktop.png"
#else
#define MBDESKTOP_ICON "mbdesktop.xpm"
#endif

  MBPixbufImage *win_top_level_img_icon = NULL;
  int *win_top_level_icon_data = NULL;
  Atom atom_net_wm_icon;


  if ((win_top_level_img_icon 
       = mb_pixbuf_img_new_from_file(mb->pixbuf, 
				     PIXMAPSDIR MBDESKTOP_ICON)) != NULL)
    {
      win_top_level_icon_data 
	= malloc(sizeof(CARD32)* ((win_top_level_img_icon->width * 
				   win_top_level_img_icon->height)+2));
      
      if (win_top_level_icon_data)
	{
	  int i = 0, idx = 0;
	  atom_net_wm_icon = XInternAtom(mb->dpy, "_NET_WM_ICON",False);
	  win_top_level_icon_data[0] = win_top_level_img_icon->width;
	  win_top_level_icon_data[1] = win_top_level_img_icon->height;
	  
	  for( i=2; 
	       i < (win_top_level_img_icon->width * 
		    win_top_level_img_icon->height)+2; 
	       i++)
	    {
	      idx = (i-2)*4;
	      //r = win_top_level_img_icon->rgba[idx] >> 16 ;
	      
	      win_top_level_icon_data[i] = 
		win_top_level_img_icon->rgba[idx] << 16 
		|  win_top_level_img_icon->rgba[idx+1] << 8 
		|  win_top_level_img_icon->rgba[idx+2] 
		|  win_top_level_img_icon->rgba[idx+3] << 24 ;  
	    }
	  
	  XChangeProperty(mb->dpy, mb->win_top_level, 
			  atom_net_wm_icon ,
			  XA_CARDINAL, 32, PropModeReplace,
			  (unsigned char *)win_top_level_icon_data, 
			  (win_top_level_img_icon->width * 
			   win_top_level_img_icon->height )+2);
	  
	  free(win_top_level_icon_data);
	}
    }
}

static void
usage(char *name)
{
  fprintf(stderr, 
	  "Usage: %s [options..]\n"
	  "Where options are:\n"
	  "   -display       Display to connect to\n"
	  "  --listview                   Display icons as list\n"
	  "  --iconsize     <int>         Icon size ( defualt: 32 )\n"
          "  --icon-padding <int>         Specify padding between icons\n"
          "                               ( defualts icon size/2 )\n"
	  "  --font         <font>        Icon font\n"
	  "  --titlefont    <font>        Title font\n"
	  "  --fontcol      <col>         Font color\n"
          "  --no-outline                 Dont outline text\n"
	  "  --bg            <background definition>, like;\n\n"
	  "\t\timg-stretched:<filename>\n"
	  "\t\timg-tiled:<filename>\n"
	  "\t\timg-centered:<filename>\n"
	  "\t\tcol-solid:<color definition>\n"
	  "\t\tcol-gradient-vertical:<start color>,<end color>\n"
          "\t\tcol-gradient-horizontal:<start color>,<end color>\n\n"
	  "Notes;\n"
	  "  <col> is specified as a color name or an rgb def in the form 'rgb:r/g/b' or '#RGB\n"
	  "\n%s is copyright Matthew Allum 2003\n",
	  name, name
	  );
  exit(1);
}


MBDesktop *
mbdesktop_init(int argc, char **argv)
{
  MBDesktop *mb;
  XGCValues gv;

#ifdef USE_DNOTIFY
  struct sigaction act;
#endif

  char *font_def       = FONT_DESC;
  char *font_title_def = FONT_TITLE_DESC;
  char *font_col_def   = FONT_COL;

  int i;
  Atom   window_type_atom, window_type_desktop_atom, desktop_manager_atom;
  char *display_name = (char *)getenv("DISPLAY");  

  signal(SIGCHLD, SIG_IGN);

  mb = malloc(sizeof(MBDesktop));
  memset(mb, 0, sizeof(MBDesktop));

  mb->dd_dir          = DD_DIR;
  mb->icon_size       = 0;
  mb->dd_dir_mtime    = 0;
#ifdef USE_XFT
  mb->xftdraw         = NULL;
#endif


  mb->top_level_name   = strdup("Home");
  mb->bg_def           = NULL;
  mb->view_type        = VIEW_ICONS;
  mb->use_text_outline = True;
  mb->icon_padding     = 0;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      display_name = argv[i];
      continue;
    }
    if (!strcmp ("--bg", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->bg_def = strdup(argv[i]);
      continue;
    }
    if (!strcmp("--help", argv[i]) || !strcmp("-h", argv[i])) {
      usage(argv[0]);
    }
    if (!strcmp ("--listview", argv[i])) {
      mb->view_type = VIEW_LIST;
      continue;
    }
    if (!strcmp ("--iconsize", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->icon_size = atoi(argv[i]);
      if (!mb->icon_size) usage(argv[0]);
      continue;
    }
    if (!strcmp ("--icon-padding", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->icon_padding = atoi(argv[i]);
      if (!mb->icon_padding) usage(argv[0]);
      continue;
    }
    if (!strcmp ("--font", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_def = argv[i];
      continue;
    }
    if (!strcmp ("--titlefont", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_title_def = argv[i];
      continue;
    }
    if (!strcmp ("--fontcol", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_col_def = argv[i];
      continue;
    }
    /*
    if (!strcmp ("--name", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->top_level_name = argv[i];
      continue;
    }
    */
    if (!strcmp ("--no-outline", argv[i])) {
      mb->use_text_outline = False;
      continue;
    }

    usage(argv[0]);
  }

  if ((mb->dpy = XOpenDisplay(display_name)) == NULL)
    {
      fprintf(stderr, "mbdesktop: unable to open display !\n");
      exit(1);
    }


  mb->scr = DefaultScreen(mb->dpy);
  mb->root = RootWindow(mb->dpy, mb->scr); 
  mb->pixbuf = mb_pixbuf_new(mb->dpy, mb->scr);
  mb->root_pxm = None;

  mb->ddfolders = NULL;

  mb->had_kbd_input = False;
  mb->theme_name    = NULL;
  mb->bg_img        = NULL;
  mb->bg = NULL;

  mb->top_head_item = NULL;

  /* Dimentions */
  mb->desktop_width  = DisplayWidth(mb->dpy, mb->scr);
  mb->desktop_height = DisplayHeight(mb->dpy, mb->scr);

  if (mb->icon_size == 0)
    {
      if (mb->desktop_width > 320)
	mb->icon_size = 48;
      else
	mb->icon_size = 32;
    }

  if (!mb->icon_padding)
    mb->icon_padding = ( mb->icon_size / 2 ); 

  if (mbdesktop_get_workarea(mb, &mb->workarea_x, &mb->workarea_y, 
			     &mb->workarea_width, 
			     &mb->workarea_height ) == False )
    {
      mb->workarea_x = 0;
      mb->workarea_y = 0;
      mb->workarea_width = DisplayWidth(mb->dpy, mb->scr);
      mb->workarea_height = DisplayHeight(mb->dpy, mb->scr);
    }


#ifdef USE_XSETTINGS

  mb->xsettings_client = xsettings_client_new(mb->dpy, mb->scr,
					      mbdesktop_xsettings_notify_cb,
					      NULL, (void *)mb );


  if (mb->xsettings_client == NULL)
    fprintf(stderr,"mbmenu: XSETTINGS manager not found\n");

#endif

  mb->folder_img_path = NULL;

   /* GC's */
  gv.function = GXcopy;
  gv.line_width = 1;
  gv.line_style = LineOnOffDash; 
  gv.fill_style = FillOpaqueStippled;
  
  
  mb->gc = XCreateGC(mb->dpy, mb->root,
		     GCFunction|GCLineWidth|GCFillStyle|GCLineStyle, &gv);

  gv.function = GXinvert;
  gv.subwindow_mode = IncludeInferiors;

  mb->invert_gc = XCreateGC(mb->dpy, mb->root,
			    GCFunction|GCSubwindowMode|GCLineWidth
			    |GCFillStyle|GCLineStyle, &gv);

  mb->atom_mb_theme = XInternAtom(mb->dpy, "_MB_THEME_NAME",False);

  mbdesktop_get_theme_via_root_prop(mb);

  mbdesktop_bg_parse_spec(mb, mb->bg_def);

  mbdesktop_set_title_font(mb, font_title_def);

  mbdesktop_set_font(mb, font_def);

  mbdesktop_set_font_color(mb, font_col_def);
  
  mbdesktop_calculate_item_dimentions(mb);

  mbdesktop_set_scroll_buttons(mb);

  /* ewmh hints */
  window_type_atom =
    XInternAtom(mb->dpy, "_NET_WM_WINDOW_TYPE" , False);
  window_type_desktop_atom =
    XInternAtom(mb->dpy, "_NET_WM_WINDOW_TYPE_DESKTOP",False);
  desktop_manager_atom =
    XInternAtom(mb->dpy, "_NET_DESKTOP_MANGER",False);



  mb->win_top_level = XCreateWindow(mb->dpy, mb->root, 
		                    0, 0,  
				    mb->desktop_width, 
				    mb->desktop_height, 0, 
				    CopyFromParent,  
				    CopyFromParent, 
				    mb->pixbuf->vis,
				    0, NULL);

  /* Make sure with mess with no other already running desktops */
  if (XGetSelectionOwner(mb->dpy, desktop_manager_atom))
    {
      fprintf(stderr, "mbdesktop: Desktop Already running on this display.\n");
      exit(1);
    }

  /* ...And they hopefully dont mess with us.... */
  XSetSelectionOwner(mb->dpy, desktop_manager_atom, 
		     mb->win_top_level, CurrentTime);

  XStoreName(mb->dpy, mb->win_top_level, "Desktop" );

  XChangeProperty(mb->dpy, mb->win_top_level, window_type_atom, XA_ATOM, 32, 
		  PropModeReplace, 
		  (unsigned char *) &window_type_desktop_atom, 1);

  mbdesktop_set_icon(mb);

#ifdef USE_DNOTIFY
  _Gmb = mb; 			/* arg, global mb for sig callback */

  act.sa_sigaction = sig_reload_handler;
  sigemptyset(&act.sa_mask);
  act.sa_flags = SA_SIGINFO;
  sigaction(SIGRTMIN, &act, NULL);
#endif 

  mb->type_register_cnt = ITEM_TYPE_CNT;

  /* get the window ready */
  XSelectInput(mb->dpy, mb->win_top_level, 
	       ExposureMask | ButtonPressMask | ButtonReleaseMask |
	       KeyPress | KeyRelease | StructureNotifyMask |
	       FocusChangeMask );

  XSelectInput(mb->dpy, mb->root, PropertyChangeMask );

  XMapWindow(mb->dpy, mb->win_top_level);

  return mb;
}

void 
mbdesktop_scroll_up(MBDesktop *mb)
{
  int i, items_per_row;

  if (mb->scroll_offset_item->item_prev_sibling)
    {
      if (mbdesktop_current_folder_view ( mb ) == VIEW_LIST)
	{
	  mb->scroll_offset_item = mb->scroll_offset_item->item_prev_sibling;

	  if (mb->kbd_focus_item->item_prev_sibling)
	    mb->kbd_focus_item = mb->kbd_focus_item->item_prev_sibling;
	}
      else
	{
	  items_per_row = mb->workarea_width / mb->item_width;
	  for(i = 0; i < items_per_row; i++)
	    {
	      mb->scroll_offset_item = mb->scroll_offset_item->item_prev_sibling;
	      if (mb->kbd_focus_item->item_prev_sibling)
		mb->kbd_focus_item = mb->kbd_focus_item->item_prev_sibling;
	    }
	}
    }
}

void 
mbdesktop_scroll_down(MBDesktop *mb)
{
  MBDesktopItem *orig = mb->scroll_offset_item;

  if (mbdesktop_current_folder_view ( mb ) == VIEW_LIST)
    {
      if (mb->scroll_offset_item->item_next_sibling != NULL)
	mb->scroll_offset_item = mb->scroll_offset_item->item_next_sibling;
    
      if (mb->kbd_focus_item->item_next_sibling)
	mb->kbd_focus_item = mb->kbd_focus_item->item_next_sibling;

      
    } else {
      if (mb->scroll_offset_item->item_next_sibling)
	{
	  int i, items_per_row = mb->workarea_width / mb->item_width;
	  for(i = 0; i < items_per_row; i++)
	    {
	      if (mb->scroll_offset_item->item_next_sibling == NULL)
		{
		  mb->scroll_offset_item = orig;
		  return;
		} 
	      else
		mb->scroll_offset_item = mb->scroll_offset_item->item_next_sibling;
	      
	      if (mb->kbd_focus_item->item_next_sibling)
		mb->kbd_focus_item = mb->kbd_focus_item->item_next_sibling;
	    }
	}
    }
}

void
handle_button_event(MBDesktop *mb, XButtonEvent *e)
{
  XEvent ev;
  MBDesktopItem *active_item = NULL;

  active_item = mbdesktop_item_get_from_coords(mb, e->x, e->y); 
  
  if (active_item)
    {
      Bool canceled = False;
      
      if (XGrabPointer(mb->dpy, mb->win_top_level, False,
		       ButtonPressMask|ButtonReleaseMask|
		       PointerMotionMask|EnterWindowMask|LeaveWindowMask,
		       GrabModeAsync,
		       GrabModeAsync, None, None, CurrentTime)
	  == GrabSuccess)
	{
	  mbdesktop_view_item_highlight (mb, active_item, HIGHLIGHT_FILL); 

	  if (mb->have_focus && mb->had_kbd_input)
	    {
	      mbdesktop_view_item_highlight (mb, mb->kbd_focus_item, 
					     HIGHLIGHT_OUTLINE); 
	    }
	  
	  for (;;) 
	    {
	      int x1 = active_item->x;
	      int x2 = active_item->x + active_item->width;
	      int y1 = active_item->y;
	      int y2 = active_item->y + active_item->height;

	      XMaskEvent(mb->dpy,
			 ButtonPressMask|ButtonReleaseMask|
			 PointerMotionMask, &ev);
	      
	      switch (ev.type)
		{
		case MotionNotify:
		  if (canceled 
		      && ev.xmotion.x > x1
		      && ev.xmotion.x < x2
		      && ev.xmotion.y > y1
		      && ev.xmotion.y < y2
		      )
		    {
		      mbdesktop_view_item_highlight (mb, active_item, 
						     HIGHLIGHT_FILL_CLEAR); 
		      canceled = False;
		      break;
		    }
		  if (!canceled 
		      && (    ev.xmotion.x < x1
			   || ev.xmotion.x > x2
			   || ev.xmotion.y < y1
			   || ev.xmotion.y > y2
			   ))
		  {
		    mbdesktop_view_item_highlight (mb, active_item, 
						   HIGHLIGHT_FILL); 
		    canceled = True;
		    break;
		  }

		  break;
		case ButtonRelease:
		  XUngrabPointer(mb->dpy, CurrentTime);
		  if (!canceled)
		    {
		      mbdesktop_view_item_highlight (mb, active_item, 
						     HIGHLIGHT_FILL_CLEAR); 
		      if (active_item->activate_cb)
			active_item->activate_cb((void *)mb, 
						 (void *)active_item); 
		      return;
		    }
		  else
		    {
		      if (mb->have_focus && mb->had_kbd_input)
			{
			  mb->kbd_focus_item = active_item;
			  mbdesktop_view_item_highlight (mb, 
							 mb->kbd_focus_item, 
							 HIGHLIGHT_OUTLINE_CLEAR); 
			}

		      return;
		    }
		}
	    }
	}
    }
  else
    {
      if (e->y < mb->workarea_y + 20)
	{
	  if (e->x > (mb->workarea_x + mb->workarea_width -24))
	    mbdesktop_scroll_up(mb);
	  else if (e->x > (mb->workarea_x + mb->workarea_width -40))
	    mbdesktop_scroll_down(mb);

	  mbdesktop_view_paint(mb, False);
	}
    }
}

static void
handle_key_event(MBDesktop *mb, XKeyEvent *e)
{
  MBDesktopItem *active_item = NULL, *tmp_item = NULL;
  KeySym         key;
  int            i = 0;
  Bool           not_scrolled = True;
  int max_items_horiz = mb->workarea_width / mb->item_width;

  if (mb->had_kbd_input == False)
    {
      mb->had_kbd_input = True;
      mbdesktop_view_paint(mb, True);      
      return;
    }

  switch (key = XKeycodeToKeysym (mb->dpy, e->keycode, 0))
    {
    case XK_Left:
      if (mb->kbd_focus_item->item_prev_sibling)
	    {
	      active_item = mb->kbd_focus_item->item_prev_sibling;
	    } else return;
      break;
    case XK_Right:
      if (mb->kbd_focus_item->item_next_sibling)
	{
	  active_item = mb->kbd_focus_item->item_next_sibling;
	} else return;
      break;
    case XK_Up:
      if (mbdesktop_current_folder_view ( mb ) == VIEW_LIST)
	{
	  if (mb->kbd_focus_item->item_prev_sibling)
	    {
	      active_item = mb->kbd_focus_item->item_prev_sibling;
	    } else return;
	} else {
	  active_item = mb->kbd_focus_item;
	  while (i++ < max_items_horiz)
	    if ((active_item = active_item->item_prev_sibling) == NULL)
	      return;
	}
      break;
    case XK_Down:	    
      if (mbdesktop_current_folder_view ( mb ) == VIEW_LIST)
	{
	  if (mb->kbd_focus_item->item_next_sibling)
	    {
	      active_item = mb->kbd_focus_item->item_next_sibling;
	    } 
	  else return;
	} else {
	  active_item = mb->kbd_focus_item;
	  while (i++ < max_items_horiz)
	    if ((active_item = active_item->item_next_sibling) == NULL)
	      return;
	}
      break;
    case XK_Return:
    case XK_KP_Enter:
      mb->kbd_focus_item->activate_cb((void *)mb, 
				      (void *)mb->kbd_focus_item);
      return;
    }
 
  if (active_item)
    {
      if ( key == XK_Down || key == XK_Right)
	{
	  /* do we need to scroll ? */
	  if (mb->last_visible_item)
	    {
	      tmp_item = active_item;
	      while ((tmp_item = tmp_item->item_prev_sibling) != NULL)
		if (tmp_item == mb->last_visible_item->item_prev_sibling )
		  {
		    not_scrolled = False;
		    mbdesktop_scroll_down(mb);
		    break;
		  }
	    }
	}
      else
	{
	  /* do we need to scroll ? */
	  if (mb->scroll_offset_item)
	    {
	      tmp_item = active_item;
	      while ((tmp_item = tmp_item->item_next_sibling) != NULL)
		if (tmp_item == mb->scroll_offset_item)
		  {
		    not_scrolled = False;
		    mbdesktop_scroll_up(mb);
		    break;
		  }
	    }
	}
      
      /* Clear the old dashed border */
      if (not_scrolled && mb->had_kbd_input)
	{
	  mbdesktop_view_item_highlight (mb, mb->kbd_focus_item, 
					 HIGHLIGHT_OUTLINE_CLEAR); 
	  mb->kbd_focus_item = active_item;
	  mbdesktop_view_paint(mb, True);
	}
      else
	{
	  mb->kbd_focus_item = active_item;
	  mbdesktop_view_paint(mb, False);
	}
    }
}

int
mbdesktop_current_folder_view ( MBDesktop *mb )
{
  //return mb->view_type; 

  return mb->current_head_item->item_parent->view; 
}

void
mbdesktop_main(MBDesktop *mb)
{
  XEvent ev;
  Atom workarea_atom = XInternAtom(mb->dpy, "_NET_WORKAREA",False);

#ifdef USE_DNOTIFY
  sigset_t block_sigset;
  sigemptyset(&block_sigset);
  sigaddset(&block_sigset, SIGRTMIN); /* XXX should also add stuff 
					 like a HUP etc .. */
#endif

  while (1)
    {
	  XNextEvent(mb->dpy, &ev);
#ifdef USE_DNOTIFY 		/* Block dnotify signal */
	  sigprocmask(SIG_BLOCK, &block_sigset, NULL); 
#endif

#ifdef USE_XSETTINGS
	  if (mb->xsettings_client != NULL)
	    xsettings_client_process_event(mb->xsettings_client, &ev);
#endif
	  switch (ev.type) 
	    {
	    case FocusIn:
	      mb->have_focus = True;
	      mbdesktop_view_paint(mb, True);
	      break;

	    case FocusOut:
	      mbdesktop_view_paint(mb, True);
	      mb->have_focus = False;
	      break;

	    case Expose:
	      mbdesktop_view_paint(mb, True);
	      break;
	    case PropertyNotify:
	      if (ev.xproperty.atom == workarea_atom)
		{
		  int wx, wy, ww, wh;
		  if (mbdesktop_get_workarea(mb, &wx, &wy, &ww, &wh))
		    {
		      if (mb->workarea_x != wx 
			  || mb->workarea_y != wy
			  || mb->workarea_width != ww 
			  || mb->workarea_height != wh)
			mbdesktop_view_configure(mb);
		    }
		}
	      else if (ev.xproperty.atom == mb->atom_mb_theme)
		{
		  if (mbdesktop_get_theme_via_root_prop(mb))
		    mbdesktop_switch_theme (mb, NULL );
		}
	      break;
	    case ConfigureNotify:
	      
	      if ( ev.xconfigure.width != mb->desktop_width
		   || ev.xconfigure.height != mb->desktop_height)
		{
		  mb->desktop_width = ev.xconfigure.width;
		  mb->desktop_height = ev.xconfigure.height;
		  mbdesktop_view_configure(mb);
		}
	      break;
	    case ButtonPress:
	      handle_button_event(mb, &ev.xbutton);
	      break;
	    case KeyPress:
	      handle_key_event(mb, &ev.xkey);
	      break;
	    }
	  
	  if (mb->current_folder_item 
	      && mb->current_folder_item->module_handle
	      && mb->current_folder_item->module_handle->folder_xevent)
	    {
	      mb->current_folder_item->module_handle->folder_xevent(mb, mb->current_folder_item, &ev);
	    }

#ifdef USE_DNOTIFY 		/* Unblock dnotify signal */
	  sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
#endif

    }

}

static char **
get_module_list(MBDesktop *mb, int *module_cnt)
{
  char **result = NULL;
  int n_modules = 0;
  FILE *fp;
  char data[1024];
  char modules_config_path[512];
  struct stat stat_info;

  int pass = 1, i = 0;

  snprintf(modules_config_path, 512, 
	   "%s/.matchbox/mbdesktop_modules", getenv("HOME"));

  if (stat(modules_config_path, &stat_info))
    {
      snprintf(modules_config_path, 512, PKGDATADIR "/mbdesktop_modules");
    }
  
  if (!(fp = fopen(modules_config_path, "r"))) return NULL;

  while (pass != 3)
    {
      while (fgets(data,1024,fp) != NULL)
	{
	  int j = 0;
	  while (isspace(data[j])) j++;
	  
	  if (data[j] == '#' || data[j] == '\n' || data[j] == '\0')
	    continue;
	  
	  if (pass == 2)
	    {
	      if (data[strlen(data)-1] == '\n')
		data[strlen(data)-1] = '\0';
	      result[i] = strdup(&data[j]);
	      i++;
	    } 
	  else n_modules++;
	}

      if (pass == 1)
	{
	  result = (char **)malloc(sizeof(char *)*n_modules);
	  rewind(fp);
	}

      pass++;
    } 

  fclose(fp);

  /*
  result[2] = strdup(PKGDATADIR "/desktop/modules/simplefilebrowser.so");
  result[1] = strdup(PKGDATADIR "/desktop/modules/tasks.so");
  result[0] = strdup(PKGDATADIR "/desktop/modules/dotdesktop.so");
  */

  *module_cnt = n_modules;
  return result;
}

static void
modules_init (MBDesktop *mb)
{
  MBDesktopFolderModule *mod_details = NULL;
  MBDesktopItem         *folder_mod_item;
  void                  *handle = NULL;
  char                  *error;
  char                 **mods   = NULL;
  int                    n_mods = 0;
  int                    i;

  mods = get_module_list(mb, &n_mods);

  if (mods == NULL)
    {
      fprintf(stderr, "mbdesktop: failed to load modules. Exiting ...\n");
      exit(1);
    }

  for (i = 0; i < n_mods; i++ )
    { 
      char *args = NULL;

      args = strchr(mods[i], ' ');

      if (args != NULL)
	{
	  *args = '\0'; args++;
	}

      printf("mbdesktop: loading %s with args %s\n", mods[i], args);

      if (i == 0 || (i > 0 && strcmp(mods[i], mods[i-1])))
	{
	  handle = dlopen (mods[i], RTLD_LAZY |RTLD_GLOBAL);

	  if (!handle) 
	    {
	      fputs (dlerror(), stderr);
	      continue;
	    }
	  
	  mod_details = dlsym(handle, "folder_module" );

	  if ((error = dlerror()) != NULL) 
	    {
	      fputs(error, stderr);
	      free(error);
	      continue;
	    }
	}

      folder_mod_item = mod_details->folder_init(mb, mod_details, args);

      if ( folder_mod_item )
	folder_mod_item->item_parent->module_handle = mod_details;

    }
}

int 
main(int argc, char **argv)
{
  MBDesktop *mb;

  mb = mbdesktop_init(argc, argv);

  mbdesktop_view_init_bg(mb);

  /* Add the root topelevel item */
  mb->top_head_item  = mbdesktop_item_new_with_params (mb, 
						       mb->top_level_name, 
						       NULL,
						       NULL,
						       ITEM_TYPE_ROOT );
  mb->current_folder_item = mb->top_head_item;

  modules_init(mb);

  mb->kbd_focus_item = mb->current_head_item 
    = mb->scroll_offset_item = mb->top_head_item->item_child;

  mbdesktop_main(mb);

  return 0;
}
