/*
 *  minisys - tiny sys monitor
 *
 *  cpu reading code based on wmbubblemon
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>

#include <libmb/mb.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef USE_PNG
#define IMG_EXT "png"
#else
#define IMG_EXT "xpm"
#endif

#define MINISYS_IMG "minisys." IMG_EXT

struct {

   /* cpu data  */
   int loadIndex;
   int samples;
   u_int64_t *load, *total;

   /* memory data  */
   u_int64_t mem_used;
   u_int64_t mem_max;
   u_int64_t swap_used;
   u_int64_t swap_max;
   unsigned int swap_percent;  /* swap used, in percent */
   unsigned int mem_percent;   /* memory used, in percent */

} msd;   

MBPixbuf *pb = NULL;
MBPixbufImage *ImgIcon = NULL, *ImgIconScaled = NULL, *ImgGraph = NULL;

int GraphHeight = 0, GraphWidth = 0;

char *ThemeName;

/* returns current CPU load in percent, 0 to 100 */
int system_cpu(void)
{
    unsigned int cpuload;
    u_int64_t load, total, oload, ototal;
    u_int64_t ab, ac, ad, ae;
    int i;
    FILE *stat;

    stat = fopen("/proc/stat", "r");
    fscanf(stat, "%*s %Ld %Ld %Ld %Ld", &ab, &ac, &ad, &ae);
    fclose(stat);

    /* Find out the CPU load */
    /* user + sys = load
     * total = total */
    load = ab + ac + ad;	/* cpu.user + cpu.sys; */
    total = ab + ac + ad + ae;	/* cpu.total; */

    /* "i" is an index into a load history */
    i = msd.loadIndex;
    oload = msd.load[i];
    ototal = msd.total[i];

    msd.load[i] = load;
    msd.total[i] = total;
    msd.loadIndex = (i + 1) % msd.samples;

    if (ototal == 0)	
	cpuload = 0;
    else
	cpuload = (100 * (load - oload)) / (total - ototal);

    return cpuload;
}

int system_memory(void)
{
    u_int64_t my_mem_used, my_mem_max;
    u_int64_t my_swap_max;

    static int mem_delay = 0;
    FILE *mem;
    static u_int64_t aa, ab, ac, ad, ae, af, ag, ah;
    /* put this in permanent storage instead of stack */
    static char not_needed[2048];

    if (mem_delay-- <= 0) {
	mem = fopen("/proc/meminfo", "r");
	fgets(not_needed, 2048, mem);
	
	fscanf(mem, "%*s %Ld %Ld %Ld %Ld %Ld %Ld", &aa, &ab, &ac,
	       &ad, &ae, &af);
	fscanf(mem, "%*s %Ld %Ld", &ag, &ah);
	fclose(mem);
	mem_delay = 25;

	/* calculate it */
	my_mem_max = aa;	/* memory.total; */
	my_swap_max = ag;	/* swap.total; */

	my_mem_used = ah + ab - af - ae;

	/* No swap on ipaq 
	if (my_mem_used > my_mem_max) {
	   my_swap_used = my_mem_used - my_mem_max;
	    my_mem_used = my_mem_max;
	} else {
	   my_swap_used = 0;
	}
	*/
	
	msd.mem_used = my_mem_used;
	msd.mem_max = my_mem_max;
	//msd.swap_used = my_swap_used;
	//msd.swap_max = my_swap_max;

	msd.mem_percent = (100 * msd.mem_used) / msd.mem_max;
	//msd.swap_percent = (100 * msd.swap_used) / msd.swap_max;
	/* memory info changed - update things */
	return 1;
	}
    /* nothing new */
    return 0;
}


void
paint_callback (MBTrayApp *app, Drawable drw )
{
  int cpusize, memsize;
  int y;

  MBPixbufImage *img_backing = NULL;

  system_memory(); 		/* Update reading */
  
  img_backing = mb_tray_app_get_background (app, pb);

  cpusize = ( system_cpu() * GraphHeight ) / 100; 
  memsize = ( msd.mem_percent * GraphHeight ) / 100; 

  mb_pixbuf_img_copy(pb, ImgGraph, ImgGraph, 
		     1, 0, GraphWidth-1, GraphHeight, 0, 0);
  for (y = GraphHeight; y >= 0; y--)
    {
      mb_pixbuf_img_plot_pixel(pb,ImgGraph, GraphWidth-1, y, 0xff, 0xff, 0xff);
      mb_pixbuf_img_set_pixel_alpha(ImgGraph, GraphWidth-1 , y, 0);
    }

  for (y = GraphHeight; y >= 0; y--)
    if ((GraphHeight-y) < memsize)
      {
	mb_pixbuf_img_plot_pixel(pb, ImgGraph, GraphWidth-1, y, 0xff, 0, 0);
	mb_pixbuf_img_set_pixel_alpha(ImgGraph, GraphWidth-1 , y, 175);
      }

  for (y = GraphHeight; y >= 0; y--)
    if ((GraphHeight-y) < cpusize)
      {
	mb_pixbuf_img_plot_pixel(pb, ImgGraph, GraphWidth-1, y, 0, 0xff, 0);
	mb_pixbuf_img_set_pixel_alpha(ImgGraph, GraphWidth-1 , y, 200);
      }

  mb_pixbuf_img_composite(pb, img_backing, ImgIconScaled, 0, 0);

  mb_pixbuf_img_copy_composite(pb, img_backing, ImgGraph,
			       0, 0,
			       GraphWidth, GraphHeight,
			       GraphWidth/2, GraphHeight/2);

  mb_pixbuf_img_render_to_drawable(pb, img_backing, drw, 0, 0);

  mb_pixbuf_img_free(pb, img_backing);
}

void
button_callback (MBTrayApp *app, int x, int y, Bool is_released )
{
  char tray_msg[256];
  int cpu = system_cpu();

  if (!is_released)
    return;

  sprintf(tray_msg, "CPU: %i %%, MEMORY: %i %%\n",cpu, msd.mem_percent);
  mb_tray_app_tray_send_message(app, tray_msg, 5000);
}

void
resize_callback (MBTrayApp *app, int w, int h )
{

 if (ImgIconScaled) mb_pixbuf_img_free(pb, ImgIconScaled);
 if (ImgGraph) mb_pixbuf_img_free(pb, ImgGraph);

 ImgIconScaled = mb_pixbuf_img_scale(pb, ImgIcon, w, h);

 GraphWidth = w/2;
 GraphHeight = h/2;

 ImgGraph = mb_pixbuf_img_rgba_new(pb, w/2, h/2);

}

void 
load_icon(void)
{
 char *icon_path = NULL;
 
 if (ImgIcon) mb_pixbuf_img_free(pb, ImgIcon);

 icon_path = mb_dot_desktop_icon_get_full_path (ThemeName, 
						32, 
						MINISYS_IMG );

 if (icon_path == NULL 
     || !(ImgIcon = mb_pixbuf_img_new_from_file(pb, icon_path)))
    {
      fprintf(stderr, "miniapm: failed to load icon %s\n", MINISYS_IMG);
      exit(1);
    }

 free(icon_path);

 return;

}

void 
theme_callback (MBTrayApp *app, char *theme_name)
{
  if (!theme_name) return;
  if (ThemeName) free(ThemeName);
  ThemeName = strdup(theme_name);
  load_icon(); 	     
  resize_callback (app, mb_tray_app_width(app), mb_tray_app_width(app) );
}

void
timeout_callback ( MBTrayApp *app )
{
  mb_tray_app_repaint (app);
}

int
main( int argc, char *argv[])
{
  MBTrayApp *app = NULL;
  struct timeval tv;

  int i;
  u_int64_t load = 0, total = 0;

  app = mb_tray_app_new ( "CPU/Mem Monitor",
			  resize_callback,
			  paint_callback,
			  &argc,
			  &argv );  

   msd.samples = 16;
   
   if (msd.load) {
      load = msd.load[msd.loadIndex];
      free(msd.load);
   }
   
   if (msd.total) {
      total = msd.total[msd.loadIndex];
      free(msd.total);
   }
   
   msd.loadIndex = 0;
   msd.load = malloc(msd.samples * sizeof(u_int64_t));
   msd.total = malloc(msd.samples * sizeof(u_int64_t));
   for (i = 0; i < msd.samples; i++) {
      msd.load[i] = load;
      msd.total[i] = total;
   }
  
  pb = mb_pixbuf_new(mb_tray_app_xdisplay(app), 
		     mb_tray_app_xscreen(app));
   
  memset(&tv,0,sizeof(struct timeval));
  tv.tv_usec = 50000;

  mb_tray_app_set_theme_change_callback (app, theme_callback );

  mb_tray_app_set_timeout_callback (app, timeout_callback, &tv); 

  mb_tray_app_set_button_callback (app, button_callback );
  
  load_icon();

  mb_tray_app_set_icon(app, pb, ImgIcon);
  
  mb_tray_app_main (app);
   
   return 1;
}
