/*
 * Algebraic manipulator program.
 *
 * Mathomatic Copyright (c) 1996 George Gesslein II.
 *
 * Currently, this program only supports binary and unary operators.
 * Unary operators are implemented as a binary operation with a dummy operand.
 *
 * In the storage format, each level of parenthesis is indicated
 * by a level number (origin 1).  The deeper the level, the
 * higher the level number.
 *
 * The storage format for expressions is a fixed size array of elements
 * "token_type", which may be a CONSTANT, VARIABLE, or OPERATOR.
 * The array always alternates between operand (CONSTANT or VARIABLE)
 * and OPERATOR.  There is a separate integer for each array which
 * contains the current length of the expression stored in the array.
 * This length is always odd.
 *
 * Only one POWER operator is allowed per level in the storage format,
 * and no other operators may be on that level.  Same with the FACTORIAL
 * operator.
 *
 * Any number of TIMES and DIVIDE operators may be on the same
 * level, because they are simple multiplicative class operators.
 * The same for PLUS and MINUS.
 */

#include "am.h"
#include <signal.h>
#if	READLINE
#include <readline/readline.h>
#endif
#if	LINUX
#include <fenv.h>
#endif
#if	UNIX
#include <sys/ioctl.h>
#include <termios.h>
#endif

void		fphandler();
void		inthandler();
void		alarmhandler();
#if	UNIX
void		resizehandler();
#endif
void		exit_program();

int		n_tokens = DEFAULT_N_TOKENS;
int		n_equations;
int		cur_equation;

int		n_lhs[N_EQUATIONS];
int		n_rhs[N_EQUATIONS];

token_type	*lhs[N_EQUATIONS];
token_type	*rhs[N_EQUATIONS];

int		n_tlhs;
int		n_trhs;

token_type	*tlhs;
token_type	*trhs;

token_type	*scratch;

#if	LONG_VAR_NAMES
char		*var_names[MAX_VAR_NAMES];
#endif

/* Set the following to "true" if you want upper and lower case variables. */
int		case_sensitive_flag;

/* Set the following to "true" to always display equations in fraction format. */
int		groupall = true;

int		preserve_roots = false;
int		export_flag;		/* true for single line listing in exportable format */
int		screen_columns = 80;
int		screen_rows = 24;
int		debug_level;
int		domain_check;
int		color_flag = true;	/* true for ANSI color mode */
int		html_flag;		/* true for HTML color mode */
int		partial_flag;		/* true for partial unfactoring of algebraic fractions */
int		symb_flag;		/* true for "simplify symbolic" */
int		high_prec;		/* true while saving equations */
int		input_column;
int		sign_flag;		/* true when all "sign" variables are to compare equal */
int		usage_flag;
int		piping_in_flag;		/* true if input is not a terminal */
char		*tmp_file = "mathxxx.tmp";
double		small_epsilon;		/* for small accumulated round-off errors */
double		epsilon;		/* for larger accumulated round-off errors */
char		var_str[MAX_VAR_LEN+80];	/* temp storage for variable names */
char		prompt_str[MAX_VAR_LEN+80];	/* temp storage for prompt strings */

double		unique[64];
int		ucnt[64];
int		uno;

sign_array_type	sign_array;

FILE		*gfp;

jmp_buf		jmp_save;

int
main(argc, argv)
int	argc;
char	**argv;
{
	extern char	*optarg;
	extern int	optind;

	int		i;
	char		*cp;
	double		numerator, denominator;
	double		multiplier;
	int		coption;
#if	UNIX
	struct winsize	ws;
#endif

	piping_in_flag = !isatty(0);
	init_gvars();
	gfp = stdout;
	setbuf(stdout, NULL);		/* make standard output unbuffered */
	small_epsilon =	0.000000000000005;	/* don't increase this */
	epsilon =	0.00000000000005;	/* the correct value for doubles */

#if	UNIX
	ws.ws_col = 0;
	ws.ws_row = 0;
	ioctl(1, TIOCGWINSZ, &ws);
	if (ws.ws_col) {
		screen_columns = ws.ws_col;
	}
	if (ws.ws_row) {
		screen_rows = ws.ws_row;
	}
#endif
	coption = false;
	while ((i = getopt(argc, argv, "chm:")) >= 0) {
		switch (i) {
		case 'c':
			coption = true;
			break;
		case 'h':
			html_flag = true;
			break;
		case 'm':
			multiplier = atof(optarg);
			if (multiplier <= 0.0 || (n_tokens = (int) (multiplier * DEFAULT_N_TOKENS)) <= 0) {
				printf(_("Invalid memory multiplier specified!\n"));
				exit(1);
			}
			break;
		default:
			exit(1);
		}
	}
	if (html_flag) {
		screen_columns = 0;
		screen_rows = 0;
		printf("<pre><font size=\"+1\">\n");
		printf(_("HTML color mode enabled.\n"));
	}
	printf(_("\nMathomatic Version %s\n"), VERSION);
	printf(_("Copyright (C) 1987-2004 George Gesslein II.\n"));
	init_mem();
	if (coption) {
		if (color_flag) {
			printf(_("ANSI color mode toggled off by the \"-c\" option.\n"));
			color_flag = false;
		} else {
			printf(_("ANSI color mode enabled.\n"));
			color_flag = true;
		}
	}
	printf("\n");
	default_color();
	if ((i = setjmp(jmp_save)) != 0) {
		clean_up();
		switch (i) {
		case 3:
			break;
		case 13:
			printf(_("Operation abruptly aborted.\n"));
			break;
		case 14:
			printf(_("Expression too big!\n"));
		default:
			printf(_("Operation aborted.\n"));
		}
	} else {
#if	LINUX
		feenableexcept(FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW);
#endif
		signal(SIGFPE, fphandler);
		signal(SIGINT, inthandler);
#if	UNIX
		signal(SIGWINCH, resizehandler);
#endif
#if	TIMEOUT_SECONDS
		signal(SIGALRM, alarmhandler);
		alarm(TIMEOUT_SECONDS);
#endif
		if (!f_to_fraction(0.5, &numerator, &denominator)
		    || !f_to_fraction(1.0/3.0, &numerator, &denominator)) {
			printf(_("Cannot convert floating point values to fractions.\n"));
		}
#if	!SECURE
		for (i = optind; i < argc; i++) {
			if (!read_in(argv[i])) {
				exit_program();
			}
		}
#endif
	}
	for (;;) {
		sprintf(prompt_str, "%d%s", cur_equation + 1, html_flag ? HTML_PROMPT : PROMPT);
		if ((cp = getstring((char *) &tlhs[0], n_tokens * sizeof(token_type))) == NULL)
			break;
		process(cp);
	}
	exit_program();
}

/*
 * Allocate the equation spaces.
 */
init_mem()
{
	int	i;
	char	*junk;

	if ((junk = malloc(8192)) == NULL
	    || (scratch = (token_type *) malloc((n_tokens + 10) * sizeof(token_type))) == NULL) {
		printf(_("Not enough memory.\n"));
		exit(1);
	}
	if ((tlhs = (token_type *) malloc(n_tokens * sizeof(token_type))) == NULL
	    || (trhs = (token_type *) malloc(n_tokens * sizeof(token_type))) == NULL) {
		printf(_("Not enough memory.\n"));
		exit(1);
	}
	for (i = 0; i < N_EQUATIONS; i++) {
		lhs[i] = (token_type *) malloc(n_tokens * sizeof(token_type));
		if (lhs[i] == NULL)
			break;
		rhs[i] = (token_type *) malloc(n_tokens * sizeof(token_type));
		if (rhs[i] == NULL) {
			free((char *) lhs[i]);
			break;
		}
	}
	free(junk);
	n_equations = i;
	if (n_equations == 0) {
		printf(_("Not enough memory.\n"));
		exit(1);
	}
	clear("all");
	printf(_("%d equation spaces allocated (total size is %ld KBytes).\n"),
	    n_equations, ((long) n_tokens * sizeof(token_type) * 2L * n_equations) / 1000L);
}

void
fphandler(sig)
int	sig;
{
#if	LINUX
	printf(_("Floating point exception!\n"));
	feenableexcept(FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW);
	signal(SIGFPE, fphandler);
#else
	printf(_("Floating point exception!\n"));
	signal(SIGFPE, fphandler);
#endif
	longjmp(jmp_save, 2);
}

#if	MATHERR		/* define this if matherr() function is supported (it usually isn't) */
int
matherr(x)
struct exception	*x;
{
	switch (x->type) {
	case DOMAIN:
		if (domain_check) {
			domain_check = false;
			return 1;
		}
		printf(_("Domain error in constant!\n"));
		break;
	case SING:
		printf(_("Singularity error in constant!\n"));
		break;
	case OVERFLOW:
		printf(_("Overflow error in constant!\n"));
		break;
	case UNDERFLOW:
		printf(_("Underflow error in constant!\n"));
		break;
	case TLOSS:
		printf(_("Total loss of significance in constant!\n"));
		break;
	case PLOSS:
		printf(_("Partial loss of significance in constant!\n"));
		break;
	default:
		printf(_("Unknown error in constant!\n"));
		break;
	}
	longjmp(jmp_save, 2);
}

check_err()
{
}
#else
check_err()
{
	switch (errno) {
	case EDOM:
		errno = 0;
		if (domain_check) {
			domain_check = false;
		} else {
			printf(_("Domain error in constant!\n"));
			longjmp(jmp_save, 2);
		}
		break;
	case ERANGE:
		errno = 0;
		printf(_("Overflow error in constant!\n"));
		longjmp(jmp_save, 2);
		break;
	}
}
#endif

/*
 * Control-C handler.
 */
void
inthandler(sig)
int	sig;
{
	exit_program();
}

/*
 * Alarm signal handler.
 */
void
alarmhandler(sig)
int	sig;
{
	printf(_("timeout\n"));
	exit_program();
}

#if	UNIX
/*
 * Window resize handler.
 */
void
resizehandler(sig)
int	sig;
{
	struct winsize	ws;

	ws.ws_col = 0;
	ws.ws_row = 0;
	ioctl(1, TIOCGWINSZ, &ws);
	if (ws.ws_col && screen_columns) {
		screen_columns = ws.ws_col;
	}
	if (ws.ws_row && screen_rows) {
		screen_rows = ws.ws_row;
	}
}
#endif

/*
 * Exit this program and return to the Operating System.
 */
void
exit_program()
{
	printf("\n");
	reset_attr();
	printf(_("Thank you for using Mathomatic!\n"));
	printf(_("Please visit \"http://www.mathomatic.com\" for info and latest versions.\n"));
	if (html_flag) {
		printf("</pre>\n");
	}
	exit(0);
}

/*
 * Common routine to output a prompt
 * and get a line of input from stdin.
 * All Mathomatic input comes from this routine,
 * except for file reading.
 *
 * Returns "string".
 */
char	*
getstring(string, n)
char	*string;	/* storage for input string */
int	n;		/* maximum size of "string" in bytes */
{
	char	*cp;

	if (piping_in_flag)
		prompt_str[0] = '\0';
	input_column = strlen(prompt_str);
#if	READLINE
	cp = readline(prompt_str);
	if (cp == NULL)
		exit_program();
	strncpy(string, cp, n);
	string[n-1] = '\0';
	if (string[0]) {
		add_history(cp);
	} else {
		free(cp);
	}
	return string;
#else
	printf("%s", prompt_str);
	if (fgets(string, n, stdin) == NULL)
		exit_program();
	return string;
#endif
}
