
#
#    ========== licence begin LGPL
#    Copyright (C) 2002 SAP AG
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#    ========== licence end
#

import string

class SchemaCompiler:
    def __init__ (self, schema = None):
        self.schema = schema

    def compileSchema (self, schema, stream, fname = ''):
        self.schema = schema
        self.writeIntro (stream, fname)
        for element in self.schema.elements:
            self.compileElement (element, stream)
        self.writeFactory (stream)

class Schema2Python (SchemaCompiler):

    def defaultName (self, base):
        return base + 'DTD.py'

    def writeIntro (self, stream, fname = ''):
        stream.write ("""# """ + fname + """
# Do not edit this file
# it was generated by fomc, the FOM compiler for Python

from xmls.FOM.runtime.FOM import CompiledElement, CompiledListElement, CompiledPCData
from xmls.FOM.runtime.Reader import AbstractReader
""")

    def compileElement (self, element, stream):
        if element.isGroup == 'true':
            return
        isList = element.isList ()
        if isList:
            superClass = 'CompiledListElement'
        elif not element.varnames ():
            superClass = 'CompiledPCData'
        else:
            superClass = 'CompiledElement'
        stream.write ("""
class %s (%s):
    name = '%s'
""" % (element.identifier (), superClass, element.identifier ()))
        if element.prefix:
            stream.write ('    namespace = ' + `element.prefix` + '\n')
        self.declareAttributes (element, stream)
        self.declareContent (element, stream)
        self.declareAttributeAccessors (element, stream)
        self.declareContentAccessors (element, stream)

    def declareAttributes (self, element, stream):
        attributes = element.attributes
        if attributes:
            stream.write ("    attributes = [\n")
            for attribute in attributes:
                stream.write ("        ('%s', %s),\n"
                    % (attribute.name, repr (attribute.default ())))
            stream.write ("        ]\n")
            for attribute in attributes:
                stream.write ("    %s = %s\n"
                    % (normalizedIdentifier (attribute.name), repr (attribute.default ())))

    def declareContent (self, element, stream):
        varnames = element.varnames ()
        if varnames:
            stream.write ("    elements = [\n")
            for name in varnames:
                stream.write ("        '" + name + "',\n")
            stream.write ("        ]\n")
            for name in varnames:
                stream.write ("    %s = None\n" % name)

    simpleAttributeSetter = """
    def set_%(name)s (self, value):
        self.%(name)s = value
"""
    convertingAttributeSetter = """
    def set_%(name)s (self, value):
        self.%(name)s = %(valueType)s (value)
"""
    simpleAccessors = """
    def add_%(name)s (self, value):
        self.%(name)s = value
"""
    listAccessors = """
    def add_%(name)s (self, value):
        if self.%(name)s is None:
            self.%(name)s = [value]
        else:
            self.%(name)s.append (value)
"""

    def declareAttributeAccessors (self, element, stream):
        attributes = element.attributes
        if attributes:
            for attribute in attributes:
                name = normalizedIdentifier (attribute.name)
                valueType = attribute.type
                if valueType:
                    stream.write (self.convertingAttributeSetter % locals ())
                else:
                    stream.write (self.simpleAttributeSetter % locals ())

    def declareContentAccessors (self, element, stream):
        varnames = element.varnames ()
        listnames = element.listnames ()
        if varnames:
            for name in varnames:
                if name in listnames:
                    stream.write (self.listAccessors % locals ())
                else:
                    stream.write (self.simpleAccessors % locals ())
                    element = self.schema.get (name)
                    if element is None:
                        continue
                    if element.isGroup:
                        stream.write ('\n')
                        for alternative in element.elements:
                            altName = alternative.name
                            stream.write ('    add_%s = add_%s\n' % (altName, name))

    def writeFactory (self, stream):
        stream.write ("""
class Reader (AbstractReader):
    mapping = {
""")
        for element in self.schema.elements:
            if element.isGroup == 'true':
                continue
            stream.write ('        %-30s: %s,\n'
                % ('"' + element.displayName () + '"', element.identifier ()))
        stream.write ("""    }

""")

def normalizedIdentifier (identifier):
    identifier = string.replace (identifier, ':', '_')
    identifier = string.replace (identifier, '-', '_')
    return identifier

class Schema2SubPython (SchemaCompiler):
    def defaultName (self, base):
        self.base = base
        return base + 'Imp.py'

    def writeIntro (self, stream, fname = ''):
        stream.write ("""# %s
from xmls.FOM.runtime.Reader import AbstractReader
import %s
superModule = %s
""" % (fname, self.base, self.base))

    def compileElement (self, element, stream):
        stream.write ("""
class %s (superModule.%s):
    pass

""" % (element.name, element.name))

    def writeFactory (self, stream):
        stream.write ("""
class Reader (AbstractReader):
    mapping = {
""")
        for element in self.schema.elements:
            stream.write ('        %-30s: %s,\n'
                % ('"' + element.displayName () + '"', element.identifier ()))
        stream.write ("""    }

""")


