/*      Copyright (C) 2002, 2003, 2004 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/


/* modes of operation.
 *    --center    vary between center and max (dag only)
 *       default: vary between self and max (works for both ite and dag)
*/

#include <string.h>

#include "report.h"

#include "impala/matrix.h"
#include "impala/vector.h"
#include "impala/ivp.h"
#include "impala/io.h"

#include "mcl/interpret.h"
#include "mcl/clm.h"

#include "util/types.h"
#include "util/err.h"
#include "util/ting.h"
#include "util/minmax.h"
#include "util/opt.h"
#include "util/array.h"


enum
{  OPT_ENSTRICT
,  OPT_OUTPUTFILE
,  OPT_INPUTFILE
,  OPT_TIGHT
,  OPT_TSTEP
,  OPT_TBOUND
,  OPT_HELP
,  OPT_USE_CENTER
,  OPT_SORT
,  OPT_DAG
,  OPT_VERSION
}  ;
   

/*    1: has argument                        */
/*    2: may occur in file argument place    */
/*    4: experimental option                 */

mcxOptAnchor options[] =
{  {  "--enstrict",        0, OPT_ENSTRICT }
,  {  "--version",         0, OPT_VERSION }
,  {  "-o",                1, OPT_OUTPUTFILE }
,  {  "-imx",              1, OPT_INPUTFILE }
,  {  "-t",                1, OPT_TIGHT }
,  {  "-tstep",            1, OPT_TSTEP }
,  {  "-tbound",           1, OPT_TBOUND }
,  {  "-h",                0, OPT_HELP }
,  {  "-sort",             1, OPT_SORT }
,  {  "-dag",              1, OPT_DAG }
,  {  "--center",          0, OPT_USE_CENTER }
,  {  NULL,                0, 0 }
}  ;

const char* usagelines[] =
{  "Usage: clmimac [options]"
,  ""
,  "Options (marked * are obligatory):"
,  "  -imx <fname> *   input matrix file, presumably dumped mcl iterand or dag"
,  "  -o <fname>       output cluster file"
,  "  -t <num>         (tightness) double in range [0.0,100.0]"
,  "  -tstep <num>     step size for multiple tightness (default off)"
,  "  -tbound <num>    bound on tightness in the presence of steps"
,  "  --center         vary tightness over [center, max] interval"
,  "  -sort <str>      sort mode for clusters, str one of <revsize|size|lex>"
,  "  -dag <fname>     output DAG associated with matrix"
,  "  --enstrict       clean up clustering if not a partition"
,  NULL
}  ;

int main
(  int                  argc
,  const char*          argv[]
)  
   {  mcxstatus parseStatus   =  STATUS_OK

   ;  mcxbool keep_overlap    =  TRUE
   ;  mcxbool use_center      =  FALSE
   ;  mclInterpretParam* ipp  =  mclInterpretParamNew()
   ;  mcxTing* fnstem         =  NULL
   ;  mcxIO* xfdag            =  NULL
   ;  const char* sortmode    =  "revsize"

   ;  int flags = 0

   ;  mcxIO* xfin
   ;  mclx *mx = NULL, *cl = NULL, *dag = NULL
   ;  int o, e, m
   ;  double t = 0.0
   ;  double toffset =  0.0         /* tight factor */
   ;  double tbound  =  100.0       /* tight factor */
   ;  double tstep   =  200.0       /* tight step */
   ;  const char* me = "clmimac"
   ;  const char* arg_fnin = NULL

   ;  mcxHash* hopts
   ;  mcxOptList* lopts

   ;  ipp->w_selfval = 0.999
   ;  ipp->w_maxval  = 0.001
   ;  ipp->w_center  = 0.0

   ;  if (argc <= 1)
      {  mcxUsage(stdout, me, usagelines)
      ;  exit(0)
   ;  }

      hopts    =  mcxOptHash(options, NULL)
   ;  lopts    =  mcxOptParse
                  (hopts, (char**) argv, argc, 1, 0, &parseStatus)

  /*  fixme: -imx argument rather than fixed suffix argument ?
  */
   ;  if (parseStatus != STATUS_OK)
      {  mcxErr(me, "initialization failed")
      ;  exit(1)
   ;  }

      while (lopts && lopts->tag)
      {  mcxOptAnchor* anch = mcxOptFind(lopts->tag, hopts)

      ;  if (!anch)                   /* huh? impossible? fixme< */
         {  lopts = lopts->next
         ;  continue
      ;  }

         if (anch->flags & 4)
         {  mcxWarn
            (  me
            ,  "EXPERIMENTAL using option <%s>, may produce unexpected results"
            ,  anch->tag
            )
      ;  }

         switch(anch->id)
         {  case OPT_ENSTRICT
         :  keep_overlap = FALSE
         ;  break
         ;

            case OPT_VERSION
         :  report_version(me)
         ;  exit(0)
         ;

            case OPT_INPUTFILE
         :  arg_fnin = lopts->val
         ;  break
         ;

            case OPT_OUTPUTFILE
         :  fnstem = mcxTingNew(lopts->val)
         ;  break
         ;

            case OPT_USE_CENTER
         :  use_center = TRUE
         ;  break
         ;

            case OPT_HELP
         :  mcxUsage(stdout, me, usagelines)
         ;  exit(0)
         ;

            case OPT_TBOUND
         :  tbound = atof(lopts->val)
         ;  break
         ;

            case OPT_TSTEP
         :  tstep = atof(lopts->val)
         ;  break
         ;

            case OPT_TIGHT
         :  toffset = atof(lopts->val)
         ;  break
         ;

            case OPT_DAG
         :  xfdag = mcxIOnew(lopts->val, "w")
         ;  break
         ;

            case OPT_SORT
         :  sortmode = lopts->val
         ;  break
         ;

            default
         :  mcxExit(1) 
      ;  }

         lopts = lopts->next
   ;  }

      if (!arg_fnin)
      {  mcxErr(me, "-imx option is required")
      ;  mcxExit(0)
   ;  }

      xfin     =  mcxIOnew(arg_fnin, "r")
   ;  mx       =  mclxRead(xfin, EXIT_ON_FAIL)
   ;  report_graph_or_exit(me, mx, xfin->fn)
   ;  mclxMakeStochastic(mx)

   ;  if (!fnstem)
      fnstem = mcxTingNew("out.imac")
   ;

      for (t = toffset; t <= tbound; t += tstep)
      {  mcxIO* xfout
      ;  int dagdepth

      ;  xfout =  mcxIOnew(fnstem->str, "w")
      ;  if (tstep <= 100.0)
         mcxTingPrintAfter(xfout->fn, "%02.0f", t)

      ;  ipp->w_maxval = t / 100.0
      ;  ipp->w_partialsum = 0.0

      ;  if (use_center)
            ipp->w_selfval =  0.0
         ,  ipp->w_center  =  (100.0 - t) / 100.0
      ;  else
            ipp->w_selfval =  (100.0 - t) / 100.0
         ,  ipp->w_center  =  0.0

      ;  cl = mclInterpret(mx, ipp, &dagdepth, &dag)

      ;  mcxTell(me, "dagdepth <%d>", dagdepth)

      ;  if (xfdag)
         mclxWrite(dag, xfdag, MCLXIO_VALUE_GETENV, RETURN_ON_FAIL)
      ;  else
         mclxFree(&dag)

      ;  if (keep_overlap)
         flags |= ENSTRICT_KEEP_OVERLAP

      ;  mclcEnstrict(cl, &o, &m, &e, flags)  

      ;  if (o)
         mcxTell
         (  me
         ,  "%s <%d> instances of overlap (t parameter %.0f)"
         ,  keep_overlap ? "found" : "removed"
         ,  (int) o
         ,  (double) t
         )

      ;  if (m)
         mcxTell
         (  me
         ,  "collected <%d> missing nodes (t parameter %.0f)"
         ,  (int) m
         ,  (double) t
         )

      ;  if (e)
         mcxTell
         (  me
         ,  "removed <%d> empty clusters (t parameter %.0f)"
         ,  (int) e
         ,  (double) t
         )

      ;  if (!strcmp(sortmode, "size"))
         mclxColumnsRealign(cl, mclvSizeCmp)
      ;  else if (!strcmp(sortmode, "revsize"))
         mclxColumnsRealign(cl, mclvSizeRevCmp)
      ;  else if (!strcmp(sortmode, "lex"))
         mclxColumnsRealign(cl, mclvLexCmp)
      ;  else if (!strcmp(sortmode, "none"))
        /* ok */
      ;  else
         mcxErr(me, "ignoring unknown sort mode <%s>", sortmode)

      ;  mclxWrite(cl, xfout, MCLXIO_VALUE_NONE, EXIT_ON_FAIL)
      ;  mcxIOfree(&xfout)
   ;  }

      return 0
;  }


