/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_DOWNLOADTASK_H_
#define MCLOUD_API_DOWNLOADTASK_H_

#include <mcloud/api/visibility.h>
#include <mcloud/api/task.h>

#include <memory>

namespace mcloud {
namespace api {

class TaskHandler;

/*!
    \struct DownloadBufferCb
    \brief DownloadBufferCb is a download request object which consists of a content id and writing callback function allows content data can be received in buffering via call back function.
 */
struct MCLOUD_API_DLL_PUBLIC DownloadBufferCb
{
    std::string content_id;
    Task::Buffer_Callback write_cb;
};

class DownloadTaskPriv;
class SyncManagerPriv;

/*!
    \class DownloadTask
    \brief DownloadTask is a task item that can be accessed from application layer to fetch
    basic download item information, also it's used for content download by sync manager.
 */
class MCLOUD_API_DLL_PUBLIC DownloadTask : public Task {
public:
    typedef std::shared_ptr<DownloadTask> Ptr;

    virtual ~DownloadTask() = default;

    DownloadTask(const DownloadTask& ) = delete;

    DownloadTask& operator=(const DownloadTask& ) = delete;

    /*!
     * Returns an unique id of download task.
     */
    const std::string & task_id() const override;

    /*!
     * \brief Returns an unique id of cloud content on mcloud.
     */
    const std::string & content_id() const override;

    /*!
     * \brief Returns a display name of cloud content on mcloud.
     */
    const std::string & content_name() const override;

    /*!
     * \brief Returns cloud content local storage file path.
     */
    const std::string & file_path() const override;

    /*!
     * \brief Returns download url assigned by mcloud for this task.
     * \note this url will be expried after a period of time.
     */
    const std::string & task_url() const override;

    /*!
     * \brief Contains the error string if an error occurs during content download.
     */
    const std::string & error_string() const override;

    /*!
     * \brief Returns current sync-up status for this task.
     * \sa Task::TaskStatus
     */
    Task::Status status() const override;

    /*!
     * \brief Handler for download progress of a task.
     * \sa Task::ProgressHandler
     */
    Task::ProgressHandler & progress_changed() override;

    /*!
     * \brief Handler for download status of a task.
     * \sa Task::StatusHandler
     */
    Task::StatusHandler & status_changed() override;

    /*! 
     * \brief cancel the task.
     * \sa Task::StatusHandler
     */
    void cancel() override;

private:
    DownloadTask(std::shared_ptr<DownloadTaskPriv> p);

    friend class SyncManagerPriv;

    std::shared_ptr<DownloadTaskPriv>   p_;
};

}
}

#endif // MCLOUD_API_DOWNLOADTASK_H_
