/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#include "uploadtask_priv.h"
#include "taskhandler.h"

#include <tinyxml2.h>
#include <boost/filesystem.hpp>

#include <iostream>

namespace http = core::net::http;
namespace fs = boost::filesystem;

using namespace mcloud::api;
using namespace std;

UploadTaskPriv::UploadTaskPriv(const tinyxml2::XMLElement *root,
         const std::string &task_id,
         const std::string &redirection_url,
         const std::string &file_path)
    : file_path_(file_path),
      file_size_(fs::file_size(file_path_)),
      upload_task_id_(task_id),
      task_url_(redirection_url),
      progress_handler_(nullptr),
      status_handler_(nullptr),
      task_handler_(std::make_shared<TaskHandler>()) {
    fillRequest(root);
}

UploadTaskPriv::UploadTaskPriv(const tinyxml2::XMLElement *root,
         const std::string &task_id,
         const std::string &redirection_url,
         size_t buffer_size,
         Task::Buffer_Callback buffer_callback)
    : file_size_(buffer_size),
      upload_task_id_(task_id),
      task_url_(redirection_url),
      progress_handler_(nullptr),
      status_handler_(nullptr),
      buffer_callback_(buffer_callback),
      task_handler_(std::make_shared<TaskHandler>()) {
    fillRequest(root);
}

void UploadTaskPriv::fillRequest(const tinyxml2::XMLElement *root) {
    content_id_ = root->FirstChildElement("contentID")->GetText();
    content_name_ = root->FirstChildElement("contentName")->GetText();
    is_need_upload_ = !strcmp(root->FirstChildElement("isNeedUpload")->GetText(), "1");

#ifndef NDEBUG
    cout <<"is_need_upload_:    " << is_need_upload_ << endl;
#endif

    //If file exists in cloud, then set status to complete to avoid duplicate upload
    if (!is_need_upload_) {
        task_handler_->set_status(Task::Status::Complete);
    }

    task_handler_->on_prepare() = [this](void* params) {
        //change status
        task_handler_->set_status(Task::Status::Running);

        //populate header info
        http::Header *header = reinterpret_cast<http::Header *>(params);

        std::string content_type;
        std::string suffix = fs::path(file_path_).extension().string();
        if (suffix == "gif" || suffix == "jpg" || suffix == "jpeg" || suffix == "png") {
            content_type = "image/"+ suffix +"; name=" + content_name_;
        } else if (suffix == "avi" || suffix == "mp4" || suffix == "mkv") {
            content_type = "video/"+ suffix +"; name=" + content_name_;
        } else if (suffix == "wav" || suffix == "ogg") {
            content_type = "audio/"+ suffix +"; name=" + content_name_;
        } else if (suffix == "mp3") {
            content_type = "audio/mpeg; name=" + content_name_;
        } else if (suffix == "ppt") {
            content_type = "application/vnd.ms-powerpoint; name=" + content_name_;
        } else if (suffix == "pptx") {
            content_type = "application/vnd.openxmlformats-officedocument.presentationml.presentation; name="
                + content_name_;
        } else if (suffix == "doc") {
            content_type = "application/msword; name=" + content_name_;
        } else if (suffix == "docx") {
            content_type =
                "application/vnd.openxmlformats-officedocument.wordprocessingml.document; name="
                + content_name_;
        } else if (suffix == "xls") {
            content_type = "application/vnd.ms-excel; name=" + content_name_;
        } else if (suffix == "xlsx") {
            content_type = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet; name="
                + content_name_;
        } else if (suffix == "odt") {
            content_type = "application/vnd.oasis.opendocument.text; name=" + content_name_;
        } else if (suffix == "ods") {
            content_type = "application/vnd.oasis.opendocument.spreadsheet; name=" + content_name_;
        } else if (suffix == "odp") {
            content_type = "application/vnd.oasis.opendocument.presentation; name=" + content_name_;
        } else if (suffix == "txt") {
            content_type = "text/plain; name=" + content_name_;
        } else {
            content_type = "application/octet-stream; name=" + content_name_;
        }
        header->add("Content-Type", content_type);

        //mcloud proprioty header spec
        std::string size_str = std::to_string(file_size_);
        header->add("Range", "bytes=0-" + size_str);
        header->add("RangeType", "2");
        header->add("ContentSize", size_str);
        header->add("UploadtaskID", upload_task_id_);
        header->add("Content-Length", size_str);

        if (buffer_callback_ != nullptr)
            return;

        if (ifs_.is_open())
            ifs_.close();

        ifs_.open(file_path_);
    };

    task_handler_->on_progress() = [this](float percent) {
        if (progress_handler_) {
            progress_handler_(percent);
        }
    };

    task_handler_->on_status() = [this](Task::Status status) {
        if (status_handler_)  {
            status_handler_(status);
        }
    };

    task_handler_->on_finished() = [this]() -> bool {
        if (ifs_.is_open())
            ifs_.close();

        //change status
        task_handler_->reset_broken_counter();
        task_handler_->set_status(Task::Status::Complete);

        return true;
    };

    task_handler_->on_ready_read() = [this](const string& data) {
        tinyxml2::XMLDocument doc;
        tinyxml2::XMLError error = doc.Parse(data.c_str());
        const tinyxml2::XMLElement *result = doc.FirstChildElement();

        if (error != tinyxml2::XML_SUCCESS && !strcmp(result->Attribute("resultCode"), "0")) {
            cerr << "upload content interrupted " << endl;
        }
    };
}

const string & UploadTaskPriv::task_id() const {
    return upload_task_id_;
}

const string & UploadTaskPriv::content_id() const {
    return content_id_;
}

const string & UploadTaskPriv::content_name() const {
    return content_name_;
}

const string & UploadTaskPriv::file_path() const {
    return file_path_;
}

const string & UploadTaskPriv::task_url() const {
    return task_url_;
}

Task::Status UploadTaskPriv::status() const {
    return task_handler_->status();
}

size_t UploadTaskPriv::file_size() const {
    return file_size_;
}

bool UploadTaskPriv::is_need_upload() const {
    return is_need_upload_;
}

std::ifstream & UploadTaskPriv::ifstream() {
    return ifs_;
}

const string & UploadTaskPriv::error_string() const {
    return error_string_; 
}

void UploadTaskPriv::set_error_string(const string &err) {
    error_string_ = err; 
}

Task::ProgressHandler & UploadTaskPriv::progress_changed() {
    return progress_handler_;
}

Task::StatusHandler & UploadTaskPriv::status_changed() {
    return status_handler_;
}

Task::Buffer_Callback & UploadTaskPriv::buffer_callback() {
    return buffer_callback_;
}

void UploadTaskPriv::cancel() {
    task_handler_->cancel();
}

std::shared_ptr<TaskHandler> UploadTaskPriv::task_handler() const {
    return task_handler_;
}

