/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "channel_manager.hh"
#include "log.hh"
#include "scope_log.hh"
#include "io.hh"
#include "application.hh"

static const DvbParameter inversion_list[] =
{
	{ "INVERSION_OFF", INVERSION_OFF },
	{ "INVERSION_ON", INVERSION_ON },
	{ "INVERSION_AUTO",  INVERSION_AUTO },
	{ NULL }
};

static const DvbParameter fec_list[] =
{
	{ "FEC_NONE", FEC_NONE },
	{ "FEC_1_2", FEC_1_2 },
	{ "FEC_2_3", FEC_2_3 },
	{ "FEC_3_4", FEC_3_4 },
	{ "FEC_4_5", FEC_4_5 },
	{ "FEC_5_6", FEC_5_6 },
	{ "FEC_6_7", FEC_6_7 },
	{ "FEC_7_8", FEC_7_8 },
	{ "FEC_8_9", FEC_8_9 },
	{ "FEC_AUTO", FEC_AUTO },
	{ NULL }
};

static const DvbParameter bw_list [] =
{
	{ "BANDWIDTH_6_MHZ", BANDWIDTH_6_MHZ },
	{ "BANDWIDTH_7_MHZ", BANDWIDTH_7_MHZ },
	{ "BANDWIDTH_8_MHZ", BANDWIDTH_8_MHZ },
	{ NULL }
};

static const DvbParameter guard_list [] =
{
	{ "GUARD_INTERVAL_AUTO", GUARD_INTERVAL_AUTO },
	{ "GUARD_INTERVAL_1_4", GUARD_INTERVAL_1_4 },
	{ "GUARD_INTERVAL_1_8", GUARD_INTERVAL_1_8 },
	{ "GUARD_INTERVAL_1_16", GUARD_INTERVAL_1_16 },
	{ "GUARD_INTERVAL_1_32", GUARD_INTERVAL_1_32 },
	{ NULL }
};

static const DvbParameter hierarchy_list [] =
{
	{ "HIERARCHY_NONE", HIERARCHY_NONE },
	{ "HIERARCHY_1", HIERARCHY_1 },
	{ "HIERARCHY_2", HIERARCHY_2 },
	{ "HIERARCHY_4", HIERARCHY_4 },
	{ "HIERARCHY_AUTO", HIERARCHY_AUTO },
	{ NULL }
};

static const DvbParameter modulation_list [] =
{
	{ "QPSK", QPSK },
	{ "QAM_16", QAM_16 },
	{ "QAM_32", QAM_32 },
	{ "QAM_64", QAM_64 },
	{ "QAM_128", QAM_128 },
	{ "QAM_256", QAM_256 },
	{ "QAM_AUTO", QAM_AUTO },
	{ "8VSB", VSB_8 },
	{ "16VSB", VSB_16 },
	{ NULL }
};

static const DvbParameter transmissionmode_list [] =
{
	{ "TRANSMISSION_MODE_2K", TRANSMISSION_MODE_2K },
	{ "TRANSMISSION_MODE_8K", TRANSMISSION_MODE_8K },
	{ "TRANSMISSION_MODE_AUTO", TRANSMISSION_MODE_AUTO },
	{ NULL }
};

ChannelManager::ChannelManager()
{
	channels_by_order = NULL;
	transponders = g_hash_table_new ( g_int_hash, g_int_equal );
	channels = g_hash_table_new ( g_str_hash, g_str_equal );
}

gint channel_sort_transponder_service_id_func(gconstpointer a, gconstpointer b)
{
	Channel* channel_a = (Channel*)a;
	Channel* channel_b = (Channel*)b;
	return (channel_a->get_transponder().get_frequency() - channel_b->get_transponder().get_frequency()) * 2 +
		channel_a->service_id - channel_b->service_id;
}

gint channel_sort_alphabetically_func(gconstpointer a, gconstpointer b)
{
	Channel* channel_a = (Channel*)a;
	Channel* channel_b = (Channel*)b;
	return g_ascii_strcasecmp(channel_a->name.c_str(), channel_b->name.c_str());
}


ChannelManager::~ChannelManager()
{
	if (transponders != NULL)
	{
		g_hash_table_destroy(transponders);
		transponders = NULL;
	}
	
	if (channels != NULL)
	{
		g_hash_table_destroy(channels);
		channels = NULL;
	}
}

void ChannelManager::load(int frontend_type, const String& path)
{
	IO::Channel channel_file(path, O_RDONLY);
	
	channel_file.set_encoding(NULL);
	
	String file_contents = channel_file.read_to_end();
	const gchar* contents = file_contents.c_str();
		
	if (!g_utf8_validate(contents, -1, NULL))
	{
		throw Exception(_("Me TV cannot use a channels.conf file that is not UTF-8 encoded."));
	}
		
	gchar** lines = g_strsplit ( contents, "\n", 1000 );
	int line_index = 0;
	while ( lines[line_index] != NULL )
	{
		StringSplitter parts(lines[line_index], ":", 100);
		line_index++;

		// Check that there are some parts
		if ( parts.get_count() == 0 )
		{
			continue;
		}			

		gsize parts_required = 0;
		switch(frontend_type)
		{
		case FE_OFDM: // Terrestrial
			parts_required = 13; 
			break;
		case FE_QAM: // Cable
			parts_required = 9; 
			break;
		case FE_QPSK: // Satellite			
			parts_required = 8; 
			break;
		case FE_ATSC: // US, Canada (Terrestrial based)
			parts_required = 6; 
			break;
		}
		
		if (parts_required != parts.get_count())
		{
			throw Exception(_("There's an invalid channel record in the channels.conf file"));
		}
		
		int frequency = parts.get_int_value(1);
		gboolean hi_band = false;

		// Need to do some massaging to get the correct frequency if DVB-S
		if (frontend_type == FE_QPSK)
		{
			frequency *= 1000;
			hi_band = HIGH_BAND(frequency);
			frequency -=  hi_band ? LOF_HI : LOF_LO;
		}
		
		Transponder* transponder = (Transponder*)g_hash_table_lookup(transponders, &frequency);
		if (transponder == NULL)
		{
			transponder = new Transponder();
			
			switch(frontend_type)
			{
			case FE_OFDM: // Terrestrial
				transponder->frontend_parameters.frequency = frequency;
				transponder->frontend_parameters.inversion = (fe_spectral_inversion_t)get_parameter_value ( parts.get_value(2), inversion_list);
				transponder->frontend_parameters.u.ofdm.bandwidth = (fe_bandwidth_t)get_parameter_value ( parts.get_value(3), bw_list);
				transponder->frontend_parameters.u.ofdm.code_rate_HP = (fe_code_rate_t)get_parameter_value ( parts.get_value(4), fec_list);
				transponder->frontend_parameters.u.ofdm.code_rate_LP = (fe_code_rate_t)get_parameter_value ( parts.get_value(5), fec_list);
				transponder->frontend_parameters.u.ofdm.constellation = (fe_modulation_t)get_parameter_value ( parts.get_value(6), modulation_list);
				transponder->frontend_parameters.u.ofdm.transmission_mode = (fe_transmit_mode_t)get_parameter_value ( parts.get_value(7), transmissionmode_list);
				transponder->frontend_parameters.u.ofdm.guard_interval = (fe_guard_interval_t)get_parameter_value ( parts.get_value(8), guard_list);
				transponder->frontend_parameters.u.ofdm.hierarchy_information = (fe_hierarchy_t)get_parameter_value ( parts.get_value(9), hierarchy_list);
				break;
			case FE_QAM: // Cable
				transponder->frontend_parameters.frequency = frequency;
				transponder->frontend_parameters.inversion = (fe_spectral_inversion_t)get_parameter_value ( parts.get_value(2), inversion_list);
				transponder->frontend_parameters.u.qam.symbol_rate = parts.get_int_value(3);
				transponder->frontend_parameters.u.qam.fec_inner = (fe_code_rate_t)get_parameter_value ( parts.get_value(4), fec_list);
				transponder->frontend_parameters.u.qam.modulation = (fe_modulation_t)get_parameter_value ( parts.get_value(5), modulation_list);
				break;
			case FE_QPSK: // Satellite			
				transponder->frontend_parameters.frequency = frequency;
				transponder->hi_band = hi_band;
				transponder->polarisation = (parts.get_value(2)[0] == 'h') ? 0 : 1;
				transponder->satellite_number = parts.get_int_value(3);
				transponder->frontend_parameters.u.qpsk.symbol_rate = parts.get_int_value(4) * 1000;
				transponder->frontend_parameters.u.qpsk.fec_inner = FEC_AUTO;
				transponder->frontend_parameters.inversion = INVERSION_AUTO;
				break;
			case FE_ATSC: // US, Canada (Terrestrial based)
				transponder->frontend_parameters.frequency = frequency;
				transponder->frontend_parameters.inversion = INVERSION_AUTO;
				transponder->frontend_parameters.u.vsb.modulation = (fe_modulation_t)get_parameter_value ( parts.get_value(2), modulation_list);
				break;
			}

			g_hash_table_insert(
				transponders,
				(gpointer)&transponder->frontend_parameters.frequency,
				(gpointer)transponder);
		}
		
		Channel* channel = new Channel(*transponder);

		// Get and fix duplicate channel names by appending a #1, #2 ...
		String channel_name = parts.get_value(0);
		channel->name = channel_name;
		
		guint index = 0;
		gpointer found = g_hash_table_lookup(channels, channel_name.c_str());
		while (found != NULL)
		{
			index++;
			channel->name = String::format("%s #%d", channel_name.c_str(), index);
			found = g_hash_table_lookup(channels, channel->name.c_str());
		}

		switch(frontend_type)
		{
		case FE_OFDM: // Terrestrial
			channel->default_video_pid = parts.get_int_value(10);
			channel->default_audio_pid = parts.get_int_value(11);
			channel->service_id = parts.get_int_value(12);
			break;
		case FE_QAM: // Cable
			channel->default_video_pid = parts.get_int_value(6);
			channel->default_audio_pid = parts.get_int_value(7);
			channel->service_id = parts.get_int_value(8);
			break;
		case FE_QPSK: // Satellite		
			channel->default_video_pid = parts.get_int_value(5);
			channel->default_audio_pid = parts.get_int_value(6);
			channel->service_id = parts.get_int_value(7);
			break;
		case FE_ATSC: // US, Canada (Terrestrial based)
			channel->default_video_pid = parts.get_int_value(3);
			channel->default_audio_pid = parts.get_int_value(4);
			channel->service_id = parts.get_int_value(5);
			break;
		}

		transponder->add_channel(channel);		
		g_hash_table_insert(channels, (gpointer)channel->name.c_str(), (gpointer)channel);
		channels_by_order = g_slist_append(channels_by_order, channel);
	}
	g_strfreev ( lines );
	
	gint channel_sort = Application::get_current().get_configuration().get_int_value("channel_sort");
	switch (channel_sort)
	{
	case CHANNEL_SORT_TRANSPONDER_SERVICE_ID:
		channels_by_order = g_slist_sort(channels_by_order, channel_sort_transponder_service_id_func);
		break;
	case CHANNEL_SORT_ALPHABETICALLY:
		channels_by_order = g_slist_sort(channels_by_order, channel_sort_alphabetically_func);
		break;
	}	
}

int ChannelManager::get_parameter_value ( const String& field, const DvbParameter* list )
{
	int i = 0;
	while (list[i].field != NULL)
	{
		if (field == list[i].field )
		{
			return list[i].value;
		}
		i++;
	}

	String message = String::format(_("Failed to parse channel file: Could not match '%s' to a value"), field.c_str());
	throw Exception(message);
}

GSList* ChannelManager::get_channels() const
{
	return channels_by_order;
}

Transponder* ChannelManager::get_transponder (int frequency) const
{
	return ( Transponder* ) g_hash_table_lookup ( transponders, &frequency );
}

Channel* ChannelManager::get_channel (int frequency, int service_id) const
{
	Channel* result = NULL;
	Transponder* t = get_transponder(frequency);
	if (t != NULL)
	{
		result = t->get_channel(service_id);
	}
	return result;
}

Channel* ChannelManager::get_channel (const String& channel_name) const
{
	return (Channel*)g_hash_table_lookup(channels, channel_name.c_str());
}
