/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "io.hh"
#include "application.hh"
#include "exception.hh"

gboolean IO::FIFO::exists(const String& path)
{
	gboolean result = false;
	
	struct stat buffer;
	if (access(path.c_str(), F_OK) == 0)
	{
		if (stat(path.c_str(), &buffer) == 0)
		{
			if (!S_ISFIFO(buffer.st_mode))
			{
				throw Exception(_("'%s' exists but is not a FIFO file"), path.c_str());
			}
			result = true;
		}
	}
	
	return result;
}

void IO::FIFO::create(const String& path)
{
	if (mkfifo(path.c_str(), S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP) != 0)
	{
		throw SystemException(_("Failed to make fifo file"));
	}
}

IO::Channel::Channel(const String& path, int oflag, mode_t mode)
{
	int fd = -1;
	if ( (fd = open(path.c_str(), oflag, mode)) < 0)
	{
		throw SystemException(_("Failed to open file '%s'"), path.c_str());
	}
	create(fd);
}

IO::Channel::Channel(const String& filename, const gchar* mode)
{
	GError* error = NULL;
	channel = g_io_channel_new_file(filename.c_str(), mode, &error);
	if (error != NULL)
	{
		throw Exception("Failed to open file '%s': %s", filename.c_str(), error->message);
	}
	pfd[0].fd = get_fd();
	pfd[0].events = POLLIN;
}

IO::Channel::Channel(int fd)
{
	create(fd);
}

IO::Channel::~Channel()
{
	g_io_channel_shutdown(channel, true, NULL);
}

void IO::Channel::create(int fd)
{
	channel = g_io_channel_unix_new (fd);
	if (channel == NULL)
	{
		throw SystemException(_("Failed to create IO channel"));
	}
	g_io_channel_set_close_on_unref (channel, true);
	pfd[0].fd = get_fd();
	pfd[0].events = POLLIN;
}

int IO::Channel::get_fd() const
{
	return g_io_channel_unix_get_fd(channel);
}

void IO::Channel::set_encoding(const gchar* encoding)
{
	GError* error = NULL;
	g_io_channel_set_encoding(channel, encoding, &error);
	if (error != NULL)
	{
		throw Exception("Failed to set encoding: %s", error->message);
	}
}

GIOStatus IO::Channel::get_status()
{
	return status;
}

gboolean IO::Channel::is_eof()
{
	return status == G_IO_STATUS_EOF;
}

String IO::Channel::read_line()
{
	String	line;
	GError*			error		= NULL;
	gchar*			str_return	= NULL;
	gsize			length		= 0;
	
	status = g_io_channel_read_line(channel, &str_return, &length, NULL, &error);
	if (error != NULL)
	{
		throw Exception("Failed to read line: %s", error->message);
	}
	
	if (status == G_IO_STATUS_ERROR)
	{
		throw Exception("Failed to read line");
	}
	
	if (str_return != NULL)
	{
		line = str_return;
		g_free(str_return);
	}

	return line;
}

gboolean IO::Channel::poll(guint timeout)
{
	return ::poll(pfd, 1, timeout);
}

gsize IO::Channel::read(gchar* buffer, gsize count)
{
	GError* error = NULL;
	gsize bytes_read = 0;

	status = g_io_channel_read_chars(channel, buffer, count, &bytes_read, &error);
	
	if (error != NULL)
	{
		throw Exception("Failed to read from channel: '%s'", error->message);
	}

	if (status == G_IO_STATUS_ERROR)
	{
		throw Exception("Failed to read from channel");
	}

	return bytes_read;
}

void IO::Channel::write(const gchar* buffer, gsize length)
{
	GError* error = NULL;
	gsize bytes_written = 0;
	
	g_io_channel_write_chars(channel, buffer, length, &bytes_written, &error);
	
	if (error != NULL)
	{
		throw Exception("Failed to write to channel: '%s'", error->message);
	}
	
	if (bytes_written != length)
	{
		throw Exception("Failed to write to channel");
	}
}

void IO::Channel::write(const String& text)
{
	write(text.c_str(), text.get_buffer_length());
}

void IO::Channel::write_line(const String& text)
{
	write(text + "\n");
}

void IO::Channel::write_formatted(const gchar* format, ...)
{
	va_list ap;
	
	va_start(ap, format);
	gchar* text = g_strdup_vprintf(format, ap);
	write(text, strlen(text));
	g_free(text);
	va_end(ap);
}

void IO::Channel::flush()
{
	GError* error = NULL;
	g_io_channel_flush(channel, &error);
	if (error != NULL)
	{
		throw Exception("Failed to flush: '%s'", error->message);
	}
}

String IO::Channel::read_to_end()
{
	String	line;
	GError*			error		= NULL;
	gchar*			str_return	= NULL;
	gsize			length		= 0;
	
	status = g_io_channel_read_to_end(channel, &str_return, &length, &error);
	if (error != NULL)
	{
		throw Exception("Failed to read line: %s", error->message);
	}
	
	if (status == G_IO_STATUS_ERROR)
	{
		throw Exception("Failed to read line");
	}
	
	if (str_return != NULL)
	{
		line = str_return;
		g_free(str_return);
	}
	
	return line;
}

gboolean IO::File::exists(const String& path)
{
	gboolean result = false;			
	struct stat buffer;
		
	if (access(path.c_str(), F_OK) == 0)
	{
		if (stat(path.c_str(), &buffer) == 0)
		{
			if (S_ISDIR(buffer.st_mode))
			{
				throw Exception(path + " exists but is a directory");
			}
			
			result = true;
		}
	}
	
	return result;
}

gboolean IO::Directory::exists(const String& path)
{
	gboolean result = false;			
	struct stat buffer;
		
	if (access(path.c_str(), F_OK) == 0)
	{
		if (stat(path.c_str(), &buffer) == 0)
		{
			if (!S_ISDIR(buffer.st_mode))
			{
				throw Exception(path + " exists but is not a directory");
			}
			
			result = true;
		}
	}
	
	return result;
}

void IO::Directory::create(const String& path)
{
	if (mkdir(path.c_str(), S_IRWXU | S_IRWXG) < 0)
	{
		throw SystemException("Failed to create directory");
	}		
}
