/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "streamer.hh"
#include "log.hh"
#include "application.hh"
#include "exception_handler.hh"
#include "mutex_lock.hh"
#include "dvb_dvr.hh"
#include "dvb_cam.hh"

#define BUFFER_SIZE	TS_PACKET_SIZE*8 // Don't exceed 64K for UDP

Streamer* Streamer::current_streamer = NULL;

Streamer::Streamer()
{
	g_static_rec_mutex_init(&mutex);
	recording_file_stream	= NULL;
	network_stream			= NULL;
	stream_thread			= NULL;
	demuxers				= NULL;
	current_streamer		= this;
	pmt_pid					= 0;
	encode_recording		= false;
	engine					= NULL;

	for(gint i = 0 ; i < 256 ; i++ )
	{
		gint k = 0;
		for (gint j = (i << 24) | 0x800000 ; j != 0x80000000 ; j <<= 1)
		{
			k = (k << 1) ^ (((k ^ j) & 0x80000000) ? 0x04c11db7 : 0);
		}
		CRC32[i] = k;
	}
}

Streamer::~Streamer()
{
	stop();
}

void Streamer::start()
{
	MutexLock mutex_lock(&mutex);
	
	Log::write("Streamer starting");

	Application& application = Application::get_current();
	engine = application.get_engine();
	if (engine != NULL)
	{
		engine->open();
	}

	stream_thread = g_thread_create((GThreadFunc)stream_thread_function, this, TRUE, NULL);
	if (stream_thread == NULL)
	{
		throw SystemException(_("Failed to create stream thread"));
	}
	Log::write(_("Stream thread created"));
}

void Streamer::stop()
{
	MutexLock mutex_lock(&mutex);

	terminate = true;
	if (stream_thread != NULL)
	{
		g_thread_join(stream_thread);
		stream_thread = NULL;
	}

	remove_all_demuxers ();

	if (engine != NULL)
	{
		engine->close();
	}
}

gboolean Streamer::is_broadcasting()
{
	MutexLock mutex_lock(&mutex);
	return network_stream != NULL;
}

void Streamer::enable_broadcasting()
{
	MutexLock mutex_lock(&mutex);

	if (network_stream == NULL)
	{
		Application& application = Application::get_current();
		String broadcast_address = application.get_configuration().get_string_value("broadcast_address");
		guint broadcast_port = application.get_configuration().get_int_value("broadcast_port");
		network_stream = new UdpSocket(broadcast_address.c_str(), broadcast_port);
	}
}

void Streamer::disable_broadcasting()
{
	MutexLock mutex_lock(&mutex);

	if (network_stream != NULL)
	{
		UdpSocket* temp = network_stream;
		network_stream = NULL;
		delete temp;
	}
}

gboolean Streamer::is_recording()
{
	MutexLock mutex_lock(&mutex);
	return recording_file_stream != NULL;
}

void Streamer::start_recording(const String& filename)
{
	MutexLock mutex_lock(&mutex);
	
	if (recording_file_stream != NULL)
	{
		throw Exception(_("Recording file stream has already been created"));
	}
	
	recording_file_stream = new IO::Channel(filename, O_CREAT | O_WRONLY);
	recording_file_stream->set_encoding(NULL);
	Log::write(_("Created recording file stream"));
}

void Streamer::stop_recording()
{
	MutexLock mutex_lock(&mutex);
	if (recording_file_stream != NULL)
	{
		IO::Channel* temp = recording_file_stream;
		recording_file_stream = NULL;
		delete temp;
	}
}

void Streamer::write(const gchar* buffer, gsize length)
{
	if (engine != NULL)
	{
		engine->write(buffer, length);
	}

	if (recording_file_stream != NULL)
	{
		recording_file_stream->write(buffer, length);
	}

	if (network_stream != NULL)
	{
		network_stream->write(buffer, length);
	}
}

void Streamer::on_write(uint8_t *buffer, int size, void *priv)
{
	current_streamer->recording_file_stream->write((gchar*)buffer, size);
}

void Streamer::stream_loop(IO::Channel& input)
{	
	Application& application = Application::get_current();

	gboolean is_video_file = application.get_video_file().get_character_length() > 0;
	
	gsize length = 0;
	gchar buffer[BUFFER_SIZE];
	encode_recording = false;
	
	gchar pat[TS_PACKET_SIZE];
	gchar pmt[TS_PACKET_SIZE];
	
	if (!is_video_file)
	{
		build_pat(pat);
		build_pmt(pmt);
	}
	
	time_t last_check_time = 0;
	
	terminate = false;
	
	while (!terminate)
	{
		try
		{
			if (!is_video_file)
			{
				// Insert PAT/PMT every second
				time_t now = time(NULL);
				if (now - last_check_time > 1)
				{
					write(pat, TS_PACKET_SIZE);
					write(pmt, TS_PACKET_SIZE);
					last_check_time = now;
				}
			}
			
			if (input.poll(1000))
			{
				length = input.read(buffer, BUFFER_SIZE);
				write(buffer, length);
			}
			else
			{
				Log::write("Read timed out after 1 second");
			}
		}
		catch(const Exception& ex)
		{
			Log::write("Exception in %s: %s", __PRETTY_FUNCTION__, ex.get_message().c_str());
		}
	}
}

gpointer Streamer::stream_thread_function(Streamer* streamer)		
{
	TRY;
	streamer->start_stream();
	THREAD_CATCH;
	Log::write(_("Stream thread stopped"));
	
	return NULL;
}

void Streamer::remove_all_demuxers()
{
	while (demuxers != NULL)
	{
		delete (DvbDemuxer*)demuxers->data;
		demuxers = g_slist_delete_link (demuxers, demuxers);
	}
}

DvbDemuxer& Streamer::add_pes_demuxer(const String& demux_path, guint pid, dmx_pes_type_t pid_type, const gchar* type_text)
{	
	DvbDemuxer* demuxer = new DvbDemuxer(demux_path);
	demuxers = g_slist_prepend (demuxers, demuxer);
	Log::write(_("Setting %s PID filter to %d (0x%X)"), type_text, pid, pid);
	demuxer->set_pes_filter(pid, pid_type);
	return *demuxer;
}

void Streamer::start_stream()
{	
	MutexLock mutex_lock(&mutex);

	remove_all_demuxers ();

	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();
	
	String demux_path = configuration.get_string_value("demux_path");
	String dvr_path = configuration.get_string_value("dvr_path");
	
	Channel& channel = application.get_video_channel();

	String video_file = application.get_video_file();
	if (video_file.get_character_length() > 0)
	{
		application.set_channel_change_complete();
		IO::Channel input(video_file, O_RDONLY);
		input.set_encoding(NULL);
		mutex_lock.unlock();
		stream_loop(input);
	}
	else if (application.get_tuner() != NULL)
	{		
		stream.clear();

		if (configuration.get_boolean_value("pid_workaround"))
		{
			VideoStream video_stream;
			video_stream.pid = channel.default_video_pid;
			stream.video_streams.push_back(video_stream);
			add_pes_demuxer(demux_path, channel.default_video_pid, DMX_PES_VIDEO, "video");
			
			AudioStream audio_stream;
			audio_stream.pid = channel.default_audio_pid;
			stream.audio_streams.push_back(audio_stream);
			add_pes_demuxer(demux_path, channel.default_audio_pid, DMX_PES_AUDIO, "audio");
		}
		else
		{
			try
			{
				Log::write(_("Trying to auto detect PIDs"));
				
				DvbDemuxer demuxer_pat(demux_path);
				DvbDemuxer demuxer_pmt(demux_path);
				DvbSectionParser parser;
							
				ProgramAssociationSection pas;
				demuxer_pat.set_filter(PAT_PID, PAT_ID);
				parser.parse_pas(demuxer_pat, pas);
				guint length = pas.program_associations.size();
				pmt_pid = 0;
				
				Log::write(_("Found %d associations"), length);
				for (guint i = 0; i < length; i++)
				{
					ProgramAssociation program_association = pas.program_associations[i];
					
					if (program_association.program_number == channel.service_id)
					{
						pmt_pid = program_association.program_map_pid;
					}
				}
				
				if (pmt_pid == 0)
				{
					throw Exception("Failed to find PMT ID for service");
				}
				demuxer_pat.stop();
				
				demuxer_pmt.set_filter(pmt_pid, PMT_ID);
				ProgramMapSection pms;
				parser.parse_pms(demuxer_pmt, pms);
				demuxer_pmt.stop();

				// Only take the first video stream
				if (pms.video_streams.size() > 0)
				{
					VideoStream video_stream = pms.video_streams[0];
					stream.video_streams.push_back(video_stream);
					add_pes_demuxer(demux_path, video_stream.pid, DMX_PES_OTHER, "video");
				}

				gsize audio_streams_size = pms.audio_streams.size();
				for (guint i = 0; i < audio_streams_size; i++)
				{
					AudioStream audio_stream = pms.audio_streams[i];
					stream.audio_streams.push_back(audio_stream);
					add_pes_demuxer(demux_path, audio_stream.pid,
									DMX_PES_OTHER,
									audio_stream.is_ac3 ? "AC3" : "audio");
				}
							
				gsize subtitle_streams_size = pms.subtitle_streams.size();
				for (guint i = 0; i < subtitle_streams_size; i++)
				{
					SubtitleStream subtitle_stream = pms.subtitle_streams[i];
					stream.subtitle_streams.push_back(subtitle_stream);
					add_pes_demuxer(demux_path, subtitle_stream.pid, DMX_PES_OTHER, "subtitle");
				}
			
				if (pms.teletext_streams.size() > 0)
				{
					TeletextStream first_teletext_stream = pms.teletext_streams[0];
					stream.teletext_streams.push_back(first_teletext_stream);
					add_pes_demuxer(demux_path, first_teletext_stream.pid, DMX_PES_OTHER, "teletext");
				}
				
				Log::write(_("PIDs read successfully"));
			}
			catch(const Exception& ex)
			{
				Log::write(_("Failed to set PIDs automatically: %s"), ex.get_message().c_str());
			}
		}
				
		application.set_channel_change_complete();

		String ca_path = configuration.get_string_value("ca_path");
		if (IO::File::exists(ca_path))
		{
			DvbDvr dvr(dvr_path);
			dvr.set_encoding(NULL);

			gboolean cam_failed = true;
			try
			{
				DvbCam cam(ca_path);
				cam.start(channel.service_id);
				cam_failed = false;
				mutex_lock.unlock();
				stream_loop(dvr);
			}
			catch(Exception& exception)
			{
				if (cam_failed)
				{
					Log::write("Failed to initialise CAM: '%s'", exception.get_message().c_str());
					mutex_lock.unlock();
					stream_loop(dvr);
				}
				else
				{
					throw exception;
				}
			}
		}
		else
		{		
			DvbDvr dvr(dvr_path);
			dvr.set_encoding(NULL);
			mutex_lock.unlock();
			stream_loop(dvr);
		}
	}
	else
	{
		throw Exception("No video file or tuner");
	}
	
	remove_all_demuxers ();
}

void Streamer::calculate_crc(guchar *p_begin, guchar *p_end)
{
	unsigned int i_crc = 0xffffffff;

	// Calculate the CRC
	while( p_begin < p_end )
	{
		i_crc = (i_crc<<8) ^ CRC32[ (i_crc>>24) ^ ((unsigned int)*p_begin) ];
		p_begin++;
	}

	// Store it after the data
	p_end[0] = (i_crc >> 24) & 0xff;
	p_end[1] = (i_crc >> 16) & 0xff;
	p_end[2] = (i_crc >>  8) & 0xff;
	p_end[3] = (i_crc >>  0) & 0xff;
}

gboolean Streamer::is_pid_used(guint pid)
{
	gboolean used = false;
	guint index = 0;
	
	if (stream.video_streams.size() > 0)
	{
		used = (pid==stream.video_streams[0].pid);
	}
	
	while (index < stream.audio_streams.size() && !used)
	{
		if (pid==stream.audio_streams[index].pid)
		{
			used = true;
		}
		else
		{
			index++;
		}
	}

	index = 0;
	while (index < stream.subtitle_streams.size() && !used)
	{
		if (pid==stream.subtitle_streams[index].pid)
		{
			used = true;
		}
		else
		{
			index++;
		}
	}
	
	return used;
}

void Streamer::build_pat(gchar* buffer)
{
	buffer[0x00] = 0x47; // sync_byte
	buffer[0x01] = 0x40;
	buffer[0x02] = 0x00; // PID = 0x0000
	buffer[0x03] = 0x10; // | (ps->pat_counter & 0x0f);
	buffer[0x04] = 0x00; // CRC calculation begins here
	buffer[0x05] = 0x00; // 0x00: Program association section
	buffer[0x06] = 0xb0;
	buffer[0x07] = 0x11; // section_length = 0x011
	buffer[0x08] = 0x00;
	buffer[0x09] = 0xbb; // TS id = 0x00b0 (what the vlc calls "Stream ID")
	buffer[0x0a] = 0xc1;
	// section # and last section #
	buffer[0x0b] = buffer[0x0c] = 0x00;
	// Network PID (useless)
	buffer[0x0d] = buffer[0x0e] = 0x00;
	buffer[0x0f] = 0xe0;
	buffer[0x10] = 0x10;
	
	// Program Map PID
	pmt_pid = 0x64;
	while (is_pid_used(pmt_pid))
	{
		pmt_pid--;
	}
	
	buffer[0x11] = 0x03;
	buffer[0x12] = 0xe8;
	buffer[0x13] = 0xe0;
	buffer[0x14] = pmt_pid;
	
	// Put CRC in buffer[0x15...0x18]
	calculate_crc( (guchar*)buffer + 0x05, (guchar*)buffer + 0x15 );
	
	// needed stuffing bytes
	for (gint i=0x19; i < 188; i++)
	{
		buffer[i]=0xff;
	}
}

void Streamer::build_pmt(gchar* buffer)
{
	int i, off=0;
	
	VideoStream video_stream;
	
	if (stream.video_streams.size() > 0)
	{
		video_stream = stream.video_streams[0];
	}

	buffer[0x00] = 0x47; //sync_byte
	buffer[0x01] = 0x40;
	buffer[0x02] = pmt_pid;
	buffer[0x03] = 0x10;
	buffer[0x04] = 0x00; // CRC calculation begins here
	buffer[0x05] = 0x02; // 0x02: Program map section
	buffer[0x06] = 0xb0;
	buffer[0x07] = 0x20; // section_length
	buffer[0x08] = 0x03;
	buffer[0x09] = 0xe8; // prog number
	buffer[0x0a] = 0xc1;
	// section # and last section #
	buffer[0x0b] = buffer[0x0c] = 0x00;
	// PCR PID
	buffer[0x0d] = video_stream.pid>>8;
	buffer[0x0e] = video_stream.pid&0xff;
	// program_info_length == 0
	buffer[0x0f] = 0xf0;
	buffer[0x10] = 0x00;
	// Program Map / Video PID
	buffer[0x11] = video_stream.type; // video stream type
	buffer[0x12] = video_stream.pid>>8;
	buffer[0x13] = video_stream.pid&0xff;
	buffer[0x14] = 0xf0;
	buffer[0x15] = 0x09; // es info length
	// useless info
	buffer[0x16] = 0x07;
	buffer[0x17] = 0x04;
	buffer[0x18] = 0x08;
	buffer[0x19] = 0x80;
	buffer[0x1a] = 0x24;
	buffer[0x1b] = 0x02;
	buffer[0x1c] = 0x11;
	buffer[0x1d] = 0x01;
	buffer[0x1e] = 0xfe;
	off = 0x1e;
	
	// Audio streams
	for (gint index = 0; index < stream.audio_streams.size(); index++)
	{
		AudioStream audio_stream = stream.audio_streams[index];
		
		gchar language_code[4] = { 0 };
		
		strncpy(language_code, audio_stream.language.c_str(), 3);

		if ( audio_stream.is_ac3 )
		{
			buffer[++off] = 0x81; // stream type = xine see this as ac3
			buffer[++off] = audio_stream.pid>>8;
			buffer[++off] = audio_stream.pid&0xff;
			buffer[++off] = 0xf0;
			buffer[++off] = 0x0c; // es info length
			buffer[++off] = 0x05;
			buffer[++off] = 0x04;
			buffer[++off] = 0x41;
			buffer[++off] = 0x43;
			buffer[++off] = 0x2d;
			buffer[++off] = 0x33;
		}
		else
		{
			buffer[++off] = 0x04; // stream type = audio
			buffer[++off] = audio_stream.pid>>8;
			buffer[++off] = audio_stream.pid&0xff;
			buffer[++off] = 0xf0;
			buffer[++off] = 0x06; // es info length
		}
		buffer[++off] = 0x0a; // iso639 descriptor tag
		buffer[++off] = 0x04; // descriptor length
		buffer[++off] = language_code[0];
		buffer[++off] = language_code[1];
		buffer[++off] = language_code[2];
		buffer[++off] = 0x00; // audio type
	}
	
	// Subtitle streams
	for (gint index = 0; index < stream.subtitle_streams.size(); index++)
	{
		SubtitleStream subtitle_stream = stream.subtitle_streams[index];
		
		gchar language_code[4] = { 0 };
		
		strncpy(language_code, subtitle_stream.language.c_str(), 3);
		
		buffer[++off] = 0x06; // stream type = ISO_13818_PES_PRIVATE
		buffer[++off] = subtitle_stream.pid>>8;
		buffer[++off] = subtitle_stream.pid&0xff;
		buffer[++off] = 0xf0;
		buffer[++off] = 0x0a; // es info length
		buffer[++off] = 0x59; // DVB sub tag
		buffer[++off] = 0x08; // descriptor length
		buffer[++off] = language_code[0];
		buffer[++off] = language_code[1];
		buffer[++off] = language_code[2];
		buffer[++off] = subtitle_stream.subtitling_type;
		buffer[++off] = subtitle_stream.composition_page_id>>8;
		buffer[++off] = subtitle_stream.composition_page_id&0xff;
		buffer[++off] = subtitle_stream.ancillary_page_id>>8;
		buffer[++off] = subtitle_stream.ancillary_page_id&0xff;
	}
	
	// TeleText streams
	for (gint index = 0; index < stream.teletext_streams.size(); index++)
	{
		TeletextStream teletext_stream = stream.teletext_streams[index];
						
		gint language_count = teletext_stream.languages.size();

		buffer[++off] = 0x06; // stream type = ISO_13818_PES_PRIVATE
		buffer[++off] = teletext_stream.pid>>8;
		buffer[++off] = teletext_stream.pid&0xff;
		buffer[++off] = 0xf0;
		buffer[++off] = (language_count * 5) + 4; // es info length
		buffer[++off] = 0x56; // DVB teletext tag
		buffer[++off] = language_count * 5; // descriptor length

		for (gint language_index = 0; language_index < language_count; language_index++)
		{
			TeletextLanguageDescriptor descriptor = teletext_stream.languages[language_index];
			gchar language_code[4] = { 0 };
			strncpy(language_code, descriptor.language.c_str(), 3);
			
			buffer[++off] = language_code[0];
			buffer[++off] = language_code[1];
			buffer[++off] = language_code[2];
			//buffer[++off] = (descriptor.type & 0x1F) | ((descriptor.magazine_number << 5) & 0xE0);
			buffer[++off] = (descriptor.magazine_number & 0x06) | ((descriptor.type << 3) & 0xF8);
			buffer[++off] = descriptor.page_number;
		}
	}

	buffer[0x07] = off-3; // update section_length

	// Put CRC in ts[0x29...0x2c]
	calculate_crc( (guchar*)buffer+0x05, (guchar*)buffer+off+1 );
	
	// needed stuffing bytes
	for (i=off+5 ; i < 188 ; i++)
	{
		buffer[i]=0xff;
	}
}

const Stream& Streamer::get_stream() const
{
	return stream;
}
