# NAME
error &mdash; Errors generated by the report module

# LIBRARY
MeasurementKit (libmeasurement_kit, -lmeasurement_kit).

# SYNOPSIS
```C++
#include <measurement_kit/report.hpp>

namespace mk {
namespace report {

MK_DEFINE_ERR(MK_ERR_REPORT(0), ReportAlreadyOpenError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(1), ReportNotOpenError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(2), ReportAlreadyClosedError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(3), ReportEofError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(4), ReportIoError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(5), ReportLogicalError, "")
MK_DEFINE_ERR(MK_ERR_REPORT(6), DuplicateEntrySubmitError, "")

} // namespace report
} // namespace mk
```

# STABILITY

2 - Stable

# DESCRIPTION

The `report` module can generate the following errors:

- *ReportAlreadyOpenError*: you called `open` on a `BaseReporter` more than once
- *ReportNotOpenError*: you called `write_entry` on a `BaseReport` that has not been openned
- *ReportAlreadyClosedError*: you called `write_entry` on a `BaseReport` that has already been closed
- *ReportEofError*: writing the report on file yields EOF
- *ReportIoError*: writing the report on file yields a I/O error
- *ReportLogicalError*: it is not possible to serialize input data into the report when writing it to file
- *DuplicateEntrySubmitError*: the previous call to the `write_entry` of a `BaseReporter` received an entry that is exacly equal to the one passed in the current call to `write_entry`

Note that the `BaseReporter` logic may hide many of these errors when it makes
sense to do so in order to make code idempotent. For example, if you call
`open` on a `BaseReporter` that is already open, the result would be `NoError`
rather than `ReportAlreadyOpenError`. This is to guarantee idempotence when
you a have a `Report` with many `BaseReporter`s registered and you want to
be able to retry `open` after if failed only on some `BaseReporters`.

To detect the case where, e.g., `open` was called twice, you should check
for the child errors of the returned no error. For example:

```C++
    report.open([=](Error err) {
        if (err) {
            size_t idx = 0;
            for (Var<Error> child_err: err.child_errors) {
                if (!child_err and child_err.child_errors.size() >= 1
                    and child_err.child_errors[0] == ReportAlreadyOpenError()) {
                    debug("- %d's open was called more than once", idx);
                    continue;
                }
            }
            // Note: given that `open` is idempotent, here you can
            // safely retry by calling `open` again
            return;
        }
        // Operations to be performed on success...
    });
```

# HISTORY

The `error` file appeared in MeasurementKit 0.2.0.
