/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

  $Id: param.cpp,v 1.11 2006/07/09 11:26:27 taku-ku Exp $;

  Copyright (C) 2001-2006 Taku Kudo <taku@chasen.org>
  Copyright (C) 2004-2006 Nippon Telegraph and Telephone Corporation

*/

#include <fstream>
#include <cstdio>
#include "param.h"
#include "common.h"
#include "utils.h"
#include "string_buffer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

namespace MeCab {

  using namespace std;

  bool Param::load(const char *filename)
  {
    std::ifstream ifs(filename);

    CHECK_FALSE(ifs) << "no such file or directory: " << filename;

    std::string line;
    while (std::getline(ifs, line)) {
      if (! line.size() || (line.size() &&(line[0] == ';' || line[0] == '#'))) continue;

      size_t pos = line.find('=');
      CHECK_FALSE(pos != std::string::npos) << "format error: " << line;

      size_t s1, s2;
      for (s1 = pos+1; s1 < line.size() && isspace(line[s1]); s1++);
      for (s2 = pos-1; static_cast<long>(s2) >= 0 && isspace(line[s2]); s2--);
      std::string value = line.substr(s1, line.size()-s1);
      std::string key   = line.substr(0, s2+1);
      setProfile(key.c_str(), value.c_str(), false);
    }

    return true;
  }

  bool Param::open(int argc, char **argv, const Option *opts)
  {
    int ind = 0;
    int _errno = 0;

#define GOTO_ERROR(n) { _errno = n; goto ERROR; } while (0)

    if (argc <= 0) {
      systemName_ = "unknown";
      return true; // this is not error
    }

    systemName_ = std::string(argv[0]);

    for (size_t i = 0; opts[i].name; ++i) {
      if (opts[i].default_value) setProfile(opts[i].name, opts[i].default_value);
    }

    for (ind = 1; ind < argc; ind++) {

      if (argv[ind][0] == '-') {

        // long options
        if (argv[ind][1] == '-') {

          char *s;
          for (s = &argv[ind][2]; *s != '\0' && *s != '='; s++);
          size_t len = (size_t)(s - &argv[ind][2]);
          if (len == 0) return true; // stop the scanning

          bool hit = false;
          size_t i = 0;
          for (i = 0; opts[i].name; ++i) {
            size_t nlen = std::strlen(opts[i].name);
            if (nlen == len && std::strncmp(&argv[ind][2], opts[i].name, len) == 0) {
              hit = true;
              break;
            }
          }

          if (!hit) GOTO_ERROR(0);

          if (opts[i].arg_description) {
            if (*s == '=') {
              if (*(s+1) == '\0') GOTO_ERROR(1);
              setProfile(opts[i].name, s+1);
            } else {
              if (argc == (ind+1)) GOTO_ERROR(1);
              setProfile(opts[i].name, argv[++ind]);
            }
          } else {
            if (*s == '=') GOTO_ERROR(2);
            setProfile(opts[i].name, 1);
          }

          // short options
        } else if (argv[ind][1] != '\0') {

          size_t i = 0;
          bool hit = false;
          for (i = 0; opts[i].name; ++i) {
            if (opts[i].short_name == argv[ind][1]) {
              hit = true;
              break;
            }
          }

          if (!hit) GOTO_ERROR(0);

          if (opts[i].arg_description) {
            if (argv[ind][2] != '\0') {
              setProfile(opts[i].name, &argv[ind][2]);
            } else {
              if (argc ==(ind+1)) GOTO_ERROR(1);
              setProfile(opts[i].name, argv[++ind]);
            }
          } else {
            if (argv[ind][2] != '\0') GOTO_ERROR(2);
            setProfile(opts[i].name, 1);
          }
        }
      } else {
        rest_.push_back(std::string(argv[ind])); // others
      }
    }

    return true;

  ERROR:
    switch (_errno) {
    case 0: WHAT << "unrecognized option `" << argv[ind] << "`"; break;
    case 1: WHAT << "`" << argv[ind] << "` requres an argument";  break;
    case 2: WHAT << "`" << argv[ind] << "` dosen't allow an argument"; break;
    }
    return false;
  }

  void Param::clear()
  {
    conf_.clear();
    rest_.clear();
  }

  bool Param::open(const char *arg, const Option *opts)
  {
    char str[BUF_SIZE];
    std::strncpy(str, arg, sizeof(str));
    char* ptr[64];
    unsigned int size = 1;
    ptr[0] = const_cast<char*>(PACKAGE);

    for (char *p = str; *p ; ) {
      while (isspace(*p)) *p++ = '\0';
      if (*p == '\0') break;
      ptr[size++] = p;
      if (size == sizeof(ptr)) break;
      while (*p && ! isspace(*p)) p++;
    }

    return open(size, ptr, opts);
  }

  const char* Param::help(const Option *opts)
  {
    help_ = std::string(COPYRIGHT) + "\nUsage: " + systemName_ + " [options] files\n";

    size_t max = 0;
    for (size_t i = 0; opts[i].name; ++i) {
      size_t l = 1 + std::strlen(opts[i].name);
      if (opts[i].arg_description) l += (1 + std::strlen(opts[i].arg_description));
      max = _max(l, max);
    }

    for (size_t i = 0; opts[i].name; ++i) {
      size_t l = std::strlen(opts[i].name);
      if (opts[i].arg_description) l += (1 + std::strlen(opts[i].arg_description));
      help_ += " -";
      help_ += opts[i].short_name;
      help_ += ", --";
      help_ += opts[i].name;
      if (opts[i].arg_description) { help_ += '='; help_ += opts[i].arg_description; }
      for (; l <= max; l++) help_ += ' ';
      help_ += opts[i].description;
      help_ += '\n';
    }

    help_ += '\n';

    return help_.c_str();
  }

  const char* Param::version(const Option *)
  {
    version_ = std::string(PACKAGE) + " of " + VERSION + '\n';;
    return version_.c_str();
  }

  int Param::help_version(const Option* long_options) {

    if (getProfileInt("help")) {
      std::cout << help(long_options);
      return 0;
    }

    if (getProfileInt("version")) {
      std::cout << version(long_options);
      return 0;
    }

    return 1;
  }

  const std::string Param::getProfileString(const char* key)
  {
    std::map<std::string,std::string>::iterator it = conf_.find(key);
    if (it == conf_.end()) return std::string("");
    return it->second;
  }

  int Param::getProfileInt(const char* key)
  {
    std::map<std::string,std::string>::iterator it = conf_.find(key);
    if (it == conf_.end()) return 0;
    return std::atoi(it->second.c_str());
  }

  double Param::getProfileFloat(const char* key)
  {
    std::map<std::string,std::string>::iterator it = conf_.find(key);
    if (it == conf_.end()) return 0.0;
    return std::atof(it->second.c_str());
  }

  void Param::setProfile(const char* key, const char* value, bool rewrite)
  {
    std::string key2 = std::string(key);
    if (rewrite || (! rewrite && conf_.find(key2) == conf_.end()))
      conf_[key2] = value;
  }

  void Param::setProfile(const char* key, int i, bool rewrite)
  {
    std::string key2 = std::string(key);
    if (rewrite || (! rewrite && conf_.find(key2) == conf_.end())) {
      char tmp[64];
      itoa(i, tmp);
      conf_[key2] = tmp;
    }
  }
}
