<?php

namespace OOUI;

/**
 * Element containing an indicator.
 *
 * Indicators are graphics, smaller than normal text. They can be used to describe unique status or
 * behavior. Indicators should only be used in exceptional cases; such as a button that opens a menu
 * instead of performing an action directly, or an item in a list which has errors that need to be
 * resolved.
 *
 * @abstract
 */
trait IndicatorElement {
	/**
	 * Symbolic indicator name
	 *
	 * @var string|null
	 */
	protected $indicatorName = null;

	/**
	 * @var Tag
	 */
	protected $indicator;

	/**
	 * @param array $config Configuration options
	 * @param string $config['indicator'] Symbolic indicator name
	 */
	public function initializeIndicatorElement( array $config = [] ) {
		// FIXME 'indicatorElement' is a very stupid way to call '$indicator'
		$this->indicator = isset( $config['indicatorElement'] )
			? $config['indicatorElement']
			: new Tag( 'span' );

		// Initialization
		$this->indicator->addClasses( [ 'oo-ui-indicatorElement-indicator' ] );
		$this->setIndicator( isset( $config['indicator'] ) ? $config['indicator'] : null );

		$this->registerConfigCallback( function( &$config ) {
			if ( $this->indicatorName !== null ) {
				$config['indicator'] = $this->indicatorName;
			}
		} );
	}

	/**
	 * Set indicator name.
	 *
	 * @param string|null $indicator Symbolic name of indicator to use or null for no indicator
	 * @return $this
	 */
	public function setIndicator( $indicator = null ) {
		if ( $this->indicatorName !== null ) {
			$this->indicator->removeClasses( [ 'oo-ui-indicator-' . $this->indicatorName ] );
		}
		if ( $indicator !== null ) {
			$this->indicator->addClasses( [ 'oo-ui-indicator-' . $indicator ] );
		}

		$this->indicatorName = $indicator;
		$this->toggleClasses( [ 'oo-ui-indicatorElement' ], (bool)$this->indicatorName );

		return $this;
	}

	/**
	 * Get indicator name.
	 *
	 * @return string Symbolic name of indicator
	 */
	public function getIndicator() {
		return $this->indicatorName;
	}

	/**
	 * Do not use outside of Theme::updateElementClasses
	 *
	 * @protected
	 * @return Tag
	 */
	public function getIndicatorElement() {
		return $this->indicator;
	}
}
