/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: common.c,v 1.12 2000/11/14 03:05:05 kos Exp $
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#include <openssl/md5.h>
#include <openssl/rand.h>

#include "xmalloc.h"
#include "array.h"
#include "llog.h"
#include "common.h"

static char hexdigit[] = "0123456789ABCDEF";
static char b64_charmap[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
#define b64_map(x) ((x)<=64?b64_charmap[x]:'.')

int 
textify(uchar *in, int ilen, uchar *out, int *olen) {
  int i;

  *olen = 0;
  for(i=0; i<ilen; i++) {
    if(in[i] == '%') {
      *out++ = '%';
      *out++ = '%';
      (*olen) += 2;
    } else if(in[i]>32 && in[i]<128) {
      *out++ = in[i];
      (*olen)++;
    } else {
      *out++ = '%';
      *out++ = hexdigit[((in[i]&0xF0) >> 4)];
      *out++ = hexdigit[((in[i]&0x0F)     )];
      (*olen) += 3;
    }
  }
  return 0;
}

int
textify_len(uchar *in, int ilen) {
  int i;
  int olen;

  for(olen=0,i=0; i<ilen; i++) {
    if(in[i]>32 && in[i]<128)
      olen++;
    else if(in[i] == '%') 
      olen+=2;
    else
      olen+=3;
  }
  return olen;
}

int
untextify(uchar *in, int ilen, uchar *out, int *olen) {
  int i;
  
  *olen = 0;
  for(i=0; i<ilen; i++) {
    if(in[i] == '%') {
      if(in[i+1] == '%') {
	i++;
	*out++ = '%';
	(*olen)++;
      } else {
	*out++ = (tmap(in[i+1])<<4 | tmap(in[i+2]));
	(*olen)++;
	i+=2;
      }
    } else {
      *out++ = in[i];
      (*olen)++;
    }
  }
  return 0;
}

int
untextify_len(uchar *in, int ilen) {
  int i;
  int olen;
  
  for(olen=0,i=0; i<ilen; i++) {
    if(in[i] == '%') {
      if(in[i+1] == '%') {
	olen++;
	i++;
      } else {
	olen++;
	i+=2;
      }
    } else 
      olen++;    
  }
  return olen;
}

#if 0
char 
b64_unmap(char c) { 
  
  if(c>='A' && c<='Z')
    c -= 'A';
  else if(c>='a' && c<='z') {
    c -= ('a' - 26);
  } else if(c>='0' && c<='9') {
    c -= ('0' - 52);
  }else if(c == '+')
    c = 62;
  else if(c == '/')
    c = 63;
  else if(c == '=')
    c = 0;
  
  return c;
}
#else

/* as per rfc1113 */
int base64_unmap(char *in_block) {
  int i;
  uchar *c;
  
  for(i=0; i<4; i++) {
    c = in_block + i;
    
    if(*c>='A' && *c<='Z') {
      *c -= 'A';
      continue;
    }
    
    if(*c>='a' && *c<='z') {
      *c -= 'a';
      *c += 26;
      continue;
    }
    
    if(*c == '+') {
      *c = 62;
      continue;
    }
    
    if(*c == '/') {
      *c = 63;
      continue;
    }
    
    if(*c == '=') {
      *c = 0;
      continue;
    }
    
    *c -= '0';
    *c += 52;
  }
  return 0;
}
#endif

int base64_decode(uchar *uin, int inlen, uchar *out) {
  int i;
  uchar *in;
  char *in_block;
  char *out_block;
  
  in = xcalloc(1, inlen);
  memcpy(in, uin, inlen);
  out_block = out;
  in_block = in;

  for(i=0; i<inlen; i+=4) {

    if(*in_block == '=') {
      xfree(in);
      return 0;
    }

    base64_unmap(in_block);
    memset(out_block, 0, 3);    
    out_block[0] = ((in_block[0]<<2) & 0xfc) | ((in_block[1]>>4) & 0x03);
    out_block[1] = ((in_block[1]<<4) & 0xf0) | ((in_block[2]>>2) & 0x0f);
    out_block[2] = ((in_block[2]<<6) & 0xc0) | ((in_block[3]   ) & 0x3f);

    out_block += 3;
    in_block += 4;
  }
  
  xfree(in);
  return 0;
}


int
base64_encode(uchar *in, int ilen, uchar *out, int *olen) {
  int i, rem;

  *olen = ((ilen/3) << 2);
  if((rem = ilen%3))
    *olen += 4;
  memset(out, 0, *olen);
  
  for(i=ilen; i>0; i-=3) {

    out[0] = b64_map(in[0]>>2);
    out[1] = b64_map(((in[0] & 0x03)<<4) | ((in[1] & 0xf0)>>4));
    switch(i) {
    case 2:
      out[2] = b64_map(((in[1] & 0x0f)<<2) | ((in[2] & 0xc0)>>6));
      out[3] = '=';      
      break;
    case 1:
      out[2] = '=';
      out[3] = '=';
      break;
    default:
      out[2] = b64_map(((in[1] & 0x0f)<<2) | ((in[2] & 0xc0)>>6));
      out[3] = b64_map(in[2] & 0x3f);
      break;
    }    

    in+= 3;
    out+=4;
  }
  
  return 0;
}


/* Binary string manipulation */

bstring *
bstring_new(void) {
  bstring *s;

  s = xcalloc(1, sizeof(bstring));
  bstring_zero(s);
  return s;
}

int
bstring_zero(bstring *s) {
  memset(s->p, '\0', BSTRING_MAXLEN);
  s->l = 0;
  return 0;
}

int
bstring_set_str(bstring *s, uchar *p, int l) {
  memset(s->p, '\0', BSTRING_MAXLEN);
  memcpy(s->p, p, l);
  s->l = l;
  return 0;
}

int
bstring_get_str(bstring *s, uchar *p, int l) {
  memcpy(p, s->p, MIN(l, BSTRING_MAXLEN));
  return 0;
}

int
bstring_cmp(bstring *s1, bstring *s2) {
  if(s1->l != s2->l)
    return 1;
  return memcmp(s1->p, s2->p, s1->l);  
}

int
bstring_set(bstring *s1, bstring *s2) {
  s1->l = s2->l;
  memset(s1->p, '\0', BSTRING_MAXLEN);
  memcpy(s1->p, s2->p, s1->l);
  return 0;
}

int
bstring_destroy(bstring *s) {
  xfree(s->p);
  return 0;
}

/* generic message support */

msg *
msg_new(void) {
  msg *s;

  s = xcalloc(1, sizeof(msg));
  s->msg = array_init(sizeof(msg_t));
  s->replace = MSG_DEF_REPLACE;
  msg_zero(s);
  return s;
}

int
msg_zero(msg *s) {
  array_zero(s->msg);
  return 0;
}

static msg_t *
msg_find(msg *m, char *lhs) {
  int i;
  msg_t *p;
  int l;

  l = strlen(lhs);
  for(i=0; (p=array_get(m->msg, i)); i++) {
    if(p->lhs && l == p->ll &&
       !strncmp(lhs, p->lhs, l))
      return p;
  }
  return (msg_t *)NULL;  
}

int
msg_add_str(msg *m, char *lhs, uchar *rhs, int rhsl) {
  msg_t p;
  msg_t *pp;

  if(!rhsl)
    rhsl = strlen(rhs);

  if(!lhs) {
    pp = &p;
    pp->ll = 0;
    pp->lhs = NULL;
  } else if(m->replace && (pp = msg_find(m, lhs))) {
    xfree(pp->rhs);
  } else {
    pp = &p;
    pp->ll = strlen(lhs);
    pp->lhs = xcalloc(1, pp->ll);
    strncpy(pp->lhs, lhs, pp->ll);
  }

  pp->rhs = xcalloc(1, rhsl+1);
  memcpy(pp->rhs, rhs, rhsl);
  pp->rl = rhsl;
  
  array_add(m->msg, pp);
  return 0;
}

int
msg_add_int(msg *m, char *lhs, int irhs) {
  char rhs[BSTRING_MAXLEN];

  snprintf(rhs, BSTRING_MAXLEN, "%d", irhs);
  return msg_add_str(m, lhs, rhs, 0);
}

int
msg_get_str(msg *m, char *lhs, uchar *rhs, int rhsl, int *rrhsl) {
  msg_t *p;
  
  if(!(p = msg_find(m, lhs)))
    return 1;  
  if(rhsl < p->rl)
    return 2;

  memset(rhs, '\0', rhsl);
  memcpy(rhs, p->rhs, p->rl);
  *rrhsl = p->rl;
  return 0;
}

int
msg_get_index(msg *m, unsigned int index, char *lhs, int lhsl, uchar *rhs, 
		  int rhsl, int *rrhsl) {
  msg_t *p;

  if(!(p = array_get(m->msg, index)))
    return 1;
  if(lhsl < p->ll)
    return 2;
  if(rhsl < p->rl)
    return 3;
  
  memset(lhs, '\0', lhsl);
  memset(rhs, '\0', rhsl);
  if(p->lhs)
    strncpy(lhs, p->lhs, p->ll);
  memcpy(rhs, p->rhs, p->rl);
  *rrhsl = p->rl;
  return 0;
}

uchar *
msg_get(msg *m, uint index) {
  msg_t *p;

  if(!(p = array_get(m->msg, index)))
    return NULL;
  return p->rhs;
}

uchar *
msg_lhs(msg *m, uint index) {
  msg_t *p;

  if(!(p = array_get(m->msg, index)))
    return NULL;
  return p->lhs;
}

uchar *
msg_get_strp(msg *m, char *lhs) {
  msg_t *p;

  if(!(p = msg_find(m, lhs)))
    return (uchar *)NULL;
  return p->rhs;
}

int
msg_get_int(msg *m, char *lhs) {
  uchar *p;

  if(!(p = msg_get_strp(m, lhs)))
    return -1;  
  return atoi(p);
}

int
msg_destroy(msg *m) {
  int i;
  msg_t *p;
  
  for(i=0; (p=array_get(m->msg, i)); i++) {
    if(p->lhs)
      xfree(p->lhs);
    xfree(p->rhs);
  }

  array_nuke(m->msg);
  return 0;
}

unsigned int
msg_getlen(msg *p, unsigned int index) {
  msg_t *m;

  if(!(m = array_get(p->msg, index)))
    return -1;
  return m->rl;
}

int
msg_getopt(msg *m, uint opt, int *val) {  
  switch(opt) {
  case MSG_REPLACE:
    *val = m->replace;
    return 0;
    break;
  default:
    return 1;
    break;
  }
}

int
msg_setopt(msg *m, uint opt, int val) {
  switch(opt) {
  case MSG_REPLACE:
    m->replace = val;
    return 0;
    break;
  default:
    return 1;
    break;
  }
}

int
msg_dump(msg *m) {
  int i;
  msg_t *p;

  llog(4, "msg_dump: message with %d elements\n", msg_nelems(m));
  for(i=0; (p=array_get(m->msg, i)); i++) {
    llog(4, " %d %s %s\n", i, p->lhs, p->rhs);
  }
  return 0;
}

/* challenge response authentication stuff */
int 
chresp_generate_nonce(bstring *nonce, int len) {
  RAND_bytes(nonce->p, len);
  nonce->l = len;
  return 0;
}

int
chresp_init(char *realpass, bstring *nonce, bstring *adminhash) { 
  MD5_CTX ctx;

  adminhash->l = 16;
  MD5_Init(&ctx);
  MD5_Update(&ctx, nonce->p, nonce->l);
  MD5_Update(&ctx, realpass, strlen(realpass));
  MD5_Final(adminhash->p, &ctx);
  return 0;
}

int
chresp_authenticate(bstring *adminhash, bstring *response) {
  return !memcmp(adminhash->p, response->p, adminhash->l);
}

/* generator options processing */
/* XXX: ought to be deprecated */
int
param_get_int(char *optstr, char *name, int *param, int def) {
  char *lopt;
  char *tokens[16];
  int i;

  if(!optstr) {
    *param = def;
    return 0;
  }

  lopt = xstrdup(optstr);
  for(i=0; i<16; i++)
    tokens[i] = NULL;
  for(i=1,tokens[0]=strtok(lopt, " ,\t\n"); (tokens[i]=strtok(NULL, " ,\t\n")); i++)
    ;
  for(i=0; tokens[i]; i++) {
    if(!strncmp(name, tokens[i], strlen(name)) && tokens[i][strlen(name)] == '=') {
      *param = atoi(&tokens[i][strlen(name)+1]);
      xfree(lopt);
      return 0;
    }
  }
  xfree(lopt);
  *param = def;
  return 1;
}

int
param_get_str(char *optstr, char *name, char *param, char *def) {
  char *lopt;
  char *tokens[16];
  int i;

  if(!optstr) {
    strcpy(param, def);
    return 0;
  }

  lopt = xstrdup(optstr);
  for(i=0; i<16; i++)
    tokens[i] = NULL;
  for(i=1,tokens[0]=strtok(lopt, " ,\t\n"); (tokens[i]=strtok(NULL, " ,\t\n")); i++)
    ;
  for(i=0; tokens[i]; i++) {
    if(!strncmp(name, tokens[i], strlen(name)) && tokens[i][strlen(name)] == '=') {
      strcpy(param, &tokens[i][strlen(name)+1]);
      xfree(lopt);
      return 0;
    }
  }
  xfree(lopt);
  strcpy(param, def);
  return 1;
}


int
isinitc(char p, char *set) {
  char *c;

  for(c = set; *c; c++)
    if(p == *c)
      return 1;
  return 0;
}

int
zap_whitespace(char *victim, int len, int mode) {
  char *oldv;
  char temp[DEF_LINELEN];

  if(mode & ZAP_LEFT) {
    oldv = victim;
    while(isinitc(*victim, WHITESPACE))
      victim++;
    strncpy(temp, victim, DEF_LINELEN);
    strncpy(oldv, temp, len);
    victim = oldv;
  }

  if(mode & ZAP_RIGHT) {
    oldv = victim + strlen(victim)-1;
    while(oldv!= victim) {
      if(isinitc(*oldv, WHITESPACE))
	*oldv-- = '\0';
      else
	break;
    }
  }
    
  return 0;
}

int
buildtime(char *p, char *str, int len) {
  struct tm stm;
  time_t t;

  if(!p || sscanf(p, "%lu", &t) != 1 || !t) {
    strncpy(str, "None", len);
    return 1;
  }

  localtime_r(&t, &stm);
  strftime(str, len, "%Y/%m/%d %H:%M", &stm);
  return 0;
}

int
buildlapse(char *p, char *str, int len) {
  struct tm stm;
  time_t t;
  char buf2[DEF_LINELEN];
  
  if(!p || sscanf(p, "%lu", &t) != 1 || !t) {
    strncpy(str, "None", len);
    return 1;
  }
  
  gmtime_r(&t, &stm);
  strftime(buf2, DEF_LINELEN, "%H:%M:%S", &stm);
  snprintf(str, len, "%lu:%s", t/(60*60*24), buf2);
  return 0;
}

#ifdef _DEBUG_COMMON
int
main(int argc, char **argv) {
  int i;
  msg *m;
  uchar *p;

  m = msg_new();
  msg_add(m, "foo", 0);
  msg_add(m, "foo", 0);
  msg_add(m, "bar", 0);
  i = msg_nelems(m);
  p = msg_get(m, 2);
  
}
#endif

