/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: random.c,v 1.3 2003/02/05 04:38:37 kos Exp $
 *
 * Essentially a crap RNG based on MD5. It is not guaranteed to give
 * results that have a one to one mapping to the index, as with any RNG.
 * Use obfuscate for that.
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#include "common.h"
#include "keyspace.h"
#include "binary.h"
#include "xmalloc.h"
#include "generator.h"
#include "random.h"

random_t *
random_init(char *params) {
  random_t *k;

  k = (random_t *)xcalloc(1, sizeof(random_t));
  param_get_str(params, "seed", k->seed, RNG_DEF_SEED);
  param_get_int(params, "len", &k->len, RNG_DEF_LENGTH);

  if(k->len > RNG_MAX_LENGTH ||
     k->len <= 0)
    return (random_t *)NULL;

  keyspace_init(k->minindex);
  keyspace_init(k->maxindex);
  keyspace_init(k->index);

  keyspace_fromint(0, k->minindex);  
  keyspace_fromint(2, k->maxindex);
  keyspace_pow(k->maxindex, k->len*8, k->maxindex);
  keyspace_set(k->index, k->minindex);
  return k;
}

key_index_t *
random_minindex(random_t *b) {
  return &b->minindex;
}

key_index_t *
random_maxindex(random_t *b) {
  return &b->maxindex;
}

key_index_t *
random_curindex(random_t *b) {
  return &b->index;
}

int
random_minlen(random_t *b) {
  return b->len;
}

int
random_maxlen(random_t *b) {
  return b->len;
}

int
random_destroy(random_t *k) {
  keyspace_destroy(k->minindex);
  keyspace_destroy(k->maxindex);
  keyspace_destroy(k->index);
  xfree(k);
  return 0;
}

int
random_fetch(random_t *k, kchar *buf, int userlen, int *len) {
  MD5_CTX ctx;
  unsigned char digest[MD5_DIGEST_LENGTH];
  int i;
  int j;

  if(userlen < k->len)
    return 1;

  MD5_Init(&ctx);
  MD5_Update(&ctx, k->x_n, RNG_MAX_LENGTH);
  MD5_Final(digest, &ctx);
  
  *len = k->len;
  memcpy(buf, digest, k->len);
  for(i=1; i<(RNG_MAX_LENGTH/k->len); i+=k->len) {
    for(j=0; j<k->len; j++)
      buf[j] += digest[i+j];
  }

  memcpy(k->x_n, digest, RNG_MAX_LENGTH);

  keyspace_inc(k->index);
  return 0;
}

int
random_set(random_t *k, key_index_t index) {
  unsigned char digest[RNG_MAX_LENGTH];
  MD5_CTX ctx;
  
  while(keyspace_cmp(k->index, index)) {
    MD5_Init(&ctx);
    MD5_Update(&ctx, k->x_n, RNG_MAX_LENGTH);
    MD5_Final(digest, &ctx);
  
    memcpy(k->x_n, digest, RNG_MAX_LENGTH);
    keyspace_inc(k->index);
  }
  return 0;
}

int 
random_done(random_t *k) {
  return (keyspace_cmp(k->index, k->maxindex) == 0);
}

generator_impl_t random_impl = {
  "random",
  (gen_init)random_init,
  (gen_set)random_set,
  (gen_fetch)random_fetch,
  (gen_done)random_done,
  (gen_destroy)random_destroy,
  (gen_minindex)random_minindex,
  (gen_maxindex)random_maxindex,
  (gen_curindex)random_curindex,
  (gen_minlen)random_minlen,
  (gen_maxlen)random_maxlen
};
