/* mg-canvas-db-relations.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "marshal.h"
#include <libmergeant/mg-database.h>
#include <libmergeant/mg-db-table.h>
#include <libmergeant/mg-db-field.h>
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-db-constraint.h>
#include <libmergeant/mg-ref-base.h>
#include "mg-canvas-db-relations.h"
#include "mg-canvas-entity.h"
#include "mg-canvas-field.h"
#include "mg-canvas-fkconstraint.h"
#include "mg-graph-item.h"

static void mg_canvas_db_relations_class_init (MgCanvasDbRelationsClass * class);
static void mg_canvas_db_relations_init       (MgCanvasDbRelations * canvas);
static void mg_canvas_db_relations_finalize   (GObject   * object);

/* virtual functions */
static void       create_canvas_items (MgCanvas *canvas);
static void       clean_canvas_items  (MgCanvas *canvas);
static void       graph_item_added    (MgCanvas *canvas, MgGraphItem *item);
static void       graph_item_dropped  (MgCanvas *canvas, MgGraphItem *item);
static GtkWidget *build_context_menu  (MgCanvas *canvas);
static void       set_pixels_per_unit (MgCanvas *canvas, gdouble n);

static void       db_table_removed_cb (MgDatabase *db, MgDbTable *table, MgCanvas *canvas);
static void       db_nullified_cb     (MgDatabase *db, MgCanvas *canvas);
static void       db_constraint_added_cb (MgDatabase *db, MgDbConstraint *cstr, MgCanvas *canvas);

static void       drag_action_dcb     (MgCanvas *canvas, MgCanvasItem *from_item, MgCanvasItem *to_item);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* HASH functions for the key of type FKTables */
typedef struct {
	MgDbTable *fk_table;
	MgDbTable *ref_pk_table;
} FKTables;

static guint
g_fktables_hash (gconstpointer key)
{
	FKTables *thekey = (FKTables *) key;
	gint h;

	h = (gint) (thekey->fk_table);
	h = (h << 5) - h + (gint) (thekey->ref_pk_table);
	
	/*g_print ("HASH: %p,%p -> %ld\n", thekey->fk_table, thekey->ref_pk_table, h);*/
	return h;
}

static gboolean
g_fktables_equal (gconstpointer a, gconstpointer b)
{
	FKTables *akey = (FKTables *) a;
	FKTables *bkey = (FKTables *) b;

	if (!a && !b)
		return TRUE;

	if (!a || !b)
		return FALSE;

	if ((akey->fk_table == bkey->fk_table) &&
	    (akey->ref_pk_table == bkey->ref_pk_table))
		return TRUE;
	else
		return FALSE;
}


struct _MgCanvasDbRelationsPrivate
{
	GSList      *ent_items;
	GHashTable  *hash_tables; /* key = MgDbTable, value = MgCanvasEntity */
	GHashTable  *hash_constraints; /* key = FKTables, value = MgCanvasFkconstrains */

	MgDatabase  *db;
};

guint
mg_canvas_db_relations_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCanvasDbRelationsClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_canvas_db_relations_class_init,
			NULL,
			NULL,
			sizeof (MgCanvasDbRelations),
			0,
			(GInstanceInitFunc) mg_canvas_db_relations_init
		};		

		type = g_type_register_static (MG_CANVAS_TYPE, "MgCanvasDbRelations", &info, 0);
	}
	return type;
}

static void
mg_canvas_db_relations_init (MgCanvasDbRelations * canvas)
{
	canvas->priv = g_new0 (MgCanvasDbRelationsPrivate, 1);
	canvas->priv->hash_tables = g_hash_table_new (NULL, NULL);
	canvas->priv->hash_constraints = g_hash_table_new_full (g_fktables_hash, g_fktables_equal, g_free, NULL);
	canvas->priv->db = NULL;
}

static void
mg_canvas_db_relations_class_init (MgCanvasDbRelationsClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	/* MgCanvas virtual functions */
	MG_CANVAS_CLASS (class)->create_canvas_items = create_canvas_items;
	MG_CANVAS_CLASS (class)->clean_canvas_items = clean_canvas_items;
	MG_CANVAS_CLASS (class)->graph_item_added = graph_item_added;
	MG_CANVAS_CLASS (class)->graph_item_dropped = graph_item_dropped;
	MG_CANVAS_CLASS (class)->build_context_menu = build_context_menu;
	MG_CANVAS_CLASS (class)->set_pixels_per_unit = set_pixels_per_unit;

	MG_CANVAS_CLASS (class)->drag_action = drag_action_dcb;
	
	object_class->finalize = mg_canvas_db_relations_finalize;
}

static void
mg_canvas_db_relations_finalize (GObject *object)
{
	MgCanvasDbRelations *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_DB_RELATIONS (object));

	canvas = MG_CANVAS_DB_RELATIONS (object);

	if (canvas->priv) {
		if (canvas->priv->db) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (canvas->priv->db),
							      G_CALLBACK (db_table_removed_cb), canvas);
			g_signal_handlers_disconnect_by_func (G_OBJECT (canvas->priv->db),
							      G_CALLBACK (db_nullified_cb), canvas);
			g_signal_handlers_disconnect_by_func (G_OBJECT (canvas->priv->db),
							      G_CALLBACK (db_constraint_added_cb), canvas);
		}

		g_hash_table_destroy (canvas->priv->hash_tables);
		g_hash_table_destroy (canvas->priv->hash_constraints);


		g_free (canvas->priv);
		canvas->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static GtkWidget *
drag_action_question_dlg (MgCanvas *canvas, GSList *cstr_list)
{
	GtkWidget *dlg, *label, *hbox, *box, *button, *bgroup;
	GSList *list = cstr_list;

	dlg = gtk_dialog_new_with_buttons (_("Constraint information request"), NULL,
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
					   NULL);
	box = GTK_DIALOG (dlg)->vbox;
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<big><b>New tables relations:</b></big>\n"
				 "You have selected a pair of table's fields to be related to each other "
				 "and must decide to either create a new constraint for it or modify an "
				 "already existing constraint."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	
	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 5);
	gtk_widget_show (label);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	box = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), box, TRUE, TRUE, 0);
	gtk_widget_show (box);

	button = gtk_radio_button_new_with_label (NULL,_("Create a new constraint"));
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 0);
	gtk_widget_show (button);
	g_object_set_data (G_OBJECT (dlg), "rb", button);
	bgroup = button;

	while (list) {
		GtkWidget *table;
		GSList *pairs, *plist;
		gint nb_pairs, i;

		pairs = mg_db_constraint_fkey_get_fields (MG_DB_CONSTRAINT (list->data));
		nb_pairs = g_slist_length (pairs);

		button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (bgroup), 
								      _("Add to constraint:"));
		gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 5);
		gtk_widget_show (button);
		g_object_set_data (G_OBJECT (button), "constraint", list->data);

		hbox = gtk_hbox_new (FALSE, 0); /* HIG */
                gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
                gtk_widget_show (hbox);
                label = gtk_label_new ("      ");
                gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
                gtk_widget_show (label);

		table = gtk_table_new (nb_pairs, 3, FALSE);
		gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
		gtk_widget_show (table);
		plist = pairs;
		i = 0;
		while (plist) {
			MgDbConstraintFkeyPair *pair = MG_DB_CONSTRAINT_FK_PAIR (pairs->data);
			gchar *str;

			str = g_strdup_printf ("%s.%s", 
					       mg_base_get_name (MG_BASE (mg_field_get_entity (pair->fkey))),
					       mg_base_get_name (MG_BASE (pair->fkey)));
			label = gtk_label_new (str);
			g_free (str);
			gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i+1, 0, 0, 0, 0);
			gtk_widget_show (label);

			label = gtk_label_new ("--");
			gtk_table_attach (GTK_TABLE (table), label, 1, 2, i, i+1, 0, 0, 5, 0);
			gtk_widget_show (label);
			
			str = g_strdup_printf ("%s.%s", 
					       mg_base_get_name (MG_BASE (mg_field_get_entity (pair->ref_pkey))),
					       mg_base_get_name (MG_BASE (pair->ref_pkey)));
			label = gtk_label_new (str);
			g_free (str);
			gtk_table_attach (GTK_TABLE (table), label, 2, 3, i, i+1, 0, 0, 0, 0);
			gtk_widget_show (label);

			g_free (plist->data);
			plist = g_slist_next (plist);
			i++;
		}
		g_slist_free (pairs);
		
		
		list = g_slist_next (list);
	}

	return dlg;
}

static void
drag_action_dcb (MgCanvas *canvas, MgCanvasItem *from_item, MgCanvasItem *to_item)
{
	MgField *fk_field = NULL, *ref_pk_field = NULL;
	MgDbTable *fk_table, *ref_pk_table;
	GSList *cstr_list;
	gboolean create_constraint = FALSE;

	if (IS_MG_CANVAS_FIELD (from_item))
		fk_field = mg_canvas_field_get_field (MG_CANVAS_FIELD (from_item));
	if (IS_MG_CANVAS_FIELD (to_item))
		ref_pk_field = mg_canvas_field_get_field (MG_CANVAS_FIELD (to_item));

	/* dismiss if we don't have two fields */
	if (!fk_field || !IS_MG_DB_FIELD (fk_field) || 
	    !ref_pk_field || !IS_MG_DB_FIELD (ref_pk_field))
		return;

	fk_table = MG_DB_TABLE (mg_field_get_entity (fk_field));
	ref_pk_table = MG_DB_TABLE (mg_field_get_entity (ref_pk_field));

	cstr_list = mg_database_get_tables_fk_constraints (MG_CANVAS_DB_RELATIONS (canvas)->priv->db,
							   fk_table, ref_pk_table, FALSE);
	if (cstr_list) {
		/* already some existing constraints */
		GSList *clist;
		MgDbConstraint *found = NULL;

		/* see if a constraint already has that pair of fields */
		clist = cstr_list;
		while (clist && !found) {
			GSList *pairs, *plist;

			pairs = mg_db_constraint_fkey_get_fields (MG_DB_CONSTRAINT (clist->data));
			plist = pairs;
			while (plist) {
				if (!found) {
					MgDbConstraintFkeyPair *pair = MG_DB_CONSTRAINT_FK_PAIR (pairs->data);
					if ((((MgField*) pair->fkey == fk_field) && ((MgField*) pair->ref_pkey == ref_pk_field)) ||
					    (((MgField*) pair->fkey == ref_pk_field) && ((MgField*) pair->ref_pkey == fk_field)))
						found = MG_DB_CONSTRAINT (clist->data);
				}
				g_free (plist->data);
				plist = g_slist_next (plist);
			}
			g_slist_free (pairs);

			clist = g_slist_next (clist);
		}
		
		if (found) {
			/* simply display a message */
			GtkWidget *dlg;

			if (mg_db_constraint_get_table (found) == fk_table)
				dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
									  GTK_BUTTONS_OK,
									  _("<big>A constraint already exists</big>\n"
									    "for the selected couple of fields."));
			else
				dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
									  GTK_BUTTONS_OK,
									  _("<big>A constraint already exists</big>\n"
									    "for the selected couple of fields, but "
									    "not in the same order of tables, please "
									    "remove that other constraint before adding "
									    "a new one."));
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}
		else {
			/* further information required */
			GSList *clist = NULL, *list;
			GtkWidget *dlg;
			gint response;

			/* keep only the constraints where the table is the FK_table */
			list = cstr_list;
			while (list) {
				if (mg_db_constraint_get_table (MG_DB_CONSTRAINT (list->data)) == fk_table)
					clist = g_slist_append (clist, list->data);
				list = g_slist_next (list);
			}
			g_slist_free (cstr_list);
			cstr_list = clist;

			dlg = drag_action_question_dlg (canvas, cstr_list);
			response = gtk_dialog_run (GTK_DIALOG (dlg));
			if (response == GTK_RESPONSE_ACCEPT) {
				GtkWidget *rb = g_object_get_data (G_OBJECT (dlg), "rb");
				GSList *group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
				MgDbConstraint *add_to = NULL;

				while (group && !add_to) {
					if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (group->data))) {
						add_to = g_object_get_data (G_OBJECT (group->data), "constraint");
					}
					group = g_slist_next (group);
				}
				
				if (add_to) {
					GSList *nlist;
					MgDbConstraintFkeyPair *pair;

					nlist = mg_db_constraint_fkey_get_fields (add_to);
					pair = g_new0 (MgDbConstraintFkeyPair, 1);
					pair->fkey = MG_DB_FIELD (fk_field);
					pair->ref_pkey = MG_DB_FIELD (ref_pk_field);
					pair->ref_pkey_repl = NULL;
					nlist = g_slist_append (nlist, pair);
					mg_db_constraint_fkey_set_fields (add_to, nlist);
					
					/* memory libreation */
					list = nlist;
					while (list) {
						g_free (list->data);
						list = g_slist_next (list);
					}
					g_slist_free (nlist);
				}
				else 
					create_constraint = TRUE;
			}
			gtk_widget_destroy (dlg);
		}
		
		g_slist_free (cstr_list);
	}
	else
		/* no existing constraint yet: create one */
		create_constraint = TRUE;


	/* actually create the constraint */
	if (create_constraint) {
		GSList *nlist, *list;
		MgDbConstraint *cstr;
		MgDbConstraintFkeyPair *pair;

		cstr = MG_DB_CONSTRAINT (mg_db_constraint_new (fk_table, CONSTRAINT_FOREIGN_KEY));
		g_object_set (G_OBJECT (cstr), "user_constraint", TRUE, NULL);
		
		/* add FK pair */
		pair = g_new0 (MgDbConstraintFkeyPair, 1);
		pair->fkey = MG_DB_FIELD (fk_field);
		pair->ref_pkey = MG_DB_FIELD (ref_pk_field);
		pair->ref_pkey_repl = NULL;

		nlist = g_slist_append (NULL, pair);
		mg_db_constraint_fkey_set_fields (cstr, nlist);
		
		/* memory libreation */
		list = nlist;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (nlist);

		mg_database_add_constraint (MG_CANVAS_DB_RELATIONS (canvas)->priv->db, cstr);
		g_object_unref (G_OBJECT (cstr));
	}
}

/**
 * mg_canvas_db_relations_new
 * @conf: a #MgConf object
 * @graph: a #MgGraph object, or %NULL
 *
 * Creates a new canvas widget to display the relations between the database's tables.
 * The database is the one managed by the #MgConf object which @graph refers.
 *
 * @graph contains all the table's graphical representations and their respective
 * locations on the canvas, or can be %NULL (in which case nothing is displayed)
 *
 * Returns: a new #MgCanvasDbRelations widget
 */
GtkWidget *
mg_canvas_db_relations_new (MgConf *conf, MgGraph *graph)
{
	GObject *obj;
	MgDatabase *db;
	MgCanvasDbRelations *canvas;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	if (graph) {
		g_return_val_if_fail (IS_MG_GRAPH (graph), NULL);
		g_return_val_if_fail (conf == mg_base_get_conf (MG_BASE (graph)), NULL);
	}

        obj = g_object_new (MG_CANVAS_DB_RELATIONS_TYPE, "aa", FALSE, NULL);
	gnome_canvas_set_center_scroll_region (GNOME_CANVAS (obj), TRUE);
	canvas = MG_CANVAS_DB_RELATIONS (obj);

	/* connecting to the MgDatabase */
	db = mg_conf_get_database (conf);
	canvas->priv->db = db;
	g_signal_connect (G_OBJECT (db), "table-removed",
			  G_CALLBACK (db_table_removed_cb), obj);
	g_signal_connect (G_OBJECT (db), "nullified",
			  G_CALLBACK (db_nullified_cb), obj);
	g_signal_connect (G_OBJECT (db), "constraint_added",
			  G_CALLBACK (db_constraint_added_cb), obj);

	/* populating the canvas */
	g_object_set (obj, "graph", graph, NULL);

	return GTK_WIDGET (obj);
}


static void
db_table_removed_cb (MgDatabase *db, MgDbTable *table, MgCanvas *canvas)
{
	MgGraphItem *item;
	MgGraph *graph;

	g_object_get (G_OBJECT (canvas), "graph", &graph, NULL);
	if (!graph)
		return;
	item = mg_graph_get_item_from_obj (graph, MG_BASE (table), FALSE);
	if (item) 
		mg_base_nullify (MG_BASE (item));
}

static void
db_nullified_cb (MgDatabase *db, MgCanvas *canvas)
{
	/* disconnecting signals */
	MG_CANVAS_DB_RELATIONS (canvas)->priv->db = NULL;
	g_signal_handlers_disconnect_by_func (G_OBJECT (db),
					      G_CALLBACK (db_table_removed_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (db),
					      G_CALLBACK (db_nullified_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (db),
					      G_CALLBACK (db_constraint_added_cb), canvas);

	/* clean the canvas */
	clean_canvas_items (canvas);
}

static void canvas_fkconstraint_destroy_cb (MgCanvasItem *canvas_item, MgCanvasDbRelations *canvas);
static void
db_constraint_added_cb (MgDatabase *db, MgDbConstraint *cstr, MgCanvas *canvas)
{
	FKTables key;
	MgCanvasItem *canvas_fk;
	GnomeCanvasItem *root, *canvas_item;

	/* ignore non FK constraints */
	if (mg_db_constraint_get_constraint_type (cstr) != CONSTRAINT_FOREIGN_KEY)
		return;

	root = GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas)));

	key.fk_table = mg_db_constraint_get_table (cstr);
	key.ref_pk_table = mg_db_constraint_fkey_get_ref_table (cstr);
	canvas_fk = g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints, &key);
	
	if (canvas_fk) 
		mg_canvas_fkconstraint_add_constraint (MG_CANVAS_FKCONSTRAINT (canvas_fk), cstr);
	else {
		if (g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, key.fk_table) &&
		    g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, key.ref_pk_table)) {
			FKTables *newkey = g_new0 (FKTables, 1);
			
			canvas_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (root),
							     MG_CANVAS_FKCONSTRAINT_TYPE,
							     "fk_constraint", cstr,
							     NULL);
			
			newkey->fk_table = key.fk_table;
			newkey->ref_pk_table = key.ref_pk_table;
			g_hash_table_insert (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints, 
					     newkey, canvas_item);
			g_object_set_data (G_OBJECT (canvas_item), "key", newkey);
			g_signal_connect (G_OBJECT (canvas_item), "destroy",
					  G_CALLBACK (canvas_fkconstraint_destroy_cb), canvas);
		}
	}
}

static void
canvas_fkconstraint_destroy_cb (MgCanvasItem *canvas_item, MgCanvasDbRelations *canvas)
{
	FKTables *key = g_object_get_data (G_OBJECT (canvas_item), "key");
	g_assert (key);
	g_hash_table_remove (canvas->priv->hash_constraints, key);
	g_object_set_data (G_OBJECT (canvas_item), "key", NULL);
}

/*
 * Add all the required GnomeCanvasItem objects for the associated #MgGraph object 
 */
static void
create_canvas_items (MgCanvas *canvas)
{
	GSList *list, *graph_items;
	MgGraph *graph = mg_canvas_get_graph (canvas);

	graph_items = mg_graph_get_items (graph);
	list = graph_items;
	while (list) {
		graph_item_added (canvas, MG_GRAPH_ITEM (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (graph_items);
}

static void
clean_canvas_items (MgCanvas *canvas)
{
	/* destroy items */
	while (GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list) 
		gtk_object_destroy (GTK_OBJECT ((GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list)->data));

	/* clean memory */
	g_hash_table_destroy (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables);
	g_hash_table_destroy (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints);
	MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables = g_hash_table_new (NULL, NULL);
	MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints = 
		g_hash_table_new_full (g_fktables_hash, g_fktables_equal, g_free, NULL);
}

/*
 * Add the MgCanvasEntity corresponding to the graph item
 */
static void
graph_item_added (MgCanvas *canvas, MgGraphItem *item)
{
	GnomeCanvasItem *canvas_item, *root;
	MgBase *ref_obj = mg_graph_item_get_ref_object (item);

	root = GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas)));
	if (IS_MG_DB_TABLE (ref_obj)) {
		GSList *fklist, *list;

		/* MgCanvasEntity for the table */
		canvas_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (root),
						     MG_CANVAS_ENTITY_TYPE,
						     "entity", ref_obj,
						     "x", 50.,
						     "y", 50.,
						     "graph_item", item,
						     NULL);
		mg_canvas_declare_item (canvas, MG_CANVAS_ITEM (canvas_item));
		g_hash_table_insert (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, ref_obj, canvas_item);
		gnome_canvas_update_now (GNOME_CANVAS (canvas)); /* forced here because of a GnomeCanvas bug */

		/* MgCanvasFkconstraint object(s) for the FK constraints using that table */
		fklist = mg_database_get_tables_fk_constraints (MG_CANVAS_DB_RELATIONS (canvas)->priv->db,
								MG_DB_TABLE (ref_obj), NULL, FALSE);
		list = fklist;
		while (list) {
			FKTables key;
			MgDbConstraint *fkcons = MG_DB_CONSTRAINT (list->data);
			MgCanvasItem *canvas_fk;

			key.fk_table = mg_db_constraint_get_table (fkcons);
			key.ref_pk_table = mg_db_constraint_fkey_get_ref_table (fkcons);
			canvas_fk = g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints, &key);
			
			if (canvas_fk)
				mg_canvas_fkconstraint_add_constraint (MG_CANVAS_FKCONSTRAINT (canvas_fk), fkcons);
			else {
				if (g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, key.fk_table) &&
				    g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, key.ref_pk_table)) {
					FKTables *newkey = g_new0 (FKTables, 1);
					
					canvas_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (root),
									     MG_CANVAS_FKCONSTRAINT_TYPE,
									     "fk_constraint", fkcons,
									     NULL);
					
					newkey->fk_table = key.fk_table;
					newkey->ref_pk_table = key.ref_pk_table;
					g_hash_table_insert (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_constraints, 
							     newkey, canvas_item);
					g_object_set_data (G_OBJECT (canvas_item), "key", newkey);
					g_signal_connect (G_OBJECT (canvas_item), "destroy",
							  G_CALLBACK (canvas_fkconstraint_destroy_cb), canvas);
				}
			}

			list = g_slist_next (list);
		}
		g_slist_free (fklist);
	}
}

/*
 * Remove the MgCanvasEntity corresponding to the graph item
 */
static void
graph_item_dropped (MgCanvas *canvas, MgGraphItem *item)
{
	MgBase *ref_obj = mg_graph_item_get_ref_object (item);

	if (IS_MG_DB_TABLE (ref_obj)) {
		GtkObject *canvas_item = g_hash_table_lookup (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables,
							      ref_obj);
		if (canvas_item) {
			gtk_object_destroy (canvas_item);
			g_hash_table_remove (MG_CANVAS_DB_RELATIONS (canvas)->priv->hash_tables, ref_obj);
		}
	}
}

static void popup_add_table_cb (GtkMenuItem *mitem, MgCanvasDbRelations *canvas);
static GtkWidget *
build_context_menu (MgCanvas *canvas)
{
	GtkWidget *menu, *mitem, *submenu, *submitem;
	GSList *tables, *list;
	MgCanvasDbRelations *dbrels = MG_CANVAS_DB_RELATIONS (canvas);
	MgRefBase *refbase;
	MgConf *conf = mg_base_get_conf (MG_BASE (dbrels->priv->db));
	gboolean other_tables = FALSE;

	tables = mg_database_get_tables (dbrels->priv->db);

	menu = gtk_menu_new ();
	submitem = gtk_menu_item_new_with_label (_("Add table"));
	gtk_widget_show (submitem);
	gtk_menu_append (menu, submitem);

	if (tables) {
		submenu = gtk_menu_new ();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (submitem), submenu);
		gtk_widget_show (submenu);
		
		/* add a menu item for each table not currently shown displayed.
		 * WARNING: if a MgDbTable is detroyed while that menu is displayed, it can still be selected
		 * from within the menu even though it will not do anything.
		 */
		list = tables;
		while (list) {
			if (! g_hash_table_lookup (dbrels->priv->hash_tables, list->data)) {
				mitem = gtk_menu_item_new_with_label (mg_base_get_name (MG_BASE (list->data)));
				gtk_widget_show (mitem);
				gtk_menu_append (submenu, mitem);
				
				refbase = MG_REF_BASE (mg_ref_base_new (conf));
				mg_ref_base_set_ref_object (refbase, MG_BASE (list->data));
				g_object_set_data_full (G_OBJECT (mitem), "table", refbase, g_object_unref);
				
				g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_add_table_cb), canvas);
				other_tables = TRUE;
			}
			list = g_slist_next (list);
		}
		g_slist_free (tables);
	}

	/* sub menu is incensitive if there are no more tables left to add */
	gtk_widget_set_sensitive (submitem, other_tables);

	return menu;
}

static void
popup_add_table_cb (GtkMenuItem *mitem, MgCanvasDbRelations *canvas)
{
	MgRefBase *refbase;
	MgDbTable *table;

	refbase = g_object_get_data (G_OBJECT (mitem), "table");
	table = (MgDbTable *) mg_ref_base_get_ref_object (refbase);

	if (table) {
		MgGraphItem *gitem;

		gitem = MG_GRAPH_ITEM (mg_graph_item_new (mg_base_get_conf (MG_BASE (canvas->priv->db)), MG_BASE (table)));
		mg_graph_item_set_position (gitem, MG_CANVAS (canvas)->xmouse, MG_CANVAS (canvas)->ymouse);
		mg_graph_add_item (mg_canvas_get_graph (MG_CANVAS (canvas)), gitem);
		g_object_unref (G_OBJECT (gitem));
	}
}

static void
set_pixels_per_unit (MgCanvas *canvas, gdouble n)
{
	GList *list;

	list = GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list;
	while (list) {
		if (IS_MG_CANVAS_ENTITY (list->data))
			g_object_set (G_OBJECT (list->data), "scale", n, NULL);
		list = g_list_next (list);
	}
}
