/* mg-entry-string.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-entry-string.h"
#include <libmergeant/mg-data-handler.h>
#include "gdk/gdkkeysyms.h"
#include <pango/pango.h>

/* 
 * Main static functions 
 */
static void mg_entry_string_class_init (MgEntryStringClass * class);
static void mg_entry_string_init (MgEntryString * srv);
static void mg_entry_string_dispose (GObject   * object);
static void mg_entry_string_finalize (GObject   * object);

static void mg_entry_string_set_property (GObject              *object,
					  guint                 param_id,
					  const GValue         *value,
					  GParamSpec           *pspec);
static void mg_entry_string_get_property (GObject              *object,
					  guint                 param_id,
					  GValue               *value,
					  GParamSpec           *pspec);

/* properties */
enum
{
	PROP_0,
	PROP_MULTILINE,
	PROP_EDITING_CANCELLED
};

/* GtkCellEditable interface */
static void mg_entry_string_cell_editable_init (GtkCellEditableIface *iface);
static void mg_entry_string_start_editing (GtkCellEditable *iface, GdkEvent *event);

/* virtual functions */
static GtkWidget *create_entry (MgEntryWrapper *mgwrap);
static void       real_set_value (MgEntryWrapper *mgwrap, const GdaValue *value);
static GdaValue  *real_get_value (MgEntryWrapper *mgwrap);
static void       connect_signals(MgEntryWrapper *mgwrap, GCallback callback);
static gboolean   expand_in_layout (MgEntryWrapper *mgwrap);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _MgEntryStringPrivate
{
	gboolean       multiline;
	GtkWidget     *vbox;

	GtkWidget     *entry;

	GtkTextBuffer *buffer;
	GtkWidget     *sw;
	GtkWidget     *view;
};

static void
mg_entry_string_cell_editable_init (GtkCellEditableIface *iface)
{
	iface->start_editing = mg_entry_string_start_editing;
}

guint
mg_entry_string_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgEntryStringClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_entry_string_class_init,
			NULL,
			NULL,
			sizeof (MgEntryString),
			0,
			(GInstanceInitFunc) mg_entry_string_init
		};
		
		static const GInterfaceInfo cell_editable_info = {
			(GInterfaceInitFunc) mg_entry_string_cell_editable_init,    /* interface_init */
			NULL,                                                 /* interface_finalize */
			NULL                                                  /* interface_data */
		};
		
		type = g_type_register_static (MG_ENTRY_WRAPPER_TYPE, "MgEntryString", &info, 0);
		g_type_add_interface_static (type, GTK_TYPE_CELL_EDITABLE, &cell_editable_info);
	}
	return type;
}

static void
mg_entry_string_class_init (MgEntryStringClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = mg_entry_string_dispose;
	object_class->finalize = mg_entry_string_finalize;

	MG_ENTRY_WRAPPER_CLASS (class)->create_entry = create_entry;
	MG_ENTRY_WRAPPER_CLASS (class)->real_set_value = real_set_value;
	MG_ENTRY_WRAPPER_CLASS (class)->real_get_value = real_get_value;
	MG_ENTRY_WRAPPER_CLASS (class)->connect_signals = connect_signals;
	MG_ENTRY_WRAPPER_CLASS (class)->expand_in_layout = expand_in_layout;

	/* Properties */
	object_class->set_property = mg_entry_string_set_property;
	object_class->get_property = mg_entry_string_get_property;

	g_object_class_install_property (object_class, PROP_MULTILINE,
					 g_param_spec_boolean ("multiline", NULL, NULL, FALSE, 
							       G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_EDITING_CANCELLED,
					 g_param_spec_boolean ("editing_cancelled", NULL, NULL, FALSE, G_PARAM_READABLE));
}

static void
mg_entry_string_init (MgEntryString * mg_entry_string)
{
	mg_entry_string->priv = g_new0 (MgEntryStringPrivate, 1);
	mg_entry_string->priv->multiline = FALSE;
	mg_entry_string->priv->vbox = NULL;
	mg_entry_string->priv->entry = NULL;
	mg_entry_string->priv->buffer = NULL;
	mg_entry_string->priv->view = NULL;
	mg_entry_string->priv->sw = NULL;
}

/**
 * mg_entry_string_new
 * @dh: the data handler to be used by the new widget
 * @type: the requested data type (compatible with @dh)
 *
 * Creates a new widget which is mainly a GtkEntry
 *
 * Returns: the new widget
 */
GtkWidget *
mg_entry_string_new (MgDataHandler *dh, GdaValueType type)
{
	GObject *obj;
	MgEntryString *mgstr;

	g_return_val_if_fail (dh && IS_MG_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, NULL);
	g_return_val_if_fail (mg_data_handler_accepts_gda_type (dh, type), NULL);

	obj = g_object_new (MG_ENTRY_STRING_TYPE, "handler", dh, NULL);
	mgstr = MG_ENTRY_STRING (obj);
	mg_data_entry_set_value_type (MG_DATA_ENTRY (mgstr), type);

	return GTK_WIDGET (obj);
}


static void
mg_entry_string_dispose (GObject   * object)
{
	MgEntryString *mg_entry_string;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_ENTRY_STRING (object));

	mg_entry_string = MG_ENTRY_STRING (object);
	if (mg_entry_string->priv) {

	}

	/* parent class */
	parent_class->dispose (object);
}

static void
mg_entry_string_finalize (GObject   * object)
{
	MgEntryString *mg_entry_string;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_ENTRY_STRING (object));

	mg_entry_string = MG_ENTRY_STRING (object);
	if (mg_entry_string->priv) {

		g_free (mg_entry_string->priv);
		mg_entry_string->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void
mg_entry_string_set_property (GObject              *object,
			      guint                 param_id,
			      const GValue         *value,
			      GParamSpec           *pspec)
{
	MgEntryString *mgstr;

	mgstr = MG_ENTRY_STRING (object);
	if (mgstr->priv) {
		switch (param_id) {
		case PROP_MULTILINE:
			if (g_value_get_boolean (value) != mgstr->priv->multiline) {
				mgstr->priv->multiline = g_value_get_boolean (value);
				if (mgstr->priv->multiline) {
					gtk_widget_hide (mgstr->priv->entry);
					gtk_widget_show (mgstr->priv->sw);
				}
				else {
					gtk_widget_show (mgstr->priv->entry);
					gtk_widget_hide (mgstr->priv->sw);
				}
			}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void
mg_entry_string_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	MgEntryString *mgstr;

	mgstr = MG_ENTRY_STRING (object);
	if (mgstr->priv) {
		switch (param_id) {
		case PROP_MULTILINE:
			g_value_set_boolean (value, mgstr->priv->multiline);
			break;
		case PROP_EDITING_CANCELLED:
			g_value_set_boolean (value, GTK_ENTRY (mgstr->priv->entry)->editing_canceled);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void widget_shown_cb (GtkWidget *wid, MgEntryString *mgstr);

static GtkWidget *
create_entry (MgEntryWrapper *mgwrap)
{
	GtkWidget *vbox;
	MgEntryString *mgstr;

	g_return_val_if_fail (mgwrap && IS_MG_ENTRY_STRING (mgwrap), NULL);
	mgstr = MG_ENTRY_STRING (mgwrap);
	g_return_val_if_fail (mgstr->priv, NULL);

	vbox = gtk_vbox_new (FALSE, 0);
	mgstr->priv->vbox = vbox;

	/* one line entry */
	mgstr->priv->entry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (vbox), mgstr->priv->entry, FALSE, TRUE, 0);
	g_signal_connect_after (G_OBJECT (mgstr->priv->entry), "show", 
				G_CALLBACK (widget_shown_cb), mgstr);

	/* multiline entry */
	mgstr->priv->view = gtk_text_view_new ();
	mgstr->priv->buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (mgstr->priv->view));
	mgstr->priv->sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (mgstr->priv->sw), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (mgstr->priv->sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (mgstr->priv->sw), mgstr->priv->view);
	gtk_widget_show (mgstr->priv->view);
	gtk_box_pack_start (GTK_BOX (vbox), mgstr->priv->sw, TRUE, TRUE, 0);
	g_signal_connect_after (G_OBJECT (mgstr->priv->sw), "show", 
				G_CALLBACK (widget_shown_cb), mgstr);


	/* show widgets if they need to be shown */
	gtk_widget_show (mgstr->priv->entry);
	gtk_widget_show (mgstr->priv->sw);

	return vbox;
}

static void
widget_shown_cb (GtkWidget *wid, MgEntryString *mgstr)
{
	if ((wid == mgstr->priv->entry) && mgstr->priv->multiline)
		gtk_widget_hide (mgstr->priv->entry);

	if ((wid == mgstr->priv->sw) && !mgstr->priv->multiline)
		gtk_widget_hide (mgstr->priv->sw);
}


static void
real_set_value (MgEntryWrapper *mgwrap, const GdaValue *value)
{
	MgEntryString *mgstr;
	MgDataHandler *dh;

	PangoLayout *layout;
	gchar *text;
	
	g_return_if_fail (mgwrap && IS_MG_ENTRY_STRING (mgwrap));
	mgstr = MG_ENTRY_STRING (mgwrap);
	g_return_if_fail (mgstr->priv);

	dh = mg_data_entry_get_handler (MG_DATA_ENTRY (mgwrap));

	/* do we need to go into multi line mode ? */
	text = mg_data_handler_get_str_from_value (dh, value);
	layout = gtk_widget_create_pango_layout (GTK_WIDGET (mgwrap), text);
	g_free (text);
	if (pango_layout_get_line_count (layout) > 1) 
		g_object_set (G_OBJECT (mgwrap), "multiline", TRUE, NULL);
	g_object_unref (G_OBJECT (layout));


	/* fill the single line widget */
	if (value) {
		if (gda_value_is_null (value))
			gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), "");
		else {
			gchar *str;

			str = mg_data_handler_get_str_from_value (dh, value);
			if (str) {
				gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), str);
				g_free (str);
			}
			else
				gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), "");
		}
	}
	else 
		gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), "");

	/* fill the multiline widget */
	if (value) {
		if (gda_value_is_null (value))
                        gtk_text_buffer_set_text (mgstr->priv->buffer, "", -1);
		else {
			gchar *str;

			str = mg_data_handler_get_str_from_value (dh, value);
			if (str) {
				gtk_text_buffer_set_text (mgstr->priv->buffer, str, -1);
				g_free (str);
			}
		}
	}
	else 
		gtk_text_buffer_set_text (mgstr->priv->buffer, "", -1);

}

static GdaValue *
real_get_value (MgEntryWrapper *mgwrap)
{
	GdaValue *value;
	MgEntryString *mgstr;
	MgDataHandler *dh;

	g_return_val_if_fail (mgwrap && IS_MG_ENTRY_STRING (mgwrap), NULL);
	mgstr = MG_ENTRY_STRING (mgwrap);
	g_return_val_if_fail (mgstr->priv, NULL);

	dh = mg_data_entry_get_handler (MG_DATA_ENTRY (mgwrap));
	if (! mgstr->priv->multiline) {
		const gchar *cstr;
		cstr = gtk_entry_get_text (GTK_ENTRY (mgstr->priv->entry));
		value = mg_data_handler_get_value_from_str (dh, cstr, mg_data_entry_get_value_type (MG_DATA_ENTRY (mgwrap)));
	}
	else {
		GtkTextIter start, end;
		gchar *str;
		gtk_text_buffer_get_start_iter (mgstr->priv->buffer, &start);
		gtk_text_buffer_get_end_iter (mgstr->priv->buffer, &end);
		str = gtk_text_buffer_get_text (mgstr->priv->buffer, &start, &end, FALSE);
		value = mg_data_handler_get_value_from_str (dh, str, mg_data_entry_get_value_type (MG_DATA_ENTRY (mgwrap)));
		g_free (str);
	}

	if (!value) {
		/* in case the mg_data_handler_get_value_from_str() returned an error because
		   the contents of the GtkEntry cannot be interpreted as a GdaValue */
		value = gda_value_new_null ();
	}

	return value;
}

static void
connect_signals(MgEntryWrapper *mgwrap, GCallback callback)
{
	MgEntryString *mgstr;

	g_return_if_fail (mgwrap && IS_MG_ENTRY_STRING (mgwrap));
	mgstr = MG_ENTRY_STRING (mgwrap);
	g_return_if_fail (mgstr->priv);

	g_signal_connect (G_OBJECT (mgstr->priv->entry), "changed",
			  callback, mgwrap);
	g_signal_connect (G_OBJECT (mgstr->priv->buffer), "changed",
			  callback, mgwrap);
}

static gboolean
expand_in_layout (MgEntryWrapper *mgwrap)
{
	MgEntryString *mgstr;

	g_return_val_if_fail (mgwrap && IS_MG_ENTRY_STRING (mgwrap), FALSE);
	mgstr = MG_ENTRY_STRING (mgwrap);
	g_return_val_if_fail (mgstr->priv, FALSE);

	return mgstr->priv->multiline;
}



/*
 * GtkCellEditable interface
 */
static void
gtk_cell_editable_entry_editing_done_cb (GtkEntry *entry, MgEntryString *mgstr) 
{
	gtk_cell_editable_editing_done (GTK_CELL_EDITABLE (mgstr));
}

static void
gtk_cell_editable_entry_remove_widget_cb (GtkEntry *entry, MgEntryString *mgstr) 
{
	gtk_cell_editable_remove_widget (GTK_CELL_EDITABLE (mgstr));
}

static void
mg_entry_string_start_editing (GtkCellEditable *iface, GdkEvent *event)
{
	MgEntryString *mgstr;

	g_return_if_fail (iface && IS_MG_ENTRY_STRING (iface));
	mgstr = MG_ENTRY_STRING (iface);
	g_return_if_fail (mgstr->priv);

	g_object_set (G_OBJECT (mgstr->priv->entry), "has_frame", FALSE, NULL);
	gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (mgstr->priv->view), GTK_TEXT_WINDOW_LEFT, 0);
	gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (mgstr->priv->view), GTK_TEXT_WINDOW_RIGHT, 0);
	gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (mgstr->priv->view), GTK_TEXT_WINDOW_TOP, 0);
	gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (mgstr->priv->view), GTK_TEXT_WINDOW_BOTTOM, 0);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (mgstr->priv->sw), GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER (mgstr->priv->sw), 0);

	gtk_cell_editable_start_editing (GTK_CELL_EDITABLE (mgstr->priv->entry), event);
	g_signal_connect (G_OBJECT (mgstr->priv->entry), "editing_done",
			  G_CALLBACK (gtk_cell_editable_entry_editing_done_cb), mgstr);
	g_signal_connect (G_OBJECT (mgstr->priv->entry), "remove_widget",
			  G_CALLBACK (gtk_cell_editable_entry_remove_widget_cb), mgstr);
	mg_entry_shell_refresh (MG_ENTRY_SHELL (mgstr));

	gtk_widget_grab_focus (mgstr->priv->entry);
	gtk_widget_queue_draw (GTK_WIDGET (mgstr));
}
