#!/bin/sh

export DOMI_DISK_FILE="domi_disk"
export DOMI_DISK_SIZE="4096"                    # in megabytes
export DOMI_SWAP_SIZE="256"                     # in megabytes
export DOMI_ROOT_SIZE=""
export DOMI_FSTYPE="ext2"

export DOMI_DISK_SPARSE=1


########################################################################
# helper functions

function init_root() {
        local dev="$1"
        local type="${2-ext2}"
        local label="${3-${DOMI_NAME}-root}"

        case "$type" in
                ext2)
                        /sbin/mke2fs -L "$label" -q -F "$dev" \
                                || exit 1
                        ;;
                ext3)
                        /sbin/mke2fs -L "$label" -j -q -F "$dev" \
                                || exit 1
                        ;;
                reiserfs)
                        /sbin/mkreiserfs --label "$label" -q -f "$dev" \
                                || exit 1
                        ;;
                xfs)
                        /sbin/mkfs.xfs -L "$label" -q -f "$dev" \
                                || exit 1
                        ;;
                *)
                        /sbin/mkfs -t "$type" "$dev" \
                                || exit 1
                        ;;
        esac
}

function init_swap() {
        local dev="$1"
        local label="${2-${DOMI_NAME}-swap}"

        mkswap -L "$label" $dev || exit 1
}

# handle 1024*1024 vs. 1000*1000 issue ...
function mib2mb() {
        local mib="$1"
        echo $(( $mib * 1024 * 1024 / 1000 / 1000 ))
}
# setup environment
export LANG="POSIX"
unset $(set | grep ^LC_ | cut -d= -f1)

# create work dir
export DOMI_WORK="${TMPDIR-/tmp}/${0##*/}-$$"
export DOMI_MNT="${DOMI_WORK}/mnt"
mkdir "${DOMI_WORK}"    || exit 1
mkdir "${DOMI_MNT}"     || exit 1


if test "$DOMI_DISK_SPARSE" != ""; then
         echo "setup disk (sparse file ${DOMI_DISK_FILE})"
         rm -f "${DOMI_DISK_FILE}"
         dd if="/dev/zero" of="${DOMI_DISK_FILE}" \
                 bs="$(( 1024 * 1024 ))" \
                 seek="$(( ${DOMI_DISK_SIZE} - 1 ))" \
                 count=1
 else
         echo  "setup disk (image file ${DOMI_DISK_FILE})"
         dd if="/dev/zero" of="${DOMI_DISK_FILE}" \
                 bs="$(( 1024 * 1024 ))" \
                 count="${DOMI_DISK_SIZE}"
 fi
 for i in $(seq 0 63); do
         DOMI_DISK_DEV="/dev/loop${i}"
         if losetup -a | grep -q "${DOMI_DISK_DEV}:"; then
                 continue
         fi
         losetup "${DOMI_DISK_DEV}" "${DOMI_DISK_FILE}" || exit 1
         break;
 done
 
 # partitioning
 if test "${DOMI_ROOT_SIZE}" = ""; then
         DOMI_ROOT_SIZE=$(( ${DOMI_DISK_SIZE} - ${DOMI_SWAP_SIZE} ))
 fi
 parted -s ${DOMI_DISK_DEV} mklabel msdos                >/dev/null
 ps="0";    pe="$(( $ps + ${DOMI_ROOT_SIZE} ))"
 parted -s ${DOMI_DISK_DEV} mkpart primary ext2 \
         $(mib2mb $ps) $(mib2mb $pe)                     >/dev/null
 ps="$pe";  pe="$(( $ps + ${DOMI_SWAP_SIZE} ))"
 parted -s ${DOMI_DISK_DEV} mkpart primary linux-swap \
         $(mib2mb $ps) $(mib2mb $pe)                     >/dev/null
 parted -s ${DOMI_DISK_DEV} set 1 boot on                >/dev/null
 parted -s ${DOMI_DISK_DEV} print

 # activate partitions
 DOMI_ROOT_DEV="/dev/mapper/$(basename ${DOMI_DISK_DEV})-part1"
 DOMI_SWAP_DEV="/dev/mapper/$(basename ${DOMI_DISK_DEV})-part2"
 kpartx -v -p "-part" -a ${DOMI_DISK_DEV}
 init_root $DOMI_ROOT_DEV $DOMI_FSTYPE
 init_swap $DOMI_SWAP_DEV
 
 mount  ${DOMI_ROOT_DEV} ${DOMI_MNT} || exit 1
 
 echo ${DOMI_MNT}