#ifndef MACROS_H
#define MACROS_H

#define ASSERT(x) { \
   if (!(x)) { \
      putlog(MMLOG_ERROR, "ASSERTION: %s:%d in function %s", __FILE__, __LINE__, __FUNCTION__); \
      abort(); \
   } \
}

#define ALIGN2(x, y) ( ((x) + (y) - 1) & ~((y) - 1) )

#define ROTATELEFT(i, b) ((i << b) | (i >> ((sizeof(i) * 8) - b)))

#define STRIP_CRLF(x) { \
	char *crlf_ptr; \
	for (crlf_ptr = x; *crlf_ptr; crlf_ptr++) { \
		if (*crlf_ptr == '\r' || *crlf_ptr == '\n') { \
			*crlf_ptr = '\0'; \
			break; \
		} \
	} \
};

#define FREE_AND_STRDUP(x, a) { \
	if ((x) != NULL) xfree(x); \
	(x) = ((a) != NULL) ? xstrdup(a) : NULL; \
};

#define FREE_AND_NULL(x) { \
	if ((x) != NULL) { \
		xfree((x)); \
		(x) = NULL; \
	} \
};

#define XML_LIST_LOOP(x, y) \
   while (x && (x = x->next) && x->item && strcasecmp(&x->item[2], &y[1]))

#define XML_LIST_CMP(x, y) \
   if (x && x->item && !strcasecmp(x->item, y))

#define PUSHN(x, y, n) { \
	do { \
		if (x == NULL) { \
			x = (struct STACK_T *)xmalloc(sizeof(struct STACK_T)); \
			x->prev = NULL; \
		} else { \
			x->next = (struct STACK_T *)xmalloc(sizeof(struct STACK_T)); \
			x->next->prev = x; \
			x = x->next; \
		} \
		x->next = NULL; \
		x->data = xstrndup(y, (n)); \
	} while(0); \
};
	
#define PUSH(x, y) PUSHN(x, y, ~0);


#define POP(x) { \
	do { \
		if (x != NULL) { \
			xfree(x->data); \
			if (x->prev != NULL) { \
				x = x->prev; \
				xfree(x->next); \
				x->next = NULL; \
			} else { \
				xfree(x); \
				x = NULL; \
			} \
		} \
	} while(0); \
};

/*
this macro will move a node from anywhere in one list to the end of another
*/
#define MOVENODE(t, a, x, y) { \
	do { \
		if (a->next != NULL) \
			a->next->prev = a->prev; \
		if (a->prev != NULL) \
			a->prev->next = a->next; \
		if (a == x) x = a->next; \
		a->next = NULL; \
		if (y == NULL) { \
			y = a; \
			y->prev = NULL; \
		} else { \
			t tmpnode; \
			for (tmpnode = y; tmpnode->next; tmpnode = tmpnode->next); \
			if (tmpnode->next != NULL) { \
				tmpnode->next->next = a; \
				tmpnode->next->next->prev = tmpnode->next; \
			} else { \
				tmpnode->next = a; \
				tmpnode->next->prev = tmpnode; \
			} \
		} \
	} while(0); \
};	

/*
this macro will shift a node in a linked list up or down one
*/
#define SHIFTNODE(t, x, a, d) { \
	do { \
		t tmp_node = NULL; \
		if (d == UP && a != (x)) { \
			if ((x) == a->prev) (x) = a; \
			if (a->prev->prev != NULL) a->prev->prev->next = a; \
			tmp_node = a->prev->prev; \
			a->prev->next = a->next; \
			a->prev->prev = a; \
			if (a->next != NULL) a->next->prev = a->prev; \
			a->next = a->prev; \
			a->prev = tmp_node; \
		} else if (d == DOWN && a->next != NULL) { \
			if ((x) == a) (x) = a->next; \
			if (a->next->next != NULL) a->next->next->prev = a; \
			tmp_node = a->next->next; \
			a->next->prev = a->prev; \
			a->next->next = a; \
			if (a->prev != NULL) a->prev->next = a->next; \
			a->prev = a->next; \
			a->next = tmp_node; \
		} \
	} while(0); \
};

/*
this macro moves a list node to top or bottom of list
*/
#define SETNODE(t, x, a, d) { \
	do { \
		t tmp_node = NULL; \
		if (d == TOP && a != (x)) { \
			if (a->next != NULL) a->next->prev = a->prev; \
			a->prev->next = a->next; \
			a->prev = NULL; \
			(x)->prev = a; \
			a->next = (x); \
			(x) = a; \
		} else if (d == DOWN && a->next != NULL) { \
			tmp_node = a; \
			while (tmp_node->next != NULL) tmp_node = tmp_node->next; \
			if (tmp_node != a) { \
				a->next->prev = a->prev; \
				if (a->prev != NULL) \
					a->prev->next = a->next; \
				else \
					(x) = a->next; \
				a->prev = tmp_node; \
				a->next = NULL; \
				tmp_node->next = a; \
			} \
		} \
	} while (0); \
};

static inline int atomic_read_rwlock(pthread_rwlock_t * lock, void *i)
{
        int ret;

        pthread_rwlock_rdlock(lock);
        ret = *(int *)i;
        pthread_rwlock_unlock(lock);

        return ret;
}

static inline int atomic_read_mutex(pthread_mutex_t * lock, void *i)
{
        int ret;

        pthread_mutex_lock(lock);
        ret = *(int *)i;
        pthread_mutex_unlock(lock);

        return ret;
}

#ifdef _DEBUG_MEMORY

#define MMAX 1024 * 256

void *xmalloc_(int, const char*, int);
void xfree_(void *, const char*, int);
#define xmalloc(size)  xmalloc_(size, __FILE__,__LINE__)
#define xfree(ptr)  xfree_(ptr, __FILE__,__LINE__)
#define xrealloc(ptr,newsize)  xrealloc_(ptr, newsize, __FILE__,__LINE__)
#define xstrdup(str) xstrdup_(str, __FILE__,__LINE__)
#define xstrndup(str, size) xstrndup_(str, size, __FILE__,__LINE__)
#define header_new() header_new_(__FILE__,__LINE__)
inline void * operator new(size_t size, const char *file, int line)
{
	return xmalloc_(size, file, line);
}
void* operator new(std::size_t size);
inline void operator delete(void *ptr)
{
	xfree_(ptr, NULL, 0);
}
#define xnew new(__FILE__, __LINE__)
#define xdelete delete
#define xpcre_compile(pattern, options, errptr, erroffset, tableptr) pcre_compile_(pattern, options, errptr, erroffset, tableptr, __FILE__,__LINE__) 
#define xpcre_study(code, options, errptr)  pcre_study_(code, options, errptr, __FILE__,__LINE__) 

pcre *pcre_compile_(const char *pattern, int options,
		    const char **errptr, int *erroffset,
		    const unsigned char *tableptr, const char* file, int line);
pcre_extra *pcre_study_(const pcre *code, int options,
			const char **errptr, const char* file, int line);
#else

#define xnew new
#define xdelete delete
#define xpcre_compile(pattern, options, errptr, erroffset, tableptr) pcre_compile(pattern, options, errptr, erroffset, tableptr) 
#define xpcre_study(code, options, errptr)  pcre_study(code, options, errptr) 

#endif //_DEBUG_MEMORY

#endif				/* MACROS_H */
