/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <time.h>
#include <utime.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/time.h>
#include "proto.h"

extern long gmtoffset;
extern GLOBAL *global;

/* hold lock when:
	calling:
		hash_add
		hash_remove
		clean
		filename
		disk_delete
		mem_delete
		journal_*
		*_unlocked
		select_store

	when accessing any members of a cachemap struct besides key, data, offset, and size.

	no other threads may change any cachemap members when it's open for writing.

	a thread with a cachemap open for writing may not change any member besides data without
	the mutex locked.

	cache->stores is protected with cache->lock
*/

/*
 * constructor of Cache Section
 */
CacheSection::CacheSection():
	Section("cache", MUTEX),
     enabled     (field_vec[0].int_value),
     violaterfc  (field_vec[1].int_value),
     maxmemsize  (field_vec[2].uint_value),
     memextra    (field_vec[3].uint_value),
     minsize     (field_vec[4].uint_value),
     maxsize     (field_vec[5].uint_value),
     maxwaitsize (field_vec[6].uint_value),
     prefetchwindow  (field_vec[7].int_value),
     icpport         (field_vec[8].int_value),
     icptimeout      (field_vec[9].int_value),
     sbalancemethod  (field_vec[10].int_value)

	
{
	hashsize = CACHEMAP_HASH_SIZE;
	maxmemsize = 0;
	mementries = 0;
	lastclean = 0;

	cachemap = (CACHEMAPLIST_T**)xmalloc(sizeof(struct CACHEMAPLIST_T *) * hashsize);
	memset(cachemap, 0, sizeof(struct CACHEMAPLIST_T *) * hashsize);
}

void CacheSection::update()
{
	store_list.clear();
	refresh_list.clear();

	ItemList::iterator sitem;
	for (sitem = sub_vec[0].item_list.begin(); sitem != sub_vec[0].item_list.end(); sitem++) {
		store_list.push_back(Store(*sitem));
	}

	ItemList::iterator ritem;
	for (ritem = sub_vec[1].item_list.begin(); ritem != sub_vec[1].item_list.end(); ritem++) {
		refresh_list.push_back(Refresh(*ritem));
	}

	/* Here we call stores_update */

	stores_update();
}

Store::Store(const Item& item):	
	enabled     (item.field_vec[0].int_value),
     comment     (item.field_vec[1].string_value),
     profiles    (item.field_vec[2].string_list_value),
     path        (item.field_vec[3].string_value),
     maxdisksize (item.field_vec[4].uint_value),
     diskextra   (item.field_vec[5].uint_value)

{
	journalentries = 0;
	journalfd = -1;
	diskentries = 0;
	disksize = 0;

}

Refresh::Refresh(const Item& item):
	enabled (item.field_vec[0].int_value),
	comment (item.field_vec[1].string_value),
	profiles (item.field_vec[2].string_list_value),
	cachable (item.field_vec[3].int_value),
	minage (item.field_vec[4].uint_value),
	maxage(item.field_vec[5].uint_value),
	validate(item.field_vec[6].uint_value),
	lmfactor(item.field_vec[7].uint_value)
{

}

int CacheSection::icptimeout_get() const
{
	return atomic_read(&this->icptimeout);
}

int CacheSection::icpport_get() const
{
	return atomic_read(&this->icpport);
}

void CacheSection::select_refresh(CONNECTION *connection, CACHEMAP *cachemap)
{
	RefreshList::iterator refresh;

	for (refresh = this->refresh_list.begin(); refresh != refresh_list.end(); refresh++) {
		if (refresh->enabled == FALSE)
			continue;

		if (connection != NULL && !profile_find(connection->profiles, refresh->profiles))
			continue;
		else if (connection == NULL && refresh->profiles.size() != 0)
			continue;

		if (refresh->cachable == FALSE)
			cachemap->flags |= CACHE_INVALID;

		cachemap->minage = refresh->minage;
		cachemap->maxage = refresh->maxage;
		cachemap->validate = refresh->validate;
		cachemap->lmfactor = refresh->lmfactor;

		break;
	}
}

void CacheSection::select_store(CONNECTION * connection, CACHEMAP * cachemap)
{
	unsigned int ptmp;
	size_t lowest = ~0;
	Store *selected = NULL;

	ASSERT(connection != NULL);

	StoreList::iterator stores;
	for (stores = this->store_list.begin(); stores != store_list.end(); stores++) {
		if (stores->enabled == FALSE)
			continue;

		if (!profile_find(connection->profiles, stores->profiles))
			continue;

		/* check if the entry is properly configured, so assumptions can be made later */
		if (stores->path == "" || stores->maxdisksize == 0)
			continue;

		switch (this->sbalancemethod) {
		case SBALANCE_FILLSIZE:
			if (stores->disksize < lowest) {
				lowest = stores->disksize;
				selected = &*stores;
			}
			break;
		case SBALANCE_FILLPERCENT:
			ptmp = (int) (((float) stores->disksize / (float) stores->maxdisksize) * 100.0);
			if (ptmp < lowest) {
				lowest = ptmp;
				selected = &*stores;
			}
			break;
		}

	}

	cachemap->store = selected;
}

void CacheSection::stores_update()
{
	int i;
	CACHEMAP *cachemap;
	struct CACHEMAPLIST_T *cl, *tmp;

	/* this routine is called whenever the cache disk store configuration is changed, it will resynchronize everything
	   in the cache hash table. */
	for (i = 0; i < this->hashsize; i++) {
		for (cl = this->cachemap[i]; cl; cl = tmp) {
			tmp = cl->next;
			cachemap = cl->cachemap;

			if (cachemap->store != NULL) {
				StoreList::iterator stores;
				/* we can only compare by pointer address now, since it may or may not point somewhere valid.
				   this is risky, because another store may have been allocated since at the same address... that's
				   extremely unlikely though so this should work fine. */
				for (stores = this->store_list.begin(); stores != store_list.end() && cachemap->store != &*stores; stores++);

				if (stores == store_list.end() || stores->path == "") {
					/* this cachemap's filesystem store information doesn't exist anymore */

					if (!(cachemap->flags & CACHE_MEM)) {
						hash_remove(cachemap);
						free(cachemap);
					} else {
						/* it's mapped in memory, so it may still be useful. */
						cl->cachemap->store = NULL;
						cl->cachemap->flags &= ~CACHE_DISK;
					}
				}
			}
		}
	}

	StoreList::iterator stores;
	for (stores = this->store_list.begin(); stores != store_list.end(); stores++) {
		if (stores->journalfd != -1) {
			close(stores->journalfd);
			stores->journalfd = -1;
		}

		if (stores->path == "")
			continue;

		stores->diskentries = stores->disksize = 0;

		/* re-reading the journals will reassociate each cache object in the hash table with a cache disk store. */
		journal_read(&*stores);
		journal_write(&*stores);
	}
}

CACHEMAP* CacheSection::hash_find(char *key)
{
	unsigned int k;
	struct CACHEMAPLIST_T *cachemaplist;
	CACHEMAP *cachemap = NULL;

	k = hash_key(this->hashsize, key);

	cachemaplist = this->cachemap[k];

	while (cachemaplist && strcmp(cachemaplist->cachemap->key, key))
		cachemaplist = cachemaplist->next;

	if (cachemaplist != NULL)
		cachemap = cachemaplist->cachemap;

	return cachemap;
}

int CacheSection::filename(CACHEMAP * cachemap, char *buf, int len)
{
	int x;
	unsigned char k;
	char *key = cachemap->key;

	k = hash_key(256, cachemap->key);
	x = snprintf(buf, len, "%s/%.2x/", cachemap->store->path.c_str(), k);
	for (; x < len - 1 && *key; x++, key++)
		buf[x] = (*key == '/') ? '\\' : *key;
	buf[x] = '\0';

	return k;
}

void CacheSection::hash_add(CACHEMAP * cachemap)
{
	unsigned int k;
	struct CACHEMAPLIST_T *cachemaplist;

	k = hash_key(this->hashsize, cachemap->key);

	if (this->cachemap[k] == NULL) {
		this->cachemap[k] = (CACHEMAPLIST_T*)xmalloc(sizeof(struct CACHEMAPLIST_T));
		this->cachemap[k]->cachemap = cachemap;
		this->cachemap[k]->next = this->cachemap[k]->prev = NULL;
	} else {
		for (cachemaplist = this->cachemap[k]; strcmp(cachemaplist->cachemap->key, cachemap->key) && cachemaplist->next; cachemaplist = cachemaplist->next);
		ASSERT(cachemaplist->next == NULL);

		cachemaplist->next = (CACHEMAPLIST_T*)xmalloc(sizeof(struct CACHEMAPLIST_T));
		cachemaplist->next->cachemap = cachemap;
		cachemaplist->next->prev = cachemaplist;
		cachemaplist->next->next = NULL;
	}
}

void CacheSection::hash_remove(CACHEMAP * cachemap)
{
	unsigned int k;
	struct CACHEMAPLIST_T *cachemaplist;

	k = hash_key(this->hashsize, cachemap->key);

	for (cachemaplist = this->cachemap[k]; cachemaplist && strcmp(cachemaplist->cachemap->key, cachemap->key); cachemaplist = cachemaplist->next);
	if (cachemaplist != NULL) {
		if (cachemaplist->next != NULL)
			cachemaplist->next->prev = cachemaplist->prev;
		if (cachemaplist->prev != NULL)
			cachemaplist->prev->next = cachemaplist->next;
		else
			this->cachemap[k] = cachemaplist->next;

		xfree(cachemaplist);
	} else
		ASSERT(0);
}



CACHEMAP* CacheSection::cache_open(CONNECTION * connection, char *key, char *header, int flags)
{
	int fd;
	time_t utctime;
	char buf[1024], *ptr;
	CACHEMAP *cachemap;
	struct stat st;

	mutex_lock();

	if (this->enabled == FALSE) {
		unlock();

		return NULL;
	}

	utctime = utc_time(NULL);

	cachemap = hash_find(key);

	if (cachemap != NULL) {
		if ((cachemap->flags & CACHE_PREFETCHED) && utctime - cachemap->mtime > this->prefetchwindow)
			cachemap->flags &= ~CACHE_PREFETCHED;

		if (flags & CACHE_WRITING) {
			/* this will solve a race condition where a URL gets added to the prefetch
			   queue more than once and several threads try to prefetch it at 
			   about the same time. */
			if (cachemap->refcount != 0 || (cachemap->flags & CACHE_PREFETCHED)) {
				if (cachemap->flags & CACHE_PREFETCHED)
					putlog(MMLOG_DEBUG, "attempted to recreate cached file within prefetch window");

				cachemap = NULL;

				goto out;
			}

			putlog(MMLOG_DEBUG, "recreating cache file");
			
			/* recreate existing cache file */
			invalidate_unlocked(cachemap);

			cachemap = cachemap_new(TRUE);
			select_refresh(connection, cachemap);

			cachemap->key = xstrdup(key);
			cachemap->size = cachemap->realsize = cachemap->offset = 0;
			cachemap->mtime = utctime;
			cachemap->flags |= flags;
			cachemap->refcount = 1;

			hash_add(cachemap);
		} else {
			if (cachemap->flags & CACHE_INVALID) {
				if (cachemap->flags & CACHE_INVALID)
					putlog(MMLOG_DEBUG, "cache object is invalid");

				cachemap = NULL;

				goto out;
			}

			if (cachemap->flags & CACHE_WRITING) {
				/* wait for thread writing the cache file to complete.
				   this isn't a very good way to do things, I would rather
				   be able to read from the cache as it is being written;
				   but this becomes very complex with threading, and I don't
				   think it's worth the trouble to handle rare cases like this */

				if (((cachemap->header->flags & HEADER_CL) && cachemap->header->content_length > this->maxwaitsize) || cachemap->size > this->maxwaitsize) {
					/* file is too large to wait on */
					putlog(MMLOG_DEBUG, "cache file too large to wait on");

					cachemap = NULL;

					goto out;
				}

				/* increase refcount incase thread downloading file
				   tries to invalidate it */
				cachemap->refcount++;

				cond_wait(&cachemap->writecompletecond);

				if (cachemap->flags & CACHE_INVALID) {
					invalidate_unlocked(cachemap);

					cachemap = NULL;

					goto out;
				}

				cachemap->refcount--;

				/* did we stop waiting because it was too large ? */
				if (cachemap->size > this->maxwaitsize) {
					putlog(MMLOG_DEBUG, "stopped waiting on large cache file");

					cachemap = NULL;

					goto out;
				}

				putlog(MMLOG_DEBUG, "waited for cache file that was being written");
			}

			if (!(cachemap->flags & CACHE_MEM) && cachemap->store != NULL) {
				ASSERT(cachemap->flags & CACHE_DISK);
				ASSERT(cachemap->data == NULL);
				ASSERT(cachemap->fd == -1);
				ASSERT(cachemap->refcount == 0);

				filename(cachemap, buf, sizeof(buf));

				fd = open(buf, O_RDONLY);
				if (fd != -1) {
					fstat(fd, &st);
					ptr = (char*)mmap(NULL, st.st_size, PROT_READ, MAP_SHARED, fd, 0);

					if (ptr != MAP_FAILED) {
						cachemap->mtime = st.st_mtime + gmtoffset;
						cachemap->atime = st.st_atime + gmtoffset;

						if (cachemap->size != st.st_size) {
							putlog(MMLOG_CACHE, "file meta information didn't match journaled data");

							/* compensate for unexpected change in file size */
							cachemap->store->disksize += (int) (st.st_size - cachemap->size);
							cachemap->size = cachemap->realsize = st.st_size;

							/* add journal entry with updated file information */
							journal_add(cachemap);
						} else
							cachemap->size = cachemap->realsize = st.st_size;

						cachemap->data = ptr;
						cachemap->flags |= CACHE_MEM;

						this->memsize += cachemap->size;
						this->mementries++;

#ifdef HAVE_MADVISE
						/* ask the kernel to asyncronously read the file from disk so it's cached
						   by the time we need it. */
						madvise(cachemap->data, cachemap->size, MADV_WILLNEED);
#endif				/* HAVE_MADVISE */
					}

					close(fd);
				}
			}

			if (!(cachemap->flags & CACHE_MEM)) {
				putlog(MMLOG_CACHE, "failed to open cache file: %s", cachemap->key);

				invalidate_unlocked(cachemap);
				cachemap = NULL;

				goto out;
			}

			cachemap->refcount++;
		}
	} else if (flags & CACHE_WRITING) {
		/* new cache item */
		cachemap = cachemap_new(TRUE);
		select_refresh(connection, cachemap);

		cachemap->key = xstrdup(key);
		cachemap->size = cachemap->realsize = cachemap->offset = 0;
		cachemap->mtime = utctime;
		cachemap->flags |= flags;
		cachemap->refcount = 1;

		hash_add(cachemap);
	}

	if (cachemap != NULL) {
		if (cachemap->flags & CACHE_INVALID) {
			/* refresh entry marked this uncachable */
			putlog(MMLOG_CACHE, "uncachable: %s", cachemap->key);
			goto invalidate;
		}

		cachemap->atime = utctime;

		if (!(cachemap->flags & CACHE_WRITING)) {
			if (cachemap->header == NULL) {
				/* refcount must be 1 at this point, so modifying any cachemap
				   member is safe */
				cachemap->offset = header_length(cachemap->data, cachemap->size);
				if (cachemap->offset != 0)
					cachemap->header = http_header_parse_response(cachemap->data);
			}
		} else {
			select_store(connection, cachemap);

			/* set header first or we open up a race condition since we unlock
			   in add */
			cachemap->header = http_header_parse_response(header);
			cachemap->offset = strlen(header);

			if ((cachemap->header->flags & HEADER_CL) && cachemap->header->content_length > this->maxsize) {
				putlog(MMLOG_CACHE, "file too large to cache (size: %d)", cachemap->header->content_length);

				goto invalidate;
			}

			/* NOTE: add_unlocked unlocks the mutex to relieve contention...
			   there may be race conditions calling it from here if you're not careful! */
			add_unlocked(cachemap, header, strlen(header));

			putlog(MMLOG_DEBUG, "cache header offset: %d", cachemap->offset);
		}

		if (cachemap->offset == 0 || cachemap->header == NULL || !cacheable(cachemap->header)) {
			/* this shouldn't happen, it may be a sign that the cache file was corrupted */
			putlog(MMLOG_CACHE, "corrupt header: %s", key);

			goto invalidate;
		}

		if (!(cachemap->flags & CACHE_WRITING))
			check_header(cachemap);

		if (!(flags & CACHE_OFFLINE) && (cachemap->flags & CACHE_VIOLATION) && this->violaterfc == FALSE) {
			close_unlocked(cachemap);
			cachemap = NULL;
		}

	}

      out:

	if (this->memsize > this->maxmemsize || (cachemap != NULL && cachemap->store != NULL && cachemap->store->disksize > cachemap->store->maxdisksize))
		clean();

	unlock();

	return cachemap;


	invalidate:

	invalidate_unlocked(cachemap);

	unlock();

	return NULL;
}

void CacheSection::cache_close(CACHEMAP * cachemap)
{
	mutex_lock();
	close_unlocked(cachemap);
	unlock();
}

void CacheSection::close_unlocked(CACHEMAP * cachemap)
{
	if (cachemap->flags & CACHE_WRITING) {
		cachemap->header->content_length = cachemap->size - cachemap->offset;

		if (cachemap->size != cachemap->realsize)
			resize(cachemap, 0);

		msync(cachemap->data, cachemap->realsize, MS_ASYNC);

		if (cachemap->fd != -1)
			close(cachemap->fd);

		cachemap->fd = -1;
	}

	cachemap->refcount--;

	if (cachemap->flags & CACHE_INVALID) {
		if (cachemap->refcount == 0)
			invalidate_unlocked(cachemap);
	} else if (cachemap->flags & CACHE_WRITING) {
		if (cachemap->size == 0 && cachemap->size < this->minsize) {
			if (cachemap->size < this->minsize)
				putlog(MMLOG_CACHE, "file too small");

			invalidate_unlocked(cachemap);
		} else {
			ASSERT(cachemap->flags & CACHE_NEW);

			cachemap->flags &= ~CACHE_WRITING;

			/* touch_unlocked does the journal update */
			touch_unlocked(cachemap, TRUE);

			if (cachemap->refcount != 0)
				pthread_cond_broadcast(&cachemap->writecompletecond);
		}
	} else
		touch_unlocked(cachemap, FALSE);
}

int CacheSection::add(CACHEMAP * cachemap, void *buf, size_t len)
{
	int ret;

	mutex_lock();

	ret = add_unlocked(cachemap, buf, len);

	unlock();

	return ret;
}

int CacheSection::add_unlocked(CACHEMAP * cachemap, void *buf, size_t len)
{
	int x;
	size_t ret;
	char path[1024], file[1024], *key;

	if (cachemap->flags & CACHE_INVALID)
		return -1;

	key = cachemap->key;

	if (!(cachemap->flags & CACHE_MEM)) {
		if (cachemap->store != NULL) {
			filename(cachemap, file, sizeof(file));

			x = hash_key(256, key);
			snprintf(path, sizeof(path), "%s/%.2x", cachemap->store->path.c_str(), x);

			mkdir(path, 0755);

			/* there may be queued unlinks for this file, remove them or the new cache file will be deleted. */
			unlink_list_zap(file);
			cachemap->fd = open(file, O_RDWR | O_CREAT, 0644);
			if (cachemap->fd != -1) {
				cachemap->store->diskentries++;
				cachemap->flags |= CACHE_DISK;

				journal_add(cachemap);
			} else
				cachemap->store = NULL;
		}

		ret = resize(cachemap, len);
		if (ret == 0)
			goto error;

		memcpy(cachemap->data, buf, len);
	} else {
		ret = resize(cachemap, len);
		if (ret == 0)
			goto error;

		memcpy(cachemap->data + (cachemap->size - len), buf, len);

	}


	this->memsize += len;
	if (cachemap->flags & CACHE_DISK)
		cachemap->store->disksize += len;

	if ((cachemap->store != NULL && cachemap->store->disksize > cachemap->store->maxdisksize) || this->memsize > this->maxmemsize)
		clean();

	if (this->maxsize != 0 && cachemap->size > this->maxsize) {
		putlog(MMLOG_CACHE, "max size exceeded (size: %u)", cachemap->size);

		goto error;
	}

	if (cachemap->size > this->maxwaitsize && cachemap->refcount > 1)
		pthread_cond_broadcast(&cachemap->writecompletecond);

	return len;

      error:
	/* we don't want to free it yet */
	cachemap->refcount++;
	invalidate_unlocked(cachemap);

	if (cachemap->refcount > 1)
		pthread_cond_broadcast(&cachemap->writecompletecond);

	return -1;
}

void CacheSection::scandisk(Store *store)
{
	int x;
	char buf[1024], buf2[1024], *ptr;
	DIR *base, *subdir;
	CACHEMAP *cachemap;
	struct dirent *basedirent, *subdirent;
	struct stat st;
	URL *url;

	base = opendir(store->path.c_str());
	if (base == NULL)
		return;

	putlog(MMLOG_CACHE, "scanning cache directory structure in %s", store->path.c_str());

	while ((basedirent = readdir(base)) != NULL) {
		if (!strcmp(basedirent->d_name, ".") || !strcmp(basedirent->d_name, ".."))
			continue;

		snprintf(buf, sizeof(buf), "%s/%s", store->path.c_str(), basedirent->d_name);
		subdir = opendir(buf);
		if (subdir != NULL) {
			while ((subdirent = readdir(subdir)) != NULL) {
				snprintf(buf, sizeof(buf), "%s/%s/%s", store->path.c_str(), basedirent->d_name, subdirent->d_name);
				x = stat(buf, &st);
				if (x != -1 && S_ISREG(st.st_mode)) {
					s_strncpy(buf2, subdirent->d_name, sizeof(buf2));
					for (x = 0; buf2[x]; x++)
						if (buf2[x] == '\\')
							buf2[x] = '/';


					/* this makes sure it's a valid URL */
					url = url_parse(buf2);
					if (url != NULL) {
						ptr = url_create(url);
						url_free(url);
					} else {
						putlog(MMLOG_CACHE, "%s doesn't appear to be a valid cache file", buf);
						continue;
					}

					cachemap = hash_find(ptr);
					if (cachemap != NULL) {
						/* this can happen if the unlink thread didn't reach this entry before the proxy
						   server was killed and the file was cached again, or if this directory
						   was used for another instance of the proxy. */
						putlog(MMLOG_CACHE, "duplicate of %s from %s found in %s", ptr, (cachemap->store->path != "") ? cachemap->store->path.c_str() : "(memory)", store->path.c_str());

						/* this could happen if scandisk is called twice on the same directory */
						if (cachemap->store == store) {
							xfree(ptr);

							continue;
						}

						/* keep other copy if it's in use or newer */
						if (cachemap->refcount != 0 || cachemap->mtime > st.st_mtime + gmtoffset) {
							unlink_list_add(buf);

							xfree(ptr);

							continue;
						}

						if (cachemap->flags & CACHE_MEM)
							mem_delete(cachemap);
						if (cachemap->flags & CACHE_DISK)
							disk_delete(cachemap);
					}

					if (cachemap == NULL) {
						cachemap = cachemap_new(FALSE);
						select_refresh(NULL, cachemap);

						cachemap->key = ptr;
						hash_add(cachemap);
					} else
						xfree(ptr);

					cachemap->store = store;
					cachemap->size = cachemap->realsize = st.st_size;
					cachemap->mtime = st.st_mtime + gmtoffset;
					cachemap->atime = st.st_atime + gmtoffset;
					cachemap->flags = CACHE_DISK;

					store->disksize += st.st_size;
					store->diskentries++;
				}
			}

			closedir(subdir);
		}
	}

	closedir(base);

	return;
}

void CacheSection::clean()
{
	int i;
	unsigned int memsizegoal, disksizegoal;
	time_t oldest = 0, newest = 0, threshold, increment, utctime;
	struct CACHEMAPLIST_T *cachemaplist, *tmp;
	CACHEMAP *cachemap;

	utctime = utc_time(NULL);

	if (this->lastclean != 0 && this->lastclean + CLEAN_INTERVAL > utctime)
		return;

	this->lastclean = utctime;

	putlog(MMLOG_CACHE, "cleaning cache");

	for (i = 0; i < this->hashsize; i++) {
		for (cachemaplist = this->cachemap[i]; cachemaplist; cachemaplist = cachemaplist->next) {
			if (cachemaplist->cachemap->refcount != 0)
				continue;

			if (cachemaplist->cachemap->atime > newest || newest == 0)
				newest = cachemaplist->cachemap->atime;
			if (cachemaplist->cachemap->atime < oldest || oldest == 0)
				oldest = cachemaplist->cachemap->atime;
		}
	}

	/* oops, nothing we can free */
	if (newest == 0)
		return;

	increment = (newest - oldest) / CACHE_INCREMENT_FRACTION + 1;
	threshold = oldest + increment;

	if (this->memsize > this->maxmemsize)
		memsizegoal = (this->memextra > this->maxmemsize) ? 0 : this->maxmemsize - this->memextra;
	else
		memsizegoal = this->maxmemsize;

	while (threshold <= newest + increment) {
		for (i = 0; i < this->hashsize; i++) {
			for (cachemaplist = this->cachemap[i]; cachemaplist; cachemaplist = tmp) {
				tmp = cachemaplist->next;
				cachemap = cachemaplist->cachemap;

				if (cachemap->refcount != 0)
					continue;

				if (cachemap->atime > threshold)
					continue;

				ASSERT(cachemap->flags & (CACHE_DISK | CACHE_MEM));
				ASSERT(!(cachemap->flags & CACHE_DISK) || cachemap->store != NULL);

				if (cachemap->flags & CACHE_DISK) {
					if (cachemap->store->disksize > cachemap->store->maxdisksize)
						disksizegoal = (cachemap->store->diskextra > cachemap->store->maxdisksize) ? 0 : cachemap->store->maxdisksize - cachemap->store->diskextra;
					else
						disksizegoal = cachemap->store->maxdisksize;

					if (cachemap->store->disksize > disksizegoal) {
						invalidate_unlocked(cachemap);
						continue;
					}
				}

				if ((cachemap->flags & CACHE_MEM) && this->memsize > memsizegoal) {
					if (!(cachemap->flags & CACHE_DISK))
						hash_remove(cachemap);

					mem_delete(cachemap);
				}
			}
		}

		/* keep increasing the threshold until enough memory can be free'd */
		threshold += increment;
	}
}

int cacheable(HEADER * header)
{
	if (header->type == HTTP_RESP) {
		if (header->code != 200)
			return FALSE;
	} else if (header->type == HTTP_PROXY) {
		if (strcasecmp(header->method, "GET"))
			return FALSE;
		if (header->authorization != NULL)
			return FALSE;
		if (header->username != NULL && strcasecmp(header->username, ANONLOGIN))
			return FALSE;
		if (header->range != NULL)
			return FALSE;
	} else
		return FALSE;

	return TRUE;
}

/* 
check if cache file needs validating based on request headers
*/
void CacheSection::check(HEADER * header)
{
	int x, expired = FALSE, i;
	time_t utctime = utc_time(NULL);
	char **args;
	CACHEMAP *cachemap = NULL;

	cachemap = cache_open(NULL, header->url, NULL, CACHE_OFFLINE);

	if (cachemap == NULL)
		return;

	mutex_lock();

	if (cachemap->flags & CACHE_PREFETCHED)
		goto out;

	if (cachemap->flags & CACHE_EXPIRED)
		expired = TRUE;

	if (header->cache_control != NULL) {
		putlog(MMLOG_DEBUG, "request cache-control header: %s", header->cache_control);

		args = string_break(header->cache_control, ',');

		for (x = 0; args[x]; x++) {
			if (!strcasecmp(args[x], "no-cache") || !strcasecmp(args[x], "no-store"))
				expired = TRUE;
			else if (!strncasecmp(args[x], "max-age=", 8)) {
				/* file can't be any older than this */
				i = atoi(&args[x][8]);

				if (utctime - cachemap->mtime > i)
					expired = TRUE;
			} else if (!strncasecmp(args[x], "min-fresh=", 10)) {
				/* file must be fresh for at least this long */
				i = atoi(&args[x][10]);

				if ((cachemap->etime != 0 && utctime + i > cachemap->etime) || (cachemap->ftime != 0 && cachemap->etime == 0 && utctime + i > cachemap->ftime))
					expired = TRUE;
			} else if (!strncasecmp(args[x], "max-stale=", 10)) {
				/* file may be stale for this long */
				i = atoi(&args[x][10]);

				if ((cachemap->etime != 0 && utctime - cachemap->etime <= i) || (cachemap->ftime != 0 && cachemap->etime == 0 && utctime - cachemap->ftime <= i))
					expired = FALSE;

			}

			xfree(args[x]);
		}

		xfree(args);
	}

	if (expired == TRUE)
		cachemap->flags |= CACHE_VALIDATE;

      out:

	unlock();
	cache_close(cachemap);
}

Filebuf *cache_to_filebuf(CACHEMAP * cachemap, size_t maxlen)
{
	Filebuf *ret;

	ret = xnew Filebuf();

	ret->Add(cachemap->data + cachemap->offset, (maxlen < cachemap->size - cachemap->offset) ? maxlen : cachemap->size - cachemap->offset);

	return ret;
}

void CacheSection::invalidate(CACHEMAP * cachemap)
{
	mutex_lock();

	invalidate_unlocked(cachemap);

	unlock();
}

void CacheSection::invalidate_unlocked(CACHEMAP * cachemap)
{
	if (!(cachemap->flags & CACHE_INVALID)) {
		putlog(MMLOG_CACHE, "invalidated: %s", cachemap->key);
		cachemap->flags |= CACHE_INVALID;

		journal_add(cachemap);
	}

	if (cachemap->refcount > 1) {
		cachemap->refcount--;

		if (cachemap->flags & CACHE_WRITING)
			pthread_cond_broadcast(&cachemap->writecompletecond);

		return;
	}

	if (cachemap->flags & CACHE_MEM)
		mem_delete(cachemap);
	if (cachemap->flags & CACHE_DISK)
		disk_delete(cachemap);

	hash_remove(cachemap);

	cache_free(cachemap);
}


void CacheSection::disk_delete(CACHEMAP * cachemap)
{
	char buf[1024];

	ASSERT(cachemap->flags & CACHE_DISK);
	ASSERT(!(cachemap->flags & CACHE_MEM));

	if (cachemap->store->path != "") {
		filename(cachemap, buf, sizeof(buf));

		unlink_list_add(buf);
	}

	cachemap->store->disksize -= cachemap->size;
	cachemap->store->diskentries--;

	cachemap->flags &= ~CACHE_DISK;
}

void CacheSection::mem_delete(CACHEMAP * cachemap)
{
	ASSERT((cachemap->flags & CACHE_MEM) && cachemap->data != NULL);
	ASSERT(cachemap->header != NULL);

	if (cachemap->flags & CACHE_NEW) {
		pthread_cond_destroy(&cachemap->writecompletecond);
		cachemap->flags &= ~CACHE_NEW;
	}

	http_header_free(cachemap->header);
	cachemap->header = NULL;

	munmap(cachemap->data, cachemap->realsize);
	this->memsize -= cachemap->size;
	this->mementries--;

	cachemap->flags &= ~CACHE_MEM;
	cachemap->data = NULL;
}



void CacheSection::touch(CACHEMAP * cachemap, int modified)
{
	mutex_lock();
	touch_unlocked(cachemap, modified);
	unlock();
}

void CacheSection::touch_unlocked(CACHEMAP * cachemap, int modified)
{
	time_t utctime;
	char buf[1024];
	struct utimbuf utb;

	utctime = utc_time(NULL);

	cachemap->atime = utctime;
	if (modified == TRUE)
		cachemap->mtime = utctime;

	if (cachemap->store != NULL) {
		filename(cachemap, buf, sizeof(buf));

		utb.actime = cachemap->atime - gmtoffset;
		utb.modtime = cachemap->mtime - gmtoffset;
		utime(buf, &utb);
	}

	journal_add(cachemap);
}

void CacheSection::cache_free(CACHEMAP * cachemap)
{
	if (cachemap->fd != -1)
		close(cachemap->fd);

	if (cachemap->header != NULL)
		http_header_free(cachemap->header);

	xfree(cachemap->key);
	xfree(cachemap);
}

int CacheSection::flag_set(CACHEMAP * cachemap, int flag)
{
	int ret;

	mutex_lock();
	cachemap->flags |= flag;
	ret = cachemap->flags;
	unlock();

	return ret;
}

int CacheSection::flag_unset(CACHEMAP * cachemap, int flag)
{
	int ret;

	mutex_lock();
	cachemap->flags &= ~flag;
	ret = cachemap->flags;
	unlock();

	return ret;
}

void CacheSection::check_header(CACHEMAP * cachemap)
{
	int i, x;
	struct tm t;
	char **args;
	time_t utctime = utc_time(NULL);

	cachemap->ftime = 0;
	cachemap->etime = 0;
	cachemap->lmtime = 0;
	cachemap->header->chunked = FALSE;
	cachemap->flags &= ~CACHE_EXPIRED;


	if (cachemap->header->expires != NULL) {
		putlog(MMLOG_DEBUG, "cache file expires header: %s", cachemap->header->expires);

		memset(&t, 0, sizeof(t));
		strptime(cachemap->header->expires, HTTPTIMEFORMAT, &t);
		cachemap->etime = mktime(&t);
	}

	if (cachemap->header->last_modified != NULL) {
		putlog(MMLOG_DEBUG, "cache file last-modified header: %s", cachemap->header->last_modified);

		memset(&t, 0, sizeof(t));
		strptime(cachemap->header->last_modified, HTTPTIMEFORMAT, &t);
		cachemap->lmtime = mktime(&t);
	}

	/* rfc says Cache-Control takes precedence over Expires: header */
	if (cachemap->header->cache_control != NULL) {
		putlog(MMLOG_DEBUG, "cache file cache-control header: %s", cachemap->header->cache_control);

		args = string_break(cachemap->header->cache_control, ',');

		for (x = 0; args[x]; x++) {
			if (!strncasecmp(args[x], "max-age=", 8)) {
				i = atoi(&args[x][8]);

				cachemap->etime = cachemap->mtime + i;
			} else if (!strcasecmp(args[x], "must-revalidate"))
				cachemap->flags |= CACHE_EXPIRED;
			else if (!strcasecmp(args[x], "no-cache") || !strcasecmp(args[x], "no-store"))
				cachemap->flags |= (CACHE_EXPIRED | CACHE_VIOLATION);

			xfree(args[x]);
		}

		xfree(args);
	}

	if (cachemap->lmtime != 0) {
		/* sanity check */
		if (cachemap->lmtime < cachemap->mtime) {
			/* calculate how long this file should be considered fresh based
			   on a percentage of the delta between the time it was last modified
			   and when it was downloaded or validated; this is used only 
			   if the server gives us nothing else to go on.
			 */
			cachemap->ftime = cachemap->mtime + ((cachemap->mtime - cachemap->lmtime) * cachemap->lmfactor / 100);

			putlog(MMLOG_DEBUG, "lmtime factor: %u", cachemap->ftime - cachemap->mtime);
		}
	}

	if ((cachemap->etime == 0 && cachemap->ftime == 0) || (cachemap->mtime + cachemap->maxage < utctime) || (cachemap->etime != 0 && cachemap->etime < utctime) || (cachemap->ftime != 0 && cachemap->etime == 0 && cachemap->ftime < utctime) || (cachemap->lmtime != 0 && cachemap->lmtime + cachemap->minage > utctime) || (cachemap->etime == 0 && cachemap->mtime + cachemap->validate < utctime)) {
		/* cache file has expired */
		cachemap->flags |= CACHE_EXPIRED;
	}
}

void CacheSection::validated(CACHEMAP * cachemap)
{
	mutex_lock();
	cachemap->flags &= ~CACHE_VALIDATE;
	touch_unlocked(cachemap, TRUE);
	unlock();
}

int CacheSection::invalidate_key(char *key)
{
	int ret = FALSE;
	CACHEMAP *cachemap;

	mutex_lock();

	cachemap = hash_find(key);
	if (cachemap != NULL) {
		cachemap->refcount++;
		invalidate_unlocked(cachemap);
		ret = TRUE;
	}

	unlock();

	return ret;
}

size_t CacheSection::resize(CACHEMAP * cachemap, size_t size)
{
	int x;
	void *ptr;
	size_t oldrs;

	if (size == 0 || (cachemap->size + size > cachemap->realsize)) {
		/* this routine can block under some circumstances, releasing the mutex
		   is safe. */
		unlock();

		oldrs = cachemap->realsize;

		if (size == 0)
			cachemap->realsize = cachemap->size;
		else {
			/* if the content length is known, just map the whole thing now */
			if ((cachemap->header->flags & HEADER_CL) && cachemap->header->content_length + cachemap->offset >= size)
				cachemap->realsize = cachemap->header->content_length + cachemap->offset;
			else {
				/* preallocation tends to hide some errors such as off-by-one's, so we won't do it if this is a debug
				   build. */
#ifdef _DEBUG
				cachemap->realsize = cachemap->size + size;
#else
				cachemap->realsize = ALIGN2(cachemap->size + size, CACHE_ALIGNMENT);
#endif
			}
		}

		if (cachemap->fd != -1) {
			x = ftruncate(cachemap->fd, cachemap->realsize);

			if (x == -1) {
				mutex_lock();
				goto error;
			}
		}

		if (!(cachemap->flags & CACHE_MEM))
			ptr = mmap(NULL, cachemap->realsize, PROT_WRITE | PROT_READ, (cachemap->fd != -1) ? MAP_SHARED : MAP_PRIVATE | MAP_ANON, cachemap->fd, 0);
		else
			ptr = p_mremap(cachemap->data, oldrs, cachemap->realsize, cachemap->fd);

		mutex_lock();

		if (ptr == MAP_FAILED)
			goto error;
		else {
			if (!(cachemap->flags & CACHE_MEM)) {
				this->mementries++;
				cachemap->flags |= CACHE_MEM;
			}

			cachemap->data = (char*)ptr;
		}
	}

	cachemap->size += size;

	return cachemap->size;

      error:

	return 0;
}

int CacheSection::exists(char *key)
{
	int ret = FALSE;

	mutex_lock();
	if (hash_find(key) != NULL)
		ret = TRUE;
	unlock();

	return ret;
}

void CacheSection::journal_write(Store *cstore)
{
	int i;
	char buf[256], buf2[256];
	FILE *fptr;
	CACHEMAP *cachemap;
	struct CACHEMAPLIST_T *cachemaplist;

	snprintf(buf, sizeof(buf), "%s/journal.new", cstore->path.c_str());
	fptr = fopen(buf, "w+");
	if (fptr == NULL)
		return;

	putlog(MMLOG_CACHE, "writing journal for %s", cstore->path.c_str());

	for (i = 0; i < this->hashsize; i++) {
		for (cachemaplist = this->cachemap[i]; cachemaplist; cachemaplist = cachemaplist->next) {
			cachemap = cachemaplist->cachemap;

			if (cachemap->store != NULL && cachemap->store == cstore)
				fprintf(fptr, "%d\t%s\t%lu\t%lu\t%lu\t%lu\t%lu\t%lu\t%lu\n", cachemap->flags, cachemap->key, cachemap->size, cachemap->mtime, cachemap->atime, cachemap->minage, cachemap->maxage, cachemap->validate, cachemap->lmfactor);
		}
	}

	fclose(fptr);

	snprintf(buf, sizeof(buf), "%s/journal.new", cstore->path.c_str());
	snprintf(buf2, sizeof(buf2), "%s/journal", cstore->path.c_str());

	unlink(buf2);
	rename(buf, buf2);

	return;

}

void CacheSection::journal_read(Store *cstore)
{
	int i;
	char buf[8192], **args;
	size_t size;
	time_t mtime, atime;
	FILE *fptr;
	CACHEMAP *cachemap;
	struct CACHEMAPLIST_T *cachemaplist, *tmp;

	snprintf(buf, sizeof(buf), "%s/journal", cstore->path.c_str());
	fptr = fopen(buf, "r");
	if (fptr == NULL) {
		scandisk(cstore);
		return;
	}

	putlog(MMLOG_CACHE, "reading journal for %s", cstore->path.c_str());

	/* read each journal entry and determine the last state a cache file was in */
	while (fgets(buf, sizeof(buf), fptr) != NULL) {
		args = string_break(buf, '\t');
		if (array_length(args) >= 5) {
			size = strtoul(args[2], NULL, 10);
			mtime = strtoul(args[3], NULL, 10);
			atime = strtoul(args[4], NULL, 10);

			cachemap = hash_find(args[1]);
			if (cachemap == NULL) {
				cachemap = cachemap_new(FALSE);
				cachemap->store = cstore;
				cachemap->key = xstrdup(args[1]);
				hash_add(cachemap);
			} else if (cachemap->refcount != 0)
				continue;
			else if (cachemap->store == NULL)
				cachemap->store = cstore;

			/* we have to deal with the case when the same file shows up in more than one
			   journal (i.e. it was invalidated in one cache store, then later created in another)...
			   we'll use the newest copy */
			if (cachemap->store == cstore || (cachemap->mtime <= mtime && cachemap->store != cstore)) {
				cachemap->size = size;
				cachemap->flags = atoi(args[0]);
				cachemap->mtime = mtime;
				cachemap->atime = atime;

				/* this is for backward compatability with the old journal format. */
				if (array_length(args) >= 9) {
					cachemap->minage = strtoul(args[5], NULL, 10);
					cachemap->maxage = strtoul(args[6], NULL, 10);
					cachemap->validate = strtoul(args[7], NULL, 10);
					cachemap->lmfactor = strtoul(args[8], NULL, 10);
				} else 
					select_refresh(NULL, cachemap);

				cachemap->store = cstore;
			}
		}

		array_free(args);
	}

	for (i = 0; i < this->hashsize; i++) {
		for (cachemaplist = this->cachemap[i]; cachemaplist; cachemaplist = tmp) {
			tmp = cachemaplist->next;
			cachemap = cachemaplist->cachemap;

			if (cachemap->store != cstore || cachemap->refcount != 0)
				continue;

			/* these flags don't carry over between sessions */
			cachemap->flags &= ~CACHE_SESSION_FLAGS;
			/* XXX: ugly hack so redoing journal_read on an active cache works */
			if (cachemap->data != NULL)
				cachemap->flags |= CACHE_MEM;

			if ((cachemap->flags & (CACHE_WRITING | CACHE_INVALID)) || !(cachemap->flags & CACHE_DISK)) {
				/* the cache object's final state was invalid... remove it. */
				putlog(MMLOG_DEBUG, "cache object %s final state was invalid: %d", cachemap->key, cachemap->flags);

				hash_remove(cachemap);

				if (cachemap->flags & CACHE_MEM)
					mem_delete(cachemap);

				if (cachemap->flags & CACHE_DISK) {
					/* disk_delete subtracts from the disksize
					   accounting, but it hasn't been added yet */
					cachemap->store->disksize += cachemap->size;
					cachemap->store->diskentries++;

					disk_delete(cachemap);
				}
				cache_free(cachemap);
			} else {
				cachemap->store->diskentries++;
				cachemap->store->disksize += cachemap->size;
			}
		}
	}

	fclose(fptr);
}

int CacheSection::journal_add(CACHEMAP * cachemap)
{
	int len;
	char buf[8192];

	if (cachemap->store == NULL)
		return TRUE;

	if (cachemap->store->journalfd == -1) {
		snprintf(buf, sizeof(buf), "%s/journal", cachemap->store->path.c_str());
		cachemap->store->journalfd = open(buf, O_WRONLY | O_CREAT | O_APPEND);

		if (cachemap->store->journalfd == -1)
			return -1;
	}

	len = snprintf(buf, sizeof(buf), "%d\t%s\t%u\t%lu\t%lu\t%lu\t%lu\t%lu\t%lu\n", cachemap->flags, cachemap->key, cachemap->size, cachemap->mtime, cachemap->atime, cachemap->minage, cachemap->maxage, cachemap->validate, cachemap->lmfactor);

	write(cachemap->store->journalfd, buf, len);

	cachemap->store->journalentries++;

	if (cachemap->store->journalentries >= JOURNAL_SIZE) {
		/* rewrite the journal, this will remove redundant entries */
		close(cachemap->store->journalfd);
		cachemap->store->journalfd = -1;

		journal_write(cachemap->store);
		cachemap->store->journalentries = 0;
	}

	return TRUE;
}

void CacheSection::force_clean()
{
	int i;
	CACHEMAP *cachemap;
	struct CACHEMAPLIST_T *cachemaplist, *tmp;

	mutex_lock();

	for (i = 0; i < this->hashsize; i++) {
		for (cachemaplist = this->cachemap[i]; cachemaplist; cachemaplist = tmp) {
			tmp = cachemaplist->next;

			if (cachemaplist->cachemap->refcount != 0)
				continue;

			putlog(MMLOG_DEBUG, "deleted %s from memory cache", cachemaplist->cachemap->key);
			
			cachemap = cachemaplist->cachemap;
			
			if (cachemap->flags & CACHE_MEM)
				mem_delete(cachemap);

			hash_remove(cachemap);
			xfree(cachemap);

			if (cachemaplist == NULL)
				break;
		}
	}

	unlock();
}

CACHEMAP* CacheSection::cachemap_new(bool writing)
{
	CACHEMAP *cachemap;

	cachemap = (CACHEMAP*)xmalloc(sizeof(CACHEMAP));
	cachemap->flags = 0;
	cachemap->header = NULL;
	cachemap->fd = -1;
	cachemap->refcount = 0;
	cachemap->data = NULL;
	cachemap->store = NULL;
	cachemap->minage = cachemap->maxage = 0;
	cachemap->validate = 0;
	cachemap->lmfactor = 0;

	if (writing == TRUE) {
		pthread_cond_init(&cachemap->writecompletecond, NULL);
		cachemap->flags |= CACHE_NEW;
	}

	return cachemap;
}

void unlink_list_add(char *file)
{
	pthread_mutex_lock(&global->unlink_lock);

	global->unlink_list->push_back(*(new string(file)));

	pthread_cond_signal(&global->unlink_cond);

	pthread_mutex_unlock(&global->unlink_lock);
}


void unlink_list_zap(char *file)
{
	UnlinkList::iterator item;

	pthread_mutex_lock(&global->unlink_lock);

	for (item = global->unlink_list->begin(); item != global->unlink_list->end(); item++) {
		if (!strcmp(item->c_str(), file)) {
			global->unlink_list->erase(item);
			break;
		}
	}

	pthread_mutex_unlock(&global->unlink_lock);
}


void unlink_thread()
{
	int ret;
	UnlinkList::iterator item;

	pthread_mutex_lock(&global->unlink_lock);

	while (1) {
		item = global->unlink_list->begin();
		if (item != global->unlink_list->end()) {
			ret = unlink(item->c_str());
			if (ret == 0)
				putlog(MMLOG_DEBUG, "unlinked %s", item->c_str());

			global->unlink_list->erase(item);

			/* ease up on disk activity and give other threads a chance to add
			   items to the unlink list */
			pthread_mutex_unlock(&global->unlink_lock);
			usleep(10000);
			pthread_mutex_lock(&global->unlink_lock);

			continue;
		}

		pthread_cond_wait(&global->unlink_cond, &global->unlink_lock);
	}
}
