/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "proto.h"

extern GeneralSection *general_section;

ExternalSection::ExternalSection():
     Section("external", RWLOCK),
     enabled  (field_vec[0].int_value)
{
}

void ExternalSection::update()
{
	external_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		external_list.push_back(External(*item));
	
}


External::External(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     exec     (item.field_vec[3].string_value),
     mime     (item.field_vec[4].string_value),
     size     (item.field_vec[5].int_value),
     newmime  (item.field_vec[6].string_value),
     type     (item.field_vec[7].int_value)
{
	me = (mime != "") ? reg_compile(mime.c_str(), REGFLAGS) : NULL;	
}

External::External(const External& e):
     enabled  (e.enabled),
     comment  (e.comment),
     profiles (e.profiles),
     exec     (e.exec),
     mime     (e.mime),
     size     (e.size),
     newmime  (e.newmime),
     type     (e.type)
{
	me = (mime != "") ? reg_compile(mime.c_str(), REGFLAGS) : NULL;	
}

External::~External()
{
	if (me != NULL)
		reg_free(me);	
}

bool ExternalSection::check_match(CONNECTION *connection) const
{
	const External *external;

	read_lock();
	external = find(connection);
	unlock();

	return (!!external);
}

void ExternalSection::process(CONNECTION *connection, Filebuf *filebuf) const
{
	Filebuf *filebuf2;
	const External *external;

	read_lock();

	external = find(connection);
	if (external != NULL && external->exec != "") {
		filebuf2 = external_exec(connection, external->exec.c_str(), filebuf, external->type);
		if (filebuf2 != NULL) {
			filebuf->data = filebuf2->data;
			filebuf->size = filebuf2->size;
			filebuf->realsize = filebuf2->realsize;

			filebuf2->data = NULL;
			
			xdelete filebuf2;

			if (external->newmime != "") {
				FREE_AND_NULL(connection->rheader->content_type);

				if (!strcasecmp(external->newmime.c_str(), "STDIN")) {
				/* try extracting Content-type header from external program's output */
					connection->rheader->content_type = external_getmime(filebuf);
				} else
					connection->rheader->content_type = xstrdup(external->newmime.c_str());
			}

			putlog(MMLOG_EXTERNAL, "parsed with %s", external->exec.c_str());
		} else
			putlog(MMLOG_EXTERNAL, "failed to execute external parser '%s'", external->exec.c_str());
	}

	unlock();
}


/*
return pointer to EXTERNAL list node matching the connection, if found
*/
const External* ExternalSection::find(CONNECTION * connection) const
{
	int ret;
	unsigned int maxbuffer;

	if (this->enabled == FALSE || connection->bypass & FEATURE_EXTERNAL)
		return NULL;

        maxbuffer = general_section->maxbuffer_get();

	ExternalList::const_iterator external;
	for (external = external_list.begin(); external != external_list.end(); external++) {
		if (external->enabled == FALSE)
			continue;


		if (!profile_find(connection->profiles, external->profiles))
			continue;

		if (external->me != NULL && connection->rheader->content_type != NULL) {
			ret = reg_exec(external->me, connection->rheader->content_type);
			if (ret)
				continue;
		} else if (external->me != NULL) {
			ret = reg_exec(external->me, "");
			if (ret)
				continue;
		}

                if ((external->size == -1 && connection->rheader->content_length > maxbuffer) ||
                    (external->size > 0 && connection->rheader->content_length > external->size))
                        continue;

		break;
	}

	if (external != external_list.end())
		return &*external;
	else
		return NULL;
}

/*
execute an external program, feed stdin the contents of filebuf if available, and
place program's output into a filebuf then return
*/
Filebuf *external_exec(CONNECTION * connection, const char *exe, Filebuf * filebuf, int flags)
{
	int x, ret, pid, ipipe[2], opipe[2], fd;
	unsigned int i = 0;
	char buf[BLOCKSIZE], *ptr, tmpfile[64], **args;
	Filebuf *ret_filebuf;
	struct pollfd pfd[2];
	HttpHeaderList::const_iterator header_list;
	StringPairMap::const_iterator stringpair_iter;

	putlog(MMLOG_DEBUG, "executing %s", exe);

	ret = pipe(ipipe);
	if (ret == -1)
		return NULL;

	ret = pipe(opipe);
	if (ret == -1)
		goto error1;


	if (flags == EXTERNAL_FILE) {
		general_section->read_lock();
		snprintf(tmpfile, sizeof(tmpfile), "%s/%s", (general_section->tempdir != "") ? general_section->tempdir.c_str() : "", TEMPMASK);
		general_section->unlock();

		fd = mkstemp(tmpfile);
		if (fd == -1)
			goto error2;

		while (i != filebuf->size) {
			x = write(fd, &filebuf->data[i], (filebuf->size - i < BLOCKSIZE) ? filebuf->size - i : BLOCKSIZE);
			if (x == -1)
				break;

			i += x;
		}

		close(fd);
	}

	pid = fork();
	if (pid == -1)
		goto error2;

	if (pid != 0) {
		close(ipipe[1]);
		close(opipe[0]);

		ret_filebuf = xnew Filebuf();

		pfd[0].fd = ipipe[0];
		pfd[0].events = POLLIN;

		if (flags == EXTERNAL_PIPE && filebuf != NULL) {
			pfd[1].fd = opipe[1];
			pfd[1].events = POLLOUT;
		} else {
			close(opipe[1]);
			pfd[1].events = 0;
			pfd[1].fd = -1;
		}

		fcntl(ipipe[0], F_SETFL, O_NONBLOCK);
		fcntl(opipe[1], F_SETFL, O_NONBLOCK);

		while (1) {
			ret = p_poll(pfd, 2, -1);

			if (ret > 0) {
				if (pfd[0].revents & POLLIN) {
					ret = read(ipipe[0], buf, sizeof(buf));
					if (ret > 0)
						ret_filebuf->Add(buf, ret);
					else
						goto cleanup;
				} else if (pfd[0].revents & (POLLHUP | POLLERR))
					goto cleanup;

				if (pfd[1].revents & POLLOUT) {
					ret = write(opipe[1], &filebuf->data[i], (filebuf->size - i < BLOCKSIZE) ? filebuf->size - i : BLOCKSIZE);
					if (ret > 0)
						i += ret;
					else {
						close(opipe[1]);

						goto cleanup;
					}

					if (i == filebuf->size) {
						close(opipe[1]);

						pfd[1].fd = -1;
						pfd[1].events = 0;
					}
				} else if (pfd[1].revents & (POLLHUP | POLLERR)) {
					close(opipe[1]);

					goto cleanup;
				}

			}
		}
	} else {
		close(opipe[1]);
		close(ipipe[0]);

		args = cmdline_parse(exe);
		if (flags == EXTERNAL_FILE) {
			for (i = 0; args[i]; i++);

			args = (char**)xrealloc(args, sizeof(char *) * (i + 2));
			args[i] = tmpfile;
			args[i + 1] = NULL;
		}

		p_clearenv();

		for (stringpair_iter = connection->variables.begin(); stringpair_iter != connection->variables.end(); stringpair_iter++)
			p_setenv(stringpair_iter->first.c_str(), stringpair_iter->second.c_str(), TRUE);

		if (connection->header != NULL) {
			if (connection->header != NULL) {
				for (header_list = connection->header->header.begin(); header_list != connection->header->header.end(); header_list++) {
					ptr = env_header_format("CLIENT_", header_list->type.c_str());

					p_setenv(ptr, header_list->value.c_str(), 1);

					xfree(ptr);
				}
			}

			if (connection->rheader != NULL) {
				for (header_list = connection->rheader->header.begin(); header_list != connection->rheader->header.end(); header_list++) {
					ptr = env_header_format("SERVER_", header_list->type.c_str());

					p_setenv(ptr, header_list->value.c_str(), 1);

					xfree(ptr);
				}
			}
		}

		p_setenv("IP", connection->ip, 1);
		p_setenv("INTERFACE", connection->interface, 1);

		snprintf(buf, sizeof(buf), "%d", connection->port);
		p_setenv("PORT", buf, 1);

		dup2(opipe[0], STDIN_FILENO);
		dup2(ipipe[1], STDOUT_FILENO);
		close(opipe[0]);
		close(ipipe[1]);

		execvp(args[0], args);

		close(opipe[0]);
		close(ipipe[1]);

		exit(EXIT_FAILURE);
	}

      error2:
	close(opipe[0]);
	close(opipe[1]);

      error1:
	close(ipipe[0]);
	close(ipipe[1]);

	return NULL;

      cleanup:

	close(ipipe[0]);
	waitpid(pid, &ret, 0);

	if (!WIFEXITED(ret) || WEXITSTATUS(ret)) {
		putlog(MMLOG_EXTERNAL, "external parser exited with error");

		/* return original content if process exits with an error */
		xdelete ret_filebuf;

		ret_filebuf = NULL;
	}

	if (flags == EXTERNAL_FILE)
		unlink(tmpfile);

	return ret_filebuf;
}

/*
extract Content-Type header from a filebuf and remove it
*/
char *external_getmime(Filebuf * filebuf)
{
	int len;
	char *ptr, *ret;

	if (filebuf->size < 14 || strncasecmp(filebuf->data, "Content-type: ", 14))
		return NULL;

	ptr = (char*)memchr(filebuf->data, '\n', filebuf->size);
	if (ptr == NULL)
		return NULL;

	len = ptr - &filebuf->data[14] - ((*(ptr - 1) == '\r') ? 1 : 0);

	ret = (char*)xmalloc(len + 1);

	s_strncpy(ret, &filebuf->data[14], len + 1);

	/* assume \r\n for second newline if first is \r\n */
	len += 14 + ((filebuf->data[14 + len] == '\r') ? 4 : 2);

	/* len can be lower if nothing but the header was sent */
	if (len <= filebuf->size) {
		memcpy(filebuf->data, &filebuf->data[len], filebuf->size - len);
		filebuf->Resize(filebuf->size - len);
	} else {
		FREE_AND_NULL(filebuf->data);
		filebuf->size = 0;
	}

	putlog(MMLOG_DEBUG, "external_getmime: %s", ret);

	return ret;
}

/*
make an http header suitable as an environment variable
*/
char *env_header_format(char *prefix, const char *type)
{
	int j = 0, i, x;
	char *ret;

	i = strlen(type);
	if (prefix != NULL)
		j = strlen(prefix);

	ret = (char*)xmalloc(i + j + 1);

	x = i + j;

	ret[x] = '\0';
	if (j > 0)
		memcpy(ret, prefix, j);

	for (; i >= 0; x--, i--) {
		if (type[i] == '-')
			ret[x] = '_';
		else
			ret[x] = toupper(type[i]);
	}

	return ret;
}
