/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include "proto.h"

extern char configfile[];

extern THREADLIST threads[];
extern GLOBAL *global;
extern TemplateSection *template_section;
extern PrefetchSection *prefetch_section;
extern CacheSection *cache_section;
extern FilterSection *filter_section;
extern MimeSection *mime_section;
extern GeneralSection *general_section;
extern char proxyhost[];
extern TiXmlDocument *document;
extern pthread_mutex_t documentlock;
extern time_t documentsynctime;

struct pages_t {
	char *file;
	void (*handler) (Filebuf *, CGIMap &, CONNECTION *);
};

struct calendarmap_t {
	char *name;
	char *html;
};

struct calendarmap_t calendar_months[] = {
	{"January", "<option value=\"January\" %s>January\n"},
	{"February", "<option value=\"February\" %s>February\n"},
	{"March", "<option value=\"March\" %s>March\n"},
	{"April", "<option value=\"April\" %s>April\n"},
	{"May", "<option value=\"May\" %s>May\n"},
	{"June", "<option value=\"June\" %s>June\n"},
	{"July", "<option value=\"July\" %s>July\n"},
	{"August", "<option value=\"August\" %s>August\n"},
	{"September", "<option value=\"September\" %s>September\n"},
	{"October", "<option value=\"October\" %s>October\n"},
	{"November", "<option value=\"November\" %s>November\n"},
	{"December", "<option value=\"December\" %s>December\n"},
	{NULL, NULL}
};

struct calendarmap_t calendar_weekdays[] = {
	{"Sunday", "<option value=\"Sunday\" %s>Sunday\n"},
	{"Monday", "<option value=\"Monday\" %s>Monday\n"},
	{"Tuesday", "<option value=\"Tuesday\" %s>Tuesday\n"},
	{"Wednesday", "<option value=\"Wednesday\" %s>Wednesday\n"},
	{"Thursday", "<option value=\"Thursday\" %s>Thursday\n"},
	{"Friday", "<option value=\"Friday\" %s>Friday\n"},
	{"Saturday", "<option value=\"Saturday\" %s>Saturday\n"},
	{NULL, NULL}
};

struct pages_t pages[] = {
	{"/connections", interface_page_connections},
	{"/dnscache", interface_page_dnscache},
	{"/headers", interface_page_headers},
	{"/config", interface_page_config},
	{"/save", interface_page_save},
	{"/load", interface_page_load},
	{"/log", interface_page_log},
	{"/cache", interface_page_cache},
	{"/pool", interface_page_pool},
	{"/prefetch", interface_page_prefetch},
	{"/stats", interface_page_stats},
	{NULL, NULL}
};

CGIMap::CGIMap(char *file)
{
	parse(file);
}

void CGIMap::parse(char *file)
{
	if (file != NULL) {
		int pos = 0, astart = 0, aequal = -1;
		
		putlog(MMLOG_DEBUG, "cgi arguments: %s", file);
		
		for (; file[pos]; pos++) {
			if (file[pos] == '=')
				aequal = pos;
			else if (file[pos] == '&' && aequal != -1) {
				
				char* name = url_decode(&file[astart], aequal - astart);
				char* value = url_decode(&file[aequal + 1], pos - aequal - 1);
				
				(*this)[name] = value;

				xfree(name);
				xfree(value);

				aequal = -1;
				astart = pos + 1;
			}
		}

		if (aequal != -1) {

			char* name = url_decode(&file[astart], aequal - astart);
			char* value = url_decode(&file[aequal + 1], pos - aequal - 1);
		
			(*this)[name] = value;
		
			xfree(name);
			xfree(value);
		
			aequal = -1;
			astart = pos + 1;
		}
	}
}

int interface_check_referer(CONNECTION * connection)
{
	char buf[128];

	/* ensure the request has a valid referer. If this check isn't performed, it opens up a security hole
	   where malious websites can load URL's that alter the proxy's configuration through the web interface. */
	if (connection->header->referer != NULL) {
		snprintf(buf, sizeof(buf), "http://%s", INTERFACEURL);
		if (!strncasecmp(connection->header->referer, buf, strlen(buf))) {
			if (connection->header->referer[strlen(buf)] == ':' || connection->header->referer[strlen(buf)] == '/')
				return TRUE;
		}

		if (proxyhost[0]) {
			snprintf(buf, sizeof(buf), "http://%s", proxyhost);
		} else {
			general_section->read_lock();
			snprintf(buf, sizeof(buf), "http://%s", general_section->hostname.c_str());
			general_section->unlock();
		}

		if (!strncasecmp(connection->header->referer, buf, strlen(buf))) {
			if (connection->header->referer[strlen(buf)] == ':' || connection->header->referer[strlen(buf)] == '/')
				return TRUE;
		}

		putlog(MMLOG_SECURITY, "web interface request from invalid referer (%s)", connection->header->referer);
	} else
		putlog(MMLOG_SECURITY, "web interface request without referer");


	return FALSE;
}

void interface_stylesheet(CONNECTION *connection, Filebuf *filebuf) {
	if (template_section->insert(connection, filebuf, "interface.css")) return;

	filebuf->Addf("<style type=\"text/css\">\n");
	filebuf->Addf("<!--\n");

	// Document body properties
	filebuf->Addf("body { color: black; background-color: #99AABB; }\n");

	// link properties
	filebuf->Addf("a:link { text-decoration: none; color: #0000FF; }\n");
	filebuf->Addf("a:visited { text-decoration: none; color: #0000FF; }\n");
	filebuf->Addf("a:hover, a:active { text-decoration: underline; color: #0066FF; }\n");

	// give some contrast between input boxes and background
	filebuf->Addf("input { background: #f1f1f1; }\n");

	// Top header
	filebuf->Addf(".header {border-style: solid; font-size: 24pt; text-align: center; width: 100%%; border-color: black; background-color: white;}\n");

	// Big dialog boxes
	filebuf->Addf(".bigdialog { border-spacing: 0; border-style: solid; width: 80%%; border-color: black; background-color: white;}\n");

	// Dialog boxes
	// 60% width should emphasise the sparation between lists
	filebuf->Addf(".dialog { border-spacing: 0; border-style: solid; width: 60%%; border-color: black; background-color: white;}\n");

	// Small dialogs (statistics)
	filebuf->Addf(".smalldialog { border-spacing: 0; border-style: solid; width: 40%%; border-color: black; background-color: white;}\n");

	// Dialog footer
	filebuf->Addf(".dialogfoot { background: #e0e0e0; }\n");

	// Lists (i.e. connection pool, log entries)
	filebuf->Addf(".list { border-spacing: 0; border-style: solid; width: 100%%; border-color: black; background-color: white;}\n");

	// List header
	filebuf->Addf(".listhead { background: #e0e0e0; font-weight: bold;}\n");

	// Dialog with centered text
	filebuf->Addf(".centerdialog { border-spacing: 0; border-top-style: solid; border-bottom-style: solid; border-left-style: solid; border-right-style: solid; text-align: center; width: 80%%; border-color: black; background-color: white;}\n");

	// styles for 'diff' url command output

	// lines added to original file
	filebuf->Addf(".diffadded { background: #aaffaa; }\n");
	// lines removed from original file
	filebuf->Addf(".diffremoved { background: #ffaaaa; }\n");
	// line number
	filebuf->Addf(".diffline { background: #99cccc; font-size: 16pt font-weight: bold;}\n");

	// Tooltip style
	filebuf->Addf(".tooltiptitle { color: #FFFFFF; background: #808080; text-decoration: none; cursor: Default; font-family: arial; font-weightt: bold; font-size: 8pt}\n");
	filebuf->Addf(".tooltipcontent { color: #000000; background: #C0C0C0; text-decoration: none; cursor: Default; font-family: arial; font-size: 8pt}\n");
	filebuf->Addf("#ToolTip { position:absolute; width: 300px; top: 0px; left: 0px; z-index:4; visibility:hidden;}\n");

	// Menu
	filebuf->Addf(".menu { border-top-style: solid; border-bottom-style: solid; border-left-style: solid; border-right-style: solid; text-align: center; width: 100%%; border-color: black; background-color: white;}\n");
	filebuf->Addf("-->\n");
	filebuf->Addf("</style>\n");

}

void interface_handle_request(CONNECTION * connection)
{
	int i;
	char *ptr = NULL, *file, buf[256];
	HEADER *header;
	Filebuf *filebuf;
	CGIMap args;

	if (!strncasecmp(&connection->header->file[(connection->header->type == HTTP_REQUEST) ? strlen(INTERFACEURL) + 1 : 0], "/template/", 9)) {
		template_section->send(&connection->header->file[(connection->header->type == HTTP_REQUEST) ? strlen(INTERFACEURL) + 11 : 10], connection, 200);

		return;
	}

	if (!(connection->access & ACCESS_CONFIG)) {
		template_section->send("noaccess", connection, 403);

		return;
	}

	header = header_new();

	header->type = HTTP_RESP;
	header->code = 200;

	if (connection->header->type == HTTP_REQUEST) {
		/* we wouldn't be here if connection->header->file wasn't this big */
		file = connection->header->file + strlen(INTERFACEURL) + 1;

		if (*file != '/') {
			/* send redirect to correct relative links */

			header->code = 302;
			header->flags |= HEADER_CL;
			header->content_length = 0;
			header->location = (char*)xmalloc(strlen(INTERFACEURL) + 2);
			sprintf(header->location, "%s/", INTERFACEURL);

			header_send(header, connection, CLIENT, HEADER_RESP);

			http_header_free(header);

			return;
		}
	} else
		file = connection->header->file;

	if (!strcasecmp(file, "/xml")) {
		char buf[128];
		FILE *fptr;
		struct tm timeval;

		fptr = fdopen(connection->client->fd, "r+");
		if (fptr == NULL) return;

		localtime_r(&documentsynctime, &timeval);
		strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &timeval);
		header->last_modified = xstrdup(buf);

		header->content_type = xstrdup("text/xml");

		header_send(header, connection, CLIENT, HEADER_RESP);

		pthread_mutex_lock(&documentlock);
		config_xml_build();
		document->Print(fptr, 0);
		pthread_mutex_unlock(&documentlock);

		connection->client->fd = dup(fileno(fptr));
		fclose(fptr);

		http_header_free(header);
		connection->keepalive_client = FALSE;

		return;
	} else
		header->content_type = xstrdup("text/html");


	if (interface_check_referer(connection) == TRUE) {
		if (strcasecmp(connection->header->method, "POST")) {
			/* retrieve cgi arguments from GET request */
			ptr = strchr(file, '?');
			if (ptr != NULL)
				args.parse(&ptr[1]);
		} else if (connection->postbody != NULL) {
			/* retrieve form data from POST request */
			filebuf = connection->postbody->Dup();
			filebuf->Add("", 1);

			args.parse(filebuf->data);
			xdelete filebuf;
		}
	}


	filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");

	if (proxyhost[0]) {
		s_strncpy(buf, proxyhost, sizeof(buf));
	} else {
		general_section->read_lock();
		s_strncpy(buf, general_section->hostname.c_str(), sizeof(buf));
		general_section->unlock();
	}
	filebuf->Addf("<html><head>\n");
	filebuf->Addf("<title>Middleman %s web interface for %s</title></head>\n", VERSION, buf);
	filebuf->Addf("<body onmousemove=\"overhere()\"><div id=\"ToolTip\"></div>\n");

	interface_stylesheet(connection, filebuf);
	template_section->insert(connection, filebuf, "tooltip");

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\">\n");
	filebuf->Addf("<tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"header\"><tr><td>Middleman Web Interface</td></tr></table>\n");

	filebuf->Addf("</td></tr><tr><td><br></td></tr>\n");

	filebuf->Addf("<tr><td><table class=\"menu\"><tr align=\"center\">\n");
	filebuf->Addf("<td><a href=\"connections\">Active connections</a></td>\n");
	filebuf->Addf("<td><a href=\"log\">View log entries</a></td>");
	filebuf->Addf("<td><a href=\"stats\">Statistics</a></td>\n");
	filebuf->Addf("<td><a href=\"dnscache\">DNS cache</a></td>\n");
	filebuf->Addf("<td><a href=\"headers\">Show headers</a></td></tr>\n");
	filebuf->Addf("<tr align=\"center\">\n");
	filebuf->Addf("<td><a href=\"cache\">View cache entries</a></td>");
	filebuf->Addf("<td><a href=\"pool\">Connection pool</a></td>\n");
	filebuf->Addf("<td><a href=\"prefetch\">Prefetch queue</a></td>\n");
	filebuf->Addf("<td><a href=\"save\">Save settings</a></td>\n");
	filebuf->Addf("<td><a href=\"load\">Load settings</a></td>\n");
	filebuf->Addf("</tr>\n");
	filebuf->Addf("<tr align=\"center\">\n");
	filebuf->Addf("<td colspan=\"5\"><a href=\"config\">Config</a></td>\n");
	filebuf->Addf("</tr>\n");

	filebuf->Addf("</table></td></tr>\n<tr><td><br></td></tr><tr><td>\n");

	for (i = 0; pages[i].file != NULL; i++) {
		if (!strncasecmp(file, pages[i].file, (ptr != NULL) ? ptr - file : ~0))
			pages[i].handler(filebuf, args, connection);
	}

	filebuf->Addf("</td></tr></table></body></html>\n");

	header->content_length = filebuf->size;
	header->flags |= HEADER_CL;

	header_send(header, connection, CLIENT, HEADER_RESP);

	if (strcasecmp(connection->header->method, "HEAD"))
		net_filebuf_send(filebuf, connection, CLIENT);

	http_header_free(header);
	xdelete filebuf;
}

void interface_page_connections(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	int i;

	filebuf->Addf("<table class=\"list\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td>PID</td><td>IP</td><td>Requests</td><td>Method</td><td>Host</td><td>Port</td><td>File</td></tr>\n");

	for (i = 0; i < MAXTHREADS; i++) {
		pthread_mutex_lock(&threads[i].lock);
		if (threads[i].flags == THREAD_BUSY)
			filebuf->Addf("<tr><td>%d</td><td>%s</td><td>%d</td><td>%s</td><td>%s</td><td>%d</td><td>%s</td></tr>\n", threads[i].pid, threads[i].ip, threads[i].requests, (threads[i].method != NULL) ? threads[i].method : "NONE", (threads[i].host != NULL) ? threads[i].host : "NONE", threads[i].port, (threads[i].file != NULL) ? threads[i].file : "NONE");

		pthread_mutex_unlock(&threads[i].lock);
	}

	filebuf->Addf("</table>\n");
}

void interface_page_dnscache(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	int i;
	time_t curtime = time(NULL);
	struct HASH_LIST *hash_list;

	filebuf->Addf("<table class=\"list\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td>Hostname</td><td>IP address</td><td>Age</td></tr>\n");

	pthread_mutex_lock(&global->dns_cache_lock);
	for (i = 0; i < global->dns_cache->size; i++) {
		hash_list = global->dns_cache->hash_list[i];

		for (; hash_list; hash_list = hash_list->next)
			filebuf->Addf("<tr><td>%s</td><td>%s</td><td>%d</td></tr>\n", hash_list->ref, (char *) hash_list->data, curtime - hash_list->age);
	}

	filebuf->Addf("</table>\n");

	pthread_mutex_unlock(&global->dns_cache_lock);
}

void interface_page_stats(Filebuf *filebuf, CGIMap &args, CONNECTION *connection)
{
	global->stats.Display(filebuf);
}

void interface_page_headers(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	HttpHeaderList::const_iterator header_list;

	filebuf->Addf("<tr><td><table class=\"list\">\n");
	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Unfiltered</td></tr>\n");
	filebuf->Addf("<tr class=\"listhead\"><td>Type</td><td>Value</td></tr>\n");

	for (header_list = connection->header->header.begin(); header_list != connection->header->header.end(); header_list++)
		filebuf->Addf("<tr><td>%s</td><td>%s</td></tr>\n", header_list->type.c_str(), header_list->value.c_str());

	filebuf->Addf("</table></td></tr>\n");

	filebuf->Addf("<tr><td><br></td></tr>\n");

	filebuf->Addf("<tr><td><table class=\"list\">\n");
	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Filtered</td></tr>\n");
	filebuf->Addf("<tr class=\"listhead\"><td>Type</td><td>Value</td></tr>\n");

	for (header_list = connection->header->header_filtered.begin(); header_list != connection->header->header_filtered.end(); header_list++)
		filebuf->Addf("<tr><td>%s</td><td>%s</td></tr>\n", header_list->type.c_str(), header_list->value.c_str());

	filebuf->Addf("</table></td></tr>\n");
}

void interface_page_save(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	bool ret;
	
	const string filename = args["filename"];

	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	if (filename == "") {
		filebuf->Addf("<table class=\"centerdialog\">\n");

		filebuf->Addf("<form action=\"save\" method=\"POST\">\n");
		filebuf->Addf("<tr><td>Filename</td><td><input type=\"text\" name=\"filename\" value=\"%s\" size=\"80\"></td></tr>\n", (*configfile) ? configfile : "");
		filebuf->Addf("<tr><td><br></td></tr> <tr class=\"dialogfoot\"><td colspan=\"2\" align=\"center\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
		filebuf->Addf("</form></table>\n");

		filebuf->Addf("</td></tr></table>\n");

		return;
	}

	ret = config_save(filename);

	filebuf->Addf("<table class=\"centerdialog\">\n");
	filebuf->Addf("<tr><td align=\"center\">\n");
	if (ret)
		filebuf->Addf("File saved\n");
	else
		filebuf->Addf("Saved failed\n");

	filebuf->Addf("</td></tr></table></td></tr></table>\n");
}

void interface_page_load(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	int ret, overwrite;
	
	const string filename = args["filename"];
	overwrite = (args["overwrite"] == "yes");


	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	if (filename == "") {
		filebuf->Addf("<table class=\"centerdialog\">\n");

		filebuf->Addf("<form action=\"load\" method=\"POST\">\n");
		filebuf->Addf("<tr><td width=\"10%%\">Filename</td><td><input type=\"text\" name=\"filename\" value=\"%s\" size=\"80\"></td></tr>\n", (*configfile) ? configfile : "");
		filebuf->Addf("<tr><td width=\"10%%\">Overwrite</td><td align=\"center\">Yes: <input type=\"radio\" name=\"overwrite\" value=\"yes\" checked> No: <input type=\"radio\" name=\"overwrite\" value=\"no\"></td></tr>\n");
		filebuf->Addf("<tr><td><br></td></tr> <tr class=\"dialogfoot\"><td colspan=\"2\" align=\"center\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
		filebuf->Addf("</form></table>\n");

		filebuf->Addf("</td></tr></table>\n");

		return;
	}

	ret = config_load(overwrite, filename);

	filebuf->Addf("<table class=\"centerdialog\">\n");
	filebuf->Addf("<tr><td align=\"center\">\n");

	if (ret == TRUE)
		filebuf->Addf("File loaded\n");
	else
		filebuf->Addf("Load failed\n");

	filebuf->Addf("</td></tr></table> </td></tr></table>\n");
}

void interface_page_log(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	int size = -1, clear = FALSE;
	char *ptr;
	struct LOGBUFFER_LIST *ll;
	regex_t *pe = NULL;

	const string pattern = args["pattern"];

	if (args["clear"] == "on") clear = TRUE;

	string s = args["size"];
	if (s != "") size = atoi(s.c_str());

	if (clear == TRUE)
		logbuffer_clear(global->logbuffer);
	if (size != -1)
		logbuffer_resize(global->logbuffer, size);

	pthread_rwlock_rdlock(&global->logbuffer->lock);

	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"centerdialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Regular expression match</td></tr>\n");
	filebuf->Addf("<form action=\"log\" method=\"POST\">\n");
	filebuf->Addf("<tr><td align=\"center\" colspan=\"2\"><input type=\"text\" name=\"pattern\" value=\"%s\" size=\"80\"></td></tr>\n", pattern.c_str());
	filebuf->Addf("<tr><td align=\"center\">Log buffer size: <input type=\"text\" name=\"size\" value=\"%d\" size=\"10\"></td>\n", global->logbuffer->size);
	filebuf->Addf("<td align=\"center\">Clear: <input type=\"checkbox\" name=\"clear\"></td></tr>");
	filebuf->Addf("<tr><td><br></td></tr> <tr class=\"dialogfoot\"><td align=\"center\" colspan=\"2\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
	filebuf->Addf("</form> </table></td></tr> <tr><td><br></td></tr>\n");

	filebuf->Addf("<tr><td align=\"center\">");

	filebuf->Addf("<table class=\"list\">\n");

	/* unlock to prevent deadlock incase the regexp pattern can't be compiled and it generates a log message */
	pthread_rwlock_unlock(&global->logbuffer->lock);

	if (pattern != "")
		pe = reg_compile(pattern.c_str(), REGFLAGS);

	pthread_rwlock_rdlock(&global->logbuffer->lock);

	for (ll = global->logbuffer->head; ll; ll = ll->next) {
		if (pe != NULL && reg_exec(pe, ll->msg))
			continue;

		ptr = string_to_html(ll->msg, HTML_NEWLINES);
		filebuf->Addf("<tr><td>%s</td></tr>\n", ptr);
		xfree(ptr);
	}

	if (pe != NULL)
		reg_free(pe);

	filebuf->Addf("</table></td></tr></table>\n");

	pthread_rwlock_unlock(&global->logbuffer->lock);
}

void interface_page_prefetch(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	struct PREFETCH_QUEUE *queue;
	char *ptr;
	URL *url;

	const string qadd = args["queue"];

	if (qadd != "") {
		url = url_parse(qadd.c_str());
		if (url != NULL)
			prefetch_section->queue_add(url, NULL, ~0, 1);
	}

	prefetch_section->write_lock();

	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Add URL to prefetch queue</td></tr>\n");
	filebuf->Addf("<form action=\"prefetch\" method=\"POST\">\n");
	filebuf->Addf("<tr><td align=\"center\" colspan=\"2\"><input type=\"text\" name=\"queue\" size=\"80\"></td></tr>\n");
	filebuf->Addf("<tr><td><br></td></tr> <tr><td align=\"center\" colspan=\"2\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
	filebuf->Addf("</form></table></td></tr><tr> </table></td></tr><td><br></td></tr>\n");

	filebuf->Addf("<tr><td align=\"center\">");

	filebuf->Addf("<table class=\"list\">\n");

	for (queue = prefetch_section->queue; queue; queue = queue->next) {
		ptr = url_create(queue->url);
		filebuf->Addf("<tr><td>%s</td></tr>\n", ptr);
		xfree(ptr);
	}

	filebuf->Addf("</table> </td></tr></table>\n");

	prefetch_section->unlock();
}

int xatoi(const string& str)
{
	if (str == "")
		return -1;
	else
		return atoi(str.c_str());
}

void interface_page_cache(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	int i, del = FALSE, show = FALSE;  /* delete is a key word in C++ */
	char *ptr, *ptr2, proto[16];
	string pattern;
	size_t minsize = 0, maxsize = ~0;
	int allsizes = FALSE;
	int mbday = -1, maday = -1;
	int mbyear = -1, mayear = -1;
	int mbhour = -1, mahour = -1;
	int mbminute = -1, maminute = -1;
	int abday = -1, aaday = -1;
	int abyear = -1, aayear = -1;
	int abhour = -1, aahour = -1;
	int abminute = -1, aaminute = -1;
	int mbcurtime = FALSE, abcurtime = FALSE;
	int maepoch = FALSE, aaepoch = FALSE;
	string mbmonth, mamonth;
	string abmonth, aamonth;
	regex_t *pe = NULL;
	CACHEMAP *cachemap;
	struct CACHEMAPLIST_T *cl, *tmp;
	time_t curtime, mbdate, madate, abdate, aadate;
	struct tm lt;

	curtime = time(NULL);

	del =  (args["delete"] == "on");
	show = (args["show"] == "yes");
	pattern = args["pattern"];

	mbday  = xatoi(args["mbday"]);
	maday  = xatoi(args["maday"]);
	mbyear = xatoi(args["mbyear"]);
	mayear = xatoi(args["mayear"]);
	mbhour = xatoi(args["mbhour"]);
	mahour = xatoi(args["mahour"]);
	mbminute = xatoi(args["mbminute"]);
	maminute = xatoi(args["maminute"]);
	mbmonth  = args["mbmonth"];
	mamonth  = args["mamonth"];
	
	abday  = xatoi(args["abday"]);
	aaday  = xatoi(args["aaday"]);
	abyear = xatoi(args["abyear"]);
	aayear = xatoi(args["aayear"]);
	abhour = xatoi(args["abhour"]);
	aahour = xatoi(args["aahour"]);
	abminute = xatoi(args["abminute"]);
	aaminute = xatoi(args["aaminute"]);
	abmonth  = args["abmonth"];
	aamonth  = args["aamonth"];
	
	mbcurtime = (args["mbcurtime"] == "on");
	maepoch   = (args["maepoch"] == "on");
	abcurtime = (args["abcurtime"] == "on");
	aaepoch   = (args["aaepoch"] == "on");

	string s = args["minsize"];
	if (s != "") minsize = string_to_filesize(s);

	s = args["maxsize"];
	if (s != "") maxsize = string_to_filesize(s);

	allsizes  = (args["allsizes"] == "on");
	

	localtime_r(&curtime, &lt);

	if (allsizes == TRUE) {
		minsize = 0;
		maxsize = ~0;
	}

	if (mbday == -1 || mbcurtime == TRUE)
		mbday = lt.tm_mday;
	if (maday == -1 || maepoch == TRUE)
		maday = 1;
	if (abday == -1 || abcurtime == TRUE)
		abday = lt.tm_mday;
	if (aaday == -1 || aaepoch == TRUE)
		aaday = 1;

	if (mbyear == -1 || mbcurtime == TRUE)
		mbyear = lt.tm_year + 1900;
	if (mayear == -1 || maepoch == TRUE)
		mayear = 1970;
	if (abyear == -1 || abcurtime == TRUE)
		abyear = lt.tm_year + 1900;
	if (aayear == -1 || aaepoch == TRUE)
		aayear = 1970;

	if (mbminute == -1 || mbcurtime == TRUE)
		mbminute = lt.tm_min;
	if (maminute == -1 || maepoch == TRUE)
		maminute = 0;
	if (abminute == -1 || abcurtime == TRUE)
		abminute = lt.tm_min;
	if (aaminute == -1 || aaepoch == TRUE)
		aaminute = 0;

	if (mbhour == -1 || mbcurtime == TRUE)
		mbhour = lt.tm_hour;
	if (mahour == -1 || maepoch == TRUE)
		mahour = 0;
	if (abhour == -1 || abcurtime == TRUE)
		abhour = lt.tm_hour;
	if (aahour == -1 || aaepoch == TRUE)
		aahour = 0;

	/* hope it's safe to assume lt.tm_mon <= 12 ... */
	if (mbmonth == "" || mbcurtime == TRUE)
		mbmonth = calendar_months[lt.tm_mon].name;
	if (mamonth == "" || maepoch == TRUE)
		mamonth = calendar_months[0].name;
	if (abmonth == "" || mbcurtime == TRUE)
		abmonth = calendar_months[lt.tm_mon].name;
	if (aamonth == "" || aaepoch == TRUE)
		aamonth = calendar_months[0].name;

	memset(&lt, 0, sizeof(lt));

	lt.tm_year = mbyear - 1900;
	lt.tm_mday = mbday;
	for (i = 0; calendar_months[i].name && strcasecmp(calendar_months[i].name, mbmonth.c_str()); i++);
	lt.tm_mon = i;
	lt.tm_hour = mbhour;
	lt.tm_min = mbminute;
	mbdate = mktime(&lt);

	lt.tm_year = mayear - 1900;
	lt.tm_mday = maday;
	for (i = 0; calendar_months[i].name && strcasecmp(calendar_months[i].name, mamonth.c_str()); i++);
	lt.tm_mon = i;
	lt.tm_hour = mahour;
	lt.tm_min = maminute;
	madate = mktime(&lt);

	lt.tm_year = abyear - 1900;
	lt.tm_mday = abday;
	for (i = 0; calendar_months[i].name && strcasecmp(calendar_months[i].name, abmonth.c_str()); i++);
	lt.tm_mon = i;
	lt.tm_hour = abhour;
	lt.tm_min = abminute;
	abdate = mktime(&lt);

	lt.tm_year = aayear - 1900;
	lt.tm_mday = aaday;
	for (i = 0; calendar_months[i].name && strcasecmp(calendar_months[i].name, aamonth.c_str()); i++);
	lt.tm_mon = i;
	lt.tm_hour = aahour;
	lt.tm_min = aaminute;
	aadate = mktime(&lt);

	cache_section->mutex_lock();

	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td align=\"center\" colspan=\"2\">Cache information</td></tr>\n");
	filebuf->Addf("<tr><td>Memory cache</td><td>%d entries / %u bytes / %u%% used</td></tr>\n", cache_section->mementries, cache_section->memsize, (cache_section->memsize == 0) ? 0 : (int) (((float) cache_section->memsize / (float)cache_section->maxmemsize) *100.0));
	StoreList::const_iterator cstore;
	for (cstore = cache_section->store_list.begin(); cstore != cache_section->store_list.end(); cstore++) {
		if (cstore->path == "")
			continue;
		filebuf->Addf("<tr><td>Disk cache %s</td><td>%d entries / %u bytes / %u%% used</td></tr>\n", cstore->path.c_str(), cstore->diskentries, cstore->disksize, (cstore->disksize == 0) ? 0 : (int) (((float) cstore->disksize / (float) cstore->maxdisksize) * 100.0));
	}
	filebuf->Addf("</table></td></tr><tr><td><br></td></tr> <tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Regular expression match</td></tr>\n");
	filebuf->Addf("<form action=\"cache\" method=\"POST\">\n");
	filebuf->Addf("<input type=\"hidden\" name=\"show\" value=\"yes\"\n");
	filebuf->Addf("<tr><td align=\"center\" colspan=\"2\"><input type=\"text\" name=\"pattern\" value=\"%s\" size=\"80\"></td></tr>\n", pattern.c_str());

	filebuf->Addf("<tr><td>Modified after</td>");
	filebuf->Addf("<td>Date: <select name=\"mamonth\">\n");
	for (i = 0; calendar_months[i].name; i++)
		filebuf->Addf(calendar_months[i].html, (!strcasecmp(mamonth.c_str(), calendar_months[i].name)) ? "selected" : "");
	filebuf->Addf("</select>\n");
	filebuf->Addf("<select name=\"maday\">\n");
	for (i = 1; i <= 31; i++)
		filebuf->Addf("<option value=\"%d\" %s>%d\n", i, (i == maday) ? "selected" : "", i);
	filebuf->Addf("</select>\n");
	filebuf->Addf("<input type=\"text\" name=\"mayear\" value=\"%d\" size=\"4\">\n", mayear);
	filebuf->Addf("Time: <input type=\"text\" name=\"mahour\" value=\"%.2d\" size=\"2\">:\n", mahour);
	filebuf->Addf("<input type=\"text\" name=\"maminute\" value=\"%.2d\" size=\"2\">\n", maminute);
	filebuf->Addf(" Epoch time: <input type=\"checkbox\" name=\"maepoch\" %s>", (maepoch == TRUE) ? "checked" : "");
	filebuf->Addf("</tr></td>\n");

	filebuf->Addf("<tr><td>Modified before</td>");
	filebuf->Addf("<td>Date: <select name=\"mbmonth\">\n");
	for (i = 0; calendar_months[i].name; i++)
		filebuf->Addf(calendar_months[i].html, (!strcasecmp(mbmonth.c_str(), calendar_months[i].name)) ? "selected" : "");
	filebuf->Addf("</select>\n");
	filebuf->Addf("<select name=\"mbday\">\n");
	for (i = 1; i <= 31; i++)
		filebuf->Addf("<option value=\"%d\" %s>%d\n", i, (i == mbday) ? "selected" : "", i);
	filebuf->Addf("</select>\n");
	filebuf->Addf("<input type=\"text\" name=\"mbyear\" value=\"%d\" size=\"4\">\n", mbyear);
	filebuf->Addf("Time: <input type=\"text\" name=\"mbhour\" value=\"%.2d\" size=\"2\">:\n", mbhour);
	filebuf->Addf("<input type=\"text\" name=\"mbminute\" value=\"%.2d\" size=\"2\">\n", mbminute);
	filebuf->Addf(" Current time: <input type=\"checkbox\" name=\"mbcurtime\" %s>", (mbcurtime == TRUE) ? "checked" : "");
	filebuf->Addf("</tr></td>\n");

	filebuf->Addf("<tr><td>Accessed after</td>");
	filebuf->Addf("<td>Date: <select name=\"aamonth\">\n");
	for (i = 0; calendar_months[i].name; i++)
		filebuf->Addf(calendar_months[i].html, (!strcasecmp(aamonth.c_str(), calendar_months[i].name)) ? "selected" : "");
	filebuf->Addf("</select>\n");
	filebuf->Addf("<select name=\"aaday\">\n");
	for (i = 1; i <= 31; i++)
		filebuf->Addf("<option value=\"%d\" %s>%d\n", i, (i == aaday) ? "selected" : "", i);
	filebuf->Addf("</select>\n");
	filebuf->Addf("<input type=\"text\" name=\"aayear\" value=\"%d\" size=\"4\">\n", aayear);
	filebuf->Addf("Time: <input type=\"text\" name=\"aahour\" value=\"%.2d\" size=\"2\">:\n", aahour);
	filebuf->Addf("<input type=\"text\" name=\"aaminute\" value=\"%.2d\" size=\"2\">\n", aaminute);
	filebuf->Addf(" Epoch time: <input type=\"checkbox\" name=\"aaepoch\" %s>", (aaepoch == TRUE) ? "checked" : "");
	filebuf->Addf("</tr></td>\n");

	filebuf->Addf("<tr><td>Accessed before</td>");
	filebuf->Addf("<td>Date: <select name=\"abmonth\">\n");
	for (i = 0; calendar_months[i].name; i++)
		filebuf->Addf(calendar_months[i].html, (!strcasecmp(abmonth.c_str(), calendar_months[i].name)) ? "selected" : "");
	filebuf->Addf("</select>\n");
	filebuf->Addf("<select name=\"abday\">\n");
	for (i = 1; i <= 31; i++)
		filebuf->Addf("<option value=\"%d\" %s>%d\n", i, (i == abday) ? "selected" : "", i);
	filebuf->Addf("</select>\n");
	filebuf->Addf("<input type=\"text\" name=\"abyear\" value=\"%d\" size=\"4\">\n", abyear);
	filebuf->Addf("Time: <input type=\"text\" name=\"abhour\" value=\"%.2d\" size=\"2\">:\n", abhour);
	filebuf->Addf("<input type=\"text\" name=\"abminute\" value=\"%.2d\" size=\"2\">\n", abminute);
	filebuf->Addf(" Current time: <input type=\"checkbox\" name=\"abcurtime\" %s>", (abcurtime == TRUE) ? "checked" : "");
	filebuf->Addf("</tr></td>\n");

	filebuf->Addf("<tr><td>File size</td>");
	filebuf->Addf("<td>All sizes: <input type=\"checkbox\" name=\"allsizes\" %s> \n", (allsizes == TRUE) ? "checked" : "");
	filebuf->Addf("<input type=\"textbox\" name=\"minsize\" value=\"%u\" size=\"10\">", minsize);
	filebuf->Addf(" - <input type=\"textbox\" name=\"maxsize\" value=\"%u\" size=\"10\"> bytes", maxsize);
	filebuf->Addf("</td></tr>\n");

	filebuf->Addf("<tr><td align=\"center\" colspan=\"2\">Delete matches: <input type=\"checkbox\" name=\"delete\"></td></tr>\n");
	filebuf->Addf("<tr><td><br></td></tr> <tr class=\"dialogfoot\"><td align=\"center\" colspan=\"2\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");

	filebuf->Addf("</form></table></td></tr><tr><td><br></td></tr>\n");

	filebuf->Addf("<tr><td align=\"center\">");

	filebuf->Addf("<table class=\"list\">\n");

	if (show == FALSE)
		goto out;

	filebuf->Addf("<tr class=\"listhead\"><td><b>URL</b></td><td><b>Size</b></td><td><b>Mem</b></td><td><b>Disk</b></td></tr>\n");

	if (pattern != "")
		pe = reg_compile(pattern.c_str(), REGFLAGS);

	for (i = 0; i < CACHEMAP_HASH_SIZE; i++) {
		for (cl = cache_section->cachemap[i]; cl; cl = tmp) {
			tmp = cl->next;
			cachemap = cl->cachemap;

			if (cachemap->mtime < madate || cachemap->mtime > mbdate)
				continue;
			if (cachemap->atime < aadate || cachemap->atime > abdate)
				continue;
			if (cachemap->size < minsize || cachemap->size > maxsize)
				continue;

			if (cachemap->refcount || (pe != NULL && reg_exec(pe, cachemap->key)))
				continue;

			if (del == TRUE) {
				/* cache_invalidate_unlocked assumes caller has
				   cache item open */
				cachemap->refcount++;
				cache_section->invalidate_unlocked(cachemap);
			} else {
				ptr2 = strstr(cachemap->key, "://");
				if (ptr2 == NULL)
					continue;
				if (ptr2 - cachemap->key + 1 > sizeof(proto))
					continue;

				s_strncpy(proto, cachemap->key, ptr2 - cachemap->key + 1);
				ptr2 += 3;

				ptr = string_to_html(cachemap->key, HTML_NEWLINES);

				filebuf->Addf("<tr><td><a href=\"%s://cache..%s\">%s</a></td><td align=\"right\">%u</td><td align=\"center\">%s</td><td align=\"center\">%s</td></tr>\n", proto, ptr2, ptr, cachemap->size, (cachemap->flags & CACHE_MEM) ? "yes" : "no", (cachemap->flags & CACHE_DISK) ? "yes" : "no");
				xfree(ptr);
			}

		}
	}

	if (pe != NULL)
		reg_free(pe);

      out:

	cache_section->unlock();

	filebuf->Addf("</table> </td></tr> </table>\n");
}


void interface_page_pool(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	time_t curtime;
	struct pool_t *pl;

	curtime = time(NULL);

	filebuf->Addf("<table class=\"list\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td>Protocol</td><td>Host</td><td>Port</td><td>Username</td><td>Age</td></tr>\n");

	pthread_mutex_lock(&global->pool->lock);

	for (pl = global->pool->pool_list; pl; pl = pl->next)
		filebuf->Addf("<tr><td>%s</td><td>%s</td><td>%d</td><td>%s</td><td>%u</td></tr>\n", pl->proto, pl->host, pl->port, (pl->username != NULL) ? pl->username : "(none)", curtime - pl->age);

	pthread_mutex_unlock(&global->pool->lock);

	filebuf->Addf("</table>");
}

void interface_page_config(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	string section, subsection, action, dialog;
	int id = -1;

	section = args["section"];
	subsection = args["subsection"];
	action     = args["action"];
	dialog     = args["dialog"];
	id         = xatoi(args["id"]);


	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"centerdialog\"><tr><td align=\"center\">\n");

	filebuf->Addf("<form action=\"config\" method=\"POST\">\n");
	filebuf->Addf("<select name=\"section\">\n");
	filebuf->Addf("<option value=\"\">Select a section\n");

	for (SectionMap::const_iterator iter = Section::map.begin(); iter != Section::map.end(); iter++) {
		Section& s = *iter->second;
		filebuf->Addf("<option value=\"%s\" %s>%s\n", s.name_get().c_str(), (s.name_get() == section) ? "selected" : "", s.name_get().c_str());
	}

	filebuf->Addf("</select></td>\n");
	filebuf->Addf("<td align=\"center\"><input type=\"submit\" value=\"Submit\"></form></td></tr>\n");
	filebuf->Addf("</table></td></tr>\n");
	filebuf->Addf("<tr><td><br></td></tr><tr><td align=\"center\">\n");

	StringList error_list;

	if (dialog != "") {
		Section* s = Section::map[section];
		s->interface_dialog(filebuf, args, connection, error_list);
	} else if (section != "") {
		Section* s = Section::map[section];

		if (action != "") {
			error_list = s->interface_action(filebuf, args, connection);
		}
		
		if (error_list.size() == 0) {
			s->interface(filebuf, args, connection, error_list);
		} else {
			if (subsection == "") 
				s->interface(filebuf, args, connection, error_list);
			else
				s->interface_dialog(filebuf, args, connection, error_list);
		}
	} 

	filebuf->Addf("</td></tr></table>\n");
}

void score_show(CONNECTION * connection, int score)
{
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Keyword score for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td align=\"center\">Keyword score for %s%s</td></tr>\n", connection->header->host, connection->header->file);
	filebuf->Addf("<tr><td align=\"center\">%d</td></tr>\n", score);
	filebuf->Addf("</table></td></tr></table></body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void cache_show(CONNECTION * connection)
{
	char *ptr;
	Filebuf *filebuf;
	CACHEMAP *cachemap;
	struct tm t;
	char buf[128];

	filebuf = xnew Filebuf();

	cachemap = cache_section->cache_open(NULL, connection->header->url, NULL, CACHE_OFFLINE);

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Cache information for %s://%s:%d%s</title></head>", connection->header->proto, connection->header->host, connection->header->port, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");

	filebuf->Addf("<table class=\"dialog\">\n");

	filebuf->Addf("<tr class=\"listhead\"><td align=\"center\" colspan=\"2\">Cache information for %s://%s:%d%s</td></tr>\n", connection->header->proto, connection->header->host, connection->header->port, connection->header->file);

	cache_section->mutex_lock();

	if (cachemap == NULL)
		filebuf->Addf("<tr><td align=\"center\" colspan=\"2\">Nothing found</td></tr>\n");
	else {
		if (cachemap->store != NULL)
			filebuf->Addf("<tr><td>Path</td><td>%s</td></tr>\n", cachemap->store->path.c_str());

		filebuf->Addf("<tr><td>Size</td><td>%d bytes</td></tr>\n", cachemap->size);

		localtime_r(&cachemap->mtime, &t);
		strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &t);

		filebuf->Addf("<tr><td>Retrieved</td><td>%s</td></tr>\n", buf);

		if (cachemap->lmtime != 0) {
			localtime_r(&cachemap->lmtime, &t);
			strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &t);
			filebuf->Addf("<tr><td>Last modified</td><td>%s</td></tr>\n", buf);
		}

		if (cachemap->etime != 0) {
			localtime_r(&cachemap->etime, &t);
			strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &t);
			filebuf->Addf("<tr><td>Expires</td><td>%s</td></tr>\n", buf);
		}

		if (cachemap->ftime != 0) {
			localtime_r(&cachemap->ftime, &t);
			strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &t);
			filebuf->Addf("<tr><td>Fresh until</td><td>%s</td></tr>\n", buf);
		}

		filebuf->Addf("<tr><td>Minimum age</td><td>%lu</td></tr>\n", cachemap->minage);
		filebuf->Addf("<tr><td>Maximum age</td><td>%lu</td></tr>\n", cachemap->maxage);
		filebuf->Addf("<tr><td>Validate age</td><td>%lu</td></tr>\n", cachemap->validate);
		filebuf->Addf("<tr><td>Last-Modified time factor</td><td>%d</td></tr>\n", cachemap->lmfactor);

		ptr = string_to_html(cachemap->header->raw, 0);
		filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">Header\n</td></tr><tr class=\"dialog\"><td colspan=\"2\"><pre>%s</pre></td></tr>\n", ptr);
		xfree(ptr);
	}

	filebuf->Addf("</table></td></tr> </table></body></html>\n");

	cache_section->unlock();

	if (cachemap != NULL)
		cache_section->cache_close(cachemap);

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}


void show_filebuf_diff(CONNECTION * connection, Filebuf * orig, Filebuf * newfb)
{
	int oline = 0, nline = 0, matches;
	char *ptr;
	Filebuf *filebuf;
	struct STACK_T *ostack = NULL, *ostackstart = NULL, *nstack = NULL, *nstackstart = NULL, *otmpstack = NULL, *ntmpstack = NULL, *tmpstack, *tmpstack2;

	filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Rewrite diff for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table class=\"list\">\n");
	/* load both Filebuf's into a linked list where each node is a seperate line */
	ostackstart = orig->to_stack();
	nstackstart = newfb->to_stack();

	ostack = ostackstart;
	nstack = nstackstart;

	while (ostack != NULL && nstack != NULL) {
		for (; ostack != NULL && nstack != NULL && !strcmp(ostack->data, nstack->data); ostack = ostack->next, nstack = nstack->next, oline++, nline++);
		/* no differences */
		if (ostack == NULL && nstack == NULL)
			break;

		/* find end of changed area by searching for first DIFF_HUNK lines which are the same in both lists */
		for (matches = 0, otmpstack = ostack, ntmpstack = NULL; otmpstack != NULL; otmpstack = otmpstack->next) {
			for (ntmpstack = nstack; ntmpstack != NULL; ntmpstack = ntmpstack->next) {
				for (matches = 0, tmpstack = otmpstack, tmpstack2 = ntmpstack; tmpstack != NULL && tmpstack2 != NULL && matches < DIFF_HUNK && !strcmp(tmpstack->data, tmpstack2->data); tmpstack = tmpstack->next, tmpstack2 = tmpstack2->next, matches++);
				if (matches >= DIFF_HUNK)
					break;
			}
			if (matches >= DIFF_HUNK)
				break;
		}

		filebuf->Addf("<tr class=\"diffline\"><td>Line %d</td></tr>\n", oline);

		filebuf->Addf("<tr class=\"diffremoved\"><td><pre>\n");
		for (; ostack != NULL && (ostack != otmpstack || otmpstack == NULL); ostack = ostack->next, oline++) {
			/* show lines removed from original file */
			ptr = string_to_html(ostack->data, 0);
			filebuf->Addf("%s\n", ptr);
			xfree(ptr);
		}
		filebuf->Addf("</pre></td></tr>\n");

		filebuf->Addf("<tr class=\"diffadded\"><td><pre>\n");
		for (; nstack != NULL && (nstack != ntmpstack || ntmpstack == NULL); nstack = nstack->next, nline++) {
			/* show lines added to original file */
			ptr = string_to_html(nstack->data, 0);
			filebuf->Addf("%s\n", ptr);
			xfree(ptr);
		}
		filebuf->Addf("</pre></td></tr>\n");

	}

	/* free up the linked lists */
	stack_free(ostackstart);
	stack_free(nstackstart);

	filebuf->Addf("</table></body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_dirlist(CONNECTION * connection, struct dir_t *dir)
{
	char *ptr, buf[128];
	Filebuf *filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Directory listing for ftp://%s:%d%s</title></head>", connection->header->host, connection->header->port, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" width=\"100%%\">\n");
	filebuf->Addf("<tr><td align=\"center\"><table class=\"header\">\n");

	ptr = path_to_url(connection->header->file);
	filebuf->Addf("<tr><td align=\"center\"><font size=\"5\">%s</font></td></tr></table></td></tr><tr><td><br></td></tr>\n", ptr);
	xfree(ptr);

	filebuf->Addf("<tr><td align=\"center\"><table class=\"list\">\n");
	filebuf->Addf("<tr class=\"listhead\">");
	filebuf->Addf("<td></td>\n");
	filebuf->Addf("<td align=\"center\">");
	filebuf->Addf("<a href=\"?order=descending&field=name\"><img src=\"http://%s/template/uparrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("<a href=\"?order=ascending&field=name\"><img src=\"http://%s/template/downarrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("</td>");
	filebuf->Addf("<td align=\"center\">");
	filebuf->Addf("<a href=\"?order=descending&field=size\"><img src=\"http://%s/template/uparrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("<a href=\"?order=ascending&field=size\"><img src=\"http://%s/template/downarrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("</td>");
	filebuf->Addf("<td align=\"center\">");
	filebuf->Addf("<a href=\"?order=descending&field=date\"><img src=\"http://%s/template/uparrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("<a href=\"?order=ascending&field=date\"><img src=\"http://%s/template/downarrow\" border=\"0\"></a>", INTERFACEURL);
	filebuf->Addf("</td>");
	filebuf->Addf("</tr>");
	filebuf->Addf("<form action=\"%s\" method=\"GET\">\n", connection->header->file);

	for (; dir; dir = dir->next) {
		filebuf->Addf("<tr>");
		filebuf->Addf("<td align=\"center\"><input type=\"checkbox\" name=\"filename\" value=\"%s\"></td>\n", dir->name);

		ptr = string_to_html(dir->name, 0);
		filebuf->Addf("<td><a href=\"%s\">%s</a></td>", ptr, ptr);
		xfree(ptr);

		if (dir->isdir == TRUE)
			filebuf->Addf("<td>DIR</td>");
		else {
			ptr = filesize(dir->size);
			filebuf->Addf("<td>%s</td>", ptr);
			xfree(ptr);
		}

		strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &dir->time);
		filebuf->Addf("<td>%s</td>", buf);
		filebuf->Addf("</tr>\n");
	}

	filebuf->Addf("<tr><td align=\"center\" colspan=\"4\">Delete: <input type=\"radio\" name=\"action\" value=\"delete\"> Rename: <input type=\"radio\" name=\"action\" value=\"rename\"> Mkdir: <input type=\"radio\" name=\"action\" value=\"mkdir\"> Rmdir: <input type=\"radio\" name=\"action\" value=\"rmdir\"> Regexp match: <input type=\"radio\" name=\"action\" value=\"rematch\"> Raw command: <input type=\"radio\" name=\"action\" value=\"raw\"></td></tr>\n");
	filebuf->Addf("<tr><td align=\"center\" colspan=\"4\">Argument: <input type=\"text\" name=\"argument\" size=\"80\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
	filebuf->Addf("</form>\n");
	filebuf->Addf("</td></tr></table></table></body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_headers(CONNECTION * connection)
{
	char *ptr;
	Filebuf *filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>HTTP headers to/from %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table class=\"dialog\">\n");
	filebuf->Addf("<tr class=\"listhead\"><td align=\"center\">Client header</td></tr>\n");

	ptr = string_to_html(connection->header->raw, 0);
	filebuf->Addf("<tr><td><pre>%s</pre></td></tr>\n", ptr);
	xfree(ptr);

	filebuf->Addf("<tr class=\"listhead\"><td align=\"center\">Server header</td></tr>\n");
	ptr = string_to_html(connection->rheader->raw, 0);
	filebuf->Addf("<tr><td><pre>%s</pre></td></tr>\n", ptr);
	xfree(ptr);

	filebuf->Addf("</table></body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_filebuf(CONNECTION * connection, Filebuf * fb)
{
	char *ptr;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();
	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Raw contents of %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	fb->Add("", 1);

	ptr = string_to_html(fb->data, 0);
	filebuf->Addf("<pre>\n");
	filebuf->Add(ptr, strlen(ptr));
	filebuf->Addf("</pre>\n");
	xfree(ptr);

	fb->Resize(fb->size - 1);
	filebuf->Addf("</body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_profiles(CONNECTION * connection)
{
	Filebuf *filebuf;

	filebuf = xnew Filebuf();
	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Active profiles for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");
	filebuf->Addf("<table class=\"list\">\n");

	StringList::const_iterator profile;
	for (profile = connection->profiles.begin(); profile != connection->profiles.end(); profile++)
		filebuf->Addf("<tr><td>%s</td></tr>\n", profile->c_str());

	filebuf->Addf("</table> </td></tr></table> </body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_proxytest(CONNECTION * connection)
{
	char *ptr;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>Proxy test for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");
	filebuf->Addf("<table class=\"centerdialog\">\n");

	if (connection->proxy_type != PROXY_NORMAL)
		filebuf->Addf("<tr><td>Not forwarding through HTTP proxy</td></tr>\n");
	else {
		proxy_test(connection);

		if (connection->rheader == NULL)
			filebuf->Addf("<tr><td>%s:%d failed to connect back</td></tr>\n", connection->proxy_host, connection->proxy_port);
		else {
			filebuf->Addf("<tr><td>Headers received from %s:%d</td></tr>\n", connection->proxy_host, connection->proxy_port);
			ptr = string_to_html(connection->rheader->raw, 0);
			filebuf->Addf("<tr><td><pre>%s</pre></td></tr>\n", ptr);
			xfree(ptr);
		}
	}

	filebuf->Addf("</table> </td></tr></table></body></html>\n");

	interface_send_response(connection, filebuf);

	http_header_free(connection->rheader);
	xdelete filebuf;
}

void show_htmltree(CONNECTION * connection)
{
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
	filebuf->Addf("<html><head><title>HTML tree for %s%s</title></head>", connection->header->host, connection->header->file);
	filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");
	filebuf->Addf("<table class=\"list\">\n");


	show_htmltree_node(filebuf, connection->htmlstream->tree, 0);

	filebuf->Addf("</table> </td></tr></table> </body></html>\n");

	interface_send_response(connection, filebuf);
	xdelete filebuf;
}

void show_htmltree_node(Filebuf * filebuf, struct htmlstream_node *node, int indent)
{
	char *ptr;
	struct htmlstream_node *n;
	struct htmlstream_tag_property *p;

	for (n = node; n; n = n->next) {
		filebuf->Addf("<tr><td>", indent);
		filebuf->Addf("<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">");

		filebuf->Addf("<tr><td width=\"%d\"></td><td>", indent * 36);

		if (n->tag != NULL) {
			filebuf->Addf("%s ", n->tag->name);

			for (p = n->tag->properties; p; p = p->next)
				filebuf->Addf("%s=\"%s\" ", p->name, p->value);
		} else if (n->text != NULL) {
			ptr = string_to_html(n->text, HTML_NEWLINES | HTML_SPACES);
			filebuf->Addf("\"%s\"", ptr);
			xfree(ptr);
		}

		filebuf->Addf("</td></tr></table></td></tr>\n");

		if (n->up != NULL)
			show_htmltree_node(filebuf, n->up, indent + 1);
	}
}

void interface_send_response(CONNECTION *connection, Filebuf *filebuf) {
	HEADER *header;

	header = header_new();
	header->type = HTTP_RESP;
	header->code = 200;
	header->content_type = xstrdup("text/html");
	header->content_length = filebuf->size;
	header->flags |= HEADER_CL;

	header_send(header, connection, CLIENT, HEADER_RESP);
	net_filebuf_send(filebuf, connection, CLIENT);

	http_header_free(header);
}

