/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include "proto.h"

extern TemplateSection *template_section;

MimeSection::MimeSection():
     Section("mime", RWLOCK),
     enabled  (field_vec[0].int_value),
     policy   (field_vec[1].int_value),
     dtempl   (field_vec[2].string_value)
{
}

void MimeSection::update()
{
	allow_list.clear();
	deny_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		allow_list.push_back(Mime(*item));
	
	for (item = sub_vec[1].item_list.begin(); item != sub_vec[1].item_list.end(); item++)
		deny_list.push_back(Mime(*item));
	
}

/* constructor for empty element */
Mime::Mime():
     enabled  (TRUE),
     comment  (""),
     profiles (""),
     host     (""),
     file     (""),
     mime     (""),
     templ    ("")
{
}



Mime::Mime(const Item& item):
	Encapsulator(item),
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     host     (item.field_vec[3].string_value),
     file     (item.field_vec[4].string_value),
     mime     (item.field_vec[5].string_value),
     templ    (item.field_vec[6].string_value)
{
	he = (host != "") ? reg_compile(host.c_str(), REGFLAGS) : NULL;
	me = (mime != "") ? reg_compile(mime.c_str(), REGFLAGS) : NULL;
	fe = (file != "") ? reg_compile(file.c_str(), REGFLAGS) : NULL;
}

Mime::Mime(const Mime& m):
	Encapsulator(m.item),
     enabled  (m.enabled),
     comment  (m.comment),
     profiles (m.profiles),
     host     (m.host),
     file     (m.file),
     mime     (m.mime),
     templ    (m.templ)
{
	he = (host != "") ? reg_compile(host.c_str(), REGFLAGS) : NULL;
	me = (mime != "") ? reg_compile(mime.c_str(), REGFLAGS) : NULL;
	fe = (file != "") ? reg_compile(file.c_str(), REGFLAGS) : NULL;
}

Mime::~Mime()
{
	if (he != NULL)
		reg_free(he);
	if (fe != NULL)
		reg_free(fe);
	if (me != NULL)
		reg_free(me);
}

bool MimeSection::check_and_block(CONNECTION *connection)
{
	const Mime *mime;

	read_lock();

	mime = check(connection);
	if (mime != NULL) {
		putlog(MMLOG_MIME, "blocked mime-type %s from %s%s", (connection->rheader->content_type != NULL) ? connection->rheader->content_type : "(none)", connection->header->host, connection->header->file);

		template_section->send((mime->templ != "") ? mime->templ.c_str() : (dtempl != "") ? dtempl.c_str() : "blocked", connection, 200);

		unlock();

		return FALSE;
	}

	unlock();

	return TRUE;
}

/*
check if a specific mime-type is allowed, and return pointer to list node
if not
*/
const Mime* MimeSection::check(CONNECTION * connection)
{
	int action = FALSE, result = TRUE, ret, i;
	MimeList* mime_list;
	const Mime* match = NULL;

	if (this->enabled == FALSE || connection->bypass & FEATURE_MIME)
		return NULL;

	for (i = 0; i < 2; i++) {
		if (i == 0) {
			if (this->policy == POLICY_ALLOW) {
				mime_list = &this->deny_list;
				action = FALSE;
				result = TRUE;
			} else {
				mime_list = &this->allow_list;
				action = TRUE;
				result = FALSE;
			}
		} else if (result == action) {
			if (this->policy == POLICY_ALLOW) {
				mime_list = &this->allow_list;
				action = TRUE;
			} else {
				mime_list = &this->deny_list;
				action = FALSE;
			}
		} else
			break;

		MimeList::const_iterator current;
		for (current = mime_list->begin(); current != mime_list->end(); current++) {
			if (current->enabled == FALSE)
				continue;

			if (!profile_find(connection->profiles, current->profiles))
				continue;

			if (current->he != NULL) {
				ret = reg_exec(current->he, connection->header->host);
				if (ret)
					continue;
			}

			if (current->fe != NULL) {
				ret = reg_exec(current->fe, connection->header->file);
				if (ret)
					continue;
			}

			if (current->me != NULL && connection->rheader->content_type != NULL) {
				ret = reg_exec(current->me, connection->rheader->content_type);
				if (ret)
					continue;
			} else if (current->me != NULL)
				continue;

			match = &*current;
			result = action;
			break;
		}
	}

	return (!result) ? (match != NULL) ? match : &this->empty : NULL;
}

void MimeSection::check_show(CONNECTION * connection)
{
        const Mime *ml;
        Filebuf *filebuf;

        filebuf = xnew Filebuf();

	read_lock();	
        ml = check(connection);

        filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
        filebuf->Addf("<html><head><title>MIME filter match for %s%s</title></head>", connection->header->host, connection->header->file);
        filebuf->Addf("<body>\n");
        interface_stylesheet(connection, filebuf);
        filebuf->Addf("<table cellpadding=\"0\" cellspacing=\"0\" border=\"0\" width=\"100%%\"><tr><td align=\"center\">\n");
        filebuf->Addf("<table class=\"dialog\">\n");
        filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\">MIME filter match for %s%s</td></tr>\n", connection->header->host, connection->header->file);

        if (ml == NULL)
                filebuf->Addf("<tr><td align=\"center\" colspan=\"2\">None</td></tr>\n");
        else {
                ml->item->display(filebuf, FALSE);
        }

        filebuf->Addf("</table></td></tr> </table></body></html>\n");

        interface_send_response(connection, filebuf);
        xdelete filebuf;

        unlock();
}

