/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "proto.h"

extern CacheSection *cache_section;
extern FtpSection *ftp_section;
extern TemplateSection *template_section;
extern GLOBAL *global;

int protocol_ftp(CONNECTION * connection)
{
	int x, cbid = 0, code, timeout, order = -1, field = -1;
	char *ptr, buf[1024];
	string action, argument;
	Filebuf *filebuf = NULL;
	struct dir_t *dir, *dstart;
	Socket *sock;
	CGIMap cgiargs;

	timeout = ftp_section->timeout_get();

	putlog((connection->flags & CONNECTION_PREFETCH) ? MMLOG_PREFETCH : MMLOG_REQUEST, "%s %s", connection->header->method, connection->header->url);

	/* look for cgi arguments */
	ptr = strchr(connection->header->file, '?');
	if (ptr != NULL) {
		cgiargs.parse(&ptr[1]);
		*ptr = '\0';
	}
	
	if (connection->header->username == NULL) {
		ftp_section->read_lock();
		connection->header->username = xstrdup(ftp_section->anonlogin.c_str());
		connection->header->password = xstrdup(ftp_section->anonpass.c_str());
		ftp_section->unlock();
	}

	if (!(connection->flags & CONNECTION_LOGGEDIN)) {
	      login:
		connection->server->PutSock("USER %s\r\n", connection->header->username);
		x = ftp_code_wait(connection, 331, 530);
		if (x == 530) {
			template_section->send("badauth", connection, 404);
			goto error;
		} else if (x == -1) {
			if (connection->keepalive_server == TRUE) {
				x = protocol_reconnect(connection);
				if (x < 0)
					goto error;

				if (!(connection->flags & CONNECTION_LOGGEDIN))
					/* fresh connection, not from pool */
					goto login;
			} else {
				template_section->send("noconnect", connection, 404);
				goto error;
			}
		}

		if (!(connection->flags & CONNECTION_LOGGEDIN)) {
			connection->server->PutSock("PASS %s\r\n", connection->header->password);
			x = connection->server->GetLine(buf, sizeof(buf), timeout);
			if (x <= 0 || atoi(buf) == 530) {
				template_section->send("badauth", connection, 404);
				goto error;
			}
		}
	}

	connection->server->PutSock("TYPE I\r\n");
	x = ftp_code_wait(connection, 200, 0);
	if (x == -1) {
		if (connection->flags & CONNECTION_LOGGEDIN) {
			/* this connection is from the pool, keep on reconnecting until 
			   we get a working pool connection or a fresh connection */
			connection->flags &= ~CONNECTION_LOGGEDIN;
			goto login;
		} else {
			template_section->send("noconnect", connection, 404);
			goto error;
		}
	}

	connection->keepalive_server = TRUE;
	connection->flags |= CONNECTION_LOGGEDIN;

	connection->server->PutSock("CWD %s\r\n", connection->header->file);
	code = ftp_code_wait(connection, 250, 550);
	if (code == -1) {
		template_section->send("noconnect", connection, 404);
		goto error;
	} else if (code == 250) {
		/* ok... user wants directory listing */

		if (connection->cachemap != NULL) {
			/* I guess this could happen if the proxy was previously
			   setup to forward FTP requests through another proxy,
			   and the index was cached.. may as well get rid of it */
			cache_section->invalidate(connection->cachemap);
			connection->cachemap = NULL;
		}

		if (connection->header->file[strlen(connection->header->file) - 1] != '/') {
			/* request for directory listing without trailing '/' on URL, 
			   we need to send a redirect so relative links work properly */

			connection->header->file = string_append(connection->header->file, "/");

			connection->rheader = header_new();
			connection->rheader->type = HTTP_RESP;
			connection->rheader->code = 302;
			connection->rheader->content_length = 0;
			connection->rheader->location = xstrdup(connection->header->file);

			header_send(connection->rheader, connection, CLIENT, HEADER_RESP);

			goto out;
		}


		action   = cgiargs["action"];
		argument = cgiargs["argument"];

		if (action != "") {
			if (argument != "") {
				if (action == "mkdir")
					connection->server->PutSock("MKD %s\r\n", argument.c_str());
				else if (action == "raw")
					connection->server->PutSock("%s\r\n", argument.c_str());
			}
			
			string filename = cgiargs["filename"];

			if (action == "delete")
				connection->server->PutSock("DELE %s\r\n", filename.c_str());
			else if (action == "rmdir")
				connection->server->PutSock("RMD %s\r\n", filename.c_str());
					
			if (argument != "") {
				if (action == "rename") {
					connection->server->PutSock("RNFR %s\r\n", filename.c_str());
					connection->server->PutSock("RNTO %s\r\n", argument.c_str());
				}
			}
		}
		

		sock = ftp_transfer_setup(connection, "LIST\r\n");
		if (sock != NULL)
			filebuf = ftp_transfer_filebuf(connection, SERVER, sock, -1);
		if (sock == NULL || filebuf == NULL) {
			if (sock != NULL)
				xdelete sock;
			template_section->send("noconnect", connection, 404);
			goto error;
		}
		xdelete sock;

		if (url_command_find(connection->url_command, "raw")) {
			show_filebuf(connection, filebuf);

			xdelete filebuf;

			goto out;
		}


		dir = dir_list_parse(filebuf);

		xdelete filebuf;

		if (action != "" && argument != "" && action == "rematch")
			dir = dir_list_filter(dir, argument.c_str());
		
		string orderstr = cgiargs["order"];
		if (orderstr == "ascending")
			order = SORT_ASCENDING;
		else if (orderstr == "descending")
			order = SORT_DESCENDING;
		else
			order = -1;
		
		string fieldstr = cgiargs["field"];
		if (fieldstr == "name")
			field = SORT_NAME;
		else if (fieldstr == "size")
			field = SORT_SIZE;
		else if (fieldstr == "date")
			field = SORT_DATE;
		else
			field = -1;

		if (order != -1 && field != -1)
			dir = dir_list_sort(dir, order, field);
		else {
			order = ftp_section->sortorder_get();
			field = ftp_section->sortfield_get();

			if (field != SORT_NONE)
				dir = dir_list_sort(dir, order, field);
		}

		show_dirlist(connection, dir);
		dir_list_free(dir);
	} else if (code == 550) {
		ptr = strrchr(connection->header->file, '/');

		s_strncpy(buf, connection->header->file, (ptr - connection->header->file) + 2);
		connection->server->PutSock("CWD %s\r\n", buf);

		x = connection->server->GetLine(buf, sizeof(buf), timeout);
		if (x <= 0)
			goto error;

		code = atoi(buf);
		if (code == 550) {
			template_section->send("nofile", connection, 404);
			goto error;
		} else if (code == 250) {
			sock = ftp_transfer_setup(connection, "LIST\r\n");
			if (sock != NULL)
				filebuf = ftp_transfer_filebuf(connection, SERVER, sock, -1);
			if (sock == NULL || filebuf == NULL) {
				if (sock != NULL)
					xdelete sock;
				template_section->send("noconnect", connection, 404);
				goto error;
			}
			xdelete sock;

			connection->rheader = header_new();
			connection->rheader->type = HTTP_RESP;
			connection->rheader->code = 200;

			dstart = dir = dir_list_parse(filebuf);

			for (; dir && strcmp(dir->name, &ptr[1]); dir = dir->next);
			if (dir == NULL) {
				/* file was not in directory listing */
				/* content length is not known, cannot do keep-alive */
				connection->keepalive_client = FALSE;

				if (connection->cachemap != NULL) {
					/* this may be a temporary error, keep the cache file
					   in case we can validate it later */
					cache_section->cache_close(connection->cachemap);
					connection->cachemap = NULL;
				}
			} else {
				strftime(buf, sizeof(buf), HTTPTIMEFORMAT, &dir->time);
				connection->rheader->last_modified = xstrdup(buf);
				connection->rheader->content_length = dir->size;
				connection->rheader->flags |= HEADER_CL;
			}

			if (connection->cachemap != NULL) {
				/* must be validating it ... */
				if (mktime(&dir->time) == connection->cachemap->lmtime) {
					putlog(MMLOG_CACHE, "validated: %s", connection->cachemap->key);

					connection->flags &= ~CONNECTION_VALIDATE;
					connection->flags |= CONNECTION_VALIDATED;
					cache_section->validated(connection->cachemap);

					http_header_free(connection->rheader);
					connection->rheader = http_header_dup(connection->cachemap->header);

					snprintf(buf, sizeof(buf), "%u", (unsigned int) (utc_time(NULL) - connection->cachemap->mtime));
					FREE_AND_NULL(connection->rheader->age);
					connection->rheader->age = xstrdup(buf);

					dir_list_free(dstart);

					if (transfer_limit_check(connection) == FALSE)
						goto out;

					xcache_header_update(connection, HIT);

					header_send(connection->rheader, connection, CLIENT, HEADER_RESP);
					net_cache_send(connection->cachemap, connection, CLIENT, 0);

					goto out;
				} else {
					/* don't count it as a hit if it failed validation */
					global->stats.Decrement("cache", "hit");
					global->stats.Increment("cache", "miss");

					putlog(MMLOG_CACHE, "validation failed: %s", connection->cachemap->key);

					cache_section->invalidate(connection->cachemap);
					connection->cachemap = NULL;
				}
			}

			/* if we can't get a directory listing, we don't have enough 
			   information about the file to cache it. */
			if (dir != NULL && !strcasecmp(connection->header->username, ANONLOGIN)) {
				filebuf = xnew Filebuf();
				header_send_filebuf(filebuf, connection->rheader, connection, CLIENT, HEADER_RESP);
				filebuf->Add("", 1);
				connection->cachemap = cache_section->cache_open(connection, connection->header->url, filebuf->data, CACHE_WRITING | ((connection->flags & CONNECTION_PREFETCH) ? CACHE_PREFETCHED : 0));
				xdelete filebuf;

				if (connection->cachemap != NULL)
					connection->flags |= CONNECTION_CACHING;
			}

			dir_list_free(dstart);

			if (transfer_limit_check(connection) == FALSE)
				goto out;

			sock = ftp_transfer_setup(connection, "RETR %s\r\n", connection->header->file);
			if (sock == NULL) {
				if (connection->cachemap != NULL) {
					cache_section->invalidate(connection->cachemap);
					connection->cachemap = NULL;
				}

				template_section->send("nofile", connection, 404);

				goto error;
			}

			xcache_header_update(connection, MISS);

			header_send(connection->rheader, connection, CLIENT, HEADER_RESP);

			cbid = connection->server->CallBackAdd(protocol_ftp_read_callback, connection, Socket::READEVENT);

			x = ftp_transfer(connection, SERVER, sock, connection->rheader->content_length);

			connection->server->CallBackRemove(cbid);

			if (x == -1)
				connection->keepalive_client = FALSE;

			xdelete sock;
		} else {
			template_section->send("badresponse", connection, 404);
			goto error;
		}
	}

      out:

	if (connection->rheader != NULL)
		http_header_free(connection->rheader);

	return TRUE;

      error:

	if (connection->rheader != NULL)
		http_header_free(connection->rheader);

	connection->keepalive_server = FALSE;

	return -1;
}

int ftp_code_wait(CONNECTION * connection, int scode, int fcode)
{
	int x, code = -1, timeout;
	char buf[128];

	timeout = ftp_section->timeout_get();

	while ((x = connection->server->GetLine(buf, sizeof(buf), timeout)) > 0) {
		code = atoi(buf);

		if (code == scode || code == fcode)
			return code;
	}

	return code;
}

Socket *ftp_transfer_setup(CONNECTION * connection, char *fmt, ...)
{
	int passive, code, x, fd = -1, timeout;
	char buf[1024], cmd[256], *ptr;
	socklen_t socklen = sizeof(struct sockaddr_in);
	struct sockaddr_in *saddr = NULL;
	struct pollfd pfd;
	va_list valist;

	va_start(valist, fmt);
	vsnprintf(cmd, sizeof(cmd), fmt, valist);
	va_end(valist);

	passive = ftp_section->passive_get();
	timeout = ftp_section->timeout_get();

	if (passive == TRUE) {
		connection->server->PutSock("PASV\r\n");

		while ((x = connection->server->GetLine(buf, sizeof(buf), timeout)) > 0) {
			code = atoi(buf);
			if (code == 227)
				break;
		}
		if (x == -1)
			goto error;

		ptr = strchr(buf, '(');
		if (ptr == NULL)
			goto error;

		saddr = ftpport_parse(&ptr[1]);

		fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (fd == -1)
			goto error;
	} else {
		saddr = (sockaddr_in*)xmalloc(sizeof(struct sockaddr_in));
		memset(saddr, 0, sizeof(struct sockaddr_in));

		/* bind to same interface connection to ftp server is made on */
		getsockname(connection->server->fd, (struct sockaddr *) saddr, &socklen);
		saddr->sin_port = 0;

		fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (fd == -1)
			goto error;

		x = bind(fd, (struct sockaddr *) saddr, sizeof(struct sockaddr_in));
		if (x == -1)
			goto error;

		x = listen(fd, 1);
		if (x == -1)
			goto error;

		/* get the port the kernel chose for us */
		getsockname(fd, (struct sockaddr *) saddr, &socklen);

		ptr = ftpport_create(saddr);
		connection->server->PutSock("PORT %s\r\n", ptr);
		xfree(ptr);

		x = ftp_code_wait(connection, 200, 0);
		if (x == -1)
			goto error;
	}

	/* ok, tranfer is ready to go... */
	connection->server->PutSock("%s", cmd);

	if (passive == TRUE) {
		x = connect(fd, (struct sockaddr *) saddr, sizeof(struct sockaddr_in));
		if (x == -1)
			goto error;
	} else {
		pfd.fd = fd;
		pfd.events = POLLIN;

		x = p_poll(&pfd, 1, ftp_section->timeout_get() * 1000);
		if (x <= 0 || !(pfd.revents & POLLIN))
			goto error;

		x = accept(fd, (struct sockaddr *) saddr, &socklen);
		if (x == -1)
			goto error;

		close(fd);
		fd = x;
	}

	xfree(saddr);

	return new Socket(fd);

      error:

	if (saddr != NULL)
		xfree(saddr);
	if (fd != -1)
		close(fd);

	return NULL;
}

int ftp_transfer(CONNECTION * connection, int which, Socket * sock, int bytes)
{
	int x;
	Socket *sock2;

	sock2 = connection->server;

	connection->server = sock;

	x = net_transfer(connection, which, bytes);

	connection->server = sock2;

	return x;
}

Filebuf *ftp_transfer_filebuf(CONNECTION * connection, int which, Socket * sock, int bytes)
{
	Socket *sock2;
	Filebuf *filebuf;

	sock2 = connection->server;

	connection->server = sock;

	filebuf = xnew Filebuf();

	net_filebuf_read(filebuf, connection, which, bytes);

	connection->server = sock2;

	return filebuf;
}

int protocol_ftp_read_callback(void *arg, int len, char *buf) {
        int i;
        CONNECTION *connection = (CONNECTION *)arg;

        if (connection->cachemap != NULL) {
                i = cache_section->add(connection->cachemap, buf, len);
                if (i == -1 && (connection->flags & CONNECTION_PREFETCH))
                        return -1;
        }

        connection->transferred += len;
        connection->transferlimit -= (len > connection->transferlimit) ? connection->transferlimit : len;

        if (connection->transferlimit == 0) {
		connection->flags |= CONNECTION_FAILED;

		return -1;
	}

        return 0;
}

