/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include "proto.h"

RedirectSection::RedirectSection():
	Section("redirect", RWLOCK),
     enabled (field_vec[0].int_value)
{

}

void RedirectSection::update()
{
	redirect_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		redirect_list.push_back(Redirect(*item));
	
}


Redirect::Redirect(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     url      (item.field_vec[3].string_value),
     redirect (item.field_vec[4].string_value),
     port     (item.field_vec[5].int_value),
     send302  (item.field_vec[6].int_value),
     options  (item.field_vec[7].int_value),
     which    (item.field_vec[8].int_value)
{
	up = (url != "") ? reg_sub_compile(url.c_str(), PCREFLAGS) : NULL;
}

Redirect::Redirect(const Redirect& r):
     enabled  (r.enabled),
     comment  (r.comment),
     profiles (r.profiles),
     url      (r.url),
     redirect (r.redirect),
     port     (r.port),
     send302  (r.send302),
     options  (r.options),
     which    (r.which)
{
	up = (url != "") ? reg_sub_compile(url.c_str(), PCREFLAGS) : NULL;
}

Redirect::~Redirect()
{
	if (up != NULL)
		reg_sub_free(up);
}


int RedirectSection::redirect_do(CONNECTION * connection, int type)
{
	int x = FALSE, offset;
	char url[1024], *ret, *ptr, *ptr2;
	regmatch_sub_t *rmatch;
	HEADER *header;

	if (connection->bypass & FEATURE_REDIRECT)
		return FALSE;

	read_lock();

	if (this->enabled == FALSE) {
		unlock();

		return FALSE;
	}

	if (type == REDIRECT_REQUEST) {
		if (connection->header->host == NULL)
			s_strncpy(url, connection->header->file, sizeof(url));
		else
			snprintf(url, sizeof(url), "%s://%s%s", connection->header->proto, connection->header->host, connection->header->file);
	} else
		s_strncpy(url, connection->rheader->location, sizeof(url));

	RedirectList::const_iterator rl;
	for (rl = redirect_list.begin(); rl != redirect_list.end(); rl++) {
		if (rl->enabled == FALSE)
			continue;

		if (!profile_find(connection->profiles, rl->profiles))
			continue;

		if (rl->up != NULL && (rl->which == REDIRECT_BOTH || (rl->which == REDIRECT_URL && type == REDIRECT_REQUEST) || (rl->which == REDIRECT_LOCATION && type == REDIRECT_HEADER))) {
			if (rl->options & URL_DECODEBEFORE) {
				/* decode url before matching */
				ret = url_decode(url, strlen(url));
				rmatch = reg_sub_exec(rl->up, ret);
				xfree(ret);
			} else
				rmatch = reg_sub_exec(rl->up, url);

			if (rmatch != NULL) {
				ret = reg_sub_parse(rmatch, rl->redirect.c_str());

				ptr2 = strstr(ret, "://");
				if (ptr2 == NULL)
					ptr2 = ret;
				else
					ptr2 += 3;

				ptr = strchr(ptr2, '/');

				if (ptr != NULL && (rl->options & (URL_ENCODE | URL_DECODEAFTER))) {
					/* encode or decode file portion of url */
					offset = ptr - ret;

					ptr2 = (rl->options & URL_ENCODE) ? url_encode(&ptr[1], strlen(&ptr[1])) : url_decode(&ptr[1], strlen(&ptr[1]));
					ret = (char*)xrealloc(ret, ptr - ret + strlen(ptr2) + 2);

					strcpy(&ret[offset + 1], ptr2);

					xfree(ptr2);
				}

				if (type == REDIRECT_REQUEST) {
					if (rl->send302 == TRUE) {
						putlog(MMLOG_REDIRECT, "sent 302 redirect for request for %s to %s", url, ret);

						/* send a 302 redirect */
						header = header_new();
						header->type = HTTP_RESP;
						header->code = 302;
						header->content_length = 0;
						header->flags |= HEADER_CL;

						if (*ret != '/') {
							if (connection->url_command != NULL) {
								ptr = url_command_create(connection->url_command);
								ptr2 = strstr(ret, "://");
								if (ptr2 != NULL) {
									s_strncpy(url, ret, ptr2 - ret + 1);
									snprintf(url + (ptr2 - ret), sizeof(url) - (ptr2 - ret), "://%s%s", ptr, ptr2 + 3);
								} else
									snprintf(url, sizeof(url), "%s%s", ptr, ret);

								xfree(ptr);
							} else
								s_strncpy(url, ret, sizeof(url));
						} else
							s_strncpy(url, ret, sizeof(url));

						header->location = xstrdup(url);

						header_send(header, connection, CLIENT, HEADER_RESP);

						http_header_free(header);

						x = TRUE;
					} else {
						/* just connect to a different host */
						ptr2 = strstr(ret, "://");
						if (ptr2 != NULL) {
							FREE_AND_NULL(connection->header->proto);
							connection->header->proto = xstrndup(ret, ptr2 - ret);
							ptr2 += 3;
						} else
							ptr2 = ret;

						if (ptr == NULL || *ret != '/') {
							FREE_AND_NULL(connection->header->host);
							connection->header->host = xstrndup(ptr2, (ptr != NULL) ? ptr - ret - ((ptr2 != NULL) ? ptr2 - ret : 0) : ~0);
						}

						FREE_AND_NULL(connection->header->file);
						connection->header->file = xstrdup((ptr != NULL) ? ptr : "/");

						if (rl->port != -1)
							connection->header->port = rl->port;

						if (connection->header->host != NULL) {
							/* if this was a HTTP request, it's now a proxy request */
							connection->header->type = HTTP_PROXY;

							putlog(MMLOG_REDIRECT, "request for %s to %s://%s%s", url, connection->header->proto, connection->header->host, connection->header->file);
						} else
							putlog(MMLOG_REDIRECT, "request for %s to %s", url, connection->header->file);
					}
				} else {
					/* change the Location: header from a 302 redirect */
					putlog(MMLOG_REDIRECT, "302 redirect for %s to %s", connection->rheader->location, ret);

					xfree(connection->rheader->location);

					connection->rheader->location = xstrdup(ret);
				}

				FREE_AND_NULL(ret);
				reg_sub_match_free(rmatch);

				break;
			} 
		}
	}

	unlock();

	return x;
}
