/***********************************************************************
*
* mimedefang.c
*
* C interface to the attachment-filter program for stripping or altering
* MIME attachments in incoming Sendmail connections.
*
* Copyright (C) 2000 Roaring Penguin Software Inc.
* http://www.roaringpenguin.com
*
* This program may be distributed under the terms of the GNU General
* Public License, Version 2, or (at your option) any later version.
*
* This program was derived from the sample mail filter included in
* libmilter/README in the Sendmail 8.11 distribution.
***********************************************************************/

static char const RCSID[] =
"$Id: mimedefang.c,v 1.216 2004/10/27 13:09:43 dfs Exp $";

/* Define this to work around an M$ Outlook bug! */
/* #define CONVERT_EMBEDDED_CRS_IN_HEADERS 1 */

#define _BSD_SOURCE 1

#ifdef __linux__
/* On Linux, we need this defined to get fdopen.  On BSD, if we define
 * it, we don't get all the u_char, u_long, etc definitions.  GRR! */
#define _POSIX_SOURCE 1
#endif

#ifdef HAVE_SOCKLEN_T
typedef socklen_t md_socklen_t;
#else
typedef int md_socklen_t;
#endif

#include "config.h"
#include "mimedefang.h"

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <pthread.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <time.h>
#include <sys/time.h>
#include <pwd.h>
#include <stdio.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "libmilter/mfapi.h"

#include <sys/socket.h>
#include <sys/un.h>

#ifdef ENABLE_DEBUGGING
#include <signal.h>
#define DEBUG(x) x
#else
#define DEBUG(x) (void) 0
#endif

extern int find_syslog_facility(char const *facility_name);

#define DEBUG_ENTER(func, line) DEBUG(syslog(LOG_DEBUG, "Entering %s (line %d)", func, line))
#define DEBUG_EXIT(func, line, ret) DEBUG(syslog(LOG_DEBUG, "Exiting %s (line %d) ret=%s", func, line, ret))

#define SCAN_BODY "MIMEDefang " VERSION

/* Call a Milter smfi_xxxx function, but syslog if it doesn't return
 * MI_SUCCESS */
#define MD_SMFI_TRY(func, args) do { if (func args != MI_SUCCESS) syslog(LOG_WARNING, "%s: %s returned MI_FAILURE", data->qid, #func); } while (0)

char *scan_body = NULL;

#define KEY_FILE CONFDIR "/mimedefang-ip-key"

/* In debug mode, we do not delete working directories. */
int DebugMode = 0;

/* Conserve file descriptors by reopening files in each callback */
int ConserveDescriptors = 0;

/* Log "eom" run-times */
int LogTimes = 0;

/* Default Backlog for "listen" */
static int Backlog = -1;

/* Run as this user */
static char *user = NULL;
extern int drop_privs(char const *user, uid_t uid, gid_t gid);

/* NOQUEUE */
static char *NOQUEUE = "NOQUEUE";

/* My IP address */
static char *MyIPAddress;

/* Header name for validating IP addresses */
static char ValidateHeader[256];

/* Additional Sendmail macros to pass along */
#define MAX_ADDITIONAL_SENDMAIL_MACROS 32
static char *AdditionalMacros[MAX_ADDITIONAL_SENDMAIL_MACROS];
static int NumAdditionalMacros = 0;

/* Keep track of private data -- file name and fp for writing e-mail body */
struct privdata {
    char *hostname;		/* Name of connecting host */
    char *hostip;		/* IP address of connecting host */
    char *myip;                 /* My IP address, from Sendmail macro */
    char *sender;		/* Envelope sender */
    char *firstRecip;		/* Address of first recipient */
    char *dir;			/* Work directory */
    char *heloArg;		/* HELO argument */
    char *qid;                  /* Queue ID */
    int fd;			/* File for message body */
    int headerFD;		/* File for message headers */
    int cmdFD;			/* File for commands */
    int numContentTypeHeaders;  /* How many Content-Type headers have we seen? */
    unsigned char validatePresent; /* Saw a relay-address validation header */
    unsigned char suspiciousBody; /* Suspicious characters in message body? */
    unsigned char lastWasCR;	/* Last char of body chunk was CR? */
    unsigned char filterFailed; /* Filter failed */
};

static void write_macro_value(SMFICTX *ctx,
			      char *macro);

static sfsistat cleanup(SMFICTX *ctx);
static sfsistat mfclose(SMFICTX *ctx);
static int do_sm_quarantine(SMFICTX *ctx, char const *reason);
static void remove_working_directory(struct privdata *data);

static char const *MultiplexorSocketName = NULL;

static int set_reply(SMFICTX *ctx, char const *first, char const *code, char const *dsn, char const *reply);

#define DATA ((struct privdata *) smfi_getpriv(ctx))

/* How many tries to get a unique directory name before giving up? */
#define MAXTRIES 8192

/* Size of chunk when replacing body */
#define CHUNK 4096

/* Number of file descriptors to close when forking */
#define CLOSEFDS 256

/* Mutex to protect mkdir() calls */
static pthread_mutex_t MkdirMutex = PTHREAD_MUTEX_INITIALIZER;

/* Do relay check? */
static int doRelayCheck = 0;

/* Do sender check? */
static int doSenderCheck = 0;

/* Do recipient check? */
static int doRecipientCheck = 0;

/* Keep directories around if multiplexor fails? */
static int keepFailedDirectories = 0;

/* Protect mkdir() with mutex? */
static int protectMkdirWithMutex = 0;

static void set_dsn(SMFICTX *ctx, char *buf2, int code);

#define NO_DELETE_DIR SPOOLDIR "/DO-NOT-DELETE-WORK-DIRS"

#ifdef ENABLE_DEBUGGING
/**********************************************************************
*%FUNCTION: handle_sig
*%ARGUMENTS:
* s -- signal number
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Handler for SIGSEGV and SIGBUS -- logs a message and returns -- hopefully,
* we'll get a nice core dump the second time around
***********************************************************************/
static void
handle_sig(int s)
{
    syslog(LOG_ERR, "WHOA, NELLY!  Caught signal %d -- this is bad news.  Core dump at 11.", s);

    /* Default is terminate and core. */
    signal(s, SIG_DFL);

    /* Return and probably cause core dump */
}
#endif

/**********************************************************************
* %FUNCTION: get_fd
* %ARGUMENTS:
*  data -- our struct privdata
*  fname -- filename to open for writing.  Relative to work directory
*  sample_fd -- the "sample" fd from "data", if we're not conserving.
* %RETURNS:
*  A file descriptor open for writing, or -1 on failure
* %DESCRIPTION:
*  If we are NOT conserving file descriptors, simply returns sample_fd.
*  If we ARE conserving file descriptors, opens fname for writing.
***********************************************************************/
static int
get_fd(struct privdata *data,
       char const *fname,
       int sample_fd)
{
    char buf[SMALLBUF];
    if (sample_fd >= 0 && !ConserveDescriptors) return sample_fd;

    snprintf(buf, SMALLBUF, "%s/%s", data->dir, fname);
    sample_fd = open(buf, O_CREAT|O_APPEND|O_RDWR, 0640);
    if (sample_fd < 0) {
	syslog(LOG_WARNING, "%s: Could not open %s/%s: %m",
	       data->qid, data->dir, fname);
    }
    return sample_fd;
}

/**********************************************************************
* %FUNCTION: put_fd
* %ARGUMENTS:
*  fd -- file descriptor to close
* %RETURNS:
*  -1 if descriptor was closed; fd otherwise.
* %DESCRIPTION:
*  If we are NOT conserving file descriptors, simply returns fd.
*  If we ARE conserving file descriptors, closes fd and returns -1.
***********************************************************************/
static int
put_fd(int fd)
{
    if (!ConserveDescriptors) return fd;

    closefd(fd);
    return -1;
}

/**********************************************************************
* %FUNCTION: set_reply
* %ARGUMENTS:
*  ctx -- filter context
*  first -- digit with which code must start
*  code -- SMTP three-digit code
*  dsn -- SMTP DSN status notification code
*  reply -- text message
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Sets the SMTP reply code and message.  code and dsn are validated.
***********************************************************************/
static int
set_reply(SMFICTX *ctx,
	  char const *first,
	  char const *code,
	  char const *dsn,
	  char const *reply)
{
    char *safe_reply;
    if (!reply || !*reply) {
	if (*first == '4') {
	    reply = "Please try again later";
	} else {
	    reply = "Forbidden for policy reasons";
	}
    }
    if (!validate_smtp_code(code, first)) {
	if (*first == '4') code = "451";
	else               code = "554";
    }
    if (!validate_smtp_dsn(dsn, first)) {
	if (*first == '4') dsn  = "4.3.0";
	else               dsn  = "5.7.1";
    }

    /* We need to double any "%" chars in reply */
    if (strchr(reply, '%')) {
	int retcode;
	char const *s;
	char *t;
	/* Worst-case, we'll double our length */
	safe_reply = malloc(strlen(reply) * 2 + 1);
	if (!safe_reply) {
	    syslog(LOG_ERR, "Out of memory to escape reply %s", reply);
	    return smfi_setreply(ctx, (char *) code, (char *) dsn,
				 "Out of memory");
	}
	s = reply;
	t = safe_reply;
	while (*s) {
	    if (*s == '%') *t++ = '%';
	    *t++ = *s++;
	}
	*t = 0;
	retcode = smfi_setreply(ctx, (char *) code, (char *) dsn, safe_reply);
	free(safe_reply);
	return retcode;
    }

    /* smfi_setreply is not const-correct, hence the (char *) casts */
    return smfi_setreply(ctx, (char *) code, (char *) dsn, (char *) reply);
}

/**********************************************************************
*%FUNCTION: closefiles
*%ARGUMENTS:
* None
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Hack -- closes a whole bunch of file descriptors.  Use after a fork()
*         to conserve descriptors.
***********************************************************************/
static void
closefiles(void)
{
    int i;
    for (i=0; i<CLOSEFDS; i++) {
	(void) close(i);
    }
}

/**********************************************************************
*%FUNCTION: mfconnect
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* hostname -- name of connecting host
* sa -- socket address of connecting host
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Allocates a private data structure for tracking this connection
***********************************************************************/
static sfsistat
mfconnect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *sa)
{
    struct privdata *data;
    int n;

    char const *tmp;
    struct sockaddr_in *insa = (struct sockaddr_in *) sa;
#if defined(AF_INET6) && defined(HAVE_INET_NTOP)
    struct sockaddr_in6 *in6sa = (struct sockaddr_in6 *) sa;
#endif

    DEBUG_ENTER("mfconnect", __LINE__);

    /* Delete any existing context data */
    mfclose(ctx);

    /* If too many running filters, reject connection at this phase */
    n = MXCheckFreeSlaves(MultiplexorSocketName);
    if (n == 0) {
	syslog(LOG_WARNING, "mfconnect: No free slaves");
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    if (n < 0) {
	syslog(LOG_WARNING, "mfconnect: Error communicating with multiplexor");
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    data = malloc_with_log(sizeof *data);
    if (!data) {
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    data->hostname = NULL;
    data->hostip   = NULL;
    data->myip     = NULL;
    data->sender   = NULL;
    data->firstRecip = NULL;
    data->dir      = NULL;
    data->heloArg  = NULL;
    data->qid      = NOQUEUE;
    data->fd       = -1;
    data->headerFD = -1;
    data->cmdFD    = -1;
    data->numContentTypeHeaders = 0;
    data->validatePresent = 0;
    data->suspiciousBody = 0;
    data->lastWasCR      = 0;
    data->filterFailed   = 0;

    /* Save private data */
    if (smfi_setpriv(ctx, data) != MI_SUCCESS) {
	free(data);
	/* Can't hurt... */
	smfi_setpriv(ctx, NULL);
	syslog(LOG_WARNING, "Unable to set private data pointer: smfi_setpriv failed");
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (hostname) {
	data->hostname = strdup_with_log(hostname);
	if (!data->hostname) {
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    } else {
	data->hostname = NULL;
    }

    /* Padding -- should be big enough for IPv6 addresses */
    if (!sa) {
	data->hostip = NULL;
    } else {
	data->hostip = malloc_with_log(65);
	if (!data->hostip) {
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
#ifdef HAVE_INET_NTOP
#ifdef AF_INET6
        if (sa->sa_family == AF_INET6) {
	    tmp = inet_ntop(AF_INET6, &in6sa->sin6_addr, data->hostip, 65);
	} else
#endif
        if (sa->sa_family == AF_INET) {
	    tmp = inet_ntop(AF_INET, &insa->sin_addr, data->hostip, 65);
	} else {
	    syslog(LOG_WARNING, "Unknown address family %d",
		   (int) sa->sa_family);
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
#else
	tmp = inet_ntoa(insa->sin_addr);
	if (tmp) strncpy(data->hostip, tmp, 64);
#endif
	if (!tmp) {
	    syslog(LOG_WARNING, "inet_ntoa or inet_ntop failed: %m");
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	data->hostip[64] = 0;
    }

    data->dir = NULL;
    data->fd = -1;
    data->headerFD = -1;
    data->cmdFD = -1;
    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    if (doRelayCheck) {
	char buf2[SMALLBUF];
	int n = MXRelayOK(MultiplexorSocketName, buf2, data->hostip,
			  data->hostname);
	if (n == 0) {
	    set_dsn(ctx, buf2, 5);
	    /* We reject connections from this relay */
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_REJECT");
	    return SMFIS_REJECT;
	}
	if (n < 0) {
	    set_dsn(ctx, buf2, 4);
	    cleanup(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	if (n == 2) {
	    set_dsn(ctx, buf2, 2);
	    cleanup(ctx);
	    return SMFIS_ACCEPT;
	}
	if (n == 3) {
	    set_dsn(ctx, buf2, 2);
	    cleanup(ctx);
	    return SMFIS_DISCARD;
	}
	if (n == 1) {
	    /* Called only in case we need to delay */
	    set_dsn(ctx, buf2, 2);
	}
    }

    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
* %FUNCTION: helo
* %ARGUMENTS:
*  ctx -- Milter context
*  helohost -- argument to "HELO" or "EHLO" SMTP command
* %RETURNS:
*  SMFIS_CONTINUE
* %DESCRIPTION:
*  Stores the HELO argument in the private data area
***********************************************************************/
static sfsistat
helo(SMFICTX *ctx, char *helohost)
{
    struct privdata *data = DATA;
    if (!data) {
	DEBUG_EXIT("helo", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    if (data->heloArg) {
	free(data->heloArg);
	data->heloArg = NULL;
    }
    data->heloArg = strdup_with_log(helohost);
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: envfrom
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* from -- list of arguments to "MAIL FROM:" SMTP command.
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Allocates a private data structure for tracking this message, and
* opens a temporary file for saving message body.
***********************************************************************/
static sfsistat
envfrom(SMFICTX *ctx, char **from)
{
    struct privdata *data;
    int tries;
    int i;
    int success;
    char buffer[SMALLBUF];
    char buf2[SMALLBUF];
    time_t now = time(NULL);
    unsigned long ulnow = (unsigned long) now;
    char *queueid;
    char *me;

    DEBUG_ENTER("envfrom", __LINE__);
    /* Get the private context */
    data = DATA;
    if (!data) {
	syslog(LOG_WARNING, "envfrom: Unable to obtain private data from milter context");
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Obtain the queue identifier */
    queueid = smfi_getsymval(ctx, "i");

    if (data->qid && data->qid != NOQUEUE) {
	free(data->qid);
	data->qid = NOQUEUE;
    }
    if (queueid && *queueid) {
	data->qid = strdup_with_log(queueid);
	if (!data->qid) {
	    data->qid = NOQUEUE;
	    cleanup(ctx);
	    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    }

    /* Copy sender */
    if (data->sender) {
	free(data->sender);
    }
    data->sender = strdup_with_log(from[0]);
    if (!data->sender) {
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Old data lying around? */
    if (data->firstRecip) {
	free(data->firstRecip);
	data->firstRecip = NULL;
    }

    /* We need a nice clean directory all our own.  If two
     * threads get the same name, one of the mkdir() calls will fail.
     * On some systems, we might need to lock this region, hence
     * the MkdirMutex */

    success = 0;

    /* Try making a directory whose name contains the Sendmail qid first */
    if (data->qid && data->qid != NOQUEUE) {
	snprintf(buffer, SMALLBUF, "%s/mdefang-%s", SPOOLDIR, data->qid);
	if (!mkdir(buffer, 0750)) {
	    success = 1;
	}
    }

    if (!success) {
	if (protectMkdirWithMutex) {
	    if (pthread_mutex_lock(&MkdirMutex)) {
		syslog(LOG_INFO, "Could not lock MkdirMutex");
		cleanup(ctx);
		DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	}

	for (tries = 0; tries < MAXTRIES; tries++) {
	    snprintf(buffer, SMALLBUF, "%s/mdefang-%lX-%d",
		     SPOOLDIR,
		     ulnow, tries);
	    if (!mkdir(buffer, 0750)) {
		success = 1;
		break;
	    }

	    /* Apparently, mkdir on Solaris 8 can fail with EBADF */
	    if ((errno != EEXIST) && (errno != EBADF)) {
		break;
	    }
	}

	if (protectMkdirWithMutex) {
	    (void) pthread_mutex_unlock(&MkdirMutex);
	}
    }

    /* Could not create temp. directory */
    if (!success) {
	syslog(LOG_WARNING, "%s: Could not create directory %s: %m",
	       data->qid, buffer);
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (data->dir) free(data->dir);
    if (data->fd >= 0) closefd(data->fd);
    if (data->headerFD >= 0) closefd(data->headerFD);
    if (data->cmdFD >= 0) closefd(data->cmdFD);

    data->dir = NULL;
    data->fd = -1;
    data->headerFD = -1;
    data->cmdFD = -1;
    data->validatePresent = 0;
    data->filterFailed = 0;
    data->numContentTypeHeaders = 0;

    data->dir = strdup_with_log(buffer);

    if (!data->dir) {
	/* Don't forget to clean up directory... */
	rmdir(buffer);
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Open command file */
    data->cmdFD = get_fd(data, "COMMANDS", data->cmdFD);
    if (data->cmdFD < 0) {
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Write the sender */
    write_mx_command(data->cmdFD, 'S', (unsigned char *) from[0]);

    /* Write some macros */
    write_macro_value(ctx, "_");
    write_macro_value(ctx, "auth_authen");
    write_macro_value(ctx, "auth_author");
    write_macro_value(ctx, "auth_ssf");
    write_macro_value(ctx, "auth_type");
    write_macro_value(ctx, "cert_issuer");
    write_macro_value(ctx, "cert_subject");
    write_macro_value(ctx, "cipher");
    write_macro_value(ctx, "cipher_bits");
    write_macro_value(ctx, "daemon_name");
    write_macro_value(ctx, "i");
    write_macro_value(ctx, "if_addr");
    write_macro_value(ctx, "if_name");
    write_macro_value(ctx, "j");
    write_macro_value(ctx, "mail_addr");
    write_macro_value(ctx, "mail_host");
    write_macro_value(ctx, "mail_mailer");
    write_macro_value(ctx, "tls_version");
    write_macro_value(ctx, "verify");

    /* Write any additional macros requested by user */
    for (i=0; i<NumAdditionalMacros; i++) {
	write_macro_value(ctx, AdditionalMacros[i]);
    }

    /* Get my IP address */
    me = smfi_getsymval(ctx, "{if_addr}");
    if (me && *me && !strcmp(me, "127.0.0.1")) {
	data->myip = strdup_with_log(me);
    } else {
	/* Sigh... use our computed address */
	data->myip = MyIPAddress;
    }


    if (doSenderCheck) {
	int n = MXSenderOK(MultiplexorSocketName, buf2,
			   from[0], data->hostip, data->hostname,
			   data->heloArg, data->dir, data->qid);
	if (n == 0) {
	    set_dsn(ctx, buf2, 5);

	    /* We reject connections from this sender */
	    cleanup(ctx);
	    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_REJECT");
	    return SMFIS_REJECT;
	}
	if (n < 0) {
	    set_dsn(ctx, buf2, 4);

	    cleanup(ctx);
	    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	if (n == 2) {
	    set_dsn(ctx, buf2, 2);
	    cleanup(ctx);
	    return SMFIS_ACCEPT;
	}
	if (n == 3) {
	    set_dsn(ctx, buf2, 2);
	    cleanup(ctx);
	    return SMFIS_DISCARD;
	}
	if (n == 1) {
	    /* Called only in case we need to delay */
	    set_dsn(ctx, buf2, 2);
	}
    }

    if (queueid) {
	write_mx_command(data->cmdFD, 'Q', (unsigned char *) queueid);
    }

    /* Write host name and host IP */
    write_mx_command(data->cmdFD, 'H', (unsigned char *) data->hostname);
    write_mx_command(data->cmdFD, 'I', (unsigned char *) data->hostip);

    /* Write HELO value */
    if (data->heloArg) {
	write_mx_command(data->cmdFD, 'E', (unsigned char *) data->heloArg);
    }

    data->cmdFD = put_fd(data->cmdFD);
    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: rcptto
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* to -- list of arguments to each RCPT_TO
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Saves recipient data
***********************************************************************/
static sfsistat
rcptto(SMFICTX *ctx, char **to)
{
    struct privdata *data = DATA;
    char ans[SMALLBUF];
    sfsistat retcode = SMFIS_CONTINUE;
    char const *rcpt_mailer, *rcpt_host, *rcpt_addr;

    DEBUG_ENTER("rcptto", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "rcptto: Unable to obtain private data from milter context");
	DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    rcpt_mailer = smfi_getsymval(ctx, "{rcpt_mailer}");
    if (!rcpt_mailer || !*rcpt_mailer) rcpt_mailer = "?";

    rcpt_host = smfi_getsymval(ctx, "{rcpt_host}");
    if (!rcpt_host || !*rcpt_host) rcpt_host = "?";

    rcpt_addr = smfi_getsymval(ctx, "{rcpt_addr}");
    if (!rcpt_addr || !*rcpt_addr) rcpt_addr = "?";

    /* Recipient check if enabled */
    if (doRecipientCheck) {
	int n;

	/* If this is first recipient, copy it */
	if (!data->firstRecip) {
	    data->firstRecip = strdup_with_log(to[0]);
	    if (!data->firstRecip) {
		DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	}
	n = MXRecipientOK(MultiplexorSocketName, ans,
			  to[0], data->sender, data->hostip,
			  data->hostname, data->firstRecip, data->heloArg,
			  data->dir, data->qid,
			  rcpt_mailer, rcpt_host, rcpt_addr);
	if (n == 0) {
	    /* We reject to this recipient */
	    set_dsn(ctx, ans, 5);

	    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_REJECT");
	    return SMFIS_REJECT;
	}
	if (n < 0) {
	    set_dsn(ctx, ans, 4);

	    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	if (n == 2) {
	    set_dsn(ctx, ans, 2);

	    retcode = SMFIS_ACCEPT;
	}
	if (n == 3) {
	    set_dsn(ctx, ans, 2);

	    cleanup(ctx);
	    return SMFIS_DISCARD;
	}
	if (n == 1) {
	    /* Called only in case we need to delay */
	    set_dsn(ctx, ans, 2);
	}
    }

    /* Write recipient line, only for recipients we accept! */
    data->cmdFD = get_fd(data, "COMMANDS", data->cmdFD);
    if (data->cmdFD < 0) {
	cleanup(ctx);
	DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    writestr(data->cmdFD, "R");
    write_percent_encoded((unsigned char *) to[0], data->cmdFD);
    writestr(data->cmdFD, " ");
    write_percent_encoded((unsigned char *) rcpt_mailer, data->cmdFD);
    writestr(data->cmdFD, " ");
    write_percent_encoded((unsigned char *) rcpt_host, data->cmdFD);
    writestr(data->cmdFD, " ");
    write_percent_encoded((unsigned char *) rcpt_addr, data->cmdFD);
    writestr(data->cmdFD, "\n");
    data->cmdFD = put_fd(data->cmdFD);
    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_CONTINUE or SMFIS_ACCEPT");
    return retcode;
}

/**********************************************************************
*%FUNCTION: header
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* headerf -- Header field name
* headerv -- Header value
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes the header to the temporary file
***********************************************************************/
static sfsistat
header(SMFICTX *ctx, char *headerf, char *headerv)
{
    struct privdata *data = DATA;
    int suspicious = 0;
    int write_header = 1;

    DEBUG_ENTER("header", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "header: Unable to obtain private data from milter context");
	DEBUG_EXIT("header", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Check for multiple content-type headers */
    if (!strcasecmp(headerf, "content-type")) {
	data->numContentTypeHeaders++;
	/* If more than one content-type header, only write the first one
	   to ensure reliable interpretation by filter! */
	if (data->numContentTypeHeaders > 1) write_header = 0;
    }

    data->fd = get_fd(data, "INPUTMSG", data->fd);
    if (data->fd < 0) {
	cleanup(ctx);
	DEBUG_EXIT("header", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (write_header) {
	/* Write the header to the message file */
	suspicious = safeWriteHeader(data->fd, (unsigned char *) headerf);
	writestr(data->fd, ": ");
	suspicious |= safeWriteHeader(data->fd, (unsigned char *) headerv);
	writestr(data->fd, "\n");
    }
    data->fd = put_fd(data->fd);

    /* Remove embedded newlines and save to our HEADERS file */
    chomp(headerf);
    chomp(headerv);
    data->headerFD = get_fd(data, "HEADERS", data->headerFD);
    if (data->headerFD < 0) {
	cleanup(ctx);
	DEBUG_EXIT("header", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    if (write_header) {
	writestr(data->headerFD, headerf);
	writestr(data->headerFD, ": ");
	writestr(data->headerFD, headerv);
	writestr(data->headerFD, "\n");
    }
    data->headerFD = put_fd(data->headerFD);

    data->cmdFD = get_fd(data, "COMMANDS", data->cmdFD);
    if (data->cmdFD < 0) {
	cleanup(ctx);
	DEBUG_EXIT("header", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    if (suspicious) {
	write_mx_command(data->cmdFD, '!', NULL);
    }
    /* Check for subject -- special case */
    if (!strcasecmp(headerf, "subject")) {
	write_mx_command(data->cmdFD, 'U', (unsigned char *) headerv);
    } else if (!strcasecmp(headerf, "message-id")) {
	write_mx_command(data->cmdFD, 'X', (unsigned char *) headerv);
    }

    /* Check for validating IP header.  If found, write a J line
       to the file to reset the SMTP host address */
    if (ValidateHeader[0] && !strcmp(headerf, ValidateHeader)) {
	/* Make sure it looks like an IP address, though... */
	int n, a, b, c, d;
	char ipaddr[32];
	n = sscanf(headerv, "%d.%d.%d.%d", &a, &b, &c, &d);
	if (n == 4 &&
	    a >= 0 && a <= 255 &&
	    b >= 0 && b <= 255 &&
	    c >= 0 && c <= 255 &&
	    d >= 0 && d <= 255) {
	    sprintf(ipaddr, "%d.%d.%d.%d", a, b, c, d);
	    write_mx_command(data->cmdFD, 'J', (unsigned char *) ipaddr);
	    data->validatePresent = 1;
	}
    }
    data->cmdFD = put_fd(data->cmdFD);

    DEBUG_EXIT("header", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: eoh
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes a blank line to indicate the end of headers.
***********************************************************************/
static sfsistat
eoh(SMFICTX *ctx)
{
    struct privdata *data = DATA;

    DEBUG_ENTER("eoh", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "eoh: Unable to obtain private data from milter context");
	DEBUG_EXIT("eoh", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* We can close headerFD to save a descriptor */
    if (data->headerFD >= 0 && closefd(data->headerFD) < 0) {
	data->headerFD = -1;
	syslog(LOG_WARNING, "%s: Error closing header descriptor: %m", data->qid);
	cleanup(ctx);
	DEBUG_EXIT("eoh", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    data->headerFD = -1;
    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    data->fd = get_fd(data, "INPUTMSG", data->fd);
    if (data->fd < 0) {
	cleanup(ctx);
	DEBUG_EXIT("eoh", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    writestr(data->fd, "\n");
    data->fd = put_fd(data->fd);
    DEBUG_EXIT("eoh", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: body
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* text -- a chunk of text from the mail body
* len -- length of chunk
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes a chunk of the body to the temporary file
***********************************************************************/
static sfsistat
body(SMFICTX *ctx, u_char *text, size_t len)
{
    struct privdata *data = DATA;

    unsigned char buf[4096];
    int nsaved = 0;

    DEBUG_ENTER("body", __LINE__);

    if (!data) {
	syslog(LOG_WARNING, "body: Unable to obtain private data from milter context");
	DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Write to file and scan body for suspicious characters */
    if (len) {
	u_char *s = text;
	size_t n;

	/* If last was CR, and this is not LF, suspicious! */
	if (data->lastWasCR && *text != '\n') {
	    data->suspiciousBody = 1;
	}

	data->lastWasCR = 0;
	data->fd = get_fd(data, "INPUTMSG", data->fd);
	if (data->fd < 0) {
	    cleanup(ctx);
	    DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}

	for (n=0; n<len; n++, s++) {
	    if (*s == '\r') {
		if (n == len-1) {
		    data->lastWasCR = 1;
		} else if (*(s+1) != '\n') {
		    data->suspiciousBody = 1;
		}
		continue;
	    }

	    /* Write char */
	    if (nsaved == sizeof(buf)) {
		if (writen(data->fd, buf, nsaved) < 0) {
		    syslog(LOG_WARNING, "%s: writen failed: %m line %d",
			   data->qid, __LINE__);
		    cleanup(ctx);
		    DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
		    return SMFIS_TEMPFAIL;
		}
		nsaved = 0;
	    }
	    buf[nsaved++] = *s;
	    /* Embedded NULL's are cause for concern */
	    if (!*s) {
		data->suspiciousBody = 1;
	    }
	}
	/* Flush buffer */
	if (nsaved) {
	    if (writen(data->fd, buf, nsaved) < 0) {
		syslog(LOG_WARNING, "%s: writen failed: %m line %d",
		       data->qid, __LINE__);
		cleanup(ctx);
		DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	}
	data->fd = put_fd(data->fd);
    }

    DEBUG_EXIT("body", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: eom
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* This is where all the action happens.  Called at end of message, it
* runs the Perl scanner which may or may not ask for the body to be
* replaced.
***********************************************************************/
static sfsistat
eom(SMFICTX *ctx)
{
    char buffer[SMALLBUF];
    char result[SMALLBUF];
    char *rbuf, *rptr, *eptr;

    int seen_F = 0;
    int res_fd;
    int n;
    struct privdata *data = DATA;
    int r;
    int problem = 0;
    int fd;
    int j;
    unsigned char chunk[CHUNK];
    char *hdr, *val, *count;
    char *code, *dsn, *reply;
    struct stat statbuf;

    struct timeval start, finish;

    DEBUG_ENTER("eom", __LINE__);
    if (LogTimes) {
	gettimeofday(&start, NULL);
    }

    /* Close output file */
    if (!data) {
	syslog(LOG_WARNING, "eom: Unable to obtain private data from milter context");
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    data->cmdFD = get_fd(data, "COMMANDS", data->cmdFD);
    if (data->cmdFD < 0) {
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    /* Signal suspicious body chars */
    if (data->suspiciousBody) {
	write_mx_command(data->cmdFD, '?', NULL);
    }

    /* Signal end of command file */
    write_mx_command(data->cmdFD, 'F', NULL);

    /* All the fd's are closed unconditionally -- no need for put_fd */
    if (data->fd >= 0       && (closefd(data->fd) < 0))       problem = 1;
    if (data->headerFD >= 0 && (closefd(data->headerFD) < 0)) problem = 1;
    if (data->cmdFD >= 0    && (closefd(data->cmdFD) < 0))    problem = 1;
    data->fd = -1;
    data->headerFD = -1;
    data->cmdFD = -1;

    if (problem) {
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    /* Run the filter */
    if (MXScanDir(MultiplexorSocketName, data->dir) < 0) {
	data->filterFailed = 1;
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Read the results file */
    snprintf(buffer, SMALLBUF, "%s/RESULTS", data->dir);
    res_fd = open(buffer, O_RDONLY);
    if (res_fd < 0) {
	syslog(LOG_WARNING, "%s: Filter did not create RESULTS file", data->qid);
	data->filterFailed = 1;
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Slurp in the entire RESULTS file in one go... */
    if (fstat(res_fd, &statbuf) < 0) {
	syslog(LOG_WARNING, "%s: Unable to stat RESULTS file: %m", data->qid);
	closefd(res_fd);
	cleanup(ctx);
	data->filterFailed = 1;
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* If file is unreasonable big, forget it! */
    if (statbuf.st_size > BIGBUF - 1) {
	syslog(LOG_WARNING, "%s: RESULTS file is unreasonably large - %ld byes; max is %d bytes",
	       data->qid, (long) statbuf.st_size, BIGBUF-1);
	closefd(res_fd);
	cleanup(ctx);
	data->filterFailed = 1;
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* RESULTS files are typically pretty small and will fit into our */
    /* SMALLBUF-sized buffer.  However, we'll allocate up to BIGBUF bytes */
    /* for weird, large RESULTS files. */

    if (statbuf.st_size < SMALLBUF) {
	rbuf = result;
    } else {
	rbuf = malloc(statbuf.st_size + 1);
	if (!rbuf) {
	    syslog(LOG_WARNING, "%s: Unable to allocate memory for RESULTS data", data->qid);
	    closefd(res_fd);
	    cleanup(ctx);
	    DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    }

    /* Slurp in the file */
    n = readn(res_fd, rbuf, statbuf.st_size);
    if (n < 0) {
	syslog(LOG_WARNING, "%s: Error reading RESULTS file: %m", data->qid);
	closefd(res_fd);
	if (rbuf != result) free(rbuf);
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Done with descriptor -- close it. */
    closefd(res_fd);
    rbuf[n] = 0;

    /* Process the commands in the results file */
    for (rptr = rbuf, eptr = rptr ; rptr && *rptr; rptr = eptr) {
	/* Get end-of-line character */
	while (*eptr && (*eptr != '\n')) {
	    eptr++;
	}
	/* Check line length */
	if (eptr - rptr >= SMALLBUF-1) {
	    syslog(LOG_WARNING, "%s: Overlong line in RESULTS file - %d chars (max %d)",
		   data->qid, eptr - rptr, SMALLBUF-1);
	    cleanup(ctx);
	    r = SMFIS_TEMPFAIL;
	    goto bail_out;
	}

	if (*eptr == '\n') {
	    *eptr = 0;
	    eptr++;
	} else {
	    eptr = NULL;
	}

	switch(*rptr) {
	case 'B':
	    /* Bounce */
	    syslog(LOG_DEBUG, "%s: Bouncing because filter instructed us to",
		   data->qid);
	    split_on_space3(rptr+1, &code, &dsn, &reply);
	    percent_decode((unsigned char *) code);
	    percent_decode((unsigned char *) dsn);
	    percent_decode((unsigned char *) reply);

	    MD_SMFI_TRY(set_reply, (ctx, "5", code, dsn, reply));
	    cleanup(ctx);
	    r = SMFIS_REJECT;
	    goto bail_out;

	case 'D':
	    /* Discard */
	    syslog(LOG_DEBUG, "%s: Discarding because filter instructed us to",
		   data->qid);
	    cleanup(ctx);
	    r = SMFIS_DISCARD;
	    goto bail_out;

	case 'T':
	    /* Tempfail */
	    syslog(LOG_DEBUG, "%s: Tempfailing because filter instructed us to",
		   data->qid);
	    split_on_space3(rptr+1, &code, &dsn, &reply);
	    percent_decode((unsigned char *) code);
	    percent_decode((unsigned char *) dsn);
	    percent_decode((unsigned char *) reply);

	    MD_SMFI_TRY(set_reply, (ctx, "4", code, dsn, reply));

	    cleanup(ctx);
	    r = SMFIS_TEMPFAIL;
	    goto bail_out;

	case 'C':
	    snprintf(buffer, SMALLBUF, "%s/NEWBODY", data->dir);
	    fd = open(buffer, O_RDONLY);
	    if (fd < 0) {
		syslog(LOG_WARNING, "%s: Could not open %s for reading: %m",
		       data->qid, buffer);
		closefd(fd);
		cleanup(ctx);
		data->filterFailed = 1;
		r = SMFIS_TEMPFAIL;
		goto bail_out;
	    }
	    while ((j=read(fd, chunk, CHUNK)) > 0) {
		MD_SMFI_TRY(smfi_replacebody, (ctx, chunk, j));
	    }
	    close(fd);
	    break;

	case 'M':
	    /* New content-type header */
	    percent_decode((unsigned char *) rptr+1);
	    if (strlen(rptr+1) > 0) {
		MD_SMFI_TRY(smfi_chgheader, (ctx, "Content-Type", 1, rptr+1));
	    }
	    MD_SMFI_TRY(smfi_chgheader, (ctx, "MIME-Version", 1, "1.0"));
	    break;

	case 'H':
	    /* Add a header */
	    split_on_space(rptr+1, &hdr, &val);
	    if (hdr && val) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) val);
		MD_SMFI_TRY(smfi_addheader, (ctx, hdr, val));
	    }
	    break;

	case 'I':
	    /* Change a header */
	    split_on_space3(rptr+1, &hdr, &count, &val);
	    if (hdr && val && count) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) count);
		percent_decode((unsigned char *) val);
		if (sscanf(count, "%d", &j) != 1 || j < 1) {
		    j = 1;
		}
		MD_SMFI_TRY(smfi_chgheader, (ctx, hdr, j, val));
	    }
	    break;

	case 'J':
	    /* Delete a header */
	    split_on_space(rptr+1, &hdr, &count);
	    if (hdr && count) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) count);
		if (sscanf(count, "%d", &j) != 1 || j < 1) {
		    j = 1;
		}
		MD_SMFI_TRY(smfi_chgheader, (ctx, hdr, j, NULL));
	    }
	    break;

	case 'R':
	    /* Add a recipient */
	    percent_decode((unsigned char *) rptr+1);
	    MD_SMFI_TRY(smfi_addrcpt, (ctx, rptr+1));
	    break;

	case 'Q':
	    /* Quarantine a message using Sendmail's facility */
	    percent_decode((unsigned char *) rptr+1);
	    MD_SMFI_TRY(do_sm_quarantine, (ctx, rptr+1));
	    break;

	case 'S':
	    /* Delete a recipient */
	    percent_decode((unsigned char *) rptr+1);
	    MD_SMFI_TRY(smfi_delrcpt, (ctx, rptr+1));
	    break;

	case 'F':
	    seen_F = 1;
	    /* We're done */
	    break;

	default:
	    syslog(LOG_WARNING, "%s: Unknown command '%c' in RESULTS file",
		   data->qid, *rptr);
	}
	if (*rptr == 'F') break;
    }

    if (!seen_F) {
	syslog(LOG_ERR, "%s: RESULTS file did not finish with 'F' line: Tempfailing",
	       data->qid);
	r = SMFIS_TEMPFAIL;
	goto bail_out;
    }
    if (scan_body && *scan_body) {
	if (data->myip) {
	    snprintf(buffer, SMALLBUF, "%s on %s", scan_body, data->myip);
	    buffer[SMALLBUF-1] = 0;
	    MD_SMFI_TRY(smfi_addheader, (ctx, "X-Scanned-By", buffer));
	} else {
	    MD_SMFI_TRY(smfi_addheader, (ctx, "X-Scanned-By", scan_body));
	}
    }

    /* Delete first validation header if it was present */
    if (ValidateHeader[0] && data->validatePresent) {
	MD_SMFI_TRY(smfi_chgheader, (ctx, ValidateHeader, 1, NULL));
    }

    /* Delete any excess Content-Type headers and log */
    if (data->numContentTypeHeaders > 1) {
	syslog(LOG_WARNING, "%s: WARNING: %d Content-Type headers found -- deleting all but first", data->qid, data->numContentTypeHeaders);
	for (j=2; j<=data->numContentTypeHeaders; j++) {
	    MD_SMFI_TRY(smfi_chgheader, (ctx, "Content-Type", j, NULL));
	}
    }

    r = cleanup(ctx);

  bail_out:
    if (rbuf != result) free(rbuf);

    if (LogTimes) {
	long sec_diff, usec_diff;

	gettimeofday(&finish, NULL);

	sec_diff = finish.tv_sec - start.tv_sec;
	usec_diff = finish.tv_usec - start.tv_usec;

	if (usec_diff < 0) {
	    usec_diff += 1000000;
	    sec_diff--;
	}

	/* Convert to milliseconds */
	sec_diff = sec_diff * 1000 + (usec_diff / 1000);
	syslog(LOG_INFO, "%s: Filter time is %ldms", data->qid, sec_diff);
    }

    DEBUG(syslog(LOG_INFO, "Exiting eom (line %d) ret=%d", __LINE__, r));
    return r;
}

/**********************************************************************
*%FUNCTION: mfclose
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_ACCEPT
*%DESCRIPTION:
* Called when connection is closed.
***********************************************************************/
static sfsistat
mfclose(SMFICTX *ctx)
{
    struct privdata *data = DATA;

    DEBUG_ENTER("mfclose", __LINE__);
    cleanup(ctx);
    if (data) {
	if (data->fd >= 0)       closefd(data->fd);
	if (data->headerFD >= 0) closefd(data->headerFD);
	if (data->cmdFD >= 0)    closefd(data->cmdFD);
	if (data->dir)           free(data->dir);
	if (data->hostname)      free(data->hostname);
	if (data->hostip)        free(data->hostip);
	if (data->myip && data->myip != MyIPAddress) free(data->myip);
	if (data->sender)        free(data->sender);
	if (data->firstRecip)    free(data->firstRecip);
	if (data->heloArg)       free(data->heloArg);
	if (data->qid && data->qid != NOQUEUE) free(data->qid);
	free(data);
    }
    smfi_setpriv(ctx, NULL);
    DEBUG_EXIT("mfclose", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: mfabort
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Called if current message is aborted.  Just cleans up.
***********************************************************************/
static sfsistat
mfabort(SMFICTX *ctx)
{
    return cleanup(ctx);
}

/**********************************************************************
*%FUNCTION: cleanup
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Cleans up temporary files.
***********************************************************************/
static sfsistat
cleanup(SMFICTX *ctx)
{
    sfsistat r = SMFIS_CONTINUE;
    struct privdata *data = DATA;

    DEBUG_ENTER("cleanup", __LINE__);
    if (!data) {
	DEBUG_EXIT("cleanup", __LINE__, "SMFIS_CONTINUE");
	return r;
    }

    if (data->fd >= 0 && (closefd(data->fd) < 0)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->fd = -1;

    if (data->headerFD >= 0 && (closefd(data->headerFD) < 0)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->headerFD = -1;

    if (data->cmdFD >= 0 && (closefd(data->cmdFD) < 0)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->cmdFD = -1;

    remove_working_directory(data);

    if (data->dir) {
	free(data->dir);
	data->dir = NULL;
    }
    if (data->sender) {
	free(data->sender);
	data->sender = NULL;
    }
    if (data->firstRecip) {
	free(data->firstRecip);
	data->firstRecip = NULL;
    }

    /* Do NOT free qid here; we need it for logging filter times */

    DEBUG(syslog(LOG_INFO, "Exiting cleanup (line %d) ret=%s", __LINE__, (r == SMFIS_TEMPFAIL ? "SMFIS_TEMPFAIL" : "SMFIS_CONTINUE")));
    return r;
}

static struct smfiDesc filterDescriptor =
{
    "MIMEDefang-" VERSION,      /* Filter name */
    SMFI_VERSION,		/* Version code */

#if SMFI_VERSION == 2
#ifdef SMFIF_QUARANTINE
    /* We can: quarantine, add a header and may alter body
       and add/delete recipients*/
    SMFIF_ADDHDRS|SMFIF_CHGBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT|SMFIF_CHGHDRS|SMFIF_QUARANTINE,
#else
    /* We can: Add headers, alter body, add/delete recipients, alter headers */
    SMFIF_ADDHDRS|SMFIF_CHGBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT|SMFIF_CHGHDRS,
#endif
#elif SMFI_VERSION == 1
    /* We can: add a header and may alter body and add/delete recipients*/
    SMFIF_MODHDRS|SMFIF_MODBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT,
#endif

    mfconnect,			/* connection */
    helo,			/* HELO */
    envfrom,			/* MAIL FROM: */
    rcptto,			/* RCPT TO: */
    header,			/* Called for each header */
    eoh,			/* Called at end of headers */
    body,			/* Called for each body chunk */
    eom,			/* Called at end of message */
    mfabort,			/* Called on abort */
    mfclose			/* Called on connection close */
};

/**********************************************************************
* %FUNCTION: usage
* %ARGUMENTS:
*  None
* %RETURNS:
*  Nothing (exits)
* %DESCRIPTION:
*  Prints usage information
***********************************************************************/
static void
usage(void)
{
    fprintf(stderr, "mimedefang version %s\n", VERSION);
    fprintf(stderr, "Usage: mimedefang [options]\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "  -h                -- Print usage info and exit\n");
    fprintf(stderr, "  -v                -- Print version and exit\n");
    fprintf(stderr, "  -U user           -- Run as user instead of root\n");
    fprintf(stderr, "  -p /path          -- Path to UNIX-domain socket for sendmail communication\n");
    fprintf(stderr, "  -d                -- Enable debugging (do not remove spool files)\n");
    fprintf(stderr, "  -k                -- Do not remove spool files if filter fails\n");
    fprintf(stderr, "  -m /path          -- Use multiplexor; use /path as UNIX-domain socket\n");
    fprintf(stderr, "  -r                -- Do relay check before processing body (requires -m)\n");
    fprintf(stderr, "  -s                -- Do sender check before processing body (requires -m)\n");
    fprintf(stderr, "  -t                -- Do recipient checks before processing body (requires -m)\n");
    fprintf(stderr, "  -P file           -- Write process-ID of daemon to specified file\n");
    fprintf(stderr, "  -T                -- Log filter times to syslog\n");
    fprintf(stderr, "  -b n              -- Set listen() backlog to n\n");
    fprintf(stderr, "  -C                -- Try very hard to conserve file descriptors\n");
    fprintf(stderr, "  -x string         -- Add string as X-Scanned-By header\n");
    fprintf(stderr, "  -X                -- Do not add X-Scanned-By header\n");
    fprintf(stderr, "  -M                -- Protect mkdir with mutex\n");
    fprintf(stderr, "  -D                -- Do not become a daemon (stay in foreground)\n");
    fprintf(stderr, "  -S facility       -- Set syslog(3) facility\n");
    fprintf(stderr, "  -a macro          -- Pass additional Sendmail macro\n");
    exit(EXIT_FAILURE);
}

/**********************************************************************
* %FUNCTION: main
* %ARGUMENTS:
*  argc, argv -- the usual suspects
* %RETURNS:
*  Whatever smfi_main returns
* %DESCRIPTION:
*  Main program
***********************************************************************/
int
main(int argc, char **argv)
{
    int c;
    int mx_alive;
    pid_t i;
    char *pidfile = NULL;
    struct passwd *pw = NULL;
    FILE *fp;
    int facility = LOG_MAIL;
    int nodaemon = 0;
    char buf[SMALLBUF];

    /* Paranoia time */
    umask(027);

    /* Paranoia time II */
    if (getuid() != geteuid()) {
	fprintf(stderr, "ERROR: %s is NOT intended to run suid! Exiting.\n",
		argv[0]);
	exit(EXIT_FAILURE);
    }

    if (getgid() != getegid()) {
	fprintf(stderr, "ERROR: %s is NOT intended to run sgid! Exiting.\n",
		argv[0]);
	exit(EXIT_FAILURE);
    }

    MyIPAddress = NULL;

    /* Determine my IP address */
    if (gethostname(buf, sizeof(buf)) >= 0) {
	struct hostent *he = gethostbyname(buf);
	struct in_addr in;
	if (he && he->h_addr) {
	    memcpy(&in.s_addr, he->h_addr, sizeof(in.s_addr));
#ifdef HAVE_INET_NTOP
	    if (inet_ntop(AF_INET, &in.s_addr, buf, sizeof(buf))) {
		if (*buf) MyIPAddress = strdup_with_log(buf);
	    }
#else
	    {
		char *s = inet_ntoa(in.sin_addr);
		if (s && *s) MyIPAddress = strdup_with_log(s);
	    }
#endif
	} else {
	    syslog(LOG_WARNING, "Could not determine my own IP address!  Ensure that %s has an entry in /etc/hosts or the DNS", buf);
	    fprintf(stderr, "Could not determine my own IP address!  Ensure that %s has an entry in /etc/hosts or the DNS\n", buf);
	}
    }

    /* Process command line options */
    while ((c = getopt(argc, argv, "a:Chp:dm:srtkP:U:Tx:MXS:Dvb:")) != -1) {
	switch (c) {
	case 'b':
	    sscanf(optarg, "%d", &Backlog);
	    if (Backlog < 5) Backlog = 5;
	    break;
	case 'C':
	    ConserveDescriptors = 1;
	    break;

	case 'v':
	    printf("mimedefang version %s\n", VERSION);
	    exit(0);

	case 'D':
	    nodaemon = 1;
	    break;
	case 'a':
	    if (strlen(optarg) > 200) {
		fprintf(stderr, "%s: Macro name too long: %s\n",
			argv[0], optarg);
		exit(EXIT_FAILURE);
	    }
	    if (NumAdditionalMacros == MAX_ADDITIONAL_SENDMAIL_MACROS) {
		fprintf(stderr, "%s: Too many Sendmail macros (max %d)\n",
			argv[0],
			MAX_ADDITIONAL_SENDMAIL_MACROS);
		exit(EXIT_FAILURE);
	    }
	    AdditionalMacros[NumAdditionalMacros] = strdup(optarg);
	    if (!AdditionalMacros[NumAdditionalMacros]) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    NumAdditionalMacros++;
	    break;
	case 'S':
	    facility = find_syslog_facility(optarg);
	    if (facility < 0) {
		fprintf(stderr, "%s: Unknown syslog facility %s\n",
			argv[0], optarg);
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'M':
	    protectMkdirWithMutex = 1;
	    break;
	case 'X':
	    if (scan_body) {
		free(scan_body);
	    }
	    scan_body = strdup("");
	    if (!scan_body) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    break;

	case 'x':
	    if (scan_body) {
		free(scan_body);
	    }
	    scan_body = strdup(optarg);
	    if (!scan_body) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    break;

	case 'T':
	    LogTimes = 1;
	    break;

	case 'U':
	    /* User to run as */
	    if (user) {
		free(user);
	    }
	    user = strdup(optarg);
	    if (!user) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'P':
	    /* Write our pid to this file */
	    if (pidfile != NULL) free(pidfile);

	    pidfile = strdup(optarg);
	    if (!pidfile) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'k':
	    keepFailedDirectories = 1;
	    break;
	case 's':
	    doSenderCheck = 1;
	    break;
	case 'r':
	    doRelayCheck = 1;
	    break;
	case 't':
	    doRecipientCheck = 1;
	    break;
	case 'h':
	    usage();
	    break;
	case 'm':
	    /* Multiplexor */
	    MultiplexorSocketName = strdup(optarg);
	    if (!MultiplexorSocketName) {
		fprintf(stderr, "%s: Out of memory\n", argv[0]);
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'd':
	    DebugMode = 1;
	    break;

	case 'p':
	    if (optarg == NULL || *optarg == '\0') {
		(void) fprintf(stderr, "%s: Illegal conn: %s\n",
			       argv[0], optarg);
		exit(EXIT_FAILURE);
	    }
	    /* Remove socket from file system */
	    (void) remove(optarg);
	    if (smfi_setconn(optarg) != MI_SUCCESS) {
		fprintf(stderr, "%s: Could not open connection %s: %s",
			argv[0], optarg, strerror(errno));
		exit(EXIT_FAILURE);
	    }
	    break;
	default:
	    usage();
	    break;
	}
    }

    if (!scan_body) {
	scan_body = SCAN_BODY;
    }

    if (Backlog > 0) {
	smfi_setbacklog(Backlog);
    }

    if (!MultiplexorSocketName) {
	fprintf(stderr, "%s: You must use the '-m' option.\n", argv[0]);
	exit(EXIT_FAILURE);
    }

    /* Look up user */
    if (user) {
	pw = getpwnam(user);
	if (!pw) {
	    fprintf(stderr, "%s: Unknown user '%s'", argv[0], user);
	    exit(EXIT_FAILURE);
	}
	if (drop_privs(user, pw->pw_uid, pw->pw_gid) < 0) {
	    fprintf(stderr, "%s: Could not drop privileges: %s",
		    argv[0], strerror(errno));
	    exit(EXIT_FAILURE);
	}
	free(user);
    }

    /* Warn */
    if (!getuid() || !geteuid()) {
	fprintf(stderr,
		"ERROR: You must not run mimedefang as root.\n"
		"Use the -U option to set a non-root user.\n");
	exit(EXIT_FAILURE);
    }


    if (chdir(SPOOLDIR) < 0) {
	fprintf(stderr, "%s: Unable to chdir(%s): %s\n",
		argv[0], SPOOLDIR, strerror(errno));
	exit(EXIT_FAILURE);
    }

    /* Read key file if present */
    fp = fopen(KEY_FILE, "r");
    if (fp) {
	fgets(ValidateHeader, sizeof(ValidateHeader), fp);
	fclose(fp);
	chomp(ValidateHeader);
    } else {
	ValidateHeader[0] = 0;
    }
    if (smfi_register(filterDescriptor) == MI_FAILURE) {
	fprintf(stderr, "%s: smfi_register failed\n", argv[0]);
	exit(EXIT_FAILURE);
    }

    /* Daemonize */
    if (!nodaemon) {
	i = fork();
	if (i < 0) {
	    fprintf(stderr, "%s: fork() failed\n", argv[0]);
	    exit(EXIT_FAILURE);
	} else if (i != 0) {
	    /* parent */
	    exit(EXIT_SUCCESS);
	}
	setsid();
	signal(SIGHUP, SIG_IGN);
	i = fork();
	if (i < 0) {
	    fprintf(stderr, "%s: fork() failed\n", argv[0]);
	    exit(EXIT_FAILURE);
	} else if (i != 0) {
	    exit(EXIT_SUCCESS);
	}
    }

    /* Write pid */
    if (pidfile) {
	FILE *fp = fopen(pidfile, "w");
	if (fp) {
	    fprintf(fp, "%d\n", (int) getpid());
	    fclose(fp);
	}
	free(pidfile);
    }

    (void) closelog();
    closefiles();

    /* Direct stdin/stdout/stderr to /dev/null */
    open("/dev/null", O_RDWR);
    open("/dev/null", O_RDWR);
    open("/dev/null", O_RDWR);

    openlog("mimedefang", LOG_PID, facility);

    if (ValidateHeader[0]) {
	syslog(LOG_DEBUG, "IP validation header is %s", ValidateHeader);
    }
#ifdef ENABLE_DEBUGGING
    signal(SIGSEGV, handle_sig);
    signal(SIGBUS, handle_sig);
#endif

    /* Wait for the multiplexor to come alive */
    mx_alive = 0;
    for (c=0; c<50; c++) {
	if (MXCheckFreeSlaves(MultiplexorSocketName) >= 0) {
	    mx_alive = 1;
	    break;
	}
	sleep(3);
    }
    if (mx_alive) {
	syslog(LOG_INFO, "Multiplexor alive - entering main loop");
    } else {
	syslog(LOG_WARNING, "Multiplexor unresponsive - entering main loop anyway");
    }
    return smfi_main();
}

/**********************************************************************
* %FUNCTION: write_macro_value
* %ARGUMENTS:
*  ctx -- Sendmail milter context
*  macro -- name of a macro
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Sends a command to Perl code to set a macro value
***********************************************************************/
static void
write_macro_value(SMFICTX *ctx,
		  char *macro)
{
    struct privdata *data;
    char *val;
    char buf[256];

    data = DATA;
    if (!data || !data->cmdFD) return;

    if (*macro && *(macro+1)) {
	/* Longer than 1 char -- use curlies */
	snprintf(buf, sizeof(buf), "{%s}", macro);
	val = smfi_getsymval(ctx, buf);
    } else {
	val = smfi_getsymval(ctx, macro);
    }
    if (!val) return;
    writestr(data->cmdFD, "=");
    write_percent_encoded((unsigned char *)macro, data->cmdFD);
    writestr(data->cmdFD, " ");
    write_percent_encoded((unsigned char *)val, data->cmdFD);
    writestr(data->cmdFD, "\n");
}

/**********************************************************************
* %FUNCTION: remove_working_directory
* %ARGUMENTS:
*  data -- our private data
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Removes working directory if appropriate.
***********************************************************************/
static void
remove_working_directory(struct privdata *data)
{
    if (!data || !data->dir || !*(data->dir)) return;

    /* Don't remove if in debug mode or various other reasons */
    if (DebugMode) {
	syslog(LOG_INFO, "%s: Not cleaning up %s because of command-line '-d' flag",
	       data->qid,
	       data->dir);
	return;
    }

    if (access(NO_DELETE_DIR, F_OK) == 0) {
	syslog(LOG_INFO, "%s: Not cleaning up %s because of %s",
	       data->qid,
	       (data->dir ? data->dir : ""),
	       NO_DELETE_DIR);
	return;
    }

    if (keepFailedDirectories && data->filterFailed) {
	syslog(LOG_WARNING, "%s: Filter failed.  Message kept in %s",
	       data->qid, data->dir);
	return;
    }

    if (rm_r(data->dir) < 0) {
	syslog(LOG_ERR, "%s: failed to clean up %s: %m",
	       data->qid, data->dir);
    }
}

/**********************************************************************
* %FUNCTION: set_dsn
* %ARGUMENTS:
*  data -- our private data area
*  ctx -- Milter context
*  buf2 -- return from a relay/sender/filter check.  Consists of
*    space-separated "reply code dsn sleep_amount" list.
*  num -- 0, 4 or 5 -- if 4 or 5, we use the code and dsn in set_reply.
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Sets SMTP reply and possibly delays
***********************************************************************/
static void
set_dsn(SMFICTX *ctx, char *buf2, int num) {
    char *reply, *code, *dsn, *sleepstr;

    if (*buf2) {
	split_on_space4(buf2, &reply, &code, &dsn, &sleepstr);
	percent_decode((unsigned char *) code);
	percent_decode((unsigned char *) dsn);
	percent_decode((unsigned char *) reply);
	percent_decode((unsigned char *) sleepstr);
	do_delay(sleepstr);
	if (num == 4 || num == 5) {
	    struct privdata *data = DATA;
	    if (num == 5) {
		MD_SMFI_TRY(set_reply, (ctx, "5", code, dsn, reply));
	    } else {
		MD_SMFI_TRY(set_reply, (ctx, "4", code, dsn, reply));
	    }
	}
    }
}

/**********************************************************************
* %FUNCTION: do_sm_quarantine
* %ARGUMENTS:
*  ctx -- Milter context
*  reason -- reason for quarantine
* %RETURNS:
*  Whatever smfi_quarantine returns
* %DESCRIPTION:
*  Quarantines a message using Sendmail's quarantine facility, if supported.
***********************************************************************/
static int
do_sm_quarantine(SMFICTX *ctx,
		 char const *reason)
{
#ifdef SMFIF_QUARANTINE
    return smfi_quarantine(ctx, reason);
#else
    syslog(LOG_WARNING, "smfi_quarantine not supported: Requires Sendmail 8.13.0 or later");
    return MI_FAILURE;
#endif

}
