/*======================================================================*\
|*		Editor mined						*|
|*		display output functions				*|
\*======================================================================*/

#include "mined.h"
#include "io.h"
#include "pc437.map"

#include <errno.h>


/*======================================================================*\
|*			Control character indication			*|
\*======================================================================*/

int
visciimode ()
{
  return mapped_text == True;
}

int
isviscii (c)
  character c;
{
  return visciimode () && (c < ' ' || c >= 0x80);
}

int
iscontrol (c)
  character c;
{
  if (visciimode ()) {
	return (c == '\177') 
		|| (c < ' ' && c != '' && c != '' && c != ''
			    && c != '' && c != '' && c != '');
  }

#ifdef pc_charset
  return (c == '\177') 
		|| (c < ' ');
#else
  if (utf8_text == True || cjk_text == True) {
	return (c == '\177') 
		|| (c < ' ');
  } else {
	return (c == '\177') 
		|| ((c & '\177') < ' ');
  }
#endif
}

character
controlchar (c)
  character c;
{
  if ((c) == '\177') {
	return '?';
  } else {
	return (c) + '@';
  }
}


/*======================================================================*\
|*			Scroll bar display				*|
\*======================================================================*/

/* mined displays last_y + 1 lines in a display area of YMAX lines */
/* the file has (approx.?) total_lines lines */
/* the file position was in line # line_number */

/* debug traces: */
#define dont_debug_scrollbar
#define dont_debug_scrollbar_calc
#define dont_debug_partial_scroll
#define dont_debug_dirty
/* visible debug scrollbar: */
#define dont_debug_lazy_scrollbar

static int prev_disp_start = 0;
static int prev_disp_end = 0;

static FLAG scrollbar_dirty = True;	/* does scrollbar need refresh ? */
static int first_dirty = -1;
static int last_dirty = -1;

#ifdef debug_dirty
#define printf_dirty(s, n)	printf ("%s %d - (prev %d..%d) [%d dirty %d..%d]\n", s, n, prev_disp_start, prev_disp_end, scrollbar_dirty, first_dirty, last_dirty);
#else
#define printf_dirty(s, n)	
#endif


static
void
set_scrollbar_dirty (scry)
  int scry;
{
  scrollbar_dirty = True;
  if (scry < first_dirty || first_dirty < 0) {
	first_dirty = scry;
  }
  if (scry > last_dirty) {
	last_dirty = scry;
  }
#ifdef debug_partial_scroll
	printf ("scrollbar_dirty @ %d, -> dirty %d..%d\n", scry, first_dirty, last_dirty);
#endif
	printf_dirty ("set_dirty", scry);
}

void
scrollbar_scroll_up (from)
  int from;
{
	int unit = utf8_screen == True && fine_scrollbar == True ? 8 : 1;

	prev_disp_start -= unit;
	prev_disp_end -= unit;
	set_scrollbar_dirty (SCREENMAX);
#ifdef debug_partial_scroll
	printf ("scrollbar_scroll_up from %d, -> prev %d..%d\n", from, prev_disp_start, prev_disp_end);
#endif
	printf_dirty ("scroll_up", from);
}

void
scrollbar_scroll_down (from)
  int from;
{
	int unit = utf8_screen == True && fine_scrollbar == True ? 8 : 1;

	prev_disp_start += unit;
	prev_disp_end += unit;
	set_scrollbar_dirty (from);
#ifdef debug_partial_scroll
	printf ("scrollbar_scroll_down from %d, -> prev %d..%d\n", from, prev_disp_start, prev_disp_end);
#endif
	printf_dirty ("scroll_down", from);
}

static
FLAG
fine_grained_scrollbar (force)
  FLAG force;
{
/* calculate scroll bar display using Unicode 
   character cell vertical eighth blocks U+2581..U+2587 ▁▂▃▄▅▆▇
   as follows:
 * screen has lines 0...last_y, screen_lines = last_y + 1,
   with fields = screen_lines * 8,
   e.g. 10 lines 0...9, 80 fields 0...79;
 * to be mapped to buffer line numbers 1...total_lines;
   scroll block size in fields is s / fields = screen_lines / total_lines,
   s = screen_lines * fields / total_lines, 
   minimum 1 (display) / 7 (Unicode) / 8 (algorithm),
   max_start = fields - s;
 * scroll block start position is in range 0...(80 - s),
   to be mapped to position of last screen line 
   last_line_number = (line_number - y + last_y):
   if last_line_number <= screen_lines ==> p = 0
   if last_line_number == total_lines ==> p = max_start
   so roughly:
   p / max_start
   	= (last_line_number - screen_lines) 
   	  / (total_lines - screen_lines);
   but to accomodate rounding problems and avoid zero division map range
   last_line_number = screen_lines + 1 ==> p = start1,
   last_line_number = total_lines - 1 ==> p = max_start - 1,
   where start1 corresponds to the delta caused by scrolling by 1 line;
   start1 / fields = 1 / (total_lines - screen_lines), min 1,
   but replace this with an adjustment (see below) for better computation;
 - distribute equally, so map
   (max_start - start1) fields to (total_lines - 1 - screen_lines) lines
   where both will taken to zero by an offset for scaling:
   fields start1 ... max_start-1 
   	==> consider result to be p - start1
   lines screen_lines + 1 ... total_lines - 1 
   	==> consider ref_last/total = last/total_line_number - (screen_lines + 1):
   p - start1 / max_start
   	= (last_line_number - screen_lines - 1) / (total_line_number - screen_lines - 1)
 * scroll block extends from p to p + s - 1
*/

  int unit = utf8_screen == True && fine_scrollbar == True ? 8 : 1;

  int screen_lines;
  long fields;
  int s;
  int max_start;
  long last_line_number;
  int disp_start;
  int disp_end;
  int i;
  int scri;
  FLAG painted = False;
  int slices = 0;
  int oldslices = 0;
  FLAG klops_started = False;
  FLAG oldklops_started = False;

  screen_lines = last_y + 1;
  fields = screen_lines * unit;
  s = fields * screen_lines / total_lines;
  if (s < unit) {
	s = unit;
  }
  max_start = fields - s;
  last_line_number = line_number - y + last_y;
  if (last_line_number <= screen_lines) {
	disp_start = 0;
  } else if (last_line_number == total_lines) {
	disp_start = max_start;
  } else {
	/* compensate by + 1 for adjustment at the beginning */
	disp_start = max_start * 
			(last_line_number - screen_lines - 1 + 1) 
			/ (total_lines - screen_lines - 1);
	/* assure distance if not quite at beginning or end */
	if (disp_start == 0) {
		disp_start = 1;
	} else if (disp_start >= max_start) {
		disp_start = max_start - 1;
	}
  }
  disp_end = disp_start + s - 1;

#ifdef debug_scrollbar_calc
	printf ("last_line_number %d/%d, screen_lines %d (0-%d)\n", last_line_number, total_lines, screen_lines, last_y);
	printf (" size %d, maxstart %d, ~ * %d / %d, pos %d-%d/%d\n", 
		s, max_start, 
		last_line_number - screen_lines, total_lines - screen_lines, 
		disp_start, disp_end, fields);
#endif

  if (disp_scrollbar == True) {
#ifdef debug_scrollbar
	if (update_scrollbar_lazy == True) {
		printf ("scrollbar (%d) %d - %d (prev. %d - %d)\n", force, disp_start, disp_end, prev_disp_start, prev_disp_end);
	}
#endif

/* last_y / SCREENMAX */
	for (i = 0; i < fields; i ++) {
		if (i >= disp_start && i <= disp_end) {
			slices ++;
		}
		if (i >= prev_disp_start && i <= prev_disp_end) {
			oldslices ++;
		}
		if (((i + 1) % unit) == 0) {
		    scri = i / unit;
		    if (slices != oldslices 
			|| klops_started != oldklops_started
			|| (force == True && scri >= first_dirty && scri <= last_dirty)
		       ) {
			painted = True;
			set_cursor (XMAX, scri);
			if (slices == 0) {
				disp_scrollbar_background ();
				putchar (' ');
				disp_scrollbar_off ();
			} else if (slices == unit) {
				disp_scrollbar_foreground ();
				putchar (' ');
				disp_scrollbar_off ();
#ifdef debug_scrollbar_calc
				if (klops_started == False) {
					printf ("@ %d (%d) ", scri, i / unit * unit);
				}
				printf ("%d", unit);
#endif
			} else {
				/* choose among the eighths blocks */
				/* U+2581..U+2587 ▁▂▃▄▅▆▇ */
				if (klops_started == True) {
					disp_scrollbar_foreground ();
					put_unichar (0x2588 - slices);
					disp_scrollbar_off ();
#ifdef debug_scrollbar_calc
					printf ("%d", slices);
#endif
				} else {
					disp_scrollbar_background ();
					put_unichar (0x2580 + slices);
					disp_scrollbar_off ();
#ifdef debug_scrollbar_calc
					if (klops_started == False) {
						printf ("@ %d (%d) ", scri, i + 1 - slices);
					}
					printf ("%d", slices);
#endif
				}
			}
#ifdef debug_lazy_scrollbar
		    } else {
			painted = True;
			set_cursor (XMAX, scri);
			if (slices > 0) {
				disp_scrollbar_foreground ();
			} else {
				disp_scrollbar_background ();
			}
			putchar ('X');
			disp_scrollbar_off ();
#endif
		    }
		    if (slices > 0) {
			klops_started = True;
		    }
		    if (oldslices > 0) {
				oldklops_started = True;
		    }
		    slices = 0;
		    oldslices = 0;
		}
	}
#ifdef debug_scrollbar_calc
	printf ("\n");
#endif
  }

  printf_dirty ("scrollbar", force);
  prev_disp_start = disp_start;
  prev_disp_end = disp_end;
  scrollbar_dirty = False;
  first_dirty = SCREENMAX;
  last_dirty = -1;
  printf_dirty ("> scrollbar", force);
  return painted;
}

#ifdef use_cell_scrollbar

static
FLAG
cell_grained_scrollbar (force)
  FLAG force;
{
/* last_y / YMAX */
  int disp_start = ((long) line_number - y) * last_y / total_lines;
  int disp_end = ((long) line_number - y + last_y + 1) * last_y / total_lines;
  int i;
  FLAG painted = False;
  FLAG isin_newklops, isin_oldklops;

  if (disp_scrollbar == True) {
#ifdef debug_scrollbar
	if (update_scrollbar_lazy == True) {
		printf ("scrollbar (%d) %d - %d (prev. %d - %d)\n", force, disp_start, disp_end, prev_disp_start, prev_disp_end);
	}
#endif

/* last_y / SCREENMAX */
	for (i = 0; i <= last_y; i ++) {
	    isin_newklops = i >= disp_start && i <= disp_end;
	    isin_oldklops = i >= prev_disp_start && i <= prev_disp_end;
	    if (isin_newklops != isin_oldklops || force == True) {
		painted = True;
		set_cursor (XMAX, i);
		if (isin_newklops) {
#ifdef CJKterm_not_coloured
			if (cjk_term == True) {
				reverse_on ();
				putchar ('<');
				reverse_off ();
			} else
#endif
			{
				disp_scrollbar_foreground ();
				putchar (' ');
				disp_scrollbar_off ();
			}
		} else {
			disp_scrollbar_background ();
			putchar (' ');
			disp_scrollbar_off ();
		}
#ifdef debug_lazy_scrollbar
	    } else {
		painted = True;
		set_cursor (XMAX, i);
		if (isin_newklops) {
			disp_scrollbar_foreground ();
		} else {
			disp_scrollbar_background ();
		}
		putchar ('X');
		disp_scrollbar_off ();
#endif
	    }
	}
  }

  prev_disp_start = disp_start;
  prev_disp_end = disp_end;
  scrollbar_dirty = False;
  first_dirty = SCREENMAX;
  last_dirty = -1;
  return painted;
}

FLAG
display_scrollbar (update)
  FLAG update;
{
  if (utf8_screen == True && fine_scrollbar == True) {
	return fine_grained_scrollbar (update == False || scrollbar_dirty == True);
  } else {
	return cell_grained_scrollbar (update == False || scrollbar_dirty == True);
  }
}

#else

FLAG
display_scrollbar (update)
  FLAG update;
{
  return fine_grained_scrollbar (update == False || scrollbar_dirty == True);
}

#endif


/*======================================================================*\
|*			UTF-8 and special indication handling		*|
\*======================================================================*/

unsigned long
isolated_alef (unichar)
  unsigned long unichar;
{
	if (unichar == 0x0622) {
		/* ALEF WITH MADDA ABOVE */
		return 0xFE81;
	} else if (unichar == 0x0623) {
		/* ALEF WITH HAMZA ABOVE */
		return 0xFE83;
	} else if (unichar == 0x0625) {
		/* ALEF WITH HAMZA BELOW */
		return 0xFE87;
	} else if (unichar == 0x0627) {
		/* ALEF */
		return 0xFE8D;
	} else {
		/* ? -> ALEF SYMBOL */
		return 0x2135;
	}
}


static
void
putnarrowchar (unichar)
  unsigned long unichar;
{
  if (iswide (unichar)) {
	if (unichar == 0xB7) {
		put_unichar ('.');
	} else if (unichar == 0x2028) {
		put_unichar ('');
	} else if (unichar == 0x2029) {
		put_unichar ('P');
	} else if (unichar == (unsigned long) '') {
		put_unichar ('0');
	} else {
		put_unichar (' ');
	}
  } else {
	put_unichar (unichar);
  }
}

static
void
putnarrowutf (utfmark)
  char * utfmark;
{
  unsigned long unichar;
  int utflen;

  utf8_info (utfmark, & utflen, & unichar);
  if (iswide (unichar)) {
	putnarrowchar (unichar);
  } else {
	put_utfchar (utfmark);
  }
}


#define illegal_UTF indicate_UTF

static
void
indicate_UTF (c)
  character c;
{
  if (char_on_status_line == True) {
	reverse_off ();
  }
  unidisp_on ();
  putnarrowchar ((unsigned long) c);
  unidisp_off ();
  if (char_on_status_line == True) {
	reverse_on ();
  }
}

static
void
indicate_special (c)
  character c;
{
  if (char_on_status_line == True) {
	reverse_off ();
  }
  unimarkdisp_on ();
  putnarrowchar ((unsigned long) c);
  unimarkdisp_off ();
  if (char_on_status_line == True) {
	reverse_on ();
  }
}

static int utfcount = 0;
static unsigned long unichar;
static unsigned short prev_cjkchar = 0L;

/*
 * put Unicode (UCS-4, actually) character to screen;
 * put as UTF-8 on UTF-8 terminal, or as Latin-1 otherwise
	7 bits	0xxxxxxx
	 8..11	110xxxxx 10xxxxxx
	12..16	1110xxxx 10xxxxxx 10xxxxxx
	17..21	11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
	22..26	111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
	27..31	1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
 */
void
_put_unichar (unichar)
  unsigned long unichar;
{
#ifdef pc_charset
  character c;
#endif

  if (unichar < ' ') {
	indicate_UTF (unichar + '@');
  } else if (unichar == 0x7F) {
	indicate_UTF ('?');
  } else if (unichar >= 0x80 && unichar <= 0x9F) {
	indicate_UTF (unichar + '@');
#ifndef pc_charset
  } else if (unichar == 0xA0 /* nbsp */ && ! visciimode ()) {
	indicate_special ('');
#endif
  } else if (utf8_screen == True) {
	if (unichar < 0x80) {
		__putchar (unichar);
	} else if (unichar < 0x800) {
		__putchar (0xC0 | (unichar >> 6));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x10000) {
		__putchar (0xE0 | (unichar >> 12));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x200000) {
		__putchar (0xF0 | (unichar >> 18));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x4000000) {
		__putchar (0xF8 | (unichar >> 24));
		__putchar (0x80 | ((unichar >> 18) & 0x3F));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x80000000) {
		__putchar (0xFC | (unichar >> 30));
		__putchar (0x80 | ((unichar >> 24) & 0x3F));
		__putchar (0x80 | ((unichar >> 18) & 0x3F));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else {
		/* special encoding of 2 Unicode chars, mapped from 1 JIS character */
		put_unichar (unichar & 0xFFFF);
		unichar = (unichar >> 16) & 0x7FFF;
		if (combining_screen == True || ! iscombining (unichar)) {
			put_unichar (unichar);
		}
	}
  } else {
	if (unichar < 0x100) {
#ifdef pc_charset
		if (unichar < 0x80) {
			__putchar (unichar);
		} else {
			c = (character) pc_charmap [unichar - 0x80];
			if (c < 0x80) {
				indicate_UTF (c);
			} else {
				__putchar (c);
			}
		}
#else
		__putchar (unichar);
#endif
	} else {
		if (iscombining (unichar)) {
			indicate_UTF ('\'');
		} else {
			if (unichar >= 0xFF01 && unichar <= 0xFF5E) {
				/* FULLWIDTH forms */
				indicate_UTF (unichar - 0xFEE0);
			} else if (unichar == 0xFFE0) {
				/* FULLWIDTH CENT SIGN */
				indicate_UTF (0xA2);
			} else if (unichar == 0xFFE1) {
				/* FULLWIDTH POUND SIGN */
				indicate_UTF (0xA3);
			} else if (unichar == 0xFFE2) {
				/* FULLWIDTH NOT SIGN */
				indicate_UTF (0xAC);
			} else if (unichar == 0xFFE3) {
				/* FULLWIDTH MACRON */
				indicate_UTF (0xAF);
			} else if (unichar == 0xFFE4) {
				/* FULLWIDTH BROKEN BAR */
				indicate_UTF (0xA6);
			} else if (unichar == 0xFFE5) {
				/* FULLWIDTH YEN SIGN */
				indicate_UTF (0xA5);
			} else if (unichar == 0x20AC) {
				/* EURO SIGN */
				indicate_UTF ('E');
			} else {
				indicate_UTF (UNI_MARK);
			}
		}
		if (iswide (unichar)) {
			indicate_UTF (' ');
		}
	}
  }
}

/**
   put CJK character to screen;
   to be called only if cjk_term == True
   returns screen length
 */
int
_put_cjkchar (cjkchar)
  unsigned long cjkchar;
{
  character cjkbytes [5];
  character * cp;
  unsigned long unichar;

  if (cjk_term == True) {
	if ((cjk_encoding == 'G' && cjkchar >= 0x80000000 && gb18030_term == False)
	 || (cjk_encoding == 'J' && cjkchar >= 0x8F0000 && euc3_term == False)
	 || (cjk_encoding == 'C' && cjkchar >= 0x8E000000 && euc4_term == False)
	 || (cjklow_term == False &&
	     ((cjkchar >= 0x8000 && cjkchar < 0xA000)
	   || ((cjkchar & 0xFF) >= 0x80 && (cjkchar & 0xFF) < 0xA0)
	     )
	    )
	   )
	{
		indicate_UTF ('@');
		indicate_UTF (' ');
	} else {
		(void) cjkencode (cjkchar, cjkbytes);
		cp = cjkbytes;
		while (* cp != '\0') {
			putchar (* cp ++);
		}
	}
  } else {
	if (cjkchar >= 0x80) {
		unichar = lookup_cjk (cjkchar);
	} else {
		unichar = cjkchar;
	}
	if (unichar != 0) {
		put_unichar (unichar);
		if (utf_cjk_wide_padding == True
		    && cjkchar >= 0x100 && ! iswide (unichar)) {
			/* simulate double screen width of CJK 
			   character even if corresponding 
			   Unicode character has only single width
			*/
			__putchar (' ');
		}
	} else {
		indicate_UTF ('#');
		if (utf_cjk_wide_padding == True && cjkchar >= 0x100) {
			indicate_UTF (' ');
		}
	}
  }

  if (cjkchar >= 0x100) {
	return 2;
  } else {
	return 1;
  }
}

static
void
put_uniend ()
{
  if (utf8_text == True && utfcount > 0) {
	illegal_UTF ('');
	utfcount = 0;
  }
}

/**
   GB18030 algorithmic mapping part
 */
static
unsigned long
gb_to_unicode (gb)
  unsigned long gb;
{
	return (((((gb >> 24) & 0xFF) - 0x90) * 10
		+ (((gb >> 16) & 0xFF) - 0x30)) * 126L
		+ (((gb >> 8) & 0xFF) - 0x81)) * 10L
		+ ((gb & 0xFF) - 0x30)
		+ 0x10000;
}

static
unsigned long
unicode_to_gb (uc)
  unsigned long uc;
{
	unsigned int a, b, c, d;
	uc -= 0x10000;
	d = 0x30 + uc % 10;
	uc /= 10;
	c = 0x81 + uc % 126;
	uc /= 126;
	b = 0x30 + uc % 10;
	uc /= 10;
	a = 0x90 + uc;
	return (a << 24) | (b << 16) | (c << 8) | d;
}

/*
   cjk () converts a Unicode value into a CJK encoded character.
   May also be used for single-byte mapped character sets.
 */
unsigned long
cjk (unichar, beep)
  unsigned long unichar;
  FLAG beep;
{
	unsigned long cjkchar;

	if (cjk_encoding == 'G' && unichar >= 0x10000) {
		return unicode_to_gb (unichar);
	}

	cjkchar = unmap_char (unichar);
	if (cjkchar != -1) {
		return cjkchar;
	}

	if (unichar < 0x80 && unichar >= 0x20) {
		/* transparently map ASCII range */
		return unichar;
	} else {
		/* notify "not found" */
		if (beep == True) {
			ring_bell ();
		}
		return quit_char;
	}
}

/*
   lookup_cjk () converts a CJK encoded character into a Unicode value.
   May also be used for single-byte mapped character sets.
 */
unsigned int
lookup_cjk (cjk)
  unsigned int cjk;
{
	unsigned int unichar;

	if (cjk_encoding == 'G' && cjk >= 0x90000000) {
		return gb_to_unicode (cjk);
	}

	unichar = map_char (cjk);
	if (unichar != -1) {
		return unichar;
	}

	if (cjk < 0x80 && cjk >= 0x20) {
		/* transparently map ASCII range */
		return cjk;
	} else {
		/* notify "not found" */
		return 0;
	}
}

/*
 * put_char () puts a character byte to the screen (via buffer)
 * If UTF-8 text is being output through put_char () but the screen 
 * is not in UTF-8 mode, put_char () transforms the byte sequences.
 * In CJK mode, this function should not be called anymore with 
 * multibyte CJK codes (Shift-JIS single-byte is OK).

 * putchar (c) is __putchar (c)
 * __putchar (c) does (void) writeout (c) (except with curses)
 */
static
void
put_char (c)
  character c;
{
  if (utf8_text == True) {
	if (c < 0x80) {
		if (utfcount > 0) {
			/* previous character not terminated properly */
			illegal_UTF ('');
			utfcount = 0;
		}
		__putchar (c);
	} else if ((c & 0xC0) == 0x80) {
		if (utfcount == 0) {
			illegal_UTF ('8');
			return;
		}
		unichar = (unichar << 6) | (c & 0x3F);
		utfcount --;

		if (utfcount == 0) {
			/* final UTF-8 byte */
			put_unichar (unichar);
		} else {
			/* character continues */
			return;
		}
	} else { /* first UTF-8 byte */
		if (utfcount > 0) {
			/* previous character not terminated properly */
			illegal_UTF ('');
			utfcount = 0;
		}
		if ((c & 0xE0) == 0xC0) {
			utfcount = 2;
			unichar = c & 0x1F;
		} else if ((c & 0xF0) == 0xE0) {
			utfcount = 3;
			unichar = c & 0x0F;
		} else if ((c & 0xF8) == 0xF0) {
			utfcount = 4;
			unichar = c & 0x07;
		} else if ((c & 0xFC) == 0xF8) {
			utfcount = 5;
			unichar = c & 0x03;
		} else if ((c & 0xFE) == 0xFC) {
			utfcount = 6;
			unichar = c & 0x01;
		} else {
			/* illegal UTF-8 code 254 (0xFE) or 255 (0xFF) */
			illegal_UTF ('4' + (c & 1));
			return;
		}
		utfcount --;
		return;
	}
  } else if (cjk_text == True && cjk_term == False) {
	/* prev_cjkchar mechanism obsolete (would not suffice anymore...) */
	if (prev_cjkchar != 0 || (c >= 0x80 && ! multichar (c))) {
		put_cjkchar (prev_cjkchar << 8 | c);
		prev_cjkchar = 0;
	} else if (multichar (c)) {
		prev_cjkchar = c;
	} else {
		__putchar (c);
	}
  } else {
#ifndef pc_charset
	if (c == 0xA0 /* nbsp */) {
		indicate_special ('');
	} else
#endif
		__putchar (c);
  }
}

/*
 * putcharacter (c) puts one non-UTF character to the screen
 */
void
putcharacter (c)
  character c;
{
  if (isviscii (c)) {
	put_unichar (lookup_cjk (c));
  } else if (utf8_screen == True && utf8_text == False) {
	put_unichar (c);
  } else {
	put_char (c);
  }
}

/*
 * put 1 UTF-8 character (from string)
 */
void
put_utfchar (s)
  char * s;
{
  unsigned long unichar;
  int utflen;

  utf8_info (s, & utflen, & unichar);
  put_unichar (unichar);
}

/*
 * put mark character, possibly switching alternate character set
 * putmarkmode and endmarkmode are only called by putmark and for 
 * display of TAB markers
 */
FLAG dim_mode = False;
FLAG mark_mode = False;
FLAG use_extracset = False;

static
void
putmarkmode (c, utfmark)
  character c;
  char * utfmark;
{
  if (dim_mode == False) {
	dimon ();
	dim_mode = True;
  }

  if (utf8_screen == True) {
	if (utfmark != NIL_PTR && * utfmark != '\0') {
		putnarrowutf (utfmark);
	} else {
		putnarrowchar ((unsigned long) c);
	}
  } else if ((c & '\200') != '\0') {
	if (mark_mode == True) {
		end_extracset ();
		mark_mode = False;
	}
#ifdef pc_charset
	put_unichar (c);
#else
	putchar (c);
#endif
  } else {
	if (mark_mode == False && use_extracset == True) {
		start_extracset ();
		mark_mode = True;
	}
	putchar (c);
  }
}

static
void
endmarkmode ()
{
  if (dim_mode == True) {
	dimoff ();
	dim_mode = False;
  }
  if (mark_mode == True) {
	end_extracset ();
	mark_mode = False;
  }
}

/*
   Output a line status marker.
 */
void
putmark (c, utfmark)
  char c;
  char * utfmark;
{
  putmarkmode (c, utfmark);
  endmarkmode ();
}

static
void
putCJKmark (umark)
  unsigned long umark;
{
	unsigned long mark = cjk (umark, True);
	dimon ();
	put_cjkchar (mark);
	dimoff ();
}

static
void
putUmark (c, utfmark)
  character c;
  char * utfmark;
{
  unimarkdisp_on ();
  if (utf8_screen == True) {
	if (utf8_text == True && utfmark != NIL_PTR && * utfmark != '\0') {
		putnarrowutf (utfmark);
	} else {
		putnarrowchar ((unsigned long) c);
	}
  } else {
#ifdef pc_charset
	put_unichar (c);
#else
	putchar (c);
#endif
  }
  unimarkdisp_off ();
}

static
void
putret (type)
  unsigned char type;
{
  if (type == lineend_LF) {
	putmark (RET_MARK, UTF_RET);
  } else if (type == lineend_CRLF) {
	/* MSDOS line separator */
	putmark (DOSRET_MARK, UTF_DOSRET);
  } else if (type == lineend_CR) {
	/* Mac line separator */
/*	putmark ('', "¥");	*/
/*	putmark ('', "¦");	*/
	putmark ('@', "@");
  } else if (type == lineend_NONE) {
	putmark ('', "¬");
  } else if (type == lineend_NUL) {
	putmark ('', "º");
  } else if (type == lineend_LS) {
	/* Unicode line separator 2028:   */
	putUmark (RET_MARK, UTF_RET);
  } else if (type == lineend_PS) {
	/* Unicode paragraph separator 2029:   */
	putUmark (PARA_MARK, UTF_PARA);
  } else {
	putmark ('', "¤");
  }
}


/*======================================================================*\
|*			Output						*|
\*======================================================================*/

/*
 * Bad_write () is called when a write failed. Notify the user.
 */
void
bad_write (fd)
  int fd;
{
  if (fd == output_fd) {	/* Cannot write to terminal? */
	panicio ("Write error on terminal", serror ());
  }

  clear_buffer (); /* out_count = 0; */
  ring_bell ();
  error ("Write aborted (File incomplete): ", serror ());
}

/*
 * Flush the I/O buffer on filedescriptor fd.
flush () is (void) flush_buffer (output_fd) unless with curses
 */
int
flush_buffer (fd)
  int fd;
{
  if (out_count <= 0) {		/* There is nothing to flush */
	return FINE;
  }
#ifdef conio
  if (fd == output_fd) {
	cputs (screen);
  }
#else
#ifdef BorlandC
  if (fd == output_fd) {
	screen [out_count] = '\0';
/* don't ask me why that crazy compiler doesn't work with write () below */
	printf ("%s", screen);
  }
#endif
#endif
  else {
	char * writepoi = screen;
	int written = 0;
	int none_count = 0;
/*	int less_count = 0;	*/

	while (out_count > 0) {
		written = write (fd, writepoi, out_count);
		if (written == -1) {
			if (geterrno () == EINTR && winchg == True) {
				/* try again */
			} else {
				bad_write (fd);
				return ERRORS;
			}
		} else if (written == 0) {
			none_count ++;
			if (none_count > 20) {
				bad_write (fd);
				return ERRORS;
			}
		} else {
			out_count -= written;
			writepoi += written;
		}
	}
  }
  clear_buffer (); /* Empty buffer: out_count = 0; */
  return FINE;
}

/*
 * writeout does a buffered output to screen.
 */
int
writeout (c)
  char c;
{
  char clow;

  if (c == '\n') {
	if (writeout ('\r') == ERRORS) {
		return ERRORS;
	}
  }
  if (translate_output == True) {
	if ((c & '\200') != '\0') {
		altcset_on ();
		clow = (c & '\177');
		if ((int) clow < translen) {
			writeout (transout [(int) clow]);
		} else {
			writeout (clow);
		}
		altcset_off ();
		return FINE;
	}
  }
  screen [out_count ++] = c;
  if (out_count == screen_BUFL) {
	return flush_buffer (output_fd);
  }
#ifdef DEBUG
  flush ();
#endif
  return FINE;
}

/*
 * Putoutstring writes the given string out to the terminal.
 * (buffered via writechar via misused screen buffer!)
putstring (str) is putoutstring (str) unless with curses
 */
void
putoutstring (text)
  register char * text;
{
  while (* text != '\0') {
	(void) writeout (* text ++);
  }
}

void
put_blanks (endpos)
  int endpos;
{
  int startpos = 0;
  while (startpos ++ <= endpos) {
	putchar (' ');
  }
}

static
void
clear_wholeline ()
{
  if (can_clear_eol == True) {
	clear_eol ();
  } else {
	put_blanks (XMAX);
  }
}

void
clear_lastline ()
{
  if (can_clear_eol == True) {
	clear_eol ();
  } else {
	put_blanks (XMAX - 1);
  }
}


/*======================================================================*\
|*			Text display (buffer oriented output)		*|
\*======================================================================*/

static
void
disp_syntax (syntax_marker_was, syntax_marker)
  char syntax_marker_was;
  char syntax_marker;
{
  if (syntax_marker != syntax_marker_was) {
	if (syntax_marker & syntax_JSP) {
		dispHTML_jsp ();
	} else if (syntax_marker & syntax_comment) {
		dispHTML_comment ();
	} else if (syntax_marker & syntax_HTML) {
		dispHTML_code ();
	} else {
		dispHTML_off ();
	}
  }
}

#define default_script -999

static
void
disp_script (script_colour)
  int script_colour;
{
  if (script_colour == default_script) {
	disp_normal ();
  } else {
	disp_colour (script_colour);
  }
}

static
struct colour_mapping {
	char * scriptname;
	int colour;
} colour_table [] =
{
#include "colours.h"
};

/**
   Determine display colour for character script.
 */
static
int
map_script_colour (scriptname)
  char * scriptname;
{
  int min = 0;
  int max = sizeof (colour_table) / sizeof (struct colour_mapping) - 1;
  int mid;
  int cmp;

  /* binary search in table */
  while (max >= min) {
    mid = (min + max) / 2;
    cmp = strcmp (colour_table [mid].scriptname, scriptname);
    if (cmp < 0) {
      min = mid + 1;
    } else if (cmp > 0) {
      max = mid - 1;
    } else {
      return colour_table [mid].colour;
    }
  }

  return default_script;
}

/*
 * Put_line prints the given line on the standard output.
 * If offset is not zero, printing will start at that x-coordinate.
 * If the FLAG clear_line is True, then the screen line will be cleared
 * when the end of the line has been reached.
line_print (ly, line) is put_line (ly, line, 0, True, False)
 */
void
put_line (ly, line, offset, clear_line, positioning)
  int ly;		/* current screen line */
  LINE * line;		/* Line to print */
  int offset;		/* Offset to start if positioning == False */
			/* position if positioning == True */
  FLAG clear_line;	/* Clear to eoln if True */
  FLAG positioning;	/* positioning inside line if True (for prop. fonts) */
{
  char * textp = line->text;
  char * parap;
  int count = get_shift (line->shift_count) * - SHIFT_SIZE;
  int count_ini = count;
  int tab_count;			/* Used in tab expansion */
  int tab_mid;
  int offset_start;
  int offset_stop;
  character c;
  unsigned long unichar;
  int follow;
  int charwidth;
  FLAG disp_separate_combining;
  FLAG disp_separate_joining;
  char syntax_marker = line->prev->syntax_marker;
  char syntax_marker_was = syntax_none;
  char syntax_marker_new = syntax_marker;
  char syntax_byte_printed;
  int script_colour = default_script;
  int last_script_colour;

  int len;
  unsigned long cjkchar;
  int charlen;
  character cjkbytes [5];

  if (positioning == True) {
	offset_start = 0;
	offset_stop = offset;
  } else {
	offset_start = offset;
	offset_stop = XBREAK;
  }

/* Skip all chars as indicated by the offset_start and the shift_count field */
  while (count < offset_start) {
	syntax_marker_new = syntax_state (syntax_marker_new, textp);
	if (* textp == '<' || * textp == '>') {
		syntax_marker_was = syntax_marker;
		syntax_marker = syntax_marker_new;
	}
	advance_char_scr (& textp, & count, line->text);
  }

  if (count == 0 && count_ini < 0 && SHIFT_BEG != '\0') {
	putmark (SHIFT_BEG, NIL_PTR);
	if (! is_tab (* textp)) {
		syntax_marker_new = syntax_state (syntax_marker_new, textp);
		if (* textp == '<' || * textp == '>') {
			syntax_marker_was = syntax_marker;
			syntax_marker = syntax_marker_new;
		}
		advance_char_scr (& textp, & count, line->text);
	} else {
		count ++;
	}
  }
/* Check for shift-out left across double-width character boundary */
  if (count_ini < 0 && count > offset_start) {
	putmark (TABchar, NIL_PTR);
  }

  if (utf8_text == True) {
	utf8_info (textp, & follow, & unichar);
	charwidth = uniscrwidth (unichar, textp, line->text);
  } else if (cjk_text == True && multichar (* textp) && (* (textp + 1)) != '\n') {
	if (utf8_screen == True) {
		cjkchar = charvalue (textp);
		unichar = lookup_cjk (cjkchar);
		if (iscombining (unichar)) {
			charwidth = 0;
		} else if (utf_cjk_wide_padding == True || iswide (unichar)) {
			charwidth = 2;
		} else {
			charwidth = 1;
		}
	} else {
		charwidth = 2;
	}
  } else {
	charwidth = 1;
  }

  if (dim_HTML == True) {
	disp_syntax (0, syntax_marker);
  }

  while (* textp != '\n' && count < offset_stop + 1 - charwidth) {
	if (dim_HTML == True) {
		syntax_marker_new = syntax_state (syntax_marker_new, textp);
		if (* textp == '<' || * textp == '>') {
			syntax_marker_was = syntax_marker;
			syntax_marker = syntax_marker_new;
		}
		if (* textp == '<') {
			disp_syntax (syntax_marker_was, syntax_marker);
		}
	}

	if (is_tab (* textp)) {		/* Expand tabs to spaces */
		tab_count = tab (count);
		tab_mid = count + (tab_count - count) / 2 + 1;
		if (cjk_term == True) {
			if ((count & 1) == 1) {
				count ++;
				__putchar (' ');
			}
			while (count < offset_stop && count < tab_count) {
				count ++;
				count ++;
				putCJKmark (0x2026);	/* … */
			}
		} else if (TABchar0 != '\0') {
			if (TABchar0 != TABchar) {
				count ++;
				putmarkmode (TABchar0, UTF_TAB0);
			}
			while (count < offset_stop && count < tab_count - 1) {
				count ++;
				putmarkmode (TABchar, UTF_TAB);
			}
			if (count < offset_stop && count < tab_count) {
				count ++;
				if (TABchar2 != '\0') {
					putmarkmode (TABchar2, UTF_TAB2);
				} else {
					putmarkmode (TABchar, UTF_TAB);
				}
			}
		} else while (count < offset_stop && count < tab_count) {
			count ++;
			if (count == tab_mid && TABcharmid != '\0') {
				putmarkmode (TABcharmid, UTF_TABmid);
			} else {
				putmarkmode (TABchar, UTF_TAB);
			}
		}
		endmarkmode ();
		if (dim_HTML == True) {
			disp_syntax (0, syntax_marker);
		}

		textp ++;
	} else if (iscontrol (* textp)
#ifndef pc_charset
		|| ((character) * textp == 0xA0 /* nbsp */
		    && utf8_text == False && cjk_text == False
		    && ! visciimode ()
		   )
#endif
		)
	{
		put_uniend ();
		if (use_vt100_block_graphics == True && display_block_graphics == True
		    && (
			   * textp == 1
			|| (* textp >= 0x0B && * textp <= 0x0F)
			|| * textp == 0x12
			|| (* textp >= 0x15 && * textp <= 0x19)
		)) {
			altcset_on ();
			putcharacter (* textp + 0x5F);
			altcset_off ();
#ifndef pc_charset
		} else if ((character) * textp == 0xA0 /* nbsp */) {
			indicate_special ('');
#endif
		} else {
			ctrldisp_on ();
			putcharacter (controlchar (* textp));
			ctrldisp_off ();
		}
		if (dim_HTML == True) {
			disp_syntax (0, syntax_marker);
		}

		textp ++;
		count ++;
	} else {
		syntax_byte_printed = * textp;

		if (utf8_text == True) {
			follow --;
			c = * textp ++;

			last_script_colour = script_colour;
			script_colour = map_script_colour (script (unichar));
			if (script_colour != last_script_colour) {
				disp_script (script_colour);
			}

			disp_separate_combining = False;
			disp_separate_joining = False;
			if (iscombined (unichar, textp - 1, line->text)) {
			    if (combining_mode == False 
				&& combining_screen == True)
			    {	/* enforce separated display */
				disp_separate_combining = True;
				combiningdisp_on ();

				if (iswide (unichar))
				{
					put_unichar (0x3000);
					count += 2;
				} else {
					if (isjoined (unichar, textp - 1, line->text)) {
						disp_separate_joining = True;
#ifdef mlterm_fixed
						/* Separate display of 
						   Arabic ligature components;
						   don't output base blank 
						   and try to suppress 
						   Arabic ligature shaping
						   with one of
						   200B;ZERO WIDTH SPACE
						   200C;ZERO WIDTH NON-JOINER
						   200D;ZERO WIDTH JOINER
						   FEFF;ZERO WIDTH NO-BREAK SPACE
						   - none works in mlterm
						 */
						put_unichar (0x200C);
#else
						/* Transform character 
						   to its isolated form;
						   see below
						 */
#endif
					} else {
						put_char (' ');
					}
					count += 1;
				}
			    } else {
				if (combining_screen == False) {
					disp_separate_combining = True;
					combiningdisp_on ();
					count += charwidth;
				}
			    }
			} else {
				count += charwidth;
			}

#ifndef mlterm_fixed
			if (disp_separate_joining == True) {
				/* prevent terminal ligature by 
				   substituting joining character 
				   with its ISOLATED FORM */
				put_unichar (isolated_alef (unichar));
				/* skip rest of joining char */
				while (follow > 0 && (* textp & 0xC0) == 0x80) {
					textp ++;
					follow --;
				}
			}
			else
#endif
			{
				put_char (c);
				while (follow > 0 && (* textp & 0xC0) == 0x80) {
					put_char (* textp ++);
					follow --;
				}
			}
			if (disp_separate_combining == True) {
				combiningdisp_off ();
				if (dim_HTML == True) {
					disp_syntax (0, syntax_marker);
				}
			}

		} else if (cjk_text == True) {
			if (multichar (* textp)) {
				len = CJK_len (textp);
				if (utf_cjk_wide_padding == True
				    || utf8_screen == False) {
					cjkchar = charvalue (textp);
				}
				charlen = cjkencode (cjkchar, cjkbytes);

				while (len > 0 && * textp != '\n' && * textp != '\0') {
					textp ++;
					len --;
					charlen --;
				}
				if (len != 0 || charlen != 0) {
					/* incomplete CJK char */
					indicate_special ('#');
					count ++;
					if (utf_cjk_wide_padding == True) {
						if (charlen == 0 || charlen == -3) {
							indicate_special ('#');
							count ++;
						}
					}
				} else {
					put_cjkchar (cjkchar);
					count += charwidth;
				}
			} else {
				put_char (* textp);
				textp ++;
				count ++;
			}
		} else {
			c = * textp ++;
			putcharacter (c);
			count ++;
		}

		if (dim_HTML == True && syntax_byte_printed == '>') {
			disp_syntax (syntax_marker_was, syntax_marker);
		}
	}

	if (utf8_text == True) {
		utf8_info (textp, & follow, & unichar);
		charwidth = uniscrwidth (unichar, textp, line->text);
	} else if (cjk_text == True && multichar (* textp) && (* (textp + 1)) != '\n') {
		if (utf8_screen == True) {
			cjkchar = charvalue (textp);
			unichar = lookup_cjk (cjkchar);
			if (iscombining (unichar)) {
				charwidth = 0;
			} else if (utf_cjk_wide_padding == True || iswide (unichar)) {
				charwidth = 2;
			} else {
				charwidth = 1;
			}
		} else {
			charwidth = 2;
		}
	} else {
		charwidth = 1;
	}
  }
  put_uniend ();

  if (script_colour != default_script) {
	disp_script (default_script);
  }

  if (dim_HTML == True) {
	disp_syntax (syntax_marker, 0);
  }

  if (positioning == True) {
	/* self-made cursor for terminals (such as xterm)
	   which have display problems with proportional screen fonts
	   and their cursor */
	reverse_on ();
	if (* textp != '\n') {
		if (utf8_text == True) {
			put_utfchar (textp);
			put_uniend ();
		} else if (cjk_text == True) {
			put_cjkchar (charvalue (textp));
		} else {
			putcharacter (* textp);
		}
	} else if (cjk_term == True) {
		putCJKmark (0x300A);	/* 《 */
	} else if (RET_MARK != '\0') {
		if (PARA_MARK != '\0' && paradisp == True
		 && line->return_type != lineend_LS
		 && line->return_type != lineend_PS) {
			parap = textp;
			parap --;
			if (* parap == ' ' || * parap == '\t') {
				putret (line->return_type);
			} else {
				putmark (PARA_MARK, UTF_PARA);
			}
		} else {
			putret (line->return_type);
		}
	} else {
		putchar (' ');
	}
	reverse_off ();
	set_cursor (0, 0);
  } else /* (positioning == False) */ {
	/* If line is longer than XBREAK chars, print the shift_mark */
	if (count <= XBREAK && * textp != '\n') {
		if (cjk_term == True) {
			/*putCJKmark (0x300B);	/* 》 */
			putmark ('>', NIL_PTR);
			count ++;
		} else {
			putmark (SHIFT_MARK, NIL_PTR);
		}
		count ++;
		if (count == XBREAK) {
			putchar (' ');
			count ++;
		}
	}

	/* Mark end of line if so desired */
	if (* textp == '\n') {
	    if (cjk_term == True) {
		putCJKmark (0x300A);	/* 《 */
		/* to assure it fits scrollbar_width is set to 1 
		   in CJK terminal mode even if scrollbar is switched off */
	    } else if (RET_MARK != '\0') {
		if (PARA_MARK != '\0' && paradisp == True
		 && line->return_type != lineend_LS
		 && line->return_type != lineend_PS) {
			parap = textp;
			parap --;
			if (* parap == ' ') {
				putret (line->return_type);
			} else {
				putmark (PARA_MARK, UTF_PARA);
			}
		} else {
			putret (line->return_type);
		}

		count ++;
		if (RET_BLANK) {
			while (count < XBREAK) {
				putmark (RET_BLANK, UTF_RETblank);
				count ++;
			}
			if (RET_BLANK2 && count <= XBREAK) {
				putmark (RET_BLANK2, UTF_RETblank2);
				count ++;
			}
		}
	    }
	}

	/* Clear the rest of the line if clear_line is True */
	if (clear_line == True) {
		set_scrollbar_dirty (ly);
		if (can_clear_eol == True) {
			if (count <= XBREAK) {
				clear_eol ();
			}
		} else {
			while (count ++ <= XBREAK) {
				/* clear up to XMAX */
				putchar (' ');
			}
		}
	}
  }
}

/*
 * set_cursor_xy sets the cursor by either directly calling set_cursor
 * or, in the case of proportional font support, reprinting the line
 * up to the x position
 */
void
set_cursor_xy ()
{
  if (proportional == True) {
	set_cursor (0, y);
	if (x != 0) {
		put_line (y, cur_line, x, False, True);
	}
	/* cur_line may still be undefined if x == 0 */
  } else {
	set_cursor (x, y);
  }
}

/*
 * Display line at screen line y_pos if it lies between y_min and y_max.
 * If it is no text line (end of file), clear screen line.
 */
static
void
display_line_at (y_pos, line, y_min, y_max, first)
  int y_pos;
  int y_min;
  int y_max;
  register LINE * line;
  FLAG first;
{
  line = proceed (line, y_pos - y_min);
  if (y_pos >= y_min && y_pos <= y_max) {
	set_cursor (0, y_pos);
	if (line == tail) {
		clear_wholeline ();
	} else {
		if (first == False) {
			if (display_delay >= 0) {
				flush ();
			}
#ifdef msdos
			if (display_delay > 0) {
				delay (display_delay);
			}
#else
#ifdef use_usleep
			if (display_delay > 0) {
				(void) usleep (1000 * display_delay);
			}
#endif
#endif
		}
		line_print (y_pos, line);
	}
  }
}

/*
 * Display () shows count + 1 lines on the terminal starting at the given 
 * coordinates. At end of file, the rest of the screen is blanked.
 * When count is negative, a backwards print from `line' will be done.
 */
void
display (y_coord, line, count, new_pos)
  int y_coord;
  int new_pos;
  register LINE * line;
  register int count;
{
  int y_max = y_coord + count;
  int y_off;

/* Find new startline if count is negative */
  if (count < 0) {
	line = proceed (line, count);
	count = - count;
  }

#ifdef obsolete
  clean_menus ();
  This led to recursive calls of display () resulting in wrong display;
  menus are now cleared before invoking functions from menu.
#endif

  display_line_at (new_pos, line, y_coord, y_max, True);
  y_off = 0;
  while (y_off < count) {
	y_off ++;
	display_line_at (new_pos - y_off, line, y_coord, y_max, False);
	display_line_at (new_pos + y_off, line, y_coord, y_max, False);
  }
#ifdef UNUSED
/* old code, building the display from top to bottom (how boring): */
/* with this code, XBREAK must be set to XMAX - 1 */
/* Print the lines */
  set_cursor (0, y_coord);
  while (line != tail && count -- >= 0) {
	line_print (y_coord, line);
	line = line->next;
	y_coord ++;
  }

/* Print the blank lines (if any) */
  if (loading == False) {
	while (count -- >= 0) {
		clear_eol ();
		putchar ('\n');
	}
  }
#endif
}


/*======================================================================*\
|*				End					*|
\*======================================================================*/
