/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 2003 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */


#include "io/fastq-mira.H"


string FastQ::FQ_blanks=" \t";


// Plain vanilla constructor
FastQ::FastQ()
{
  FUNCSTART("FastQ::FastQ()");

  FQ_linebuffer.reserve(2000);
  FQ_name.reserve(100);
  FQ_comment.reserve(100);
  FQ_seq.reserve(100);
  FQ_quals.reserve(100);
  FQ_dummy1.reserve(100);
  FQ_dummy2.reserve(100);
  FQ_tracktellg=0;
  FQ_filesize=0;

  FUNCEND();
}

FastQ::~FastQ()
{
  FUNCSTART("FastQ::~FastQ()");

  discard();

  FUNCEND();
}


void FastQ::discard()
{
  FUNCSTART("FastQ::discard()");

  FUNCEND();
}


//// Copy constructor
////  no discard needed as this object will be freshly created when
////  called through this constructor
//FastQ::FastQ(FastQ const &other)
//{
//  FUNCSTART("FastQ::FastQ(FastQ const &other)");
//
//  ??_valid=0;
//
//  *this=other;                               // call the copy operator
//
//  FUNCEND();
//}
//
//// Copy operator, needed by copy-constructor
//FastQ const & FastQ::operator=(FastQ const & other)
//{
//  FUNCSTART("FastQ const & FastQ::operator=(FastQ const & other)");
//  ERROR("Not implemented yet.");
//  FUNCEND();
//  return *this;
//}

//ostream & operator<<(ostream &ostr, FastQ const &???)
//{
//  FUNCSTART("friend ostream & FastQ::operator<<(ostream &ostr, const  &???)");
//  ERROR("Not implemented yet.");
//
//  FUNCEND();
//  return ostr;
//}


void FastQ::openFile(const string & filename, bool tracktellg)
{
  FUNCSTART("void FastQ::openFile(string & filename, bool tracktellg)");

  if(FQ_fin.is_open()){
    FQ_fin.close();
  }
  FQ_fin.clear();
  FQ_tracktellg=tracktellg;

  FQ_fin.open(filename.c_str(),ios::in|ios::ate|ios::binary);
  if(!FQ_fin){
    MIRANOTIFY(Notify::FATAL, "File not found: " << filename);
  }
  FQ_filesize=FQ_fin.tellg();
  if(FQ_filesize==static_cast<streampos>(0)){
    MIRANOTIFY(Notify::FATAL, "Zero length file: " << filename);
  }
  FQ_fin.seekg(0, ios::beg);

  FQ_linecount=0;
  FQ_lasttellg=0;
  FQ_tellgofread=0;

  FUNCEND();
}

/*************************************************************************
 *
 * Careful: at function end, may already have preloaded one line after
 *  the next read into FQ_linebuffer
 *
 *************************************************************************/

int64_t FastQ::loadNext(streampos sp)
{
  FUNCSTART("int64_t FastQ::loadNext(streampos sp)");
  BUGIFTHROW(!FQ_fin.is_open(),"Tried to load a sequence while no file is opened.");
  FQ_fin.clear();
  FQ_fin.seekg(sp, ios::beg);
  FQ_linebuffer.clear();
  FUNCEND();
  return loadNext();
}

int64_t FastQ::loadNext()
{
  FUNCSTART("void FastQ::loadNext()");

  BUGIFTHROW(!FQ_fin.is_open(),"Tried to load a sequence while no file is opened.");

  FQ_name.clear();
  FQ_comment.clear();
  FQ_seq.clear();
  FQ_quals.clear();

  if(FQ_linebuffer.empty()) {
    safeGetLineIntoLineBuffer();
  }
  FQ_tellgofread=FQ_lasttellg;

  // allow for empty lines??
  if(!FQ_linebuffer.empty()){
    if(FQ_linebuffer[0]!='@'){
      MIRANOTIFY(Notify::FATAL,"First character of line " << FQ_linecount << " is not a '@'. That file is not in FASTQ format.");
    }
    splitNameLine(FQ_name,FQ_comment);
    while(safeGetLineIntoLineBuffer() && FQ_linebuffer[0] != '@' && FQ_linebuffer[0] != '+'){
      FQ_seq+=FQ_linebuffer;
    }
    if(!FQ_linebuffer.empty() && FQ_linebuffer[0] == '+'){
      if(FQ_linebuffer.size()>1){
	splitNameLine(FQ_dummy1,FQ_dummy2);
	if(FQ_name!=FQ_dummy1){
	  MIRANOTIFY(Notify::FATAL,"Quality name of read '" << FQ_dummy1 << "' is not equal to sequence name '" << FQ_name << "'. Around line " << FQ_linecount);
	}
      }
      while(safeGetLineIntoLineBuffer() && FQ_quals.size()<FQ_seq.size()){
	FQ_quals+=FQ_linebuffer;
      }
      if(!FQ_quals.empty() && FQ_quals.size() != FQ_seq.size()){
	MIRANOTIFY(Notify::FATAL,"Read " << FQ_name << ": length of quality string (" << FQ_quals.size() << ") is not equal to sequence size (" << FQ_seq.size() << "') Around line " << FQ_linecount);
      }
    }
  }

  FUNCEND();
  if(FQ_seq.empty() && FQ_fin.eof()) return -1;
  return static_cast<int64>(FQ_seq.size());
}

size_t FastQ::safeGetLineIntoLineBuffer()
{
  FQ_linebuffer.clear();
  if(FQ_tracktellg) FQ_lasttellg=FQ_fin.tellg();
  while(getline(FQ_fin,FQ_linebuffer)){
    ++FQ_linecount;
    // get rid of '\r' from DOS
    while(!FQ_linebuffer.empty() && FQ_linebuffer[FQ_linebuffer.size()-1]=='\r') FQ_linebuffer.resize(FQ_linebuffer.size()-1);
    // allow for empty lines??
    if(!FQ_linebuffer.empty()) break;
  }
  return FQ_linebuffer.size();
}

void FastQ::splitNameLine(string & name, string & comment)
{
  name.clear();
  comment.clear();

  string::size_type nameend=string::npos;

  nameend=FQ_linebuffer.find_first_of(FQ_blanks,0);
  if(nameend==string::npos){
    // do not use swap: FQ_linebuffer may have a large memory
    //  block reserved due to long lines read in the past
    //  keep that with FQ_linebuffer to avoid reallocations
    name=FQ_linebuffer.substr(1, FQ_linebuffer.size()-1);
  }else{
    name=FQ_linebuffer.substr(1, nameend-1);
    comment=FQ_linebuffer.substr(nameend+1,FQ_linebuffer.size()-nameend);
  }
 
}


streampos FastQ::getStreamposOfRead() const {
  FUNCSTART("streampos FastQ::getStreamposOfRead()");
  BUGIFTHROW(!FQ_tracktellg,"Trying to get stream pos of a read, but file opened without tracking option");
  FUNCEND();
  return FQ_tellgofread;
}
