/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */

#include "contig.H"
#include "util/misc.H"
#include "assembly_output.H"


//#define PARANOIABUGTRACKFLAG
#ifdef PARANOIABUGTRACKFLAG
#define paranoiaBUGSTAT(statement) { statement;}
#define paranoiaBUGIF(ifcond, statement) { if(ifcond) {statement;}}
#else
#define paranoiaBUGSTAT(statement)
#define paranoiaBUGIF(ifcond, statement)
#endif



enum { CON_CONSTARTSIZE=4000,
       CON_VECTORINCSIZE=4000,
       CON_READSSTARTSIZE=100,
       CON_READSINCSIZE=100};



bool Contig::CON_abortflag=false;

uint32 Contig::CON_id_counter=1;
uint32 Contig::CON_static_ok=0;

uint32 Contig::CON_railcounter=0;

uint32 Contig::CON_cer_idcounter=0;



vector<multitag_t::mte_id_t> Contig::CON_danger_zones_ids;
vector<multitag_t::mte_id_t> Contig::CON_baselock_ids;
vector<multitag_t::mte_id_t> Contig::CON_snplock_ids;



#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
const Contig::consensus_counts_t Contig::CON_concounts_zero=
{0,0,0,0,0,0,0, {0,0,0,0,0,0}, 0,0,0,
 0,{0,0},{0,0},{0,0},'@'};
const Contig::consensus_counts_t Contig::CON_concounts_zero_nobb=
{0,0,0,0,0,0,0, {0,0,0,0,0,0}, 0,0,0};


uint8 Contig::CON_outtype=AS_TEXT;


#ifdef DEVELOPMENTVERSION
//bool  Contig::CON_outputRails=true;
bool  Contig::CON_outputRails=false;
#else
bool  Contig::CON_outputRails=false;
#endif

void Contig::foolCompiler()
{
#include "stdinc/foolcompiler.C"
}


//void Contig::addConsensusSequenceAsReadToContig(int32 strainid)
//{
//  string consseq;
//  vector<base_quality_t> consqual;
//  
//  getConsensus(consseq, consqual, false,0,0, strainid);
//
//  contigread_t newread;
//  newread.orpid=-1;
//  newread.direction=1;
//  newread.offset=0;
//  // newread.read=CON_readpool->getRead(id); // NO! we don't copy it twice
//
//  // now insert the read into the readpool of the contig 
//  CON_reads.push_back(newread);
//
//  CON_reads.back().read.setName("bla");
//  CON_reads.back().read.setSequenceFromString(consseq.getSequence());
//
//  hier weiter 
//
//
//}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/


// Plain vanilla constructor
Contig::Contig(vector<MIRAParameters> * params, ReadPool & readpool)
{
  FUNCSTART("Contig::Contig(MIRAParameters * params, ReadPool & readpool)");

  CON_miraparams=params;

  //CON_aligncache = new Align(params, true);

  CON_readpool=&readpool;

  init();

  CON_id=CON_id_counter++;

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::init()
{
  if(CON_static_ok==0){
    CON_static_ok=1;

    //CON_danger_zones_ids.push_back(CON_tagALUS);
    //CON_danger_zones_ids.push_back(CON_tagREPT);
    //
    //CON_baselock_ids.push_back(Read::REA_tagSRMr);
    //CON_baselock_ids.push_back(Read::REA_tagCRMr);
    //CON_baselock_ids.push_back(Read::REA_tagWRMr);
    //
    //CON_snplock_ids.push_back(Read::REA_tagSIOr);

    // TODO: need to use some strings here
    CON_danger_zones_ids.push_back(multitag_t::newIdentifier("ALUS"));
    CON_danger_zones_ids.push_back(multitag_t::newIdentifier("REPT"));

    CON_baselock_ids.push_back(multitag_t::newIdentifier("SRMr"));
    CON_baselock_ids.push_back(multitag_t::newIdentifier("CRMr"));
    CON_baselock_ids.push_back(multitag_t::newIdentifier("WRMr"));

    CON_snplock_ids.push_back(multitag_t::newIdentifier("SIOr"));

    CON_cer_idcounter=0;
  }

  CON_valid=CON_VALID;

  zeroVars();

  CON_tmpcons_from_backbone=false;
  CON_mergenewsrreads=false;
  CON_hasforcemergeareas=false;

  CON_cebugflag=false;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::discard()
{
  FUNCSTART("Contig::discard()");

  nukeSTLContainer(CON_2tmpcons);

  //CON_aligncache->discard();

  nukeSTLContainer(CON_counts);
  nukeSTLContainer(CON_reads);
  nukeSTLContainer(CON_templates_present);
  nukeSTLContainer(CON_consensus_tags);
  nukeSTLContainer(CON_fixedconsseq);
  nukeSTLContainer(CON_fixedconsqual);
  nukeSTLContainer(CON_outputorder);
  nukeSTLContainer(CON_last_dangerous_overlaps);
  nukeSTLContainer(CON_maprpids_to_conreadsids);
  nukeSTLContainer(CON_targetcoverageperst);
//  nukeSTLContainer(CON_cheat_intelcons);
//  nukeSTLContainer(CON_cheat_intelconsqual);
  nukeSTLContainer(CON_allconsseq);
  nukeSTLContainer(CON_allconsqual);
  nukeSTLContainer(CON_alladjustments);
  nukeSTLContainer(CON_strainconsseq);
  nukeSTLContainer(CON_strainconsqual);
  nukeSTLContainer(CON_strainadjustments);
  nukeSTLContainer(CON_2tmpcons);
  nukeSTLContainer(CON_allowedrefids);

  zeroVars();

  FUNCEND();
}


void Contig::zeroVars()
{

  CON_readsperstrain.clear();

//  CON_cheat_intelcons_markspecials=false;
//  CON_cheat_intelcons_mincov=999999;
//  CON_cheat_intelcons_minqual=99;
//  CON_cheat_intelcons_strainidtotake=999999;

  CON_conscalc_mincov=999999;
  CON_conscalc_minqual=101;
  CON_conscalc_missingchar=0;
  zeroStats();

  CON_specialsraddconditions=false;
  CON_ssrc_maxtotalerrors=-1;
  CON_ssrc_maxgaps=-1;
  CON_ssrc_maxmismatches=-1;

  for(uint32 i=0; i<NUMMERGESEQTYPES; i++){
    CON_nummergedreads_perseqtype[i]=0;
  }

  CON_index_leftestbbread=0;
  CON_index_rightestbbread=0;

  CON_nameprefix="stdname";
  CON_outtype=AS_TEXT;

  CON_contains_long_repeats_only=false;
  
  definalise();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

Contig::~Contig()
{
  FUNCSTART("Contig::~Contig()");

  discard();

  //delete CON_aligncache;

  FUNCEND();
}



/*************************************************************************
 *
 * Dump some internal things on the contig
 *
 * note: incomplete
 *
 *************************************************************************/

void Contig::dumpStatus(ostream & ostr)
{
  ostr << "ContigDump: " << CON_id << " " << CON_nameprefix << " " << CON_name << endl
       << "CON_finalised: " << CON_finalised << endl;
  if(CON_readpool==NULL){
    ostr << "CON_readpool: NULL !\n";
  }else{
    ostr << "CON_readpool: set\n";
  }
  ostr << "CON_reads: " << CON_reads.size() << endl
       << "CON_counts: " << CON_counts.size() << endl
       << "CON_templates_present: " << CON_templates_present.size() << endl
       << "CON_consensus_tags: " << CON_consensus_tags.size() << endl
       << "CON_targetcoverageperst: " << CON_targetcoverageperst.size();
  for(uint32 i=0; i<CON_targetcoverageperst.size();++i){
    ostr << "\t" << CON_targetcoverageperst[i] << endl;
  }

  ostr << "CON_maprpids_to_conreadsids: " << CON_maprpids_to_conreadsids.size() << endl
       << "CON_allowedrefids: " << CON_allowedrefids.size() << endl
       << "CON_2tmpcons: " << CON_2tmpcons.size() << endl
       << "CON_tmpcons_from_backbone: " <<  CON_tmpcons_from_backbone << endl
       << "CON_specialsraddconditions: " << CON_specialsraddconditions << endl
       << "CON_ssrc_maxtotalerrors: " << CON_ssrc_maxtotalerrors << endl
       << "CON_ssrc_maxgaps: " << CON_ssrc_maxgaps << endl
       << "CON_ssrc_maxmismatches: " << CON_ssrc_maxmismatches << endl;

  for(uint32 i=0; i<NUMMERGESEQTYPES; ++i){
    ostr << "CON_nummergedreads_perseqtype[" << i << "]: " << CON_nummergedreads_perseqtype[i] << endl;
  }

  ostr << "CON_fixedconsseq: " << CON_fixedconsseq.size() << endl
       << "CON_fixedconsqual: " << CON_fixedconsqual.size() << endl
       << "CON_conscalc_mincov: " << CON_conscalc_mincov << endl
       << "CON_strainconsqual: " <<  CON_strainconsqual.size() << endl;

  for(uint32 i=0; i<CON_strainconsqual.size(); ++i){
    ostr << "\t" << i << " " << CON_strainconsqual[i].size() << endl;
  }

  ostr << "CON_strainadjustments: " << CON_strainadjustments.size() << endl;
  for(uint32 i=0; i<CON_strainadjustments.size(); ++i){
    ostr << "\t" << i << " " << CON_strainadjustments[i].size() << endl;
  }

  ostr << "CON_readsperstrain: " << CON_readsperstrain.size() << endl;
  for(uint32 i=0; i<CON_readsperstrain.size(); ++i){
    ostr << "\t" << i << " " << CON_readpool->getStrainOfStrainID(i) << "\t" << CON_readsperstrain[i] << endl;
  }

  ostr << "CON_outputorder: " << CON_outputorder.size() << endl
       << "CON_last_dangerous_overlaps: " << CON_last_dangerous_overlaps.size() << endl
       << "CON_contains_long_repeats_only: " << CON_contains_long_repeats_only << endl;

}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::reserveCapacity(uint32 expectedlen, uint32 expectedreads)
{
  //CON_counts.reserve(expectedlen);
  CON_reads.reserve(expectedreads);
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::definalise()
{
//  CON_cheat_intelcons.clear();
//  CON_cheat_intelconsqual.clear();
  
  CON_allconsseq.clear();
  CON_allconsqual.clear();
  CON_alladjustments.clear();
  CON_strainconsseq.clear();
  CON_strainconsqual.clear();
  CON_strainadjustments.clear();

  CON_stats.statsvalid=false;
  CON_finalised=false;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::saveMem()
{
  FUNCSTART("void Contig::saveMem()");

  //CON_aligncache->discard();

  nukeSTLContainer(CON_2tmpcons);
  nukeSTLContainer(CON_outputorder);

  definalise();
//  nukeSTLContainer(CON_cheat_intelcons);
//  nukeSTLContainer(CON_cheat_intelconsqual);

  nukeSTLContainer(CON_allconsseq);
  nukeSTLContainer(CON_allconsqual);
  nukeSTLContainer(CON_alladjustments);
  nukeSTLContainer(CON_strainconsseq);
  nukeSTLContainer(CON_strainconsqual);
  nukeSTLContainer(CON_strainadjustments);

  nukeSTLContainer(CON_maprpids_to_conreadsids);

  nukeSTLContainer(CON_allowedrefids);

  FUNCEND();
  return;
}


// Copy constructor
//  no discard needed as this object will be freshly created when
//  called through this constructor
Contig::Contig(Contig const &other)
{
  FUNCSTART("Contig::Contig(Contig const &other)");

  //CON_aligncache=new Align(CON_miraparams,true);

  init();

  *this=other;                               // call the copy operator

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// Copy operator, needed by copy-constructor
Contig const & Contig::operator=(Contig const & other)
{
  FUNCSTART("Contig const & Contig::operator=(Contig const & other)");

  if(this != &other){
    TRACE("copying\n");
    discard();

    // CON_maprpids_to_conreadsids not copied on purpose

    CON_cebugflag=other.CON_cebugflag;

    CON_miraparams=other.CON_miraparams;

    CON_id=other.CON_id;
    CON_nameprefix=other.CON_nameprefix;
    CON_name=other.CON_name;
    CON_finalised=other.CON_finalised;

    CON_readpool=other.CON_readpool;
    CON_reads=other.CON_reads;
    CON_counts=other.CON_counts;
    CON_templates_present=other.CON_templates_present;
    CON_consensus_tags=other.CON_consensus_tags;

    CON_targetcoverageperst=other.CON_targetcoverageperst;

    CON_maprpids_to_conreadsids.clear();

    CON_2tmpcons=other.CON_2tmpcons;

    CON_tmpcons_from_backbone=other.CON_tmpcons_from_backbone;
    CON_mergenewsrreads=other.CON_mergenewsrreads;
    CON_hasforcemergeareas=other.CON_hasforcemergeareas;

    CON_specialsraddconditions=other.CON_specialsraddconditions;
    CON_ssrc_maxtotalerrors=other.CON_ssrc_maxtotalerrors;
    CON_ssrc_maxgaps=other.CON_ssrc_maxgaps;
    CON_ssrc_maxmismatches=other.CON_ssrc_maxmismatches;

    for(uint32 i=0; i<NUMMERGESEQTYPES; i++){
      CON_nummergedreads_perseqtype[i]=other.CON_nummergedreads_perseqtype[i];
    }
    
    CON_index_leftestbbread=other.CON_index_leftestbbread;
    CON_index_rightestbbread=other.CON_index_rightestbbread;

    CON_fixedconsseq=other.CON_fixedconsseq;
    CON_fixedconsqual=other.CON_fixedconsqual;

    // 
    CON_conscalc_mincov=other.CON_conscalc_mincov;
    CON_conscalc_minqual=other.CON_conscalc_minqual;
    CON_conscalc_missingchar=other.CON_conscalc_missingchar;
    CON_allconsseq=other.CON_allconsseq;
    CON_allconsqual=other.CON_allconsqual;
    CON_alladjustments=other.CON_alladjustments;
    CON_strainconsseq=other.CON_strainconsseq;
    CON_strainconsqual=other.CON_strainconsqual;
    CON_strainadjustments=other.CON_strainadjustments;

    //// TODO: phase out this block
    //CON_cheat_intelcons=other.CON_cheat_intelcons;
    //CON_cheat_intelconsqual=other.CON_cheat_intelconsqual;
    //CON_cheat_intelcons_markspecials=other.CON_cheat_intelcons_markspecials;
    //CON_cheat_intelcons_mincov=other.CON_cheat_intelcons_mincov;
    //CON_cheat_intelcons_minqual=other.CON_cheat_intelcons_minqual;
    //CON_cheat_intelcons_strainidtotake=other.CON_cheat_intelcons_strainidtotake;


    CON_allconsseq=other.CON_allconsseq;
    CON_allconsqual=other.CON_allconsqual;
    CON_strainconsseq=other.CON_strainconsseq;
    CON_strainconsqual=other.CON_strainconsqual;
    CON_readsperstrain=other.CON_readsperstrain;


    CON_outputorder=other.CON_outputorder;
    CON_last_dangerous_overlaps=other.CON_last_dangerous_overlaps;

    CON_contains_long_repeats_only=other.CON_contains_long_repeats_only;

    CON_stats=other.CON_stats;

    CON_valid=other.CON_valid;

  }

  FUNCEND();
  return *this;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Contig::setParams(vector<MIRAParameters> * params)
{
  FUNCSTART("Contig::setParams(MIRAParameters * params)");
  CON_miraparams=params;
  FUNCEND();
}



/*************************************************************************
 *
 *  TODO: rework for strain dependend data
 *
 *
 *************************************************************************/
int32 Contig::paddedPos2UnpaddedPos(uint32 padpos)
{
  FUNCSTART("int32 Contig::paddedPos2UnpaddedPos(uint32 padpos)");

  BUGIFTHROW(CON_alladjustments.empty(),"CON_alladjustments.empty()?");

  ensureConsensus(-1);

//  if( CON_finalised == false || CON_adjustments.size()!=CON_counts.size()){
//    finalise();
//    makeIntelligentConsensus(CON_cheat_intelcons,
//			     CON_cheat_intelconsqual,
//			     0,
//			     CON_counts.size(),
//			     false,
//			     0,
//			     0,
//			     -1,
//			     '@',
//			     NULL);
//  }

  BUGIFTHROW(CON_alladjustments.empty(),"CON_alladjustments.empty()?");
  if(padpos>=CON_alladjustments.size()) padpos=CON_alladjustments.size()-1;

  FUNCEND();
  return CON_alladjustments[padpos];
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla) {if(CON_cebugflag) cout << bla;} 


// new ... calculate main consensus and cache it
// consensus for strains: moved as "on-demand" calulation to getConsensus()

//#define CEBUG(bla) {cout << bla;} 
void Contig::calcConsensi(int32 mincoverage, base_quality_t minqual, char missingcoveragechar)
{

  FUNCSTART("void Contig::calcConsensi(int32 mincoverage, base_quality_t minqual, char missingcoveragechar)");

  //CON_cebugflag=true;

  CEBUG("calcConsensi(). mincov: " << mincoverage << "\tminqual: " << static_cast<uint16>(minqual) << "\tmissingcchar: " << missingcoveragechar << endl);

  // loading from different files (backbones reads etc) make the number of strain change over time
  // i.e., contigs loaded earlier may have a smaller CON_readsperstrain.size() than is good
  if(CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool()){
    CON_readsperstrain.resize(CON_readpool->getNumOfStrainInReadpool(),0);
  }

  CEBUG("Rebuild for " << CON_readsperstrain.size() << " strain in readpool\n");

  if(CON_conscalc_mincov!=mincoverage
     || CON_conscalc_minqual!=minqual
     || CON_conscalc_missingchar!=missingcoveragechar
     || CON_strainconsseq.size()==0
     || CON_readsperstrain.size() >= CON_strainconsseq.size()){

    CEBUG("need recalc. old values: " << CON_conscalc_mincov << " " << static_cast<uint16>(CON_conscalc_minqual) << " " << CON_conscalc_missingchar << endl);

    CON_conscalc_mincov=mincoverage;
    CON_conscalc_minqual=minqual;
    CON_conscalc_missingchar=missingcoveragechar;

    // make a common consensus for all strains
    // cache them in CON_allconsseq and *qual
    makeIntelligentConsensus(CON_allconsseq,
			     CON_allconsqual,
			     &CON_alladjustments,
			     NULL,
			     0,
			     CON_counts.size(),
			     0,
			     0,
			     -1,
			     missingcoveragechar);
    
    CEBUG("CON_readsperstrain.size(): " << CON_readsperstrain.size() << endl);

    CON_strainconsseq.clear();
    CON_strainconsqual.clear();
    CON_strainadjustments.clear();
    CON_strainconsseq.resize(CON_readsperstrain.size());
    CON_strainconsqual.resize(CON_readsperstrain.size());
    CON_strainadjustments.resize(CON_readsperstrain.size());

    uint32 numstrains=0;
    for(uint32 si=0; si<CON_readsperstrain.size(); si++){
      if(CON_readsperstrain[si]>0) numstrains++;
    }
    
    // now for all strains.
    // strains not present in contig will have sequence, quality and adjustments 
    //  pre-filled with default values (@, 0 and -1)
    // strains present will have empty seq+qual+adj ... to be claulated on demand

    // Change: no precalculated strains, only pre-filled for clear cases!
    // Should getConsensus() ask for them later, they
    //  will be calculated on demand (using CON_conscalc_* values)
    for(uint32 si=0; si<CON_readsperstrain.size(); si++){
      if(CON_readsperstrain[si]==0) {
	CON_strainconsseq[si].resize(CON_allconsseq.size(),'@');
	CON_strainconsqual[si].resize(CON_allconsqual.size(),0);
	CON_strainadjustments[si].resize(CON_allconsqual.size(),-1);
      }else{
	if(numstrains==1 && mincoverage==0 && minqual==0){
	  // take over from allstrains in this very special case
	  CON_strainconsseq[si]=CON_allconsseq;
	  CON_strainconsqual[si]=CON_allconsqual;
	  CON_strainadjustments[si]=CON_alladjustments;
	}
      }
    }
  }else{
  }

  FUNCEND();

  return;
}  
//#define CEBUG(bla)


// makes sure consensus and all adjoining structures are calculated and valid
// works like newConsensusGet() below, but does not return the consensus nor qualities
//
// trick: calling newConsensusGet() with the src==target will make sure things are not
//  unnecessarily copied
void Contig::ensureConsensus(int32 strainidtotake)
{
  FUNCSTART("void Contig::ensureConsensus(int32 strainidtotake)");

  CEBUG("ensureConsensus(): " << strainidtotake << endl);
  if(CON_allconsseq.empty() || CON_strainconsseq.empty()){
    CEBUG("ensureConsensus(): cons empty, need recalc" << endl);
    calcConsensi();
  }
  if(strainidtotake<0){
    newConsensusGet(CON_allconsseq, CON_allconsqual, strainidtotake);
  }else{
    newConsensusGet(CON_strainconsseq[strainidtotake], CON_strainconsqual[strainidtotake], strainidtotake);
  }
  FUNCEND();
}

void Contig::newConsensusGet(string & target, vector<base_quality_t> & qual, int32 strainidtotake)
{
  FUNCSTART("void Contig::newConsensusGet(string & target, vector<base_quality_t> & qual, int32 strainidtotake)");

//  if(CON_abortflag) {
//    uint16 * bombme=NULL;
//    *bombme=0xdead;
//    cout << bombme;
//  }

  CEBUG("newConsensusGet(): gimme strain " << strainidtotake << endl);

  // loading from different files (backbones reads etc) make the number of strain change over time
  // i.e., contigs loaded earlier may have a smaller CON_readsperstrain.size() than is good
  if(CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool()){
    CON_readsperstrain.resize(CON_readpool->getNumOfStrainInReadpool(),0);
  }
  //BUGIFTHROW(CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool(), "CON_readsperstrain.size() (" << CON_readsperstrain.size() << ") < CON_readpool->getNumOfStrainInReadpool() (" << CON_readpool->getNumOfStrainInReadpool() << ") ?");

  BUGIFTHROW(strainidtotake>=static_cast<int32>(CON_readsperstrain.size()),"strainidtotake>=CON_readsperstrain.size() ?");

  if(CON_fixedconsseq.size() && CON_fixedconsqual.size() && strainidtotake<0){
    CEBUG("newConsensusGet(): get fixed" << endl);
    target=CON_fixedconsseq;
    qual=CON_fixedconsqual;
  }else{
    CEBUG("CON_allconsseq.size(): " << CON_allconsseq.size() << endl);
    CEBUG("CON_strainconsseq.size(): " << CON_strainconsseq.size() << endl);
    if(CON_allconsseq.empty() || CON_strainconsseq.empty()
       || (strainidtotake>=0 && strainidtotake>=CON_strainconsseq.size())){
      CEBUG("something's empty, need recalc" << endl);
      calcConsensi();
    }
    if(strainidtotake<0){
      CEBUG("newConsensusGet(): get allcons" << endl);
      CEBUG("CON_allconsseq.size(): " << CON_allconsseq.size() << endl);
      CEBUG("CON_allconsqual.size(): " << CON_allconsqual.size() << endl);
      target=CON_allconsseq;
      qual=CON_allconsqual;
    }else{
      CEBUG("newConsensusGet(): get strain " << strainidtotake << endl);
      BUGIFTHROW(strainidtotake>=CON_strainconsseq.size(),"something's utterly wrong: strainidtotake>=CON_strainconsseq.size() ???");
      // on demand calculation
      if(CON_strainconsseq[strainidtotake].empty()){
	CEBUG("check on demand calculation\n");
	uint32 numstrains=0;
	for(uint32 si=0; si<CON_readsperstrain.size(); si++){
	  if(CON_readsperstrain[si]>0) numstrains++;
	}
	if(numstrains==1){
	  CEBUG("only 1 strain, can take main consensus\n");
	  CON_strainconsseq[strainidtotake]=CON_allconsseq;
	  CON_strainconsqual[strainidtotake]=CON_allconsqual;
	  CON_strainadjustments[strainidtotake]=CON_alladjustments;
	}else{
	  CEBUG("must do calculation\n");
	  makeIntelligentConsensus(CON_strainconsseq[strainidtotake],
				   CON_strainconsqual[strainidtotake],
				   &CON_strainadjustments[strainidtotake],
				   NULL,
				   0,
				   CON_counts.size(),
				   CON_conscalc_mincov,
				   CON_conscalc_minqual,
				   strainidtotake,
				   CON_conscalc_missingchar);
	}
      }else{
	CEBUG("take cached\n");
      }
      target=CON_strainconsseq[strainidtotake];
      qual=CON_strainconsqual[strainidtotake];
    }
  }

  FUNCEND();
}
//#define CEBUG(bla)


//void Contig::OLDgetConsensus1(string & target, vector<base_quality_t> & qual, bool markspecials, int32 mincoverage, base_quality_t minqual, int32 strainidtotake, char missingcoveragechar, ostream * ostr, bool contagsintcs)
//{
//  FUNCSTART("void Contig::getConsensus(string & target, vector<base_quality_t> & qual, bool markspecials, int32 mincoverage, int32 strainidtotake, ostream * ostr, bool contagsintcs)");
//
//  CEBUG("getCons()\n");
//
//  CON_cebugflag=true;
//
//  if(CON_cheat_intelcons.empty() 
//     || CON_cheat_intelcons_markspecials!=markspecials 
//     || CON_cheat_intelcons_mincov!=mincoverage
//     || CON_cheat_intelcons_minqual!=minqual
//     || CON_cheat_intelcons_strainidtotake!=strainidtotake
//     || ostr != NULL) {
//    bool mustcompute=true;
//    if(!CON_cheat_intelcons.empty() 
//       && CON_cheat_intelcons_mincov==mincoverage
//       && CON_cheat_intelcons_minqual==minqual
//       && CON_cheat_intelcons_strainidtotake == strainidtotake) {
//      if(markspecials) {
//	if(CON_cheat_intelcons_markspecials) {
//	  mustcompute=false;
//	}
//      } else {
//	mustcompute=false;
//	if(CON_cheat_intelcons_markspecials) {
//	  // just make the sequence make uppercase
//	  for(uint32 i=0; i<CON_cheat_intelcons.size(); i++) {
//	    CON_cheat_intelcons[i]=toupper(CON_cheat_intelcons[i]);
//	  }
//	} 
//      }
//    } 
//    if(mustcompute || ostr != NULL) {
//      makeIntelligentConsensus(CON_cheat_intelcons,
//			       CON_cheat_intelconsqual,
//			       0,
//			       CON_counts.size(),
//			       markspecials,
//			       mincoverage,
//			       minqual,
//			       strainidtotake,
//			       missingcoveragechar,
//			       ostr,
//			       contagsintcs);
//    }
//    CON_cheat_intelcons_markspecials=markspecials;
//    CON_cheat_intelcons_mincov=mincoverage;
//    CON_cheat_intelcons_minqual=minqual;
//    CON_cheat_intelcons_strainidtotake=strainidtotake;
//  }
//
//  target=CON_cheat_intelcons;
//  qual=CON_cheat_intelconsqual;
//
//  FUNCEND();
//
//  return;
//}  






/*************************************************************************
 *
 * if the routines to decide for a base (helper2 routines) could not get 
 *  clear base but the user wants one, this is doing a shootout based on
 *  majority vote.
 *
 * groups with forward and reverse count double in the read count
 *
 * if majority vote still does not work (all the same), then the last
 *  one wins. I.e., * takes precedence over T, this over G, over C, over A
 *
 *************************************************************************/

void Contig::makeIntelligentConsensus_helper3(char & thisbase, base_quality_t & thisqual, const vector<nngroups_t> & groups, const vector<char> & IUPACbasegroups)
{
  uint32 maxcount=0;

  for(uint32 actgroup=0; actgroup<groups.size(); actgroup++){
    for(uint32 actbase=0; actbase<IUPACbasegroups.size(); actbase++){
      if(IUPACbasegroups[actbase]==groups[actgroup].base){
	uint32 groupcount=groups[actgroup].ids.size();
	if(groups[actgroup].forwarddircounter>0
	   && groups[actgroup].complementdircounter>0){
	  groupcount*=2;
	}
	if(groupcount>=maxcount){
	  maxcount=groupcount;
	  thisbase=groups[actgroup].base;
	  thisqual=groups[actgroup].groupquality;
	}
      }
    }
  }

}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
//#define FUNCSTART(bla)  static const char * THISFUNC = bla"  "; {cout << "enter " << THISFUNC << "\n"; cout.flush();}
//#define FUNCEND() {cout << "exit " << THISFUNC << "\n"; cout.flush();}


//#define CEBUG(bla) {cout << bla;} 
void Contig::makeIntelligentConsensus_helper2_calcSOLEXA(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, cccontainer_t::const_iterator ccI, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups, const base_quality_t maxqual, const uint32 maxcount, int32 strainidtotake)
{
  FUNCSTART("void Contig::makeIntelligentConsensus_helper2_calcSOLEXA(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, cccontainer_t::const_iterator ccI, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups, const base_quality_t maxqual, const uint32 maxcount, int32 strainidtotake)");

  //makeIntelligentConsensus_helper3(thisbase,
  //				   thisqual,
  //				   groups,
  //				   IUPACbasegroups);
  //return;

  IUPACbasegroups.clear();


  // TODO: clarify whether to fill IUPACbasegroups also with merged base??

  bool hasmergedbases=(ccI->backbonechar!='@') & (ccI->bbcounts[0]>0);
  bool mergedalsogroup=false;
  bool mergedisvalid=false;

  CEBUG("bbchar: " << ccI->backbonechar << "\tbbcounts: " << ccI->bbcounts[0] << "\tbbbestquals: " << static_cast<uint16>(ccI->bbbestquals[0]) << "\tbbstrains: " << hex << static_cast<uint16>(ccI->bbstrains[0]) << dec << "\tHasmergedb: " << hasmergedbases <<'\n');

  // Ok, check whether the merged bases belong to this strain
  //  if not, well then no merged bases exist
  uint8 strainmask=255;

  CEBUG("Strainidtotake: " << strainidtotake << '\n');
  if(strainidtotake>=0) strainmask=getBBStrainMask(strainidtotake);
  if(!(ccI->bbstrains[0] & strainmask)) hasmergedbases=false;

  uint32 numgroups=0;
  base_quality_t lastseengroupqual=0;
  char lastseenbase='&';

  bool groupschosen[groups.size()];

  // search for excellent groups
  for(uint32 i=0; i<groups.size(); i++){
    groupschosen[i]=false;
    CEBUG("Lookatgroup: " << groups[i] << '\n');
    if(hasmergedbases && ccI->backbonechar == groups[i].base){
      if((groups[i].ids.size() + ccI->bbcounts[0]) >= 10
	 && (ccI->bbbestquals[0] >=35
	     || (groups[i].groupquality >40
		 && groups[i].forwarddircounter>0
		 && groups[i].complementdircounter>0))){
	// >= 6 solexa reads with qual >=30
	CEBUG("Excellent1: " << groups[i].base << '\n');
	numgroups++;
	groupschosen[i]=true;
	lastseengroupqual=groups[i].groupquality;
	lastseenbase=groups[i].base;
	mergedisvalid=true;
	if(!groups[i].ids.empty()) mergedalsogroup=true;
      }
    }else{
      // or >=10 normal reads, gqual>=40 and forward/reverse directions
      //  are at least 25% of all reads
      if(groups[i].ids.size()>=10
	 && groups[i].groupquality >=40
	 && groups[i].forwarddircounter>=groups[i].ids.size()/4
	 && groups[i].complementdircounter>=groups[i].ids.size()/4) {
	CEBUG("Excellent2: " << groups[i].base << '\n');
	numgroups++;
	groupschosen[i]=true;
	lastseengroupqual=groups[i].groupquality;
	lastseenbase=groups[i].base;
      }
    }
  }
  CEBUG("Numgroups excel: " << numgroups << '\n');
  CEBUG("Hasmergedbases: " << hasmergedbases << "\tMergedalsogroup: " << mergedalsogroup << '\n');


  // search for high groups
  if(numgroups==0){
    for(uint32 i=0; i<groups.size(); i++){
      CEBUG("Lookatgroup: " << groups[i] << '\n');
      if(hasmergedbases && ccI->backbonechar == groups[i].base){
	if((groups[i].ids.size() + ccI->bbcounts[0]) >= 6
	   && (ccI->bbbestquals[0] >=30
	       || (groups[i].groupquality >=30
		   && groups[i].forwarddircounter>0
		   && groups[i].complementdircounter>0))){
	  // >= 6 solexa reads with qual >=30
	  CEBUG("High1: " << groups[i].base << '\n');
	  numgroups++;
	  groupschosen[i]=true;
	  lastseengroupqual=groups[i].groupquality;
	  lastseenbase=groups[i].base;
	  mergedisvalid=true;
	  if(!groups[i].ids.empty()) mergedalsogroup=true;
	}
      }else{
	// or >=10 normal reads, gqual>=30 and forward/reverse
	if(groups[i].ids.size()>=6
	   && groups[i].groupquality >=30
	   && groups[i].forwarddircounter>0
	   && groups[i].complementdircounter>0) {
	  CEBUG("High2: " << groups[i].base << '\n');
	  numgroups++;
	  groupschosen[i]=true;
	  lastseengroupqual=groups[i].groupquality;
	  lastseenbase=groups[i].base;
	}
      }
    }
    CEBUG("Numgroups high: " << numgroups << '\n');
    CEBUG("Hasmergedbases: " << hasmergedbases << "\tMergedalsogroup: " << mergedalsogroup << '\n');
  }

  // TODO: insert good. fwd/rev and qual vs non-fwd/rev and qual

  // search for good groups. Same as high, but without mincount
  if(numgroups==0){
    for(uint32 i=0; i<groups.size(); i++){
      CEBUG("Lookatgroup: " << groups[i] << '\n');
      if(hasmergedbases && ccI->backbonechar == groups[i].base){
	if(ccI->bbbestquals[0] >=30
	    || (groups[i].groupquality >=30
		&& groups[i].forwarddircounter>0
		&& groups[i].complementdircounter>0)){
	  // >= 6 solexa reads with qual >=30
	  CEBUG("Good1: " << groups[i].base << '\n');
	  numgroups++;
	  groupschosen[i]=true;
	  lastseengroupqual=groups[i].groupquality;
	  lastseenbase=groups[i].base;
	  mergedisvalid=true;
	  if(!groups[i].ids.empty()) mergedalsogroup=true;
	}
      }else{
	// or gqual>=30 and forward/reverse
	if(groups[i].groupquality >=30
	   && groups[i].forwarddircounter>0
	   && groups[i].complementdircounter>0) {
	  CEBUG("Good2: " << groups[i].base << '\n');
	  numgroups++;
	  groupschosen[i]=true;
	  lastseengroupqual=groups[i].groupquality;
	  lastseenbase=groups[i].base;
	}
      }
    }
    CEBUG("Numgroups good: " << numgroups << '\n');
    CEBUG("Hasmergedbases: " << hasmergedbases << "\tMergedalsogroup: " << mergedalsogroup << '\n');
  }

  // search for medium groups
  if(numgroups==0){
    for(uint32 i=0; i<groups.size(); i++){
      if(!groups[i].ids.empty() && groups[i].groupquality >=30) {
	numgroups++;
	groupschosen[i]=true;
	lastseengroupqual=groups[i].groupquality;
	lastseenbase=groups[i].base;
	if(hasmergedbases 
	   && ccI->backbonechar == groups[i].base
	   && ccI->bbbestquals[0] >=30) {
	  mergedisvalid=true;
	  mergedalsogroup=true;
	}
      }else{
	if(hasmergedbases && ccI->backbonechar == groups[i].base
	   && ccI->bbbestquals[0] >=30){
	  mergedisvalid=true;
	  numgroups++;
	  groupschosen[i]=true;
	}
      }
    }
    CEBUG("Numgroups medium: " << numgroups << '\n');
    CEBUG("Hasmergedbases: " << hasmergedbases << "\tMergedalsogroup: " << mergedalsogroup << '\n');
  }

  bool badquals=false;
  if(numgroups==0){
    // no read in the groups? still might be ... but with bad quals.
    CEBUG("No group first time, redo without qual.\n");

    for(uint32 i=0; i<groups.size(); i++){
      if(!groups[i].ids.empty()) {
	numgroups++;
	groupschosen[i]=true;
	lastseengroupqual=groups[i].groupquality;
	lastseenbase=groups[i].base;
	if(hasmergedbases 
	   && ccI->backbonechar == groups[i].base) {
	  mergedisvalid=true;
	  mergedalsogroup=true;
	}
      }else{
	if(hasmergedbases && ccI->backbonechar == groups[i].base) {
	  mergedisvalid=true;
	  numgroups++;
	  groupschosen[i]=true;
	}
      }
    }
    badquals=true;

    CEBUG("Numgroups: " << numgroups << "\tbadquals: " << badquals << '\n');
    CEBUG("Hasmergedbases: " << hasmergedbases << "\tMergedalsogroup: " << mergedalsogroup << '\n');
  }

  char maxcount_base='%';
  base_quality_t maxcount_qual=0;

  if(numgroups==1){
    // just one group
    // see whether it is a valid merged base
    if(hasmergedbases && mergedisvalid) {
      // if merged, is it also covered by a readgroup?
      if(mergedalsogroup){
	// yes
	thisbase=ccI->backbonechar;
	thisqual=max(lastseengroupqual, ccI->bbbestquals[0]);
      }else{
	// no, only a merged base
	thisbase=ccI->backbonechar;
	thisqual=ccI->bbbestquals[0];
      }
    }else{
      // no, it's only one normal read group
      thisbase=lastseenbase;
      IUPACbasegroups.push_back(thisbase);
      thisqual=lastseengroupqual;
    }
  } else if(numgroups==0){
    thisbase='N';
    thisqual=0;
  } else {
    // let the fun begin ... *sigh*
    // at least two groups were detected, which means any combination of
    //  normal read groups (1-5) and eventually also a merged base
    // the groups were either detected all at good qual or all at bad qual

    // first decision taken on counts of bases
    vector<uint32> counts(groups.size());
    vector<base_quality_t> quals(groups.size());
    uint32 maxcounts=0;
    base_quality_t maxquals=0;      // carefull, this is not maxqual 
    // from the parameter (rename that)
    uint32 totalcounts=0;
    bool hasgapchosen=false;

    for(uint32 i=0; i<groups.size(); i++){
      counts[i]=groups[i].ids.size();
      quals[i]=groups[i].groupquality;
      if(groupschosen[i] && groups[i].base == '*') hasgapchosen=true;
      if(hasmergedbases
	 && ccI->backbonechar == groups[i].base) {
	counts[i]+=ccI->bbcounts[0];
	quals[i]=max(quals[i],ccI->bbbestquals[0]);
      }
      totalcounts+=counts[i];
      maxcounts=max(maxcounts,counts[i]);
      maxquals=max(maxquals,quals[i]);
      if(maxcounts==counts[i]){
	maxcount_base=groups[i].base;
	maxcount_qual=quals[i];
      }
    }

    CEBUG("Maxcounts: " << maxcounts << "\ttotalcounts: " << totalcounts << '\n');

    thisbase=' ';
    thisqual=0;
    uint32 numtaken=0;
    for(uint32 i=0; i<groups.size(); i++){
      if(groupschosen[i]){
	numtaken++;
	if(thisbase==' ') {
	  thisbase=groups[i].base;
	  thisqual=quals[i];
	}else{
	  // take different approaches depending on whether user forces
	  //  a non-IUPAC or not
	  //  or whether a gap is part of the best groups
	  if(counts[i] == maxcounts
	     && (hasgapchosen
		 || (*CON_miraparams)[Read::SEQTYPE_SOLEXA].getContigParams().con_force_nonIUPACconsensus_perseqtype)) {
	    thisbase=groups[i].base;
	    thisqual=quals[i];
	    numtaken=1;
	  }else{
	    thisbase=dptools::calcIUPACConsensus(thisbase, groups[i].base);
	    thisqual+=quals[i];
	  }
	}
	if(!groups[i].ids.empty()) IUPACbasegroups.push_back(thisbase);
      }
    }
    if(numtaken) thisqual/=numtaken;
  }

  CEBUG("Settled on: '" << thisbase << "' " << static_cast<uint16>(thisqual) << '\n');

  if(!dptools::isValidACGTStarBase(thisbase)
     && maxcount_base!='%'
     && (*CON_miraparams)[Read::SEQTYPE_SOLEXA].getContigParams().con_force_nonIUPACconsensus_perseqtype) {
    thisbase=maxcount_base;
    thisqual=maxcount_qual;
    CEBUG("User forced non-IUPAC per seqtype: '" << thisbase << "' " << static_cast<uint16>(thisqual) << '\n');
  }

  FUNCEND();
  return;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::makeIntelligentConsensus_helper2_calcPACBIO(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups)
{

  // no info. atm, same as 454 without 40:60 rule

  // Idea: pure coverage, maximum wins.
  //       if two or more with same maximum: 
  //          IUPAC (star goes under, sorry) or
  //          if non-IUPAC is wished, take the last
  // TODO: is there a better way?

  IUPACbasegroups.clear();

  int32 maxsize=0;
  int32 maxsize_i=-1;
  base_quality_t maxqual=0;
  int32 runnerup=0;
  int32 runnerup_i=-1;
  base_quality_t runnerupqual=0;

  size_t totalsize=0;

  for(uint32 i=0; i<groups.size(); i++){
    totalsize+=groups[i].ids.size();
    if(static_cast<int32>(groups[i].ids.size())>=maxsize){
      runnerup=maxsize;
      runnerup_i=maxsize_i;
      runnerupqual=maxqual;
      maxsize=static_cast<int32>(groups[i].ids.size());
      maxsize_i=i;
      maxqual=groups[i].groupquality;
    }else if(static_cast<int32>(groups[i].ids.size())>=runnerup){
      runnerup=static_cast<int32>(groups[i].ids.size());
      runnerup_i=i;
      runnerupqual=groups[i].groupquality;
    }
  }

  int32 avgqual=0;
  for(uint32 i=0; i<groups.size(); i++){
    if(maxsize==static_cast<int32>(groups[i].ids.size())){
      IUPACbasegroups.push_back(groups[i].base);
      avgqual+=groups[i].groupquality;
    }
  }

  if(totalsize==0 || IUPACbasegroups.empty()){
    /// Oooops? just an N???
    thisbase='N';
    thisqual=0;
    return;
  }

  if(IUPACbasegroups.size()==1) {
    thisbase=IUPACbasegroups[0];
    // reduce quality if there are doubts
    if(runnerup>0 && maxsize-runnerup < 10){
      avgqual-=runnerupqual;
      if(avgqual<0) avgqual=max(abs(avgqual),10);
    }
    thisqual=static_cast<base_quality_t>(avgqual);
  }else{
    if((*CON_miraparams)[Read::SEQTYPE_PACBIO].getContigParams().con_force_nonIUPACconsensus_perseqtype) {
      thisbase=IUPACbasegroups[IUPACbasegroups.size()-1];
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }else{
      thisbase=dptools::calcIUPACConsensus(IUPACbasegroups);
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }
  }

  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::makeIntelligentConsensus_helper2_calc454GS20(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups)
{
  // Idea: pure coverage, maximum wins.
  //       if two or more with same maximum: 
  //          IUPAC (star goes under, sorry) or
  //          if non-IUPAC is wished, take the last
  // TODO: is there a better way?

  IUPACbasegroups.clear();

  int32 maxsize=0;
  int32 maxsize_i=-1;
  base_quality_t maxqual=0;
  int32 runnerup=0;
  int32 runnerup_i=-1;
  base_quality_t runnerupqual=0;

  size_t totalsize=0;

  for(uint32 i=0; i<groups.size(); i++){
    totalsize+=groups[i].ids.size();
    if(static_cast<int32>(groups[i].ids.size())>=maxsize){
      runnerup=maxsize;
      runnerup_i=maxsize_i;
      runnerupqual=maxqual;
      maxsize=static_cast<int32>(groups[i].ids.size());
      maxsize_i=i;
      maxqual=groups[i].groupquality;
    }else if(static_cast<int32>(groups[i].ids.size())>=runnerup){
      runnerup=static_cast<int32>(groups[i].ids.size());
      runnerup_i=i;
      runnerupqual=groups[i].groupquality;
    }
  }

  // if max count is gap, but there are other bases
  if(maxsize_i==4 && runnerup>0){
    // apply 40:60 rule
    if(100*runnerup/(maxsize+runnerup) >= 40){
      swap(maxsize,runnerup);
      swap(maxsize_i,runnerup_i);
      swap(maxqual,runnerupqual);
    }
  }

  int32 avgqual=0;
  for(uint32 i=0; i<groups.size(); i++){
    if(maxsize==static_cast<int32>(groups[i].ids.size())){
      IUPACbasegroups.push_back(groups[i].base);
      avgqual+=groups[i].groupquality;
    }
  }

  if(totalsize==0 || IUPACbasegroups.empty()){
    /// Oooops? just an N???
    thisbase='N';
    thisqual=0;
    return;
  }

  if(IUPACbasegroups.size()==1) {
    thisbase=IUPACbasegroups[0];
    // reduce quality if there are doubts
    if(runnerup>0 && maxsize-runnerup < 10){
      avgqual-=runnerupqual;
      if(avgqual<0) avgqual=max(abs(avgqual),10);
    }
    thisqual=static_cast<base_quality_t>(avgqual);
  }else{
    if((*CON_miraparams)[Read::SEQTYPE_454GS20].getContigParams().con_force_nonIUPACconsensus_perseqtype) {
      thisbase=IUPACbasegroups[IUPACbasegroups.size()-1];
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }else{
      thisbase=dptools::calcIUPACConsensus(IUPACbasegroups);
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }
  }

  //// TODO: testing
  //if(maxsize_i>=0 && runnerup_i>=0){
  //  if(maxsize+runnerup == 0){
  //    //addTagToConsensus(actcontigpos, actcontigpos,'=',"P454","DOH?");
  //  }else if(100*runnerup/(maxsize+runnerup) >= 30){
  //    //ostringstream ostr;
  //    //ostr << static_cast<char>(groups[maxsize_i].base) << ": " << maxsize;
  //    //ostr << " " << static_cast<char>(groups[runnerup_i].base) << ": " << runnerup;
  //    //ostr << "  -  " << 100*runnerup/(maxsize+runnerup) << "%";
  //    //
  //    //addTagToConsensus(actcontigpos, actcontigpos,'=',"P454",ostr.str().c_str());
  //  }
  //}

  return;
}



/*************************************************************************
 *
 * atm a pure copy of 454
 *
 *
 *************************************************************************/

void Contig::makeIntelligentConsensus_helper2_calcIonTorrent(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups)
{
  // Idea: pure coverage, maximum wins.
  //       if two or more with same maximum: 
  //          IUPAC (star goes under, sorry) or
  //          if non-IUPAC is wished, take the last
  // TODO: is there a better way?

  IUPACbasegroups.clear();

  int32 maxsize=0;
  int32 maxsize_i=-1;
  base_quality_t maxqual=0;
  int32 runnerup=0;
  int32 runnerup_i=-1;
  base_quality_t runnerupqual=0;

  size_t totalsize=0;

  for(uint32 i=0; i<groups.size(); i++){
    totalsize+=groups[i].ids.size();
    if(static_cast<int32>(groups[i].ids.size())>=maxsize){
      runnerup=maxsize;
      runnerup_i=maxsize_i;
      runnerupqual=maxqual;
      maxsize=static_cast<int32>(groups[i].ids.size());
      maxsize_i=i;
      maxqual=groups[i].groupquality;
    }else if(static_cast<int32>(groups[i].ids.size())>=runnerup){
      runnerup=static_cast<int32>(groups[i].ids.size());
      runnerup_i=i;
      runnerupqual=groups[i].groupquality;
    }
  }

  // if max count is gap, but there are other bases
  // Test, maybe IonTorrent is a bit different.
  // Actually, it's almost the best rule there is ...
  //  35 perhaps better, but needs to be verified.
  // Astonishing
  if(maxsize_i==4 && runnerup>0){
    // apply 40:60 rule
    if(100*runnerup/(maxsize+runnerup) >= 40){
      swap(maxsize,runnerup);
      swap(maxsize_i,runnerup_i);
      swap(maxqual,runnerupqual);
    }
  }

  int32 avgqual=0;
  for(uint32 i=0; i<groups.size(); i++){
    if(maxsize==static_cast<int32>(groups[i].ids.size())){
      IUPACbasegroups.push_back(groups[i].base);
      avgqual+=groups[i].groupquality;
    }
  }

  if(totalsize==0 || IUPACbasegroups.empty()){
    /// Oooops? just an N???
    thisbase='N';
    thisqual=0;
    return;
  }

  if(IUPACbasegroups.size()==1) {
    thisbase=IUPACbasegroups[0];
    // reduce quality if there are doubts
    if(runnerup>0 && maxsize-runnerup < 10){
      avgqual-=runnerupqual;
      if(avgqual<0) avgqual=max(abs(avgqual),10);
    }
    thisqual=static_cast<base_quality_t>(avgqual);
  }else{
    if((*CON_miraparams)[Read::SEQTYPE_454GS20].getContigParams().con_force_nonIUPACconsensus_perseqtype) {
      thisbase=IUPACbasegroups[IUPACbasegroups.size()-1];
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }else{
      thisbase=dptools::calcIUPACConsensus(IUPACbasegroups);
      thisqual=static_cast<base_quality_t>(avgqual/IUPACbasegroups.size());
    }
  }

  //// TODO: testing
  //if(maxsize_i>=0 && runnerup_i>=0){
  //  if(maxsize+runnerup == 0){
  //    //addTagToConsensus(actcontigpos, actcontigpos,'=',"P454","DOH?");
  //  }else if(100*runnerup/(maxsize+runnerup) >= 30){
  //    //ostringstream ostr;
  //    //ostr << static_cast<char>(groups[maxsize_i].base) << ": " << maxsize;
  //    //ostr << " " << static_cast<char>(groups[runnerup_i].base) << ": " << runnerup;
  //    //ostr << "  -  " << 100*runnerup/(maxsize+runnerup) << "%";
  //    //
  //    //addTagToConsensus(actcontigpos, actcontigpos,'=',"P454",ostr.str().c_str());
  //  }
  //}

  return;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::makeIntelligentConsensus_helper2_calcSangerQual(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, const vector<nngroups_t> & groups, vector<char> & IUPACbasegroups, const vector<char> & columnbases, const base_quality_t maxqual, const uint32 maxcount)
{
  // Idea: all groups with qual >= 30
  //    or groups with qual within (X) (with X==5 now) of maxqual
  //    or groups with count >= 3 and quality >=20 and forward/reverse

  IUPACbasegroups.clear();
  int32 avgqual=0;
  int32 groupstaken=0;
  int32 verygoodgroups=0;

  // New: if force non-IUPAC consensus is wished
  // decrease the level with which we look at the groups
  // gradually until we have to find something (qual == 0)
  base_quality_t goodgroupqual=35;
  base_quality_t lessergroupqual=goodgroupqual;
  if(lessergroupqual>=10) lessergroupqual-=10;

  do{
    if(goodgroupqual>=5) goodgroupqual-=5;
    if(lessergroupqual>=5) lessergroupqual-=5;
    for(uint32 i=0; i<groups.size(); i++){
      if(groups[i].ids.size() 
	 &&(groups[i].groupquality >= goodgroupqual
	    || groups[i].groupquality+5 >= maxqual
	    || (groups[i].ids.size() >= 3          // TODO: check 2 or 3 (was 3, testing with 2)
		&& groups[i].groupquality >= lessergroupqual
		&& groups[i].forwarddircounter>0
		&& groups[i].complementdircounter>0))) {
	avgqual+=groups[i].groupquality;
	IUPACbasegroups.push_back(groups[i].base);
	groupstaken++;
	
	if(groups[i].groupquality >= goodgroupqual
	   && groups[i].forwarddircounter>0
	   && groups[i].complementdircounter>0) {
	  verygoodgroups++;
	}
      }
    }
  } while((*CON_miraparams)[Read::SEQTYPE_SANGER].getContigParams().con_force_nonIUPACconsensus_perseqtype
	  && groupstaken==0 && goodgroupqual>0);
  
  CEBUG("groups taken: " << groupstaken << "\t");
#ifdef CEBUGFLAG
  {
    for(uint32 i=0; i< IUPACbasegroups.size(); i++){
      CEBUG(IUPACbasegroups[i]);
    }
  }
#endif
  CEBUG(endl);
  if(groupstaken == 0) {
    // this can still happen if we do not force a non-IUPAC
    // or if it's a column entirely made of N (or X) (or both)

    // well, calculate the base from all columnbases
    thisbase=dptools::calcIUPACConsensus(columnbases);
    avgqual=0;
    int32 n=0;
    uint32 totalids=0;
    for(uint32 i=0; i<groups.size()-1; i++){
      if(groups[i].ids.size()) {
	totalids+=groups[i].ids.size();
	avgqual+=groups[i].groupquality;
	n++;
      }
    }
    // test if there are more stars than other bases
    if(groups[groups.size()-1].ids.size() > totalids) {
      // oh well, more stars than bases, make it a star
      thisbase='*';
      thisqual=groups[groups.size()-1].groupquality;
    } else if(n) {
      thisqual=avgqual/n;
    } else {
      thisqual=0;
    }
  } else if(groupstaken == 1) {
    // only one group taken (perfect)
    thisbase=IUPACbasegroups.front();
    int32 bahqual=maxqual;
    for(uint32 i=0; i<groups.size(); i++){
      if(groups[i].base!=thisbase) {
	bahqual-=groups[i].groupquality;
      }
    }
    if(bahqual<0) bahqual=0;
    thisqual=bahqual;
  } else {
    // ouch, more than one group was taken as good

    bool needIUPACconsensus=true;

    CEBUG("cpos: " << actcontigpos << "\tpossible IUPAC\tvgg: " << verygoodgroups << endl);
    //for(uint32 i=0; i< groups.size(); i++){
    //  cout << groups[i] << endl;
    //}

    // First, check whether we have a very good group
    //  as searched for above
    if(verygoodgroups==1){
      // if yes, rebuild IUPACbasegroups with very good groups
      // then, if only one group remains, perfect. Else it'll be
      //  a IUPAC

      CEBUG("recheck group: ");

      vector<char> newIUPACbasegroups;
      int32 newavgqual=0;
      int32 newgroupstaken=0;
      for(uint32 i=0; i<groups.size(); i++){
	if(groups[i].ids.size() 
	   && (groups[i].groupquality >= goodgroupqual
	       && groups[i].forwarddircounter>0
	       && groups[i].complementdircounter>0)){
	  newavgqual+=groups[i].groupquality;
	  newIUPACbasegroups.push_back(groups[i].base);
	  newgroupstaken++;
	}
      }
      
      if(newgroupstaken == 1) {
	CEBUG("success!");
	needIUPACconsensus=false;

	IUPACbasegroups=newIUPACbasegroups;
	avgqual=newavgqual;
	groupstaken=newgroupstaken;

	// only one group taken (perfect)
	thisbase=IUPACbasegroups.front();
	int32 bahqual=maxqual;
	for(uint32 i=0; i<groups.size(); i++){
	  if(groups[i].base!=thisbase) {
	    bahqual-=groups[i].groupquality;
	  }
	}
	if(bahqual<0) bahqual=0;
	thisqual=bahqual;
      }
    }
    CEBUG(endl);

    if(needIUPACconsensus){
      if((*CON_miraparams)[Read::SEQTYPE_SANGER].getContigParams().con_force_nonIUPACconsensus_perseqtype){
	// well, can't get a good decision, but user wants a non-IUPAC
	// make a majority-vote shootout among the chosen
	//   groups
	makeIntelligentConsensus_helper3(thisbase,
					 thisqual,
					 groups,
					 IUPACbasegroups);
      }else{
	// we'll have to make 
	//  a IUPAC consensus here. Alas, if a star (*) is part of this group,
	//  it will 'go under' (the good bases always win). So check if the star
	//  has max quality.
	bool isstar=false;
	if(groups[groups.size()-1].groupquality == maxqual) {
	  // yep, now check if one of the bases has the same qual
	  isstar=true;
	  for(uint32 i=0; i<groups.size()-1; i++){
	    if(groups[i].groupquality == maxqual) {
	      // in dubio pro reo: it could be a base 
	      isstar=false;
	    }
	  }
	}
	CEBUG("isstar: " << isstar << endl);
	if(isstar) {
	  thisbase='*';
	  thisqual=groups[groups.size()-1].groupquality;
	} else {
	  // really several different good bases.
	  thisqual=avgqual/groupstaken;
	  thisbase=dptools::calcIUPACConsensus(IUPACbasegroups);
	}
      }
    }
  }
}



/*************************************************************************
 *
 * For a given readtype, return thisbase and thisqual as result for this
 *  position in the contig
 * Also return the potential number of solutions (bases), the one which
 *  was chosen plus the ones which were not
 *
 * all other arguments are passovers from the main function (saving STL
 *  setup and memory allocation time)
 *
 * Side effects: manifold. Most notably: all the passovers from the
 *  main function which are not const will be overwritten with values
 *  reflecting the calculation of base and quality of this read type
 *
 * E.g.: the bases that were considered to make the consensus are in 
 *  IUPACbasegroups, the base-groups in groups etc.
 *
 *************************************************************************/

//#define CEBUG(bla) {cout << bla;} 

void Contig::makeIntelligentConsensus_helper1(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, cccontainer_t::const_iterator ccI, const int32 mincoverage, vector<nngroups_t> & groups, vector<nngroups_t> & maskedshadowgroups, vector<char> & IUPACbasegroups, vector<char> & columnbases, const vector<int32> & read_ids_in_col, vector<int8> & maskshadow, uint8 actreadtype, int32 strainidtotake, char missingcoveragechar)
{
  FUNCSTART("void Contig::makeIntelligentConsensus_helper1(char & thisbase, base_quality_t & thisqual, const uint32 actcontigpos, cccontainer_t::const_iterator ccI, const int32 mincoverage, vector<nngroups_t> & groups, vector<nngroups_t> & maskedshadowgroups, vector<char> & IUPACbasegroups, vector<char> & columnbases, const vector<int32> & read_ids_in_col, vector<int8> & maskshadow, uint8 actreadtype, char missingcoveragechar)");

#if 0
  if(read_ids_in_col.empty()){
    thisbase='!';
    FUNCEND();
    return;
  }
  if(CON_counts[actcontigpos].total_cov < mincoverage){
    thisbase='N';
    FUNCEND();
    return;
  }

#else

  size_t nummapped=0;
  if(actreadtype == Read::SEQTYPE_SOLEXA){
    // Ok, check whether the merged bases belong to this strain
    //  if not, well then no merged bases exist
    if(ccI->bbcounts[0]){
      uint8 strainmask=255;
      if(strainidtotake>=0) strainmask=getBBStrainMask(strainidtotake);
      if(ccI->bbstrains[0] & strainmask){
	nummapped=ccI->bbcounts[0];
//      cout << "looked at: " << *ccI << endl
//	   << "nummapped: " << nummapped << endl;
	//nummapped=0;
      }
    }
  }
  if(read_ids_in_col.size()+nummapped == 0){
    thisbase='!';
    FUNCEND();
    return;
  }
  if(read_ids_in_col.size()+nummapped < mincoverage){
    thisbase='N';
    FUNCEND();
    return;
  }
#endif

  const contig_parameters & con_rt_params = (*CON_miraparams)[actreadtype].getContigParams();

  CEBUG("conpos: " << actcontigpos << "\tnumreads: " << read_ids_in_col.size() << endl);
  columnbases.clear();
    
  /* 
     slowwwwwwwwwwwwwwwwwwwwwww
       
     on a 170MB CAF file, caf2fasta goes from 1:46 to 1:40 when
     replacing these two lines with the loop below (from 49s for
     output down to 43s)

     groups=emptygroups;
     maskedshadowgroups=emptygroups;
       
  */
    
  for(uint32 groupi=0; groupi < groups.size(); groupi++){
    groups[groupi].groupquality=0;
    groups[groupi].valid=false;
    groups[groupi].forwarddircounter=0;
    groups[groupi].complementdircounter=0;
    groups[groupi].ids.clear();
    groups[groupi].quals.clear();
    groups[groupi].directions.clear();
    //groups[groupi].strainids.clear();
    //groups[groupi].ids_sorted.clear();
    //groups[groupi].uniqstrainids.clear();
    //groups[groupi].countstrainids.clear();
      
    maskedshadowgroups[groupi].groupquality=0;
    maskedshadowgroups[groupi].valid=false;
    maskedshadowgroups[groupi].forwarddircounter=0;
    maskedshadowgroups[groupi].complementdircounter=0;
    maskedshadowgroups[groupi].ids.clear();
    maskedshadowgroups[groupi].quals.clear();
    maskedshadowgroups[groupi].directions.clear();
    //maskedshadowgroups[groupi].strainids.clear();
    //maskedshadowgroups[groupi].ids_sorted.clear();
    //maskedshadowgroups[groupi].uniqstrainids.clear();
    //maskedshadowgroups[groupi].countstrainids.clear();
  }
    
    
  bool xonly=true;

  if(nummapped>0 && ccI->backbonechar != 'X') xonly=false;

  for(int32 readnr=0; static_cast<int32>(readnr<read_ids_in_col.size()); readnr++){
    char           base;
    base_quality_t qual;
    int32          actreadid=read_ids_in_col[readnr];
    contigread_t & ric =CON_reads[actreadid];

    //if(ric.read.isShortReadMapping()) continue;
    
    int32 readpos=contigPosToUnclippedReadPos(actcontigpos, ric);
      
    CEBUG("read: " << ric.read.getName());
      
    //if(actcontigpos==682){
    //  int32 lenit=ric.read.getLenClippedSeq();
    //  cout << ric.read.getName() << "\t" << ric.offset << "\t"  << readpos << endl;
    //  //if(readnr==49) cout << ric.read << endl;
    //  if(lenit != ric.read.getLenClippedSeq()) {
    //    cout << "newlen??? " << ric.read.getLenClippedSeq();
    //    exit(0);
    //  }
    //}
      
    int32 realreadpos;
    if(ric.direction>0){
      base=toupper(ric.read.getBaseInSequence(readpos));
      qual=ric.read.getQualityInSequence(readpos);
      realreadpos=readpos;
    }else{
      base=toupper(ric.read.getBaseInComplementSequence(readpos));
      qual=ric.read.getQualityInComplementSequence(readpos);
      realreadpos=ric.read.calcComplPos(readpos);
    }
      
    columnbases.push_back(base);
      
    // this one gives a problem 
    //   if(base != 'X' && base != '*') xonly=false;
    // when one retrieves the consensus of a single strain in multiple
    //  strain assemblies: it may well be only a star!
    // preliminary fix: back to checking of X only
      
    if(base != 'X') xonly=false;
      
    CEBUG("\t" << base << "\t" << (uint16) qual << endl);
      
    // bases near start/end of reads might be dangerous because 
    //  of possible vector leftovers or still bad quality
    // so, if there is more than one read in this column, 
    // ...
    // - if within 10 bases of vector clip, max quality = 2*distance to svec 
    // - if not, lower the quality values of the bases that are near ends
    //   (con_endreadmarkexclusionarea bases) of (eventually clipped) reads
    // must be regular read (i.e. not backbone, rail or shortreadmapping

    if(!ric.read.isRail()
       && !ric.read.isBackbone()
       && !ric.read.isCoverageEquivalentRead()){
      if(read_ids_in_col.size() >1) {
	if((ric.read.getLSClipoff() >0
	    && realreadpos < ric.read.getLSClipoff()+10)
	   || (ric.read.getRSClipoff() < ric.read.getLenSeq()
	       && realreadpos > ric.read.getRSClipoff()-10)){
	  CEBUG(ric.read.getName()<< ": near seq vec in read, lowering the quality.\n");
	  int32 distance;
	  if(realreadpos < ric.read.getLSClipoff()+10){
	    distance=realreadpos-ric.read.getLSClipoff();
	  }else{
	    distance=ric.read.getRSClipoff()-realreadpos;
	  }
	  if(qual > distance*2){
	    qual=distance*2;
	  }
	} else if(realreadpos <= ric.read.getLeftClipoff()+con_rt_params.con_endreadmarkexclusionarea
		  || realreadpos >= ric.read.getRightClipoff()-con_rt_params.con_endreadmarkexclusionarea) {
	  CEBUG(ric.read.getName()<< ": near end of read, lowering the quality.\n");

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
	  switch(ric.read.getSequencingType()){
	  case Read::SEQTYPE_SANGER :
	  case Read::SEQTYPE_454GS20 :
	  case Read::SEQTYPE_IONTORRENT:
	  case Read::SEQTYPE_PACBIO : {
	    if(qual>=10) {
	      qual-=10;
	    } else {
	      qual=5;
	    }
	    if(qual<5) qual=5;
	    break;
	  }
	  case Read::SEQTYPE_SOLEXA :
	  case Read::SEQTYPE_ABISOLID : {
	    // decrease by one
	    if(qual) qual--;
	    break;
	  }
	  default: {
	  }
	  }
	}
      }
    }
      
    // No railreads at all, have been taken out earlier!
    //// Quality of bases from railreads are set to 0 so as not 
    ////  to be counted twice (as bases are the same as in backbone)
    //if(ric.read.isRail()) qual=0;
      
      
      
    bool maskedset=false;


    // TODO: rework the if(...hasTag(Read::REA_tagFpAS,realreadpos))
    //  to check for all mask taks in a masktagstring vector

//    if(maskshadow[actcontigpos]) {
//      // remember that the readpos computing routine doesn't take care
//      //  of direction, so we have to complement that position in reverse cases
//      int32 realreadpos=readpos;
//      if(ric.direction<0){
//	realreadpos=ric.read.calcComplPos(readpos);
//      }
//	
//      CEBUG("MASKED: " << actcontigpos << "\t");
//      CEBUG(ric.read.getName() << "\t" << realreadpos << "\t");
//      if(ric.read.hasTag(Read::REA_tagFpAS,realreadpos)) {
//	CEBUG("in" << endl);
//	for(uint32 bindex=0; bindex<maskedshadowgroups.size(); bindex++) {
//	  if(maskedshadowgroups[bindex].base==base) {
//	    maskedshadowgroups[bindex].ids.push_back(actreadid);
//	    maskedshadowgroups[bindex].quals.push_back(qual);
//	    maskedshadowgroups[bindex].directions.push_back(ric.direction);
//	    if(ric.direction>0){
//	      maskedshadowgroups[bindex].hasforwarddir=true;
//	    }else{
//	      maskedshadowgroups[bindex].hascomplementdir=true;
//	    }
//	    // special case: treat short read mapping as both forward and reverse
//	    if(ric.read.isCoverageEquivalentRead()){
//	      maskedshadowgroups[bindex].hasforwarddir=true;
//	      maskedshadowgroups[bindex].hascomplementdir=true;
//	    }
//	    maskedset=true;
//	    break;
//	  }
//	}
//      }
//    }


    if(maskedset==false) {
      CEBUG("out" << endl);
      for(uint32 bindex=0; bindex<groups.size(); bindex++) {
	if(groups[bindex].base==base) {
	  groups[bindex].ids.push_back(actreadid);
	  groups[bindex].quals.push_back(qual);
	  groups[bindex].directions.push_back(ric.direction);
	  if(ric.direction>0){
	    groups[bindex].forwarddircounter++;
	  }else{
	    groups[bindex].complementdircounter++;
	  }

	  // special case: treat short read mapping as both forward and reverse
	  if(ric.read.isCoverageEquivalentRead()){
	    // we already counted forward
	    //groups[bindex].forwarddircounter++;
	    groups[bindex].complementdircounter++;
	  }
	  break;
	}
      }
    }
  }
      
  {
    // check whether there is anything in groups 
    //  and maskedshadowgroups selected
    // if groups emty and masked not, copy masked to groups
    bool groupsempty=true;
    bool maskedempty=true;
    for(uint32 bindex=0; bindex<groups.size(); bindex++) {
      if( groups[bindex].ids.size()) groupsempty=false;
      if( maskedshadowgroups[bindex].ids.size()) maskedempty=false;
    }
    if(groupsempty && maskedempty==false) {
      CEBUG("EMPTY: " << actcontigpos << endl);
      groups=maskedshadowgroups;
    }
  }


  //{
  //  // Cheat!
  //  bool hasmergedbases=(ccI->backbonechar!='@') & (ccI->bbcounts[0]>0);
  //  bool mergedalsogroup=false;
  //  
  //  // Ok, check whether the merged bases belong to this strain
  //  //  if not, well then no merged bases exist
  //  uint8 strainmask=255;
  //
  //  if(strainidtotake>=0) strainmask=getBBStrainMask(strainidtotake);
  //  if(!(ccI->bbstrains[0] & strainmask)) hasmergedbases=false;
  //
  //
  //}





  if(xonly) {
    thisbase='X';
  } else {
    base_quality_t maxqual=0;
    uint32 maxcount=0;
    for(uint32 i=0; i<groups.size(); i++){
      calcGroupQual(groups[i]);
      if(groups[i].groupquality>maxqual){
	maxqual=groups[i].groupquality;
      }
      if(groups[i].ids.size()>maxcount){
	maxcount=groups[i].ids.size();
      }
      //if(actcontigpos>830 && actcontigpos <920) {
      CEBUG(actcontigpos << "\tb: " << groups[i].base << "\tgq: " << (uint16) groups[i].groupquality << "\ts: " << groups[i].ids.size() << endl);
      //}
    }
    //if(actcontigpos>830 && actcontigpos <920) {
    CEBUG(actcontigpos << "maxqual: " << (uint16) maxqual << "\t" << "maxcount: " << maxcount << endl);
    //}
    
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

    switch(actreadtype) {
    case Read::SEQTYPE_SANGER : {
      makeIntelligentConsensus_helper2_calcSangerQual(thisbase,
						      thisqual,
						      actcontigpos,
						      groups,
						      IUPACbasegroups,
						      columnbases,
						      maxqual,
						      maxcount);
      break;
    }
    case Read::SEQTYPE_454GS20 : {
      makeIntelligentConsensus_helper2_calc454GS20(thisbase,
						   thisqual,
						   actcontigpos,
						   groups,
						   IUPACbasegroups);
      break;
    }
    case Read::SEQTYPE_IONTORRENT : {
      makeIntelligentConsensus_helper2_calcIonTorrent(thisbase,
						      thisqual,
						      actcontigpos,
						      groups,
						      IUPACbasegroups);
      break;
    }
    case Read::SEQTYPE_PACBIO : {
      makeIntelligentConsensus_helper2_calcPACBIO(thisbase,
						  thisqual,
						  actcontigpos,
						  groups,
						  IUPACbasegroups);
      break;
    }
    case Read::SEQTYPE_SOLEXA : {
      // the static cast for the strain id 
//TODO: hier weiter
      makeIntelligentConsensus_helper2_calcSOLEXA(
	thisbase,
	thisqual,
	actcontigpos,
	ccI,
	groups,
	IUPACbasegroups,
	maxqual,
	maxcount,
	strainidtotake
	);
      break;
    }
    case Read::SEQTYPE_ABISOLID : {
      cout << "Actreadtype: " << static_cast<uint16>(actreadtype) << endl;
      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support.");
      break;
    }
    default: {
      cout << "Actreadtype: " << static_cast<uint16>(actreadtype) << endl;
      throw Notify(Notify::INTERNAL, THISFUNC, "Unknown read type.");
    }
    }

    if(!dptools::isValidIUPACStarBase(thisbase)){
      CEBUG("ALERT! This is not a valid base: '" << thisbase << "'\t: " << static_cast<uint16>(thisbase) << '\n');
    }

  }


  FUNCEND();
}






/*************************************************************************
 *
 * Calculate the 'true' consensus and gives back a string with consensus
 *  and a vector with the base quality of each base
 *
 * strainidtotake: <0 means "all", >=0 means "exactly those reads with that id"
 *
 * If the ostream parameter is != NULL, also writes a .tcs live file
 *  to it
 *
 *************************************************************************/

//#define CEBUG(bla) {cout << bla;} 

void Contig::makeIntelligentConsensus(string & target, vector<base_quality_t> & qual, vector<int32> * targetadjustments, vector<int32> * targetadjustments_bla, int32 from, int32 to, int32 mincoverage, base_quality_t minqual, int32 strainidtotake, char missingcoveragechar)//, ostream * ostr, bool contagsintcs)
{
  FUNCSTART("void Contig::makeIntelligentConsensus(string & target, vector<base_quality_t> & qual, int32 from, int32 to, int32 mincoverage, base_quality_t minqual, int32 strainidtotake)");//, ostream * ostr, bool contagsintcs)");

  cout << "makeIntelligentConsensus() complete calc\n";

  //CON_cebugflag=true;

  CEBUG("MIC\n");
  CEBUG("from " << from << endl);
  CEBUG("to " << to << endl);
  CEBUG("mincoverage " << mincoverage << endl);
  CEBUG("minqual " << static_cast<uint16>(minqual) << endl);
  CEBUG("missingcovchar " << missingcoveragechar << endl);
  CEBUG("strainidtotake " << strainidtotake << endl);


  BUGIFTHROW(from>to,"from>to?");
  BUGIFTHROW(from<0, "from < 0 ?");


  // Finalising the contig initialises the output order structure vector
  finalise();

  //const contig_parameters & con_params = CON_miraparams->getContigParams();

  if( to > static_cast<int32>(CON_counts.size())) to=CON_counts.size();
  int32 len_target=to-from;
  
  //target.resize(len_target);
  target.clear();
  target.reserve(len_target+10);
  qual.clear();
  qual.reserve(len_target+10);

  // for calculating the adjustments, only to this when whole
  //  consensus is calculated
  if(targetadjustments != NULL && from==0 && to==CON_counts.size()) {
    targetadjustments->clear();
    targetadjustments->reserve(len_target+10);
  }
  int32 unpaddedposcounter=0;


  nngroups_t emptygroup;
  emptygroup.base='!';
  emptygroup.valid=false;
  // TODO: make also use of bothdirectionspresent in this routine?
  emptygroup.forwarddircounter=0;
  emptygroup.complementdircounter=0;
  emptygroup.groupquality=0;
  vector<nngroups_t> emptygroups;
  for(uint32 i=0; i<5; i++) {
    emptygroups.push_back(emptygroup);
    emptygroups[i].base= "ACGT*"[i];
  }

/*
  // if there's a stream, we're dumping TCS
  // initialise a quick lookup vector to point at the positions in the
  //  consensus that have a tag
  // a for any tag
  // d for dangerous tag
  vector<bool> tcs_aconstagpositions;
  vector<bool> tcs_dconstagpositions;
  if(ostr != NULL && contagsintcs){
    tcs_aconstagpositions.resize(CON_counts.size(),false);
    tcs_dconstagpositions.resize(CON_counts.size(),false);
    vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
    for(; I!=CON_consensus_tags.end(); I++) {
      for(uint32 i=I->from; i<=I->to; i++) tcs_aconstagpositions[i]=true;
      if(I->identifier == CON_tagentry_idSRMc
	 || I->identifier == CON_tagentry_idWRMc
	 || I->identifier == CON_tagentry_idDGPc
	 || I->identifier == CON_tagentry_idUNSc
	 || I->identifier == CON_tagentry_idIUPc){
	for(uint32 i=I->from; i<=I->to; i++) tcs_dconstagpositions[i]=true;
      }
    }
  }
  // temporary vectors for TCS
  vector<int32> tcs_totalgroupcount;
  tcs_totalgroupcount.reserve(emptygroups.size());
  vector<int32> tcs_totalgroupqual;
  tcs_totalgroupqual.reserve(emptygroups.size());
*/


  vector<nngroups_t> maskedshadowgroups=emptygroups;

  vector<char> IUPACbasegroups;
  IUPACbasegroups.reserve(10);

  vector<char> columnbases;
  columnbases.reserve(1000);

  // for each read type, we will predict a base and quality and such
  vector<char> predictedbases(Read::getNumSequencingTypes());
  vector<base_quality_t> predictedquality(Read::getNumSequencingTypes());

  // as well as the possible alternative bases
  vector<vector<char> > possiblebases;
  possiblebases.resize(Read::getNumSequencingTypes());

  // as well as a goodness level estimate of the prediction
  vector<int8> predictlevel(Read::getNumSequencingTypes());
  vector<int8> predictlevelused(10);


  // for each read type a vector<nngroups_t> to hold all estimates
  //  regarding bases and qualities
  vector<vector<nngroups_t> > groupsvec;
  groupsvec.reserve(Read::getNumSequencingTypes());

  // a vector (read_ids_in_col) keeps the ids of the reads that are
  //  covering a specific position of the contig
  // reserving 1000 position should be enough 99.9% of all cases,
  //  is automatically extended by STL if needed.

  // now with different read types (sanger, 454 etc, we need a vector
  //  of read_ids_in_col

  vector<vector<int32> > read_ids_in_col;
  read_ids_in_col.resize(Read::getNumSequencingTypes());

  // ok, fill in some
  for(uint32 i=0; i<Read::getNumSequencingTypes(); i++){
    read_ids_in_col[i].reserve(1000);
    groupsvec.push_back(emptygroups);
  }

  // fill the vector in case we are starting within the contig
  if(from>0) {
    // TODO: this is wrong when going only after specific strainids!
    // correct that ASAN (as soon as needed)

    throw Notify(Notify::INTERNAL, THISFUNC, "starting with from > 0 not available yet.");

    //getReadIDsAtContigPosition(read_ids_in_col,from);
  }


  vector<int8> maskshadow;
  vector<multitag_t::mte_id_t> masktagstrings;

  // Bach: 17.08.2008
  // the new strategy of tagging poly-AT sites and keeping them in 
  //  the read (no clipping) makes it necessary to keep sequence
  //  under FpAS tags as full valid member of consensus somputation
  // Therefore, FpAS may NOT be put into the masktagstrings anymore!
  // masktagstrings.push_back(Read::REA_tagFpAS);

  buildMaskShadow(maskshadow,masktagstrings,false);

  vector<out_order_t>::const_iterator outorderI = CON_outputorder.begin();

  cccontainer_t::const_iterator ccI=CON_counts.begin();
  advance(ccI,from);

  // this is the loop that updates the vector that 
  //  keeps track of the reads that are
  //  covering a specific position of the contig
  // works quite simple: reads in the vector that are ending get
  //  thrown out, new reads starting are entered.

  CEBUG("CON_counts.size(): " << CON_counts.size() << endl);

  for(uint32 actcontigpos=from; actcontigpos<to ;actcontigpos++, ccI++){

    //cout << "conposx: " << actcontigpos << endl;
    //if(actcontigpos>1000000 && strainidtotake == 1){
    //  cout << "profile exit" << endl;
    //  exit(0);
    //}
    //if(actcontigpos >= 2698850
    //   && actcontigpos <= 2698870){
    //  CON_cebugflag=true;
    //}else{
    //  CON_cebugflag=false;
    //}
  
    {
      // updating the ids of the reads at that position

      // first delete those who fall out at this new position
      for(uint32 actreadtype=0; actreadtype<Read::getNumSequencingTypes(); actreadtype++){ 
	vector<int32>::iterator Ifrom=read_ids_in_col[actreadtype].begin();
	vector<int32>::iterator Ito=Ifrom;
	for(;Ifrom != read_ids_in_col[actreadtype].end(); Ifrom++){
	  *Ito=*Ifrom;
	  if(CON_reads[*Ifrom].offset+CON_reads[*Ifrom].read.getLenClippedSeq() > actcontigpos) {
	    Ito++;
	  }
	}
	if(Ito != Ifrom) {
	  read_ids_in_col[actreadtype].resize(Ito-read_ids_in_col[actreadtype].begin());
	}
      }

      // now insert ids of reads that have newly started at this position
      // Don't take railreads, backbone is there.
      for(;outorderI != CON_outputorder.end() && outorderI->offset_start == actcontigpos; outorderI++){
	if( ! CON_reads[outorderI->original_index].read.isRail()) {
	  
	  // new! one can also make a choice which strains to take into account
	  // either all (strainidtotake<0 or exactly the one chosen)
	  if(strainidtotake < 0
	     || CON_reads[outorderI->original_index].read.getStrainID() == strainidtotake) {
	    read_ids_in_col[CON_reads[outorderI->original_index].read.getSequencingType()].push_back(outorderI->original_index);
	  }
	}
      }
    }
  
    // for each read type, predict a base and a quality,
    //  but also get back the potential solutions and number thereof

    //vector<uint32> numsolutionsvec(Read::getNumSequencingTypes(),0);

    uint32 numreadtypeswithsolution=0;
    for(uint32 actreadtype=0; actreadtype<Read::getNumSequencingTypes(); actreadtype++){
      predictedbases[actreadtype]='!';
      predictedquality[actreadtype]=0;
      makeIntelligentConsensus_helper1(predictedbases[actreadtype],
				       predictedquality[actreadtype],
				       actcontigpos,
				       ccI,
				       mincoverage,
				       groupsvec[actreadtype],
				       maskedshadowgroups,
				       IUPACbasegroups,
				       columnbases,
				       read_ids_in_col[actreadtype],
				       maskshadow,
				       actreadtype,
				       strainidtotake,
				       missingcoveragechar
	);

      possiblebases[actreadtype]=IUPACbasegroups;

      //numsolutionsvec[actreadtype]=IUPACbasegroups.size();
      //if(numsolutionsvec[actreadtype] > 0) numreadtypeswithsolution++;
      if(predictedbases[actreadtype] != '!') numreadtypeswithsolution++;
    }

    /* after the previous loop, we have the following important things
       per read type (other things too, less important at the moment):
       - the picked base and quality in predictedbases[] and 
         predictedquality[]
       - possible alternatives (including picked one) in possiblebases[]
       - all group calculations for all read types in groupsvec[]
     */

    CEBUG("preda: ");
    for(uint32 actreadtype=0; actreadtype<Read::getNumSequencingTypes(); actreadtype++){
      CEBUG(predictedbases[actreadtype]);
    }
    CEBUG('\n');

    bool mismatchesintyperesults=false;
    if(numreadtypeswithsolution>1) {
      CEBUG("numreadtypes: " << numreadtypeswithsolution << endl);
      char basepick='!';
      for(uint32 actreadtype=0; actreadtype<Read::getNumSequencingTypes(); actreadtype++){
	CEBUG("pred[" << actreadtype << "]: " << predictedbases[actreadtype] << endl);
	if(predictedbases[actreadtype] != '!'){
	  if(basepick == '!'){
	    basepick=toupper(predictedbases[actreadtype]);
	  }else if(toupper(predictedbases[actreadtype]) != basepick){
	    if(basepick=='X'){
	      if(dptools::isValidIUPACStarBase(predictedbases[actreadtype])){
		basepick=toupper(predictedbases[actreadtype]);
	      }
	    }else if(basepick=='N'){
	      if(toupper(predictedbases[actreadtype]) != 'X'){
		basepick=toupper(predictedbases[actreadtype]);
	      }
	    }else{
	      if(toupper(predictedbases[actreadtype]) != 'X'
		 && toupper(predictedbases[actreadtype]) != 'N'){
		mismatchesintyperesults=true;
	      }
	    }
	  }
	}
      }
      CEBUG("Mismatch1: " << mismatchesintyperesults << endl);
    }


    // Party time, let's find out which base we take
    char thisbase='@';
    base_quality_t thisqual=0;

    bool hasSOLiD=false;
    if(hasSOLiD){
      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 2.");
      break;
    }

    // if we have a mismatch: shootout
    if(mismatchesintyperesults){

      CEBUG("Mismatch. ");

      // look first at at uniqueness and quality

      // set default level 100 (not used) for all seqtypes
      predictlevel.clear();
      predictlevel.resize(Read::getNumSequencingTypes(),100);
      
      // count how many predictions of each level are present
      predictlevelused.clear();
      predictlevelused.resize(10,0);

      int8 bestpred=100;
      for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	// just do this for sanger, 454, PacBio, solexa, solid equally (might change later)

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

	switch(actseqtype){
	case Read::SEQTYPE_SANGER :
	case Read::SEQTYPE_454GS20 :
	case Read::SEQTYPE_IONTORRENT :
	case Read::SEQTYPE_PACBIO :
	case Read::SEQTYPE_SOLEXA : 
	case Read::SEQTYPE_ABISOLID : {
	  if(predictedbases[actseqtype] != '!') {
	    // TODO: still not optimal, should perhaps test all
	    //  goups of IUPAC *sigh*
	    if(dptools::isValidACGTStarBase(predictedbases[actseqtype])){
	      predictlevel[actseqtype]=rateGoodnessLevelOfConsensus(
		ccI,
		groupsvec[actseqtype][dptools::getIndexOfBase(predictedbases[actseqtype])],
		possiblebases[actseqtype].size(),
		actseqtype
		);
	      BUGIFTHROW(predictlevel[actseqtype]<0, "problem in hybrid cons calc: level < 0");
	      BUGIFTHROW(predictlevel[actseqtype]>=predictlevelused.size(), "level > expected maxiumum of 9");
	    }else{
	      predictlevel[actseqtype]=5;
	    }
	    predictlevelused[predictlevel[actseqtype]]++;
	    if(predictlevel[actseqtype] < bestpred) bestpred=predictlevel[actseqtype];
	    CEBUG("plevel " << predictedbases[actseqtype] << " [" << actseqtype << "]: " << static_cast<int16>(predictlevel[actseqtype]) << '\n');
	  }
	  break;
	}
	default: {
	  //throw Notify(Notify::INTERNAL, THISFUNC, "unexpected sequencing type 1");
	}
	}
      }

      CEBUG("bestpred: " << static_cast<int16>(bestpred) << '\n');

      ostringstream tagstr;
      ostringstream tmdstr; // tag method details
      
      // if there's more than one prediction with quality level of best
      //  prediction or one after, it's unresolved
      // else we'll take the best prediction
      if(predictlevelused[bestpred]+predictlevelused[bestpred+1] >= 2){
	/* if this happens, we have different base predictions for each
	   sequencing type. That's bad. Could be due to sequencing or 
	   alignment or assembly error. Or when the strain sequenced
	   with one sequencing type was not 100% identical to the sequenced 
	   strain with the other sequencing type(s).

	   But there's one saving attempt: if one is a normal base and the other(s)
	   an IUPAC and the normal base is contained in the IUPAC, then we'll
	   take this base as result and not the IUPAC. (gaps kill the saving
	   process, sorry)
	*/


	// saving attempt, pre-check
	thisbase = '!';
	bool canbesaved=true;
	for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	  if(predictlevel[actseqtype]==bestpred
	     || predictlevel[actseqtype]==bestpred-1) {
	    // make sure that gaps kill the whole saving process
	    if(predictedbases[actseqtype]=='*') canbesaved=false;

	    if(dptools::isValidBase(predictedbases[actseqtype])){
	      tmdstr << " / " << Read::getShortNameOfSequencingType(actseqtype) 
		     << " (" << static_cast<uint16>(predictlevel[actseqtype])
		     << ") " << predictedbases[actseqtype]
		     << " " << static_cast<uint16>(predictedquality[actseqtype]);

	      if(thisbase=='!'){
		thisbase=predictedbases[actseqtype];
	      }else{
		thisbase=dptools::calcIUPACConsensus(thisbase,
						     predictedbases[actseqtype]);
	      }
	    }
	  }
	}

	canbesaved&=dptools::isValidBase(thisbase);

	// saving attempt, post-check (does the base fit in all IUPACs)
	if(canbesaved) {
	  tmdstr << "  is in  ";
	  for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	    if(predictlevel[actseqtype]==bestpred
	       || predictlevel[actseqtype]==bestpred-1) {
	      if(!dptools::isValidBase(predictedbases[actseqtype])){
		if(!dptools::hasNucleicAcidInIUPAC(thisbase,predictedbases[actseqtype])){
		  canbesaved=false;
		} else {
		  tmdstr << " / " << Read::getShortNameOfSequencingType(actseqtype) 
			 << " (" << static_cast<uint16>(predictlevel[actseqtype])
			 << ") " << predictedbases[actseqtype]
			 << " " << static_cast<uint16>(predictedquality[actseqtype]);
		}
	      }
	    }
	  }
	}

	CEBUG("TMDstr: " << tmdstr.str() << '\n');

	multitag_t::mte_id_t tagtype=CON_tagentry_idSTMU;
	if(canbesaved){
	  tagtype=CON_tagentry_idSTMS;
	  tagstr << "Solved vote: " << tmdstr.str();
	  // TODO: refine this: other quality and tell which sequencing types with which
	  //  results were taken into account
	  thisqual=10;
	}else{
	  tagstr << "Unresolved mismatch in seq. type consensus";
	  
	  thisbase = '!';
	  thisqual=0;
	  for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	    if(predictlevel[actseqtype]==bestpred
	       || predictlevel[actseqtype]==bestpred-1) {
	      tagstr << " / " << Read::getShortNameOfSequencingType(actseqtype)
		     << ": (";
	      if(thisbase=='!'){
		thisbase=predictedbases[actseqtype];
	      }else{
		thisbase=dptools::calcIUPACConsensus(thisbase,
						     predictedbases[actseqtype]);
	      }
	      tagstr << static_cast<uint16>(predictlevel[actseqtype])
		     << ") " << predictedbases[actseqtype]
		     << " " << static_cast<uint16>(predictedquality[actseqtype]);
	    }
	  }
	  tagstr << " || " << thisbase << " " << static_cast<uint16>(thisqual);
	}
	CEBUG("TAGstr1: " << tagstr.str() << '\n');
	addTagToConsensus(actcontigpos,
			  actcontigpos,
			  '=',
			  multitag_t::getIdentifierStr(tagtype).c_str(),
			  tagstr.str().c_str(),
			  true);
      }else{
	// well, shootout: take the best one
	tagstr << "Solved mismatch: ";

	for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	  if(predictlevel[actseqtype]==bestpred) {
	    tagstr << ' ' << Read::getShortNameOfSequencingType(actseqtype)
		   << ": (";
	    thisbase=predictedbases[actseqtype];
	    thisqual=predictedquality[actseqtype];

	    tagstr << static_cast<uint16>(predictlevel[actseqtype])
		   << ") " << thisbase
		   << ' ' << static_cast<uint16>(thisqual);

	    break;
	  }
	}
	CEBUG("TAGstr2: " << tagstr.str() << '\n');
	addTagToConsensus(actcontigpos,
			  actcontigpos,
			  '=',
			  multitag_t::getIdentifierStr(CON_tagentry_idSTMS).c_str(),
			  tagstr.str().c_str(),
			  true);
      }
    }else{
      // no mismatch in type results, good
      // quality = sum(.75*quality of each seqtype) or best qual
      CEBUG("No mismatch. ");

      uint32 compqual=0;
      uint32 bestqual=0;
      bool hassomething=false;
      bool has_n=false;
      bool has_x=false;
      for(uint32 actseqtype=0; actseqtype<Read::getNumSequencingTypes(); actseqtype++){
	if(predictedbases[actseqtype] == 'N') {
	  has_n=true;
	  continue;
	}
	if(predictedbases[actseqtype] == 'X') {
	  has_x=true;
	  continue;
	}
	if(predictedbases[actseqtype] != '!') {

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
	  switch(actseqtype){
	  case Read::SEQTYPE_SANGER : 
	  case Read::SEQTYPE_454GS20 :
	  case Read::SEQTYPE_IONTORRENT :
	  case Read::SEQTYPE_PACBIO :
	  case Read::SEQTYPE_SOLEXA :
	  case Read::SEQTYPE_ABISOLID: {
	    hassomething=true;
	    thisbase=predictedbases[actseqtype];
	    compqual+=(static_cast<uint32>(predictedquality[actseqtype])*75)/100;
	    if(predictedquality[actseqtype]>bestqual) bestqual=predictedquality[actseqtype];
	    break;
	  }
	  default: {
	    // do nothing???
	    //thisbase='x';
	  }
	  }
	}
      }

      if(!hassomething){
	if(has_x) {
	  thisbase='x';
	  hassomething=true;
	}
	if(has_n) {
	  thisbase='N';
	  hassomething=true;
	}
      }

      if(hassomething){
	BUGIFTHROW(thisbase=='@',"Coverage present but base is '@' ... this is not healthy.\n");
      }else{
	BUGIFTHROW(thisbase!='@',"No coverage present but base is not '@' ... this is not healthy.\n");
	thisbase=missingcoveragechar;
	thisqual=0;
      }

      if(bestqual>compqual) compqual=bestqual;
      if(compqual>90) compqual=90;
      thisqual=static_cast<base_quality_t>(compqual);
    }

    CEBUG("thisbase " << thisbase << "\tthisqual " << (uint16) thisqual << endl);

    thisbase=tolower(thisbase);

    if(thisqual<minqual && thisbase!=missingcoveragechar){
      thisbase='N';
      thisqual=0;
      CEBUG("Minqual not reached, changed to: thisbase " << thisbase << "\tthisqual " << (uint16) thisqual << endl);
    }

    if(thisqual>90) thisqual=90;

    target+=thisbase;
    qual.push_back(thisqual);

    // calc the adjustments
    if(targetadjustments!= NULL && from==0 && to==CON_counts.size()) {
      if(thisbase=='*') {
	targetadjustments->push_back(-1);
      }else{
	targetadjustments->push_back(unpaddedposcounter);
	unpaddedposcounter++;
      }

      BUGIFTHROW(targetadjustments->size()!=target.size(), "gna1");
      BUGIFTHROW(targetadjustments->size()!=qual.size(), "gna2");

    }

/*
    // dump out .tcs file to ostr if given
    if(ostr!=NULL) {
      //throw Notify(Notify::INTERNAL, THISFUNC, "must be adapted to multiple read types");

      // name and position
      // don't call paddedPos2UnpaddedPos as this would lead to recursion!
      *ostr << setw(20) << left << getContigName() 
	    << setw(9) << right << actcontigpos
	    << setw(9) << CON_adjustments.back()
	    << " | " << thisbase
	    << setw(3) << static_cast<uint16>(thisqual)
	    << " |" << setw(5) << read_ids_in_col.size();

      tcs_totalgroupcount.clear();
      tcs_totalgroupcount.resize(emptygroups.size(),0);
      tcs_totalgroupqual.clear();
      tcs_totalgroupqual.resize(emptygroups.size(),0);
      for(uint32 actseqtype=0; actseqtype<groupsvec.size(); actseqtype++){
	for(uint32 grpi=0; grpi<groupsvec[actseqtype].size(); grpi++){
	  tcs_totalgroupcount[grpi]+=groupsvec[actseqtype][grpi].ids.size();
	  tcs_totalgroupqual[grpi]+=groupsvec[actseqtype][grpi].groupquality;
	  if(tcs_totalgroupqual[grpi]>90) tcs_totalgroupqual[grpi]=90;
	}
      }
      for(uint32 grpi=0; grpi<tcs_totalgroupcount.size(); grpi++){
	*ostr << setw(5) << tcs_totalgroupcount[grpi];
      }

      *ostr << " |";
      uint32 numgroupswithqual=0;
      for(uint32 grpi=0; grpi<tcs_totalgroupcount.size(); grpi++){
	if(tcs_totalgroupcount[grpi]) {
	  *ostr << setw(3) << tcs_totalgroupqual[grpi];
	  numgroupswithqual++;
	}else{
	  *ostr << " --";
	}
      }

      // TODO: different characters for different cases?
      char bstatus='?';
      if(!dptools::isValidBase(thisbase)
	 && dptools::isValidIUPACBase(thisbase)){
	bstatus='M';
      }else if(thisbase=='*'){
	if(thisqual>=40){
	  bstatus=':';
	}else if(thisqual<30){
	  bstatus='m';
	}else if(numgroupswithqual>2){
	  bstatus='m';
	}else{
	  bstatus=':';
	}
      }else if(thisqual<30 && numgroupswithqual>1){
	bstatus='m';
      }else{
	bstatus=':';
      }
      if(!tcs_aconstagpositions.empty() && tcs_dconstagpositions[actcontigpos]){
	bstatus='$';
      }
      if(bstatus!=':'){
	*ostr << " | !" << bstatus << " |";
      }else{
	*ostr << " |  : |";
      }

      if(!tcs_aconstagpositions.empty() && tcs_aconstagpositions[actcontigpos]){
	vector<consensustag_t>::const_iterator I=CON_consensus_tags.begin();
	bool doneoutput=false;
	for(; I!=CON_consensus_tags.end(); I++) {
	  if((I->from <= actcontigpos) && ((I->to) >= actcontigpos)) {
	    if(doneoutput){
	      *ostr << ' ' << I->getIdentifierStr();
	    }else{
	      *ostr << " \"" << I->getIdentifierStr();
	      doneoutput=true;
	    }
	  }
	}
	if(doneoutput) *ostr << "\"";

      }
      *ostr << "\n";
    }
*/

  }


  CEBUG("target (first 100): " << target.substr(0,100) << endl);

  FUNCEND();

  return;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *
 *
 *************************************************************************/

int8 Contig::rateGoodnessLevelOfConsensus(cccontainer_t::const_iterator ccI, nngroups_t & group, uint32 numpossiblebases, uint8 seqtype)
{
  FUNCSTART("int8 Contig::rateGoodnessLevelOfConsensus(cccontainer_t::const_iterator ccI, nngroups_t & group, uint32 numpossiblebases, uint8 seqtype)");

  // basically, we'll prefer read types that have forward and
  //  complement direction present.

  CEBUG("predicting level for " << group.base << " seqtype " << static_cast<uint16>(seqtype) << '\n' << group << '\n');
  int8 level=-1;

  if(seqtype==Read::SEQTYPE_ABISOLID){
    throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 3.");
  }


  bool hasfr=(group.forwarddircounter>0) && (group.complementdircounter>0);

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  switch(seqtype) {
  case Read::SEQTYPE_SANGER :{
    if(hasfr &&
       group.groupquality >= 35){
      level=0;
    }else if(hasfr
	     && group.groupquality >= 25){
      level=1;
    } else if(hasfr){
      level=2;
    } else if(group.ids.size()>=3
	      && numpossiblebases == 1){
      level=3;
    } else {
      level=4;
    }
    break;
  }
  case Read::SEQTYPE_454GS20 :{
    if(hasfr
       && group.ids.size() >= 12
       && group.groupquality >= 35){
      level=0;
    } else if(hasfr
	      && group.ids.size() >= 8
	      && group.groupquality >= 25){
      level=1;
    } else if(hasfr){
      level=2;
    } else if(group.ids.size()>=6
	      && numpossiblebases == 1){
      level=3;
    } else {
      level=4;
    }
    break;
  }
  case Read::SEQTYPE_IONTORRENT :{
    if(hasfr
       && group.ids.size() >= 8
       && group.groupquality >= 35){
      level=0;
    } else if(hasfr
	      && group.ids.size() >= 6
	      && group.groupquality >= 25){
      level=1;
    } else if(hasfr){
      level=2;
    } else if(group.ids.size()>=4
	      && numpossiblebases == 1){
      level=3;
    } else {
      level=4;
    }
    break;
  }
  case Read::SEQTYPE_PACBIO :{
    // No info. atm use same as 454
    if(hasfr
       && group.ids.size() >= 12
       && group.groupquality >= 35){
      level=0;
    } else if(hasfr
	      && group.ids.size() >= 8
	      && group.groupquality >= 25){
      level=1;
    } else if(hasfr){
      level=2;
    } else if(group.ids.size()>=6
	      && numpossiblebases == 1){
      level=3;
    } else {
      level=4;
    }
    break;
  }
  case Read::SEQTYPE_SOLEXA :{
    // Solexa special case: we need to take the merged reads
    //  into account
    uint32 realsize=group.ids.size();
    if(ccI->backbonechar==group.base){
      realsize+=ccI->bbcounts[0];
      if(ccI->bbcounts[0]>=6){
	hasfr=true;
      }
    }
    if(hasfr
       && realsize >= 12
       && group.groupquality >= 35){
      level=0;
    } else if(hasfr
	      && realsize >= 8
	      && group.groupquality >= 30){
      level=1;
    } else if(hasfr 
	      && group.groupquality >= 25){
      level=2;
    } else if(group.groupquality >= 20
	      && realsize>=4
	      && numpossiblebases == 1){
      level=3;
    } else if(group.groupquality >= 10){
      level=4;
    } else {
      level=5;
    }
    break;
  }
  default : {
    cout << "seqtype: " << seqtype << endl;
    throw Notify(Notify::INTERNAL, THISFUNC, "Uknown seqtype to rate?");
  }
  }

  FUNCEND();
  return level;
}

//#define CEBUG(bla)





/*************************************************************************
 *
 * builds a vector which shows at which positions in the consensus there
 *  are reads that have a given tag 
 *
 *   .........TTTTTTTT..................
 *      ...........TTTTTTT.................  <- reads
 *
 *   00000000011111111111100000000000000000  <- maskshadow
 *
 *************************************************************************/

void Contig::buildMaskShadow(vector <int8> & maskshadow, vector<multitag_t::mte_id_t> & masktagstrings, bool onlybackbone)
{
  FUNCSTART("void Contig::buildMaskShadow(vector <int8> & maskshadow, vector<string> & masktagstrings)");

  //cout << "MASK #############" << endl;
  
  maskshadow.clear();
  maskshadow.resize(CON_counts.size(),0);
  vector<contigread_t>::const_iterator ACR=CON_reads.begin();
  for( ; ACR!=CON_reads.end(); ACR++) {
    if(onlybackbone && !ACR->read.isBackbone()) continue;
    const Read & actread = ACR->read;
    for(uint32 tagnum=0; tagnum<actread.getNumOfTags(); tagnum++) {
      const multitag_t & acttag=actread.getTag(tagnum);
      vector<multitag_t::mte_id_t>::const_iterator I= masktagstrings.begin();
      for( ; I!=masktagstrings.end(); I++) {
	if(acttag.identifier == *I) {
	  // ok, this tag is one of those we search
	  // get the consensusposition from the read tag positions
	  //  (may be outside of bounds as tags can be in clipped parts!)
	  int32 cfrom=unclippedReadPosToContigPos(acttag.from, *ACR);
	  int32 cto=unclippedReadPosToContigPos(acttag.to, *ACR);

	  // positions may be reverse, swap then
	  if(cfrom > cto) swap(cfrom,cto);

	  //cout << actread.getName() << "\t" << acttag.from << "\t" << acttag.to << "\t" << cfrom << "\t" << cto << endl;

	  // if both positions are left, right out of bounds, next tag
	  if(cfrom <0 && cto<0) continue;
	  if(cfrom >= CON_counts.size() && cto>=CON_counts.size()) continue;
	  // those positions ot of bounds -> set to bounds
	  if(cfrom < 0) cfrom=0;
	  if(cto >= CON_counts.size()) cto=CON_counts.size()-1;
	  // Tag from .. to are INCLUDING!
	  for(int32 i=cfrom; i<=cto; i++) {
	    maskshadow[i]=1;
	  }
	}
      }
    }
  }

  FUNCEND();
  return;
}







///*************************************************************************
// *
// * g must be a valid group structure
// *
// *************************************************************************/
//#if 1
//
//void Contig::calcGroupQual(const groups_t & g)
//{
//  if(g.ids.empty()){
//    const_cast<nngroups_t &>(g).groupquality=0;
//    return;
//  }
//
//  /* errorrate for this group is computed as follows:
//     Best quality for a base in a direction makes basic rate = 100%
//     add to this: 10% of next best base quality
//     
//     Same procedure for other direction, then add both qualities
//
//     In general, the values are almost the same (mostly a tad higher) as
//     with the more complicated (and time consuming) old variant.
//     
//     Cap at 90
//     
//     e.g.
//     + A 30     -> 30       \
//     + A 20     ->  2        \
//     + A 20                  /+ = 32    \
//     + A 20                 /            \
//     .                                    > + = 60
//     - A 26     -> 26     \              /
//     - A 20     ->  2      >  + = 28    /
//     - A 15               /
//  */
//
//  base_quality_t max1f=0;
//  base_quality_t max2f=0;
//  base_quality_t max1r=0;
//  base_quality_t max2r=0;
//
//  // find two highest values for each direction
//
//  for(uint32 j=0; j<g.ids.size(); j++) {
//    if(CON_reads[g.ids[j]].read.isRail()) continue;
//    if(g.directions[j]>0) {
//      if(g.quals[j] >= max1f){
//	max2f=max1f;
//	max1f=g.quals[j];
//      }else if(g.quals[j] > max2f){
//	max2f=g.quals[j];
//      }
//    } else {
//      if(g.quals[j] >= max1r){
//	max2r=max1r;
//	max1r=g.quals[j];
//      }else if(g.quals[j] > max2r){
//	max2r=g.quals[j];
//      }
//    }
//  }
//
//  // compute quality
//
//  const_cast<nngroups_t &>(g).groupquality=max1f+max1r+(max2f+max2r)/10;
//  if((g).groupquality>90) const_cast<nngroups_t &>(g).groupquality=90;
//}
//
//
//#else
//
///*************************************************************************
// *
// * g must be a valid group structure
// * errorvector must just be an vector<vector<double> >, holding this vector
// *   external saves _a lot_ of time for subsequent calls to calcGroupQual
// *   (eliminates zilliions of memory allocations)
// *
// *************************************************************************/
//
//void Contig::calcGroupQual(const groups_t & g,  vector<vector<double> > & errorvector)
//{
//  base_quality_t bq=0;
//
//  // compute the group quality
//  if(!g.ids.empty()){
//    if(errorvector.size()!=2) errorvector.resize(2);
//    errorvector[0].clear();
//    errorvector[1].clear();
//
//    // first, collect the error probabilities of each group for forward
//    //  and complement direction
//    // Leave out rails here!
//
//    for(uint32 j=0; j<g.ids.size(); j++) {
//      if(CON_reads[g.ids[j]].read.isRail()) continue;
//      if(g.directions[j]>0) {
//	errorvector[0].push_back(Read::qualityToErrorRate(g.quals[j]));
//	//cout << "ErrRate + " << errorvector[0].back() << endl;
//      } else {
//	errorvector[1].push_back(Read::qualityToErrorRate(g.quals[j]));
//	//cout << "ErrRate - " << errorvector[1].back() << endl;
//      }
//    }
//
//    /* errorrate for this group is computed as follows:
//       Best probability for a base in a direction makes basic rate = 100%
//       add to this: the other rates of the same direction each - 10% from
//         the previous, that is: 2nd = 90%, 3rd = 80% etc.
//       Same procedure for other direction, then multiply both errorrates
//       e.g.
//       + A 30    1000       ->          \
//       + A 20     100 \                  \
//       + A 20     100  >(90+80+70)       /+ = 1240  \
//       + A 20     100 /                 /            \
//       .                                              >* = 636120; log10(636120)*10
//       - A 26     398       ->        \              /               = 58
//       - A 20     100 \                >398+115=513 /
//       - A 15      32 / (90+25)= 115  /
//    */
//    double errorrate=1.0;
//    for(uint32 j=0;j<2; j++){
//      //cout << "  " << j << ") "; 
//      if(!errorvector[j].empty()){
//	sort(errorvector[j].begin(),errorvector[j].end(),greater<double>());
//	
//	double tmperrorrate=1.0;
//	uint32 multiplier=10;
//	for(uint32 k=0; k<errorvector[j].size() && multiplier>0; k++, multiplier--) {
//	  tmperrorrate+=(errorvector[j][k]*static_cast<double>(multiplier))/10.0;
//	  //cout << " " << errorvector[j][0];
//	}
//	errorrate*=tmperrorrate;
//	//cout << "erate: " << errorrate;
//      }
//    }
//    if(errorrate>1.0){
//      bq=Read::errorRateToQuality(errorrate);;
//      if(bq>95) bq=95;
//    }
//  }
//
//  const_cast<nngroups_t &>(g).groupquality=bq;
//}
//
//#endif




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::zeroStats()
{
  CON_stats.statsvalid=false;

  CON_stats.conlength=0;
  CON_stats.conlength_nogap=0;

  CON_stats.AinC=0;
  CON_stats.CinC=0;
  CON_stats.GinC=0;
  CON_stats.TinC=0;
  CON_stats.NinC=0;
  CON_stats.XinC=0;
  CON_stats.starInC=0;
  CON_stats.IUPACinC=0;
  CON_stats.FunnyInC=0;
  CON_stats.starInR=0;
  CON_stats.NinR=0;

  CON_stats.gccontent=0.0;

  CON_stats.numSRMc=0;
  CON_stats.numWRMc=0;
  CON_stats.numSTMU=0;
  CON_stats.numSTMS=0;

  CON_stats.avg_coverage=0.0;
  CON_stats.avg_conqual=0;

  CON_stats.numnocoverage=0;

  CON_stats.max_coverage=0;

  CON_stats.total_reads=0;
  CON_stats.numreads_withqual=0;
  CON_stats.numreads_noqual=0;

  for(uint32 i=0; i<Read::SEQTYPE_END; i++){
    CON_stats.max_covperst[i]=0;
    CON_stats.avg_covperst[i]=0.0;
    CON_stats.readsperst[i]=0;
    CON_stats.readlenperst[i]=0;
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::calcStats()
{
  FUNCSTART("void Contig::calcStats()");

  if(CON_valid==0){
    throw Notify(Notify::INTERNAL, THISFUNC, "Object not valid.");
  }
  if(CON_valid!=CON_VALID){
    throw Notify(Notify::INTERNAL, THISFUNC, "Object probably trashed.");
  }

  if(CON_stats.statsvalid) return;

  zeroStats();
  finalise();

  string consseq;
  vector<base_quality_t> consqual;
  
  newConsensusGet(consseq, consqual);

  CON_stats.conlength=CON_counts.size();

  // avg. coverage of solexa and solid must be made differently than 
  //  for sanger & 454 (because of merged reads)
  uint64 actcovsxa=0;
  uint64 sumcovsxa=0;
  uint64 actcovsid=0;
  uint64 sumcovsid=0;

  // using a pointer to CON_counts is (a lot) faster than accessing it 
  //  via [] each time
  cccontainer_t::iterator ccI=CON_counts.begin();

  //cout << "tililili! " << CON_name << '\n';
  for(uint32 ci=0; ci<CON_counts.size(); ci++, ccI++){
    if(consseq[ci]!='*') CON_stats.conlength_nogap++;
    //cout << ci << '\t' << consseq[ci] << '\n';
    switch(toupper(consseq[ci])){
    case 'A':{
      CON_stats.AinC++;
      break;
    }
    case 'C':{
      CON_stats.CinC++;
      break;
    }
    case 'G':{
      CON_stats.GinC++;
      break;
    }
    case 'T':{
      CON_stats.TinC++;
      break;
    }
    case '*':{
      CON_stats.starInC++;
      break;
    }
    case '-':
    case 'N': {
      CON_stats.NinC++;
      break;
    }
    case 'X':{
      CON_stats.XinC++;
      break;
    }
    default: {
      if(dptools::isValidIUPACBase(toupper(consseq[ci]))){
	CON_stats.IUPACinC++;
      }else{
	CON_stats.FunnyInC++;
      }
    }
    }

    // in backbone assemblies, acount for positions not covered
    // this is only good for backbones with one read, but cannot
    //   be helped at the moment
    // alternative would be a pre-analysis: use array size of CON_counts,
    //   go thorugh all reads (non backbone, non rail) and set to 1 
    //   for position covered
    if(ccI->backbonechar!='@' && ccI->total_cov==1){
      CON_stats.numnocoverage++;
    }

    CON_stats.max_coverage=max(CON_stats.max_coverage,
			       static_cast<uint32>(ccI->total_cov));
    for(uint32 i=0; i<Read::SEQTYPE_END; i++){
      CON_stats.max_covperst[i]=max(CON_stats.max_covperst[i],
				    static_cast<uint32>(ccI->seqtype_cov[i]));
    }

    // 100% mapped Solexa/SOLiD reads must be accounted for a bit differently

    actcovsxa=static_cast<uint64>(ccI->seqtype_cov[Read::SEQTYPE_SOLEXA]);
    sumcovsxa+=actcovsxa;

    CON_stats.max_covperst[Read::SEQTYPE_SOLEXA]=
      max(CON_stats.max_covperst[Read::SEQTYPE_SOLEXA],static_cast<uint32>(actcovsxa));
	  
    actcovsid=static_cast<uint64>(ccI->seqtype_cov[Read::SEQTYPE_ABISOLID]);
    sumcovsid+=actcovsid;

    CON_stats.max_covperst[Read::SEQTYPE_ABISOLID]=
      max(CON_stats.max_covperst[Read::SEQTYPE_ABISOLID],static_cast<uint32>(actcovsid));

    CON_stats.starInR+=ccI->star;
    CON_stats.NinR+=ccI->N;
  }

  // ok, in the section above, rail reads would have been also counted in 
  //  (in .coverage)
  // compute reads numbers

  {
    vector<contigread_t>::const_iterator cI=CON_reads.begin();
    for(;cI!=CON_reads.end();cI++){
      if(!cI->read.isRail()){

	// Nono, backbones must be counted in!
	// Scenario: assembly at 6x used as backbone, assembled further 1x
	//  If backbones were not counted, then it would look like all contigs
	//  had only 1x. Well, it'd be 1x new, but that's not what most people
	//  would want
	// && !cI->read.isBackbone()) {

	CON_stats.total_reads++;
	CON_stats.readsperst[cI->read.getSequencingType()]++;
	if(cI->orpid>=0){
	  CON_stats.readlenperst[cI->read.getSequencingType()]+=CON_readpool->getRead(cI->orpid).getLenClippedSeq();
	  if(cI->read.hasQuality()){
	    CON_stats.numreads_withqual++;
	  }else{
	    CON_stats.numreads_noqual++;
	  }
	}else{
	  CON_stats.readlenperst[cI->read.getSequencingType()]+=cI->read.getLenClippedSeq();
	}
      }
    }
  }

  CON_stats.readlenperst[Read::SEQTYPE_SOLEXA]=static_cast<uint32>(sumcovsxa);
  CON_stats.readlenperst[Read::SEQTYPE_ABISOLID]=static_cast<uint32>(sumcovsid);

  if(!CON_counts.empty()){
    CON_stats.avg_coverage=0.0;
    for(uint32 i=0; i<Read::SEQTYPE_END; i++){
      if(CON_stats.readlenperst[i]>0.0){
	CON_stats.avg_covperst[i]=static_cast<double>(CON_stats.readlenperst[i])/(CON_counts.size()- CON_stats.numnocoverage);
	CON_stats.avg_coverage+=CON_stats.avg_covperst[i];
      }
    }
  }

  {
    vector<base_quality_t>::const_iterator I=consqual.begin();
    double qual=0.0;
    while(I!=consqual.end()){
      qual+=static_cast<double>(*I);
      I++;
    }
    CON_stats.avg_conqual=static_cast<uint16>((qual/consseq.size())+.5);
  }

  if(CON_stats.AinC+CON_stats.CinC+CON_stats.GinC+CON_stats.TinC>0.0){
    CON_stats.gccontent=100.0/(CON_stats.AinC+CON_stats.CinC+CON_stats.GinC+CON_stats.TinC)*(CON_stats.CinC+CON_stats.GinC);
  }

  // IUPACs are perhaps not set yet
  updateStatsFromConsensusTags(true, true, false, true, true);

  CON_stats.statsvalid=true;

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::updateStatsFromConsensusTags(bool countSRMcs, bool countWRMcs, bool countIUPACs, bool countSTMUs, bool countSTMSs)
{
  if(countSRMcs) CON_stats.numSRMc=0;
  if(countWRMcs) CON_stats.numWRMc=0;
  if(countIUPACs) CON_stats.IUPACinC=0;
  if(countSTMUs) CON_stats.numSTMU=0;
  if(countSTMSs) CON_stats.numSTMS=0;
  vector<consensustag_t>::const_iterator ctI=CON_consensus_tags.begin();
  for(; ctI != CON_consensus_tags.end(); ctI++){
    if(countSRMcs && ctI->identifier == CON_tagentry_idSRMc) CON_stats.numSRMc++;
    if(countWRMcs && ctI->identifier == CON_tagentry_idWRMc) CON_stats.numWRMc++;
    if(countIUPACs && ctI->identifier == CON_tagentry_idIUPc) CON_stats.IUPACinC++;
    if(countSTMUs && ctI->identifier == CON_tagentry_idSTMU) CON_stats.numSTMU++;
    if(countSTMSs && ctI->identifier == CON_tagentry_idSTMS) CON_stats.numSTMS++;
  }
  
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::setContigName(const string & name)
{
  FUNCSTART("void Contig::setContigName(const string & name)");
  
  CON_name=name;

  FUNCEND();
  return;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const string & Contig::getContigName() const
{
  FUNCSTART("string Contig::getContigName() const");

  //if(!CON_name.empty()) {
  //  FUNCEND();
  //  return CON_name;
  //}

  if(!CON_name.empty()) {
    return CON_name;
  }

  ostringstream ostr;

  if(!CON_nameprefix.empty()) {
    ostr << CON_nameprefix;
  }else{
    ostr << "contig";
  }

  if(CON_reads.size()>1) {
    if(CON_contains_long_repeats_only){
      ostr << "_rep_c" << CON_id;
    }else{
      ostr << "_c" << CON_id;
    }
  }else{
    ostr << "_s" << CON_id;
  }

  //string cname=ostr.str();
  CON_name=ostr.str();
  
  FUNCEND();
  return CON_name;
}


///*************************************************************************
// *
// *
// *
// *
// *************************************************************************/
//
//bool Contig::hasConsensusTag(const string & identifier) const
//{
//  FUNCSTART("bool Contig::hasTag(const string & identifier) const");
//
//  bool returnit=false;
//  for(uint32 i=0; i<CON_consensus_tags.size();i++){
//    if(CON_consensus_tags[i].identifier==identifier){
//      returnit=true;           // we found this id in the consensus tags
//      break;                  // we don't need to examine further
//    }
//  }
//
//  FUNCEND();
//
//  return returnit;
//}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

bool Contig::hasConsensusTag(const multitag_t::mte_id_t identifier) const
{
  FUNCSTART("bool Contig::hasConsensusTag(const  multitag_t::mte_id_t identifier) const");

  bool returnit=false;
  for(uint32 i=0; i<CON_consensus_tags.size();i++){
    if(CON_consensus_tags[i].identifier==identifier){
      returnit=true;           // we found this id in the consensus tags
      break;                  // we don't need to examine further
    }
  }

  FUNCEND();

  return returnit;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::setSpecialSRAddConditions(const int32 maxtotalerrors, const int32 maxgaps, const int32 maxmismatches)
{
  CON_ssrc_maxtotalerrors=maxtotalerrors;
  CON_ssrc_maxgaps=maxgaps;
  CON_ssrc_maxmismatches=maxmismatches;
  if(maxtotalerrors<0 && maxgaps<0 && maxmismatches<0) {
    CON_specialsraddconditions=false;
  }else{
    CON_specialsraddconditions=true;
  }
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::addFirstRead(int32 id, uint32 direction)
{
  FUNCSTART("void Contig::addFirstRead(uint32 id, uint32 direction)");

  if(!CON_reads.empty()){
    throw Notify(Notify::INTERNAL, THISFUNC, "used addFirstRead while there are already reads in contig.");
  }

  definalise();
  CON_last_dangerous_overlaps.clear();

  uint32 len=CON_readpool->getRead(id).getLenClippedSeq();
  CEBUG("Len of clipped read: " << len << endl);

  CON_counts.resize(len,CON_concounts_zero);
  CON_reads.reserve(CON_READSSTARTSIZE);

  //check in this first read
  {
    contigread_t tmp;

    tmp.offset=0;
    tmp.orpid=id;
    tmp.direction=direction;
    
    CON_reads.push_back(tmp);

    CON_reads[0].read=CON_readpool->getRead(id);
  }

  if(CON_reads[0].read.getTemplateID()>=0){
    CON_templates_present.insert(CON_reads[0].read.getTemplateID());
  }

  if(direction>0){
    updateCountVectors(0,
		       len,
		       CON_readpool->getRead(id).getClippedSeqIterator(),
		       CON_readpool->getRead(id).getSequencingType(),
		       true);
    // Put base locks in CON_counts that this read produces
    initialiseBaseLocks();
  }else{
    throw Notify(Notify::INTERNAL, THISFUNC, "untested direction < 0");
  }

  //cout << "JJJJJJJ " << CON_readsperstrain.size() << " " << CON_readpool->getNumOfStrainInReadpool() << endl;
  if(CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool()){
    CON_readsperstrain.resize(CON_readpool->getNumOfStrainInReadpool(),0);
  }
  CON_readsperstrain[CON_readpool->getRead(id).getStrainID()]=1;
  if(CON_fixedconsseq.size()){
    nukeSTLContainer(CON_fixedconsseq);
    nukeSTLContainer(CON_fixedconsqual);
  }

  FUNCEND();
}







/*************************************************************************
 *
 * addRead_wrapped might change some alignment parameters
 * as it returns from a number of different points within the loop,
 *  setting back original this function takes care to set everything back 
 *  to normal if needed.
 *
 *************************************************************************/

//#define BUGHUNT

void Contig::addRead(vector<Align> & aligncache, const AlignedDualSeqFacts * initialads, int32 refid, int32 newid, int32 direction_frnid, bool newid_ismulticopy, int32 forcegrow, errorstatus_t & errstat)
{
  FUNCSTART("void Contig::addRead(vector<Align> & aligncache, const AlignedDualSeqFacts * initialads, int32 refid, int32 newid, int32 direction_frnid, errorstatus_t & errstat)");

  paranoiaBUGIF(aligncache.size()!=Read::SEQTYPE_END, 
		throw Notify(Notify::INTERNAL, THISFUNC, "aligncache is not size of available sequencing types???"));

  // this is ugly: save the align parames because
  //  addRead_wrapped() might change some *sigh*
  align_parameters oldalignparams=
    (*CON_miraparams)[CON_readpool->getRead(newid).getSequencingType()].getAlignParams();

  try {
#ifdef BUGHUNT
    {
      //rotate
      string system_rmdir = static_cast<string>("mv addRead_debug_pre.log addRead_debug_pre.log1");
      int tmp=system(system_rmdir.c_str()); 
      system_rmdir = static_cast<string>("mv addRead_debug_post.log addRead_debug_post.log1");
      tmp=system(system_rmdir.c_str()); 
      (void) tmp;

      ofstream eout("addRead_debug_pre.log", ios::out | ios::trunc);
      Read::setCoutType(Read::AS_MAF);
      setCoutType(AS_MAF);
      eout << "This file is for debugging and should never be seen by users!\n\n"
	"If you do see it, contact the author (bach@chevreux.org) immediately\n\n"
	"addRead_debug_pre.log!\n\n";
      if(initialads!=NULL) eout << "\n\n\ninitialads:\n" << *initialads;
      eout << "\n\nrefid: " << refid << " (" << CON_readpool->getRead(refid).getName()
	   << ")\nnewid: " << newid << " (" << CON_readpool->getRead(newid).getName()
	   << ")\ndirection_frnid: " << direction_frnid
	   << "\nnewid_ismulticopy: " << newid_ismulticopy
	   << "\nforcegrow: " << forcegrow
	   << "\n\nTrying read :\n" << CON_readpool->getRead(newid)
	   << endl
	   << "\nCurrent contig:\n" << *this
	   << endl;
    }
#endif

    addRead_wrapped(aligncache,
		    initialads,
		    refid,
		    newid,
		    direction_frnid,
		    newid_ismulticopy,
		    forcegrow,
		    errstat);

#ifdef BUGHUNT
    {
      ofstream eout("addRead_debug_post.log", ios::out | ios::trunc);
      Read::setCoutType(Read::AS_MAF);
      setCoutType(AS_MAF);
      eout << "This file is for debugging and should never be seen by users!\n\n"
	"If you do see it, contact the author (bach@chevreux.org) immediately\n\n"
	"addRead_debug_post.log!\n\n";
      if(initialads!=NULL) eout << "\n\n\ninitialads:\n" << *initialads;
      eout << "\n\nrefid: " << refid << " (" << CON_readpool->getRead(refid).getName()
	   << ")\nnewid: " << newid << " (" << CON_readpool->getRead(newid).getName()
	   << ")\ndirection_frnid: " << direction_frnid
	   << "\nnewid_ismulticopy: " << newid_ismulticopy
	   << "\nforcegrow: " << forcegrow
	   << "\n\nUsed read :\n" << CON_readpool->getRead(newid)
	   << endl
	   << "\nNew contig:\n" << *this
	   << endl;
    }
#endif

  }
  catch(Notify n){
    ofstream eout("addReaderror_REPLAY_do_not_delete.log", ios::out | ios::trunc);
    Read::setCoutType(Read::AS_MAF);
    setCoutType(AS_MAF);
    eout << "This file allows to replay an error that MIRA just encountered\n\n"
      "Please do not delete but contact the author (bach@chevreux.org) immediately\n\n"
      "addRead_Error!\n\n"
	 << "\n\n\ninitialads:\n" << *initialads
	 << "\n\nrefid: " << refid << " (" << CON_readpool->getRead(refid).getName()
	 << ")\nnewid: " << newid << " (" << CON_readpool->getRead(newid).getName()
	 << ")\ndirection_frnid: " << direction_frnid
	 << "\nnewid_ismulticopy: " << newid_ismulticopy
	 << "\nforcegrow: " << forcegrow
	 << "\n\nOffending read:" << CON_readpool->getRead(newid)
	 << endl
	 << "\nOffending contig: " << *this
	 << endl;
    eout.close();

    cout << "\n\n\nA file named 'addReaderror_REPLAY_do_not_delete.log' has been written"
      "\nto the working directory. Please send it to the author to get that bug fixed!\n\n\n";
    n.handleError(THISFUNC);
  }

  //CON_miraparams->setAlignParams(oldalignparams);
  (*CON_miraparams)[CON_readpool->getRead(newid).getSequencingType()].setAlignParams(oldalignparams);

  FUNCEND();
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
//#define CLOCK_STEPS
#ifdef CLOCK_STEPS
  enum{USCLO_PRE=0,
	 USCLO_DIR, USCLO_XCUT, USCLO_TEMPL1, USCLO_SWALIGN,
	 USCLO_PREINSCHK, 
	 USCLO_INSCON, 
	 USCLO_UPDBLOCKS, USCLO_DELREAD,
	 USCLO_ANRMBZ, USCLO_END};
#endif

//#define CEBUG(bla)   {cout << bla; cout.flush();}
//#define CEBUGF(bla)  {cout << bla; cout.flush();}
//#define CEBUGFLAG

//#define CEBUG(bla)   {if(docebug) {cout << bla; cout.flush();}}

void Contig::addRead_wrapped(vector<Align> & aligncache, const AlignedDualSeqFacts * initialads, int32 refid, int32 newid, int32 direction_frnid, bool newid_ismulticopy, int32 forcegrow, errorstatus_t & errstat)
{
  FUNCSTART("void Contig::addRead_wrapped(vector<Align> & aligncache, const AlignedDualSeqFacts * initialads, int32 refid, int32 newid, int32 direction_frnid, bool newid_ismulticopy, int32 forcegrow, errorstatus_t & errstat)");

#ifdef CLOCK_STEPS
  vector<suseconds_t> us_steps(USCLO_END,0);
  timeval us_start;
  gettimeofday(&us_start,NULL);
#endif

//#define ALIGNCHECK
//#define EXTRATRACEHELP

//  bool docebug=false;
//  if(CON_readpool->getRead(newid).getName()=="E0K6C4E02GJT3U") {
//    docebug=true;
//    assout::saveAsCAF(*this, "debug_co_addread.caf", true);
//  }

  if(!CON_reads.empty() && CON_reads[0].read.isBackbone()) CON_tmpcons_from_backbone=true;

  bool docebug=false;

//  docebug=true;
//  if(CON_id>=195) {
//    docebug=true;
//  }

  errstat.code=ENOERROR;
  errstat.reads_affected.clear();
  errstat.srmb_reads_affected.clear();
  errstat.proposed_cutback=-1;

  uint8 newreadseqtype=CON_readpool->getRead(newid).getSequencingType();
  MIRAParameters & rt_params=(*CON_miraparams)[newreadseqtype];

  //CEBUG(*this);
  CEBUG("\nCON_counts.size(): " << CON_counts.size() << '\n');

#ifndef PUBLICQUIET
  cout << "###Adding: " <<CON_readpool->getRead(newid).getName() << " ("
       << Read::getShortNameOfSequencingType(newreadseqtype)
       << "," << newid << ")\t"
       << static_cast<uint16>(CON_readpool->getRead(newid).getStrainID()) << " (" << CON_readpool->getRead(newid).getStrain() << ") ";
#endif

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  switch(newreadseqtype){
  case Read::SEQTYPE_SANGER : 
  case Read::SEQTYPE_454GS20 :
  case Read::SEQTYPE_IONTORRENT :
  case Read::SEQTYPE_PACBIO :
  case Read::SEQTYPE_SOLEXA : {
    break;
  }
  case Read::SEQTYPE_ABISOLID : {
    throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 4.");
    break;
  }
  default : {
    cout << "\nSeqtype: " << newreadseqtype << endl;
    throw Notify(Notify::INTERNAL, THISFUNC, "Trying to add unknown seqtype? Needs more code 4.");
  }
  }


  CEBUG("\tNewid: " << newid << '\t' << CON_readpool->getRead(newid).getName() << endl);

#ifdef PARANOIABUGTRACKFLAG
  cout << "Warning: PARANOIABUGTRACKFLAG is compiled!";
  if(CON_readpool->getRead(newid).hasValidData()==false){
    cout << "Read has no valid data???";
    throw Notify(Notify::INTERNAL, THISFUNC, "Read has no valid data???") ;
  }
  if(CON_readpool->getRead(newid).checkRead()){
    cout << "Read " << newid << " in readpool is trashed\n";
    cout << CON_readpool->getRead(newid);
    throw Notify(Notify::INTERNAL, THISFUNC, "*sigh* read trashed") ;
  }
#endif

  definalise();
  CON_last_dangerous_overlaps.clear();

//  CEBUG("I've got this contig:\n"<<*this);
//  CEBUG("Must align this:\n");

  if(CON_reads.empty()){
	addFirstRead(refid, 1);
	CEBUG(*this);
	FUNCEND();
	return;
  }
  // should be set by addFirstRead
  BUGIFTHROW(CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool(), "CON_readsperstrain.size() < CON_readpool->getNumOfStrainInReadpool() ?");

#ifndef PUBLICQUIET
  cout << "From: " <<CON_readpool->getRead(refid).getName() << " ("
       << Read::getShortNameOfSequencingType(CON_readpool->getRead(refid).getSequencingType())
       << "," << refid << ")\t";

  cout << initialads->getOverlapLen() << '/';
  cout << static_cast<uint16>(initialads->getScoreRatio()) << "%\t";
#endif

  {
    bool strangeerror=false;
    if( !(initialads->getID1() == refid
	  || initialads->getID2() == refid)) {
      cout << "refid not in initialads???\n";
      strangeerror=true;
    }
    if( !(initialads->getID1() == newid
	  || initialads->getID2() == newid)) {
      cout << "newid not in initialads???\n";
      strangeerror=true;
    }
    if(strangeerror){
      MIRANOTIFY(Notify::INTERNAL,"Strange error while adding read, investigate!");
    }
  }


#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  switch(CON_readpool->getRead(refid).getSequencingType()){
  case Read::SEQTYPE_SANGER :
  case Read::SEQTYPE_454GS20 :
  case Read::SEQTYPE_IONTORRENT :
  case Read::SEQTYPE_PACBIO :
  case Read::SEQTYPE_SOLEXA : {
    break;
  }
  case Read::SEQTYPE_ABISOLID : {
    throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 4a.");
    break;
  }
  default : {
    cout << "\nSeqtype: " << CON_readpool->getRead(refid).getSequencingType() << endl;
    throw Notify(Notify::INTERNAL, THISFUNC, "Having unknown seqtype as reference? Needs more code 4a.");
  }
  }

#ifdef PARANOIABUGTRACKFLAG
  // Test, if new read already in contig
  {
    vector<contigread_t>::const_iterator D= CON_reads.begin();
    uint32 i;
    for(i=0; i< CON_reads.size(); i++, D++){
      //      cout << " Checking:" << D->id;
      if(D->orpid==newid) break;
    }
    if(i!=CON_reads.size()){
      dumpConReads();
      Read::setCoutType(Read::AS_TEXTSHORT);
      cout << CON_readpool->getRead(newid);
      throw Notify(Notify::INTERNAL, THISFUNC, "new ID already in contig???");
    }
  }

  // Test, if ref read already in contig
  {
    vector<contigread_t>::const_iterator D= CON_reads.begin();
    uint32 i;
    for(i=0; i< CON_reads.size(); i++, D++){
      //      cout << " Checking:" << D->id;
      if(D->orpid==refid) break;
    }
    if(i==CON_reads.size()){
      dumpConReads();
      Read::setCoutType(Read::AS_TEXTSHORT);
      cout << CON_readpool->getRead(newid);
      throw Notify(Notify::INTERNAL, THISFUNC, "ref ID not in contig???");
    }
  }
#endif

  CEBUG("Refid len:" << CON_readpool->getRead(refid).getLenClippedSeq() << '\n');
  CEBUG("Newid len:" << CON_readpool->getRead(newid).getLenClippedSeq() << '\n');

  CEBUG("Refid: " << refid << endl);
  CEBUG('\n'<< *initialads);


  //int32 offsetrefid=-1;
  //int32 direction_refid=0;
  //{
  //  for(uint32 i=0;i<CON_reads.size(); i++){
  //    if(CON_reads[i].id==refid){
  //	offsetrefid=CON_reads[i].offset;
  //	direction_refid=CON_reads[i].direction;
  //	break;
  //    }
  //  }
  //  if(offsetrefid==-1){
  //    throw Notify(Notify::INTERNAL, THISFUNC, "RefID not found on contig?");
  //  }
  //}

  // initialise CON_maprpids_to_conreadsids if it's dirty (==empty)
  if(CON_maprpids_to_conreadsids.empty()){
    CON_maprpids_to_conreadsids.resize(CON_readpool->size(),-1);
    for(int32 i=0;i<CON_reads.size(); i++){
      if(CON_reads[i].orpid>=0){
	CON_maprpids_to_conreadsids[CON_reads[i].orpid]=i;
      }
    }
  }

#ifdef CLOCK_STEPS
  us_steps[USCLO_PRE]=diffsuseconds(us_start);
  gettimeofday(&us_start,NULL);
#endif

  contigread_t & conactrefread=CON_reads[CON_maprpids_to_conreadsids[refid]];
  int32 offsetrefid=conactrefread.offset;
  if(offsetrefid==-1){
    throw Notify(Notify::INTERNAL, THISFUNC, "RefID not found on contig?");
  }

  //Read::setCoutType(Read::AS_FASTA);
  //cout << conactrefread.read;



  //gna if(errstat.reads_affected.empty())
  
  if(!CON_allowedrefids.empty()){
    if(!CON_allowedrefids[refid]){
      errstat.code=EREFIDNOTALLOWED;
      if(CON_readpool->getRead(refid).isRail()) {
	// we did not compute xcut and ycut yet.
	// to save time, we already stop here and take the positions
	//  of the refid rail as xcut and ycut
	getRailsAsReadsAffected(refid, errstat.reads_affected,
				conactrefread.offset,
				conactrefread.offset+conactrefread.read.getLenClippedSeq());
      }else{
	errstat.reads_affected.push_back(refid);
      }
      FUNCEND();
      return;
    }
  }



  int32 direction_refid=conactrefread.direction;


#ifdef CLOCK_STEPS
  us_steps[USCLO_DIR]=diffsuseconds(us_start);
#endif


  int32 direction_newid_incontig=direction_refid*direction_frnid;

  CEBUG("dir_frnid: " << direction_frnid);
  CEBUG("\tdir_refid: " << direction_refid<< endl;);
  CEBUG("Offset Refid:" << offsetrefid<<endl;);

  list<AlignedDualSeq>::const_iterator I;
  //Align bla(CON_miraparams);
  list<AlignedDualSeq> madsl;


  // xcut and ycut are the posintions in the contig where a
  //  sequence must be made from for SW alignment
  // xcut and ycut are defined [...[  (xcut including, ycut excluding)

  // the ycut is less critical, but also influences slightly the
  //  speed of the banded SW ... there might be a little bit more to 
  //  compute

  int32 xcut=offsetrefid; // will be adapted

  int32 xcutinc=1;
  int32 deltax;
  // same direction in contig and reference ads?
  if(direction_refid*initialads->getSequenceDirection(refid)>0){
    // yes
    if(initialads->getOffsetInAlignment(refid)==0){
      CEBUG("R1 offset.\n");
      deltax=initialads->getOffsetInAlignment(newid);

    }else{
      CEBUG("R2 offset.\n");
      deltax=initialads->getOffsetInAlignment(refid);
      xcutinc=-1;
    }
  }else{
    if(initialads->getRightOffsetInAlignment(refid)==0){
      CEBUG("R3 offset.\n");
      deltax=initialads->getRightOffsetInAlignment(newid);
    }else{
      CEBUG("R4 offset.\n");
      deltax=initialads->getRightOffsetInAlignment(refid);
      xcutinc=-1;
    }
  }

  CEBUG("deltax: " << deltax << '\n');
  CEBUG("xcutinc: " << xcutinc << '\n');

#ifdef CLOCK_STEPS
  gettimeofday(&us_start,NULL);
#endif

  int32 runlength=CON_readpool->getRead(newid).getLenClippedSeq();
  int32 ycut;

  // search for xcut: use refread sequence when possible,
  //  else the concount_t structure
  {
    int32 refpos=0;
    const char * refseq;
    {
      if(direction_refid>0){
	refseq=conactrefread.read.getClippedSeqAsChar();
      }else{
	refseq=conactrefread.read.getClippedComplementSeqAsChar();
      }
      cccontainer_t::iterator ccI=CON_counts.begin();
      BOUNDCHECK(xcut, 0, CON_counts.size()+1);
      if(xcutinc<0 && xcut>0) xcut--;
      advance(ccI, xcut);
      for(;deltax>=0; xcut+=xcutinc, refpos+=xcutinc, refseq+=xcutinc){
	CEBUG("deltax: " << deltax << "\txcut: " << xcut);
	if(refpos>=0 && refpos<conactrefread.read.getLenClippedSeq()){
	  CEBUG("\tRead: " << *refseq << '\n');
	  if(*refseq!='*'){
	    deltax--;
	  }
	  ccI+=xcutinc;
	}else{
	  if(xcut>=0 && xcut<CON_counts.size()+1){
	    // this part mimicks the makeTmpConsenus() behaviour of calling stars
	    if(CON_tmpcons_from_backbone
	       && ccI->backbonechar!='N'
	       && ccI->backbonechar!='X'
	       && ccI->backbonechar!='@'){
	      if(ccI->backbonechar!='*') {
		CEBUG("\tBB: (base)\n");
		deltax--;
	      }else{
		CEBUG("\tBB: *\n");
	      }
	    }else{
	      ccctype_t maximum= max(ccI->A, max(ccI->C, max(ccI->G, ccI->T)));
	      if(maximum >0 && maximum > ccI->star) {
		//if(maximum/4 >= ccI->star) {
		if(maximum/4 >= (ccI->star)*2) {
		  deltax--;
		  CEBUG("\tCON: (base)\n");
		}else{
		  CEBUG("\tCON: *\n");
		}
	      }else{
		if(!((ccI->star >= ccI->X)
		     && (ccI->star >= ccI->X))){
		  deltax--;
		  CEBUG("\tCON: (base)\n");
		}else{
		  CEBUG("\tCON: *\n");
		}
	      }
	    }
	    ccI+=xcutinc;
	  }else{
	    // TODO: replace with one += and a break out of loop
	    CEBUG("\tccI out of contig bounds.\n");
	    deltax--;
	    runlength--;
	  }
	}
      }
    }
    
    CEBUG("xcut 1: " << xcut <<'\n');
    
    // xcut, and therefore the initial ycut, may be way negative, beware!
    ycut=xcut;
    if(ycut<0) ycut=0;
    
    {
      cccontainer_t::iterator ccI=CON_counts.begin();
      if(ycut>0) advance(ccI, ycut);
      for(;runlength>=0; ycut++, refpos++, refseq++){
	CEBUG("runlength: " << runlength << "\tycut: " << ycut);
	if(0){
	  //if(refpos>=0 && refpos<conactrefread.read.getLenClippedSeq()){
	  CEBUG("\tRead: " << *refseq << '\n');
	  if(*refseq!='*'){
	    runlength--;
	  }
	}else{
	  // Changed 16.05.2010:
	  // Found by using IndexedDeque segfaulting due to out of bounds
	  // Why on earth did I have +1 
	  //  if(ycut>=0 && ycut<CON_counts.size()+1){
	  if(ycut>=0 && ycut<CON_counts.size()){
	    // this part mimicks the makeTmpConsenus() behaviour of calling stars
	    if(CON_tmpcons_from_backbone 
	       && ccI->backbonechar!='@'
	       && ccI->backbonechar!='N'
	       && ccI->backbonechar!='X'){
	      if(ccI->backbonechar!='*') {
		CEBUG("\tBB: (base)\n");
		runlength--;
	      }else{
		CEBUG("\tBB: *\n");
	      }
	    }else{
	      ccctype_t maximum= max(ccI->A, max(ccI->C, max(ccI->G, ccI->T)));
	      if(maximum >0 && maximum > ccI->star) {
		//if(maximum/4 >= ccI->star) {
		if(maximum/4 >= (ccI->star)*2) {
		  runlength--;
		  CEBUG("\tCON1: (base): " << *ccI << endl);
		}else{
		  CEBUG("\tCON1: *: " << *ccI << endl);
		}
	      }else{
		if(!((ccI->star >= ccI->X)
		     && (ccI->star >= ccI->X))){
		  runlength--;
		  CEBUG("\tCON2: (base): " << *ccI << endl);
		}else{
		  CEBUG("\tCON2: *: " << *ccI << endl);
		}
	      }
	    }
	    ccI++;
	  }else{
	    CEBUG("\tccI out of bounds.\n");
	    runlength--;
	  }
	}
      }
    }
  }

  CEBUG("ycut 1: " << ycut <<'\n');

  ycut+=10; // add safety distance at the end

  // -2  as safety distance in front
  xcut-=2;

  CEBUG("xcut 2: " << xcut <<'\n');


  // in some cases, xcut may be > size of contig (and ycut anyway)
  //  or ycut < 0 (and xcut anyway)
  // this can happen when a read is edited over and over again during 
  //  contig assembly and then the expected offset in the adsfact
  //  is way off target
  // occurs with short matches at end of reads.
  //
  // only possibility to handle this at this stage: reject alignment

  if(xcut> static_cast<int32>(CON_counts.size()) || ycut < 0){
#ifndef PUBLICQUIET
    cout << "rej: no align found (bounds 1)\t";
    cout.flush();
#endif
    errstat.code=ENOALIGN;
    errstat.reads_affected.push_back(refid);
#ifdef CLOCK_STEPS
    us_steps[USCLO_XCUT]=diffsuseconds(us_start);
    coutAddReadClocks(us_steps);
#endif
    FUNCEND();
    return;
  }


  // but check that we're not hitting a gap base at the xcut position
  //  if yes, go back as far as needed to find a non-gap
  // Can happen because of the xcut-=2 above:
  if(xcut>0){
    cccontainer_t::iterator ccI=CON_counts.begin();
    BOUNDCHECK(xcut, 0, CON_counts.size()+1);
    advance(ccI, xcut);
    while(xcut>0){
      ccctype_t maximum= max(ccI->A, max(ccI->C, max(ccI->G, ccI->T)));
      if(maximum >0 && maximum > ccI->star) {
  	if(maximum/4 >= ccI->star) {
  	  // base
  	  break;
  	}
      }else{
  	if(!((ccI->star >= ccI->X)
  	     && (ccI->star >= ccI->X))){
  	  // base
  	  break;
  	}
      }
      ccI--;
      xcut--;
    }
  }

  CEBUG("xcut final: " << xcut <<'\n');
  CEBUG("ycut final: " << ycut <<'\n');

#ifdef CLOCK_STEPS
  us_steps[USCLO_XCUT]=diffsuseconds(us_start);
#endif

  

#ifdef CLOCK_STEPS
  gettimeofday(&us_start,NULL);
#endif


  // Template handling1

  bool havematchingtemplatepartner=false;

  if(rt_params.getContigParams().con_use_template_information > 0){
    int32 acttid=CON_readpool->getRead(newid).getTemplateID();
    if(acttid >=0){
#ifndef PUBLICQUIET
      cout << " tmplhand1...";
      cout.flush();
#endif
      int32 tidcount=CON_templates_present.count(acttid);
      // get that out ... strobed reads can have more than two reads per template
      //  Sanger multiple sequencing of template too
      //if(tidcount>1){
      //	cout << *this;
      //	throw Notify(Notify::INTERNAL, THISFUNC, "Template handling error, already present twice.");
      //}
      if(tidcount>0){
	vector<contigread_t>::const_iterator I=CON_reads.begin();
	while(I != CON_reads.end() && I->read.getTemplateID() != acttid) I++;
	if(I==CON_reads.end()){
	  cout << *this;
	  throw Notify(Notify::INTERNAL, THISFUNC, "Another template handling error *sigh*.");
	}

	// ok, we found the read with the template which corresponds to the newly 
	//  to insert read. Now check if they meet the constraints. 
	//  If not, do not insert

	if(direction_newid_incontig*I->direction != CON_readpool->getRead(newid).getTemplateBuildDirection()){
	  // not direction wanted, not good
#ifndef PUBLICQUIET
	  cout << "templ in wrong dir";
#endif
	  errstat.code=ETEMPLATEDIRECTION;
	  //errstat.reads_affected.push_back(refid);

	  if(xcut<0) xcut=0;
	  if(static_cast<uint32>(ycut)>CON_counts.size()){
	    ycut=CON_counts.size();
	  }

	  if(CON_readpool->getRead(refid).isRail())  {
	    getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
	  }else{
	    getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
	  }
#ifdef CLOCK_STEPS
	  coutAddReadClocks(us_steps);
#endif
	  FUNCEND();
	  return;
	}


	// Note: play it safe while adding right extend ... take only 
	//  part of it
	int32 leftrx=xcut;
	int32 leftry=ycut;
	if(direction_newid_incontig>0) {
	  leftry+=CON_readpool->getRead(newid).getRightExtend()*3/2;
	}else{
	  leftrx-=CON_readpool->getRead(newid).getRightExtend()*3/2;
	}

	int32 rightrx=I->offset;
	int32 rightry=I->offset+I->read.getLenClippedSeq();
	if(direction_refid>0) {
	  rightry+=CON_readpool->getRead(newid).getRightExtend()*3/2;
	}else{
	  rightrx-=CON_readpool->getRead(newid).getRightExtend()*3/2;
	}

	if(rightrx<leftrx) {
	  swap(rightrx,leftrx);
	  swap(rightry,leftry);
	}

	int32 actinsertsize=rightry - leftrx;
	if(rightry<leftry) actinsertsize=leftry - leftrx;

	// allow 15% error in calculated insert size
	int32 aisp10=actinsertsize+actinsertsize*15/100;
	int32 aism10=actinsertsize-actinsertsize*15/100;

	int32 tif=I->read.getInsizeFrom();
	int32 tit=I->read.getInsizeTo();

	// Not anymore
	//// allow 10% error in given insert sizes, but at least 1kb
	//int32 tif=I->read.getInsizeFrom();
	//if(tif/10 >= 1000){
	//  tif-=tif/10;
	//} else {
	//  tif-=1000;
	//}
	//if(tit/10 >= 1000){
	//  tit+=tit/10;
	//} else {
	//  tit+=1000;
	//}


	if(I->read.getInsizeFrom() >=0 && aisp10 < tif){
	  // distance too small
#ifndef PUBLICQUIET
	  cout << "templ too small: " << tif << " min allowed, got " << aisp10;
#endif
	  errstat.code=ETEMPLATESIZE;
	  //errstat.reads_affected.push_back(refid);

	  if(xcut<0) xcut=0;
	  if(static_cast<uint32>(ycut)>CON_counts.size()){
	    ycut=CON_counts.size();
	  }
	  if(CON_readpool->getRead(refid).isRail()) {
	    getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
	  }else{
	    getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
	  }

#ifdef CLOCK_STEPS
	  coutAddReadClocks(us_steps);
#endif
	  FUNCEND();
	  return;
	}
	if(I->read.getInsizeTo() >=0 && aism10 > tit){
	  // distance too big
#ifndef PUBLICQUIET
	  cout << "templ too big: " << tit << " max allowed, got " << aism10;
#endif

	  errstat.code=ETEMPLATESIZE;
	  //errstat.reads_affected.push_back(refid);

	  if(xcut<0) xcut=0;
	  if(static_cast<uint32>(ycut)>CON_counts.size()){
	    ycut=CON_counts.size();
	  }
	  if(CON_readpool->getRead(refid).isRail())  {
	    getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
	  }else{
	    getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
	  }
#ifdef CLOCK_STEPS
	  coutAddReadClocks(us_steps);
#endif
	  FUNCEND();
	  return;
	}

	// ok, we have a matching template partner
	havematchingtemplatepartner=true;
      }
#ifndef PUBLICQUIET
      cout << "done";
      cout.flush();
#endif
    }
  }

#ifdef CLOCK_STEPS
  us_steps[USCLO_TEMPL1]=diffsuseconds(us_start);
  gettimeofday(&us_start,NULL);
#endif


  bool doneit;
  bool maymapthisread;
  uint32 foolsafe=0;


#ifdef ALIGNCHECK
  Align checkbla(CON_miraparams);
#endif
  
  do{
    CEBUG("xcut: " << xcut << "\t");
    CEBUG("ycut: " << ycut << endl);
    CEBUG("CON_counts size: " << CON_counts.size() << endl);
	
    int32 eoffset=0;
    if(xcut<0){
      eoffset=xcut;
      xcut=0;
    }
    if(static_cast<uint32>(ycut)>CON_counts.size()){
      ycut=CON_counts.size()+1;
    }

    CEBUG("xcut: " << xcut << "\t");
    CEBUG("ycut: " << ycut << endl);
    CEBUG("eoffset: " << eoffset << "\t");
	
    // The following should happen only very, very rarely
    // Normally almost impossible, but maybe triggered in projects where
    //  reads get edited over and over again
    if(xcut> static_cast<int32>(CON_counts.size()) 
       || ycut < 0
       || ycut <= xcut){
#ifndef PUBLICQUIET
      cout << "rej: no align found (bounds 2)\t";
      cout.flush();
#endif
      errstat.code=ENOALIGN;
      errstat.reads_affected.push_back(refid);
#ifdef CLOCK_STEPS
      us_steps[USCLO_XCUT]=diffsuseconds(us_start);
      coutAddReadClocks(us_steps);
#endif
      FUNCEND();
      return;
    }

    // if makeTmpConsenus() returns true, a N or X was encountered in the consensus
    //  and we may not map this read
    maymapthisread=!(makeTmpConsensus(xcut, ycut));

    if(CON_2tmpcons.size()==0){
      // This should never happen here (the above checks should have made sure of that)
      // If it does ... we'll simply make it easy:
      //  dump out error to log and continue, ignoring it the best we can
      cout << "Sheeesh, length of temporary consensus is 0? Error, but continuing as this probably will not affect contig-building anyway."
	   << "\nxcut: " << xcut
	   << "\nycut: " << ycut
	   << "\neoffset: " << eoffset
	   << endl;
      madsl.clear();

      errstat.code=EUNSPECIFIED;
      errstat.reads_affected.push_back(refid);

      // rather not do this here ... we really do not know what xcut and ycut are
      //
      //// this can throw ... if it does, things are really, really, really botched
      //try {
      //	if(CON_readpool->getRead(refid).isRail())  {
      //	  getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
      //	}else{
      //	  getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
      //	}
      //}
      //catch (Notify n) {
      //	n.gravity=Notify::WARNING;
      //	n.handleError("internal to addRead_wrapped()");
      //	errstat.reads_affected.clear();
      //	errstat.reads_affected.push_back(refid);
      //}
#ifdef CLOCK_STEPS
      us_steps[USCLO_XCUT]=diffsuseconds(us_start);
      coutAddReadClocks(us_steps);
#endif
      FUNCEND();
      return;
    }

    try{
      if(direction_frnid>=0){
	if(direction_refid>=0){
	  CEBUG("C1 align.\n");
#ifdef EXTRATRACEHELP
	  cout << "Read sequence:\n";
	  for(uint32 qwer=0; qwer<CON_readpool->getRead(newid).getLenClippedSeq();qwer++){
	    cout << CON_readpool->getRead(newid).getClippedSeqAsChar()[qwer];
	  }
#endif
	  aligncache[newreadseqtype].acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, 1, true, eoffset);
#ifdef ALIGNCHECK
	  checkbla.acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, 1);
#endif
	}else{
	  CEBUG("C2 align.\n");
#ifdef EXTRATRACEHELP
	  cout << "Read sequence:\n";
	  for(uint32 qwer=0; qwer<CON_readpool->getRead(newid).getLenClippedSeq();qwer++){
	    cout << CON_readpool->getRead(newid).getClippedComplementSeqAsChar()[qwer];
	  }
#endif
	  aligncache[newreadseqtype].acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedComplementSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, -1, true, eoffset);
	
	  //if(xcut == 468 && ycut == 1251) {
	  //  cout << "Read sequence:\n";
	  //  for(uint32 qwer=0; qwer<CON_readpool->getRead(newid).getLenClippedSeq();qwer++){
	  //    cout << CON_readpool->getRead(newid).getClippedComplementSeqAsChar()[qwer];
	  //  }
	  //  cout << endl;
	  //}


#ifdef ALIGNCHECK
	  checkbla.acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedComplementSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, -1);
#endif
	}    
      }else{
	if(direction_refid>=0){
	  CEBUG("C3 align.\n");
#ifdef EXTRATRACEHELP
	  cout << "Read sequence:\n";
	  for(uint32 qwer=0; qwer<CON_readpool->getRead(newid).getLenClippedSeq();qwer++){
	    cout << CON_readpool->getRead(newid).getClippedSeqAsChar()[qwer];
	  }
#endif
	  aligncache[newreadseqtype].acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedComplementSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, -1, true, eoffset);
#ifdef ALIGNCHECK
	  checkbla.acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedComplementSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, -1);
#endif
	}else{
	  CEBUG("C4 align.\n");
#ifdef EXTRATRACEHELP
	  cout << "Read sequence:\n";
	  for(uint32 qwer=0; qwer<CON_readpool->getRead(newid).getLenClippedSeq();qwer++){
	    cout << CON_readpool->getRead(newid).getClippedSeqAsChar()[qwer];
	  }
#endif
	  aligncache[newreadseqtype].acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, 1, true, eoffset);
#ifdef ALIGNCHECK
	  checkbla.acquireSequences(
	    CON_2tmpcons.c_str(),
	    CON_2tmpcons.size(),
	    CON_readpool->getRead(newid).getClippedSeqAsChar(),
	    CON_readpool->getRead(newid).getLenClippedSeq(),
	    -1, newid, 1, 1);
#endif
	}
      }

#ifdef EXTRATRACEHELP
      cout << "\ntmpcons sequence:\n";
      for(uint32 qwer=0, cpl=0; qwer<CON_2tmpcons.size(); qwer++, cpl++){
	if(cpl==60) {
	  cout << endl;
	  cpl=0;
	}
	cout << CON_tmpcons[qwer];
      }
      cout << endl;
#endif

      CEBUG("Done acquiring.\n");
      madsl.clear();
      CEBUG("madsl cleared.\n");

      bool enforce_clean_ends=(*CON_miraparams)[CON_readpool->getRead(newid).getSequencingType()].getAlignParams().ads_enforce_clean_ends;
      // if the reference read is a rail or backbone, do not
      //  enforce clean ends! (to find SNPs!)
      if(CON_readpool->getRead(refid).isBackbone()
	 || CON_readpool->getRead(refid).isRail()) enforce_clean_ends=false;

      /* BaCh 29.03.2009
	  enforcing clean ends here is ... dangerous.
          reads is a contig that contain a sequencing error
          otherwise prevent the correct extension of the contig
          
                  |    .    |    .    |    .    |   
          ID1:-1  GGGCATTGTCTGCCACCTCTAACTCTACCTAA
          ID2:500 GGGCATTGTCTGCCAGCTCTAAC
          480-                   X

	 where the -1 contig is a single read with a sequencing error
	 near the end (C instead of G)

	 Possible ways out:
	  - enforcing clean ends needs at least k mismatches
	    IMPLEMENTED.  (presently k = 5 in ads.C)
	  - enforce clean ends only when overlap contains
	    above avg frequencies (to reduce false positives due to
	    really too many sequencing erros near an end). 
	    IMPLEMENTED (a bit differently: switch off if refid or
	    newid does not have repeat frequencies)
	  - performing hash based correction of obvious sequencing
	    errors (replacing wrong bases with N). Problematic for
	    "low coverage" Sanger cases or when different strains are 
	    used.
	    This would also need the contig editor to perform "obvious"
	    corrections in an alignment (replacing gaps with a base or N).
	    NOT IMPLEMENTED yet, quite some work.

	  CURRENTLY SWITCHED OFF! more harm than good
      */

      if(!(CON_readpool->getRead(newid).hasFreqRept()
	   || CON_readpool->getRead(refid).hasFreqRept())) enforce_clean_ends=false;

      bool dontpenalisengaps=false;
      if(CON_readpool->getRead(newid).isSequencingType(Read::SEQTYPE_PACBIO)
	 || CON_readpool->getRead(newid).isSequencingType(Read::SEQTYPE_PACBIO)){
	dontpenalisengaps=true;
      }

      aligncache[newreadseqtype].fullAlign(&madsl,enforce_clean_ends, dontpenalisengaps);
    }
    catch(Notify n){
      cout << dec << "Ouch ... error in alignment detected.\n";
      cout << "xcut: " << xcut << "\t";
      cout << "ycut: " << ycut << endl;
      cout << "eoffset: " << eoffset << "\n";

      cout << "dir_frnid: " << direction_frnid;
      cout << "\tdir_refid: " << direction_refid<< endl;
      cout << "Offset Refid:" << offsetrefid<<endl;

      Read::setCoutType(Read::AS_TEXTSHORT);
      cout << CON_readpool->getRead(newid);

      aligncache[newreadseqtype].coutWhatWasGiven();

      // throw again so that addRead() catches and we get info to replay
      throw Notify(n);
    }

    CEBUG("Done full align.\n");


//#ifdef EXTRATRACEHELP
//    if(CLASS_debug_counter==112){
//      setCoutType(AS_TEXT);
//      cout << *this;
//      setCoutType(AS_DEBUG);
//      cout << *this;
//    }
//#endif

	
#ifdef ALIGNCHECK
    CEBUG("ALC");
    if(madsl.empty()){

      // scrap this, AlignedDualSeqFacts does not drag this along
      //  (saving memory)
      //if(initialads->getOverlap()>50) {
      //	cout << "Missed badly?\n";
      //	cout << *initialads;
      //	cout << "Tmp cons.: \n" << CON_tmpcons << endl;
      //	cout << *this;
      //}

      list<AlignedDualSeq> tadsl;
      checkbla.fullAlign(&tadsl);
      if(!tadsl.empty()){
	CEBUG("Waaaaah! BSW failed!" << endl);

	//CEBUG("I had this contig:\n"<<*this);
	CEBUG("Had to align this:\n");
	  CEBUG("Refid: " << refid);
	  CEBUG("\tNewid: " << newid);
	  CEBUG("dir_frnid: " << direction_frnid);
	  CEBUG("\tdir_refid: " << direction_refid<< endl;);
	  CEBUG("Offset Refid:" << offsetrefid<<endl;);
	  CEBUG("xcut: " << xcut << "\t");
	  CEBUG("ycut: " << ycut << endl);
	  CEBUG("eoffset: " << eoffset << "\t");

#ifdef CEBUGFLAG
	  CEBUG(" ----------------------------------------------------- \n");
	  CEBUG("# solutions found: "<< tadsl.size() << endl);
	
	  {
	    list<AlignedDualSeq>::const_iterator Itmp=tadsl.begin();
	    while(Itmp!=tadsl.end()){
	      CEBUG(*Itmp<<endl<<endl<<endl); Itmp++;
	    }
	  }
	
	  CEBUG(" ----------------------------------------------------- \n");
#endif
      }
    }
#endif

#ifdef CEBUGFLAG
    CEBUG(" ----------------------------------------------------- \n");
    CEBUG("# solutions found: "<< madsl.size() << endl);
	  
	  
    {
	list<AlignedDualSeq>::const_iterator Itmp=madsl.begin();
	while(Itmp!=madsl.end()){
	  CEBUG(*Itmp); Itmp++;
	}
    }
	  
    CEBUG(" ----------------------------------------------------- \n");
#endif
	
    if(madsl.empty()){
      //    throw Notify(Notify::INTERNAL, THISFUNC, "No solution?!? At least one expected.\n");    

#ifndef PUBLICQUIET
      cout << "rej: no align found";
      if(aligncache[newreadseqtype].wasBandHit()) cout << " BH";
      cout << '\t';
      //cout << "Ov: " << initialads->getOverlap() << "\t";
      //cout << "ESR: " << static_cast<uint16>(initialads->getScoreRatio()) << "\t";
      //cout << "S: " << initialads->getScore() << "\t";
      //cout << "ES: " << initialads->getExpectedScore() << "\t";
      cout.flush();
#endif
      errstat.code=ENOALIGN;
      //errstat.reads_affected.push_back(refid);
      if(CON_readpool->getRead(refid).isRail())  {
	getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
      }else{
	getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
      }
#ifdef CLOCK_STEPS
      us_steps[USCLO_SWALIGN]=diffsuseconds(us_start);
      coutAddReadClocks(us_steps);
#endif
      FUNCEND();
      return;
    }

    I=madsl.begin();
	
    // TODO: was tun?
    // z.Zt. erst einmal den besten nehmen. wird wahrsch. der richtige sein.
    // darauf achten, dass, wenn mglich, offset==0 ist
    {
      list<AlignedDualSeq>::const_iterator Ibest=I;
      int32 bestratio=0;
      int32 bestweight=0;
      int32 offset=10000;

      while(I!=madsl.end()){
	if(I->getScoreRatio()>=bestratio){
	  CEBUG(offset << "\t" << I->getOffsetInAlignment(newid));
	  if(I->getScoreRatio()>=bestratio){
	    //|| (offset>0 && I->getOffsetInAlignment(newid)<offset)){
	    if(I->getWeight() > bestweight) {
	      offset=I->getOffsetInAlignment(newid);
	      bestratio=I->getScoreRatio();
	      bestweight=I->getWeight();
	      Ibest=I;
	    }
	  }
	}
	I++;
      }    
      I=Ibest;
    }

    CEBUG(" ----------------------------------------------------- \n");
    CEBUG("# solution chosen: " << endl);
    CEBUG(*I);
    CEBUG(" ----------------------------------------------------- \n");

    doneit=true;

    foolsafe++;

    // should the alignment not be perfect, calculate new xcut (ycut)
    //  coordinates
    // IF we're in the last foolsaferedone loop, make sure we still go another
    //  loop
    if(xcut >0){
      if(I->getOffsetInAlignment(-1)!=0){
	CEBUG("Redo because of wrong left cut. Too far right.\n");
	doneit=false;
	if(foolsafe==5) foolsafe--;
	xcut-=I->getOffsetInAlignment(-1)+7;
      }
      //else if(I->getOffsetInAlignment(newid) > 5){
      //	// TODO: check whether we could accept this regardless of
      //	//  offset. check: right offsets.
      //	CEBUG("Redo because of wrong left cut. Too far left.\n");
      //	doneit=false;
      //	if(foolsafe<5) xcut+=I->getOffsetInAlignment(newid);
      //}
    }

    if(I->getRightOffsetInAlignment(-1)!=0
       && ycut < CON_counts.size()+1){
      CEBUG("Redo because of wrong right cut. Not taken enough cons.\n");
      doneit=false;
      if(foolsafe==5) foolsafe--;
      ycut+=I->getRightOffsetInAlignment(-1)+10;
    }
    if(aligncache[newreadseqtype].wasBandHit()){
      CEBUG("Redo because of band hit in banded SW.\n");
      rt_params.setAlignBandwidthMin(200);
      rt_params.setAlignBandwidthMax(400);
      rt_params.setAlignBandwidthInPercent(80);
      doneit=false;
    }

    if(foolsafe>1){
#ifndef PUBLICQUIET
      cout << "fsc (" << foolsafe << ")\t";
      cout.flush();
#endif
    }
    if(foolsafe>=5){
#ifndef PUBLICQUIET
      cout << "already 5 iterations to find optimum ads and still not found, using suboptimum.\n";
      cout.flush();
      
      CEBUG(*I);
#endif
      doneit=true;
    }

  }while(doneit==false);

#ifdef CLOCK_STEPS
  us_steps[USCLO_SWALIGN]=diffsuseconds(us_start);
  gettimeofday(&us_start,NULL);
#endif

  // need this, because if foolsafe kicks in, xcut may be <0 and ycut > size
  if(xcut<0){
    xcut=0;
  }
  if(static_cast<uint32>(ycut)>CON_counts.size()){
    ycut=CON_counts.size();
  }


  CEBUG("CON_specialsraddconditions: " << CON_specialsraddconditions << endl);
  CEBUG("newreadseqtype: " << Read::getNameOfSequencingType(newreadseqtype) << endl);

  // check for short reads whether we are violating against special rules for mismatch numbers
  if(CON_specialsraddconditions
    && (newreadseqtype==Read::SEQTYPE_SOLEXA
	|| newreadseqtype==Read::SEQTYPE_ABISOLID)){
    CEBUG("CON_ssrc_maxtotalerrors: " << CON_ssrc_maxtotalerrors << endl);
    if(CON_ssrc_maxtotalerrors >=0){
      if(static_cast<int32>(I->getNumMismatches()) 
	 + static_cast<int32>(I->getNumGaps()) > CON_ssrc_maxtotalerrors){
#ifndef PUBLICQUIET
	cout << "saf:mte " << (static_cast<int32>(I->getNumMismatches()) + static_cast<int32>(I->getNumGaps())) << ">" << CON_ssrc_maxtotalerrors;
#endif
	errstat.code=ESPECIALSRADDFAIL;
      }
    }
    CEBUG("CON_ssrc_maxmismatches: " << CON_ssrc_maxmismatches << endl);
    if(CON_ssrc_maxmismatches >=0){
      if(static_cast<int32>(I->getNumMismatches()) > CON_ssrc_maxmismatches){
#ifndef PUBLICQUIET
	cout << " saf:mnm " << I->getNumMismatches() << ">" << CON_ssrc_maxmismatches;
#endif
	errstat.code=ESPECIALSRADDFAIL;
      }
    }
    CEBUG("CON_ssrc_maxgaps: " << CON_ssrc_maxgaps << endl);
    if(CON_ssrc_maxgaps >=0){
      if(static_cast<int32>(I->getNumGaps()) > CON_ssrc_maxgaps){
#ifndef PUBLICQUIET
	cout << " saf:mng " << static_cast<int32>(I->getNumGaps()) << ">" << CON_ssrc_maxgaps;
#endif
	errstat.code=ESPECIALSRADDFAIL;
      }
    }
    if(errstat.code==ESPECIALSRADDFAIL){
#ifndef PUBLICQUIET
      cout << " specialsradd failed\t";
      //cout << *I;
#endif

      try{
	errstat.proposed_cutback=analyseADSForCuttingBackCERMap(*I, direction_frnid);
      }
      catch (Notify n){
	cout << *I;
	// throw again so that addRead() catches and we get info to replay
	throw Notify(n);
      }
      if(CON_readpool->getRead(refid).isRail())  {
	getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
      }else{
	getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
      }
#ifdef CLOCK_STEPS
      coutAddReadClocks(us_steps);
#endif

      FUNCEND();
      return;
    }
//    else{
//      cout << "accepting this: " << endl;
//      cout << "maxtotal: " << CON_ssrc_maxtotalerrors;
//      cout << "maxmis: " << CON_ssrc_maxmismatches;
//      cout << "maxgap: " << CON_ssrc_maxgaps;
//      cout << *I;
//    }
  }

#ifndef PUBLICQUIET
    cout << "\tASR: " << static_cast<uint16>(I->getScoreRatio()) << '\t';
    //cout << *I;
#endif

  // First check on rodirs: relaxed parameters (rodirs*2)
  // Reason: we haven't checked template restriction yet
  //  if we lateron see that we do not have a matching template,
  //  then we'll use the 'stricter' rodirs value
  if(initialads->getScoreRatio() > 
     I->getScoreRatio()+(2*rt_params.getContigParams().con_reject_on_drop_in_relscore)){
    // REMOVEME
#ifndef PUBLICQUIET
    cout << "Dead end (even lax)\t";
    cout << "ESR: " << static_cast<uint16>(initialads->getScoreRatio());
    cout << "\tASR: " << static_cast<uint16>(I->getScoreRatio()) << '\t';
    //cout << *I;
#endif
    //    cout << *I;

    errstat.code=EDROPINRELSCORE;
    //errstat.reads_affected.push_back(refid);

    // Diese entscheidung hier ist ... gefhrlich, wenn nicht gar falsch,
    //  denn es kann sein, dass ein potentiell guter match auch dieselben
    //  Fehler hatte wie dieses Align.
    // falsch!? ist legitim? diese bans werden ja nicht auf permbans uebertragen,
    //  weshalb es fuer _diesen_ Contig ja stimmen wuerde. 

    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    FUNCEND();
    return;
  }


  // compute whether the contig will grow (left or right)

  // check left side
  int32 expect_growleft=0;
  if(I->getOffsetInAlignment(newid)==0 && I->getOffsetInAlignment(-1)!=0){
    expect_growleft=-(xcut-I->getOffsetInAlignment(-1));
    if(expect_growleft<0) expect_growleft=0;
  }
  int32 expect_growright=0;
  if(ycut==CON_counts.size()
     && I->getRightOffsetInAlignment(newid)==0
     && I->getRightOffsetInAlignment(-1)!=0){
    expect_growright=I->getRightOffsetInAlignment(-1);
  }

  if(forcegrow>0
     && max(expect_growleft, expect_growright) < forcegrow){
    errstat.code=EFORCEDGROWTHNOTREACHED;

    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    FUNCEND();
    return;
  }

  if(forcegrow<0
     && (expect_growleft > 0
	 || expect_growright > 0)){
    errstat.code=EGROWTHNOTALLOWED;

    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    FUNCEND();
    return;
  }

  // now check whether we're aligning in an area where we already 
  //  reached max coverage
  //
  // New 14.12.2008: if contig grows, skip that as we will very
  //  probably be in a multicopy environment and if we add
  //  so much coverage that maxcoverage allowed is reached,
  //  the building stops (which is bad).
  // TODO: do better, perhaps before the alignment?
  //  furthermore, change reduceSKim() and Pathfinder to 
  //  select links where multicopy is involved to have at least
  //  20 to 30 bases extension!
  // New 03.01.2009: scrap the above, leads to heavy overcompression
  // New 29.04.2011:
  // taken out unconditional check:
  //  old routine always gave back "true" anyway except if coverage 
  //  reached 16384
  //  with new 32 bit counters, will never be really reached anyway

  if(newid_ismulticopy
     && !havematchingtemplatepartner
//     && expect_growleft==0
//     && expect_growright==0
     && (*CON_miraparams)[0].getAssemblyParams().as_uniform_read_distribution
     && !checkFreeCoverageForAddingRead(newreadseqtype, xcut, ycut)){
    errstat.code=EMAXCOVERAGEREACHED;

    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    FUNCEND();
    return;
  }

#ifdef CLOCK_STEPS
  us_steps[USCLO_PREINSCHK]=diffsuseconds(us_start);
  gettimeofday(&us_start,NULL);
#endif


  // forcemerge: for reads not mapping 100%
  bool forcemerge=false;
  if(CON_hasforcemergeareas
     && CON_mergenewsrreads
     && newreadseqtype==Read::SEQTYPE_SOLEXA
     && rt_params.getContigParams().con_mergeshortreads){
    //// for testing
    //forcemerge=true;

    // TODO: is using xcut & ycut OK? Probably yes.
    cccontainer_t::iterator ccI=CON_counts.begin();
    advance(ccI,xcut);
    for(uint32 ii=xcut; ii<ycut; ++ii, ++ccI){
      if(ccI->forcemergearea) {
	forcemerge=true;
	break;
      }
    }
  }

  // now check whether we can merge this read
  // if we want to map, we need 100% score ratio
  CEBUG("CON_mergenewsrreads: " << CON_mergenewsrreads << endl);
  CEBUG("maymapthisread: " << maymapthisread << endl);
  CEBUG("newreadseqtype==Read::SEQTYPE_SOLEXA: " << (newreadseqtype==Read::SEQTYPE_SOLEXA) << endl);
  CEBUG("rt_params.getContigParams().con_mergeshortreads: " << rt_params.getContigParams().con_mergeshortreads << endl);
  CEBUG("I->getScoreRatio() == 100: " << (I->getScoreRatio() == 100) << endl);

  if(forcemerge 
     || (CON_mergenewsrreads
	 && maymapthisread
	 && newreadseqtype==Read::SEQTYPE_SOLEXA
	 && rt_params.getContigParams().con_mergeshortreads
	 && I->getScoreRatio() == 100)){

    CEBUG("Try map.\n");

    bool canmap=true;

    // sometimes the 100% matches have some weirdnesses against N's
    // ADS should nowadays take care of this, but just to be sure:
    if(!forcemerge){
      if(canmap &&
	 (I->getNumMismatches()>0
	  || I->getNumGaps()>0)){
#ifndef PUBLICQUIET
	cout << " has mismatches";
#endif
	canmap=false;
      }
    }

    // furthermore, we may not be extending the contig left or right

    if(expect_growleft>0) {
      canmap=false;
#ifndef PUBLICQUIET
      cout << " growl";
#endif
    }
    if(expect_growright>0) {
      canmap=false;
#ifndef PUBLICQUIET
      cout << " growr";
#endif
    }

    // ok, consensus does not grow ... but it might be in a part
    //  that initially was not in the contig.
    if(canmap &&
       (CON_counts[xcut].backbonechar=='@'
	|| CON_counts[ycut-1].backbonechar=='@')) {
#ifndef PUBLICQUIET
//      cout << "xcut: " << xcut << "\tycut: " << ycut << endl;
//      dumpAsDebug(cout);
      cout << " ingrown";
#endif
      canmap=false;
    }

    // maybe we'd like to have reads at contig ends not mapped (e.g.
    //  for scaffolding)
    if(canmap
       && rt_params.getContigParams().con_msr_keependsunmapped!=0){
      int32 dist=rt_params.getContigParams().con_msr_keependsunmapped;
      if(dist<0){
	dist=CON_readpool->getRead(newid).getInsizeTo();
      }

//      cout << "lri: " << CON_index_leftestbbread
//	   << "\trri: " << CON_index_rightestbbread
//	   << "\nxcut: " << xcut
//	   << "\tycut: " << ycut
//	   << "\tdist: " << dist
//	   << "\nr: " << dist+CON_reads[CON_index_leftestbbread].offset
//	   << "\tl: " << (CON_reads[CON_index_rightestbbread].offset
//			+ CON_reads[CON_index_rightestbbread].read.getLenClippedSeq())-dist
//	;

      // if no distance or when read has no paired-end partner, we could map
      //  anyway and would not need to check further
      if(dist>0 && CON_readpool->getRead(newid).getTemplatePartnerID() >= 0){
	// check whether we are in boundaries, if yes, do not map
	if(xcut <= dist+CON_reads[CON_index_leftestbbread].offset 
	   || ycut >= (CON_reads[CON_index_rightestbbread].offset
		       + CON_reads[CON_index_rightestbbread].read.getLenClippedSeq())-dist){
	  canmap=false;
	}
      }

      //cout << "\ncanmap: " << canmap << endl;
    }

    // map if possible, else the read will be normally added
    if(canmap){
#ifndef PUBLICQUIET
      cout << " mapping ...";
      cout.flush();
#endif

      bool wasmapped=true;
      try{
	wasmapped=insertMappedReadInContig(*I, newreadseqtype, xcut, direction_frnid,forcemerge);
      }
      catch (Notify n){
	cout << *I;
	// throw again so that addRead() catches and we get info to replay
	throw Notify(n);
      }

      if(wasmapped){
#ifdef CLOCK_STEPS
	us_steps[USCLO_INSCON]=diffsuseconds(us_start);
	coutAddReadClocks(us_steps);
#endif

	// count this in the readsperstrain statistics
	++CON_readsperstrain[CON_readpool->getRead(newid).getStrainID()];
	return;
      }else{
#ifndef PUBLICQUIET
	cout << " failedmap";
	cout.flush();
#endif	
      }
    }else{
#ifndef PUBLICQUIET
      cout << " cantmap";
      cout.flush();
#endif
    }
  }

  //cout << *I;

#ifndef PUBLICQUIET
  cout << " inscon...";
  cout.flush();
#endif
#ifdef CLOCK_STEPS
  gettimeofday(&us_start,NULL);
#endif
  insertReadInContig(*I, xcut,direction_frnid, direction_refid);
#ifdef CLOCK_STEPS
  us_steps[USCLO_INSCON]=diffsuseconds(us_start);
#endif
#ifndef PUBLICQUIET
  cout << "done";
  cout.flush();
#endif
  CON_maprpids_to_conreadsids[newid]=CON_reads.size()-1;
  
  
#ifdef CLOCK_STEPS
  gettimeofday(&us_start,NULL);
#endif
#ifndef PUBLICQUIET
  cout << " updbasloc...";
  cout.flush();
#endif
  // Put base locks in CON_counts that this read produces
  updateBaseLocks(CON_reads.back(), true);
#ifndef PUBLICQUIET
  cout << "done";
  cout.flush();
#endif
#ifdef CLOCK_STEPS
  us_steps[USCLO_UPDBLOCKS]=diffsuseconds(us_start);
#endif
  
#ifndef PUBLICQUIET
  cout << " chkcon...";
  cout.flush();
#endif
  paranoiaBUGSTAT(checkContig());
#ifndef PUBLICQUIET
  cout << "done";
  cout.flush();
#endif
  
  
  
  // Template handling, part 2

  if(!havematchingtemplatepartner){
#ifndef PUBLICQUIET
    cout << " tmplhand2...";
    cout.flush();
#endif
  
    // Second check on rodirs: strict parameters (rodirs)
    // Reason: if there was no matching tpartner, apply strict parameter

    if(initialads->getScoreRatio()>I->getScoreRatio()+rt_params.getContigParams().con_reject_on_drop_in_relscore){
      // REMOVEME
      
#ifndef PUBLICQUIET
      cout << "Dead end (strict)\t";
      cout << "ESR: " << static_cast<uint16>(initialads->getScoreRatio());
      cout << "\tASR: " << static_cast<uint16>(I->getScoreRatio()) << '\t';
      // cout << *I;
#endif
      
#ifdef CLOCK_STEPS
      gettimeofday(&us_start,NULL);
#endif
      deleteRead(newid);
#ifdef CLOCK_STEPS
      us_steps[USCLO_DELREAD]=diffsuseconds(us_start);
#endif

      errstat.code=EDROPINRELSCORE;

      //errstat.reads_affected.push_back(refid);

      // Diese entscheidung hier ist ... gefhrlich, wenn nicht gar falsch,
      //  denn es kann sein, dass ein potentiell guter match auch dieselben
      //  Fehler hatte wie dieses Align.
      // falsch!? ist legitim? diese bans werden ja nicht auf permbans uebertragen,
      //  weshalb es fuer _diesen_ Contig ja stimmen wuerde. 
      if(CON_readpool->getRead(refid).isRail())  {
	getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
      }else{
	getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
      }
#ifdef CLOCK_STEPS
      coutAddReadClocks(us_steps);
#endif

      FUNCEND();
      return;
    }
  }

#ifdef CLOCK_STEPS
  gettimeofday(&us_start,NULL);
#endif
#ifndef PUBLICQUIET
  cout << " ansrmbzone...";
  cout.flush();
#endif
  if(analyseRMBZones(CON_reads.back(), errstat.srmb_reads_affected) == true){
#ifdef CLOCK_STEPS
    us_steps[USCLO_ANRMBZ]=diffsuseconds(us_start);
#endif
#ifndef PUBLICQUIET
    cout << "rej: srmb zone\t";
#endif
    //if(CON_reads.back().read.getName()=="GBIBI26TF") {
    //  cout << "\nMUSTSAVE!\n";
    //  saveAsGAP4DA("error_out.gap4da", cout);
    //  exit(0);
    //}
    // remove the read from the contig as we don't want it
#ifdef CLOCK_STEPS
    gettimeofday(&us_start,NULL);
#endif
    deleteRead(newid);
#ifdef CLOCK_STEPS
    us_steps[USCLO_DELREAD]=diffsuseconds(us_start);
#endif

    errstat.code=ESRMBMISMATCH;
    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    // errstat.srmb_reads_affected will already contains the ids of the reads
    //  that have RMB mismatches with that read
    FUNCEND();
    return;
  }
#ifdef CLOCK_STEPS
  us_steps[USCLO_ANRMBZ]=diffsuseconds(us_start);
#endif
#ifndef PUBLICQUIET
  cout << "done";
  cout.flush();
#endif

#ifndef PUBLICQUIET
  cout << " andngrzone...";
  cout.flush();
#endif
  if(analyseDangerZones(CON_reads.back()) == true){
    // TODO: in ALUS & co strengere Kriterien.
    // remove the read from the contig as we don't want it

#ifndef PUBLICQUIET
    cout << "rej: danger zone\t";
#endif

#ifdef CLOCK_STEPS
    gettimeofday(&us_start,NULL);
#endif
    deleteRead(newid);
#ifdef CLOCK_STEPS
    us_steps[USCLO_DELREAD]=diffsuseconds(us_start);
#endif
    
    errstat.code=EDANGERZONE;
    if(CON_readpool->getRead(refid).isRail())  {
      getRailsAsReadsAffected(refid, errstat.reads_affected, xcut, ycut);
    }else{
      getReadIDsAtContigPosition(errstat.reads_affected, xcut, ycut);
    }
#ifdef CLOCK_STEPS
    coutAddReadClocks(us_steps);
#endif

    FUNCEND();
    return;
  }
#ifndef PUBLICQUIET
  cout << "done";
  cout.flush();
#endif



//  //TODO: parametrieren!
//  if(analyseAllZones(CON_reads.back()) == true){
//    // todo: in ALUS & co strengere Kriterien.
//    // remove the read from the contig as we don't want it
//
//    deleteRead(newid);
//    
//    throw Flow(Flow::POSSIBLE_DEAD_END);
//  }

  // count this in the readsperstrain statistics
  ++CON_readsperstrain[CON_readpool->getRead(newid).getStrainID()];

  if(CON_fixedconsseq.size()){
    nukeSTLContainer(CON_fixedconsseq);
    nukeSTLContainer(CON_fixedconsqual);
  }

#ifdef CLOCK_STEPS
  coutAddReadClocks(us_steps);
#endif

  FUNCEND();
  return;
}
//#define CEBUG(bla)
//#define CEBUGF(bla)
//#undef CEBUGFLAG

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// coffset:  the offset of the contig part in the contig used in the ads
void Contig::coutAddReadClocks(const vector<suseconds_t> & suv)
{
#ifdef CLOCK_STEPS
  cout << "\nctimings: " << 
    suv[USCLO_PRE] << " / " <<
    suv[USCLO_DIR] << " / " <<
    suv[USCLO_XCUT] << " / " <<
    suv[USCLO_TEMPL1] << " / " <<
    suv[USCLO_SWALIGN] << " / " <<
    suv[USCLO_PREINSCHK] << " / " <<
    suv[USCLO_INSCON] << " / " <<
    suv[USCLO_UPDBLOCKS] << " / " <<
    suv[USCLO_DELREAD] << " / " <<
    suv[USCLO_ANRMBZ] << '\n';
#endif
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::setContigCoverageTarget(vector<uint32> covtarget)
{
  FUNCSTART("void Contig::setContigCoverageTarget(vector<uint32)");
  
  CON_targetcoverageperst.clear();
  if(covtarget.empty()) return;

  BUGIFTHROW(covtarget.size()!=Read::SEQTYPE_END, "covtarget.size()!=Read::SEQTYPE_END ??");
  for(uint32 i=0; i<covtarget.size(); i++){
    CON_targetcoverageperst.push_back(static_cast<uint16>(covtarget[i]));
  }

  FUNCEND();
}


/*************************************************************************
 *
 * looks whether adding a read between the given positions would not
 *  exceed the maximum allowed coverage
 *
 * return true for "it's ok to add"
 * false for "maximum already reached"
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}
bool Contig::checkFreeCoverageForAddingRead(const uint8 newreadseqtype, const int32 xcut, const int32 ycut)
{
  FUNCSTART("void Contig::bool Contig::checkFreeCoverageForAddingRead(const uint8 newreadseqtype, const int32 xcut, const int32 ycut)");

  CEBUG("\ncovcheck: st:" << static_cast<uint16>(newreadseqtype) << "\txcut: " << xcut << "\tycut: " << ycut << '\n');

  ccctype_t maxcovallowed=0;
  if(!CON_targetcoverageperst.empty()){
    ccctype_t tmp=static_cast<ccctype_t>(CON_targetcoverageperst[newreadseqtype]) * ((*CON_miraparams)[newreadseqtype]).getAssemblyParams().as_urd_cutoffmultiplier;
    maxcovallowed=max(tmp,maxcovallowed);
  }

  cccontainer_t::const_iterator ccI=CON_counts.begin();
  advance(ccI,xcut);
  
  if(maxcovallowed<10) maxcovallowed=10;

  // The 16383 number stems from the 16 bit base counters in CON_concounts[]
  //  which use increments by 4 
  // 1073741823 is for 32 bit counters (== 2^32 / 4 -1)
  for(int32 ci=xcut; ci<ycut && ccI!=CON_counts.end(); ci++, ccI++){
    CEBUG("ci: " << ci << '\t' << ccI->seqtype_cov[0] << '\t' << ccI->seqtype_cov[1] << '\t' << ccI->seqtype_cov[2] << '\t' << ccI->seqtype_cov[3] << "\tc: " << ccI->total_cov);
    if(ccI->seqtype_cov[newreadseqtype]>maxcovallowed
      || ccI->total_cov == 1073741823) {
      return false;
    }
    CEBUG('\n');
  }

  FUNCEND();
  return true;
}
//#define CEBUG(bla)



/*************************************************************************
 *
 * Starting from a reference id that is a rail, takes reads in readpool 
 *  before and after that are rails and that fall into the same xcut,
 *  ycut region as given and pushed them into the vector
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}
void Contig::getRailsAsReadsAffected(const int32 refid, vector<int32> & reads_affected, const int32 xcut, const int32 ycut)
{
  FUNCSTART("void Contig::getRailsAsReadsAffected(const int32 refid, vector<int32> & reads_affected)");

  if(CON_readpool->getRead(refid).isRail()){
    CEBUG("Ref Rail: " << CON_readpool->getRead(refid).getName() << '\n');
    reads_affected.push_back(refid);

    // initialise CON_maprpids_to_conreadsids if it's dirty (==empty)
    if(CON_maprpids_to_conreadsids.empty()){
      CON_maprpids_to_conreadsids.resize(CON_readpool->size(),-1);
      for(int32 i=0;i<CON_reads.size(); i++){
	if(CON_reads[i].orpid>=0){
	  CON_maprpids_to_conreadsids[CON_reads[i].orpid]=i;
	}
      }
    }

    int32 rpid_refid=CON_maprpids_to_conreadsids[refid];
    if(rpid_refid==-1){
      throw Notify(Notify::INTERNAL, THISFUNC, "RefID not found on contig?");
    }

    for(int32 actid=rpid_refid-1; actid>=0 && CON_reads[actid].orpid>=0 && CON_reads[actid].read.isRail(); actid--){
      if((xcut>=CON_reads[actid].offset && xcut <CON_reads[actid].offset+CON_reads[actid].read.getLenClippedSeq())
	 || (ycut>=CON_reads[actid].offset && ycut <CON_reads[actid].offset+CON_reads[actid].read.getLenClippedSeq())){
	CEBUG("also rail-: " << CON_readpool->getRead(CON_reads[actid].orpid).getName() << '\n');
	reads_affected.push_back(CON_reads[actid].orpid);
      }else{
	// get out, we won't find more downstream anyway
	break;
      }
    }
    for(int32 actid=rpid_refid+1; actid<CON_reads.size() && CON_reads[actid].orpid>=0 && CON_reads[actid].read.isRail(); actid++){
      if((xcut>=CON_reads[actid].offset && xcut <CON_reads[actid].offset+CON_reads[actid].read.getLenClippedSeq())
	 || (ycut>=CON_reads[actid].offset && ycut <CON_reads[actid].offset+CON_reads[actid].read.getLenClippedSeq())){
	CEBUG("also rail+: " << CON_readpool->getRead(CON_reads[actid].orpid).getName() << '\n');
	reads_affected.push_back(CON_reads[actid].orpid);
      }else{
	// get out, we won't find more upstream anyway
	break;
      }
    }
  }

  FUNCEND();
  return;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 * Meant for Solexa data
 *
 * looks at ADS for short reads, looks how far it can cut back the read
 * cutback rule: base quality <6, mismatches not farther than 3 apart
 *
 *************************************************************************/

//#define CEBUG(bla)   {if(CON_reads.size()>=9 && CON_reads.size()<=11 ){cout << bla; cout.flush();}}
//#define CEBUG(bla)   {if(CON_counts.size()>=5900 && CON_counts.size()<=6100 ){cout << bla; cout.flush();}}

//#define CEBUG(bla)   {cout << bla; cout.flush();}

int32 Contig::analyseADSForCuttingBackCERMap(const AlignedDualSeq & ads, int32 direction_frnid)
{
  FUNCSTART("void Contig::analyseADSForCuttingBackCERMap(const AlignedDualSeq & ads, int32 direction_frnid)");

  //BUGIFTHROW(ads.getOffsetInAlignment(-1)>0, "ads.getOffsetInAlignment(-1) > 0 ?");
  //BUGIFTHROW(ads.getRightOffsetInAlignment(-1)>0, "ads.getRightOffsetInAlignment(-1) > 0 ?");

  CEBUG(ads);

  // This is a read that claims it wants to be mapped but has too many errors
  int32 id=ads.getOtherID(-1);

  if(CON_readpool->getRead(id).getSequencingType() != Read::SEQTYPE_SOLEXA
     && CON_readpool->getRead(id).getSequencingType() != Read::SEQTYPE_ABISOLID){
    throw Notify(Notify::INTERNAL, THISFUNC, "Trying to cut back other than Solexa / SOLiD ?");
  }

  // get pointer to the aligned contig 
  const char * contigptr;
  const char * readptr;
  //int32 indexincontig;
  //int32 indexinread;

  // ... and advance it to the beginning of the read
  // we made sure above that (ads.getOffsetInAlignment(-1) == 0) is true

  /*
    in this routine, we will run always in read forward direction, i.e., the two
    forward sequences

    ID1:-1  GTTCCGGATTATGGCTTCACGCCCGCTACGCCCGATAATGTCGCCAAGA
    ID2:83   TTCCGGATTATGGCTTCACGCCCGCTCCGCACGTTA
    0-                                 X   X  X

    will run read 83 (and the contig) from left to right and if the read is
    reversed (as is 84)

    ID1:-1  GATCATATCCCGCCGGTCCCAGGCATGGTTGCCCAGCGTAATCACATCG
    ID2:84   AACAAACCCCGCCGGTCCCAGGCATGGTTGCCCAGC
    0-        X  X X

    will run from right to left.
   */

  int32 ptrincr=1;
  if(direction_frnid>0){
    contigptr= ads.getAlignedSequence(-1)+ads.getOffsetInAlignment(id);
    readptr= ads.getAlignedSequence(id);
    //indexinread=0;
    //indexincontig= offsetnewread;
  }else{
    contigptr= ads.getAlignedSequence(-1)+ads.getOffsetInAlignment(id)+ads.getOverlapLen()-1;
    readptr= ads.getAlignedSequence(id)+ads.getOverlapLen()-1;
    ptrincr=-1;
  }

  const vector<base_quality_t> & qv=CON_readpool->getRead(id).getQualities();
  vector<base_quality_t>::const_iterator qvI=qv.begin();
  advance(qvI,CON_readpool->getRead(id).getLeftClipoff());

  int32 canstartcutpos=-1;
  int32 lastcutpos=-1;
  int32 currentreadpos=CON_readpool->getRead(id).getLeftClipoff();

  for(uint32 ioverlap=0; ioverlap<ads.getOverlapLen(); ioverlap++, contigptr+=ptrincr, readptr+=ptrincr, qvI++, currentreadpos++){
    //CEBUG("io: " << ioverlap << "\ti_c: " << indexincontig);
    //CEBUG("\ti_r: " << indexinread);
    CEBUG("\nio: " << ioverlap);
    CEBUG("\t*rptr: " << *readptr);
    CEBUG("\t*qvI: " << static_cast<uint16>(*qvI));
    CEBUG("\t*cptr: " << *contigptr);
    CEBUG("\tcrp: " << currentreadpos);

    BUGIFTHROW(qvI==qv.end(),"quality out of bounds? 2");

    if(*readptr=='*') {
      currentreadpos--;
      // the following works only because we do not have (I hope)
      //  gaps at the very end of sequences in an ADS
      qvI--;
    }

    if(*contigptr=='#' && *readptr=='*') continue;
    if(*contigptr!=*readptr
       && *contigptr != 'N'
       && *readptr != 'N'){
      if(*readptr=='*'){
	// the following works only because we do not have (I hope)
	//  gaps at the very end of sequences in an ADS
	if((*qvI+*(qvI+1))/2<6) {
	  if(canstartcutpos<0) canstartcutpos=currentreadpos;
	  lastcutpos=currentreadpos;
	}
      }else{
	if(*qvI<6) {
	  if(canstartcutpos<0) canstartcutpos=currentreadpos;
	  lastcutpos=currentreadpos;
	}
      }
    }else{
      if((currentreadpos-lastcutpos)>3) {
	canstartcutpos=-1;
	lastcutpos=-1;
      }
    }
    CEBUG("\tcscp: " << canstartcutpos << "\tlcp: " << lastcutpos);
  }
  CEBUG("\ncutback result:\ncscp: " << canstartcutpos << "\tlcp: " << lastcutpos);

  canstartcutpos-=2;
  if(canstartcutpos<20) canstartcutpos=-1;

  return canstartcutpos;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 * "maps" a read, return true if successful
 *
 * if read or contig contain IUPACs that differ, mapping cannot be done
 *  (return false)
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {if(CON_reads.size()>=9 && CON_reads.size()<=11 ){cout << bla; cout.flush();}}
//#define CEBUG(bla)   {if(CON_counts.size()>=5900 && CON_counts.size()<=6100 ){cout << bla; cout.flush();}}

//#define CEBUG(bla)   {cout << bla; cout.flush();}

bool Contig::insertMappedReadInContig(const AlignedDualSeq & ads, const uint8 newreadseqtype, const uint32 coffset, const int32 direction_frnid, bool forcemerge)
{
  //hier weiter: gaps anders behandeln (mitzaehlen) desgleichen auch in insertRead()

  FUNCSTART("void Contig::insertMappedReadInContig(AlignedDualSeq & ads, uint32 coffset, int32 direction_frnid)");

  CEBUG("CON_reads:" << CON_reads.size() << endl);
  CEBUG("CON_counts:" << CON_counts.size() << endl);

  // This is a read that claims it can be mapped, so better be sure
  //  the consensus is completely covering it!
  BUGIFTHROW(ads.getOffsetInAlignment(-1)>0, "ads.getOffsetInAlignment(-1) > 0 ?");
  BUGIFTHROW(ads.getRightOffsetInAlignment(-1)>0, "ads.getRightOffsetInAlignment(-1) > 0 ?");

  // TODO: do we need that?
  definalise();

  // precalculate the offset into CON_counts.bbcounts[] according to 
  //  sequencing type
  unsigned int sr_seqtypeoffset=0;
  if(newreadseqtype == Read::SEQTYPE_SOLEXA){
    sr_seqtypeoffset=0;
  }else if(newreadseqtype == Read::SEQTYPE_ABISOLID){
    sr_seqtypeoffset=1;
  }else{
    throw Notify(Notify::INTERNAL, THISFUNC, "Trying to map other than Solexa / SOLiD ?");
  }

  //// precalculate the bbstrains bitmask
  //uint8 strainmask=0;
  //{
  //  int32 strainid=CON_readpool->getRead(id).getStrainID();
  //  if(strainid>7 || strainid<0) {
  //    throw Notify(Notify::FATAL, THISFUNC, "Mapping more than 8 strains? Sorry, not possible yet.");
  //  }
  //  BITSET(strainid, strainmask);
  //}

  int32 id=ads.getOtherID(-1);

  // precalculate the bbstrains bitmask
  uint8 strainmask=0;
  if(newreadseqtype==Read::SEQTYPE_SOLEXA){
    strainmask=getBBStrainMask(CON_readpool->getRead(id).getStrainID());
  }

  //CEBUG(*this);

  CEBUG("coffset: " << coffset << endl);

  // compute the offset of this read in the contig.
  uint32 offsetnewread=coffset+ads.getOffsetInAlignment(id);


  // mapping is done in two steps
  // first loop is simulation to determine if read can really be mapped,
  // second loop the mapping is really done
  // if we force the merge ... well, no first loop

  for(uint32 loopi=0; loopi<2; loopi++){
    if(forcemerge) loopi=1;

    CEBUG("loopi: " << loopi << endl);

    // get pointer to the aligned contig 
    const char * contigptr;
    const char * readptr;
    int32 indexincontig;
    int32 indexinread;

    // ... and advance it to the beginning of the overlap
    // we made sure above that (ads.getOffsetInAlignment(-1) == 0) is true

    contigptr= ads.getAlignedSequence(-1)+ads.getOffsetInAlignment(id);
    readptr= ads.getAlignedSequence(id);
    indexincontig= offsetnewread;
    indexinread=0;

    cccontainer_t::iterator ccI=CON_counts.begin();
    advance(ccI,offsetnewread);

    // get the qualities for this read and position a pointer on the first
    const vector<base_quality_t> & qv=CON_readpool->getRead(id).getQualities();
    vector<base_quality_t>::const_iterator qvI=qv.begin();
    if(direction_frnid>0){
      CEBUG("advancef " << CON_readpool->getRead(id).getLeftClipoff() << endl);
      advance(qvI,CON_readpool->getRead(id).getLeftClipoff());
    }else{
      CEBUG("advancer " << CON_readpool->getRead(id).getRightClipoff()-1 << endl);
      advance(qvI,CON_readpool->getRead(id).getRightClipoff()-1);
    }

    for(uint32 ioverlap=0; ioverlap<ads.getOverlapLen(); ioverlap++, contigptr++, readptr++, indexincontig++, indexinread++, ccI++, qvI+=direction_frnid){
      CEBUG("io: " << ioverlap << "\ti_c: " << indexincontig);
      CEBUG("\ti_r: " << indexinread);
      CEBUG("\t*rptr: " << *readptr);
      CEBUG("\t*qvI: " << (uint16) *qvI);
      CEBUG("\t*cptr: " << *contigptr);

      /* WARNING!!!!!!!!!!!!!!!!!!!!!!!!!
       
	 These two checks below which look at the boundaries of the quality
	 iterator only work because at the moment MIRA enforces a left clip of 1 on
	 all short reads to hide the tag in GAP4
    
	 rethink that strategy should this change one day!
      */
      if(forcemerge){
	// on forced merge, do everything so that we don't stop
	if(qvI==qv.begin() || qvI==qv.end()) qvI-=direction_frnid;
      }else{
	BUGIFTHROW(qvI==qv.begin(),"quality out of bounds? 1");
	BUGIFTHROW(qvI==qv.end(),"quality out of bounds? 2");
      }

      if(indexincontig<0 || indexincontig >= CON_counts.size()+1) {
	// force it ... somehow ... to continue the for loop;
	if(forcemerge) continue;

	cout << "indexincontig: " << indexincontig << endl;
	cout << "\nCON_counts.size()+1: " << CON_counts.size()+1;
      
	cout << "\n\n" << ads << endl;
      
	setCoutType(AS_CAF);
	cout << *this;
	setCoutType(AS_TEXT);
	cout << *this;
	throw Notify(Notify::INTERNAL, THISFUNC, "BOUNDCHECK error");
      }

      if(loopi){
	ccI->seqtype_cov[newreadseqtype]++;
	ccI->total_cov++;
      }

      BUGIFTHROW(!forcemerge && *contigptr=='*',"!forcemerge && *contigptr=='*' ?");

      // we may have a gap in the read, this is ok as long as the 
      //   consensus has the code for "old gap" ('#')
      if(*readptr=='*'){
	if(!forcemerge && *contigptr!='#'){
	  // return only if we do not try to force the merge
	  // if we force ... do nothing

	  //cout << "\n\n" << ads;
	  //cout << "Alert: read has " << static_cast<char>(*readptr);
	  //cout << "\tcontig has " << static_cast<char>(*contigptr);
	  //cout << "\nio: " << ioverlap << "\ti_c: " << indexincontig;
	  //cout << "\ti_r: " << indexinread;
	  //cout << "\n"; 
	  //throw Notify(Notify::INTERNAL, THISFUNC, "logical error");
	  
	  return false;
	}else{
	  // in the mean time, the qvI iterator must be decreased/increased as
	  //  no quality value exists at the moment in the read and we must counter
	  //  the increase/decrease in the for loop
	  qvI-=direction_frnid;

	  if(loopi){
	    // increase the gap count
	    ccI->bbcounts[sr_seqtypeoffset]+=1;
	    
	    // set the strain bitmask
	    ccI->bbstrains[sr_seqtypeoffset]|=strainmask;
	    
	    // see whether we need to adapt the gap quality
	    uint16 gapqual=*qvI;
	    // move the quality pointer back by one to counterbalance
	    //  the for loop (illegally) munching up one quality value
	    //  as this was a *new* gap in the read
	    qvI-=direction_frnid;
	    if(qvI!=qv.end()) gapqual=(gapqual+(*qvI))/2;
	    if(gapqual>ccI->bbbestquals[sr_seqtypeoffset]){
	      ccI->bbbestquals[sr_seqtypeoffset]=static_cast<base_quality_t>(gapqual);
	    }
	  }
	}
      }else{
	if(forcemerge
	   || toupper(*readptr) == ccI->backbonechar
	   || toupper(*readptr) == 'N'){            // in dubio pro reo
	  if(loopi){
	    ccI->bbcounts[sr_seqtypeoffset]+=1;
	    ccI->bbstrains[sr_seqtypeoffset]|=strainmask;
	    if(*qvI > ccI->bbbestquals[sr_seqtypeoffset]){
	      ccI->bbbestquals[sr_seqtypeoffset]=*qvI;
	    }
	  }
	}else if(!forcemerge && *contigptr!='#'){
	  //cout << "\n\n" << ads;
	  //
	  //cout << "Alert2: read has " << static_cast<char>(*readptr);
	  //cout << "\tcontig has " << static_cast<char>(*contigptr);
	  //cout << "\nio: " << ioverlap << "\ti_c: " << indexincontig;
	  //cout << "\ti_r: " << indexinread;
	  //cout << "\nbackbone has: " << static_cast<char>(ccI->backbonechar) << endl;
	  //throw Notify(Notify::INTERNAL, THISFUNC, "logical error 2");

	  // return only if we do not try to force the merge
	  // if we force ... do nothing
	  return false;
	}
      }
    }    
  }

  CON_nummergedreads_perseqtype[sr_seqtypeoffset]++;

  return true;
}

//#define CEBUG(bla)



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}

//#define CLOCK_STEPS
//#define IRICTRACE

// coffset:  the offset of the contig part in the contig used in the ads
void Contig::insertReadInContig(const AlignedDualSeq & ads, uint32 coffset, int32 direction_frnid, int32 direction_refid)
{
  FUNCSTART("void Contig::insertReadInContig(AlignedDualSeq & ads, uint32 coffset, int32 direction_frnid, int32 direction_refid)");

#ifdef CLOCK_STEPS
  timeval us_start;
  gettimeofday(&us_start,NULL);
#endif

  CEBUG("CON_reads:" << CON_reads.size() << endl);
  CEBUG("CON_counts:" << CON_counts.size() << endl);

#ifdef IRICTRACE
  cout << "predef "; cout.flush();
#endif

  definalise();

#ifdef IRICTRACE
  cout << "postdef "; cout.flush();
#endif


  int32 id=ads.getOtherID(-1);

  //if(CON_readpool->getRead(id).getName()=="125691_2930_3355"){
  //  cout << "Dingodong\n";
  //  //cout << *this;
  //}

  //CEBUG(*this);

  CEBUG("coffset: " << coffset << endl);

  // compute the offset of this read in the contig.
  //  it might change later if the contig has grown to the left, but
  //  basically it's like this:
  uint32 offsetnewread;
  offsetnewread=coffset+ads.getOffsetInAlignment(id);

  BUGIFTHROW(coffset > CON_counts.size(),"Something is wrong with coffset: " << coffset);
  BUGIFTHROW(ads.getOffsetInAlignment(id) > CON_counts.size(),"Something is wrong with that ADS!\n" << ads);
  BUGIFTHROW(offsetnewread>CON_counts.size(),"offsetnewread>CON_counts.size() ???");

  // Has the consensus grown to the left?
  int32 grownleft=0;
  if(ads.getOffsetInAlignment(id)==0 && ads.getOffsetInAlignment(-1)!=0){
    grownleft=-(coffset-ads.getOffsetInAlignment(-1));
    CEBUG("Grown left: " << grownleft << endl);

    // WRONG: BUGIFTHROW(grownleft<0, "grownleft <0 ???");
    // if grownleft <0 -> routine was not called with 'optimal' leftaligned ads
    // This can happen when the iteration in addRead() did not find the optimum
    //  and stoppped.
    // No big deal, this ain't an error.

    if(grownleft>0){
      // Yes, it has. So insert the needed positions into the 
      //  contig and ...
	
      CON_counts.insert(CON_counts.begin(), grownleft, CON_concounts_zero);

      // ... this new read is the (an) anchor in the contig, so offset=0 ...
      offsetnewread=0;

      // ... and adjust the beginning of the reads which are behind.
      for(uint32 k=0; k<CON_reads.size(); k++){
	CON_reads[k].offset+=grownleft;
	CEBUG("========" << CON_reads[k].offset);
      }

      // ... and adjust the consensus tags which are behind.
      for(uint32 k=0; k<CON_consensus_tags.size(); k++){
	CON_consensus_tags[k].from+=grownleft;
	CON_consensus_tags[k].to+=grownleft;
      }

    }else{
      grownleft=0;
    }
  }

#ifdef IRICTRACE
  cout << "postl1 "; cout.flush();
#endif

#ifdef CLOCK_STEPS
  cout << "\nirictimings 1: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

  // template for new read (initialise what we already know,
  //  except the read itself, because we don't want it to be copied
  //  twice :)
  contigread_t newread;
  newread.orpid=id;
  newread.direction=direction_frnid*direction_refid;
  newread.offset=offsetnewread;
  // newread.read=CON_readpool->getRead(id); // NO! we don't copy it twice

  // now insert the read into the readpool of the contig 
  CON_reads.push_back(newread);

#ifdef IRICTRACE
  cout << "postreadpush "; cout.flush();
#endif

// Superceded by Read strategy of moderate growth
//  // insertions and / or deletions may very well occur in that read 
//  //  due to other reads in the contig at that position.
//  // therefore, reserve some additional space in the read for these
//  //  insertions so that the internal containers don't double at the first 
//  //  insertion
//  // 
//  // for that, set the adjustment flag of the newread because reserve()
//  //  uses that too
//
//  {
//    if(!CON_readpool->getRead(id).usesAdjustments()){
//      CON_reads.back().read.disallowAdjustments();
//    }
//
//    // TODO: perhaps as parameter?
//#if CPP_READ_SEQTYPE_END != 5
//#error "This code is made for 5 sequencing types, adapt!"
//#endif
//
//    uint32 lenclipped=CON_readpool->getRead(id).getLenClippedSeq();
//    uint32 additionalreserve=0;
//    switch(CON_readpool->getRead(id).getSequencingType()){
//    case Read::SEQTYPE_SANGER : {
//      // 10% for Sanger
//      additionalreserve=lenclipped/10;
//      break;
//    }
//    case Read::SEQTYPE_454GS20 : {
//      // 20% for 454 (better safe than sorry for high coverage projects)
//      additionalreserve=lenclipped/15;
//      break;
//    }
//    case Read::SEQTYPE_PACBIO : {
//      // 20% for PacBio (strobed reads, dark strobe editing)
//      additionalreserve=lenclipped/10;
//      break;
//    }
//    case Read::SEQTYPE_SOLEXA : {
//      // 6% for Solexa as overwhelming majority has no indel
//      additionalreserve=lenclipped/15;
//    }
//    case Read::SEQTYPE_ABISOLID : {
//      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support c1.");
//    }
//    default: {
//      throw Notify(Notify::INTERNAL, THISFUNC, "Whoooooops? Unknown sequencing type?");
//    }
//    }
//
//    CON_reads.back().read.reserve(CON_readpool->getRead(id).getLenSeq()+additionalreserve);
//  }
  

#ifdef CLOCK_STEPS
  cout << "irictimings 2: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

  // ... and set the read object itself.
  CON_reads.back().read=CON_readpool->getRead(id);

#ifdef CLOCK_STEPS
  cout << "irictimings 3: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

  // add template if any
  if(CON_reads.back().read.getTemplateID()>=0){
    CON_templates_present.insert(CON_reads.back().read.getTemplateID());
  }

#ifdef IRICTRACE
  cout << "postti "; cout.flush();
#endif


  CEBUG("Offset of new read: " << offsetnewread << endl);
  CEBUG("Direction: " << newread.direction << endl);

  // get pointer to the aligned contig 
  const char * contigptr;
  const char * readptr;
  int32 indexincontig;
  int32 indexinread;

  // ... and advance it to the beginning of the overlap
  if(ads.getOffsetInAlignment(-1) == 0){
    contigptr= ads.getAlignedSequence(-1)+ads.getOffsetInAlignment(id);
    readptr= ads.getAlignedSequence(id);
    indexincontig= newread.offset;
    indexinread=0;
  }else{
    contigptr= ads.getAlignedSequence(-1);
    readptr= ads.getAlignedSequence(id)+ads.getOffsetInAlignment(-1);
    //WRONG! causes Heisenbug
    // indexincontig=newread.offset+ads.getOffsetInAlignment(-1);

    //also wrong
    //indexincontig=newread.offset;

    if(grownleft>0) {
      indexincontig=newread.offset+ads.getOffsetInAlignment(-1);
    }else{
      indexincontig=newread.offset;
    }

    indexinread=ads.getOffsetInAlignment(-1);
  }

  CEBUG("Index in read: " << indexinread << endl);
  CEBUG("Index in contig: " << indexincontig << endl);
  BUGIFTHROW(indexinread<0, "indexinread < 0 ?");
  BUGIFTHROW(indexincontig<0, "indexincontig < 0 ?");

  cccontainer_t::iterator ccI;

#ifdef CLOCK_STEPS
  cout << "irictimings 4: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

  // Now go through the aligned contig and read base per base
  //  and see whether there must be inserted a gap in either one

  {
#ifdef CLOCK_STEPS
    suseconds_t cs_ccinsert=0;
    suseconds_t cs_rloop=0;

    timeval us_loop;
#endif

    base_quality_t gapqual;

    for(int32 i=0; i<ads.getOverlapLen(); i++, contigptr++, readptr++, indexincontig++, indexinread++){
      CEBUG("i: " << i << "\ti_c: " << indexincontig);
      CEBUG("\ti_r: " << indexinread);
      CEBUG("\t*rptr: " << *readptr);
      CEBUG("\t*cptr: " << *contigptr);


      //BOUNDCHECK(indexincontig, 0, CON_counts.size()+1);
      
      if(indexincontig<0 || indexincontig >= CON_counts.size()+1) {
	cout << "indexincontig: " << indexincontig << endl;
	cout << "\nCON_counts.size()+1: " << CON_counts.size()+1;
	
	cout << "\n\n" << ads << endl;

	setCoutType(AS_CAF);
	cout << *this;
	setCoutType(AS_TEXT);
	cout << *this;
	throw Notify(Notify::INTERNAL, THISFUNC, "BOUNDCHECK error");
      }
      CEBUG(" CC: " << CON_counts[indexincontig].A << "\t" 
	    <<CON_counts[indexincontig].C << "\t" 
	    <<CON_counts[indexincontig].G << "\t" 
	    <<CON_counts[indexincontig].T << "\t" 
	    <<CON_counts[indexincontig].star << "\t" 
	    <<CON_counts[indexincontig].coverage << endl);


      // is there a gap in the contig?
      // TODO: wrong?
      if(*contigptr=='*'){
	// yes, is there also a gap in the read?
	BUGIFTHROW(*readptr=='*',"Both strands in ads have a '*' ?");
	CEBUG("\t* in con\n");

#ifdef CLOCK_STEPS
	gettimeofday(&us_loop,NULL);
#endif

	// luckily (well, this should really be the normal case), only
	//  the contig has a *
	// so we must insert this into the consensus counts
	ccI=CON_counts.begin();

	CEBUG("Increase CON_counts: " << CON_counts.size());
	BUGIFTHROW(indexincontig>CON_counts.size(),"indexincontig (" << indexincontig << ") > CON_counts.size() (" << CON_counts.size() << ") ???");
	advance(ccI, indexincontig);
	ccI=CON_counts.insert(ccI, CON_concounts_zero);

#ifdef CLOCK_STEPS
	cs_ccinsert+=diffsuseconds(us_loop);
	gettimeofday(&us_start,NULL);
#endif

	// CON_concounts_zero has 'N' as standard backbone char, change to *
	ccI->backbonechar='*';
	interpolateSRMValuesInCONcounts(ccI);

	CEBUG("\t" << CON_counts.size() << endl);
	
#ifdef IRICTRACE
	cout << "preuTBI "; cout.flush();
#endif
	// push up the consensus tags
	updateTagBaseInserted(indexincontig);

#ifdef IRICTRACE
	cout << "postuTBI "; cout.flush();
#endif

	// now, we must insert a * in all the reads at that are at this
	//  position, and update 'star' and 'coverage' counters
	// for that, go through all the reads (except this new inserted one,
	//  that's what the -1 is for as the new read has been inserted at
	//  the back) ...

#ifdef CLOCK_STEPS
	gettimeofday(&us_loop,NULL);
#endif
//#define CEBUG(bla)   {cout << bla; cout.flush();}

	vector<contigread_t>::iterator rI=CON_reads.begin();
	for(uint32 k=0; k<CON_reads.size()-1; k++, rI++) {

	  // ... and insert a * in those covering this position ...
	  if(indexincontig > rI->offset &&                   // TODO: >, >=?
	     indexincontig < rI->offset+rI->read.getLenClippedSeq()){ // checked: <!, not <=
	    CEBUG("Read id: " << rI->orpid);
	    CEBUG("\tfrom: " << rI->offset);
	    CEBUG("\tto: " << rI->offset+rI->read.getLenClippedSeq());
	    CEBUG("\tin!");
	    CEBUG(ccI->A << "\t" <<ccI->C << "\t" <<ccI->G << "\t" <<ccI->T << "\t" <<ccI->star << "\t" <<ccI->total_cov << endl);
	    ccI->star+=1;
	    ccI->total_cov+=1;
	    ccI->seqtype_cov[rI->read.getSequencingType()];
	    CEBUG(ccI->A << "\t" <<ccI->C << "\t" <<ccI->G << "\t" <<ccI->T << "\t" <<ccI->star << "\t" <<ccI->total_cov << endl);

	    int32 posinread;
	    posinread= indexincontig - rI->offset;
	    CEBUG("posinread: " << posinread);

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
	    // TODO: what are gapquals for PacBio???
	    if(rI->direction>0){
	      if(rI->read.isSequencingType(Read::SEQTYPE_454GS20)
		 || rI->read.isSequencingType(Read::SEQTYPE_IONTORRENT)) {
		gapqual=1;
	      }else{
		CEBUG("preqAQICS1 ");
		gapqual=rI->read.queryAverageQualInClippedSequence(posinread-1, posinread, false, true);
		CEBUG("postqAQICS2 ");
	      }
	      CEBUG("preqiBICS ");
	      rI->read.insertBaseInClippedSequence('*', gapqual, posinread, true);
	      CEBUG("postiBICS ");
	    }else{
	      if(rI->read.isSequencingType(Read::SEQTYPE_454GS20)
		 || rI->read.isSequencingType(Read::SEQTYPE_IONTORRENT)) {
		gapqual=1;
	      }else{
		CEBUG("preqAQICS2 ");
		gapqual=rI->read.queryAverageQualInClippedComplementSequence(posinread-1, posinread, false, true);
		CEBUG("postqAQICS2 ");
	      }
	      CEBUG("preqiBICCS ");
	      rI->read.insertBaseInClippedComplementSequence('*', gapqual, posinread, true);
	      CEBUG("postqiBICCS ");
	    }
	  }

//#define CEBUG(bla)
	  // ... and push the reads being behind this position one up
	  // >= tested
	  if(rI->offset >= indexincontig){
	    CEBUG("\tpushed up.");
	    rI->offset+=1;
	  }
	  CEBUG(endl);
	}

#ifdef IRICTRACE
	cout << "postfl1 "; cout.flush();
#endif

#ifdef CLOCK_STEPS
	cs_rloop+=diffsuseconds(us_loop);
#endif

	CEBUG(ccI->A << "\t" <<ccI->C << "\t" <<ccI->G << "\t" <<ccI->T << "\t" <<ccI->star << "\t" <<ccI->coverage << endl);
      }else{
	// No gap in the contig. Could there be one in the read?
	// If not, do nothing ...

	if(*readptr=='*'){
	  CEBUG("\t* in rea");
	  // if yes, we must insert a gap exactly in this read
	  int32 posinread;
	  vector<contigread_t>::iterator rI=CON_reads.end();
	  rI--;
	  posinread= indexincontig - rI->offset;

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

	  // TODO: what are gapquals for PacBio???
	  if(rI->direction>0){
	    if(rI->read.isSequencingType(Read::SEQTYPE_454GS20)
	      || rI->read.isSequencingType(Read::SEQTYPE_IONTORRENT)) {
	      gapqual=1;
	    }else{
	      gapqual=rI->read.queryAverageQualInClippedSequence(posinread-1, posinread, false, true);
	    }
	    rI->read.insertBaseInClippedSequence('*', gapqual, posinread, true);
	  }else{
	    if(rI->read.isSequencingType(Read::SEQTYPE_454GS20)
	       || rI->read.isSequencingType(Read::SEQTYPE_IONTORRENT)) {
	      gapqual=1;
	    }else{
	      gapqual=rI->read.queryAverageQualInClippedComplementSequence(posinread-1, posinread, false, true);
	    }
	    rI->read.insertBaseInClippedComplementSequence('*', gapqual, posinread, true);
	  }
	}
      }
      CEBUG("\n");
    }

#ifdef CLOCK_STEPS
    cout << "irictimings 4a ccinsert: " << cs_ccinsert << endl;    
    cout << "irictimings 4b rloop: " << cs_rloop << endl;    
#endif
  }

#ifdef CLOCK_STEPS
  cout << "irictimings 5: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

#ifdef IRICTRACE
  cout << "prex1 "; cout.flush();
#endif
  // good, now see if we have to add something to the right of the contig
  //  (happens when the read grows over the right of the contig)
  CEBUG("CON_reads.back().offset: " << CON_reads.back().offset);
  CEBUG("\tCON_reads.back().read.getLenClippedSeq(): " << CON_reads.back().read.getLenClippedSeq());
  CEBUG("\tCON_counts.size(): " << CON_counts.size() << endl);
  int32 grownright=CON_reads.back().offset
    +CON_reads.back().read.getLenClippedSeq()
    -CON_counts.size();
  if(grownright>0){
    CEBUG("Grown right: " << grownright << endl);
    CON_counts.resize(CON_counts.size()+grownright,CON_concounts_zero);
  }


#ifdef IRICTRACE
  cout << "prex2 "; cout.flush();
#endif

  vector<contigread_t>::iterator D= CON_reads.end();
  D--;
  int32 offset=D->offset;
  int32 len=D->read.getLenClippedSeq();
  vector<char>::const_iterator sI;
  if(D->direction > 0){
    sI=D->read.getClippedSeqIterator();
  }else{
    sI=D->read.getClippedComplementSeqIterator();
  }

#ifdef CLOCK_STEPS
  cout << "irictimings 6: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

#ifdef IRICTRACE
  cout << "preuCV "; cout.flush();
#endif

  updateCountVectors(offset,
		     len,
		     sI,
		     D->read.getSequencingType(),
		     true);

#ifdef IRICTRACE
  cout << "postuCV "; cout.flush();
#endif

#ifdef CLOCK_STEPS
  cout << "irictimings 7: " << diffsuseconds(us_start) << endl;
  gettimeofday(&us_start,NULL);
#endif

  return;
}
//#define CEBUG(bla)
//#define CEBUG(bla)

//#undef CLOCK_STEPS



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Contig::interpolateSRMValuesInCONcounts(cccontainer_t::iterator ccI)
{
  FUNCSTART("void Contig::interpolateSRMValuesInCONcounts(cccontainer_t::iterator ccI)");
  // We need to adjust the counts of short read mappings:
  //  take the mean of the positions left and right (if possible)
  // Also, the bbstrains must be guessed

  for(unsigned int actsrtype=0; actsrtype<2; actsrtype++){
    uint16 poslooked=0;
    ccctype_t bbcountnewpos=0;
    base_quality_t qual=0;
    // theoretically, the "if" should not be needed
    if(ccI != CON_counts.begin()){
      bbcountnewpos=(ccI-1)->bbcounts[actsrtype];
      qual=(ccI-1)->bbbestquals[actsrtype];
      poslooked++;
    }
    // theoretically, the "if" should not be needed
    if((ccI+1) != CON_counts.end()){
      bbcountnewpos+=(ccI+1)->bbcounts[actsrtype];
      qual+=(ccI+1)->bbbestquals[actsrtype];
      poslooked++;
    }
    if(poslooked){
      ccI->bbcounts[actsrtype]=bbcountnewpos/poslooked;
      ccI->bbbestquals[actsrtype]=qual/poslooked;
    }else{
      // should never happen (reads of size 1???), but anyway
      ccI->bbcounts[actsrtype]=0;
      throw Notify(Notify::INTERNAL, THISFUNC, "I'm on a branch I shouldn't be. Really!");
    }


    // theoretically, the "if" should not be needed
    if(ccI != CON_counts.begin()
       && (ccI+1) != CON_counts.end()){
      ccI->bbstrains[actsrtype]=((ccI-1)->bbstrains[actsrtype]) 
	& ((ccI+1)->bbstrains[actsrtype]);
    }


  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::updateReadTagsFromReadPool()
{
  FUNCSTART("void Contig::updateReadTagsFromReadPool()");

  vector<contigread_t>::const_iterator crI=CON_reads.begin();
  for(;crI != CON_reads.end(); crI++){
    if(crI->orpid<0 &&
       (crI->read.isRail() || crI->read.isBackbone())){
      BUGIFTHROW(crI->direction < 0, "Backbone or rail read in reverse direction ... rather unexpected. Really.");
      Read & rpread=CON_readpool->getRead(crI->orpid);
      BUGIFTHROW(crI->read.getName() != rpread.getName(),"different read names? not good.");
      for(uint32 tagnr=0; tagnr<rpread.getNumOfTags(); tagnr++){
	const multitag_t & acttagrpread=rpread.getTag(tagnr);
	if(acttagrpread.identifier == Read::REA_tagentry_idSRMr 
	   || acttagrpread.identifier == Read::REA_tagentry_idCRMr 
	   || acttagrpread.identifier == Read::REA_tagentry_idWRMr 
	   || acttagrpread.identifier == Read::REA_tagentry_idSAOr 
	   || acttagrpread.identifier == Read::REA_tagentry_idSROr
	   || acttagrpread.identifier == Read::REA_tagentry_idSIOr) {

	  //hier weiter


	}
      }
    }
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::initialiseBaseLocks()
{
  FUNCSTART("void Contig::initialiseBaseLocks()");

  cccontainer_t::iterator ccI=CON_counts.begin();
  for(; ccI != CON_counts.end(); ccI++){
    ccI->baselock=0;
    ccI->snplock=0;
  }

  vector<contigread_t>::const_iterator crI=CON_reads.begin();
  for(;crI != CON_reads.end(); crI++){
    updateBaseLocks(*crI,true);
  }

  FUNCEND();
}


/*************************************************************************
 *
 * When adding: transfers base locks from read tag into the CON_counts[].baselock
 * When removing: decreases CON_counts[].baselock where baselock tags are present in read
 *
 *************************************************************************/

void Contig::updateBaseLocks(const contigread_t & ric, bool addiftrue)
{

  FUNCSTART("void Contig::updateBaseLocks(const contigread_t & ric, bool addiftrue)");

  if(addiftrue){
    CEBUG("Adding base locks in " << ric.read.getName() << endl);
  }else{
    CEBUG("Removing base locks in " << ric.read.getName() << endl);
  }

  int32 one=1;
  if(addiftrue==false){
    one=-one;
  }

  for(int32 i=0; i<ric.read.getNumOfTags(); i++){
    multitag_t acttag=ric.read.getTag(i);
    bool baselock=false;
    bool snplock=false;
    for(int32 j=0; j<CON_baselock_ids.size(); j++){
      if(acttag.identifier==CON_baselock_ids[j]){
	baselock=true;
	break;
      }
    }
    for(int32 j=0; j<CON_snplock_ids.size(); j++){
      if(acttag.identifier==CON_snplock_ids[j]){
	snplock=true;
	break;
      }
    }

    if(baselock || snplock){
      CEBUG("Tag #" << i << "\tfrom: " << acttag.from << "\tto: " << acttag.to << "\t" << acttag.identifier);
      int32 contigpos=unclippedReadPosToContigPos(acttag.from, ric);
      CEBUG("\tconpos: " << contigpos << endl);
      // no, bad, it _might_ really be outside the contig
      //  e.g. if tags are in clipped part
      //BOUNDCHECK(contigpos, 0, CON_counts.size());
      if(contigpos>=0 && contigpos<CON_counts.size()){
	cccontainer_t::iterator C=CON_counts.begin();
	advance(C, contigpos);
	for(int32 j=0; j<=acttag.to-acttag.from; j++, contigpos++, C++){
	  // We mind only for the parts of the lock in the used read
	  if(contigpos>=ric.offset && contigpos < ric.offset+ric.read.getLenClippedSeq()){
	    if(baselock) C->baselock+=one;
	    if(snplock) C->snplock+=one;
	  }
	}
      }
    }
  }

  FUNCEND();

}


/*************************************************************************
 *
 * update the count vectors
 * the CON_counts vector mus have enough space to contain the updated seq
 * from    is a position in the contig
 * len     is the length of the updated portion
 * updateI is the sequence which updates into the contig
 * seqtype is the seqtype of the read that is added
 *
 * Does NOT update the baselock! This is done in updateBaseLocks
 *
 *************************************************************************/

void Contig::updateCountVectors(const int32 from, const int32 len, vector<char>::const_iterator updateI, const uint32 seqtype, const bool addiftrue)
{

  FUNCSTART("void Contig::updateCountVectors(uint32 from, uint32 len, vector<char>::const_iterator updateI, bool addiftrue)");

  CEBUG("From: "<<from<<"\tLen: "<<len<<"\tCON_counts.size():" << CON_counts.size() << endl);


  if(from<0){
    throw Notify(Notify::INTERNAL, THISFUNC, "from <0 ?");
  }

  if(from+len>CON_counts.size()){
    cout << "Error:\n";
    cout << "from: " << from << endl;
    cout << "len: " << len << endl;
    cout << "size of contig: " << CON_counts.size() << endl;
    throw Notify(Notify::INTERNAL, THISFUNC, "from + len > size of contig?");
  }

  cccontainer_t::iterator I= CON_counts.begin();
  BOUNDCHECK(from, 0, CON_counts.size());
  advance(I, from);

  int32 one=1;
  int32 two=2;
  int32 four=4;

  if(addiftrue==false){
    one=-one;
    four=-four;
  }

  for(int32 i=0; i<len; i++, I++, updateI++){
    char thechar=*updateI;
    switch(toupper(thechar)){
    case '-':
    case 'N': {
      I->A+=one;
      I->C+=one;
      I->G+=one;
      I->T+=one;
      I->N+=one;
      break;
    }
    case 'X': {I->X+=one; break;}
    case 'A': {I->A+=four; break;}
    case 'C': {I->C+=four; break;}
    case 'G': {I->G+=four; break;}
    case 'T': {I->T+=four; break;}

    case 'M': {I->A+=two; I->C+=two; break;}
    case 'R': {I->A+=two; I->G+=two; break;}
    case 'W': {I->A+=two; I->T+=two; break;}
    case 'S': {I->C+=two; I->G+=two; break;}
    case 'Y': {I->C+=two; I->T+=two; break;}
    case 'K': {I->G+=two; I->T+=two; break;}

    case 'V': {I->A+=one; I->C+=one; I->G+=one; break;}
    case 'H': {I->A+=one; I->C+=one; I->T+=one; break;}
    case 'D': {I->A+=one; I->G+=one; I->T+=one; break;}
    case 'B': {I->C+=one; I->G+=one; I->T+=one; break;}
      

    case '*': {I->star+=one; break;}
    default: {
      cout << "WHY? Illegal char: " << (uint16) thechar << " >>" << thechar << "<<\n";
      throw Notify(Notify::FATAL, THISFUNC, "Unexpected base.");
    }
    }
    I->total_cov+=one;
    I->seqtype_cov[seqtype]+=one;
  }
  
  FUNCEND();
}


/*************************************************************************
 *
 * makes a temporary char * of a portion of the consensus
 * from and to positions in the consensus
 * from inclusive
 * to exclusive
 * appending a 0 char
 *
 * returns whether an N or X has been put in the tmp consensus
 *
 *************************************************************************/

bool Contig::makeTmpConsensus(int32 from, int32 to)
{
//#define CEBUG(bla)   {cout << bla; cout.flush();}

  FUNCSTART("void Contig::makeTmpConsensus(int32 from, int32 to)");

  definalise();
  CEBUG("\nFrom: " << from << "\tTo: " << to<<"\tCON_counts.size(): "<<CON_counts.size());

  BUGIFTHROW(from>to,"from>to?");
  BUGIFTHROW(from<0, "from < 0 ?");

  if(to>CON_counts.size()) to=CON_counts.size();
  uint32 len_tmpcons=to-from;

  if(CON_2tmpcons.capacity()<2000 || CON_2tmpcons.capacity()<len_tmpcons){
    CON_2tmpcons.reserve(max(len_tmpcons,static_cast<uint32>(2000)));
  }
  CON_2tmpcons.resize(len_tmpcons);

  bool hasNorX=false;
  {
    string::iterator toptr=CON_2tmpcons.begin();

    // using a pointer to CON_counts is (a lot) faster than accessing it 
    //  via [] each time
    cccontainer_t::iterator ccI=CON_counts.begin();
    BOUNDCHECK(from, 0, CON_counts.size()+1);
    advance(ccI, from);
    
    for(uint32 i=from; i<to ;i++, toptr++, ccI++){
      *toptr=0;
      if(CON_tmpcons_from_backbone && ccI->backbonechar!='@'){
	*toptr=ccI->backbonechar;
	if(toupper(*toptr)=='N' || toupper(*toptr)=='X'){
	  hasNorX=true;
	}
      }else{
	ccctype_t maximum= max(ccI->A, max(ccI->C, max(ccI->G, ccI->T)));
	uint8 counts=0;
	//CEBUGF(ccI->A << "\t" << ccI->C << "\t" << ccI->G << "\t" << ccI->T << "\t" << ccI->N << "\t" << ccI->star << "\n"); 
      
	// is any ACGT set?
	if(maximum >0 && maximum > ccI->star) {
#if 0
// Somehow U13 does not like this, recheck
	  if(ccI->A==maximum){
	    counts++;
	    *toptr='A';
	  }
	  if(ccI->C==maximum){
	    counts++;
	    if(counts>1){
	      *toptr=dptools::calcIUPACConsensus(*toptr,'C');
	    }else{
	      *toptr='C';
	    }
	  }
	  if(ccI->G==maximum){
	    counts++;
	    if(counts>1){
	      *toptr=dptools::calcIUPACConsensus(*toptr,'G');
	    }else{
	      *toptr='G';
	    }
	  }
	  if(ccI->T==maximum){
	    counts++;
	    if(counts>1){
	      *toptr=dptools::calcIUPACConsensus(*toptr,'T');
	    }else{
	      *toptr='T';
	    }
	  }
	  //if(counts>1) *toptr='N';
#else
	  if(ccI->A==maximum){
	    counts++;
	    *toptr='A';
	  }
	  if(ccI->C==maximum){
	    counts++;
	    *toptr='C';
	  }
	  if(ccI->G==maximum){
	    counts++;
	    *toptr='G';
	  }
	  if(ccI->T==maximum){
	    counts++;
	    *toptr='T';
	  }
	  if(counts>1) {
	    *toptr='N';
	    hasNorX=true;
	  }
#endif
	  //// can be somewhat problematic with 454 data
	  //// calls the base until 50/50, then the gap
	  //if(maximum/4 < ccI->star) *toptr='*';

	  // this prefers to call gaps
	  // calls the base until 1/3 base, 2/3 gap, then the gap
	  // this should help, together with the "expected gap" # in
	  //  alignments, to further reduce to a maximum this kind of 
	  //  base jiggling in homopolymers
	  //
	  //     ...*AAAAAAAAA...
	  //     ...*AAAAAAAAA...
	  //     ...AAAAAAAAA*...
	  //     ...AAAAAAAAA*...
	  //     ...AAAAAAAAA*...
	  //     ...*AAAAAAAAA...
	  //     ...*AAAAAAAAA...
	  //     ...AAAAAAAAA*...
	  //     ...*AAAAAAAAA...

	  if(maximum/4 < (ccI->star)*2) {
	    switch(*toptr){
	    case 'A': {
	      *toptr='1';
	      break;
	    }
	    case 'C': {
	      *toptr='2';
	      break;
	    }
	    case 'G': {
	      *toptr='3';
	      break;
	    }
	    case 'T': {
	      *toptr='4';
	      break;
	    }
	    default: {
	      *toptr='*';
	    }
	    }
	  }
	} else {
	  if((ccI->star >= ccI->X)
	     && (ccI->star >= ccI->X)){
	    *toptr='*';
	  } else if(ccI->N){
	    *toptr='N';
	    hasNorX=true;
	  }else{
	    *toptr='X';
	    hasNorX=true;
	  }
	}
      }
      if(*toptr==0){
	MIRANOTIFY(Notify::INTERNAL,"Ooooops? makeTmpConsensus encountered the unexpected situation of an uncalled base? Please contact the author immediately.");
      }
    }
    // not needed for string, terminated by 0 anyway
    //*(toptr)=0;
  }
  CEBUG("Tmp_cons: >>>" <<CON_2tmpcons << "<<<" << endl);

  FUNCEND();

  return hasNorX;
}
//#define CEBUG(bla)



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::finalise()
{
  FUNCSTART("void Contig::finalise()");

  //checkContig();

  if(CON_finalised==false){
    makeTmpConsensus(0, CON_counts.size());
    
    CON_outputorder.resize(CON_reads.size());

    CEBUG("finalise sizes: " <<CON_reads.size() << " " <<CON_outputorder.size() << endl) 

    CEBUG("offset\tindex\tread_id\tt_id\ttempl\n");
    for(uint32 i=0; i<CON_reads.size(); i++){
      CON_outputorder[i].offset_start=CON_reads[i].offset;
      CON_outputorder[i].original_index=i;
      CON_outputorder[i].orpid=CON_reads[i].orpid;
      
      CEBUG(CON_reads[i].offset);
      CEBUG("\t" << i);
      CEBUG("\t" << CON_reads[i].id);
      CEBUG("\t" << CON_reads[i].read.getTemplateID());
      //CEBUG("\t" << CON_reads[i].read.getTemplate());
      CEBUG("\n");
      BUGIFTHROW(CON_outputorder[i].offset_start<0, "offset_start < 0 ?");

    }
    
    BUGIFTHROW(CON_outputorder.size()!=CON_reads.size(), "CON_outputorder.size()!=CON_reads.size()???");

    sort(CON_outputorder.begin(), CON_outputorder.end(), Contig::out_order_t_comparator);
    
    CON_finalised=true;
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::deleteRead(int32 id)
{
  //TODO: What happens, when a contig breaks? (no more coverage in the middle)

  FUNCSTART("void Contig::deleteRead(uint32 id)");

  if(CON_fixedconsseq.size()){
    nukeSTLContainer(CON_fixedconsseq);
    nukeSTLContainer(CON_fixedconsqual);
  }

  CEBUG("Deleting: " << id << "\nContig before:\n");
  CEBUG(*this);

  definalise();

  paranoiaBUGSTAT(checkContig());

  vector<contigread_t>::iterator D= CON_reads.begin();

  if(CON_maprpids_to_conreadsids.empty()){
    uint32 i;
    for(i=0; i<CON_reads.size(); i++, D++){
      if(D->orpid==id) break;
    }
    if(i==CON_reads.size()){
      throw Notify(Notify::INTERNAL, THISFUNC, "ID not in contig.");
    }
  }else{
    advance(D,CON_maprpids_to_conreadsids[id]);
  }

  // Remove the base locks in CON_counts that this read produced
  updateBaseLocks(*D, false);

  // Remove the read from the CON_counts
  int32 offset=D->offset;
  int32 len=D->read.getLenClippedSeq();
  vector<char>::const_iterator sI;
  if(D->direction > 0){
    sI=D->read.getClippedSeqIterator();
  }else{
    sI=D->read.getClippedComplementSeqIterator();
  }

  CEBUG("Offset: " << offset);
  CEBUG("\nlen: " << len);

  updateCountVectors(offset,
		     len,
		     sI,
		     D->read.getSequencingType(),
		     false);

  CEBUG(*this);


  int32 frontdeletions=0;
  int32 enddeletions=0;
  {
    // Readjust the ends of the contig if the removed read was there

    // Readjust front
    //   readjust the offsets if needed
    int32 maxchecklen=len;
    BUGIFTHROW(maxchecklen<0, "front: maxchecklen < 0 ?");
    if(maxchecklen>0){
      if(D->offset==0){
	cccontainer_t::iterator I= CON_counts.begin();
	if(I->total_cov==0){
	  CEBUG("CONTIG-- delread at beginning!\n");
	  while(I->total_cov==0 &&
		frontdeletions<maxchecklen){
	    frontdeletions++;
	    I++;
	  }
	  if(frontdeletions>0){
	    // ok, we've got to delete bases at the front of the consensus
	    CON_counts.erase(CON_counts.begin(), I);
	    CEBUG("Deleted << " << frontdeletions << " at front.\n");

	    // push consensus tags down, deleting all those which are completely 
	    //  in the deleted part, adjusting others.
	    {
	      vector<consensustag_t>::iterator ctI=CON_consensus_tags.begin();
	      while(ctI!=CON_consensus_tags.end()){
		ctI->from-=frontdeletions;
		ctI->to-=frontdeletions;
		if(ctI->to < 0){
		  ctI=CON_consensus_tags.erase(ctI);
		}else{
		  if(ctI->from < 0) ctI->from=0;
		  ctI++;
		}
	      }
	    }

	    // push down the offsets of the other reads
	    for(uint32 i=0; i<CON_reads.size(); i++){
	      // Safetycheck
	      if(CON_reads[i].offset<frontdeletions){
		if(CON_reads[i].orpid!=id){
		  cout << "i: " << i << "\tfrontdeletions: " << frontdeletions << "\treads[i].offset: " << CON_reads[i].offset << "\n";
		  throw Notify(Notify::INTERNAL, THISFUNC, "Offset wrong: too low.");
		}
	      }
	      if(CON_reads[i].orpid!=id) CON_reads[i].offset-=frontdeletions;
	    }
	  }
	}
      }
    }
    
    BUGIFTHROW(frontdeletions<0, "frontdeletions < 0 ?");

    if(offset>0){
      offset=offset-frontdeletions;
      if(offset<0){
	throw Notify(Notify::INTERNAL, THISFUNC, "offset >0 and offset-frontdeletions < 0 ?");
      }
    }
    
    BUGIFTHROW(offset<0, "offset < 0 ?");

    // Readjust end
    {
      maxchecklen=len-frontdeletions;
      BUGIFTHROW(maxchecklen<0, "end: maxchecklen< 0 ?");
      if(maxchecklen>0){
	cccontainer_t::iterator I= CON_counts.end();
	I--;
	if(I->total_cov==0){
	  CEBUG("CONTIG-- delread at end!\n");
	  while(I->total_cov==0 &&
		enddeletions<maxchecklen){
	    CEBUG(".");
	    enddeletions++;
	    I--;
	  }
	  if(enddeletions>0){
	    I++;
	    CEBUG("Counts.size() before: " << CON_counts.size() << endl);
	    CON_counts.erase(I,CON_counts.end());
	    CEBUG("Counts.size() after: " << CON_counts.size() << endl);
	    CEBUG("Deleted << " << enddeletions << " at end.\n");

	    // and delete consensus tags there
	    int32 newsize=CON_counts.size()-1;
	    vector<consensustag_t>::iterator ctI=CON_consensus_tags.begin();
	    while(ctI!=CON_consensus_tags.end()){
	      if((ctI->from > newsize)){
		ctI=CON_consensus_tags.erase(ctI);
	      }else{
		if(ctI->to > newsize) ctI->to=newsize;
		ctI++;
	      }
	    }
	  }
	}
      }
    }
  }

  BUGIFTHROW(enddeletions<0, "enddeletions < 0 ?");

  // remove template if any
  {
    int32 tid=D->read.getTemplateID();
    if(tid>=0){
      multiset<int32>::iterator msI=CON_templates_present.find(tid);
      if(msI==CON_templates_present.end()){
	throw Notify(Notify::INTERNAL, THISFUNC, "Template not present in list though read has one?");
      }
      CON_templates_present.erase(msI);
    }
  }

  // Remove read
  D=CON_reads.erase(D);

  // remove mapping
  if(!CON_maprpids_to_conreadsids.empty()){
    // instead of clear, recalc the maps for the reads after
    // the deleted one
    //CON_maprpids_to_conreadsids.clear();

    int32 conreadid=CON_maprpids_to_conreadsids[id];
    CON_maprpids_to_conreadsids[id]=-1;
    for(; D != CON_reads.end(); D++, conreadid++){
      CON_maprpids_to_conreadsids[D->orpid]=conreadid;
    }
  }

#if 1
  deleteStarOnlyColumns(offset, offset+len-frontdeletions-enddeletions);
#endif 

  paranoiaBUGSTAT(checkContig());
  
  CEBUG("Contig after:\n" << *this);

  FUNCEND();

}






/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::checkContig()
{
#ifdef PARANOIABUGTRACK
  FUNCSTART("void Contig::checkContig()");

  // TODO: ich muss, da ansonsten nicht als CAF abspeicherbar
  //         (erlaube ich 0 lngenreads oder nicht?)

  uint32 i;
  vector<contigread_t>::iterator D= CON_reads.begin();
  for(i=0; i<CON_reads.size(); i++, D++){
    if(D->offset<0){
      throw Notify(Notify::INTERNAL, THISFUNC, "offset < 0?");
    }
    if(!CON_counts.empty()){
      if(D->offset>=static_cast<int32>(CON_counts.size())){
	cout << "Error. Contig_id: " << CON_id << endl;
	cout << "First read: " << CON_reads[0].read;
	throw Notify(Notify::INTERNAL, THISFUNC, "offset >= contigsize?");
      }
      if(D->offset+D->read.getLenClippedSeq()>CON_counts.size()){
	throw Notify(Notify::INTERNAL, THISFUNC, "offset+readlen > contigsize?");
      }
    }
  }

  if(CON_miraparams->size() != Contig::SEQTYPE_END){
    throw Notify(Notify::INTERNAL, THISFUNC, "CON_miraparams->size() != Contig::SEQTYPE_END");
  }

  //#ifndef CEBUGFLAG
#if 1
//  if(CON_counts.size()>0 &&
//     CON_counts.back().A == 0 &&
//     CON_counts.back().C == 0 &&
//     CON_counts.back().G == 0 &&
//     CON_counts.back().T == 0
//     ){
//    throw Notify(Notify::INTERNAL, THISFUNC, "0-0-0-0 N at end?");
//  }
  if(!CON_counts.empty() &&
     CON_counts.front().coverage == 0
     ){
    throw Notify(Notify::INTERNAL, THISFUNC, "0-0-0-0-0 N (zero coverage) at front?");
  }
  if(!CON_counts.empty() &&
     CON_counts.back().coverage == 0
     ){
    throw Notify(Notify::INTERNAL, THISFUNC, "0-0-0-0-0 N (zero coverage) at end?");
  }
#endif

#ifdef CEBUGFLAG
  {
    cccontainer_t::iterator I=CON_counts.begin();
    while(I!=CON_counts.end()){
      if(I->A + I->C + I->G + I->T + (I->star)*4 != (I->coverage)*4){
	throw Notify(Notify::INTERNAL, THISFUNC, "Gna, error in Con_counts :/.");
      }
//	if(I->star>10){
//	  throw Notify(Notify::INTERNAL, THISFUNC, "Gna, suspect coverage:/.");
//	}
      I++;
    }
  }
#endif

  FUNCEND();
#endif
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// TODO Fehler, wenn clips sich gegenber dem original verschieben

//#define CEBUG(bla)   {cout << bla; cout.flush();}

void Contig::initialiseContig(uint32 contig_length,
			      const list<contig_init_read_t> & rlist, 
			      vector<tag_t> & tags,
			      const string & contigname,
			      string & fixedseq,
			      vector<base_quality_t> & fixedqual)
{
  FUNCSTART("void Contig::initialiseContig(uint32 contig_length, const list<contig_init_read_t> & rlist, vector<tag_t> & tags)");

  discard();

  CON_name=contigname;
  CON_fixedconsseq=fixedseq;
  CON_fixedconsqual=fixedqual;

  CEBUG("::initC1 fixed " << CON_fixedconsseq.size() << endl);

  CON_counts.resize(contig_length, CON_concounts_zero);

  CEBUG("Contig ini-size: " << contig_length << endl);

  // make space for reads
  CON_reads.reserve(rlist.size()+5);

  //cout << "KKKKKKKKK " << CON_readpool->getNumOfStrainInReadpool() << endl;
  // ... and strain tracking
  BUGIFTHROW(CON_readpool->getNumOfStrainInReadpool()==0,"CON_readpool->getNumOfStrainInReadpool()==0 ???");
  CON_readsperstrain.resize(CON_readpool->getNumOfStrainInReadpool(),0);

  // Correct for weird things:
  // 1) joining contigs in gap5 inserts gap bases having a quality of 0, leading to the possibility
  //    that a single wrong bases overrules hundred of valid gaps.
  //    Counterstrategy: in reads, re-assign quality to gap bases with quality 0
  // 2) no "two" atm

  {
    CEBUG("Fixing zero-gap qualities\n");
    list<contig_init_read_t>::const_iterator rlI=rlist.begin();
    for(;rlI!=rlist.end();rlI++){
      CON_readpool->getRead(rlI->id).fixZeroGapQuals();
    }
  }


  // 
  CEBUG("Going through list (size " << rlist.size() << "):\n");
  list<contig_init_read_t>::const_iterator rlI=rlist.begin();
  for(;rlI!=rlist.end();rlI++){
    
    CEBUG("New element:\t");
    {
      contigread_t tmp;
      
      tmp.offset=rlI->offset_in_contig;
      tmp.orpid=rlI->id;
      tmp.direction=rlI->direction;
      
      CON_reads.push_back(tmp);
      
      CON_reads.back().read=CON_readpool->getRead(tmp.orpid);

      CEBUG("Offset: " << tmp.offset << "\tID: " << tmp.orpid);
      CEBUG("\tDir: " << tmp.direction << endl);
      CEBUG("Read:\n" << CON_reads.back().read);
    }

    // track strain IDs
    ++CON_readsperstrain[CON_reads.back().read.getStrainID()];

    // TODO: check this. Took it out because when assembling a project
    //  without base qualities, then doing a convert_project from caf to caf,
    //  all the reads got a base quality of 0
    //
    // what would this setting of base qualities be for anyway? what did I
    //  have in mind?
    //
    //// set default quality if read has no qual
    //if(!CON_reads.back().read.hasQuality()){
    //  CON_reads.back().read.setQualities((*CON_miraparams)[0].getAssemblyParams().as_backbone_basequals);
    //}

    // add template if any
    if(CON_reads.back().read.getTemplateID()>=0){
      CON_templates_present.insert(CON_reads.back().read.getTemplateID());
    }

    //CEBUG("Setting clipoffs: " << rlI->read_lclip << "\t" << rlI->read_rclip << endl);
    //CON_reads.back().read.setClipoffs(rlI->read_lclip, rlI->read_rclip, true);
    //    CEBUG("done. Read is now:\n" << CON_reads.back().read);

    try{
      if(rlI->direction>0){
	updateCountVectors(rlI->offset_in_contig,
			   CON_reads.back().read.getLenClippedSeq(),
			   CON_readpool->getRead(rlI->id).getClippedSeqIterator(),
			   CON_readpool->getRead(rlI->id).getSequencingType(),
			   true);
      }else{
	updateCountVectors(rlI->offset_in_contig,
			   CON_reads.back().read.getLenClippedSeq(),
			   CON_readpool->getRead(rlI->id).getClippedComplementSeqIterator(),
			   CON_readpool->getRead(rlI->id).getSequencingType(),
			   true);
      }
    }
    catch(Notify n){
      cout << "Readpool id: " << rlI->id << endl;
      cout << "Error while adding this read (1):\n" << CON_reads.back().read;
      cout << "In this contig (output might crash):" << endl << *this << endl;
      n.handleError(THISFUNC);
    }


    // initialise backbone and short read mapping structures if this read was used as a
    //  backbone
    if(CON_reads.back().read.isBackbone()){
      cccontainer_t::iterator ccI=CON_counts.begin();
      advance(ccI,CON_reads.back().offset);
      const char * consb=CON_reads.back().read.getSeqAsChar();
      for(uint32 ci=0; ci<CON_reads.back().read.getLenClippedSeq(); ccI++, consb++, ++ci){
	BUGIFTHROW(ccI == CON_counts.end(), "ccI == CON_counts.end() ???");
	BUGIFTHROW(*consb == 0, "*consb == 0 ?");
	// Don't init bbcounts[], bbbestquals[] and bbstrains[] as they
	//  might already have been initialised by CER reads ... well, 
	//  will be when this functionality is implemented.
	// So, only initialise backbone chars
	ccI->backbonechar=toupper(*consb);
      }
    }
  }

  trimEnds();
    
  // must copy each tag as we have to "upgrade" a tag_t to a consensustag_t
  {
    for(uint32 i=0;i!=tags.size();i++){
      CON_consensus_tags.push_back(tags[i]);
      //cout << "copied ";
      //tags[i] << endl;
    }
  }

  CEBUG(*this);
  
  try{
    checkContig();
  }
  catch(Notify n){
    n.handleError(THISFUNC);
  }

  definalise();

  if(CON_fixedconsseq.size() != CON_fixedconsqual.size()
     || CON_fixedconsseq.empty()
     || CON_fixedconsqual.empty()){
    string dummy1;
    vector<base_quality_t> dummy2;
    newConsensusGet(dummy1, dummy2);
  }

  CEBUG("::initC2 fixed " << CON_fixedconsqual.size() << endl);

  FUNCEND();
}
//#define CEBUG(bla)



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::trimEnds()
{
  FUNCSTART("void Contig::trimEnds()");

  // trim end of contig if needed
  while(!CON_counts.empty() &&
	CON_counts.back().total_cov == 0
	){
    CON_counts.pop_back();
  }

  // trim start of contig if needed
  while(!CON_counts.empty() &&
	CON_counts.back().total_cov == 0
	){
    CON_counts.pop_front();
  }

  definalise();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

vector<int32> &
Contig::getCRIndicesAtContigPosition(vector<int32> & vec, int32 pos1, int32 pos2) const
{
  FUNCSTART("vector<int32> & getReadIDsAtContigPosition(vector<int32> & vec, int32 pos1, int32 pos2) const");

  BUGIFTHROW(pos1 < 0, "pos1 < 0?");
  BUGIFTHROW(pos1 >= static_cast<int32>(CON_counts.size()), "pos1 > size of contig?");

  if(pos2==-1) pos2=pos1;

  BUGIFTHROW(pos2 < pos1, "pos2 < pos1");

  if(pos2 > static_cast<int32>(CON_counts.size())) pos2=static_cast<int32>(CON_counts.size()-1);

  vec.clear();
  vec.reserve(100);

  int32 i=0;
  vector<contigread_t>::const_iterator I= CON_reads.begin();

  for(I= CON_reads.begin(); I != CON_reads.end(); I++, i++){
    if(I->offset > pos2 
       || static_cast<int32>(I->offset+I->read.getLenClippedSeq()) <= pos1) continue;
    vec.push_back(i);
  }

  FUNCEND();

  return vec;
}




/*************************************************************************
 *
 * Returns the original readpool-IDs of the reads between contig positions
 *  pos1 and pos2
 *
 *************************************************************************/

vector<int32> &
Contig::getReadIDsAtContigPosition(vector<int32> & vec, int32 pos1, int32 pos2) const
{
  FUNCSTART("vector<int32> & getReadIDsAtContigPosition(vector<int32> & vec, int32 pos1, int32 pos2) const");

  //cout << "getReadIDsAtContigPosition(): search between: " << pos1 << "\t" << pos2 << '\n';

  BUGIFTHROW(pos1 < 0, "pos1 < 0?");
  BUGIFTHROW(pos1 >= static_cast<int32>(CON_counts.size()), "pos1 (" << pos1 << ") >= size of contig (" << CON_counts.size() << ")?");

  if(pos2==-1) pos2=pos1;

  BUGIFTHROW(pos2 < pos1, "pos2 < pos1");

  if(pos2 > static_cast<int32>(CON_counts.size())) pos2=static_cast<int32>(CON_counts.size()-1);

  vec.clear();

  vector<contigread_t>::const_iterator I;

  for(I= CON_reads.begin(); I != CON_reads.end(); I++){
    if(I->offset > pos2 
       || static_cast<int32>(I->offset+I->read.getLenClippedSeq()) < pos1) continue;
    if(I->orpid>=0) vec.push_back(I->orpid);
  }

  FUNCEND();

  //cout << "Searched between: " << pos1 << "\t" << pos2 << "\nFound: " << vec.size();
  //cout << endl;
  //for(uint32 i=0; i<vec.size(); i++) {
  //  cout << vec[i] << endl;
  //}

  return vec;
}






/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::deleteBaseInRead(int32 contigposition, int32 readindex)
{
  FUNCSTART("void Contig::deleteBaseInRead(int32 contigposition, int32 readindex)");

  paranoiaBUGSTAT(checkContig());

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  contigread_t & theread=CON_reads[readindex];

  BUGIFTHROW((contigposition < theread.offset ||
	      contigposition >= theread.offset+theread.read.getLenClippedSeq()),
	     "Read does not cover given contigposition");

  definalise();

  int32 posinread=contigposition-theread.offset;
  int32 updatelen=theread.read.getLenClippedSeq()-posinread;
  
  CEBUG("contigpos: " << contigposition);
  CEBUG("\nPPPosinread: " << posinread);
  CEBUG("\nupdatelen: " << updatelen<<endl);

  bool deletein=false;
  bool deleteend=false;

  // now only clip contig when size is affected, never ever at all move
  // other reads around ---> this may lead to a contig break :(
  if(theread.offset+theread.read.getLenClippedSeq()==CON_counts.size()
     && CON_counts.back().total_cov==1) deleteend=true;

  CEBUG("deletein: " << deletein << endl);
  CEBUG("deleteend: " << deleteend << endl);

  vector<char>::const_iterator I;

  if(theread.direction>0){
    I=theread.read.getClippedSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq());
    advance(I, posinread);
    updateCountVectors(contigposition, 
		       updatelen, 
		       I,
		       theread.read.getSequencingType(),
		       false);
    theread.read.deleteBaseFromClippedSequence(posinread);
    I=theread.read.getClippedSeqIterator();
  }else{
    I=theread.read.getClippedComplementSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq());
    advance(I, posinread);
    updateCountVectors(contigposition, 
		       updatelen, 
		       I, 
		       theread.read.getSequencingType(),
		       false);
    theread.read.deleteBaseFromClippedComplementSequence(posinread);
    I=theread.read.getClippedComplementSeqIterator();
  }

  if(deletein==true){
    cccontainer_t::iterator cI=CON_counts.begin();
    BOUNDCHECK(contigposition, 0, CON_counts.size());
    advance(cI, contigposition);
    CON_counts.erase(cI);
    for(uint32 i=0; i<CON_reads.size(); i++){
      if(CON_reads[i].offset>contigposition) CON_reads[i].offset--;
    }
    // adjust tags in consensus
    // TODO: untested (ok?)
    updateTagBaseDeleted(contigposition);
  }
  if(deleteend==true){
    CON_counts.pop_back();
    // adjust tags in consensus
    // TODO: untested (ok?)
    updateTagBaseDeleted(contigposition);
  }

  // BOUNDS checked above
  advance(I, posinread);
  updateCountVectors(contigposition, 
		     updatelen-1, 
		     I, 
		     theread.read.getSequencingType(),
		     true);

  paranoiaBUGSTAT(checkContig());

//  if(contigposition==4711){
//    cout << *this;
//  }
  
  FUNCEND();

}



/*************************************************************************
 *
 * Adds/Subtracts the offset to the offset of the reads in the contig
 *  which occur _after_ contigposition
 *
 *************************************************************************/

void Contig::adjustReadOffsets(int32 contigposition, int32 offset)
{
  FUNCSTART("void Contig::adjustReadOffsets(int32 contigposition, int32 offset)");

  //  if(contigposition==3713) cout << *this;

  paranoiaBUGSTAT(checkContig());

  BUGIFTHROW(contigposition<0,"contigposition<0 ?");

  //  bool resizeit=false;
  
  if(offset!=0){
    uint32 maxsize=0;
    for(uint32 i=0; i<CON_reads.size(); i++){
      if(CON_reads[i].offset>contigposition) {
	CON_reads[i].offset+=offset;
      }
      if(CON_reads[i].offset+CON_reads[i].read.getLenClippedSeq()>maxsize) maxsize=CON_reads[i].offset+CON_reads[i].read.getLenClippedSeq();
    }
    if(maxsize!=CON_counts.size()) CON_counts.resize(maxsize);
    rebuildConCounts();
  }

  //if(contigposition==3713) cout << *this;

  paranoiaBUGSTAT(checkContig());

  FUNCEND();

}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::rebuildConCounts()
{
  FUNCSTART("void Contig::rebuildConsensus()");
  
  // using a pointer to CON_counts is (a lot) faster than accessing it 
  //  via [] each time
  cccontainer_t::iterator ccI=CON_counts.begin();
  for(;ccI!=CON_counts.end();ccI++){
    ccI->A=0;             // ACGT are extended counters
    ccI->C=0;             // ccctype_t is enough for a coverage of 16k reads.
    ccI->G=0;
    ccI->T=0;
    ccI->N=0;             // N, X, star and coverage are normal counters.
    ccI->X=0;
    ccI->star=0;
    ccI->total_cov=0;
    ccI->baselock=0;     // if > 0 then only one of A, C, T, G may be set, otherwise it's a misassembly
    ccI->snplock=0;
    for(uint32 i=0; i<Read::SEQTYPE_END; ++i){
      ccI->seqtype_cov[i]=0;
    };
  }
  
  for(uint32 i=0;i<CON_reads.size(); i++){
    if(CON_reads[i].direction>0){
      updateCountVectors(CON_reads[i].offset, 
			 CON_reads[i].read.getLenClippedSeq(), 
			 CON_reads[i].read.getClippedSeqIterator(), 
			 CON_reads[i].read.getSequencingType(),
			 true);
    }else{
      updateCountVectors(CON_reads[i].offset, 
			 CON_reads[i].read.getLenClippedSeq(), 
			 CON_reads[i].read.getClippedComplementSeqIterator(), 
			 CON_reads[i].read.getSequencingType(),
			 true);
    }
  }
  
  ccI=CON_counts.begin();
  for(;ccI!=CON_counts.end();ccI++){
    ccI->seqtype_cov[Read::SEQTYPE_SOLEXA]+=ccI->bbcounts[0];
    ccI->total_cov+=ccI->bbcounts[0];
    ccI->seqtype_cov[Read::SEQTYPE_ABISOLID]+=ccI->bbcounts[1];
    ccI->total_cov+=ccI->bbcounts[1];
  }

  FUNCEND();
}





/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// TODO: base and quality check? Offsets after?
void Contig::insertBaseInRead(char base,
			      int32 contigposition,
			      int32 readindex,
			      base_quality_t quality)
{
//  if(contigposition==4709){
//    cout << *this;
//  }
  

  FUNCSTART("void Contig::insertBaseInRead(char base, base_quality_t quality, int32 contigposition, int32 readindex)");

  paranoiaBUGSTAT(checkContig());

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  contigread_t & theread=CON_reads[readindex];

  BUGIFTHROW((contigposition < theread.offset ||
	      contigposition >= theread.offset+theread.read.getLenClippedSeq()),
	     "Read does not cover given contigposition");

  definalise();

  int32 posinread=contigposition-theread.offset;
  int32 updatelen=theread.read.getLenClippedSeq()-posinread;


  bool insertin=false;
  bool insertend=false;

  // now only extend contig when size is affected, never ever at all move
  // other reads around
  if(theread.offset+theread.read.getLenClippedSeq()==CON_counts.size()
     //    FALSCH!!!  && CON_counts.back().coverage==1)
     ) insertend=true;


  CEBUG("insertin: " << insertin << endl);
  CEBUG("insertend: " << insertend << endl);

  base_quality_t rqual=0;

  vector<char>::const_iterator I;

  if(theread.direction>0){
    if(quality==BQ_UNDEFINED || quality>100){
      quality=theread.read.queryAverageQualInClippedSequence(posinread-1, posinread, false, true);
    }
    rqual=quality;

    I=theread.read.getClippedSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq()+1);
    advance(I, posinread);
    updateCountVectors(contigposition, 
		       updatelen, 
		       I,
		       theread.read.getSequencingType(),
		       false);
    theread.read.insertBaseInClippedSequence(base, rqual, posinread,true);
    I=theread.read.getClippedSeqIterator();
  }else{
    if(quality==BQ_UNDEFINED || quality>100){
      quality=theread.read.queryAverageQualInClippedComplementSequence(posinread-1, posinread, false, true);
    }
    rqual=quality;

    I=theread.read.getClippedComplementSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq()+1);
    advance(I, posinread);
    updateCountVectors(contigposition, 
		       updatelen,
		       I,
		       theread.read.getSequencingType(),
		       false);
    theread.read.insertBaseInClippedComplementSequence(base, rqual, posinread,true);
    I=theread.read.getClippedComplementSeqIterator();
  }

  if(insertin==true){
    cccontainer_t::iterator ccI=CON_counts.begin();
    BOUNDCHECK(contigposition, 0, CON_counts.size()+1);
    advance(ccI, contigposition);
    ccI=CON_counts.insert(ccI,CON_concounts_zero);
    ccI->backbonechar='*';
    interpolateSRMValuesInCONcounts(ccI);

    for(uint32 i=0; i<CON_reads.size(); i++){
      if(CON_reads[i].offset>contigposition) CON_reads[i].offset++;
    }
    // adjust tags in consensus
    // TODO: untested (ok?)
    updateTagBaseInserted(contigposition);
  }
  if(insertend==true){
    CON_counts.push_back(CON_concounts_zero);
  }

  advance(I, posinread);
  updateCountVectors(contigposition, 
		     updatelen+1,
		     I,
		     theread.read.getSequencingType(),
		     true);

  paranoiaBUGSTAT(checkContig());

//  if(contigposition==4709){
//    cout << *this;
//  }
  


  FUNCEND();

  return;
}








/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// untested, should work

char Contig::getBaseInRead(int32 contigposition,
			   int32 readindex) const
{
  FUNCSTART("char Contig::getBaseInRead(int32 contigposition, int32 readindex)");

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  //  Read & ric=CON_reads[readindex].read;
  const contigread_t & ric=CON_reads[readindex];

  BUGIFTHROW((contigposition < ric.offset ||
	      contigposition >= ric.offset+ric.read.getLenClippedSeq()),
	     "Read does not cover given contigposition");


  //int32 posinread=contigposition-CON_reads[readindex].offset;

  int32 readpos=contigPosToUnclippedReadPos(contigposition, ric);
  FUNCEND();
  if(ric.direction>0){
    return toupper(ric.read.getBaseInSequence(readpos));
  }else{
    return toupper(ric.read.getBaseInComplementSequence(readpos));
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// untested, should work

base_quality_t Contig::getQualityInRead(int32 contigposition,
			   int32 readindex) const
{
  FUNCSTART("char Contig::getQualityInRead(int32 contigposition, int32 readindex)");

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  //  Read & ric=CON_reads[readindex].read;
  const contigread_t & ric=CON_reads[readindex];

  BUGIFTHROW((contigposition < ric.offset ||
	      contigposition >= ric.offset+ric.read.getLenClippedSeq()),
	     "Read does not cover given contigposition");


  //int32 posinread=contigposition-CON_reads[readindex].offset;

  int32 readpos=contigPosToUnclippedReadPos(contigposition, ric);
  FUNCEND();
  if(ric.direction>0){
    return ric.read.getQualityInSequence(readpos);
  }else{
    return ric.read.getQualityInComplementSequence(readpos);
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// TODO: check base and qualities?
void Contig::changeBaseInRead(char base,
			      int32 contigposition,
			      int32 readindex,
			      base_quality_t quality)
{
  FUNCSTART("void Contig::changeBaseInRead(char base, base_quality_t quality, int32 contigposition, int32 readindex)");

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  //  Read & theread=CON_reads[readindex].read;
  contigread_t & theread=CON_reads[readindex];

  BUGIFTHROW((contigposition < theread.offset ||
	      contigposition >= theread.offset+theread.read.getLenClippedSeq()),
	     "Read does not cover given contigposition");


  definalise();

  base_quality_t rqual=0;

  int32 posinread=contigposition-CON_reads[readindex].offset;
  vector<char>::const_iterator I;

  if(theread.direction>0){
    if(quality==BQ_UNDEFINED || quality>100){
      quality=theread.read.queryAverageQualInClippedSequence(posinread-1, posinread+1, true, true);
    }
    rqual=quality;

    I=theread.read.getClippedSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq());
    advance(I, posinread);
    updateCountVectors(contigposition, 
		       1 ,
		       I,
		       theread.read.getSequencingType(),
		       false);
    theread.read.changeBaseInClippedSequence(base, rqual, posinread);
  }else{
    if(quality==BQ_UNDEFINED || quality>100){
      quality=theread.read.queryAverageQualInClippedComplementSequence(posinread-1, posinread+1, true, true);
    }
    rqual=quality;

    I=theread.read.getClippedComplementSeqIterator();
    BOUNDCHECK(posinread, 0, theread.read.getLenClippedSeq());
    advance(I, posinread);
    updateCountVectors(contigposition,
		       1,
		       I,
		       theread.read.getSequencingType(),
		       false);
    theread.read.changeBaseInClippedComplementSequence(base, rqual, posinread);
  }

  updateCountVectors(contigposition, 
		     1,
		     I,
		     theread.read.getSequencingType(),
		     true);
  
  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const Read & Contig::getReadAtIndex(int32 index) const
{
  FUNCSTART("void Contig::getReadAtIndex(int32 index)");

  BUGIFTHROW((index<0 || index>=CON_reads.size()),
	     "index <0 || >= CON_reads.size() -> Read not in this contig!");

  FUNCEND();

  return CON_reads[index].read;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const Read & Contig::getReadWithID(int32 readid) const
{
  FUNCSTART("int32 Contig::getReadWithID(int32 readid) const");

  vector<contigread_t>::const_iterator crI=CON_reads.begin();

  for(;crI!=CON_reads.end(); crI++){
    if(crI->orpid == readid) {
      FUNCEND();
      return crI->read;
    }
  }
  throw Notify(Notify::INTERNAL, THISFUNC, "id of read not found in contig?");
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::addTagToRead(uint32 contigpositionfrom,
			  uint32 contigpositionto,
			  int32 readindex,
			  const string & identifier,
			  const string & comment)
{
  FUNCSTART("void Contig::addTagToRead(int32 contigpositionfrom, int32 contigpositionto, int32 readindex, const string & identifier, const string & comment)");

  BUGIFTHROW((readindex<0 || readindex>=CON_reads.size()),
	     "Readindex <0 || >= CON_reads.size() -> Read not in this contig!");

  //  Read & theread=CON_reads[readindex].read;
  contigread_t & theread=CON_reads[readindex];

  BUGIFTHROW((contigpositionfrom < theread.offset ||
	      contigpositionfrom > theread.offset+theread.read.getLenClippedSeq()),
	     "Read does not cover given contigposition (from)");
  BUGIFTHROW((contigpositionto < static_cast<uint32>(theread.offset) ||
	      contigpositionto > theread.offset+theread.read.getLenClippedSeq()),
	     "Read does not cover given contigposition (to)");


  definalise();
  
  //TODO: check

  if(contigpositionto < contigpositionfrom) swap(contigpositionfrom,
						 contigpositionto);

  uint32 posinreadfrom;
  uint32 posinreadto;
  uint32 len=(contigpositionto-contigpositionfrom);
  if(CON_reads[readindex].direction>0){
    posinreadfrom=contigpositionfrom-theread.offset+theread.read.getLeftClipoff();
    posinreadto=posinreadfrom+len;
  }else{
    posinreadto=theread.read.getLenClippedSeq()-(contigpositionfrom-theread.offset)+theread.read.getLeftClipoff()-1;
    posinreadfrom=posinreadto-len;
  }

  theread.read.addTag(posinreadfrom,
		      posinreadto,
		      identifier,
		      comment);

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::addTagToRead(const uint32 contigpositionfrom,
			  const uint32 contigpositionto,
			  const int32 readindex,
			  const char * identifier,
			  const char * comment)
{
  string id=identifier;
  string co=comment;
  addTagToRead(contigpositionfrom, contigpositionto, readindex, id, co);
}


// TODO: write version with multitag_t::mte_id_t??

/*************************************************************************
 *
 * Adds a tag to the consensus if this tag is not already present
 *
 *
 *************************************************************************/

void Contig::addTagToConsensus(const uint32 contigposfrom,
			       const uint32 contigposto,
			       const char strand, 
			       const char * identifier,
			       const char * comment,
			       const bool doublecheck,
			       const bool additionalinfo,
			       const base_quality_t qualA,
			       const base_quality_t qualC,
			       const base_quality_t qualG,
			       const base_quality_t qualT,
			       const base_quality_t qualStar
			       )
{
  FUNCSTART("void Contig::addTagToConsensus(int32 contigposfrom, int32 contigposto, const char * identifier, const char * comment)");

  BUGIFTHROW((contigposto < contigposfrom),
	     "contigposto < contigposfrom not allowed");
  //BUGIFTHROW((contigposfrom < 0),
  //	     "contigposfrom < 0 not allowed");
  BUGIFTHROW((contigposfrom >= CON_counts.size()),
	     "contigposfrom > size of contig");
  //BUGIFTHROW((contigposto < 0),
  //	     "contigposto < 0 not allowed");
  BUGIFTHROW((contigposto >= CON_counts.size()),
	     "contigposto > size of contig");

  BUGIFTHROW(!(strand=='+' || strand=='-' || strand=='='),
	     "strand is not +, - or =");

  // TODO: commented out 22.04.2007, check whether ok 
  // not needed? and disturbs setting tags in makeIntelligentConsensus()
  // definalise();

  consensustag_t thetag;

  thetag.identifier=multitag_t::newIdentifier(identifier);
  thetag.comment=multitag_t::newComment(comment);
  thetag.from=contigposfrom;
  thetag.to=contigposto;
  thetag.strand=strand;
  thetag.additionalinfo_initialised=additionalinfo;
  thetag.qualACGTStar[0]=qualA;
  thetag.qualACGTStar[1]=qualC;
  thetag.qualACGTStar[2]=qualG;
  thetag.qualACGTStar[3]=qualT;
  thetag.qualACGTStar[4]=qualStar;

  bool found=false;
  if(doublecheck){
    for(uint32 i=0; i<CON_consensus_tags.size(); i++){
      if(CON_consensus_tags[i].from==thetag.from
	 && CON_consensus_tags[i].to==thetag.to
	 && CON_consensus_tags[i].strand==thetag.strand
	 && CON_consensus_tags[i].identifier==thetag.identifier
	) {
	found=true;
	
	// replace with new comments and additional info
	CON_consensus_tags[i]=thetag;
      }
    }
  }

  if(!found) CON_consensus_tags.push_back(thetag);  

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::updateTagBaseDeleted(uint32 contigpos)
{
  FUNCSTART("void Contig::updateTagBaseDeleted(uint32 contigpos)");
  //BUGIFTHROW((contigpos < 0),
  //	     "contigpos < 0 not allowed");
  BUGIFTHROW((contigpos >= CON_counts.size()),
	     "contigpos > size of contig");

  vector<consensustag_t>::iterator I=CON_consensus_tags.begin();
  while(I!=CON_consensus_tags.end()){
    if(I->to == I->from && I->to == contigpos){
      I=CON_consensus_tags.erase(I);
    }else{
      if(contigpos <= I->to) I->to-=1;
      if(contigpos < I->from) I->from-=1;
      I++;
    }
  }
  
  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Contig::updateTagBaseInserted(uint32 contigpos)
{
  FUNCSTART("void Contig::updateTagInsert(uint32 contigpos)");

  //BUGIFTHROW((contigpos < 0),
  //	     "contigpos < 0 not allowed");
  BUGIFTHROW((contigpos >= CON_counts.size()),
	     "contigpos > size of contig");

  vector<consensustag_t>::iterator I=CON_consensus_tags.begin();
  while(I!=CON_consensus_tags.end()){
    if(I->from >= contigpos) I->from+=1;
    if(I->to >= contigpos) I->to+=1;
    I++;
  }

  FUNCEND();
}




// *******************************************************************
//  unclippedReadPosToContigPos
//  We have a position in an UNCLIPPED read and we want to know the corresponding
//  position in the contig
//
//  ******************************************************************        

int32 Contig::unclippedReadPosToContigPos(const int32 readpos, const contigread_t &ric) const
{
  if (ric.direction > 0) {
    return readpos - ric.read.getLeftClipoff() + ric.offset ;
  } else {
    // doch richtig? FALSCH return ric.offset + ric.read.getRightClipoff() -1 - readpos;
    // FALSCH, das ist unclipped?return readpos+ric.offset-(ric.read.getLenSeq()-ric.read.getRightClipoff());
    return ric.read.getRightClipoff()-readpos+ric.offset -1 ;
  }
}


//  ******************************************************************        
//  contigPosToUnclippedReadPos
//  We have a contig position and we want to know which position in the
//  UNCLIPPED FORWARD read corresponds to the contig position
//
//  ******************************************************************        

int32 Contig::contigPosToUnclippedReadPos(const int32 contigpos, const contigread_t &ric) const
{
  if (ric.direction > 0) {
    return contigpos - ric.offset + ric.read.getLeftClipoff();
  } else {
    // FALSCH return ric.offset + ric.read.getRightClipoff() -1 - contigpos;
    return ric.read.getLenSeq()-ric.read.getRightClipoff()+contigpos-ric.offset;
  }
}


int32 Contig::getRealReadPos(const int32 contigpos, const uint32 readindex) const
{
  FUNCSTART("int32 Contig::getRealReadPos(const int32 contigpos, const uint32 readindex) const");
  BUGIFTHROW(readindex>=CON_reads.size(),"readindex>=CON_reads.size()?");
  FUNCEND();
  return getRealReadPos(contigpos,CON_reads[readindex]);
}

int32 Contig::getRealReadPos(const int32 contigpos, const contigread_t &ric) const
{
  if(ric.direction>0){
    return contigPosToUnclippedReadPos(contigpos, ric);
  }
  return ric.read.calcComplPos(contigPosToUnclippedReadPos(contigpos, ric));
}



//int32 Contig::contigPosToComplReadPos(const int32 contigpos, const contigread_t &ric)
//{
//  if (ric.direction > 0) {
//    return ric.read.getLenSeq() -1 - (contigpos - ric.offset + ric.read.getLeftClipoff());   
//  } else {
//    return ric.read.getLenSeq() -1 - (ric.offset + ric.read.getRightClipoff() -1 - contigpos);
//    // return contigpos-ric.offset+ric.read.getLenSeq()-ric.read.getRightClipoff();
//  }
//}
//





/*************************************************************************
 *
 * Transfers SRMr tags in reads to a ORMB tag in the contig
 *
 *
 *************************************************************************/
void Contig::transposeReadSRMTagsToContig()
{
  FUNCSTART("void Contig::transposeReadSRMTagsToContig()");

  //cout << "We are in contig: " << getContigName() << endl;

  vector<contigread_t>::const_iterator rI=CON_reads.begin();
  for(;rI!=CON_reads.end();rI++){

    const contigread_t & ric =*rI;

    //cout << "We are in read: " << ric.read.getName() << endl;

    for(uint32 i=0; i<ric.read.getNumOfTags(); i++){
      multitag_t acttag=ric.read.getTag(i);
      //cout << "Tag #" << i << "\tfrom: " << acttag.from << "\tto: " << acttag.to << "\t" << acttag.identifier[0] << acttag.identifier[1] << acttag.identifier[2] << acttag.identifier[3] << endl;
      if(acttag.identifier==Read::REA_tagentry_idSRMr
	 || acttag.identifier==Read::REA_tagentry_idWRMr ) {
	
	int32 tleftbound;
	int32 trightbound;
	
	if(ric.direction>0){
	  tleftbound=acttag.from-ric.read.getLeftClipoff();
	  trightbound=acttag.to-ric.read.getLeftClipoff();
	}else{
	  tleftbound=ric.read.getRightClipoff()-acttag.to-1;
	  trightbound=ric.read.getRightClipoff()-acttag.from-1;
	}
	if(tleftbound<0) tleftbound=0;
	if(trightbound<0) trightbound=0;
	
	int32 lpos=ric.offset+tleftbound;
	int32 rpos=ric.offset+trightbound;
	//cout << "Tagging bounds " << ric.offset+tleftbound << "\t" << ric.offset+trightbound << endl;
	// this is needed as some RMB tags might be in presently clipped 
	//  parts of a read 
	if(lpos < static_cast<int32>(CON_counts.size())
	   && rpos>=0 ) {
	  // adjust for partly present tags
	  if(lpos<0) lpos=0;
	  if(rpos >= static_cast<int32>(CON_counts.size())) rpos=CON_counts.size()-1;
	  //cout << "Tagging bounds cleaned" << lpos << "\t" << rpos << endl;
	  addTagToConsensus(lpos,
			    rpos,
			    '=',
			    "ORMB",
			    "Old repeat marker base",
			    true);
	}
      }
    }
  }

  FUNCEND();
}



/*************************************************************************
 *
 * actualises template and strain ids in the contig reads from the readpool
 * also refills the template ids present vector of the contig
 * also refills the CON_readsperstrain vector
 *
 * (routine needed for backbone assembly. data loaded for backbone
 *  would conflict with data for assembly)
 *
 *************************************************************************/

void Contig::recalcTemplateAndStrainIDsPresent()
{
  FUNCSTART("void Contig::recalcTemplateAndStrainIDsPresent()");

  CON_templates_present.clear();
  CON_readsperstrain.clear();
  CON_readsperstrain.resize(CON_readpool->getNumOfStrainInReadpool(),0);

  vector<contigread_t>::iterator I=CON_reads.begin();
  for(; I!=CON_reads.end(); I++){
    if(I->orpid>=0){
      I->read.setTemplateID(CON_readpool->getRead(I->orpid).getTemplateID());
      I->read.setTemplatePartnerID(CON_readpool->getRead(I->orpid).getTemplatePartnerID());
      I->read.setTemplate(CON_readpool->getRead(I->orpid).getTemplate());
      if(I->read.getTemplateID()>=0){
	CON_templates_present.insert(I->read.getTemplateID());
      }
      I->read.setStrainID(CON_readpool->getRead(I->orpid).getStrainID());
      I->read.setStrain(CON_readpool->getRead(I->orpid).getStrain());

      ++CON_readsperstrain[CON_readpool->getRead(I->orpid).getStrainID()];
    }
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 * build the "rails" for a backbone (except for singlets)
 *   the rails are pseudo reads of a given length that are
 *    deduced from the consensus of the backbone contigs
 *   they are inserted into the contig to cover it without
 *    gaps. Eventually with overlaps.
 * Schematic drawing (a-f reads, 1-4 rails), overlap 2.
 *
 *   aaaaaaaaaaaaaaaaaaaaa
 *            bbbbbbbbbbbbbbbbbbbbb
 *	          cccccccccccccccccccccc
 *		   ddddddddddddddddddddddddddddddddd
 *		      eeeeeeeeeeeeeeeeeeeeee
 *		              ffffffffffffffffffffffffffffff
 *   1111111111111111
 *                 22222222222222222
 *                                333333333333333
 *                                             4444444444444
 *
 *   Aims:
 *    - allow backbones at all :-) 
 *    - reduce the work for the assembler a lot
 *    - hold back automatic editor to insert or delete whole
 *      columns
 *
 * Also inserts rails as reads into the readpool (not when simulateonly 
 *  is true)
 *
 * Returns number of rails inserted (or simulated to insert)
 *
 * Also initialise values in CON_counts structure for backbone info
 *
 * Additional functionality later: add the reads as "backbonelets"
 *  to circumvent gap4 maximum read length for directed assembly
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}

size_t Contig::addRails(const uint32 raillength, const uint32 railoverlap, const string & straintxt, const bool forcestrainset, const string & railfromstrain, const bool simulateonly)
{
  vector<base_quality_t> tmp;

  if(!simulateonly){
    cout << "Adding rails: length " << raillength << " and overlap " << railoverlap << endl;
  }

  return addSubsequences(raillength, 
			 railoverlap,
			 straintxt, 
			 forcestrainset,
			 railfromstrain,
			 false, "", tmp, "", true,
			 simulateonly);
}


size_t Contig::addSubsequences(const uint32 raillength, const uint32 railoverlap, const string & straintxt, const bool forcestrainset, const string & railfromstrain, const bool asbackbone, const string & bbseq, const vector<base_quality_t> & bbqualvec, const string & backbonename, const bool initccbbvalues, const bool simulateonly)
{
  FUNCSTART("void Contig::addRails()");

  // make sure we get the latest consensus
  definalise();

  CEBUG("railfromstrain: " << railfromstrain << endl);
  CEBUG("len: " << raillength << endl);
  CEBUG("overlap: " << railoverlap << endl);


  size_t numrailsinserted=0;
  string consseq;
  vector<base_quality_t> consqual;

  if(asbackbone) {
    consseq=bbseq;
    consqual=bbqualvec;
  }else{
    //getConsensus(consseq, consqual, true,0,0);
    if(!railfromstrain.empty()){
      int32 strainidtotake=0;
      if(!CON_readpool->getStrainIDOfStrain(railfromstrain,strainidtotake)){
	cout.flush();
	cout << "Error: did not find strain \"" << railfromstrain << "\" in readpool." << endl;
	throw Notify(Notify::FATAL, THISFUNC, "Have been asked to use non-existing strain for building rails.");
      }
      //string strainseq;
      //vector<base_quality_t> strainqual;

      CEBUG("Strainidtotake: " << strainidtotake << endl);

      CEBUG(">>>> strainidtotake: " << strainidtotake << endl);

      newConsensusGet(consseq, consqual, strainidtotake);
      
      // TODO: eventually "merge" strainseq with consseq
      //hier weiter: strainseq und consseq mergen nach consseq

    }else{
      newConsensusGet(consseq, consqual);
    }

    //// fill quality vector with qual (if given)
    //{
    //  vector<base_quality_t>::iterator bI=consqual.begin();
    //  for(; bI!=consqual.end(); bI++) *bI=bbquals;
    //}
  }

  CEBUG(">>>> " << consseq.substr(0,200) << endl);

  // init the CON_counts backbone values if needed
  if(initccbbvalues && !simulateonly){
    cccontainer_t::iterator ccI=CON_counts.begin();
    const char * consb=&consseq[0];
    for(; ccI != CON_counts.end(); ccI++, consb++){
      ccI->bbcounts[0]=0;
      ccI->bbcounts[1]=0;
      ccI->bbbestquals[0]=0;
      ccI->bbbestquals[1]=0;
      ccI->bbstrains[0]=0;
      ccI->bbstrains[1]=0;
      ccI->backbonechar=toupper(*consb);
    }
  }

  // keep readname here so that at the end we can easily print out the name of the last rail
  string readname;

  // make the railreads
  {
    string cutseq;
    cutseq.reserve(raillength+10);
    vector<base_quality_t> cutqual;
    cutqual.reserve(raillength+10);
    
    uint32 transi=0;
    uint32 actoffset=0;
    char tmpcons;
    for(;transi<consseq.size(); transi++){
      tmpcons=consseq[transi];
      if(tmpcons=='@') tmpcons='N';
      cutseq+=tmpcons;
      cutqual.push_back(consqual[transi]);

      // check whether one rail is ready
      // be sure that last rail is not < ... bases
      if((cutseq.size()==raillength && consseq.size()-transi>raillength)
	 || cutseq.size()>=29900
	 || transi==consseq.size()-1){
	
	numrailsinserted++;
	
	if(!simulateonly){
	  CON_railcounter++;
	  ostringstream ostr;
	  
	  if(asbackbone) {
	    ostr << "bb_" << CON_railcounter << "_" << backbonename;
	  } else {
	    ostr << "rr_####" << CON_readpool->size() << "####";
	  }

	  readname=ostr.str();

	  if(numrailsinserted==1 && !simulateonly){
	    cout << getContigName() << " first rail: " << readname << endl;
	  }

	  CON_readpool->addNewEmptyRead();
	  
	  Read & newread=CON_readpool->back();
	  
	  try{
	    //newread.setFileNamesFromFASTAName(readname);
	    newread.setName(readname);
	    newread.setSequenceFromString(cutseq);
	    newread.setQualities(cutqual);
	    newread.setStrain(straintxt.c_str());
	    
	    if(asbackbone){
	      newread.setBackbone(true);
	      newread.setRail(false);
	    }else{
	      newread.setBackbone(false);
	      newread.setRail(true);
	    }
	  }
	  catch(Notify n){
	    cout << "\nOuch ... error while adding subsequence.\n";
	    cout << "cs.size: " << consseq.size() << endl;
	    cout << "transi: " << transi << endl;
	    cout << "actoffset: " << actoffset << endl;
	    cout << "readname: " << readname << endl;
	    cout << "strain: " << straintxt << endl;
	    cout << "seq: " << cutseq << endl;
	    n.handleError(THISFUNC);
	  }
	  
	  //CEBUG("New element:\t");
	  {
	    contigread_t tmp;
	    
	    tmp.offset=actoffset;
	    tmp.orpid=CON_readpool->size()-1;
	    tmp.direction=1;
	    
	    CON_reads.push_back(tmp);
	    
	    CON_reads.back().read=newread;
	    
	    CEBUG(readname << "\t" << tmp.offset << '\t' << newread.getLenClippedSeq() << endl);
	    
	    //CEBUG(tmp.offset << endl);
	    //CEBUG(tmp.id << endl);
	    //CEBUG(tmp.direction << endl);
	    //CEBUG(CON_reads.back().read << endl);
	  }
	  
	  //try{
	  //  updateCountVectors(actoffset,
	  //		     CON_reads.back().read.getLenClippedSeq(),
	  //		     newread.getClippedSeqIterator(),
	  //		     true);
	  //}
	  //catch(Notify n){
	  //  cout << "Readpool id: " << CON_reads.back().id << endl;
	  //  cout << "Error while adding this read (2):\n" << CON_reads.back().read;
	  //  cout << "In this contig (output might crash):" << endl << *this << endl;
	  //  n.handleError(THISFUNC);
	  //}
	  
	  
	  //definalise();
	  //cout << "tada: " << CON_reads.size() << endl;
	  //setCoutType(AS_TEXT);
	  //cout << *this;
	  
	}
	cutseq.clear();
	cutqual.clear();
	if(transi!=consseq.size()-1) transi-=railoverlap;
	actoffset=transi+1;
      }
    }
  }

  if(!simulateonly){
    cout << getContigName() << " last rail: " << readname << endl;
  }

  definalise();

  FUNCEND();
  return numrailsinserted;
}


//#define CEBUG(bla)



/*************************************************************************
 *
 * Remove all rails from a contig
 *
 *************************************************************************/

void Contig::removeRails()
{
  FUNCSTART("void Contig::removeRails()");

  vector<contigread_t>::iterator cfromI=CON_reads.begin();
  vector<contigread_t>::iterator ctoI=cfromI;

  for(; cfromI!=CON_reads.end(); cfromI++){
    if(cfromI->read.isRail()){
//      updateCountVectors(cfromI->offset,
//			 cfromI->read.getLenClippedSeq(),
//			 cfromI->read.getClippedSeqIterator(),
//			 false);
    }else{
      // TODO: it'd be better so collect all index of reads that
      //  get freed and at the end copy from back of vector
      //  into the free places. Then again, I currently don't care:
      // It's done only once per contig before output.
      // Perhaps once we reach 10 mio reads or so.

      if(ctoI!=cfromI){
	*ctoI=*cfromI;
      }
      ctoI++;
    }
  }

  CON_reads.resize(ctoI-CON_reads.begin());

  definalise();

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush();}

void Contig::transformCERMappingsToCoverageReads()
{
  FUNCSTART("void Contig::transformCERMappingsToCoverageReads()");

  // go through all short read types
  // srtype==0 Solexa, 1==SOLiD

  string srmseq;
  srmseq.reserve(CON_counts.size());

  vector<base_quality_t> srmqual;
  srmqual.reserve(CON_counts.size());

  vector<int32> strainsinsrm;

  for(unsigned int actsrtype=0; actsrtype<2; actsrtype++){
    // go through complete CON_counts, looking for areas which have a
    //  a short read coverage of the actsrtype

    bool needanotherpass=true;
    uint32 passnum=0;

    while(needanotherpass){
      passnum++;
      needanotherpass=false;
      srmseq.clear();
      srmqual.clear();
      cccontainer_t::iterator ccI=CON_counts.begin();
      int32 rangestart=-1;       // -1, not inrange
      int32 ccpos=0;
      uint8 thisstrainmask=0;
      bool needsave=true;
      uint32 rangesthispass=0;
      for(; ccI!=CON_counts.end(); ccI++, ccpos++){
	CEBUG("ccpos: " << ccpos << '\t' << *ccI << '\n');
	if(rangestart>=0){
	  // do we need to finish?
	  if(ccI->bbstrains[actsrtype]!= thisstrainmask
	     || ccI->bbcounts[actsrtype]==0 
	     || ccI->backbonechar=='@'
	     || ccpos+1==CON_counts.size()){
	    // yes, look whether stretch is long enough or we need to 
	    //  save this part
	    // FIXME:
	    // the first line (>5) is a lame fix for almost empty reads
	    //  (only a gap) being added (with no strain): "N*"
	    if(ccpos-rangestart>5
	       && ((needsave && ccpos-rangestart>0) || ccpos-rangestart>30)){
	      CEBUG("Would add: " << rangestart << "\t" <<ccpos);
	      CEBUG("\n" << srmseq << "\n");
	      
	      ostringstream ostr;
	      if(actsrtype==0){
		ostr << "_cer_sxa_" << CON_cer_idcounter << '_';
	      }else{
		ostr << "_cer_sid_" << CON_cer_idcounter << '_';
	      }

	      definalise();

	      CON_reads.resize(CON_reads.size()+1);

	      CON_reads.back().offset=-1;
	      CON_reads.back().orpid=-1;
	      CON_reads.back().direction=1;	      
	      CON_reads.back().offset=rangestart;
  
	      CEBUG("Adding:\n" << ostr.str() << '\n');
	      CEBUG("Orpid: " << CON_reads.back().orpid);
	      CEBUG("\nOffset: " << CON_reads.back().offset);
	      CEBUG("\nDirection: " << CON_reads.back().direction);

	      Read & newread=CON_reads.back().read;

	      // we won't need adjustments, let's save some memory
	      newread.disallowAdjustments();

	      newread.setName(ostr.str());
	      newread.setSequenceFromString(srmseq);
	      newread.setLQClipoff(1);
	      if(actsrtype==0){
		newread.setSequencingType(Read::SEQTYPE_SOLEXA);
	      }else if(actsrtype==1){
		newread.setSequencingType(Read::SEQTYPE_ABISOLID);
	      }else{
		throw Notify(Notify::INTERNAL, THISFUNC, "Unknown actsrtype? Do we have more than Solexa and SOLiD?.");
	      }
	      newread.setCoverageEquivalentRead(true);
	      //newread.setStrain(straintxt.c_str());
	      if(srmqual.empty()){
		newread.setQualities(1);
		newread.setQualityFlag(false);
	      }else{
		newread.setQualities(srmqual);
	      }

	      CEBUG("\nLen: " << newread.getLenClippedSeq());
	      CEBUG(endl);

	      // set a strain, if possible
	      getMappedBBStrainIDsFromMask(strainsinsrm,thisstrainmask);

	      // TODO: think of something for several strains
	      if(!strainsinsrm.empty()){
		CON_reads.back().read.setStrainID(strainsinsrm[0]);
		CON_reads.back().read.setStrain(
		  CON_readpool->getStrainOfStrainID(strainsinsrm[0])
		  );
	      }

	      //assout::saveAsCAF(*this, "transfsrmdebug.caf", true);

	      //CON_reads.back().read.exchangeNsWithGaps();
	      CON_cer_idcounter++;
	      rangesthispass++;
	      needanotherpass=true;
	    }
	    rangestart=-1;
	    ccI--;
	    ccpos--;
	    needsave=true;
	    srmseq.clear();
	    srmqual.clear();
	  }else{
	    //// no, just append current bb char to sequence
	    srmseq.push_back(tolower(ccI->backbonechar));
	    //srmseq.push_back('n');

	    if(passnum==1) srmqual.push_back(ccI->bbbestquals[actsrtype]);
	  }
	}else{
	  if(ccI->bbcounts[actsrtype]>0){
	    thisstrainmask=ccI->bbstrains[actsrtype];
	    rangestart=ccpos;
	    srmseq.clear();
	    srmqual.clear();
	    srmseq.push_back('X');
	    srmseq.push_back(tolower(ccI->backbonechar));
	    //srmseq.push_back('n');
	    if(passnum==1) {
	      srmqual.push_back(0);
	      srmqual.push_back(ccI->bbbestquals[actsrtype]);
	    }
	  }else{
	    //if(rangesthispass>0) 
	    needsave=false;
	  }
	}
	if(ccI->bbcounts[actsrtype]>0) ccI->bbcounts[actsrtype]--;
      }
    }
  }
  FUNCEND();
  return;
}

//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Contig::getMappedBBStrainIDsFromMask(vector<int32> & strains, uint8 mask)
{
  strains.clear();
  for(uint8 i=0; i<8; i++){
    if(mask & 1) strains.push_back(i);
    mask=mask>>1;
  }
  return;
}

/*************************************************************************
 *
 * copy first all Genbank features of all reads into one big vector
 *  if allowedfeatures not empty: only the features named there
 *  if forbiddenfeatures not empty: do not take features named there
 *
 * reads that have  Fsrc, Fgen, FCDS, Fexn, or Fint can get additional
 *  intergenic regions simulated if wished
 *
 * list is sorted ascending before given back by contigfrom, to and lastly
 *  by identifier (Fsrc, Fgen, FCDS, rest)
 *
 *************************************************************************/
//#define CEBUGF2(bla)  {cout << bla; cout.flush();}
#define CEBUGF2(bla)

void Contig::getGBFSummary(list<gbfsummary_t> & allGBfeatures, const vector<multitag_t::mte_id_t> & allowedfeatures, const vector<multitag_t::mte_id_t> & forbiddenfeatures, bool simulateintergenics) const
{
  allGBfeatures.clear();

  vector<contigread_t>::const_iterator R=CON_reads.begin();
  vector<int8> featuremask;
  uint32 igrcounter=0;

  for(; R != CON_reads.end(); R++){
    featuremask.clear();

    //Read::setCoutType(Read::AS_TEXTCLIPS);
    //CEBUGF2(R->read);
    bool mustsimulateigr=false;
    for(uint32 t=0; t<R->read.getNumOfTags(); t++){
      const multitag_t & thetag=R->read.getTag(t);
      if(thetag.identifier==Read::REA_tagentry_idFsrc
    	 || thetag.identifier==Read::REA_tagentry_idFgen
    	 || thetag.identifier==Read::REA_tagentry_idFCDS
    	 || thetag.identifier==Read::REA_tagentry_idFexn
    	 || thetag.identifier==Read::REA_tagentry_idFint) mustsimulateigr=simulateintergenics;
    }

    for(uint32 t=0; t<R->read.getNumOfTags(); t++){
      const multitag_t & thetag=R->read.getTag(t);
      if(!GBF::checkIfGAP4GBFfeaure(thetag.getIdentifierStr())) continue;

      // next tag if tag is completely outside of the good read part
      if(!((thetag.from >= R->read.getLeftClipoff()
	    && thetag.from < R->read.getRightClipoff())
	   || (thetag.to >= R->read.getLeftClipoff()
	       && thetag.to < R->read.getRightClipoff()))) continue;	

      bool musttakeit=true;
      if(!allowedfeatures.empty()) {
	musttakeit=false;
	for(uint32 i=0;i<allowedfeatures.size();i++){
	  if(thetag.identifier == allowedfeatures[i]){
	    musttakeit=true;
	    break;
	  }
	}
      }


      if(!forbiddenfeatures.empty()) {
	for(uint32 i=0;i<forbiddenfeatures.size();i++){
	  if(thetag.identifier == forbiddenfeatures[i]){
	    musttakeit=false;
	    break;
	  }
	}
      }

      bool isgene=false;
      if(thetag.identifier==Read::REA_tagentry_idFgen
	 || thetag.identifier==Read::REA_tagentry_idFCDS
	 || thetag.identifier==Read::REA_tagentry_idFexn
	 || thetag.identifier==Read::REA_tagentry_idFint
	 || thetag.identifier==Read::REA_tagentry_idFmRN
	 || thetag.identifier==Read::REA_tagentry_idFm_R
	 || thetag.identifier==Read::REA_tagentry_idFpRN
	 || thetag.identifier==Read::REA_tagentry_idFrRN
	 || thetag.identifier==Read::REA_tagentry_idFscR
	 || thetag.identifier==Read::REA_tagentry_idFsnR
	 || thetag.identifier==Read::REA_tagentry_idFtRN
	){
	isgene=true;
      }

      if(musttakeit) {
	gbfsummary_t onefeature;
	string extracted;

	CEBUG("Have this:\n"<<thetag<<endl);

	onefeature.identifier=thetag.getIdentifierStr();

	onefeature.mustbetranslated=false;
	if(thetag.identifier==Read::REA_tagentry_idFCDS){
	  onefeature.mustbetranslated=true;
	}
	onefeature.cfrom=unclippedReadPosToContigPos(thetag.from,*R);
	onefeature.cto=unclippedReadPosToContigPos(thetag.to, *R);
	if(thetag.strand=='-'){
	  onefeature.direction=-1;
	} else {
	  onefeature.direction=1;
	}
	onefeature.readindex=R-CON_reads.begin();
	onefeature.strainid=R->read.getStrainID();
	if(thetag.extractGenBankKeyValueFromComment("/locus_tag",extracted)) {
	  onefeature.locustag=extracted;
	}
	if(thetag.extractGenBankKeyValueFromComment("/gene",extracted)) {
	  onefeature.gene=extracted;
	}
	if(thetag.extractGenBankKeyValueFromComment("/function",extracted)) {
	  onefeature.function=extracted;
	}
	if(thetag.extractGenBankKeyValueFromComment("/EC_number",extracted)) {
	  onefeature.ecnumber=extracted;
	}
	if(thetag.extractGenBankKeyValueFromComment("/product",extracted)) {
	  onefeature.product=extracted;
	}
	if(thetag.extractGenBankKeyValueFromComment("/note",extracted)) {
	  onefeature.note=extracted;
	}
	onefeature.translationtable=11;
	if(thetag.extractGenBankKeyValueFromComment("/transl_table",extracted)) {
	  onefeature.translationtable=static_cast<int8>(atoi(extracted.c_str()));
	}
	onefeature.codonstart=1;
	if(thetag.extractGenBankKeyValueFromComment("/codon_start",extracted)) {
	  onefeature.codonstart=static_cast<int8>(atoi(extracted.c_str()));
	}

	onefeature.isgene=isgene;

	CEBUGF2("Extracted feature:\n" << onefeature);
	
	allGBfeatures.push_back(onefeature);
      }

      // preparing for intergenic regions: mask away this feature from read 
      if(isgene){
	if(featuremask.empty()) featuremask.resize(R->read.getLenSeq(),1);
	for(uint32 pos=thetag.from; pos<=thetag.to; pos++){
	  if(pos<R->read.getLenSeq()) featuremask[pos]=0;
	}
      }
    }

    // if there were GenBank features in this read, simulate intergenic
    //  features
    if(!featuremask.empty() && mustsimulateigr){
      CEBUGF2("mustsimulateigr: " << mustsimulateigr << '\n');
      CEBUGF2("Simulate IGR for " << R->read.getName() << '\n');
      bool inintergenic=false;
      uint32 runstart=0;
      for(uint32 pos=0; pos <= R->read.getLenSeq(); pos++){
	if(pos < R->read.getLenSeq()){
	  CEBUGF2("featuremask[" << pos << "]:\t" << static_cast<uint16>(featuremask[pos]) << endl);
	}
	if(pos == R->read.getLenSeq() || featuremask[pos]==0){
	  if(inintergenic){
	    gbfsummary_t onefeature;
	    
	    onefeature.identifier="Figr";
	    
	    onefeature.mustbetranslated=false;
	    int32 tmp=unclippedReadPosToContigPos(runstart,*R);
	    if(tmp<0) tmp=0;
	    onefeature.cfrom=tmp;
	    if(pos == R->read.getLenSeq()){
	      // carefull: if at end of sequence, we must take -1
	      //  or else the tag will be one too long
	      onefeature.cto=unclippedReadPosToContigPos(pos-1,*R);
	    }else{
	      onefeature.cto=unclippedReadPosToContigPos(pos,*R);
	    }
	    onefeature.direction=1;
	    onefeature.readindex=R-CON_reads.begin();
	    onefeature.strainid=R->read.getStrainID();
	    onefeature.isgene=false;
	    allGBfeatures.push_back(onefeature);

	    CEBUGF2("Made IGR:\n" << onefeature);
	    
	    inintergenic=false;
	    runstart=0;
	  }
	}else{
	  if(!inintergenic){
	    inintergenic=true;
	    runstart=pos;
	  }
	}
      }
    }
  }

  CEBUGF2("going to sort\n");
  allGBfeatures.sort(Contig::gbfsummary_t_comparator);
  CEBUGF2("sorted\n");

  // assign names to the intergenic regions
  if(simulateintergenics){
    string gene;
    string function;
    string ecnumber;
    string product;
    string note;

    string concatstring="; ";

    list<gbfsummary_t>::iterator gbfsI=allGBfeatures.begin();
    for(; gbfsI != allGBfeatures.end(); gbfsI++){
      if(gbfsI->identifier == "Figr"){
      	ostringstream newlocusostr;
      	ostringstream newnoteostr;
      
      	newlocusostr << "IGR_";
      	newnoteostr << "Intergenic between ";

CEBUGF2("#1\n");

      	bool foundprevious=false;
      	if(gbfsI != allGBfeatures.begin()){
	  list<gbfsummary_t>::iterator gbfsImm=gbfsI;

CEBUGF2("#2\n");
	  while (gbfsImm != allGBfeatures.begin()
		 && gbfsI->readindex == gbfsImm->readindex
		 && (gbfsImm == gbfsI || gbfsImm->isgene==false)) gbfsImm--;

CEBUGF2("#3\n");
	  if(gbfsI->readindex == gbfsImm->readindex) {
	    newlocusostr << gbfsImm->locustag << '_';
CEBUGF2("#4\n");
	    concatAllGBFInfoForLocus(allGBfeatures, gbfsImm, concatstring, gene, function, ecnumber, product, note);
CEBUGF2("#5\n");
	    if(gene.empty()){
	      newnoteostr << gbfsImm->locustag << " and ";
	    }else{
	      newnoteostr << gene << " and ";
	    }
CEBUGF2("#6\n");
	    foundprevious=true;
	  }
	}
CEBUGF2("#7\n");
      	if(!foundprevious){
      	  newlocusostr << "seqstart_";
      	  newnoteostr << "sequence start and ";
      	}
      	
      	if(gbfsI != allGBfeatures.end()) {
CEBUGF2("#8\n");
	  list<gbfsummary_t>::iterator gbfsImm=gbfsI;
	  while (gbfsImm != allGBfeatures.end()
		 && gbfsI->readindex == gbfsImm->readindex
		 && (gbfsImm == gbfsI || gbfsImm->isgene==false)) gbfsImm++;

CEBUGF2("#9\n");
	  foundprevious=false;
	  if(gbfsImm != allGBfeatures.end()
	     && gbfsI->readindex == gbfsImm->readindex) {
CEBUGF2("#a\n");
	    newlocusostr << gbfsImm->locustag;
CEBUGF2("#b\n");
	    concatAllGBFInfoForLocus(allGBfeatures, gbfsImm, concatstring, gene, function, ecnumber, product, note);
CEBUGF2("#c\n");
	    if(gene.empty()){
	      newnoteostr << gbfsImm->locustag;
	    }else{
	      newnoteostr << gene;
	    }
CEBUGF2("#d\n");
	    foundprevious=true;
	  }
	}
	if(!foundprevious){
      	  newlocusostr << "seqend";
      	  newnoteostr << "sequence end";
      	}
      	
CEBUGF2("#e\n");
      	gbfsI->locustag=newlocusostr.str();
      	gbfsI->note = newnoteostr.str();
CEBUGF2("#f\n");
      }
    }
  }
}
//#define CEBUGF2(bla)


/*************************************************************************
 *
 * append b to a if b != a;
 * if a was not empty, also add a concat string inbetween
 *
 *************************************************************************/

void Contig::myappend(string & a, const string & b, const string & concatstring) const
{
  if(!b.empty() && b != a) {
    if(!a.empty()) a+=concatstring;
    a+=b;
  }
}


/*************************************************************************
 *
 * GenBank annotations are sometimes scattered across different identifiers
 *  of a locus. Starting from a known locus, this function collects
 *  all information for gene, function, ecnumber, product and note
 * 
 * Expects the list to be sorted
 *
 *************************************************************************/

void Contig::concatAllGBFInfoForLocus(const list<gbfsummary_t> & allGBfeatures, list<gbfsummary_t>::const_iterator gbelementI, const string & concatstring, string & gene, string & function, string & ecnumber, string & product, string & note) const
{
  gene.clear();
  function.clear();
  ecnumber.clear();
  product.clear();
  note.clear();

  if(gbelementI == allGBfeatures.end()) return;

//  list<gbfsummary_t>::const_iterator gbfsI=gbelementI;
//  // search first element with that locus
//  while (gbfsI != allGBfeatures.begin() && gbfsI->locustag == gbelementI->locustag) gbfsI--;
//  gbfsI++;
//
//  for(; gbfsI != allGBfeatures.end() && gbfsI->locustag == gbelementI->locustag; gbfsI++){
//    myappend(gene, gbfsI->gene, concatstring);
//    myappend(function, gbfsI->function, concatstring);
//    myappend(ecnumber, gbfsI->ecnumber, concatstring);
//    myappend(product, gbfsI->product, concatstring);
//    myappend(note, gbfsI->note, concatstring);
//  }


  list<gbfsummary_t>::const_iterator gbfsI=gbelementI;
  // search first element with that locus
  while (gbfsI != allGBfeatures.begin() && gbfsI->cfrom >= gbelementI->cfrom) gbfsI--;
  gbfsI++;
  list<gbfsummary_t>::const_iterator gbfsE=gbfsI;
  while (gbfsE != allGBfeatures.end() && gbfsE->cfrom <= gbelementI->cto) gbfsE++;

  for(; gbfsI != gbfsE; gbfsI++){
    if(gbfsI->locustag == gbelementI->locustag
      && gbfsI->cfrom == gbelementI->cfrom
      && gbfsI->cto == gbelementI->cto) {
      myappend(gene, gbfsI->gene, concatstring);
      myappend(function, gbfsI->function, concatstring);
      myappend(ecnumber, gbfsI->ecnumber, concatstring);
      myappend(product, gbfsI->product, concatstring);
      myappend(note, gbfsI->note, concatstring);
    }
  }

  return;
}


/*************************************************************************
 *
 * Helper functions to iterate through the contig in forward direction
 *  and always know which reads cover which part (fast)
 *
 * Warning: after initialising the rcci, never ever perform contig
 *  changing operations, this will invalidate the rcci!
 *
 *************************************************************************/

void Contig::readColContigIteratorInit(rcci_t & rcci, const vector<int32> & allowedstrainids, const vector<uint8> & allowedreadtypes, bool takerails, bool takebackbones, bool takereadswithoutreadpool)
{
  FUNCSTART("void Contig::readColContigIteratorInit(rcci_t & rcci, const vector<int32> & allowedstrainids, const vector<uint8> & allowedreadtypes, bool takerails, bool takebackbones, bool takereadswithoutreadpool)");

  rcci.actcontigpos=0;
  rcci.allowedstrainids=allowedstrainids;
  rcci.allowedreadtypes=allowedreadtypes;
  rcci.takerails=takerails;
  rcci.takebackbones=takebackbones;
  rcci.takereadswithoutreadpool=takereadswithoutreadpool;

  // Finalising the contig initialises the output order structure vector
  finalise();

  // a vector (read_ids_in_col) keeps the ids of the reads that are
  //  covering a specific position of the contig
  // reserving 1000 position should be enough for 99.9% of all cases,
  //  but is automatically extended by STL if needed.
  rcci.read_ids_in_col.clear();
  rcci.read_ids_in_col.reserve(1000);

  rcci.outorderI = CON_outputorder.begin();

  readColContigIteratorUpdate(rcci);

  FUNCEND();
}

void Contig::readColContigIteratorUpdate(rcci_t & rcci)
{
  FUNCSTART("void Contig::readColContigIteratorUpdate(rcci_t & rcci)");

  // update the vector that 
  //  keeps track of the reads that are
  //  covering a specific position of the contig
  // works quite simple: reads in the vector that are ending get
  //  thrown out, new reads starting are entered.

  // first delete those who fall out at this new position
  vector<int32>::iterator Ifrom=rcci.read_ids_in_col.begin();
  vector<int32>::iterator Ito=Ifrom;
  for(;Ifrom != rcci.read_ids_in_col.end(); Ifrom++){
    *Ito=*Ifrom;
    if(CON_reads[*Ifrom].offset+CON_reads[*Ifrom].read.getLenClippedSeq() > rcci.actcontigpos) {
      Ito++;
    }
  }
  if(Ito != Ifrom) {
    rcci.read_ids_in_col.resize(Ito-rcci.read_ids_in_col.begin());
  }
  
  // now insert ids of reads that have newly started at this position
  // Don't take railreads or backbones on demand
  for(;rcci.outorderI != CON_outputorder.end() && (rcci.outorderI)->offset_start == static_cast<int32>(rcci.actcontigpos); (rcci.outorderI)++){
    bool takeit=true;
    if(CON_reads[(rcci.outorderI)->original_index].read.isRail()){ 
      if(!rcci.takerails) takeit=false;
    }else if(CON_reads[(rcci.outorderI)->original_index].read.isBackbone()){
      if(!rcci.takebackbones) takeit=false;
    }else if(!rcci.allowedstrainids.empty()){
      takeit=false;
      vector<int32>::const_iterator I=rcci.allowedstrainids.begin();
      for(;I!=rcci.allowedstrainids.end(); I++){
	if(CON_reads[(rcci.outorderI)->original_index].read.getStrainID() == *I){
	  takeit=true;
	  break;
	}
      }
    }else if(!rcci.allowedreadtypes.empty()){
      takeit=false;
      vector<uint8>::const_iterator I=rcci.allowedreadtypes.begin();
      for(;I!=rcci.allowedreadtypes.end(); I++){
	if(CON_reads[(rcci.outorderI)->original_index].read.getSequencingType() == *I){
	  takeit=true;
	  break;
	}
      }
    }
    if(takeit){
      rcci.read_ids_in_col.push_back((rcci.outorderI)->original_index);
    }
  }

  FUNCEND();
  return;
}

void Contig::readColContigIteratorAdvance(rcci_t & rcci)
{
  rcci.actcontigpos++;
  readColContigIteratorUpdate(rcci);
}


///*************************************************************************
// *
// * Simply go through all reads and delet tags with specified tagid
// *
// *
// *************************************************************************/
//
//void Contig::deleteTagsInReads(const string & identifier)
//{
//  for(uint32 i=0; i<CON_reads.size(); i++){
//    CON_reads[i].read.deleteTag(identifier);
//  }
//  return;
//}


/*************************************************************************
 *
 * Simply go through all reads and delet tags with specified tagid
 *
 *
 *************************************************************************/

void Contig::deleteTagsInReads(const multitag_t::mte_id_t identifier)
{
  for(uint32 i=0; i<CON_reads.size(); i++){
    CON_reads[i].read.deleteTag(identifier);
  }
  return;
}



/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Contig::setupAsBackBoneContig()
{
  // first, trash all consensus caches so that MIRA uses newly computed consensus
  //  sequences later on. Especially important when working with several strains
  trashConsensusCache(false);

  const string & strainname=(*CON_miraparams)[0].getAssemblyParams().as_backbone_strainname;

  vector<Contig::contigread_t>::iterator J=CON_reads.begin();
  for(; J!=CON_reads.end(); ++J){
    if(J->orpid==-1) continue;
    
    CON_readpool->getRead(J->orpid).setBackbone(true);
    J->read.setBackbone(true);

    //Read::setCoutType(Read::AS_TEXTSHORT);
    //cout << AS_readpool.getRead(J->orpid);
    
    if((*CON_miraparams)[0].getAssemblyParams().as_backbone_strainname_forceforall
       || CON_readpool->getRead(J->orpid).getStrain().empty()){
      CON_readpool->getRead(J->orpid).setStrain(strainname.c_str());
      J->read.setStrain(strainname.c_str());
    }
  }


  // setup quick lookup for -CO:msrkceu (keep contig ends unmerged)
  int32 leftestoff=0xffffffff;
  int32 rightestoff=-1;

  CON_index_leftestbbread=0;
  CON_index_rightestbbread=0;

  J=CON_reads.begin();
  for(uint32 ri=0; J!=CON_reads.end(); ++J, ++ri){
    if(J->offset + J->read.getLenClippedSeq() > rightestoff) {
      rightestoff=J->offset + J->read.getLenClippedSeq();
      CON_index_rightestbbread=ri;
    }
    if(J->offset < leftestoff) {
      leftestoff=J->offset;
      CON_index_leftestbbread=ri;
    }
  }


  // setup data for areas in genome were we want to force merge
  // how memory-inefficient having it done by buildMaskShadow()
  //  instead of writing directly to CON_counts
  // but I don't care atm
  CON_hasforcemergeareas=false;
  {
    cccontainer_t::iterator ccI=CON_counts.begin();
    for(; ccI != CON_counts.end(); ++ccI){
      ccI->forcemergearea=0;
    }

    vector<int8> maskshadow;
    vector<multitag_t::mte_id_t> masktagstrings;
    masktagstrings.push_back(Read::REA_tagentry_idMFSM);

    buildMaskShadow(maskshadow,masktagstrings,true);
    ccI=CON_counts.begin();
    vector<int8>::const_iterator mI=maskshadow.begin();
    for(; ccI != CON_counts.end(); ++ccI, ++mI){
      ccI->forcemergearea=*mI;
      CON_hasforcemergeareas|=*mI;
    }
  }

  return;
}



/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Contig::blindContig()
{
  FUNCSTART("void Contig::blindContig()");

  definalise();

  vector<contigread_t>::iterator cI=CON_reads.begin();
  for(; cI != CON_reads.end(); ++cI){
    if(cI->direction>0){
      cI->read.blindSeqData('c');
    }else{
      cI->read.blindSeqData('g');
    }
  }

  FUNCEND();
}
