/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */


// TODO: mutable?

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */

#include <mira/read.H>


//#define PARANOIABUGTRACKFLAG
#ifdef PARANOIABUGTRACKFLAG
#define paranoiaBUGIF(ifcond, statement) { if(ifcond) {statement;}}
#else
#define paranoiaBUGIF(ifcond, statement)
#endif


const Read::bposhashstat_t Read::REA_bposhashstat_default;

StringContainer<uint32> Read::REA_sc_readname("Read:: read name");

StringContainer<uint8> Read::REA_sc_scf_pathname("Read:: scf path name");
StringContainer<uint8> Read::REA_sc_exp_pathname("Read:: exp path name");

StringContainer<uint8> Read::REA_sc_machine_type("Read:: machine type");
StringContainer<uint8> Read::REA_sc_primer("Read:: primer");
StringContainer<uint8> Read::REA_sc_strain("Read:: strain");
StringContainer<uint8> Read::REA_sc_basecaller("Read:: base caller");
StringContainer<uint8> Read::REA_sc_dye("Read:: dye");
StringContainer<uint8> Read::REA_sc_processstatus("Read:: process status");

StringContainer<uint16> Read::REA_sc_clonevec_name("Read:: clone vector name");
StringContainer<uint16> Read::REA_sc_seqvec_name("Read:: sequencing vector name");

StringContainer<uint32> Read::REA_sc_asped("asped");

vector<Read::insizelib_t> Read::REA_insize_staticlib;





const vector<string> Read::REA_namesofseqtypes;
const vector<string> Read::REA_shortnamesofseqtypes;


// initialise the filename logic
// the read expects the name of the EXP file from which it gets 
//  the SCF and CAF names

const char Read::REA_zerostring=0;
uint8      Read::REA_cut_givenname=0;
string Read::REA_add_4expfn=".exp";
string Read::REA_add_4caffn=".caf";
string Read::REA_add_4scffn=".scf";

uint8 Read::REA_outtype=AS_TEXT;



/*************************************************************************
 *
 * a static class variables initialised by static class initialiser
 * 
 *
 *************************************************************************/

vector<double> Read::REA_bqual2errorrate;
// keep this last!
const bool Read::REA_initialisedstatics=Read::staticInitialiser();


void Read::foolCompiler()
{
#include "stdinc/foolcompiler.C"
}



/*************************************************************************
 *
 * a static class initialiser
 *
 * initialises a few static variables / arrays _before_ main is called
 *
 *
 *************************************************************************/

bool Read::staticInitialiser()
{
  if(getNumSequencingTypes()>6) {
    cerr << "Internal error: more than 6 readtypes defined in class Read!\nNeed to rework Contig::makeIntelligentConsensus()\n";
    abort();
  }

  REA_bqual2errorrate.clear();
  REA_bqual2errorrate.reserve(101);
  REA_bqual2errorrate.push_back(1);
  for(base_quality_t i=1; i<=100; i++){
    REA_bqual2errorrate.push_back(qualityToErrorRate_compute(i));
  }

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  // ugly ugly ugly
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("Sanger");
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("454");
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("IonTor");
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("PacBio");
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("Solexa");
  const_cast<vector<string> *>(&REA_namesofseqtypes)->push_back("Solid");

  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("san");
  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("454");
  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("ion");
  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("pbs");
  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("sxa");
  const_cast<vector<string> *>(&REA_shortnamesofseqtypes)->push_back("sid");


  // initialise library with insert sizes
  REA_insize_staticlib.reserve(255);
  REA_insize_staticlib.resize(1);
  REA_insize_staticlib.back().insize_from=-1;
  REA_insize_staticlib.back().insize_to=-1;

  return true;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// Plain vanilla constructor
Read::Read()
{
  FUNCSTART("Read::Read()");

  // Hmmm, do what?
  zeroVars();

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::zeroVars()
{
  //nukeSTLContainer(REA_name);
  nukeSTLContainer(REA_scf_filename);
  //nukeSTLContainer(REA_exp_filename);
  //nukeSTLContainer(REA_caf_filename);
  
  nukeSTLContainer(REA_padded_sequence);
  nukeSTLContainer(REA_padded_complementsequence);

  REA_ps_dirty=false;
  REA_pcs_dirty=false;

  REA_has_quality=false;
  REA_has_basehashstats=false;
  REA_has_freqavg=false;
  REA_has_freqrept=false;

  REA_scf_available=false;
  REA_scf_loadattempted=false;

  REA_name_scheme_valid=false;

  REA_has_valid_data=false;

  REA_used_in_assembly=false;
  REA_isbackbone=false;
  REA_israil=false;
  REA_iscoverageequivalentread=false;

  REA_uses_adjustments=true;
  //REA_uses_adjustments=false;

  REA_seqtype=SEQTYPE_SANGER;
  REA_readnaming_scheme=SCHEME_SANGER;
  REA_template_end='N';

  nukeSTLContainer(REA_qualities);
  nukeSTLContainer(REA_adjustments);
  nukeSTLContainer(REA_bposhashstats);
  nukeSTLContainer(REA_tags);

  REA_ql=0;
  REA_sl=0;
  REA_cl=0;
  REA_ml=0;

  REA_qr=0;
  REA_sr=0;
  REA_cr=0;
  REA_mr=0;

  //REA_leftclip=0;
  //REA_rightclip=0;
  //REA_len_clipped=0;

  REA_insize_libid=0;
  //REA_stadenid=-1;
  REA_templateid=-1;
  REA_templatepartnerid=-1;
  REA_templatebuilddirection=false;
  REA_strainid=0;

  nukeSTLContainer(REA_template);
  //nukeSTLContainer(REA_clonevec_name);
  //nukeSTLContainer(REA_seqvec_name);
  //nukeSTLContainer(REA_strain);
  //nukeSTLContainer(REA_basecaller);
  //nukeSTLContainer(REA_stolen);
  //nukeSTLContainer(REA_asped);
  //nukeSTLContainer(REA_dye);
  //nukeSTLContainer(REA_processstatus);
  //nukeSTLContainer(REA_primer);
  //nukeSTLContainer(REA_machine_type);
  //nukeSTLContainer(REA_scf_pathname);
  //nukeSTLContainer(REA_exp_pathname);
  //nukeSTLContainer(REA_scf_fullpathname);
  //nukeSTLContainer(REA_exp_fullpathname);

  REA_nameentry=REA_sc_readname.emptyEntry();
  REA_asped=REA_sc_asped.emptyEntry();
  REA_clonevec_name=REA_sc_clonevec_name.emptyEntry(); 
  REA_seqvec_name=REA_sc_seqvec_name.emptyEntry();   
		                        
  REA_machine_type=REA_sc_machine_type.emptyEntry();  
  REA_primer=REA_sc_primer.emptyEntry();	     
  REA_strain=REA_sc_strain.emptyEntry();	     
  REA_basecaller=REA_sc_basecaller.emptyEntry();    
  REA_dye=REA_sc_dye.emptyEntry();	     
  REA_processstatus=REA_sc_processstatus.emptyEntry(); 
		                        
  REA_scf_pathname=REA_sc_scf_pathname.emptyEntry();  
  REA_exp_pathname=REA_sc_exp_pathname.emptyEntry();  
  

  //REA_template_seqtry=" ";
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

Read::~Read()
{
  FUNCSTART("Read::~Read()");

  discard();

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// Copy constructor
//  no discard needed as this object will be freshly created when
//  called through this constructor
Read::Read(Read const &other)
{
  FUNCSTART("Read::Read(Read const &other)");
  
  zeroVars();
  *this=other;                               // call the copy operator

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpStringContainerStats(ostream & ostr)
{
  REA_sc_readname.status(ostr);
  REA_sc_scf_pathname.status(ostr);
  REA_sc_exp_pathname.status(ostr);
  REA_sc_machine_type.status(ostr);
  REA_sc_primer.status(ostr);
  REA_sc_strain.status(ostr);
  REA_sc_basecaller.status(ostr);
  REA_sc_dye.status(ostr);
  REA_sc_processstatus.status(ostr);
  REA_sc_clonevec_name.status(ostr);
  REA_sc_seqvec_name.status(ostr);
  REA_sc_asped.status(ostr);
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
const char * Read::sanityCheck() const
{
  if(getName().empty()) return NULL;

  if(REA_seqtype >= getNumSequencingTypes()) return "Undefined read type.";

  if(REA_has_valid_data==false) return "Read has no valid data?";

  if(REA_ps_dirty==true && REA_pcs_dirty==true) return "REA_ps_dirty and REA_pcs_dirty both true?";
  if(REA_ps_dirty==false && REA_pcs_dirty==false){
    if(REA_padded_sequence.size()!=REA_padded_complementsequence.size())
      return "Sizes of forward and complement padded differ.";
  }
  if(REA_ql<0) return "REA_ql<0 ?";
  if(REA_cl<0) return "REA_cl<0 ?";
  if(REA_sl<0) return "REA_sl<0 ?";
  if(REA_ml<0) return "REA_ml<0 ?";
  // do not call getLenSeq() ... recursion!
  int32 actlen;
  if(REA_ps_dirty==false){
    actlen=static_cast<int32>(REA_padded_sequence.size());
  }else{
    actlen=static_cast<int32>(REA_padded_complementsequence.size());
  }
  if(REA_cr>actlen+1) return "REA_cr > size() ?";
  if(REA_qr>actlen+1) return "REA_qr > size() ?";
  if(REA_sr>actlen+1) return "REA_sr > size() ?";
  if(REA_mr>actlen+1) return "REA_mr > size() ?";

  //if(REA_leftclip != max(REA_ql, REA_sl)) return "REA_leftclip!=max(REA_ql, REA_sl) ?";
  //if(REA_rightclip != min(REA_qr, REA_sr)) return "REA_rightclip!=min(REA_qr, REA_sr) ?";
  //if(REA_len_clipped != REA_rightclip-REA_leftclip){
  //  if(REA_rightclip-REA_leftclip>0) return "REA_len_clipped != REA_rightclip-REA_leftclip ?";
  //}

  if(REA_uses_adjustments
     && static_cast<int32>(REA_adjustments.size()) != actlen) return "REA_adjustments != expected size?";
  if(static_cast<int32>(REA_qualities.size()) != actlen) return "REA_qualities != expected size?";
  if(static_cast<int32>(REA_bposhashstats.size()) != actlen) return "REA_bposhashstats != expected size?";

  switch(REA_template_end) {
  case 'F':
  case 'R':
  case 'N':{
    break;
  }
  default : {
    return "REA_template_end is not F/R/N ?";
  }
  }

  if(REA_seqtype>=getNumSequencingTypes()){
    return "Illegal sequencing type?";
  }
  // TODO: configure this only when bughunting
#if 0
  {
    vector<multitag_t>::const_iterator T=REA_tags.begin();
    for(;T!=REA_tags.end();T++){
      if(T->from > actlen) {
	T->dump();
	return "From > actlen?";
      }
      if(T->to > actlen) {
	T->dump();
	return "To > actlen?";
      }
    }
  }
#endif

  return NULL;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
const char * Read::checkRead() const
{
  //  return NULL;


  if(sanityCheck()) {
    cout << "Sanity check of read '" << getName() << "' failed" << endl;
    return sanityCheck();
  }

#ifdef PARANOIABUGTRACKFLAG
  {
    vector<char>::const_iterator I;
    for(I=REA_padded_sequence.begin(); I!=REA_padded_sequence.end(); I++){
      if(!dptools::isValidStarBase(*I)){
	cout << "Invalid base: " << *I << "\t(" << static_cast<uint16>(*I) << ")" << endl;
	cout << "At pos: " << I-REA_padded_sequence.begin() << endl;
	return "Invalid base in padded sequence!";
      }
    }
    for(I=REA_padded_complementsequence.begin(); I!=REA_padded_complementsequence.end(); I++){
      if(!dptools::isValidStarBase(*I)){
	cout << "Invalid base: " << *I << "\t(" << static_cast<uint16>(*I) << ")" << endl;
	cout << "At pos: " << I-REA_padded_complementsequence.begin() << endl;
	return "Invalid base in padded complement sequence!";
      }
    }
  }

  {
    vector<base_quality_t>::const_iterator I;
    for(I=REA_qualities.begin(); I!=REA_qualities.end(); I++){
      if(*I>100){
	cout << "Invalid quality: " << static_cast<uint16>(*I) << endl;
	cout << "At pos: " << I-REA_qualities.begin() << endl;
	return "Invalid quality value!";
      }
    }
  }

  if(REA_uses_adjustments){
    vector<int32>::const_iterator I;
    for(I=REA_adjustments.begin(); I!=REA_adjustments.end(); I++){
      if(*I<-1 || *I >= static_cast<int32>(REA_qualities.size())){
	cout << "Invalid adjustment: " << *I << endl;
	cout << "At pos: " << I-REA_adjustments.begin() << endl;
	return "Invalid adjustment value!";
      }
    }
  }else{
    cout << "Does not use adjustments.\n";
  }
#endif

  return NULL;
}


void Read::integrityCheck() const 
{
  FUNCSTART("void Read::integrityCheck() const ");

  if(REA_has_valid_data==false){
    MIRANOTIFY(Notify::INTERNAL, "Read " << getName() << " has no valid data?");
  }

  if(checkRead()){
    throw Notify(Notify::INTERNAL, THISFUNC, checkRead());
  }
  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::reserve(uint32 lentoreserve) 
{
  REA_padded_sequence.reserve(lentoreserve);
  REA_padded_complementsequence.reserve(lentoreserve);
  REA_qualities.reserve(lentoreserve);
  REA_bposhashstats.reserve(lentoreserve);
  if(REA_uses_adjustments) REA_adjustments.reserve(lentoreserve);
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
size_t Read::getWastage() const
{
  size_t w=REA_padded_sequence.capacity()-REA_padded_sequence.size();
  w+=REA_padded_complementsequence.capacity()-REA_padded_complementsequence.size();
  w+=REA_qualities.capacity()-REA_qualities.size();
  w+=sizeof(bposhashstat_t)*(REA_bposhashstats.capacity()-REA_bposhashstats.size());
  if(REA_adjustments.capacity()) {
    w+=4*(REA_adjustments.capacity()-REA_adjustments.size());
  }
  return w;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
size_t Read::capacity() const
{
  return REA_bposhashstats.capacity();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::moderateContainerGrowth() 
{
  size_t divval=10; // == 10%
  size_t minim=5;

  if(REA_padded_sequence.capacity()==REA_padded_sequence.size()){
    REA_padded_sequence.reserve(
      max(minim,
	  REA_padded_sequence.size()+REA_padded_sequence.size()/divval)
      );
  }
  if(REA_padded_complementsequence.capacity()==REA_padded_complementsequence.size()){
    REA_padded_complementsequence.reserve(
      max(minim,
	  REA_padded_complementsequence.size()+REA_padded_complementsequence.size()/divval)
      );
  }
  if(REA_qualities.capacity()==REA_qualities.size()){
    REA_qualities.reserve(
      max(minim,
	  REA_qualities.size()+REA_qualities.size()/divval)
      );
  }
  if(REA_uses_adjustments){
    if(REA_adjustments.capacity()==REA_adjustments.size()){
      REA_adjustments.reserve(
	max(minim,
	    REA_adjustments.size()+REA_adjustments.size()/divval)
	);
    }
  }

  if(REA_bposhashstats.capacity()==REA_bposhashstats.size()){
    REA_bposhashstats.reserve(
      max(minim,
	  REA_bposhashstats.size()+REA_bposhashstats.size()/divval)
      );
  }

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// Copy operator, needed by copy-constructor
Read const & Read::operator=(Read const & other)
{
  FUNCSTART("Read const & Read::operator=(Read const & other)");

  if(this != &other){
    TRACE("copying\n");
    discard();
    
    REA_uses_adjustments=other.REA_uses_adjustments;

    //uint32 lentoreserve=0;
    //if(REA_ps_dirty==false){
    //  lentoreserve=other.REA_padded_sequence.size();
    //} else if(REA_pcs_dirty==false){
    //  lentoreserve=other.REA_padded_complementsequence.size();
    //}
    //reserve(lentoreserve+lentoreserve/10);

    //REA_name=        other.REA_name;
    REA_nameentry=        other.REA_nameentry;

    REA_scf_filename=other.REA_scf_filename;

    //REA_exp_filename=other.REA_exp_filename;
    //REA_caf_filename=other.REA_caf_filename;
    
    REA_scf_pathname=    other.REA_scf_pathname;	   
    REA_exp_pathname=	   other.REA_exp_pathname;	   
    
    REA_ps_dirty=other.REA_ps_dirty;
    if(REA_ps_dirty==false){
      REA_padded_sequence=other.REA_padded_sequence;
    }
    REA_pcs_dirty=other.REA_pcs_dirty;
    if(REA_pcs_dirty==false){
      REA_padded_complementsequence=other.REA_padded_complementsequence;
    }
      
    REA_qualities=other.REA_qualities;

    REA_adjustments=other.REA_adjustments;
    REA_bposhashstats=other.REA_bposhashstats;
      
    REA_ql=other.REA_ql;
    REA_qr=other.REA_qr;
    REA_sl=other.REA_sl;
    REA_sr=other.REA_sr;
    REA_cl=other.REA_cl;
    REA_cr=other.REA_cr;
    REA_ml=other.REA_ml;
    REA_mr=other.REA_mr;

    //REA_leftclip=other.REA_leftclip;
    //REA_rightclip=other.REA_rightclip;
    //REA_len_clipped=other.REA_len_clipped;
      
    REA_tags=other.REA_tags;
      
    REA_has_quality=other.REA_has_quality;
    REA_has_basehashstats=other.REA_has_basehashstats;
    REA_has_freqavg=other.REA_has_freqavg;
    REA_has_freqrept=other.REA_has_freqrept
;
    REA_scf_available=other.REA_scf_available;
    REA_scf_loadattempted=other.REA_scf_loadattempted;
    REA_name_scheme_valid=other.REA_name_scheme_valid;
    REA_has_valid_data=other.REA_has_valid_data;
    REA_used_in_assembly=other.REA_used_in_assembly;
    REA_isbackbone=other.REA_isbackbone;
    REA_israil=other.REA_israil;
    REA_iscoverageequivalentread=other.REA_iscoverageequivalentread;
    REA_seqtype=other.REA_seqtype;

    REA_readnaming_scheme=other.REA_readnaming_scheme;

    REA_insize_libid=other.REA_insize_libid;
    //REA_stadenid=other.REA_stadenid;
    REA_templateid=other.REA_templateid;
    REA_templatepartnerid=other.REA_templatepartnerid;
    REA_template_end=other.REA_template_end;
    REA_templatebuilddirection=other.REA_templatebuilddirection;
    REA_strainid=other.REA_strainid;

    REA_seqvec_name=other.REA_seqvec_name;
    REA_clonevec_name=other.REA_clonevec_name;
    REA_template=other.REA_template;
    REA_strain=other.REA_strain;
    REA_basecaller=other.REA_basecaller;
    //REA_stolen=other.REA_stolen;
    REA_asped=other.REA_asped;
    REA_dye=other.REA_dye;
    REA_processstatus=other.REA_processstatus;
    REA_primer=other.REA_primer;
    REA_machine_type=other.REA_machine_type;
  }

  FUNCEND();
  return *this;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// Copy operator, needed by copy-constructor
size_t Read::estimateMemoryUsage() const
{
  FUNCSTART("size_t Read::estimateMemoryUsage()");

  size_t components=0;

  size_t cnum,cbytes,freecap,clba;
  
  //components+=estimateMemoryUsageOfContainer(REA_name	  ,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_scf_filename,false,cnum,cbytes,freecap,clba);
  //components+=estimateMemoryUsageOfContainer(REA_exp_filename,false,cnum,cbytes,freecap,clba);
  //components+=estimateMemoryUsageOfContainer(REA_caf_filename,false,cnum,cbytes,freecap,clba);
  
  components+=estimateMemoryUsageOfContainer(REA_padded_sequence,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_padded_complementsequence,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_qualities,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_adjustments,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_bposhashstats,false,cnum,cbytes,freecap,clba);
  components+=estimateMemoryUsageOfContainer(REA_tags,false,cnum,cbytes,freecap,clba);

  FUNCEND();
  return components;
}




void Read::setCoutType(uint8 type)
{
  switch(type){
  case AS_TEXT: 
  case AS_TEXTSHORT: 
  case AS_TEXTCLIPS: 
  case AS_CAF: 
  case AS_MAF: 
  case AS_ACE: 
  case AS_ACE_COMPLEMENT: 
  case AS_FASTA: 
  case AS_FASTQ: 
  case AS_GAP4DA: 
  case AS_CLIPPEDFASTA: 
  case AS_SEQVECMASKEDFASTA: 
  case AS_MASKEDMASKFASTA:
  case AS_FASTAQUAL: 
  case AS_CLIPPEDFASTAQUAL: 
  case AS_SEQVECMASKEDFASTAQUAL:
  case AS_MASKEDMASKFASTAQUAL: {
    REA_outtype=type;
    break;
  }
  default:{
    throw Notify(Notify::INTERNAL, "void Read::setCoutType(uint8 type)", "Wrong type is not one of TEXT or CAF.");
  }
  }

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
ostream & operator<<(ostream &ostr, Read const &read)
{
  FUNCSTART("friend ostream & Read::operator<<(ostream &ostr, const  &read)");

  BUGSTAT(read.checkRead());

//  if(read.REA_outtype==Read::AS_CAF){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsCAF(ostr);
//    FUNCEND();
//    return ostr;
//  } else if(read.REA_outtype==Read::AS_MASKEDFASTA){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsFASTA(ostr,false,true);
//    FUNCEND();
//    return ostr;
//  } else if(read.REA_outtype==Read::AS_CLIPPEDFASTA){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsFASTA(ostr,true, false);
//    FUNCEND();
//    return ostr;
//  } else if(read.REA_outtype==Read::AS_FASTA){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsFASTA(ostr,false, false);
//    FUNCEND();
//    return ostr;
//  } else if(read.REA_outtype==Read::AS_ACE){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsACE(ostr, 1);
//    FUNCEND();
//    return ostr;
//  } else if(read.REA_outtype==Read::AS_ACE_COMPLEMENT){
//    Read & nonconstread = const_cast<Read &>(read);
//    nonconstread.dumpAsACE(ostr, -1);
//    FUNCEND();
//    return ostr;
//  }

  bool stopit=true;
  switch(read.REA_outtype) {
  case Read::AS_CAF : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsCAF(ostr);
    break;
  }
  case Read::AS_MAF : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsMAF(ostr);
    break;
  }
  case Read::AS_BAF : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsBAF(ostr);
    break;
  }
  case Read::AS_FASTA : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTA(ostr,false, false, false);
    break;
  }
  case Read::AS_FASTQ : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTQ(ostr,false, false, false);
    break;
  }
  case Read::AS_SEQVECMASKEDFASTA : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTA(ostr,false,true, false);
    break;
  }
  case Read::AS_MASKEDMASKFASTA : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTA(ostr,false,false, true);
    break;
  }
  case Read::AS_CLIPPEDFASTA : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTA(ostr,true, false, false);
    break;
  }
  case Read::AS_FASTAQUAL : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTAQual(ostr,false, false, false);
    break;
  }
  case Read::AS_SEQVECMASKEDFASTAQUAL : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTAQual(ostr,false,true, false);
    break;
  }
  case Read::AS_MASKEDMASKFASTAQUAL : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTAQual(ostr,false,false, true);
    break;
  }
  case Read::AS_CLIPPEDFASTAQUAL : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsFASTAQual(ostr,true, false, false);
    break;
  }
  case Read::AS_GAP4DA : {
    Read & nonconstread = const_cast<Read &>(read);
    string gna;
    nonconstread.dumpAsGAP4DA(ostr, gna, true);
    break;
  }
  case Read::AS_ACE : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsACE(ostr, 1);
    break;
  }
  case Read::AS_ACE_COMPLEMENT : {
    Read & nonconstread = const_cast<Read &>(read);
    nonconstread.dumpAsACE(ostr, -1);
    break;
  }
  default: {
    stopit=false;
  }
  }

  if(stopit) {
    FUNCEND();
    return ostr;
  }




  ostr << dec;

  ostr << "\nName: " << read.getName();
  ostr << "\nql: " << read.REA_ql << "\tqr: " << read.REA_qr;
  ostr << "\nsl: " << read.REA_sl << "\tsr: " << read.REA_sr;
  ostr << "\ncl: " << read.REA_cl << "\tcr: " << read.REA_cr;
  ostr << "\nml: " << read.REA_ml << "\tmr: " << read.REA_mr;
  ostr << "\nLeftclip: " << read.getLeftClipoff() << "\tRightclip: " << read.getRightClipoff();
  ostr << "\tLen: " << read.getLenClippedSeq() << " (" << (read.getRightClipoff()-read.getLeftClipoff()) << ")\tLenSeq: " << read.getLenSeq() << '\n';
  ostr << "\nLeftextend: " << read.getLeftExtend() << "\tRightextend: " << read.getRightExtend();
  ostr << "\nTemplate end: " << read.REA_template_end;
  ostr << "\tTemplate: ";
  if(!read.REA_template.empty()){
    ostr << read.REA_template;
  }
  ostr << "\tInt. Tname: " << const_cast<Read &>(read).getInternalTemplateName();
  ostr << "\nT-ID: " << read.REA_templateid;
  ostr << "\tTPartner-ID: " << read.REA_templatepartnerid << endl;
  ostr << "\nStrain: " << read.REA_sc_strain.getEntry(read.REA_strain);

  ostr << "\tS-ID: " << static_cast<int32>(read.REA_strainid) << endl;
  ostr << "Seqtype: " << read.getShortNameOfSequencingType(read.REA_seqtype) << endl;
  ostr << "IsBB: " << read.isBackbone()
       << "\tIsRail: " << read.isRail()
       << "\tIsCER: " << read.isCoverageEquivalentRead() << endl;

  ostr << "\nTags:\n";
  {
    Read & nonconstread=const_cast<Read &>(read);
    nonconstread.sortTags();
  }
  for(uint32 i=0; i < read.getNumOfTags(); i++) {
    const multitag_t & acttag=read.getTag(i);
    ostr << "Tag " << i << ":\t" << acttag.getIdentifierStr() << ' ' << acttag.from << ' ' << acttag.to;
    ostr << " \"" << acttag.getCommentStr() << "\"\n";	
  }

  if(read.REA_outtype!=Read::AS_TEXTCLIPS
    && read.REA_outtype!=Read::AS_TEXTSHORT){

    if(read.REA_ps_dirty==false){
      ostr << "\n\nRead size padded: " << read.REA_padded_sequence.size();
      ostr << "\nRead padded sequence:\n";
      {
	Read & nonconstread=const_cast<Read &>(read);
	const vector<char> & seq=nonconstread.getActualSequence();
	vector<char>::const_iterator I=seq.begin();
	while(I!=read.REA_padded_sequence.end()) ostr << *I++;
	ostr << endl;
      }
    }else{
      ostr << "Forward padded dirty.\n";
    }
    
    
    if(read.REA_pcs_dirty==false){
      ostr << "\n\nRead complement size padded: " << read.REA_padded_complementsequence.size();
      ostr << "\nRead padded complement sequence:\n";
      {
	Read & nonconstread=const_cast<Read &>(read);
	const vector<char> & seq=nonconstread.getActualComplementSequence();
	vector<char>::const_iterator I=seq.begin();
	while(I!=read.REA_padded_complementsequence.end()) ostr << *I++;
	ostr << endl;
      }
    }else{
      ostr << "Complement padded dirty.\n";
    }
    
    if(read.REA_outtype==Read::AS_TEXT) {
      ostr << "\nRead padded sequence, adjustment, quality, baseflags:\n";
      {
	Read & nonconstread=const_cast<Read &>(read);
	const vector<char> & seq=nonconstread.getActualSequence();
	const vector<int32> & adj=read.getAdjustments();
	const vector<base_quality_t> & qual=read.getQualities();
	const vector<Read::bposhashstat_t> & bhstat=read.getBPosHashStats();
	
	vector<char>::const_iterator sI=seq.begin();
	vector<int32>::const_iterator aI=adj.begin();
	vector<base_quality_t>::const_iterator qI=qual.begin();
	vector<Read::bposhashstat_t>::const_iterator fI=bhstat.begin();
	
	uint32 actpos=0;
	for(;sI!=read.REA_padded_sequence.end(); sI++,qI++,fI++,actpos++){
	  ostr << actpos << ":\t" << *sI;
	  if(!dptools::isValidStarBase(*sI)) ostr << " inv!";
	  if(read.REA_uses_adjustments) {
	    ostr << '\t' << *aI;
	    // manually counting up aI. Not in for loop as adjustments 
	    //  may be empty
	    aI++;
	  }else{
	    ostr << "\tNoAdj";
	  }
	  ostr << '\t' << static_cast<uint16>(*qI)
	       << '\t' << *fI << '\n';
	}
      }
    }
  }

  FUNCEND();
  return ostr;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsFASTA(ostream & ostr, bool clippedonly, bool maskedseqvec, bool maskedmask)
{
  FUNCSTART("void Read::dumpAsFASTA(ostream & ostr, bool clippedonly, bool maskedseqvec)");

  if(checkRead()!=NULL || getName().empty()){
    // This is some kind of invalid read ... try to save something minimal

    if(getName().empty()){
      char buffer[24] ;
      sprintf(buffer, "%x", this) ;

      ostr << ">ivld_" << buffer << "\nn\n";
    }else{
      ostr << ">ivld_" << getName() << "\nn\n";
    }

    FUNCEND();
    return;
  }

  //cout << "f:\t" << clippedonly << maskedseqvec << maskedmask << endl;

  if(getLenSeq()>0) {
    if(!clippedonly || (clippedonly && getLenClippedSeq() > 0)){

      // TODO: u.U. die ComplSeq nutzen
      refreshPaddedSequence();
      {
	ostr << ">" << getName() << endl;
	vector<char>::const_iterator I=REA_padded_sequence.begin();
	vector<char>::const_iterator J=REA_padded_sequence.end();
	//if(clippedonly) {
	//  J=I;
	//  advance(J,getRightClipoff());
	//  advance(I,getLeftClipoff());
	//}
	int32 seqcharcount=0;
	uint32 cpl=0;
	bool dooutput;
	while(I!=J){
	  dooutput=true;
	  if(clippedonly){
	    if(seqcharcount<getLeftClipoff() || seqcharcount >= getRightClipoff()) dooutput=false;
	  }
	  if(dooutput){
	    if((maskedseqvec || maskedmask)
	       && (seqcharcount<REA_sl || seqcharcount >= REA_sr
		 || seqcharcount<REA_ml || seqcharcount >=REA_mr)) {
	      if(*I=='x'){
		ostr << "x";
	      } else {
		ostr << "X";
	      }
	    } else {
	      ostr << *I;
	    }
	    if(cpl++==59){
	      cpl=0;
	      ostr << '\n';
	    }
	  }
	  I++;
	  seqcharcount++;
	}
	if(cpl!=0) ostr << '\n';
      }
    }
  }

  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsFASTQ(ostream & ostr, bool clippedonly, bool maskedseqvec, bool maskedmask)
{
  FUNCSTART("void Read::dumpAsFASTQ(ostream & ostr, bool clippedonly, bool maskedseqvec)");

  if(checkRead()!=NULL || getName().empty()){
    // This is some kind of invalid read ... try to save something minimal

    if(getName().empty()){
      char buffer[24] ;
      sprintf(buffer, "%x", this) ;

      ostr << "@ivld_" << buffer << "\nn\n";
    }else{
      ostr << "@ivld_" << getName() << "\nn\n";
    }

    FUNCEND();
    return;
  }

  //cout << "f:\t" << clippedonly << maskedseqvec << maskedmask << endl;

  if(getLenSeq()>0) {
    if(!clippedonly || (clippedonly && getLenClippedSeq() > 0)){

      // TODO: u.U. die ComplSeq nutzen
      refreshPaddedSequence();
      {
	ostr << "@" << getName() << '\n';
	vector<char>::const_iterator I=REA_padded_sequence.begin();
	vector<char>::const_iterator J=REA_padded_sequence.end();

	int32 seqcharcount=0;
	bool dooutput;
	while(I!=J){
	  dooutput=true;
	  if(clippedonly){
	    if(seqcharcount<getLeftClipoff() || seqcharcount >= getRightClipoff()) dooutput=false;
	  }
	  if(dooutput){
	    if((maskedseqvec || maskedmask)
	       && (seqcharcount<REA_sl || seqcharcount >= REA_sr
		 || seqcharcount<REA_ml || seqcharcount >=REA_mr)) {
	      if(*I=='x'){
		ostr << "x";
	      } else {
		ostr << "X";
	      }
	    } else {
	      ostr << *I;
	    }
	  }
	  I++;
	  seqcharcount++;
	}
      }
      {
	ostr << "\n+\n";
	vector<base_quality_t>::const_iterator I=REA_qualities.begin();
	vector<base_quality_t>::const_iterator J=REA_qualities.end();

	int32 seqcharcount=0;
	bool dooutput;
	while(I!=J){
	  dooutput=true;
	  if(clippedonly){
	    if(seqcharcount<getLeftClipoff() || seqcharcount >= getRightClipoff()) dooutput=false;
	  }
	  if(dooutput){
	    ostr << static_cast<char>((*I)+33);
	  }
	  I++;
	  seqcharcount++;
	}
      }
      ostr << '\n';
    }
  }

  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsFASTAQual(ostream & ostr, bool clippedonly, bool maskedseqvec, bool maskedmask)
{
  FUNCSTART("void Read::dumpAsFASTAQual(ostream & ostr, bool clippedonly, bool maskedseqvec)");

  ostr << dec;

  if(checkRead()!=NULL || getName().empty()){
    // This is some kind of invalid read ... try to save something minimal

    if(getName().empty()){
      char buffer[24] ;
      sprintf(buffer, "%x", this) ;

      ostr << ">ivld_" << buffer << "\n0\n";
    }else{
      ostr << ">ivld_" << getName() << "\n0\n";
    }

    FUNCEND();
    return;
  }

  //cout << "fq:\t" << clippedonly << maskedseqvec << maskedmask << endl;

  if(getLenSeq()>0) {
    if(!clippedonly || (clippedonly && getLenClippedSeq() > 0)){

      {
	ostr << ">" << getName() << endl;
	vector<base_quality_t>::const_iterator I=REA_qualities.begin();
	vector<base_quality_t>::const_iterator J=REA_qualities.end();

	int32 seqcharcount=0;
	uint32 cpl=0;
	bool dooutput;
	while(I!=J){
	  dooutput=true;
	  if(clippedonly){
	    if(seqcharcount<getLeftClipoff() || seqcharcount >= getRightClipoff()) dooutput=false;
	  }
	  if(dooutput){
	    if((maskedseqvec || maskedmask)
	       && (seqcharcount<REA_sl || seqcharcount >= REA_sr
		 || seqcharcount<REA_ml || seqcharcount >=REA_mr)) {
	      ostr << "00 ";
	    } else {
	      ostr << static_cast<uint16>(*I) << ' ';
	    }
	    if(cpl++==19){
	      cpl=0;
	      ostr << '\n';
	    }
	  }
	  I++;
	  seqcharcount++;
	}
	if(cpl!=0) ostr << '\n';
      }      
    }
  }

  FUNCEND();

  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsCAF(ostream & ostr)
{
  FUNCSTART("void Read::dumpAsCAF(ostream & ostr)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if(checkRead()!=NULL || getName().empty()){
    if(getName().empty()){
      char buffer[24] ;
      sprintf(buffer, "%x", this) ;
      ostr << "Sequence : ivld_" << buffer;
    }else{
      ostr << "Sequence : ivld_" << getName();
    }
    ostr << "Is_read\n\n";
    FUNCEND();
    return;
  }

  // TODO: u.U. die ComplSeq nutzen
  refreshPaddedSequence();
  {
    ostr << "\nDNA : " << getName() << '\n';
    vector<char>::const_iterator I=REA_padded_sequence.begin();
    uint32 cpl=0;
    while(I!=REA_padded_sequence.end()){
      if(*I=='*'){
	ostr << '-';
      }else{
	ostr << *I;
      }
      if(cpl++==59){
	cpl=0;
	ostr << '\n';
      }
      I++;
    }
    ostr << '\n';
  }

#ifdef PARANOIABUGTRACKFLAG
  if(checkRead()!=NULL){
    cout << "Ouch, failed second check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }
#endif

  {
    ostr << "\nBaseQuality : " << getName() << '\n';

    vector<base_quality_t>::const_iterator I=REA_qualities.begin();
    uint32 cpl=0;
    while(I!=REA_qualities.end()){
      ostr << static_cast<uint16>(*I);
	if(cpl++==30){
	  ostr << '\n';
	  cpl=0;
	}else{
	  ostr << ' ';
	}
      I++;
    }
    ostr << "\n\n";
  }


  ostr << "Sequence : " << getName() << '\n';
  ostr << "Is_read\nPadded\n";

  if(!REA_sc_seqvec_name.getEntry(REA_seqvec_name).empty()){
    ostr << "Sequencing_vector \"" << REA_sc_seqvec_name.getEntry(REA_seqvec_name) << "\"" << '\n';
  }
  if(!REA_template.empty()){
    ostr << "Template \"" << REA_template <<  "\"" << '\n';
  }
  switch(REA_template_end){
  case 'F' : {
    ostr << "Strand Forward\n";
    break;
  }
  case 'R' : {
    ostr << "Strand Reverse\n";
    break;
  }
  case 'N' : {
    //ostr << "//template end == 'N'\n";
    // do nothing
    break;
  }
  default : {
    throw Notify(Notify::FATAL, THISFUNC, "Illegal template end?");
  }
  }
  if(REA_insize_staticlib[REA_insize_libid].insize_to!=-1 
     && REA_insize_staticlib[REA_insize_libid].insize_from!=-1){
    ostr << "Insert_size " << REA_insize_staticlib[REA_insize_libid].insize_from 
	 << ' ' << REA_insize_staticlib[REA_insize_libid].insize_to << '\n';
  }
  if(!REA_scf_filename.empty()){
    ostr << "SCF_File \"" << REA_scf_filename <<  "\"" << '\n';
  }else if(REA_seqtype==SEQTYPE_454GS20){
    ostr << "SCF_File \"" << getName() <<  "\"" << '\n';
  }

  if(!REA_sc_basecaller.getEntry(REA_basecaller).empty()){
    ostr << "Base_caller \"" << REA_sc_basecaller.getEntry(REA_basecaller) <<  "\"" << '\n';
  }
  //if(!REA_stolen.empty()){
  //  ostr << "Stolen \"" << REA_stolen <<  "\"" << '\n';
  //}
  //if(REA_stadenid!=-1){
  //  ostr << "Staden_id " << REA_stadenid <<  '\n';
  //}
  if(!REA_sc_asped.getEntry(REA_asped).empty()){
    ostr << "Asped \"" << REA_sc_asped.getEntry(REA_asped) <<  "\"" << '\n';
  }
  if(!REA_sc_processstatus.getEntry(REA_processstatus).empty()){
    ostr << "ProcessStatus \"" << REA_sc_processstatus.getEntry(REA_processstatus) <<  "\"" << '\n';
  }
  if(!REA_sc_dye.getEntry(REA_dye).empty()){
    ostr << "Dye \"" << REA_sc_dye.getEntry(REA_dye) <<  "\"" << '\n';
  }
  if(!REA_sc_primer.getEntry(REA_primer).empty()){
    ostr << "Primer " << REA_sc_primer.getEntry(REA_primer) << '\n';
  }

  
  if(REA_sl>0){
    ostr << "Seq_vec SVEC 1 " << REA_sl;
    if(!REA_sc_seqvec_name.getEntry(REA_seqvec_name).empty()) ostr << " \"" << REA_sc_seqvec_name.getEntry(REA_seqvec_name) << "\"";
    ostr << '\n';;
  }
  if(REA_sr < static_cast<int32>(getLenSeq())){
    ostr << "Seq_vec SVEC "<< REA_sr+1 <<' ' << getLenSeq();
    if(!REA_sc_seqvec_name.getEntry(REA_seqvec_name).empty()) ostr << " \"" << REA_sc_seqvec_name.getEntry(REA_seqvec_name) << "\"";
    ostr << '\n';
  }
  
  ostr << "Clipping QUAL " << REA_ql+1 << ' ' << REA_qr << '\n'; //<<"NameOfSeq\n";
  
  if(REA_cl>0){
    ostr << "Clone_vec CVEC 1 " << REA_cl;
    if(!REA_sc_clonevec_name.getEntry(REA_clonevec_name).empty()) ostr << " \"" << REA_sc_clonevec_name.getEntry(REA_clonevec_name) <<  "\"";
    ostr << '\n';
  }
  if(REA_cr < static_cast<int32>(getLenSeq())){
    ostr << "Clone_vec CVEC "<< REA_cr+1 <<' ' << getLenSeq();
    if(!REA_sc_clonevec_name.getEntry(REA_clonevec_name).empty()) ostr << " \"" << REA_sc_clonevec_name.getEntry(REA_clonevec_name) <<  "\"";
    ostr << '\n';
  }

//  if(REA_cl > 0){
//    ostr << "Clone_vec CVEC "<< REA_cl+1 <<' ' << REA_cr <<  ' ' << tmpexp.getCF() << '\n';;
//  }

  //  ostr << "// Need the align_to_scf\n";

  if(REA_uses_adjustments){
    uint32 i;
    uint32 imin=0;
    for(i=1; i< getLenSeq(); i++){
      if(REA_adjustments[i-1]!=-1){
	if(REA_adjustments[i]!=REA_adjustments[i-1]+1){
	  ostr << "Align_to_SCF " << imin+1 << ' ' << i << ' ' << REA_adjustments[imin]+1 << ' ' << REA_adjustments[i-1]+1 << '\n';
	  imin=i;
	}
      }else{
	imin=i;
      }
    }
    if(REA_adjustments[i-1]!=-1){
      ostr << "Align_to_SCF " << imin+1 << ' ' << i << ' ' << REA_adjustments[imin]+1 << ' ' << REA_adjustments[i-1]+1 << '\n';
    }
  }else{
    ostr << "Align_to_SCF " << 1 << ' ' << getLenSeq() << ' ' << 1 << ' ' << getLenSeq() << '\n';
  }


  if(!REA_tags.empty()){
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    while(I!=REA_tags.end()){
      if(I->strand=='-') {
	ostr << "Tag " << I->getIdentifierStr() << ' ' << (I->to)+1 << ' ' << (I->from)+1 << " \"" ;
      }else{
	ostr << "Tag " << I->getIdentifierStr() << ' ' << (I->from)+1 << ' ' << (I->to)+1 << " \"" ;
      }
      
      // cannot use serialiseComment() here because Quotes are bad
      // in CAFs
      const string & tmpcs=I->getCommentStr();
      for(uint32 i=0; i< tmpcs.size(); i++){
	if(tmpcs[i]!='\n'){
	  if(tmpcs[i]!='\"') ostr << tmpcs[i];
	}else{
	  ostr << " :: ";
	}
      }
      ostr << "\"\n";
      I++;
    }
  }
  
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
  // Store read type and strain name in a special tag
  {
    ostr << "Tag MINF 1 1 \"ST=";
    switch(REA_seqtype) {
    case SEQTYPE_SANGER : 
    case SEQTYPE_454GS20 : 
    case SEQTYPE_IONTORRENT : 
    case SEQTYPE_PACBIO : 
    case SEQTYPE_SOLEXA : {
      ostr << getNameOfSequencingType(REA_seqtype);
      break;
    }
    case SEQTYPE_ABISOLID : {
      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 7.");
      break;
    }
    default : {
      cerr << "For read " << getName() << ": unknown seqtype code " << static_cast<uint16>(REA_seqtype) << " ?\n";
      cerr << "Allowed are: Sanger, 454, IonTor, PacBio, Solexa, SOLiD.\n";
      throw Notify(Notify::FATAL, THISFUNC, "Illegal readtype?");
    }
    }
    //if(REA_strainid > 0){
    if(!REA_sc_strain.getEntry(REA_strain).empty()){
      ostr << " :: SN=" << REA_sc_strain.getEntry(REA_strain);
    }
    if(!REA_sc_machine_type.getEntry(REA_machine_type).empty()){
      ostr << " :: MT=" << REA_sc_machine_type.getEntry(REA_machine_type);
    }
    if(isBackbone()){
      ostr << " :: BB=1";
    }
    if(isCoverageEquivalentRead()){
      ostr << " :: CER=1";
    }
    ostr << "\"\n";
  }

  ostr << "\n\n";


  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsBAF(ostream & ostr)
{
  FUNCSTART("void Read::dumpAsBAF(ostream & ostr)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if(checkRead()!=NULL){
    cout << "Ouch, failed first check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }

  /*
    To be written outside:
    AP=
    DR=
    SI= & SS=
    SO=

    Not written by class yet:
    AL=


    Problems:
    - need hard clips (SL/SR)
    - SO as definition and used in reads: bad
    - RD/CO in annotations: bad, double RD from definition)
    - RD/CO in annotations: bad, consensus annotations? AT= (attach to)
    - direction of annotation
    - multiline annotations in TX?
    
    Perhaps include???
    optional: LN= (for preassigned sizes)


    Propose:
    DR=+/-/=
   */


  // TODO: u.U. die ComplSeq nutzen
  refreshPaddedSequence();

  ostr << "RD=" << getName()
       << "\nSQ=";

  {
    vector<char>::const_iterator I=REA_padded_sequence.begin();
    for(;I!=REA_padded_sequence.end(); ++I) ostr << *I;
  }

#ifdef PARANOIABUGTRACKFLAG
  if(checkRead()!=NULL){
    cout << "Ouch, failed second check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }
#endif

  ostr << "\nFQ=";
  {
    vector<base_quality_t>::const_iterator I=REA_qualities.begin();
    for(;I!=REA_qualities.end();++I) ostr << (*I)+33;
  }
  ostr << '\n';

  if(!REA_scf_filename.empty()){
    ostr << "TR=" << REA_scf_filename << '\n';
  }else if(REA_seqtype==SEQTYPE_454GS20){
    ostr << "TR=" << getName() << '\n';
  }

  // for the time being, we can't use REA_ql an qr as MIRA
  //  has other clips that may meddle
  if(getLeftClipoff()>0) ostr << "QL=" << getLeftClipoff() << '\n';
  if(getRightClipoff()<getLenSeq()) ostr << "QR=" << getRightClipoff()+1 << '\n';

  if(!REA_template.empty()){
    ostr << "TN=" << REA_template << '\n';
  }


  if(!REA_tags.empty()){
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    for(;I!=REA_tags.end(); I++){
      ostr << "AN=" << I->getIdentifierStr()
	   << "\nLO=" << (I->from)+1;
      if((I->to)-(I->from) > 0) ostr << "\nLL=" << (I->to)-(I->from)+1 << "\n" ;
      if(I->strand=='-') {
	ostr << "DR=-1\n";
      }

      // use serialiseComment()?
      const string & tmpcs=I->getCommentStr();
      if(!tmpcs.empty()){
	ostr << "TX=";
	for(uint32 i=0; i< tmpcs.size(); i++){
	  if(tmpcs[i]!='\n'){
	    ostr << tmpcs[i];
	  }else{
	    ostr << " :: ";
	  }
	}
	ostr << "\n";
      }
    }
  }
  
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
  // Store read type and strain name in a special tag
  {
    ostr << "AN=MINF\nTX=ST=";
    switch(REA_seqtype) {
    case SEQTYPE_SANGER :
    case SEQTYPE_454GS20 :
    case SEQTYPE_IONTORRENT :
    case SEQTYPE_PACBIO :
    case SEQTYPE_SOLEXA : {
      ostr << getNameOfSequencingType(REA_seqtype) << '\n';
      break;
    }
    case SEQTYPE_ABISOLID : {
      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 7.");
      break;
    }
    default : {
      cerr << "For read " << getName() << ": unknown readtype code " << static_cast<uint16>(REA_seqtype) << " ?\n";
      cerr << "Allowed are: Sanger, 454, IonTor, Solexa, PacBio, SOLiD.\n";
      throw Notify(Notify::FATAL, THISFUNC, "Illegal readtype?");
    }
    }
    //if(REA_strainid > 0){
    if(!REA_sc_strain.getEntry(REA_strain).empty()){
      ostr << " :: SN=" << REA_sc_strain.getEntry(REA_strain);
    }
    if(!REA_sc_machine_type.getEntry(REA_machine_type).empty()){
      ostr << " :: MT=" << REA_sc_machine_type.getEntry(REA_machine_type);
    }
    if(isBackbone()){
      ostr << " :: BB=1";
    }
    if(isCoverageEquivalentRead()){
      ostr << " :: CER=1";
    }
    ostr << "\n";
  }

  ostr << "\n\n";


  FUNCEND();
}



/*************************************************************************
 *
 * MAF is mixture of CAF & EXP, but single line per entry and adapted to MIRA
 *
 *
 *************************************************************************/

void Read::dumpAsMAF(ostream & ostr)
{
  FUNCSTART("void Read::dumpAsMAF(ostream & ostr)");

  //BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if(checkRead()!=NULL || getName().empty()){
    // This is some kind of invalid read ... try to save something minimal

    if(getName().empty()){
      char buffer[24] ;
      sprintf(buffer, "%x", this) ;

      ostr << "RD\tivld_" << buffer << "\nER\n";
    }else{
      ostr << "RD\tivld_" << getName() << "\nER\n";
    }

    FUNCEND();
    return;
  }

  // TODO: u.U. die ComplSeq nutzen
  refreshPaddedSequence();

  // RD = Read name
  // LR = Read length (length of unclipped sequence) (optional, only written on len >=2000)
  // RS = sequence (in one line, gaps as '*')
  // RQ = base quality (in one line, FASTQ-33 format)
  // SV = sequencing vector (name)
  // TN = template (name)
  // DI = direction (strand; 'F' or 'R' for forward/reverse)

  // TF = template size from (integer>=0 or -1 for "don't care")
  // TT = template size to (integer>=0 or -1 for "don't care")

  // SF = Sequencing File (file name with raw data like SCF file etc.pp)
  // BC = basecaller (name)

  // SL, SR = sequencing vector left / right cutoffs
  // QL, QR = quality left / right cutoffs
  // CL, CR = (other) clipping left / right cutoffs

  // AO = Align to Original (is align_to_scf from CAF: from to from to)

  // RT = Read Tag (identifier from to comment).

  // ST = sequencing type (Sanger, 454, PacBio, Solexa, SOLiD)
  // SN = strain name
  // MT = machine name

  // IB = is backbone (0 or 1)
  // IR = is rail (0 or 1)
  // IC = is Coverage Equivalent Read (0 or 1)

  // ER = End Read (marker for MAF parsing, mandatory)
  {
    ostr << "RD\t" << getName() << '\n';
    if(getLenSeq()>=2000) ostr << "LR\t" << getLenSeq() << '\n';
    ostr << "RS\t";
    vector<char>::const_iterator I=REA_padded_sequence.begin();
    for(;I!=REA_padded_sequence.end(); I++) ostr << *I;
    ostr << '\n';
  }

#ifdef PARANOIABUGTRACKFLAG
  if(checkRead()!=NULL){
    cout << "Ouch, failed second check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }
#endif

  {
    ostr << "RQ\t";
    vector<base_quality_t>::const_iterator I=REA_qualities.begin();
    for(;I!=REA_qualities.end();++I) ostr << static_cast<char>((*I)+33);
    ostr << '\n';
  }

  if(!REA_sc_seqvec_name.getEntry(REA_seqvec_name).empty()){
    ostr << "SV\t" << REA_sc_seqvec_name.getEntry(REA_seqvec_name) << '\n';
  }
  if(!REA_template.empty()){
    ostr << "TN\t" << REA_template << '\n';
  }
  switch(REA_template_end){
  case 'F' : {
    ostr << "DI\tF\n";
    break;
  }
  case 'R' : {
    ostr << "DI\tR\n";
    break;
  }
  case 'N' : {
    //ostr << "//template end == 'N'\n";
    // do nothing
    break;
  }
  default : {
    throw Notify(Notify::FATAL, THISFUNC, "Illegal template end?");
  }
  }
  if(REA_insize_staticlib[REA_insize_libid].insize_to!=-1 
     && REA_insize_staticlib[REA_insize_libid].insize_from!=-1){
    ostr << "TF\t" << REA_insize_staticlib[REA_insize_libid].insize_from 
	 << "\nTT\t" << REA_insize_staticlib[REA_insize_libid].insize_to << '\n';
  }
  if(!REA_scf_filename.empty()){
    ostr << "SF\t" << REA_scf_filename << '\n';
  }

  if(!REA_sc_basecaller.getEntry(REA_basecaller).empty()){
    ostr << "BC\t" << REA_sc_basecaller.getEntry(REA_basecaller) << '\n';
  }

  if(REA_sl>0){
    ostr << "SL\t" << REA_sl+1 << '\n';
  }
  if(REA_sr < static_cast<int32>(getLenSeq())){
    ostr << "SR\t"<< REA_sr  << '\n';
  }
  
  if(REA_ql>0){
    ostr << "QL\t" << REA_ql+1 << '\n';
  }
  if(REA_qr < static_cast<int32>(getLenSeq())){
    ostr << "QR\t"<< REA_qr  << '\n';
  }

  if(REA_cl>0){
    ostr << "CL\t" << REA_cl+1 << '\n';
  }
  if(REA_cr < static_cast<int32>(getLenSeq())){
    ostr << "CR\t"<< REA_cr  << '\n';
  }
  
  if(REA_uses_adjustments){
    uint32 i;
    uint32 imin=0;
    for(i=1; i< getLenSeq(); i++){
      if(REA_adjustments[i-1]!=-1){
	if(REA_adjustments[i]!=REA_adjustments[i-1]+1){
	  ostr << "AO\t" << imin+1 << ' ' << i << ' ' << REA_adjustments[imin]+1 << ' ' << REA_adjustments[i-1]+1 << '\n';
	  imin=i;
	}
      }else{
	imin=i;
      }
    }
    if(REA_adjustments[i-1]!=-1){
      ostr << "AO\t" << imin+1 << ' ' << i << ' ' << REA_adjustments[imin]+1 << ' ' << REA_adjustments[i-1]+1 << '\n';
    }
  }else{
    ostr << "AO\t" << 1 << ' ' << getLenSeq() << ' ' << 1 << ' ' << getLenSeq() << '\n';
  }


  if(!REA_tags.empty()){
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    while(I!=REA_tags.end()){
      if(I->strand=='-') {
	ostr << "RT\t" << I->getIdentifierStr() << ' ' << (I->to)+1 << ' ' << (I->from)+1 << " " ;
      }else{
	ostr << "RT\t" << I->getIdentifierStr() << ' ' << (I->from)+1 << ' ' << (I->to)+1 << " " ;
      }

      const string & tmpcs=I->getCommentStr();
      for(uint32 i=0; i< tmpcs.size(); i++){
	if(tmpcs[i]=='\n'){
	  ostr << "\\n";
	}else if(tmpcs[i]=='\\'){
	  ostr << "\\\\";
	}else{
	  ostr << tmpcs[i];
	}
      }
      ostr << '\n';
      I++;
    }
  }
  
  // Store read type and strain name in a special tag
  {
    ostr << "ST\t";
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    switch(REA_seqtype) {
    case SEQTYPE_SANGER :
    case SEQTYPE_454GS20 : 
    case SEQTYPE_IONTORRENT : 
    case SEQTYPE_PACBIO :
    case SEQTYPE_SOLEXA : {
      ostr << getNameOfSequencingType(REA_seqtype) << '\n';
      break;
    }
    case SEQTYPE_ABISOLID : {
      throw Notify(Notify::INTERNAL, THISFUNC, "Type ABI SOLiD needs more support 7.");
      break;
    }
    default : {
      cerr << "For read " << getName() << ": unknown readtype code " << static_cast<uint16>(REA_seqtype) << " ?\n";
      cerr << "Allowed are: Sanger, 454, IonTor, Solexa, PacBio, SOLiD.\n";
      throw Notify(Notify::FATAL, THISFUNC, "Illegal readtype?");
    }
    }
    //if(REA_strainid > 0){
    if(!REA_sc_strain.getEntry(REA_strain).empty()){
      ostr << "SN\t" << REA_sc_strain.getEntry(REA_strain) << '\n';
    }
    if(!REA_sc_machine_type.getEntry(REA_machine_type).empty()){
      ostr << "MT\t" << REA_sc_machine_type.getEntry(REA_machine_type) << '\n';
    }
    if(isBackbone()){
      ostr << "IB\t1\n";
    }
    if(isRail()){
      ostr << "IR\t1\n";
    }
    if(isCoverageEquivalentRead()){
      ostr << "IC\t1\n";
    }
  }

  ostr << "ER\n";

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::dumpAsACE(ostream & ostr, int32 direction)
{
  FUNCSTART("void Read::dumpAsACE(ostream & ostr, int32 direction)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if(checkRead()!=NULL){
    cout << "Ouch, failed first check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }

  {
    vector<char>::const_iterator I;
    vector<char>::const_iterator IEnd;
    if (direction > 0) {
      refreshPaddedSequence();
      ostr << "\nRD " << getName() << ' ' << REA_padded_sequence.size() << " 0 0" << endl;
      I=REA_padded_sequence.begin();
      IEnd=REA_padded_sequence.end();
    } else {
      refreshPaddedComplementSequence();
      ostr << "\nRD " << getName() << ' ' << REA_padded_complementsequence.size() << " 0 0" << endl;
      I=REA_padded_complementsequence.begin();
      IEnd=REA_padded_complementsequence.end();
    }

    uint32 cpl=0;
    while(I!=IEnd){
      ostr << *I;
      if(cpl++==59){
	cpl=0;
	ostr << '\n';
      }
      I++;
    }
    ostr << '\n';
  }

  if(checkRead()!=NULL){
    cout << "Ouch, failed second check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }

  // ACE has no base qualities *sigh*

  if (direction > 0) {
    ostr << "\nQA " << getLeftClipoff()+1 << ' ' << getRightClipoff();
    ostr << ' ' << getLeftClipoff()+1 << ' ' << getRightClipoff() << endl;
  } else {
    ostr << "\nQA " << REA_padded_complementsequence.size()-getRightClipoff()+1 << ' ' << REA_padded_complementsequence.size()-getLeftClipoff();
    ostr << ' ' << REA_padded_complementsequence.size()-getRightClipoff()+1 << ' ' << REA_padded_complementsequence.size()-getLeftClipoff() << endl;
  }

  ostr << "\nDS ";
  if(!REA_template.empty()){
    ostr << "TEMPLATE: " << REA_template << ' ';
  }
  if(!REA_scf_filename.empty()){
    ostr << "SCF_FILE: " << REA_scf_filename <<  ' ' ;
  }
  ostr << "CHROMAT_FILE: " << getName() << ' ';
  ostr << "PHD_FILE: " << getName() << ".phd.1 ";
  ostr << "TIME: Sat Jan  1 11:11:11 MEST 2000";
  ostr << '\n';

  // TODO: consed does not support tags in reverse direction
  //  MIRA sets tags always in forward direction or on both
  //  strands ('='), but tags read in from GenBank files
  //  NEED to allow for reverse (genes can be in reverse direction
  //  on the genome after all).
  // So, this is something that consed must fix.
  if(!REA_tags.empty()){
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    string serialc;
    while(I!=REA_tags.end()){
      ostr << "RT{\n" << getName() << ' ' << I->getIdentifierStr();
      if(I->strand=='-'){
	ostr << " MIRA " << (I->to)+1 << ' ' << (I->from)+1;
      } else{
	ostr << " MIRA " << (I->from)+1 << ' ' << (I->to)+1;
      }
      // TODO: make it a real date
      ostr << " 020202:121212\n";
      
      const string & tmpcs=I->getCommentStr();
      if(!tmpcs.empty()){
	I->serialiseComment(serialc);
	if(!serialc.empty()) {
	  ostr << "COMMENT{\n" << serialc << "\nC}\n";
	}
      }
      ostr << "}\n\n";
      I++;
    }
  }else{
    ostr << '\n';
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::dumpAsGAP4DA(ostream & ostr, string & APline, bool outputTags)
{
  FUNCSTART("void Read::dumpAsGAP4DA(ostream & ostr, string & APline, bool outputTags)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if(checkRead()!=NULL){
    cout << "Ouch, failed first check." << endl;
    setCoutType(AS_TEXT);
    cout << *this;
    throw Notify(Notify::FATAL, THISFUNC, checkRead());
  }

  ostr << "ID   " << getName() << endl;
  ostr << "EN   " << getName() << endl;
  ostr << "DR   +\n";
  if(REA_ql>0) {
    ostr << "QL   " << REA_ql << endl;
  }
  if(REA_qr<static_cast<int32>(REA_qualities.size())) {
    ostr << "QR   " << REA_qr+1 << endl;
  }
  if(REA_sl>0) {
    ostr << "SL   " << REA_sl << endl;
  }
  if(REA_sr<static_cast<int32>(REA_qualities.size())) {
    ostr << "SR   " << REA_sr+1 << endl;
  }
  if(!REA_sc_seqvec_name.getEntry(REA_seqvec_name).empty()){
    ostr << "SV   " << REA_sc_seqvec_name.getEntry(REA_seqvec_name) << endl;
  }
  if(!REA_sc_clonevec_name.getEntry(REA_clonevec_name).empty()){
    ostr << "CV   " << REA_sc_clonevec_name.getEntry(REA_clonevec_name) << endl;
  }

  if(!REA_template.empty()){
    ostr << "TN   " << REA_template << endl;
  }
  if(REA_insize_staticlib[REA_insize_libid].insize_to!=-1 
     && REA_insize_staticlib[REA_insize_libid].insize_from!=-1){
    ostr << "SI   " << REA_insize_staticlib[REA_insize_libid].insize_from 
	 << ".." << REA_insize_staticlib[REA_insize_libid].insize_to << endl;
  }
  if(!REA_scf_filename.empty()){
    ostr << "LN   " << REA_scf_filename;
    ostr << "\nLT   SCF" << endl;
  }
  if(!REA_sc_basecaller.getEntry(REA_basecaller).empty()){
    ostr << "BC   " << REA_sc_basecaller.getEntry(REA_basecaller) << endl;
  }
  if(!REA_sc_processstatus.getEntry(REA_processstatus).empty()){
    ostr << "PS   " << REA_sc_processstatus.getEntry(REA_processstatus) << endl;
  }
  if(!REA_sc_primer.getEntry(REA_primer).empty()){
    ostr << "PN   " << REA_sc_primer.getEntry(REA_primer) << endl;
  }

  // ON
  // Basically, that's the align to scf from CAF
  // except it is not written if the read has no adjustments
  if(REA_uses_adjustments){
    ostr << "ON   ";
    bool hasrange=false;
    uint32 i;
    int32 rmin=-1;

    // James said there is a line limit for EXP
    //  so I need this to make an output of at most 
    //  "8" (may be a few more) values per line 
    bool mustwriteON=false;
    uint32 valuesinline=0;
    uint32 maxvaluesinline=8;


    for(i=0; i< getLenSeq(); i++){
      if(REA_adjustments[i]==-1){
	if(mustwriteON){
	  ostr << "\nON        ";
	  valuesinline=0;
	  mustwriteON=false;
	}
	if(hasrange) {
	  ostr << rmin+1 << ".." << REA_adjustments[i-1]+1 << ' ';
	  valuesinline++;
	  hasrange=false;
	}
	ostr << "0 ";
	valuesinline++;
      } else {
	if(hasrange) {
	  if(REA_adjustments[i]!=REA_adjustments[i-1]+1) {
	    if(mustwriteON){
	      ostr << "\nON        ";
	      valuesinline=0;
	      mustwriteON=false;
	    }
	    ostr << rmin+1 << ".." << REA_adjustments[i-1]+1 << ' ';
	    valuesinline++;
	    rmin=REA_adjustments[i];
	  }
	} else {
	  rmin=REA_adjustments[i];
	  hasrange=true;
	}
      }
      if(valuesinline>=maxvaluesinline) {
	mustwriteON=true;
      }
    }
    if(hasrange) {
      ostr << rmin+1 << ".." << REA_adjustments[i-1]+1 << ' ';
    }
    ostr << endl;
  }

  // AV
  {
    vector<base_quality_t>::const_iterator I=REA_qualities.begin();
    uint32 cpl=21;
    if (I!=REA_qualities.end()) ostr << "AV   ";
    while(I!=REA_qualities.end()){
      ostr << static_cast<uint16>(*I);
      if(--cpl){
	ostr << ' ';
      }else{
	ostr << "\nAV        ";
	cpl=19;
      }
      I++;
    }
    if(cpl) ostr << '\n';
  }

  if(APline.size()) ostr << "AP   " << APline << endl;

  // SQ
  {
    refreshPaddedSequence();
    ostr << "SQ\n";
    vector<char>::const_iterator I=REA_padded_sequence.begin();

    uint32 cpl=0;
    while(I!=REA_padded_sequence.end()){
      if(cpl==0) ostr << "    ";
      if(cpl%10==0) ostr << ' ';
      if(*I=='N' || *I=='n') {
	ostr << "-";
      }else{
	ostr << *I;
      }
      if(cpl++==59){
	cpl=0;
	ostr << '\n';
      }
      I++;
    }
    if(cpl) ostr << '\n';
    ostr << "//\n";
  }

  // TG
  if(outputTags && !REA_tags.empty()){
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    while(I!=REA_tags.end()){
      ostr << "TG   " << I->getIdentifierStr();
      ostr << ' ' << I->strand << ' ' << (I->from)+1 << ".." << (I->to)+1;
      const char * cs=I->getCommentStr().c_str();
      if (*cs!=0) ostr << "\nTG        ";
      while(*cs) {
	ostr << *cs;
	if(*cs == '\n') ostr << "TG        ";
	cs++;
      }
      ostr << endl;
      I++;
    }
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
// frees everything possible, making the class ready for reuse
void Read::discard()
{
  FUNCSTART("Read::discard()");

  //nukeSTLContainer(REA_name);         
  nukeSTLContainer(REA_scf_filename); 
  //nukeSTLContainer(REA_exp_filename); 
  //nukeSTLContainer(REA_caf_filename); 
  nukeSTLContainer(REA_template);  

  
  nukeSTLContainer(REA_padded_sequence);
  nukeSTLContainer(REA_padded_complementsequence);
  nukeSTLContainer(REA_qualities);
  nukeSTLContainer(REA_adjustments);
  nukeSTLContainer(REA_bposhashstats);
  nukeSTLContainer(REA_tags);

  zeroVars();

  FUNCEND();
}





///*************************************************************************
// *
// *
// *
// *
// *************************************************************************/
//void Read::updateClipoffs()
//{
////  REA_rightclip=min(REA_qr, min(REA_sr, REA_cr));
////  REA_leftclip=max(REA_ql, max(REA_sl, REA_cl));
//
//  REA_rightclip=min(REA_qr, REA_sr);
//  REA_leftclip=max(REA_ql, REA_sl);
//  REA_len_clipped=REA_rightclip-REA_leftclip;
//  if(REA_len_clipped<0) REA_len_clipped=0;
//}






/*************************************************************************
 *
 * starting from the actual quality clips, clip more from the ends while
 *  the characters occuring are X, N, *. That is, results will differ 
 *  whether or not a previous quality clipping might have happened or not
 *
 * stretches with other bases will be merged while being <= 'gapsize', 
 *  the very first stretch may be 'frontgs' away and the last
 *  'endgs' away from the ends
 *
 *************************************************************************/
void Read::setClipoffsToMaskedChars(int32 gapsize, int32 frontgs, int32 endgs, bool allowN)
{
  FUNCSTART("void Read::setClipoffstoMaskedChars(int32 gapsize, int32 frontgs, int32 endgs, bool allowN)");

  BUGIFTHROW(gapsize < 0, getName() << ": gapsize (" << gapsize << ") < 0 ?");

  refreshPaddedSequence();

  if(frontgs < 0) frontgs=gapsize;
  if(endgs < 0) endgs=gapsize;

  int32 grace;

  // calc the left clip
  int32 icount;
  int32 ql;

  // 1st loop: calc from ends of read
  // 2nd loop: calc from quality clips
  for(uint32 loop=0; loop<2; loop++){
    grace=frontgs+1;
    if(loop) {
      ql=REA_ql;
    } else {
      ql=0;
    }
    icount=ql+1;
    vector<char>::const_iterator I=REA_padded_sequence.begin();
    advance(I,ql);
    bool inXrun=false;
    bool foundX=false;
    char actbase;
    while(grace>0 && I!=REA_padded_sequence.end()) {
      actbase=toupper(*I);
      if(allowN && actbase=='N') actbase='X';
      switch(actbase) {
      case '*' :
      case 'N' : {
	if(inXrun) {
	  ql=icount;
	  grace=gapsize+1;
	}
	break;
      }
      case 'X' : {
	inXrun=true;
	foundX=true;
        ql=icount;
	grace=gapsize+1;
	break;
      }
      default: {
	inXrun=false;
	grace--;
      }
      }
      I++;
      icount++;
    }
    REA_ql=max(ql,REA_ql);
    if(foundX) REA_ml=ql;
  }

  // getLenSeq needs sane clipoffs
  updateClipoffs();


  // calc the right clip

  int32 qr;
  // 1st loop: calc from ends of read
  // 2nd loop: calc from quality clips
  for(uint32 loop=0; loop<2; loop++){
    grace=endgs+1;
    if(loop) {
      qr=REA_qr;
    } else {
      qr=getLenSeq();
    }
    icount=qr-1;
    vector<char>::const_iterator I=REA_padded_sequence.begin();
    if(qr > 0) advance(I,qr-1);
    bool inXrun=false;
    bool foundX=false;
    char actbase;
    while(grace>0 && I!=REA_padded_sequence.begin()) {
      actbase=toupper(*I);
      if(allowN && actbase=='N') actbase='X';
      switch(actbase) {
      case 'N' : 
      case '*' : {
	if(inXrun) {
	  qr=icount;
	  grace=gapsize+1;
	}
	break;
      }
      case 'X' : {
	inXrun=true;
	foundX=true;
        qr=icount;
	grace=gapsize+1;
	break;
      }
      default: {
	inXrun=false;
	grace--;
      }
      }
      I--;
      icount--;
    }
    if(qr==-1) REA_qr=0;       // to check: necessary?

    REA_qr=min(qr,REA_qr);
    if(foundX) REA_mr=qr;
  }

  if(REA_qr<REA_ql) REA_qr=REA_ql;
  updateClipoffs();

  //if(REA_qr<REA_mr && REA_qr < REA_sr ) cout << getName() << " rextend possible " << REA_qr << ' ' <<REA_mr << ' ' <<REA_sr << endl;

  FUNCEND();

  return;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::setSequenceFromString(const char * sequence)
{
  FUNCSTART("void Read::setSequenceFromString(const char * sequence)");

  REA_has_valid_data=false;

  REA_padded_sequence.clear();
  REA_padded_sequence.reserve(strlen(sequence)+3);
  
  // initialise the sequence vector
  {
    const char * fasta_sequence=sequence;
    do{
      char tmpc=*fasta_sequence++;
      if(dptools::isValidIUPACStarBase(tmpc)){
	REA_padded_sequence.push_back(tmpc);
      }else{
	switch(toupper(tmpc)){
	case 'N':
	case 'X':{ 
	  REA_padded_sequence.push_back(tmpc);
	  break;
	}
	  //case '*': 
	case '-': {
	  REA_padded_sequence.push_back('N');
	  break;
	}
	// be lenient: jump over spaces, tabs, newlines, ^M from DOS, etc.pp
	case ' ' :
	case '\t' :
	case '\n' :
	case '\r' :{
	  break;
	}
	default: {
	  fasta_sequence=NULL;
	}
	}
      }
    }while(fasta_sequence!=NULL);
  }
  
  REA_ql=0;                           
  REA_qr=static_cast<int32>(REA_padded_sequence.size());
  REA_sl=0;                           
  REA_sr=static_cast<int32>(REA_padded_sequence.size());
  REA_cl=0;                           
  REA_cr=static_cast<int32>(REA_padded_sequence.size());
  REA_ml=0;                           
  REA_mr=static_cast<int32>(REA_padded_sequence.size());
  
  updateClipoffs();
  
  postLoadEXPFASTA();
    
  REA_has_valid_data=true;

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::checkQualities()
{
  FUNCSTART("void Read::checkQualities()");

  vector<base_quality_t>::const_iterator I=REA_qualities.begin();
  vector<base_quality_t>::const_iterator E=REA_qualities.end();

  for(;I!=E; I++){
    if(*I>100){
      MIRANOTIFY(Notify::FATAL,"Read " << getName() << ": tried to set quality <0 or > 100???\n");
    }
  }

  FUNCEND();
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::setQualities(const vector<base_quality_t> & quals)
{
  FUNCSTART("void Read::setQuality(vector<base_quality_t> & quals)");

  // TODO: check, if that is ok!
  if(quals.size() != REA_qualities.size()) {
    MIRANOTIFY(Notify::FATAL,"Read " << getName() << ": tried to set " << quals.size() << " qualities although the read has " << REA_qualities.size() << " bases.\n");
  }


  REA_qualities=quals;

  setQualityFlag(true);

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::setQualities(base_quality_t qual)
{
  FUNCSTART("void Read::setQuality(base_quality_t qual)");

  vector<base_quality_t>::iterator I=REA_qualities.begin();
  for(; I != REA_qualities.end(); I++) *I=qual;

  setQualityFlag(true);

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::loadDataFromSCF(const string & scfname)
{
  FUNCSTART("void Read::loadDataFromSCF(const string & scfname)");

  discard();

  if(scfname.empty()) {
    throw Notify(Notify::FATAL, THISFUNC, "No scfname given?");
  }

  REA_scf_filename=scfname;
  //setFileNamesFromSCFFileName(scfname);

  SCF scf;
  scf.load(REA_scf_filename.c_str());

  // initialise the sequence vector
  {
    REA_padded_sequence.clear();
    REA_padded_sequence.reserve(scf.getNumBases()+3);

    for(uint32 i=0; i<scf.getNumBases(); i++) {
      char tmpc=scf.getBase(i);
      if(dptools::isValidIUPACBase(tmpc)){
	REA_padded_sequence.push_back(tmpc);
      }else{
	switch(toupper(tmpc)){
	case 'N':
	case 'X':{ 
	  REA_padded_sequence.push_back(tmpc);
	  break;
	}
	case '*': 
	case '-': {
	  REA_padded_sequence.push_back('N');
	  break;
	}
	default: {
	  cerr << "At base position " << i << ": base " << tmpc << " unknown IUPAC code?\n";
	  MIRANOTIFY(Notify::FATAL, "Illegal base found: " << scfname);
	}
	}
      }
    }
  }

  REA_ql=0;                           
  REA_qr=static_cast<int32>(REA_padded_sequence.size());
  REA_sl=0;                           
  REA_sr=static_cast<int32>(REA_padded_sequence.size());
  REA_cl=0;                           
  REA_cr=static_cast<int32>(REA_padded_sequence.size());
  REA_ml=0;                           
  REA_mr=static_cast<int32>(REA_padded_sequence.size());
  
  updateClipoffs();

  postLoadEXPFASTA();
    
  for(uint32 i=0; i<scf.getNumBases(); i++) {
    REA_qualities[i]=scf.getCalledBaseProb(i);
  }

  REA_has_valid_data=true;

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
  return;

}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::loadDataFromEXP(const string & filename)
{
  FUNCSTART("void Read::loadDataFromEXP()");

  REA_has_valid_data=false;

  EXP tmpexp;

  string dummy;
  if(!REA_sc_exp_pathname.getEntry(REA_exp_pathname).empty()) {
    dummy=REA_sc_exp_pathname.getEntry(REA_exp_pathname)+"/"+filename;
  }else{
    dummy=filename;
  }

  tmpexp.load(dummy.c_str());

  // ok, get the name and SCFname from the exp file
  {
    // First, the name of the experiment
    //REA_name.clear();
    //REA_name=tmpexp.getID();
    setName(tmpexp.getID());
    if(getName().empty()){
      MIRANOTIFY(Notify::FATAL, "The experiment has no name (no ID field found): " << filename);
    }

    // Now the SCF filename of the experiment
    REA_scf_filename.clear();
    REA_scf_filename=tmpexp.getLN();
    //if(REA_scf_filename.empty()){
    //  // FIXME: how to treat exp files without SCF data?
    //  //throw Notify(Notify::FATAL, THISFUNC, REA_exp_filename, ": The experiment has no SCF file (no LN field found)");
    //}
  }


  REA_ql=tmpexp.getQL();
  REA_qr=tmpexp.getQR();
  REA_sl=tmpexp.getSL();
  REA_sr=tmpexp.getSR();
  REA_cl=tmpexp.getCL();
  REA_cr=tmpexp.getCR();

  REA_machine_type=REA_sc_machine_type.addEntry(tmpexp.getMA());

  setInsize(tmpexp.getSIfrom(),tmpexp.getSIto());

  // TODO: reactivate/emulate
  //tmpexp.swapTags(REA_tags);      // the EXP will be destructed, so save time

  // subtract 1 from the positions read in exp file (which are basis 1)
  for(uint32 i=0; i<REA_tags.size(); i++ ){
    REA_tags[i].from--;
    REA_tags[i].to--;
  }
  
  if(REA_ql>REA_qr){
#ifndef PUBLICQUIET
    WARNING("WARNING: " << getName() << "   left quality clip > right quality clip (QL > QR), adjusted.");
#endif
    REA_qr=REA_ql;
  }
  if(REA_sl>REA_sr){
#ifndef PUBLICQUIET
    WARNING("WARNING: " << getName() << "   sequence vector left clip > sequence vector right clip (SL > SR), adjusted.");
#endif
    REA_sr=REA_sl;
  }

  updateClipoffs();

  if(getRightClipoff()-getLeftClipoff()<0){
#ifndef PUBLICQUIET
    WARNING("WARNING: " << getName() << "   quality and sequence vector clippings form a negative length (clip right - clip left <0), adjusted");
#endif
    REA_qr=REA_sl;
    updateClipoffs();
  }

  // initialise the sequence vector
  {
    const char * exp_sequence=tmpexp.getSequence().c_str();
    if(exp_sequence==NULL){
      MIRANOTIFY(Notify::WARNING, "No bases were found: " << getName());
    }

    REA_padded_sequence.clear();
    REA_padded_sequence.reserve(strlen(exp_sequence)+3);
    do{
      char tmpc=*exp_sequence++;
      if(dptools::isValidIUPACBase(tmpc)){
	REA_padded_sequence.push_back(tmpc);
      }else{
	switch(toupper(tmpc)){
	case 'N':
	case 'X':
	case '*':{
	  REA_padded_sequence.push_back(tmpc);
	  break;
	}
	case '-': {
	  REA_padded_sequence.push_back('N');
	  break;
	}
	case 0 : {
	  // FIXME: how to treat emtpy reads????
	  //throw Notify(Notify::WARNING, THISFUNC, REA_exp_filename, ": No bases were found?");
	  // cannot throw, isn't gentle enough for big projects :(
	  WARNING("WARNING: " << tmpexp.getID() << "   has no bases in the experiment file?");
	  // FIXME: pfusch!
	  REA_padded_sequence.push_back('N');
	  REA_ql=0;
	  REA_qr=1;
	  REA_sl=0;
	  REA_sr=1;
	  updateClipoffs();
	  exp_sequence--;
	  break;
	}
	default: {
	  cout << "Illegal base: " << tmpc << "(" << hex << static_cast<uint16>(tmpc) << dec << ")" << endl;
	  MIRANOTIFY(Notify::FATAL, "Illegal base found: " << tmpexp.getID());
	}
	}
      }
      
    }while(*exp_sequence!=0);
  }

  postLoadEXPFASTA();


  if(tmpexp.getSV().empty()){
    REA_seqvec_name=REA_sc_seqvec_name.addEntry(tmpexp.getSF());
  }else{
    REA_seqvec_name=REA_sc_seqvec_name.addEntry(tmpexp.getSV());
  }
  REA_template=tmpexp.getTN();
  REA_basecaller=REA_sc_basecaller.addEntry(tmpexp.getBC());
  REA_asped=REA_sc_asped.addEntry(tmpexp.getDT());
  //  REA_dye;
  REA_processstatus=REA_sc_processstatus.addEntry(tmpexp.getPS());
  REA_primer=REA_sc_primer.addEntry(tmpexp.getPN());

  //  read AV qualities if any
  {
    const vector<base_quality_t> & equal=tmpexp.getAV();
    uint32 expavsize=static_cast<uint32>(equal.size());
    if(expavsize>0 && expavsize!=REA_padded_sequence.size()){
      cout << "Sequence length: " << REA_padded_sequence.size() << endl;
      cout << "Number of quality values: " << expavsize << endl;
      MIRANOTIFY(Notify::FATAL, "The experiment has an unequal number of bases and quality values (SQ vs AV fields): " << filename);
    }
    if(expavsize>0){
      REA_has_quality=true;
      for(uint32 i=0; i<REA_qualities.size(); i++){
	REA_qualities[i]=equal[i];
      }
    }
  }

  //  read ON adjustments
  if(REA_uses_adjustments){
    const vector<int32> onvals= tmpexp.getON();
    if(onvals.size()){
      if(onvals.size()%2) {
	MIRANOTIFY(Notify::INTERNAL, "there should be a even number of ON adjustments: " << filename);
      }
      //vector<int32> newadjustments[REA_adjustments.size()];
      vector<int32> newadjustments;
      newadjustments.resize(REA_adjustments.size());
      uint32 nai=0;
      uint32 ovi=0;
      while(ovi<onvals.size()){
	//cout << onvals[ovi] << "  " << onvals[ovi+1] << endl;
	if(onvals[ovi]==0){
	  newadjustments[nai]=-1;
	  nai++;
	} else {
	  uint32 onlow=onvals[ovi];
	  uint32 onhigh=onvals[ovi+1];
	  for(uint32 i=onlow; i<=onhigh; i++, nai++){
	    if(nai>=newadjustments.size()) {
	      MIRANOTIFY(Notify::FATAL, "the adjustments given in the ON tag exceed the size of the sequence: " << filename);
	    }
	    newadjustments[nai]=i-1;
	  }
	}
	ovi+=2;
      }
      if(nai!=newadjustments.size()){
	cout << "nai: " << nai << "  newadjustments.size(): " << newadjustments.size() << endl;
	MIRANOTIFY(Notify::FATAL, "the adjustments given in the ON tag do not cover the whole range of sequence: " << filename);
      }
      REA_adjustments.swap(newadjustments);
    }
  }

  // transfer Mira INFormation in MINF tags to read variables
  transferMINFTagsToReadInfo();

  REA_has_valid_data=true;

  REA_ml=0;                           
  REA_mr=getLenSeq();

  if(REA_qr<static_cast<int32>(REA_qualities.size())) REA_qr--;
  if(REA_sr<static_cast<int32>(REA_qualities.size())) REA_sr--;
  if(REA_cr<static_cast<int32>(REA_qualities.size())) REA_cr--;
  updateClipoffs();

  //  CEBUG(*this);
  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //tmpexp.dump();
  //cout << *this;

  FUNCEND();
}



/*************************************************************************
 *
 * void postLoadEXPFASTA()
 * build complements, build padded vectors, set qualities to defaults
 *  set adjustments to defaults, set bposhashstats to default
 *  set template if necessary and possible
 *
 *************************************************************************/
void Read::postLoadEXPFASTA()
{
  FUNCSTART("void postLoadEXPFASTA();");

  // build the complement padded sequence
  makeComplement(REA_padded_sequence, REA_padded_complementsequence);

  REA_ps_dirty=false;
  REA_pcs_dirty=false;


  // When loading from an EXP-file, there might be no qualities given
  // So, fill the quality-vector with 0, which will be overwritten if 
  //  quals are available
  REA_qualities.clear();
  REA_qualities.reserve(REA_padded_sequence.capacity());
  REA_qualities.resize(REA_padded_sequence.size(),0);

  // base flags
  REA_bposhashstats.clear(); 
  REA_bposhashstats.reserve(REA_padded_sequence.capacity());
  clearAllBPosHashStats();

  // When loading an EXP-file, the bases correspond to the bases in the
  //  SCF file.
  if(REA_uses_adjustments){
    REA_adjustments.reserve(REA_padded_sequence.capacity());
    REA_adjustments.resize(REA_padded_sequence.size());
    vector<int32>::iterator aI=REA_adjustments.begin();
    vector<char>::const_iterator sI=REA_padded_sequence.begin();
    int32 actadjust=0;
    for(; aI<REA_adjustments.end(); aI++, sI++){
      if(*sI!='*') {
	*aI=actadjust;
	actadjust++;
      }else{
	*aI=-1;
      }
    }
  }

  if(REA_template.empty()){
    string tmp=getInternalTemplateName();
    if(REA_template_end!='N') REA_template=tmp;
  }

  FUNCEND();
}


/*************************************************************************
 *
 * TODO; check 'frommaskedchar'
 *
 *
 *************************************************************************/

void Read::setMinimumLeftClipoff(int32 minreq, int32 setto, bool fromseqvec, bool frommaskedchar)
{
  FUNCSTART("void Read::setMinimumLeftClipoff(int32 minreq, int32 setto, bool fromseqvec, bool frommaskedchar)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if( REA_ql < setto ) {
    if(REA_ql < minreq) {
      REA_ql=setto;
    }
    if(fromseqvec && REA_sl < minreq) {
      REA_ql=setto;
    }
    if(REA_ml>0 && REA_ml < minreq) {
      REA_ql=setto;
    }
    if(REA_ql>=static_cast<int32>(REA_qualities.size())) REA_ql=static_cast<int32>(REA_qualities.size())-1;
    if(REA_ql<0) REA_ql=0;
    if(REA_qr<REA_ql) REA_qr=REA_ql;
    updateClipoffs();
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::setMinimumRightClipoff(int32 minreq, int32 setto)
{
  FUNCSTART("void Read::setMinimumRightClipoff(int32 minreq, int32 setto)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  if( REA_qr > static_cast<int32>(REA_qualities.size())-minreq ) {
    REA_qr=static_cast<int32>(REA_qualities.size())-setto;
    if(REA_qr<0) REA_qr=0;
    if(REA_qr<REA_ql) REA_qr=REA_ql;
    updateClipoffs();
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

bool Read::hasSCFData(bool loadfailnoerror)
{
  FUNCSTART("bool Read::hasSCFData(bool loadfailnoerror=false)");

  if(!REA_has_valid_data
     || REA_seqtype != SEQTYPE_SANGER) {
    FUNCEND();
    return false;
  }

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  checkSCFAndLoadQual(true,loadfailnoerror);

  FUNCEND();
  return REA_scf_available;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::checkSCFAndLoadQual(bool justcheck, bool loadfailnoerror)
{
  FUNCSTART("void Read::checkSCFAndLoadQual(bool justcheck)");

  if(hasValidData()==false){
    throw Notify(Notify::INTERNAL, THISFUNC, "Trying to load SCF from unitialised Read.");          
  }

  if(REA_scf_loadattempted
     || REA_scf_filename.empty()
     || REA_israil
     || REA_iscoverageequivalentread
     || REA_seqtype!=SEQTYPE_SANGER){
    FUNCEND();
    return;
  }

  REA_scf_loadattempted=true;

  SCF scf;

  try{
    string dummy;
    getSCFFullPathName(dummy);
    scf.load(dummy.c_str());
    scf.transposeAmbiguityCodes();
    //    scf.recalcProbs();
  }
  catch(Notify n){
    if(!loadfailnoerror) {
      n.setGravity(Notify::WARNING);
      n.handleError(THISFUNC);
    }
    FUNCEND();
    return;
  }

  if(justcheck==false){
    if(scf.getNumBases()!=getLenSeq()){
      MIRANOTIFY(Notify::WARNING,"Number of bases in SCF file (" << REA_scf_filename << ") does not correspond to the number of bases expected in the read (read from fasta, exp, phd or caf file).\nRead will _not_ be used in assembly!");
    }
    for(uint32 i=0; i<REA_padded_sequence.size(); i++){
      if(toupper(scf.getBase(i))!=toupper(REA_padded_sequence[i])){
	if(!(scf.getBase(i)=='-' || toupper(REA_padded_sequence[i])=='N' || toupper(REA_padded_sequence[i])=='X')){
	  MIRANOTIFY(Notify::WARNING,"A base (" << scf.getBase(i) << ") in the SCF (" << REA_scf_filename << ") does not correspond to the the one (" << REA_padded_sequence[i] << ") in the read read from fasta, exp, phd or caf (position: " << i << ").\nRead will _not_ be used in assembly!");
	}
      }
      if(justcheck==false){
	REA_qualities[i]=scf.getCalledBaseProb(i);
      }
    }
    setQualityFlag(true);
  }
  REA_scf_available=true;

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::makeComplement(const vector<char> & source, vector<char> & destination)
{
  FUNCSTART("void Read::makeComplement(vector<char> & source, vector<char> & destination);");

  destination.clear();
  destination.reserve(source.capacity());
  vector<char>::const_reverse_iterator I=source.rbegin();
  while(I!=source.rend()){
    char base=*I++;
    char cbase=dptools::getComplementIUPACBase(base);
    if(cbase!=0){
      destination.push_back(cbase);
    }else{
      cout << "Argh! Found illegal base " << base << " and there's no complement for it!\n";
      throw Notify(Notify::INTERNAL, THISFUNC, "Illegal base in a phase where there should be none.");
    }
  }
  if(destination.size()!=source.size()){
    throw Notify(Notify::INTERNAL, THISFUNC, "While making complement sequence: source and destination have different sizes afterwards?");
  }

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

int32 Read::calcComplPos(int32 pos) const
{
  FUNCSTART("int32 Read::calcComplPos(int32 pos) const");
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  FUNCEND();
  return static_cast<int32>(REA_qualities.size())-1-pos;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

int32 Read::calcClippedPos2RawPos(int32 pos) const
{
  FUNCSTART("int32 Read::calcComplPos2RawPos(int32 pos) const");
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  FUNCEND();
  return (pos+getLeftClipoff());
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

int32 Read::calcRawPos2ClippedPos(int32 pos) const
{
  FUNCSTART("int32 Read::calcComplPos2RawPos(int32 pos) const");
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  FUNCEND();
  return (pos-getLeftClipoff());
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

int32 Read::calcClippedComplPos2RawPos(int32 pos) const
{
  FUNCSTART("int32 Read::calcClippedComplPos2RawPos(int32 pos) const");
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  FUNCEND();
  return (getRightClipoff()-1-pos);
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::setName(const string & name)
{
  FUNCSTART("void Read::setName(const string & name)");

  const char * c=name.c_str();
  const char * emsg=NULL;
  const char * wmsg=NULL;
  char wchar=' ';
  for(; *c !=0; ++c){
    if(*c <32 || *c==127){
      emsg="This is a control code, names should not contain that!";
    }
    switch(*c){
    case 32 : {
      emsg="This is the space character, names should not contain that!";
      break;
    }
    case 34 : //"
    case 38 : //&
    case 39 : //'
    case 59 : //;
    case 63 : //?
    case 96 : //`
    {
      wchar=*c;
      wmsg="This character may pose problems in downstream processing by other programs, it is suggested you choose to change that name.";
      break;
    }
    default : {
    }
    }
    if(emsg != NULL) break;
  }

  if(emsg!=NULL){
    MIRANOTIFY(Notify::INTERNAL,"While trying to set the name of read \n" << name << "\nEncountered character with ASCII code " << static_cast<uint16>(*c) << ". " << *emsg << "\nIt is probably due to your input data, but normally, MIRA should have caught that earlier!");
  }
  if(wmsg!=NULL){
    cout << "Warning while setting the name of read '" 
	 << name 
	 << "':\nEncountered character with ASCII code " 
	 << static_cast<uint16>(wchar) << " (" << wchar << ").\n" 
	 << *wmsg << "\n";
  }

  REA_nameentry=REA_sc_readname.addEntryNoDoubleCheck(name);

  FUNCEND();
}


void Read::setDirectories(const string & exp, const string & scf)
{
  REA_exp_pathname=REA_sc_exp_pathname.addEntry(exp);
  REA_scf_pathname=REA_sc_scf_pathname.addEntry(scf);

  REA_scf_loadattempted=false;
  REA_scf_available=false;
}

void Read::setSCFDirectory(const string & scf)
{
  REA_scf_pathname=REA_sc_scf_pathname.addEntry(scf);

  REA_scf_loadattempted=false;
  REA_scf_available=false;
}

void Read::setSCFFileName(const string & scf)
{
  REA_scf_filename=scf;

  REA_scf_loadattempted=false;
  REA_scf_available=false;
}

void Read::getSCFFullPathName(string & path) const
{
  path.clear();
  if(!REA_sc_scf_pathname.getEntry(REA_scf_pathname).empty()) {
    path=REA_sc_scf_pathname.getEntry(REA_scf_pathname)+"/"+REA_scf_filename;
  }else{
    path=REA_scf_filename;
  }
  return;
}

//void Read::getEXPFullPathName(string & path) const
//{
//  path.clear();
//  string expfn;
//  getEXPName(expfn);
//  if(!REA_sc_exp_pathname.getEntry(REA_exp_pathname).empty()) {
//    path=REA_sc_exp_pathname.getEntry(REA_exp_pathname)+"/"+expfn;
//  }else{
//    path=expfn;
//  }
//  return;
//}



///*************************************************************************
// *
// * Constructs the EXP and CAF filenames for the read, taking the reads
// *  given name as source.
// * Evtl. construct an SCF name.
// *
// *************************************************************************/
//
//void Read::setFileNamesFromEXPFileName(const string & filename)
//{
//  FUNCSTART("void Read::setFileNamesFromEXPFileName(const string & filename)");
//
//  REA_name=filename;
//
//  string::size_type bpos = filename.rfind(".");
//
//  if (bpos != string::npos) {
//    REA_name=filename.substr(0,bpos);
//  }
//
//  //REA_exp_filename=filename;
//  //REA_caf_filename=REA_name+REA_add_4caffn;
//  if (REA_scf_filename.empty()) {
//    REA_scf_filename=REA_name+REA_add_4scffn;
//  }
//
//  REA_scf_loadattempted=false;
//  REA_scf_available=false;
//
//  FUNCEND();
//  return;
//}


///*************************************************************************
// *
// * Constructs the EXP and CAF filenames for the read, taking the reads
// *  given name as source.
// *
// *************************************************************************/
//
//void Read::setFileNamesFromSCFFileName(const string & filename)
//{
//  FUNCSTART("void Read::setFileNamesFromSCFFileName(const string & filename)");
//
//  REA_scf_filename=filename;
//
//  string tmpname=filename;
//  for(uint32 i=0; i<tmpname.size(); i++){
//    tmpname[i]=static_cast<char>(tolower(tmpname[i]));
//  }
//
//  string::size_type bpos = tmpname.rfind(".scf");
//
//  if (bpos != string::npos) {
//    REA_name=filename.substr(0,bpos);
//  } else {
//    REA_name=filename;
//  }
//
//  REA_exp_filename=REA_name+REA_add_4expfn;
//  //REA_caf_filename=REA_name+REA_add_4caffn;
//
//  FUNCEND();
//  return;
//}


//string str16 = "abcdefghi";
//string str17 = "def";
//string::size_type pos = str16.find (str17,0);
//cout << pos << endl; // 3
//pos = str16.find ("AB",0);
//if (pos == string::npos) cout << "Not found" << endl;




void Read::setFileNamesFromFASTAName(const string & givenname)
{
  FUNCSTART("void Read::setFileNamesFromFASTAName(const string & givenname)");

  //REA_name=givenname;
  //REA_exp_filename=givenname+REA_add_4expfn;
  //REA_caf_filename=givenname+REA_add_4caffn;

  setName(givenname);
  if (REA_scf_filename.empty()) {
    REA_scf_filename=givenname+REA_add_4scffn;
    REA_scf_loadattempted=false;
    REA_scf_available=false;
  }

  //cout << REA_name << endl;
  //cout << REA_exp_filename << endl;
  //cout << REA_scf_filename << endl;
  //cout << REA_caf_filename << endl;

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::initialiseRead(bool preserve_originals,
			  bool iscomplement,
			  bool ispadded,
			  vector<char>           & sequence,
			  vector<base_quality_t> & qualities,
			  vector<int32>          & adjustments,
			  vector<tag_t>          & tags,
			  const string & name,
			  const string & SCFname,
			  int32 ql, int32 qr,    // quality clipping left/right
			  int32 sl, int32 sr,   // sequence vector clipping
			  int32 cl, int32 cr   // clone vector clipping
			  )
{
  FUNCSTART("void Read::initialiseRead(bool preserve_originals, ..., ...<base_quality_t>, ...)");

  discard();

//  cout << "got ql: " << ql << endl;
//  cout << "got qr: " << qr << endl;
//  cout << "got sl: " << sl << endl;
//  cout << "got sr: " << sr << endl;
//  cout << "got cl: " << cl << endl;
//  cout << "got cr: " << cr << endl;


  //cout << name << "\t" << sequence.size() << "\t" << qualities.size() << endl;

  uint32 size=static_cast<uint32>(sequence.size());
  if(qualities.size() != size){
    cerr << "Sequence size: " << size << "\tQualities size: " << qualities.size() << endl;
    MIRANOTIFY(Notify::INTERNAL, "qualities.size() != sequence.size() : " << name);
  }
  if(adjustments.size() != size){
    cerr << "Sequence size: " << size << "\tAdjustments size: " << qualities.size() << endl;
    MIRANOTIFY(Notify::INTERNAL, "adjustments.size() != sequence.size() ?:" << name);
  }

  if(name.empty()){
    throw Notify(Notify::INTERNAL, THISFUNC," no name given");
  }
  //if(SCFname==NULL){
  //  //throw Notify(Notify::INTERNAL, THISFUNC," no SCFname given");
  //  SCFname=' ';
  //}

  // handle the sequence
  if(iscomplement==true){
    if(ispadded==true){
      if(preserve_originals==true){
	REA_padded_complementsequence=sequence;
      }else{
	REA_padded_complementsequence.swap(sequence);
      }
      makeComplement(REA_padded_complementsequence, REA_padded_sequence);
    }else{
      if(preserve_originals==true){
	REA_padded_complementsequence=sequence;
      }else{
	REA_padded_complementsequence.swap(sequence);
      }
      makeComplement(REA_padded_complementsequence, REA_padded_sequence);

    }
  }else{
    if(ispadded==true){
      if(preserve_originals==true){
	REA_padded_sequence=sequence;
      }else{
	REA_padded_sequence.swap(sequence);
      }
      makeComplement(REA_padded_sequence, REA_padded_complementsequence);
    }else{
      if(preserve_originals==true){
	REA_padded_sequence=sequence;
      }else{
	REA_padded_sequence.swap(sequence);
      }
      makeComplement(REA_padded_sequence, REA_padded_complementsequence);

    }
  }

  // handle the qualities
  if(iscomplement==true){
    REA_qualities.reserve(qualities.size());
    vector<base_quality_t>::reverse_iterator I=qualities.rbegin();
    while(I!=qualities.rend()){
      REA_qualities.push_back(*I++);
    }
  }else{
    if(preserve_originals==true){
      REA_qualities=qualities;
    }else{
      REA_qualities.swap(qualities);
    }
  }
  setQualityFlag(true);

  // handle the adjustments
  if(REA_uses_adjustments){
    if(iscomplement==true){
      REA_adjustments.reserve(adjustments.size());
      vector<int32>::reverse_iterator I=adjustments.rbegin();
      while(I!=adjustments.rend()){
	REA_adjustments.push_back(*I++);
      }
    }else{
      if(preserve_originals==true){
	REA_adjustments=adjustments;
      }else{
	REA_adjustments.swap(adjustments);
      }
    }
  }

  // handle the tags

  {
    REA_tags.resize(tags.size());
    vector<tag_t>::const_iterator fI=tags.begin();
    vector<multitag_t>::iterator tI=REA_tags.begin();
    for(; fI != tags.end(); ++tI, ++fI){
      *tI=*fI;
    }
  }

  // handle the baseflags (not given, so set everything to default)
  clearAllBPosHashStats();

  // handle the names
  setName(name);
  REA_scf_filename=SCFname;


  // handle the normal clippings
  
  REA_qr=qr;
  REA_ql=ql;
  REA_sr=sr;
  REA_sl=sl;
  REA_cr=cr;
  REA_cl=cl;
  updateClipoffs();

  // transfer Mira INFormation in MINF tags to read variables
  transferMINFTagsToReadInfo();

  // getLenSeq needs a valid read!
  REA_has_valid_data=true;
  REA_ml=0;                           
  REA_mr=getLenSeq();
  
  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::removeGapsFromRead()
{
  FUNCSTART("void Read::removeGapsFromRead()");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  refreshPaddedSequence();

  // TODO: check what happens with gaps at front of reads?
  //bool del=false;
  for(int32 i=0; i<static_cast<int32>(REA_padded_sequence.size());i++){
    if(REA_padded_sequence[i]=='*'){
      //del=true;
      deleteBaseFromSequence(i);
      i--;
    }
  }

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::fixZeroGapQuals()
{
  FUNCSTART("void Read::fixZeroGapQuals()");

  // cannot do anything in these cases
  if(REA_qualities.empty() || getLenSeq()<2) return;

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  refreshPaddedSequence();

  for(int32 i=0; i<static_cast<int32>(REA_padded_sequence.size());i++){
    if(REA_padded_sequence[i]=='*' && REA_qualities[i]==0){
      int32 lowpos=i;
      for(; lowpos>=0 && REA_padded_sequence[lowpos]=='*'; --lowpos);
      if(lowpos<0) lowpos=0;
      int32 highpos=i;
      for(; highpos<REA_padded_sequence.size() && REA_padded_sequence[highpos]=='*'; ++highpos);
      if(highpos>=REA_padded_sequence.size()) highpos=REA_padded_sequence.size()-1;
      REA_qualities[i]=(REA_qualities[lowpos]+REA_qualities[highpos])/2;
    }
  }

  FUNCEND();
  return;
}



/*************************************************************************
 *
 * for strobed PacBio reads which have stretches like ...x*****NNNNNNNNNx...
 *
 * in stretch given, insert or delete the number of Ns given
 *
 * from, to: including! from can be > to
 * changeestim <0: delete Ns; >0 insert Ns
 *
 *************************************************************************/

void Read::correctNStretch(int32 from, int32 to, int32 changeestim)
{
  FUNCSTART("void Read::correctNStretch(const int32 from, const int32 to, int32 changeestim)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  
  if(from>to) swap(from,to);

  refreshPaddedSequence();
	     
  BUGIFTHROW(from<0,getName() << ": from (" << from << ") <0 ?");
  BUGIFTHROW(from>=REA_padded_sequence.size(),getName() << ": from (" << from << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");
  BUGIFTHROW(to<0,getName() << ": to (" << to << ") <0 ?");
  BUGIFTHROW(to>=REA_padded_sequence.size(),getName() << ": to (" << to << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");
			
  if(changeestim==0) return;

  bool ok=true;
  for(int acti=from; acti <= to; acti++){
    if(REA_padded_sequence[acti] != '*'
       && REA_padded_sequence[acti] != 'N'
       && REA_padded_sequence[acti] != 'n'){
      ok=false;
      break;
    }
  }

  BUGIFTHROW(!ok, getName() << " from: " << from << " to: " << to << " not exclusively N or *?");

  if(changeestim>0){
    // insert Ns
    // first, try to convert gaps to N
    // TODO: slow, inefficient
    for(int acti=from; acti <= to && changeestim!=0; acti++){
      if(REA_padded_sequence[acti] == '*'){
	changeBaseInSequence('N', 1, static_cast<uint32>(acti));
	--changeestim;
      }
    }
    // if still need to insert, insert them
    // TODO: slow, inefficient
    if(changeestim!=0){
      for(; changeestim != 0; --changeestim){
	insertBaseInSequence('N', 1, static_cast<uint32>(from),false);
      }
    }
  }else{
    // delete Ns
    for(int acti=from; acti <= to && changeestim!=0; acti++){
      if(REA_padded_sequence[acti] == 'N'
	 || REA_padded_sequence[acti] == 'n'){
	deleteBaseFromSequence(static_cast<uint32>(acti));
	--to;
	++changeestim;
      }
    }
  }

  FUNCEND();
}



/*************************************************************************
 *
 * for strobed PacBio reads which have stretches like ...x*****NNNNNNNNNx...
 *
 * if minimum of (10) N in a gap/N stretch, transform all gaps of that
 *  stretch to N
 *
 *************************************************************************/

void Read::transformGapsToNs()
{
  FUNCSTART("void Read::transformGapsToNs()");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  refreshPaddedSequence();

  const uint32 minns=10;

  vector<char>::iterator cS=REA_padded_sequence.begin();
  vector<base_quality_t>::iterator qS=REA_qualities.begin();
  for(; cS != REA_padded_sequence.end(); cS++, qS++){
    if(*cS == '*'
       || *cS == 'N'
       || *cS == 'n'){
      vector<char>::iterator cE=cS;
      uint32 ncount=0;
      for(; cE != REA_padded_sequence.end(); cE++){
	if(*cE == 'N' || *cE == 'n') {
	  ncount++;
	}else if(*cE != '*'){
	  break;
	}
      }
      uint32 ccount=0;
      for(; cS != cE; cS++, qS++){
	if(ncount>=minns
	   && *cS == '*'
	   && ccount <5 ) {
	  *cS='N';
	  *qS=1;
	  ccount++;
	}
      }
      // we might have arrived at the end of the read in the inner loop
      //  decrease by one so that the outer for will arrive at the end
      if(cS==REA_padded_sequence.end()){
	--cS;
	--qS;
      }
    }
  }

  FUNCEND();
  return;
}



/*************************************************************************
 *
 * return the adjustment position of a read position
 *
 *************************************************************************/

int32 Read::getAdjustmentPosOfReadPos(const uint32 position) const
{
  FUNCSTART("int32 getAdjustmentPosOfReadPos(int32 position)");

  if(REA_uses_adjustments){
    BUGIFTHROW(position>=REA_adjustments.size(), getName() << ": readpos (" << position << ") >= size of read (" << REA_adjustments.size() << ")?");
    
    FUNCEND();
    return REA_adjustments[position];
  }

  const_cast<Read *>(this)->refreshPaddedSequence();

  if(REA_padded_sequence[position]=='*') return -1;
  return position;
}


/*************************************************************************
 *
 * return the nearest adjustment position that is not a -1 of a read position
 * (searching toward lower)
 *
 *************************************************************************/

int32 Read::getLowerNonGapAdjustmentPosOfReadPos(uint32 position) const
{
  FUNCSTART("int32 Read::getLowerNonGapAdjustmentPosOfReadPos(const uint32 position) const");

  if(REA_uses_adjustments){
    BUGIFTHROW(position>=REA_adjustments.size(), getName() << ": readpos (" << position << ") >= size of read (" << REA_adjustments.size() << ")?");

    while(position>0 && REA_adjustments[position] == -1) position--;
    
    FUNCEND();
    return REA_adjustments[position];
  }

  const_cast<Read *>(this)->refreshPaddedSequence();
  while(position>0 && REA_padded_sequence[position] == '*') position--;

  int32 adjpos=0;
  vector<char>::const_iterator cI=REA_padded_sequence.begin();
  for(int32 i=0; i<position; i++, cI++){
    if(*cI!='*') adjpos++;
  }

  FUNCEND();
  return adjpos;
}


/*************************************************************************
 *
 * return the nearest adjustment position that is not a -1 of a read position
 * (searching toward upper)
 *
 *************************************************************************/

int32 Read::getUpperNonGapAdjustmentPosOfReadPos(uint32 position) const
{
  FUNCSTART("int32 Read::getUpperNonGapAdjustmentPosOfReadPos(const uint32 position) const");

  if(REA_uses_adjustments){
    BUGIFTHROW(position>=REA_adjustments.size(), getName() << ": readpos (" << position << ") >= size of read (" << REA_adjustments.size() << ")?");
    
    while(position<REA_adjustments.size()-1 && REA_adjustments[position] == -1) position++;
    
    FUNCEND();
    return REA_adjustments[position];
  }

  const_cast<Read *>(this)->refreshPaddedSequence();
  while(position<REA_padded_sequence.size()-1 && REA_padded_sequence[position] == '*') position++;

  int32 adjpos=0;
  vector<char>::const_iterator cI=REA_padded_sequence.begin();
  for(int32 i=0; i<position; i++, cI++){
    if(*cI!='*') adjpos++;
  }

  return adjpos;
}




/*************************************************************************
 *
 * return the nearest position that is not a gap
 * (searching toward lower)
 *
 *************************************************************************/

int32 Read::getLowerNonGapPosOfReadPos(uint32 position) const
{
  FUNCSTART("int32 Read::getLowerNonGapPosOfReadPos(const uint32 position) const");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(position>=REA_padded_sequence.size(),getName() << ": position (" << position << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  while(position>0 && REA_padded_sequence[position] == '*') --position;

  FUNCEND();
  return position;
}


/*************************************************************************
 *
 * return the nearest position that is not a gap
 * (searching toward upper)
 *
 *************************************************************************/

int32 Read::getUpperNonGapPosOfReadPos(uint32 position) const
{
  FUNCSTART("int32 Read::getUpperNonGapPosOfReadPos(const uint32 position) const");

  const_cast<Read *>(this)->refreshPaddedSequence();
  BUGIFTHROW(position>=REA_padded_sequence.size(),getName() << ": position (" << position << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  while(position<REA_padded_sequence.size()-1 && REA_padded_sequence[position] == '*') ++position;

  return position;
}







/*************************************************************************
 *
 * given an adjustment position, return a read position (if it exists)
 * else -1
 *
 *************************************************************************/

int32 Read::getReadPosOfAdjustmentPos(const int32 position) const
{
  FUNCSTART("int32 getReadPosOfAdjustmentPos(int32 position)");

  if(REA_uses_adjustments){
    vector<int32>::const_iterator adjI=REA_adjustments.begin();
    for(;adjI!=REA_adjustments.end(); adjI++) {
      if(*adjI==position) {
	FUNCEND();
	return static_cast<int32>(adjI-REA_adjustments.begin());
      }
    }

    FUNCEND();
    return -1;
  }

  return position;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::setTags(const vector<tag_t> & tags)
{
  FUNCSTART("void setTags(const vector<tag_t> & tags)");

  vector<tag_t>::const_iterator I=tags.begin();
  for(;I!=tags.end();I++){
    try{
      addTag(*I,false);
    }
    catch (Notify n){
      cout << *I;
      n.handleError(THISFUNC);
    }
  }
  FUNCEND();
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::setTags(const vector<multitag_t> & tags)
{
  FUNCSTART("void setTags(const vector<multitag_t> & tags)");

  vector<multitag_t>::const_iterator I=tags.begin();
  for(;I!=tags.end();I++){
    try{
      addTag(*I,false);
    }
    catch (Notify n){
      cout << *I;
      n.handleError(THISFUNC);
    }
  }
  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::addTag(const tag_t & tag, bool onlygoodqual)
{
  FUNCSTART("void Read::addTag(tag_t & tag)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  BUGIFTHROW(tag.to>=getLenSeq(), getName() << ": tag to (" << tag.to << ") >= len of sequence (" << getLenSeq() << ")?");
  BUGIFTHROW(tag.from>tag.to, getName() << ": tag from (" << tag.from << ") > tag.to (" << tag.to << ")?");

  //REMOVEME: ?? at the moment, i decided to go for optional parameter
  if(onlygoodqual) {
    BUGIFTHROW(static_cast<int32>(tag.from)<getLeftClipoff(), getName() << ": from (" << tag.from << ") < getLeftClipoff() (" << getLeftClipoff() << ")?");
    BUGIFTHROW(static_cast<int32>(tag.to)>=getRightClipoff(), getName() << ": to (" << tag.to << " >= getRightClipoff() (" << getRightClipoff() << ")?");
  }

  // simple check for tag duplicates 
  // TODO: MIRA started to use multiple base tags at this time, 
  //  so this should be perhaps reworked.
  // New: if tag is already there, then backpack information (comment
  //  at the time) is copied into the location.
  //  Cannot use string.swap() as tag might be present multiple times
  //   (should not happen, but might be due to loading non-MIRA tags)

  bool mustaddtag=true;
  vector<multitag_t>::iterator I=REA_tags.begin();
  for(;I!=REA_tags.end();I++) {
    if(I->from == tag.from
       && I->to == tag.to
       && I->strand == tag.strand
       && I->getIdentifierStr() == tag.identifier) {
      mustaddtag=false;
      I->comment=multitag_t::newComment(tag.comment.c_str());
    }
  }

  if(mustaddtag) {
    REA_tags.resize(REA_tags.size()+1);
    REA_tags.back()=tag;
  }

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::addTag(const multitag_t & tag, bool onlygoodqual)
{
  FUNCSTART("void Read::addTag(multitag_t & tag)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  BUGIFTHROW(tag.to>=getLenSeq(), getName() << ": tag to (" << tag.to << ") >= len of sequence (" << getLenSeq() << ")?");
  BUGIFTHROW(tag.from>tag.to, getName() << ": tag from (" << tag.from << ") > tag.to (" << tag.to << ")?");

  //REMOVEME: ?? at the moment, i decided to go for optional parameter
  if(onlygoodqual) {
    BUGIFTHROW(static_cast<int32>(tag.from)<getLeftClipoff(), getName() << ": from (" << tag.from << ") < getLeftClipoff() (" << getLeftClipoff() << ")?");
    BUGIFTHROW(static_cast<int32>(tag.to)>=getRightClipoff(), getName() << ": to (" << tag.to << " >= getRightClipoff() (" << getRightClipoff() << ")?");
  }

  // simple check for tag duplicates 
  // TODO: MIRA started to use multiple base tags at this time, 
  //  so this should be perhaps reworked.
  // New: if tag is already there, then backpack information (comment
  //  at the time) is copied into the location.
  //  Cannot use string.swap() as tag might be present multiple times
  //   (should not happen, but might be due to loading non-MIRA tags)

  bool mustaddtag=true;
  vector<multitag_t>::iterator I=REA_tags.begin();
  for(;I!=REA_tags.end();I++) {
    if(I->from == tag.from
       && I->to == tag.to
       && I->strand == tag.strand
       && I->identifier == tag.identifier) {
      mustaddtag=false;
      I->comment=tag.comment;
    }
  }

  if(mustaddtag) {
    REA_tags.resize(REA_tags.size()+1);
    REA_tags.back()=tag;
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::addTag(uint32 from, uint32 to,
		  const string & identifier,
		  const string & comment)
{
  FUNCSTART("void Read::addTag(uint32 from, uint32 to, const string & identifier, const string & comment)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  BUGIFTHROW(to>=getLenSeq(), getName() << ": to (" << to << ") >= len of sequence (" << getLenSeq() << ") ?");
  BUGIFTHROW(from>to, getName() << ": from (" << from << ") > to (" << to << ") ?");

  ////REMOVEME: ?? at the moment, i decided to go for optional parameter
  //if(onlygoodqual) {
  //  BUGIFTHROW(static_cast<int32>(from)<getLeftClipoff(), "from < getLeftClipoff()?");
  //  BUGIFTHROW(static_cast<int32>(to)>=getRightClipoff(), "to >= getRightClipoff()?");
  //}

  bool mustaddtag=true;
  vector<multitag_t>::iterator I=REA_tags.begin();
  for(;I!=REA_tags.end();I++) {
    if(I->from == from
       && I->to == to
       && I->getIdentifierStr() == identifier) {
      mustaddtag=false;
      if(!I->getCommentStr().empty()
	 || !comment.empty()){
	I->comment=multitag_t::newComment(comment);
      }
    }
  }

  if(mustaddtag){
    REA_tags.resize(REA_tags.size()+1);
    REA_tags.back().from=from;
    REA_tags.back().to=to;
    REA_tags.back().identifier=multitag_t::newIdentifier(identifier);
    REA_tags.back().strand='=';
    if(!comment.empty()){
      REA_tags.back().comment=multitag_t::newComment(comment);
    }
  }

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::addTag(uint32 from, uint32 to,
		  const multitag_t::mte_id_t identifier,
		  const multitag_t::mte_co_t comment)
{
  FUNCSTART("void Read::addTag(...)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  BUGIFTHROW(to>=getLenSeq(), getName() << ": to (" << to << ") >= len of sequence (" << getLenSeq() << ") ?");
  BUGIFTHROW(from>to, getName() << ": from (" << from << ") > to (" << to << ") ?");

  ////REMOVEME: ?? at the moment, i decided to go for optional parameter
  //if(onlygoodqual) {
  //  BUGIFTHROW(static_cast<int32>(from)<getLeftClipoff(), "from < getLeftClipoff()?");
  //  BUGIFTHROW(static_cast<int32>(to)>=getRightClipoff(), "to >= getRightClipoff()?");
  //}

  bool mustaddtag=true;
  vector<multitag_t>::iterator I=REA_tags.begin();
  for(;I!=REA_tags.end();I++) {
    if(I->from == from
       && I->to == to
       && I->identifier == identifier) {
      mustaddtag=false;
      if(I->comment!=comment){
	I->comment=comment;
      }
    }
  }

  if(mustaddtag){
    REA_tags.resize(REA_tags.size()+1);
    REA_tags.back().from=from;
    REA_tags.back().to=to;
    REA_tags.back().identifier=identifier;
    REA_tags.back().strand='=';
    REA_tags.back().comment=comment;
  }

  FUNCEND();
}





/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::addTag(uint32 from, uint32 to,
		  const multitag_t::mte_id_t identifier,
		  const string & comment)
{
  FUNCSTART("void Read::addTag(...)");

  addTag(from,to,
	 identifier,
	 multitag_t::newComment(comment));

  FUNCEND();
}





///*************************************************************************
// *
// *
// *
// *
// *************************************************************************/
//
//bool Read::hasTag(const string & identifier, int32 pos) const
//{
//  FUNCSTART("bool Read::hasTag(const string & identifier, uint32 pos)");
//
//  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
//  
//  BUGIFTHROW(pos>=0 && pos>=static_cast<int32>(REA_qualities.size()),
//	     getName() << ": pos(" << pos << ") > read size " << REA_qualities.size() << ")?");
//
//  // handle case where we don't care about tag position
//  if(pos<0) {
//    vector<multitag_t>::const_iterator I=REA_tags.begin();
//    for(;I!=REA_tags.end();I++) {
//      if(I->getIdentifierStr()==identifier){
//	// we found this id in the read, we don't need to examine further
//	FUNCEND();
//	return true;
//      }
//    }
//    FUNCEND();
//    return false;
//  }
//
//  // here we need to care about tag position
//  vector<multitag_t>::const_iterator I=REA_tags.begin();
//  for(;I!=REA_tags.end();I++) {
//    if(static_cast<int32>(I->to) < pos
//       || static_cast<int32>(I->from) > pos) continue;
//    // tag at  specified position, check identifier
//    if(I->getIdentifierStr()==identifier){
//      // we found this id in the read at that pos, 
//      //  we don't need to examine further
//      FUNCEND();
//      return true;
//    }
//  }
//
//  FUNCEND();
//  return false;
//}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

multitag_t::mte_id_t Read::getTagID(const string & identifier)
{
  return multitag_t::hasIdentifier(identifier);
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

bool Read::hasTag(const string & identifier, int32 pos) const
{
  multitag_t::mte_id_t id=multitag_t::hasIdentifier(identifier);
  if(!multitag_t::getIdentifierStr(id).empty()){
    return hasTag(id,pos);
  }
  return false;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

bool Read::hasTag(const multitag_t::mte_id_t identifier, int32 pos) const
{
  FUNCSTART("Read::hasTag(const multitag_t::mte_id_t identifier, int32 pos) const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  
  BUGIFTHROW(pos>=0 && pos>=static_cast<int32>(REA_qualities.size()),
	     getName() << ": pos(" << pos << ") > read size " << REA_qualities.size() << ")?");

  // handle case where we don't care about tag position
  if(pos<0) {
    vector<multitag_t>::const_iterator I=REA_tags.begin();
    for(;I!=REA_tags.end();I++) {
      if(I->identifier==identifier){
	// we found this id in the read, we don't need to examine further
	FUNCEND();
	return true;
      }
    }
    FUNCEND();
    return false;
  }

  // here we need to care about tag position
  vector<multitag_t>::const_iterator I=REA_tags.begin();
  for(;I!=REA_tags.end();I++) {
    if(static_cast<int32>(I->to) < pos
       || static_cast<int32>(I->from) > pos) continue;
    // tag at  specified position, check identifier
    if(I->identifier==identifier){
      // we found this id in the read at that pos, 
      //  we don't need to examine further
      FUNCEND();
      return true;
    }
  }

  FUNCEND();
  return false;
}

///*************************************************************************
// *
// *
// *
// *
// *************************************************************************/
//
//uint32  Read::countTags(const string & identifier, int32 pos) const
//{
//  FUNCSTART("bool Read::countTags(const string & identifier, uint32 pos)");
//
//  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
//  
//  BUGIFTHROW(pos>=0 && pos>=static_cast<int32>(REA_qualities.size()),
//	     getName() << ": pos(" << pos << ") > read size " << REA_qualities.size() << ")?");
//
//  // handle case where we don't care about tag position
//  uint32 retcount=0;
//  vector<multitag_t>::const_iterator I=REA_tags.begin();
//  if(pos<0) {
//    for(;I!=REA_tags.end();I++) {
//      if(I->getIdentifierStr()==identifier){
//	// we found this id in the read, we don't need to examine further
//	retcount++;
//      }
//    }
//  }else{
//    // here we need to care about tag position
//    for(;I!=REA_tags.end();I++) {
//      if(static_cast<int32>(I->to) < pos
//	 || static_cast<int32>(I->from) > pos) continue;
//      // tag at  specified position, check identifier
//      if(I->getIdentifierStr()==identifier){
//	retcount++;
//      }
//    }
//  }
//
//  FUNCEND();
//  return retcount;
//}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

uint32 Read::countTags(const multitag_t::mte_id_t identifier, int32 pos) const
{
  FUNCSTART("uint32 Read::countTags(const multitag_t::mte_id_t identifier, int32 pos) const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  
  BUGIFTHROW(pos>=0 && pos>=static_cast<int32>(REA_qualities.size()),
	     getName() << ": pos(" << pos << ") > read size " << REA_qualities.size() << ")?");

  // handle case where we don't care about tag position
  uint32 retcount=0;
  vector<multitag_t>::const_iterator I=REA_tags.begin();
  if(pos<0) {
    for(;I!=REA_tags.end();I++) {
      if(I->identifier==identifier){
	// we found this id in the read, we don't need to examine further
	retcount++;
      }
    }
  }else{
    // here we need to care about tag position
    for(;I!=REA_tags.end();I++) {
      if(static_cast<int32>(I->to) < pos
	 || static_cast<int32>(I->from) > pos) continue;
      // tag at  specified position, check identifier
      if(I->identifier==identifier){
	retcount++;
      }
    }
  }

  FUNCEND();
  return retcount;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const multitag_t & Read::getTag(uint32 tagnumber) const 
{
  FUNCSTART("const multitag_t & Read::getTag(uint32 tagnumber) const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  BUGIFTHROW(tagnumber >= REA_tags.size(),
	     getName() << ": tagnumber (" << tagnumber << ") >= REA_tags.size (" << REA_tags.size() << ") ?");

  FUNCEND();

  return REA_tags[tagnumber];
}





/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::refreshPaddedSequence()
{
  FUNCSTART("void Read::refreshPaddedSequence()");
  
  if(REA_ps_dirty==true){
    if(REA_pcs_dirty==true){
      throw Notify(Notify::FATAL, THISFUNC, "Both seq and compl.seq. are tagged dirty.");
    }
    BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

    makeComplement(REA_padded_complementsequence, REA_padded_sequence);
    CEBUG("Refreshed.\n");
    REA_ps_dirty=false;

    BUGIF(sanityCheck()!=NULL, cout << *this);
    BUGIF(sanityCheck()!=NULL, throw Notify(Notify::FATAL, THISFUNC, sanityCheck()));

  }

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Read::refreshPaddedComplementSequence()
{
  FUNCSTART("void Read::refreshPaddedComplementSequence()");
  
  if(REA_pcs_dirty==true){
    if(REA_ps_dirty==true){
      throw Notify(Notify::FATAL, THISFUNC, "Both seq and compl.seq. are tagged dirty.");
    }
    BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

    CEBUG("Refreshed.\n");
    makeComplement(REA_padded_sequence, REA_padded_complementsequence);
    REA_pcs_dirty=false;
    BUGIF(sanityCheck()!=NULL, cout << *this);
    BUGIF(sanityCheck()!=NULL, throw Notify(Notify::FATAL, THISFUNC, sanityCheck()));

  }

  FUNCEND();
}




// Tested: Insert/Delete/Change with caching methods + clips

/*************************************************************************
 *
 * Inserts a base at the given position with the given base qualities.
 *    E.g.:   ATGC   inserting N at pos 2 gives    ATNGC
 *
 * If extends_clipped_area is true and the position given is on the border
 * of a clip, the clip will be moved in that way to extend the clipped
 * area.
 *    E.g.:   ATGC with CL=2   (meaning AT GC)
 *            inserting N at pos 2 and extends_clipped_area=true gives
 *            ATNGC with CL=2   (meaning AT NGC)
 *    E.g.:   inserting N at pos 2 and extends_clipped_area=false gives
 *            ATNGC with CL=3   (meaning ATN GC)
 *    E.g.:   ATGC with CR=2   (meaning AT GC)    Note the R for Right!!!
 *            inserting N at pos 2 and extends_clipped_area=true gives
 *            ATNGC with CR=3   (meaning ATN GC)
 *
 * Tags will NEVER! be extended!
 *
 *tested
 *************************************************************************/

void Read::insertBaseInSequence(char base, base_quality_t quality, uint32 position, bool extends_clipped_area)
{
  FUNCSTART("void Read::insertBaseInSequence(char base, base_quality_t quality, int32 position, bool extends_clipped_area)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  CEBUG("Position: " << position << endl);

  if(REA_ps_dirty==true && REA_pcs_dirty==false){
    insertBaseInComplementSequence(dptools::getComplementIUPACBase(base),
				   quality,
				   static_cast<uint32>(REA_padded_complementsequence.size())-position,
				   extends_clipped_area);
  }else{
    if(position>REA_padded_sequence.size()){
      throw Notify(Notify::FATAL, THISFUNC, "Position > size of read?");
    }

    refreshPaddedSequence();

    moderateContainerGrowth();

    {
      vector<char>::iterator I=REA_padded_sequence.begin();
      BUGIFTHROW(position>REA_padded_sequence.size(),getName() << ": position (" << position << ") > REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");
      advance(I, position);
      REA_padded_sequence.insert(I, base);
    }

    {
      vector<base_quality_t>::iterator J=REA_qualities.begin();
      BUGIFTHROW(position>REA_qualities.size(),getName() << ": position (" << position << ") > REA_qualities.size (" << REA_qualities.size() << ") ?");
      advance(J,position);
      REA_qualities.insert(J, quality);
    }

    // insert -1 into adjustment vector, as this base is not in the
    //  original sequence
    if(REA_uses_adjustments){
      vector<int32>::iterator K=REA_adjustments.begin();
      BUGIFTHROW(position > REA_adjustments.size(),getName() << ": position (" << position << " > REA_adjustments.size (" << REA_adjustments.size() << ") ?");
      advance(K, position);
      REA_adjustments.insert(K, -1);
    }

    // insert baseflags into baseflags_t vector (with defaults)
    {
      vector<bposhashstat_t>::iterator B=REA_bposhashstats.begin();
      BUGIFTHROW(position > REA_bposhashstats.size(),getName() << ": position (" << position << " > REA_bposhashstats.size (" << REA_bposhashstats.size() << ") ?");
      advance(B, position);
      REA_bposhashstats.insert(B, REA_bposhashstat_default);
    }


    REA_pcs_dirty=true;

    if(extends_clipped_area==true){
      if(static_cast<int32>(position) <  REA_ql) REA_ql++;
      if(static_cast<int32>(position) <  REA_sl) REA_sl++;
      if(static_cast<int32>(position) <  REA_ml) REA_ml++;
      if(static_cast<int32>(position) <= REA_qr) REA_qr++;
      if(static_cast<int32>(position) <= REA_sr) REA_sr++;
      if(static_cast<int32>(position) <= REA_mr) REA_mr++;
      if(REA_cl>=0 &&
	 static_cast<int32>(position) <  REA_cl) REA_cl++;
      if(REA_cr>=0 &&
	 static_cast<int32>(position) <= REA_cr) REA_cr++;
    }else{
      if(static_cast<int32>(position) <= REA_ql) REA_ql++;
      if(static_cast<int32>(position) <= REA_sl) REA_sl++;
      if(static_cast<int32>(position) <= REA_ml) REA_ml++;
      if(static_cast<int32>(position) <  REA_qr) REA_qr++;
      if(static_cast<int32>(position) <  REA_sr) REA_sr++;
      if(static_cast<int32>(position) <  REA_mr) REA_mr++;
      if(REA_cl>=0 &&
	 static_cast<int32>(position) <= REA_cl) REA_cl++;
      if(REA_cr>=0 &&
	 static_cast<int32>(position) <  REA_cr) REA_cr++;
    }

    updateClipoffs();
    updateTagBaseInserted(position);

  }

//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  BUGIF(sanityCheck()!=NULL, throw Notify(Notify::FATAL, THISFUNC, sanityCheck()));

  FUNCEND();
}


// TODO: check and test tag handling
/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::updateTagBaseInserted(uint32 position)
{
  FUNCSTART("void Read::updateTagInsert(uint32 position)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  vector<multitag_t>::iterator tI=REA_tags.begin();
  for(;tI!=REA_tags.end();tI++){
    if(tI->from >= position) tI->from++;
    if(tI->to >= position) tI->to++;
  }

  FUNCEND();
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::updateTagBaseDeleted(uint32 position)
{
  FUNCSTART("void Read::updateTagBaseDeleted(int32 position)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  vector<multitag_t>::iterator tI=REA_tags.begin();
  while(tI!=REA_tags.end()){
    if(tI->from == tI->to &&
       tI->to == position){
      tI=REA_tags.erase(tI);
    }else{
      if(position < tI->from) tI->from--;
      if(position <= tI->to) tI->to--;
      tI++;
    }
  }

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *tested
 *************************************************************************/
void Read::deleteBaseFromSequence(uint32 position)
{
  FUNCSTART("void Read::deleteBaseFromSequence(int32 position)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  CEBUG("Position: " << position << endl);

  if(REA_ps_dirty==true && REA_pcs_dirty==false){
    deleteBaseFromComplementSequence(static_cast<uint32>(REA_padded_complementsequence.size())-position-1);
  }else{
    if(position>=REA_padded_sequence.size()){
      throw Notify(Notify::FATAL, THISFUNC, "Position >= size of read?");
    }

    refreshPaddedSequence();

    {
      vector<char>::iterator I=REA_padded_sequence.begin();
      //BOUNDCHECK(position, 0, REA_padded_sequence.size());
      BUGIFTHROW(position>=REA_padded_sequence.size(),getName() << ": position (" << position << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");
      advance(I, position);
      REA_padded_sequence.erase(I);
    }

    {
      vector<base_quality_t>::iterator J=REA_qualities.begin();
      //BOUNDCHECK(position, 0, REA_qualities.size());
      BUGIFTHROW(position>=REA_qualities.size(), getName() << ": position (" << position << ") >= REA_qualities.size (" << REA_qualities.size() << ") ?");
      advance(J,position);
      REA_qualities.erase(J);
    }

    if(REA_uses_adjustments){
      vector<int32>::iterator K=REA_adjustments.begin();
      //BOUNDCHECK(position, 0, REA_adjustments.size());
      BUGIFTHROW(position >= REA_adjustments.size(),getName() << ": position (" << position << " >= REA_adjustments.size (" << REA_adjustments.size() << ") ?");
      advance(K, position);
      REA_adjustments.erase(K);
    }

    {
      vector<bposhashstat_t>::iterator B=REA_bposhashstats.begin();
      //BOUNDCHECK(position, 0, REA_bposhashstats.size());
      BUGIFTHROW(position >= REA_bposhashstats.size(),getName() << ": position (" << position << " >= REA_bposhashstats.size (" << REA_bposhashstats.size() << ") ?");
      advance(B, position);
      REA_bposhashstats.erase(B);
    }

    REA_pcs_dirty=true;

    // FALSCH! was <, but I think that was wrong. <= should be right 
    // TODO: was <, but I think that was wrong. <= should be right 
    if(static_cast<int32>(position)<REA_ql && REA_ql > 0) REA_ql--;
    if(static_cast<int32>(position)<REA_qr && REA_qr > 0) REA_qr--;
    if(static_cast<int32>(position)<REA_ml && REA_ml > 0) REA_ml--;
    if(static_cast<int32>(position)<REA_mr && REA_mr > 0) REA_mr--;
    if(static_cast<int32>(position)<REA_sl && REA_sl > 0) REA_sl--;
    if(static_cast<int32>(position)<REA_sr && REA_sr > 0) REA_sr--;
    if(REA_cl>=0 &&
       static_cast<int32>(position)<REA_cl && REA_cl > 0) REA_cl--;
    if(REA_cr>=0 &&
       static_cast<int32>(position)<REA_cr && REA_cr > 0) REA_cr--;

    updateClipoffs();
    updateTagBaseDeleted(position);

  }

//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *tested
 *************************************************************************/
void Read::insertBaseInComplementSequence(char base, base_quality_t quality, uint32 position, bool extends_clipped_area)
{
  FUNCSTART("void Read::insertBaseInComplementSequence(char base, base_quality_t quality, int32 position, bool extends_clipped_area)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  CEBUG("Position: " << position << endl);

  if(REA_pcs_dirty==true && REA_ps_dirty==false){
    insertBaseInSequence(dptools::getComplementIUPACBase(base),
			 quality,
			 static_cast<uint32>(REA_padded_sequence.size())-position,
			 extends_clipped_area);
  }else{
    if(position>REA_padded_complementsequence.size()){
      cout << *this;
      throw Notify(Notify::FATAL, THISFUNC, "Position > size of read?");
    }

    refreshPaddedComplementSequence();

    moderateContainerGrowth();

    uint32 complement_position=static_cast<uint32>(REA_padded_complementsequence.size())-position;

    CEBUG("compl_pos: " << complement_position << endl);

    {
      vector<char>::iterator I=REA_padded_complementsequence.begin();
      //BOUNDCHECK(position, 0, REA_padded_complementsequence.size()+1);
      BUGIFTHROW(position>REA_padded_complementsequence.size(),getName() << ": position (" << position << ") > REA_padded_complementsequence.size (" << REA_padded_complementsequence.size() << ") ?");
      advance(I, position);
      REA_padded_complementsequence.insert(I, base);
    }

    vector<base_quality_t>::iterator J=REA_qualities.begin();
    //BOUNDCHECK(complement_position, 0, REA_qualities.size()+1);
    BUGIFTHROW(complement_position>REA_qualities.size(),getName() << ": complement_position (" << complement_position << " > REA_qualities.size (" << REA_qualities.size() << ") ?");
    advance(J,complement_position);
    REA_qualities.insert(J, quality);

    // insert -1 into adjustment vector, as this base is not in the
    //  original sequence
    if(REA_uses_adjustments){
      vector<int32>::iterator K=REA_adjustments.begin();
      //BOUNDCHECK(complement_position, 0, REA_adjustments.size()+1);
      BUGIFTHROW(complement_position > REA_adjustments.size(),getName() << ": complement_position (" << complement_position << ") > REA_adjustments.size (" << REA_adjustments.size() << ")");
      advance(K, complement_position);
      REA_adjustments.insert(K, -1);
    }

    // insert baseflags into baseflags_t vector (with defaults)
    {
      vector<bposhashstat_t>::iterator B=REA_bposhashstats.begin();
      BUGIFTHROW(complement_position > REA_bposhashstats.size(),getName() << ": complement_position (" << position << " > REA_bposhashstats.size (" << REA_bposhashstats.size() << ") ?");
      advance(B, complement_position);
      REA_bposhashstats.insert(B, REA_bposhashstat_default);
    }

    REA_ps_dirty=true;

    if(extends_clipped_area==true){
      if(static_cast<int32>(complement_position) <  REA_ql) REA_ql++;
      if(static_cast<int32>(complement_position) <  REA_sl) REA_sl++;
      if(static_cast<int32>(complement_position) <  REA_ml) REA_ml++;
      if(static_cast<int32>(complement_position) <= REA_qr) REA_qr++;
      if(static_cast<int32>(complement_position) <= REA_sr) REA_sr++;
      if(static_cast<int32>(complement_position) <= REA_mr) REA_mr++;
      if(REA_cl>=0 &&
	 static_cast<int32>(complement_position) <  REA_cl) REA_cl++;
      if(REA_cr>=0 &&
	 static_cast<int32>(complement_position) <= REA_cr) REA_cr++;
    }else{
      if(static_cast<int32>(complement_position) <= REA_ql) REA_ql++;
      if(static_cast<int32>(complement_position) <= REA_sl) REA_sl++;
      if(static_cast<int32>(complement_position) <= REA_ml) REA_ml++;
      if(static_cast<int32>(complement_position) <  REA_qr) REA_qr++;
      if(static_cast<int32>(complement_position) <  REA_sr) REA_sr++;
      if(static_cast<int32>(complement_position) <  REA_mr) REA_mr++;
      if(REA_cl>=0 &&
	 static_cast<int32>(complement_position) <= REA_cl) REA_cl++;
      if(REA_cr>=0 &&
	 static_cast<int32>(complement_position) <  REA_cr) REA_cr++;
    }

    updateClipoffs();
    updateTagBaseInserted(complement_position);

  }

//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
}





/*************************************************************************
 *
 *
 *
 *
 *tested
 *************************************************************************/
void Read::deleteBaseFromComplementSequence(uint32 position)
{
  FUNCSTART("void Read::deleteBaseFromComplementSequence(int32 uposition)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  CEBUG("Position: " << position << endl);

  if(REA_pcs_dirty==true && REA_ps_dirty==false){
    deleteBaseFromSequence(static_cast<uint32>(REA_padded_sequence.size())-position-1);
  }else{
    if(position > REA_padded_complementsequence.size()){
      throw Notify(Notify::FATAL, THISFUNC, "Position >= size of read?");
    }

    refreshPaddedComplementSequence();

    uint32 complement_position=static_cast<uint32>(REA_padded_complementsequence.size())-position-1;

    {
      vector<char>::iterator I=REA_padded_complementsequence.begin();
      //BOUNDCHECK(position, 0, REA_padded_complementsequence.size()+1);
      BUGIFTHROW(position>REA_padded_complementsequence.size(),getName() << ": position (" << position << ") > REA_padded_complementsequence.size (" << REA_padded_complementsequence.size() << ") ?");
      advance(I, position);
      REA_padded_complementsequence.erase(I);
    }

    {
      vector<base_quality_t>::iterator J=REA_qualities.begin();
      //BOUNDCHECK(complement_position, 0, REA_qualities.size());
      BUGIFTHROW(complement_position>=REA_qualities.size(),getName() << ": complement_position (" << complement_position << ") >= REA_qualities.size (" << REA_qualities.size() << ") ?")
	advance(J,complement_position);
      REA_qualities.erase(J);
    }

    if(REA_uses_adjustments){
      vector<int32>::iterator K=REA_adjustments.begin();
      //BOUNDCHECK(complement_position, 0, REA_adjustments.size());
      BUGIFTHROW(complement_position>=REA_adjustments.size(),getName() << " complement_position (" << complement_position << ") >= REA_adjustments.size (" << REA_adjustments.size() << ") ?");
      advance(K, complement_position);
      REA_adjustments.erase(K);
    }

    {
      vector<bposhashstat_t>::iterator B=REA_bposhashstats.begin();
      //BOUNDCHECK(complement_position, 0, REA_bposhashstats.size());
      BUGIFTHROW(complement_position>=REA_bposhashstats.size(),getName() << " complement_position (" << complement_position << ") >= REA_bposhashstats.size (" << REA_bposhashstats.size() << ") ?");
      advance(B, complement_position);
      REA_bposhashstats.erase(B);
    }

    REA_ps_dirty=true;

    // TODO: was <, but I think that was wrong. <= should be right 
    // update: <= is wrong, due to new complpos calc, < should be right 
    if(static_cast<int32>(complement_position)<REA_ql && REA_ql > 0) REA_ql--;
    if(static_cast<int32>(complement_position)<REA_sl && REA_sl > 0) REA_sl--;
    if(static_cast<int32>(complement_position)<REA_ml && REA_ml > 0) REA_ml--;
    if(static_cast<int32>(complement_position)<REA_qr && REA_qr > 0) REA_qr--;
    if(static_cast<int32>(complement_position)<REA_sr && REA_sr > 0) REA_sr--;
    if(static_cast<int32>(complement_position)<REA_mr && REA_mr > 0) REA_mr--;
    if(REA_cl>=0 &&
       static_cast<int32>(complement_position)<REA_cl && REA_cl> 0) REA_cl--;
    if(REA_cr>=0 &&
       static_cast<int32>(complement_position)<REA_cr && REA_cr > 0) REA_cr--;

    updateClipoffs();
    updateTagBaseDeleted(complement_position);

  }

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  FUNCEND();
}






/*************************************************************************
 *
 *
 *
 *
 *tested
 *************************************************************************/

void Read::changeBaseInSequence(char base, base_quality_t quality, uint32 position)
{
  FUNCSTART("void changeBaseInSequence(char base, base_quality_t quality, int32 position)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  CEBUG("Position: " << position << endl);

  if(REA_ps_dirty==false){
    CEBUG("ps clean\n");
    if(position>=REA_padded_sequence.size()){
      throw Notify(Notify::FATAL, THISFUNC, "Position >= size of read?");
    }
    vector<char>::iterator I=REA_padded_sequence.begin();
    //BOUNDCHECK(position, 0, REA_padded_sequence.size());
    advance(I, position);
    *I=base;

    vector<base_quality_t>::iterator J=REA_qualities.begin();
    //BOUNDCHECK(position, 0, REA_qualities.size());
    BUGIFTHROW(position>=REA_qualities.size(),getName() << ": position (" << position << ") >= REA_qualities.size (" << REA_qualities.size() << ") ?");
    advance(J,position);
    *J=quality;
  }

  if(REA_pcs_dirty==false){
    CEBUG("pcs clean\n");
    BUGIFTHROW(position>=REA_padded_complementsequence.size(),getName() << ": position (" << position << ") >= REA_padded_complementsequence.size (" << REA_padded_complementsequence.size() << ") ?");

    uint32 cposition=static_cast<uint32>(REA_padded_complementsequence.size())-position-1;
    vector<char>::iterator I=REA_padded_complementsequence.begin();

    //BOUNDCHECK(cposition, 0, REA_padded_complementsequence.size());
    BUGIFTHROW(cposition>=REA_padded_complementsequence.size(),getName() << ": cposition (" << " ) >= REA_padded_complementsequence.size (" << REA_padded_complementsequence.size() << ") ?");

    advance(I, cposition);
    *I=dptools::getComplementIUPACBase(base);

    vector<base_quality_t>::iterator J=REA_qualities.begin();
    //BOUNDCHECK(position, 0, REA_qualities.size());
    BUGIFTHROW(position>=REA_qualities.size(),getName() << ": position (" << position << ") >= REA_qualities.size (" << REA_qualities.size() << ") ?");
    advance(J, position);
    *J=quality;
  }


  {
    vector<bposhashstat_t>::iterator B=REA_bposhashstats.begin();
    advance(B, position);
    *B=REA_bposhashstat_default;
  }

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
}





/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

void Read::changeBaseInComplementSequence(char base, base_quality_t quality, uint32 position)
{
  FUNCSTART("void changeBaseInComplementSequence(char base, base_quality_t quality, int32 position)");

  CEBUG("Position: " << position << endl);

  changeBaseInSequence(dptools::getComplementIUPACBase(base), quality, getLenSeq()-position-1);

//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *untested
 *************************************************************************/

void Read::changeAdjustment(uint32 position, int32 newadjustment)
{
  FUNCSTART("void Read::changeAdjustment(uint32 position, int32 newadjustment)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  BUGIFTHROW(REA_adjustments.empty(), getName() << ": trying to change adjustment in read which does not have any?");
  BUGIFTHROW(position>=REA_adjustments.size(), getName() << ": trying to change adjustment at position " << position << " but size of adjustment is only " << REA_adjustments.size());

  REA_adjustments[position]=newadjustment;

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

void Read::insertBaseInClippedSequence(char base, base_quality_t quality, uint32 position, bool extends_clipped_area)
{
  FUNCSTART("void Read::insertBaseInClippedSequence(char base, base_quality_t quality, uint32 position, bool extends_clipped_area)");
  
  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position > len of clipped read ?");
  }

  insertBaseInSequence(base, quality, position+getLeftClipoff(),
		       extends_clipped_area);

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

void Read::deleteBaseFromClippedSequence(uint32 position)
{
  FUNCSTART("void Read::deleteBaseFromClippedSequence(uint32 position)");

  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>=getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position >= len of clipped read ?");
  }

  deleteBaseFromSequence(position+getLeftClipoff());

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

void Read::insertBaseInClippedComplementSequence(char base, base_quality_t quality, uint32 position, bool extends_clipped_area)
{
  FUNCSTART("void Read::insertBaseInClippedComplementSequence(char base, base_quality_t quality, int32 position, bool extends_clipped_area)");
  
  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position > len of clipped read ?");
  }

  insertBaseInComplementSequence(base, quality,
				 getLenSeq()-getRightClipoff()+position,
				 extends_clipped_area);

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

void Read::deleteBaseFromClippedComplementSequence(uint32 position)
{
  FUNCSTART("void Read::deleteBaseFromClippedComplementSequence(int32 position)");

  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>=getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position >= len of clipped read ?");
  }

  deleteBaseFromComplementSequence(getLenSeq()-getRightClipoff()+position);

  FUNCEND();
}





/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::changeBaseInClippedSequence(char base, base_quality_t quality, uint32 position)
{
  FUNCSTART("void Read::changeBaseInClippedSequence(char base, base_quality_t quality, uint32 position)");

  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>=getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position >= len of clipped read ?");
  }

  changeBaseInSequence(base, quality, getLeftClipoff()+position);
  
  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
void Read::changeBaseInClippedComplementSequence(char base, base_quality_t quality, uint32 position)
{
  FUNCSTART("void Read::changeBaseInClippedComplementSequence(char base, base_quality_t quality, uint32 position)");

  CEBUG("Position: " << position << endl);

  if(static_cast<int32>(position)>=getLenClippedSeq()){
    throw Notify(Notify::FATAL, THISFUNC, "Position >= len of clipped read ?");
  }

  changeBaseInComplementSequence(base, quality, getLenSeq()-getRightClipoff()+position);
  
  FUNCEND();
}






/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/
uint32 Read::getLenSeq() const
{
  FUNCSTART("uint32 Read::getLenSeq() const");
  
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();
  if(REA_ps_dirty==false){
    return static_cast<uint32>(REA_padded_sequence.size());
  }else{
    return static_cast<uint32>(REA_padded_complementsequence.size());
  }
}

/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

// This should be
//    const vector<char> & Read::getActualSequence()  const
// too, but SGI CC 7.1 doesn't know the keyword mutable
const vector<char> & Read::getActualSequence() const
{
  FUNCSTART("const vector<char> & Read::getActualSequence() const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  const_cast<Read *>(this)->refreshPaddedSequence();

  FUNCEND();
  return REA_padded_sequence;
}


// This should be
//    const vector<char> & Read::getActualComplementSequence  const
// too, but SGI CC 7.1 doesn't know the keyword mutable
const vector<char> & Read::getActualComplementSequence() const
{
  FUNCSTART("const vector<char> & Read::getActualComplementSequence() const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
  const_cast<Read *>(this)->refreshPaddedComplementSequence();

  FUNCEND();
  return REA_padded_complementsequence;
}



/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/
// This should be const
// too, but SGI CC 7.1 doesn't know the keyword mutable
const char * Read::getClippedSeqAsChar() const
{
  FUNCSTART("const char * Read::getClippedSeqAsChar()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  const_cast<Read *>(this)->refreshPaddedSequence();

  if(REA_padded_sequence.empty()) return &REA_zerostring;

  vector<char>::const_iterator cI=REA_padded_sequence.begin();
  BOUNDCHECK(getLeftClipoff(), 0, static_cast<int32>(REA_padded_sequence.size()));
  advance(cI, getLeftClipoff());
  return &(*cI);
}



/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/
// This should be const
// too, but SGI CC 7.1 doesn't know the keyword mutable
const char * Read::getSeqAsChar() const
{
  FUNCSTART("const char * Read::getSeqAsChar()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  const_cast<Read *>(this)->refreshPaddedSequence();

  if(REA_padded_sequence.empty()) return &REA_zerostring;

  vector<char>::const_iterator cI=REA_padded_sequence.begin();
  return &(*cI);
}




/*************************************************************************
 *
 *
 *
 *tested
 *************************************************************************/

// This should be const
// too, but SGI CC 7.1 doesn't know the keyword mutable
const char * Read::getClippedComplementSeqAsChar() const
{
  FUNCSTART("const char * Read::getClippedComplementSeqAsChar()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  const_cast<Read *>(this)->refreshPaddedComplementSequence();

  if(REA_padded_complementsequence.empty()) return &REA_zerostring;

  vector<char>::const_iterator cI=REA_padded_complementsequence.begin();
  BOUNDCHECK(static_cast<int32>(REA_padded_complementsequence.size())-getRightClipoff(), 0, static_cast<int32>(REA_padded_complementsequence.size()));
  advance(cI, REA_padded_complementsequence.size()-getRightClipoff());
  
  return &(*cI);
}





/*************************************************************************
 *
 * Returns an iterator to the sequencing vector and quality clipped read
 *
 *tested
 *************************************************************************/

vector<char>::const_iterator Read::getClippedSeqIterator() const
{
  FUNCSTART("const vector<char>::iterator Read::getClippedSeqIterator()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //  const_cast<Read *>(this)->refreshPaddedSequence();
  const_cast<Read *>(this)->refreshPaddedSequence();
  
  vector<char>::const_iterator I=REA_padded_sequence.begin();

  if(getLeftClipoff() < 0 ||
     getLeftClipoff() >= static_cast<int32>(REA_padded_sequence.size())){
    setCoutType(AS_TEXT);
    cout << '\n' << *this << '\n';
    cout.flush();
  }

  BOUNDCHECK(getLeftClipoff(), 0, static_cast<int32>(REA_padded_sequence.size()));
  advance(I, getLeftClipoff());

  FUNCEND();

  return I;
}



/*************************************************************************
 *
 * Returns an iterator to the read
 *
 *tested
 *************************************************************************/

vector<char>::const_iterator Read::getSeqIteratorBegin() const
{
  FUNCSTART("const vector<char>::iterator Read::getSeqIteratorBegin()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //  const_cast<Read *>(this)->refreshPaddedSequence();
  const_cast<Read *>(this)->refreshPaddedSequence();
  
  FUNCEND();

  return REA_padded_sequence.begin();
}

vector<char>::const_iterator Read::getSeqIteratorEnd() const
{
  FUNCSTART("const vector<char>::iterator Read::getSeqIteratorEnd()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //  const_cast<Read *>(this)->refreshPaddedSequence();
  const_cast<Read *>(this)->refreshPaddedSequence();
  
  FUNCEND();

  return REA_padded_sequence.end();
}



/*************************************************************************
 *
 * Returns an iterator to the 
 *  complement read
 *
 *tested
 *************************************************************************/

vector<char>::const_iterator Read::getComplementSeqIteratorBegin() const
{
  FUNCSTART("const vector<char>::iterator Read::getComplementSeqIteratorBegin()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //const_cast<Read *>(this)->refreshPaddedComplementSequence();
  const_cast<Read *>(this)->refreshPaddedComplementSequence();
  
  FUNCEND();

  return REA_padded_complementsequence.begin();
}

vector<char>::const_iterator Read::getComplementSeqIteratorEnd() const
{
  FUNCSTART("const vector<char>::iterator Read::getComplementSeqIteratorEnd()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //const_cast<Read *>(this)->refreshPaddedComplementSequence();
  const_cast<Read *>(this)->refreshPaddedComplementSequence();
  
  FUNCEND();

  return REA_padded_complementsequence.end();
}



/*************************************************************************
 *
 * Returns an iterator to the sequencing vector and quality clipped 
 *  complement read
 *
 *tested
 *************************************************************************/

vector<char>::const_iterator Read::getClippedComplementSeqIterator() const
{
  FUNCSTART("const vector<char>::iterator Read::getClippedComplementSeqIterator()");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  //const_cast<Read *>(this)->refreshPaddedComplementSequence();
  const_cast<Read *>(this)->refreshPaddedComplementSequence();
  
  vector<char>::const_iterator I=REA_padded_complementsequence.begin();
  
  BOUNDCHECK(static_cast<int32>(REA_padded_complementsequence.size())-getRightClipoff(), 0, static_cast<int32>(REA_padded_complementsequence.size()+1));
  advance(I, REA_padded_complementsequence.size()-getRightClipoff());

  FUNCEND();

  return I;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

char Read::getBaseInSequence(uint32 pos) const
{
  FUNCSTART("char Read::getBaseInSequence(uint32 pos)");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_padded_sequence.size(), getName() << ": pos (" << pos << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  FUNCEND();

  return REA_padded_sequence[pos];
}

char Read::getBaseInClippedSequence(uint32 pos) const
{
  FUNCSTART("char Read::getBaseInClippedSequence(uint32 pos)");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_padded_sequence.size(), getName() << ": pos (" << pos << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  FUNCEND();

  return REA_padded_sequence[pos+getLeftClipoff()];
}

char Read::getBaseInComplementSequence(uint32 pos) const
{
  FUNCSTART("char Read::getBaseInComplementSequence(int32 pos)");

  const_cast<Read *>(this)->refreshPaddedComplementSequence();

  FUNCEND();

  return REA_padded_complementsequence[pos];
}

base_quality_t Read::getQualityInSequence(uint32 pos) const
{
  FUNCSTART("char Read::getQualityInSequence(int32 pos)");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_qualities.size(), getName() << ": pos (" << pos << ") >= REA_qualities.size (" << REA_qualities.size()  << ") ?");

  FUNCEND();

  return REA_qualities[pos];
}

base_quality_t Read::getQualityInComplementSequence(uint32 pos) const
{
  FUNCSTART("char Read::getQualityInSequence(int32 pos)");

  const_cast<Read *>(this)->refreshPaddedComplementSequence();

  BUGIFTHROW(pos >= static_cast<uint32>(REA_qualities.size()), getName() << ": pos (" << pos << ") >= REA_qualities.size (" << REA_qualities.size()  << ") ?");

  FUNCEND();

  return REA_qualities[REA_qualities.size()-1-pos];
}


/*************************************************************************
 *
 * pos in unclipped sequence
 *
 *
 *************************************************************************/

Read::bposhashstat_t Read::getBPosHashStats(uint32 pos) const
{
  FUNCSTART("baseflags_t Read::getBaseFlags(uint32 pos) const");

  BUGIFTHROW(pos >= REA_bposhashstats.size(), getName() << ": pos (" << pos << ") >= REA_bposhashstats.size (" << REA_bposhashstats.size() << ") ?");

  FUNCEND();

  return REA_bposhashstats[pos];
}


/*************************************************************************
 *
 * returns the position in unclipped sequence of first base of a 
 *  consecutive run of bases (including)
 *
 *
 *
 *************************************************************************/

uint32 Read::getLowerBoundPosOfBaseRun(uint32 pos, char base, const bool alsotakegap) const
{
  FUNCSTART("uint32 Read::getLowerBoundPosOfBaseRun(uint32 pos, char base, const bool alsotakegap) const");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_padded_sequence.size(), getName() << ": pos (" << pos << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  if(pos==0) return 0;
  if(!alsotakegap && REA_padded_sequence[pos]=='*') return pos;

  base=static_cast<char>(toupper(base));
  for(; pos>0; pos--){
    if(!alsotakegap && REA_padded_sequence[pos-1]=='*') return pos;
    if(REA_padded_sequence[pos-1]!='*'
       && toupper(REA_padded_sequence[pos-1]) !=base) return pos;
  }

  FUNCEND();
  return 0;
}

/*************************************************************************
 *
 * returns the position in unclipped sequence of last base of a 
 *  consecutive run of bases (including)
 *
 *
 *************************************************************************/

uint32 Read::getUpperBoundPosOfBaseRun(uint32 pos, char base, const bool alsotakegap) const
{
  FUNCSTART("uint32 Read::getUpperBoundPosOfBaseRun(uint32 pos, char base, const bool alsotakegap) const");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_padded_sequence.size(), getName() << ": pos (" << pos << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");

  if(pos==REA_padded_sequence.size()-1) return pos;
  if(!alsotakegap && REA_padded_sequence[pos]=='*') return pos;

  base=static_cast<char>(toupper(base));
  for(; pos<REA_padded_sequence.size()-1; pos++){
    if(!alsotakegap && REA_padded_sequence[pos+1]=='*') return pos;
    if(REA_padded_sequence[pos+1]!='*'
       && toupper(REA_padded_sequence[pos+1])!=base) return pos;
  }

  FUNCEND();
  return static_cast<uint32>(REA_padded_sequence.size())-1;
}



/*************************************************************************
 *
 *
 *
 *************************************************************************/

uint32 Read::getLenOfGapRun(uint32 pos) const
{
  FUNCSTART("uint32 Read::getLenOfGapRun(uint32 pos) const");

  const_cast<Read *>(this)->refreshPaddedSequence();

  BUGIFTHROW(pos >= REA_padded_sequence.size(), getName() << ": pos (" << pos << ") >= REA_padded_sequence.size (" << REA_padded_sequence.size() << ") ?");
  
  if(REA_padded_sequence[pos]!='*') {
    FUNCEND();
    return 0;
  }

  while(pos>0 && REA_padded_sequence[pos-1]=='*') pos--;
  uint32 spos=pos;
  while(pos<REA_padded_sequence.size()-1 && REA_padded_sequence[pos+1]=='*') pos++;

  FUNCEND();
  return pos-spos+1;
}

/*************************************************************************
 *
 * set insize (mate pair size) for this read via static insize lib
 * 
 * looks through existing lib whether already present
 *  if not, adds this size to lib
 *
 *
 *************************************************************************/

void Read::setInsize(int32 from, int32 to)
{
  FUNCSTART("void Read::setInsize(int32 from, int32 to)");
  BUGIFTHROW(from<-1 || to < -1, getName() << ": negative values < -1? " << from << ' ' << to << '?');
  BUGIFTHROW(to<from, getName() << ": to<from? Tried to set insert sizes where the minimum library size (" << from << ") is larger than the maximum library size (" << to << ')');

  for(uint32 libid=0; libid<REA_insize_staticlib.size(); libid++){
    if(REA_insize_staticlib[libid].insize_from == from
       && REA_insize_staticlib[libid].insize_to == to){
      REA_insize_libid=libid;
      FUNCEND();
      return;
    }
  }

  BUGIFTHROW(REA_insize_staticlib.size()>=255, getName() << ": Uh ... more than 255 libraries with different mate pair sizes? Very unusual, contact author!");

  REA_insize_libid=REA_insize_staticlib.size();
  REA_insize_staticlib.resize(REA_insize_staticlib.size()+1);
  REA_insize_staticlib.back().insize_from=from;
  REA_insize_staticlib.back().insize_to=to;

  FUNCEND();
  return;
}


/*************************************************************************
 *
 * Set the read naming scheme 
 * for the time being uses in getInternalTemplateName
 *
 *************************************************************************/

void Read::setReadNamingScheme(uint8 scheme)
{
  FUNCSTART("void Read::setReadNamingScheme(uint8 scheme)");
  BUGIFTHROW(scheme >= SCHEME_UNKNOWN, getName() << ": Tried to set unknown read naming scheme (" << static_cast<uint16>(scheme) << ") ?");
  REA_readnaming_scheme=scheme;
  FUNCEND();
  return;
}


/*************************************************************************
 *
 * Set the template end
 *
 *************************************************************************/

void Read::setTemplateEnd(char e)
{
  FUNCSTART("void Read::setTemplateEnd(char e)");
  switch(e) {
  case 'F':
  case 'R':
  case 'N':{
    break;
  }
  default : {
    MIRANOTIFY(Notify::INTERNAL, getName() << ": template end/strand is not F/R/N ?");
  }
  }

  REA_template_end=e;

  FUNCEND();
  return;
}




/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from centre naming scheme
 *
 *************************************************************************/

string Read::getInternalTemplateName()
{
  FUNCSTART("string Read::getInternalTemplateName() const");
  switch(REA_readnaming_scheme) {
  case SCHEME_SANGER: {
    return getInternalTemplateName_Sanger();
  }
  case SCHEME_TIGR: {
    return getInternalTemplateName_TIGR();
  }
  case SCHEME_SOLEXA: {
    return getInternalTemplateName_Solexa();
  }
  case SCHEME_FR: {
    return getInternalTemplateName_FR();
  }
  case SCHEME_STLOUIS: {
    return getInternalTemplateName_StLouis();
  }
  case SCHEME_NONE: {
    return getName();
  }
  default: {
    throw Notify(Notify::INTERNAL, THISFUNC, "Unknown read naming scheme set???");
  }
  }

  FUNCEND();
  return "";
}

/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from sanger centre naming scheme
 *
 *************************************************************************/

string Read::getInternalTemplateName_Sanger()
{
    string templbase;
    string sangerext;

    REA_name_scheme_valid=false;

    string::size_type bpos = getName().rfind(".");

    if (bpos != string::npos) {
      templbase=getName().substr(0,bpos);
      sangerext=getName().substr(bpos,getName().size());
      if(!sangerext.empty()){
	if(toupper(sangerext[1])=='P'
	   || toupper(sangerext[1])=='F'){
	  REA_name_scheme_valid=true;
	  if(REA_template_end == 'N') REA_template_end='F';
	}else if(toupper(sangerext[1])=='Q'
		 || toupper(sangerext[1])=='R'){
	  REA_name_scheme_valid=true;
	  if(REA_template_end == 'N') REA_template_end='R';
	}
      }
    } else {
      if(!REA_template.empty()) return REA_template;
      return getName();
    }
    
    if(!REA_template.empty()) {
      templbase=REA_template;
    }
    
    if(sangerext.size()<2) {
      return (templbase+' ');
    }
    
    // test: if not sanger centre naming scheme, it's read name
    //  or template base if it exists
    if(sangerext.size()<3) {
      if(!REA_template.empty()) return REA_template;
      return getName();
    } 
    //return (templbase+sangerext[sangerext.size()-1]);

    //cout << "getName(): " << getName()<< endl;
    //cout << "templbase: " << templbase << endl;
    //cout << "sangerext: " << sangerext << endl;
    //cout << "sangerext.substr(2,sangerext.size()): " << sangerext.substr(2,sangerext.size()) << endl;
    //cout << "return; " << (templbase+sangerext.substr(2,sangerext.size())) << endl;

    return (templbase+'_'+sangerext.substr(2,sangerext.size()));
}



/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from TIGR centre naming scheme
 *
 *************************************************************************/

string Read::getInternalTemplateName_TIGR()
{
    string templbase;
    string tigrext;

    //cout << "Scheme TIGR";

    REA_name_scheme_valid=false;

    string::size_type bpos = getName().find("TF");

    if (bpos != string::npos) {
      templbase=getName().substr(0,bpos);
      REA_name_scheme_valid=true;
      if(REA_template_end == 'N') REA_template_end='F';
    } else {
      bpos = getName().find("TR");
      if (bpos != string::npos) {
	templbase=getName().substr(0,bpos);
	REA_name_scheme_valid=true;
	if(REA_template_end == 'N') REA_template_end='R';
      } else {
	if(!REA_template.empty()) return REA_template;
	return getName();
      }
    }
    tigrext=getName().substr(bpos,getName().size());
    
    if(!REA_template.empty()) {
      templbase=REA_template;
    }

    //cout << "REA_name: " << getName()<< endl;
    //cout << "templbase: " << templbase << endl;
    //cout << "tigrext: " << tigrext << endl;

    if(tigrext.size()<=2) {
      return (templbase+' ');
    }

    //cout << "tigrext.substr(2,tigrext.size()): " << tigrext.substr(2,tigrext.size()) << endl;
    //cout << "return; " << (templbase+tigrext.substr(2,tigrext.size())) << endl;

    return (templbase+'_'+tigrext.substr(2,tigrext.size()));
}



/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from FR naming scheme (.f* / .r*)
 *
 *************************************************************************/

string Read::getInternalTemplateName_FR()
{
    string templbase;
    string otherext;

    REA_name_scheme_valid=false;

    string::size_type bpos = getName().rfind(".");

    if (bpos != string::npos) {
      templbase=getName().substr(0,bpos);
      otherext=getName().substr(bpos,getName().size());

      if(REA_template_end == 'N' && !otherext.empty()){
	if(toupper(otherext[1])=='F'){
	  REA_template_end='F';
	}else if(toupper(otherext[1])=='R'){
	  REA_template_end='R';
	}else{
	  return getName();
	}
      }
    } else {
      REA_template_end='F';
      if(!REA_template.empty()) return REA_template;
      return getName();
    }
    
    if(!REA_template.empty()) {
      templbase=REA_template;
    }
    
    // if it was .f or .r  (well, .? to be precise)
    if(otherext.size()<3) {
      return templbase;
    }
    
    // else it was .f* .r*
    return (templbase+'_'+otherext.substr(2,otherext.size()));
}



/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from Solexa naming scheme ( /1 or /2)
 *
 *************************************************************************/

string Read::getInternalTemplateName_Solexa()
{
    string templbase;
    string otherext;

    REA_name_scheme_valid=false;

    string::size_type bpos = getName().rfind("/");

    if (bpos != string::npos) {
      templbase=getName().substr(0,bpos);
      otherext=getName().substr(bpos,getName().size());

      if(REA_template_end == 'N' && !otherext.empty()){
	if(toupper(otherext[1])=='1'){
	  REA_template_end='F';
	}else if(toupper(otherext[1])=='2'){
	  REA_template_end='R';
	}
      }
    } else {
      REA_template_end='F';
      if(!REA_template.empty()) return REA_template;
      return getName();
    }
    
    if(!REA_template.empty()) {
      templbase=REA_template;
    }
    
    // if it was /1 or /2  (well, .? to be precise)
    if(otherext.size()<3) {
      return templbase;
    }
    
    // else it was /1* /2*
    return (templbase+'_'+otherext.substr(2,otherext.size()));
}



/*************************************************************************
 *
 * get template name either from internal template name (if present)
 *  or deduce from stlouis naming scheme
 *
 *************************************************************************/

string Read::getInternalTemplateName_StLouis()
{
    string templbase;
    string stlext;

    REA_name_scheme_valid=false;

    string::size_type bpos = getName().rfind(".");

    if (bpos != string::npos) {
      templbase=getName().substr(0,bpos);
      stlext=getName().substr(bpos,getName().size());
      if(!stlext.empty()){
	switch(tolower(stlext[1])){
	case 'r' :
	case 'y' :
	case 'g' : {
	  if(REA_template_end == 'N') REA_template_end='R';
	  REA_name_scheme_valid=true;
	  break;
	}
	case 's' :
	case 'f' :
	case 'x' :
	case 'z' :
	case 'i' :
	case 'b' : {
	  if(REA_template_end == 'N') REA_template_end='F';
	  REA_name_scheme_valid=true;
	}
	default: {
	  // everything else treat as unknown
	  REA_template_end='N';
	  REA_name_scheme_valid=false;
	}
	}
      }
    } else {
      if(!REA_template.empty()) return REA_template;
      return getName();
    }
    
    if(!REA_template.empty()) {
      templbase=REA_template;
    }
    
    if(stlext.size()<2) {
      return (templbase+' ');
    }
    
    //cout << "getName(): " << getName()<< endl;
    //cout << "templbase: " << templbase << endl;
    //cout << "stlext: " << stlext << endl;
    //cout << "stlext.substr(2,stlext.size()): " << stlext.substr(2,stlext.size()) << endl;
    //cout << "return; " << (templbase+stlext.substr(2,stlext.size())) << endl;

    return (templbase+'_'+stlext.substr(2,stlext.size()));
}




/*************************************************************************
 *
 * looks at the tags of the read: if SVEC is present, looks if the read
 *  clips left and right have to be adjusted to cover those tags
 *
 * first: go from left(right) clip and search within 'tolerance' distance
 *  if there's some SVEC
 * if nothing found there, search within first(last) 'clipsearchlen' for
 *  SVEC tags
 *************************************************************************/

void Read::transferSVTagsToClip(int32 tolerancelen, int32 clipsearchlen) {
  FUNCSTART("void Read::transferSVTagsToClip(int32 tolerancelen)");

  if(tolerancelen<0){
    throw Notify(Notify::FATAL, THISFUNC, "tolerancelen <0");
  }
  if(clipsearchlen<0){
    throw Notify(Notify::FATAL, THISFUNC, "clipsearchlen <0");
  }
  if(REA_has_valid_data==false) return;


  refreshPaddedSequence();

  if(REA_tags.empty()) {
    FUNCEND();
    return;
  }

  //string tagSVEC="SVEC";  

  bool found=false;
  for(uint32 i=0; i<REA_tags.size(); i++){
    if(REA_tags[i].identifier==REA_tagentry_idSVEC) found=true;
  }
  if(found){
    vector<char> tagfield;
    tagfield.resize(REA_padded_sequence.size(),0);
    for(uint32 i=0; i<REA_tags.size(); i++){
      if(REA_tags[i].identifier==REA_tagentry_idSVEC) {
	uint32 from=REA_tags[i].from;
	uint32 to=REA_tags[i].to;
	if(to<from) swap(from,to);
	if(from>=REA_padded_sequence.size()) continue;
	if(to>=REA_padded_sequence.size()) to=static_cast<uint32>(REA_padded_sequence.size());
	for(uint32 index=from; index<to; index++) {
	  tagfield[index]=1;
	}
	
	if(getLeftClipoff()<static_cast<int32>(REA_padded_sequence.size())){
	  // first, search within left clip + tolerance
	  vector<char>::const_iterator I=tagfield.begin();
	  I+=getLeftClipoff();
	  found=false;
	  for(int32 c=0; c<tolerancelen && I!=tagfield.end(); c++, I++) {
	    if(*I==1) {
	      found=true;
	      break;
	    }
	  }
	  // if no SV found, search from begining + some length
	  if(!found) {
	    I=tagfield.begin();
	    for(int32 c=0; c<clipsearchlen && I!=tagfield.end(); c++, I++) {
	      if(*I==1) {
		found=true;
		break;
	      }
	    }
	  }
	  if(found) {
	    while(found) {
	      if(I==tagfield.end()) break;
	      if(*I==0) break;
	      I++;
	    }
	    //REA_sl=I-tagfield.begin();
	    // +1 experimentell ermittelt :)
	    REA_sl=static_cast<int32>(I-tagfield.begin())+1;
	    if(REA_sl>REA_sr) REA_sl=REA_sr;
	    updateClipoffs();
	  }
	}

	// right clip
	if(getRightClipoff()>0){
	  // first, search within right clip - tolerance
	  vector<char>::const_iterator I=tagfield.begin();
	  I+=getRightClipoff();
	  found=false;
	  for(int32 c=0; c<tolerancelen && I!=tagfield.begin(); c++) {
	    if(*(--I)==1) {
	      found=true;
	      break;
	    }
	  }
	  if(!found) {
	    I=tagfield.begin();
	    for(int32 c=0; c<clipsearchlen && I!=tagfield.begin(); c++) {
	      if(*(--I)==1) {
		found=true;
		break;
	      }
	    }
	  }
	  if(found) {
	    while(found) {
	      if(I==tagfield.begin()) break;
	      if(*I==0) {
		I++;
		break;
	      }
	      I--;
	    }
	    REA_sr=static_cast<int32>(I-tagfield.begin());
	    if(REA_sr<REA_sl) REA_sr=REA_sl;
	    updateClipoffs();
	  }
	}
      }
    }
  }

  

  FUNCEND();
}




/*************************************************************************
 *
 * Sets the (quality) clipoffs to the given values. If force is true,
 *  then the sequence vector clipoffs are also set if needed.
 *
 *************************************************************************/

void Read::setClipoffs(int32 lclip, int32 rclip, bool force)
{
  FUNCSTART("void Read::setClipoffs(uint32 lclip, uint32 rclip, bool force)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(lclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "lclip < 0?"));
  BUGIF(rclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "rclip < 0?"));

  if(rclip>static_cast<int32>(REA_padded_sequence.size())){
    rclip=static_cast<int32>(REA_padded_sequence.size());
    //throw Notify(Notify::INTERNAL, THISFUNC, "rclip > REA_padded_sequence.size().");
  }

  if(lclip>static_cast<int32>(REA_padded_sequence.size())){
    lclip=static_cast<int32>(REA_padded_sequence.size());
  }

  REA_ql=lclip;
  REA_qr=rclip;
  if(force==true){
    if(REA_sl>lclip) REA_sl=lclip;
    if(REA_sr<rclip) REA_sr=rclip;
  }
//  REA_char_seq_dirty=true;
//  REA_char_cseq_dirty=true;

  updateClipoffs();

  FUNCEND();
}



/*************************************************************************
 *
 * Sets the lq quality clipoff to the given values
 *
 *************************************************************************/

void Read::setLQClipoff(int32 lclip)
{
  FUNCSTART("void Read::setLQClipoffs(int32 lclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(lclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "lclip < 0?"));

  if(lclip>static_cast<int32>(REA_padded_sequence.size())){
    lclip=static_cast<int32>(REA_padded_sequence.size());
  }

  REA_ql=lclip;
  updateClipoffs();

  FUNCEND();
}

/*************************************************************************
 *
 * Sets the rq quality clipoff to the given values
 *
 *************************************************************************/

void Read::setRQClipoff(int32 rclip)
{
  FUNCSTART("void Read::setRQClipoffs(int32 rclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(rclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "rclip < 0?"));

  if(rclip>static_cast<int32>(REA_padded_sequence.size())){
    rclip=static_cast<int32>(REA_padded_sequence.size());
  }

  REA_qr=rclip;
  updateClipoffs();

  FUNCEND();
}


/*************************************************************************
 *
 * Sets the l seq clipoff to the given values
 *
 *************************************************************************/

void Read::setLSClipoff(int32 lclip)
{
  FUNCSTART("void Read::setLSClipoffs(int32 lclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(lclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "lclip < 0?"));

  if(lclip>static_cast<int32>(REA_padded_sequence.size())){
    lclip=static_cast<int32>(REA_padded_sequence.size());
  }

  REA_sl=lclip;
  updateClipoffs();

  FUNCEND();
}

/*************************************************************************
 *
 * Sets the r seq clipoff to the given values
 *
 *************************************************************************/

void Read::setRSClipoff(int32 rclip)
{
  FUNCSTART("void Read::setRSClipoffs(int32 rclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(rclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "rclip < 0?"));

  if(rclip>static_cast<int32>(REA_padded_sequence.size())){
    rclip=static_cast<int32>(REA_padded_sequence.size());
  }

  REA_sr=rclip;
  updateClipoffs();

  FUNCEND();
}



/*************************************************************************
 *
 * Sets the masked (and quality) clipoffs to the given values. 
 *
 *************************************************************************/

void Read::setLMClipoff(int32 lclip) 
{
  FUNCSTART("int32 Read::setLMClipoff(int32 lclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(lclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "lclip < 0?"));

  if(lclip>static_cast<int32>(REA_padded_sequence.size())){
    lclip=static_cast<int32>(REA_padded_sequence.size());
    //throw Notify(Notify::INTERNAL, THISFUNC, "rclip > REA_padded_sequence.size().");
  }
  REA_ml=lclip;
  REA_ql=lclip;
  updateClipoffs();
}

void Read::setRMClipoff(int32 rclip) 
{
  FUNCSTART("int32 Read::setRMClipoff(int32 rclip)");

  BUGIF(checkRead()!=NULL, throw Notify(Notify::INTERNAL, THISFUNC, checkRead()));
  BUGIF(rclip<0, throw Notify(Notify::INTERNAL, THISFUNC, "rclip < 0?"));

  if(rclip>static_cast<int32>(REA_padded_sequence.size())){
    rclip=static_cast<int32>(REA_padded_sequence.size());
    //throw Notify(Notify::INTERNAL, THISFUNC, "rclip > REA_padded_sequence.size().");
  }
  REA_mr=rclip;
  REA_qr=rclip;
  updateClipoffs();
}


/*************************************************************************
 *
 * Returns the number of bases the read could be extended to the left
 *  before the sequence vector or masked bases clipped part begins. 0 if none.
 *
 *************************************************************************/

int32 Read::getLeftExtend() const
{
  FUNCSTART("int32 Read::getLeftExtend() const");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();

  int32 hardclip=max(REA_sl,REA_ml);
  if(REA_ql > hardclip) return REA_ql-hardclip;
  return 0;
}

/*************************************************************************
 *
 * Returns the number of bases the read could be extended to the right
 *  before the sequence vector or masked bases clipped part begins. 0 if none.
 *
 *************************************************************************/

int32 Read::getRightExtend() const
{
  FUNCSTART("int32 Read::getRightExtend() const");
  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  FUNCEND();

  int32 hardclip=min(REA_sr,REA_mr);
  if(REA_qr < hardclip) return hardclip-REA_qr;
  return 0;
}



// Routines for phred style quality<->errProbability conversion.

/*************************************************************************
 *
 *
 *
 *************************************************************************/

double Read::qualityToErrorRate_compute(base_quality_t qual) 
{
  return pow(10.0,static_cast<double>(qual)/10.0);
}


// Qual is caped at 100
base_quality_t Read::errorRateToQuality(double errProb) 
{
  //FUNCSTART("static const double Read::qualityToErrorProb(base_quality_t qual)");
  if(errProb>1000000000.0) errProb=1000000000.0;
  base_quality_t tmp=static_cast<base_quality_t>(log10(errProb)*10.0+.5);
  return tmp;

  //FUNCEND();
}


// Alert! These routines are not prepared for phred-style qualities!
//  (but work quite well anyway)

/*************************************************************************
 *
 *
 *
 *************************************************************************/

base_quality_t Read::queryAverageQualInSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)
{
  FUNCSTART("base_quality_t Read::queryAverageQualInSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)");

  CEBUG("posl: " << posl << endl);
  CEBUG("posr: " << posr << endl);
  CEBUG("skipNs: " << skipNs << endl);
  CEBUG("skipStars: " << skipStars << endl);

  uint32 avgqual=0;
  uint32 countavg=0;
  base_quality_t retval=0;

  if(REA_ps_dirty==true && REA_pcs_dirty==false){
    retval=queryAverageQualInComplementSequence(
      static_cast<int32>(REA_padded_complementsequence.size())-posr-1,
      static_cast<int32>(REA_padded_complementsequence.size())-posl-1,
      skipNs,
      skipStars);
  }else{
    refreshPaddedSequence();

    if(posl<0) posl=0;
    if(posr >= static_cast<int32>(REA_padded_sequence.size())) posr=static_cast<int32>(REA_padded_sequence.size())-1;

    vector<base_quality_t>::const_iterator qI;
    vector<char>::const_iterator sI;
    
    qI=REA_qualities.begin();
    BOUNDCHECK(posl, 0, static_cast<int32>(REA_qualities.size()));
    advance(qI, posl);
    sI=REA_padded_sequence.begin();
    BOUNDCHECK(posl, 0, static_cast<int32>(REA_padded_sequence.size()));
    advance(sI, posl);
    
    while((((*sI=='N'
	       || *sI=='n'
	       || *sI=='X'
	       || *sI=='x'
	       || *sI=='-') && skipNs)
	     || (*sI=='*' && skipStars))
	    && sI!=REA_padded_sequence.begin()){
	sI--; qI--;
    }
    
    vector<char>::const_iterator tsI;
    tsI=REA_padded_sequence.begin();
    BOUNDCHECK(posr, 0, static_cast<int32>(REA_padded_sequence.size()));
    advance(tsI, posr);
    
    while((((*tsI=='N'
	       || *tsI=='n'
	       || *tsI=='X'
	       || *tsI=='x'
	       || *tsI=='-') && skipNs)
	     || (*tsI=='*' && skipStars))
	    && (tsI+1)!=REA_padded_sequence.end()){
	tsI++;
    }
    
    for(;sI<=tsI; sI++, qI++){
      CEBUG("char: " << *sI<< endl);
      if(dptools::isValidIUPACBase(*sI)){
	if(*qI!=BQ_UNDEFINED){
	  avgqual+=static_cast<uint32>(*qI);
	  countavg++;
	}
      }else{
	switch(toupper(*sI)){
	case '-':
	case 'N':
	case 'X':{
	  if(!skipNs && *qI!=BQ_UNDEFINED){
	    avgqual+=static_cast<uint32>(*qI);
	    countavg++;
	  }
	  break;
	}
	case '*':{
	  if(!skipStars && *qI!=BQ_UNDEFINED){
	    avgqual+=static_cast<uint32>(*qI);
	    countavg++;
	  }
	  break;
	}
	default:{
	  cout << "Illegal base: " << *sI << "(" << hex << static_cast<uint16>(*sI) << dec << ")" << endl;
	  MIRANOTIFY(Notify::FATAL, "Illegal base found: " << getName());
	}
	}
      }
    }
  }

  if(countavg>0) retval=static_cast<base_quality_t>(avgqual/countavg);

  CEBUG("Counted " << countavg << " non-skipped bases.\n");
  CEBUG("Returning: " << (uint16) retval << endl);

  FUNCEND();

  return retval;
}



/*************************************************************************
 *
 *
 *
 *************************************************************************/

base_quality_t Read::queryAverageQualInClippedSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)
{
  FUNCSTART("base_quality_t Read::queryAverageQualInClippedSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)");

  CEBUG("posl: " << posl << endl);
  CEBUG("posr: " << posr << endl);
  CEBUG("skipNs: " << skipNs << endl);
  CEBUG("skipStars: " << skipStars << endl);

  posl+=getLeftClipoff();
  posr+=getLeftClipoff();
  if(posl<getLeftClipoff()) posl=getLeftClipoff();
  if(posr>=getRightClipoff()) posr=getRightClipoff();

  FUNCEND();

  return queryAverageQualInSequence(posl, posr, skipNs, skipStars);
}




/*************************************************************************
 *
 *
 *
 *************************************************************************/

base_quality_t Read::queryAverageQualInComplementSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)
{
  FUNCSTART("base_quality_t Read::queryAverageQualInComplementSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)");

  CEBUG("posl: " << posl << endl);
  CEBUG("posr: " << posr << endl);
  CEBUG("skipNs: " << skipNs << endl);
  CEBUG("skipStars: " << skipStars << endl);

  uint32 avgqual=0;
  uint32 countavg=0;
  base_quality_t retval=0;

  if(REA_pcs_dirty==true && REA_ps_dirty==false){
    retval=queryAverageQualInSequence(
      static_cast<int32>(REA_padded_complementsequence.size())-posr-1,
      static_cast<int32>(REA_padded_complementsequence.size())-posl-1,
      skipNs,
      skipStars);
  }else{
    refreshPaddedComplementSequence();

    if(posl<0) posl=0;
    if(posr >= static_cast<int32>(REA_padded_complementsequence.size())) posr=static_cast<int32>(REA_padded_complementsequence.size())-1;
    
    uint32 complement_posl=static_cast<uint32>(REA_padded_complementsequence.size())-posl-1;

    CEBUG("complement_posl: " << complement_posl << endl);

    vector<base_quality_t>::const_iterator qI;
    vector<char>::const_iterator sI;
    
    qI=REA_qualities.begin();
    //BOUNDCHECK(complement_posl, 0, REA_qualities.size());
    BUGIFTHROW(complement_posl>=REA_qualities.size(),getName() << ": complement_posl (" << complement_posl << ") >= REA_qualities.size(" << REA_qualities.size() << ") ?");
    advance(qI, complement_posl);
    sI=REA_padded_complementsequence.begin();
    BOUNDCHECK(posl, 0, static_cast<int32>(REA_padded_complementsequence.size()));
    advance(sI, posl);
    
    while((((*sI=='N'
	       || *sI=='n'
	       || *sI=='X'
	       || *sI=='x'
	       || *sI=='-') && skipNs)
	     || (*sI=='*' && skipStars))
	    && sI!=REA_padded_complementsequence.begin()){
	sI--; qI++;
    }
    
    vector<char>::const_iterator tsI;
    tsI=REA_padded_complementsequence.begin();
    BOUNDCHECK(posr, 0, static_cast<int32>(REA_padded_complementsequence.size()));
    advance(tsI, posr);
    
    while((((*tsI=='N'
	       || *tsI=='n'
	       || *tsI=='X'
	       || *tsI=='x'
	       || *tsI=='-') && skipNs)
	     || (*tsI=='*' && skipStars))
	    && (tsI+1)!=REA_padded_complementsequence.end()){
	tsI++;
    }
    
    for(;sI<=tsI; sI++, qI--){
      CEBUG("char: " << *sI<< endl);
      if(dptools::isValidIUPACBase(*sI)){
	if(*qI!=BQ_UNDEFINED){
	  avgqual+=static_cast<int32>(*qI);
	  countavg++;
	}
      }else{
	switch(toupper(*sI)){
	case '-':
	case 'N':
	case 'X':{
	  if(!skipNs && *qI!=BQ_UNDEFINED){
	    avgqual+=static_cast<int32>(*qI);
	    countavg++;
	  }
	  break;
	}
	case '*':{
	  if(!skipStars && *qI!=BQ_UNDEFINED){
	    avgqual+=static_cast<int32>(*qI);
	    countavg++;
	  }
	  break;
	}
	default:{
	  cout << "Illegal base: " << *sI << "(" << hex << static_cast<uint16>(*sI) << dec << ")" << endl;
	  MIRANOTIFY(Notify::FATAL, "Illegal base found: " << getName());
	}
	}
      }
    }
  }

  if(countavg>0) retval=static_cast<base_quality_t>(avgqual/countavg);

  CEBUG("Counted " << countavg << " non-skipped bases.\n");
  CEBUG("Returning: " << (uint16) retval << endl);

  FUNCEND();

  return retval;
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

base_quality_t Read::queryAverageQualInClippedComplementSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)
{
  FUNCSTART("base_quality_t Read::queryAverageQualInClippedComplementSequence(int32 posl, int32 posr, bool skipNs, bool skipStars)");

  CEBUG("posl: " << posl << endl);
  CEBUG("posr: " << posr << endl);
  CEBUG("skipNs: " << skipNs << endl);
  CEBUG("skipStars: " << skipStars << endl);

  FUNCEND();

  return queryAverageQualInComplementSequence(getLenSeq()-getRightClipoff()+posl, 
						getLenSeq()-getRightClipoff()+posr,
						skipNs,
						skipStars);

}



/*************************************************************************
 *
 * Clip off bad qual at start and end of sequence
 *
 *
 * forward:
 *  -------F2-----------------------F1-----------    winlen, minqual (avg)
 *         >>                       >>
 *
 *  then
 *  -------F2-----RF1-----RF2-------F1-----------    winlen/2, minqual-5 (avg)
 *                <<<     <<<
 *  then
 *  -------F2-----RF1-X---RF2-------F1-----------
 *                >>>     
 *
 * backward:
 *  -------F1-----------------------F2-----------
 *         <<                       <<
 *
 *  then
 *  -------F1-----RF2-----RF1-------F2-----------
 *                >>>     >>>
 *  then
 *  -------F1-----RF2--X--RF1-------F2-----------
 *                        <<<
 *
 *************************************************************************/

void Read::performQualityClip(uint32 minqual, uint32 winlen)
{
  FUNCSTART("void Read::performQualityClip(uint32 avgqual, uint32 winlen)");

  if(!REA_has_valid_data) {
    FUNCEND();
    return;
  }


  if(!REA_has_quality){
    FUNCEND();
    return;
  }
  
  if(minqual<5) minqual=5;
  if(winlen<10) winlen=10;

  double minquald=minqual;
  double winlend=winlen;

  uint32 winlen_2=winlen/2;
  uint32 minqualm5=minqual-5;
  double minqualm5d=minqualm5;
  double winlen_2d=winlen_2;

  bool exitprematurely=false;

  if(REA_qualities.size() > winlen) {
    uint32 qualtotal=0;
    vector<base_quality_t>::const_iterator F1=REA_qualities.begin();
    for(uint32 i=0; i<winlen; i++, F1++) {
      qualtotal+=*F1;
    }
    vector<base_quality_t>::const_iterator F2=REA_qualities.begin();
    double avgquald=static_cast<double>(qualtotal)/winlend;
    while(F1 != REA_qualities.end() && avgquald < minquald){
      qualtotal+=*F1;
      qualtotal-=*F2;
      avgquald=static_cast<double>(qualtotal)/winlend;
      F1++;
      F2++;
    }
    if(avgquald < minquald) {
      // Sheesh, we went through that Read without finding good quality, 
      //  set the REA_ql and REA_qr accordingly 
      REA_ql=static_cast<int32>(REA_qualities.size());
      REA_qr=0;
      exitprematurely=true;
    } else {
      // some good qual start found ... iterate a bit backward to find
      //  an acceptable start
      // halve winlen and take minqual -5
      if(F1==REA_qualities.end()) F1--;
      vector<base_quality_t>::const_iterator RF1=F1;
      qualtotal=0;
      for(uint32 i=0; i<winlen_2; i++, RF1--) {
	qualtotal+=*RF1;
      }
      vector<base_quality_t>::const_iterator RF2=F1;
      avgquald=static_cast<double>(qualtotal)/winlen_2d;
      while(RF1 >= F2 && avgquald < minqualm5d){
	qualtotal+=*RF1;
	qualtotal-=*RF2;
	avgquald=static_cast<double>(qualtotal)/winlend;
	RF1--;
	RF2--;
      }
      // ok, average qual on halved window is now below minqual-5
      // reiterate forward until first base with qual >= minqual-5
      while(RF1!=F1 && *RF1<minqualm5) RF1++;

      // RF1 now points to the first base of good region
      REA_ql=static_cast<int32>(RF1-REA_qualities.begin());
    }

    // ok, we have start of good sequence at the front, now we'll look at the end
    if(!exitprematurely){
      qualtotal=0;
      F1=REA_qualities.end()-1;
      for(uint32 i=0; i<winlen; i++, F1--) {
	qualtotal+=*F1;
      }
      F2=REA_qualities.end()-1;
      avgquald=static_cast<double>(qualtotal)/winlend;
      while(F1 >= REA_qualities.begin() && avgquald < minquald){
	qualtotal+=*F1;
	qualtotal-=*F2;
	avgquald=static_cast<double>(qualtotal)/winlend;
	F1--;
	F2--;
      }
      if(avgquald < minquald) {
	// Sheesh, we went through that Read without finding good quality, 
	//  this ... eh ... cannot be! (remember, we found forward!)
	updateClipoffs();
	refreshPaddedSequence();
	cout << *this;
	throw Notify(Notify::INTERNAL, THISFUNC, "Quality clipping error: no backward cutoff, but forward found?");
      } else {
	if(F1<REA_qualities.begin()) F1=REA_qualities.begin();
	vector<base_quality_t>::const_iterator RF1=F1;
	qualtotal=0;
	for(uint32 i=0; i<winlen_2; i++, RF1++) {
	  qualtotal+=*RF1;
	}
	vector<base_quality_t>::const_iterator RF2=F1;
	avgquald=static_cast<double>(qualtotal)/winlen_2d;
	while(RF1 <= F2 && avgquald < minqualm5d){
	  qualtotal+=*RF1;
	  qualtotal-=*RF2;
	  avgquald=static_cast<double>(qualtotal)/winlend;
	  RF1++;
	  RF2++;
	}
	if(RF1==REA_qualities.end()) RF1--;
	while(RF1!=F1 && *RF1<minqualm5) RF1--;

	// RF1 now points to the first base of good region
	REA_qr=static_cast<int32>(RF1-REA_qualities.begin());
      }
    }
  }

  // it might happen that no good qualities were found,
  //  in this case, REA_ql=sizeofseq and REA_qr=0
  // we still want a read that won't throw errors if we ask it
  //  to give us the sequence and other things
  // so: set ql=qr=0
  if(REA_ql>REA_qr) REA_ql=REA_qr;

  updateClipoffs();

  FUNCEND();
  return;
}




/*************************************************************************
 *
 * physically trims the read to the left and right clips
 * tags completely outside new bounds are erased, others get positions adjusted
 *
 *************************************************************************/

void Read::performHardTrim()
{
  FUNCSTART("void Read::performHardTrim()");

  refreshPaddedSequence();

  //setCoutType(AS_TEXT);
  //cout << *this; cout.flush();

  // see whether to completely delete
  if(getLenClippedSeq()==0){
    discard();
  }else{
    // see whether to trim on right
    if(static_cast<int32>(getLenSeq()) - getRightClipoff() >0){
      BUGIFTHROW(REA_padded_sequence.size()<getRightClipoff()-1,"1 REA_padded_sequence.size()<getRightClipoff()-1 ? " << REA_padded_sequence.size() << " " << getRightClipoff());
      REA_padded_sequence.resize(getRightClipoff()-1);
      REA_padded_complementsequence.clear();
      REA_pcs_dirty=true;
      if(!REA_qualities.empty()){
	BUGIFTHROW(REA_qualities.size()<getRightClipoff()-1,"REA_qualities.size()<getRightClipoff()-1 ?");
	REA_qualities.resize(getRightClipoff()-1);      
      }
      if(!REA_adjustments.empty()){
	BUGIFTHROW(REA_adjustments.size()<getRightClipoff()-1,"REA_adjustments.size()<getRightClipoff()-1 ?");
	REA_adjustments.resize(getRightClipoff()-1);      
      }
      if(!REA_bposhashstats.empty()){
	BUGIFTHROW(REA_bposhashstats.size()<getRightClipoff()-1,"REA_bposhashstats.size()<getRightClipoff()-1 ?");
	REA_bposhashstats.resize(getRightClipoff()-1);      
      }
      
      REA_qr=REA_padded_sequence.size();
      REA_sr=REA_padded_sequence.size(); 
      REA_cr=REA_padded_sequence.size();
      REA_mr=REA_padded_sequence.size();
      
      vector<multitag_t>::iterator tI=REA_tags.begin();
      for(uint32 tagpos=0; tI != REA_tags.end(); ++tagpos){
	if(tI->from >= REA_padded_sequence.size()) {
	  REA_tags.erase(tI);
	  tI=REA_tags.begin();
	  advance(tI,tagpos);
	  --tagpos;
	} else {
	  if(tI->to > REA_padded_sequence.size()) tI->to=REA_padded_sequence.size();
	  ++tI;
	}
      }
    }
    
    
    // see whether to trim on left
    if(getLeftClipoff()>0){
      BUGIFTHROW(REA_padded_sequence.size()<getRightClipoff()-1,"2 REA_padded_sequence.size()<getRightClipoff()-1 ?");
      {
	vector<char>::iterator bI=REA_padded_sequence.begin();
	vector<char>::iterator eI=bI;
	advance(eI,getLeftClipoff());
	REA_padded_sequence.erase(bI,eI);
      }
      REA_padded_complementsequence.clear();
      REA_pcs_dirty=true;
      if(!REA_qualities.empty()){
	BUGIFTHROW(REA_qualities.size()<getRightClipoff()-1,"REA_qualities.size()<getRightClipoff()-1 ?");
	vector<base_quality_t>::iterator bI=REA_qualities.begin();
	vector<base_quality_t>::iterator eI=bI;
	advance(eI,getLeftClipoff());
	REA_qualities.erase(bI,eI);
      }
      if(!REA_adjustments.empty()){
	BUGIFTHROW(REA_adjustments.size()<getRightClipoff()-1,"REA_adjustments.size()<getRightClipoff()-1 ?");
	vector<int32>::iterator bI=REA_adjustments.begin();
	vector<int32>::iterator eI=bI;
	advance(eI,getLeftClipoff());
	REA_adjustments.erase(bI,eI);
	
	vector<int32>::iterator aI=REA_adjustments.begin();
	for(; aI != REA_adjustments.end(); ++aI){
	  if(*aI >=0 ) (*aI)-=getLeftClipoff();
	}
      }
      if(!REA_bposhashstats.empty()){
	BUGIFTHROW(REA_bposhashstats.size()<getRightClipoff()-1,"REA_bposhashstats.size()<getRightClipoff()-1 ?");
	vector<bposhashstat_t>::iterator bI=REA_bposhashstats.begin();
	vector<bposhashstat_t>::iterator eI=bI;
	advance(eI,getLeftClipoff());
	REA_bposhashstats.erase(bI,eI);
      }
      
      vector<multitag_t>::iterator tI=REA_tags.begin();
      for(uint32 tagpos=0; tI != REA_tags.end(); ++tagpos){
	if(tI->to < getLeftClipoff()) {
	  REA_tags.erase(tI);
	  tI=REA_tags.begin();
	  advance(tI,tagpos);
	  --tagpos;
	} else {
	  tI->from-=getLeftClipoff();
	  tI->to-=getLeftClipoff();
	  ++tI;
	}
      }
      
      REA_qr-=getLeftClipoff(); 
      REA_sr-=getLeftClipoff(); 
      REA_cr-=getLeftClipoff();
      REA_mr-=getLeftClipoff();
      
      REA_ql=0; 
      REA_sl=0; 
      REA_cl=0;
      REA_ml=0;
    }

  }

  FUNCEND();
}



/*************************************************************************
 *
 * deletes the weakest base of a base run 
 * can also insert a gap with quality 0 at the *original* position
 *  given in the run (helps to delete a weak base also in aligned reads
 *  without breaking the alignment)
 *
 *************************************************************************/

void Read::deleteWeakestBaseInRun(const char base, const uint32 position, const bool insertgap)
{
  FUNCSTART("");

  uint32 p1=position;
  bool found=getPosOfWeakestBaseInRun(base,p1);
  if(found){
    deleteBaseFromSequence(p1);
    if(insertgap) {
      insertBaseInSequence('*',0,position,true);
    }
  }

  FUNCEND();
}


/*************************************************************************
 *
 * Why give a char base to the function if it simply could read out
 * the base from its sequence?
 *
 * Easy: imagine ..AAA*TTT... and we gave the position of the gap character
 *  as only parameter? (this can happen during assembly in later stages)
 *
 *
 *************************************************************************/

bool Read::getPosOfWeakestBaseInRun(char base, uint32 & position)
{
  FUNCSTART("");

  if(!REA_has_valid_data) {
    FUNCEND();
    return false;
  }

  // force a check of quality status
  //paranoiaBUGIF(setQualityFlag(),);

  BUGIFTHROW(position>=getLenSeq(), getName() << ": position (" << position << ") >= size of read (" << getLenSeq() << ") ?");

  base=static_cast<char>(toupper(base));

  refreshPaddedSequence();

  base_quality_t minqual=101;
  uint32 minqualpos=position;
  bool foundmin=false;

  // check stretch of char base to left 
  // if several bases have the same loq qual, takes farthest right
  {
    int32 rrpi=position-1;
    
    for(; rrpi>=0; rrpi--){
      if(REA_padded_sequence[rrpi] != '*') {
	if(toupper(REA_padded_sequence[rrpi]) == base) {
	  if(REA_qualities[rrpi] < minqual) {
	    minqual=REA_qualities[rrpi];
	    minqualpos=rrpi;
	    foundmin=true;
	  }
	} else {
	  break;
	}
      }
    }
  }
  
  // check stretch of char base to right
  // if several bases have the same loq qual, takes farthest right
  {
    int32 rrpi=position;
    
    for(; rrpi < static_cast<int32>(REA_padded_sequence.size()); rrpi++){
      if(REA_padded_sequence[rrpi] != '*'){
	if(toupper(REA_padded_sequence[rrpi]) == base) {
	  if(REA_qualities[rrpi] <= minqual) {
	    minqual=REA_qualities[rrpi];
	    minqualpos=rrpi;
	    foundmin=true;
	  }
	}else{
	  break;
	}
      }
    }
  }

  if(foundmin){
    position=minqualpos;
    FUNCEND();
    return true;
  }

  FUNCEND();
  return false;
}




/*************************************************************************
 *
 * transfers read type and strain names from MINF tags into REA_... variables
 * deletes the tags after processing
 *
 *************************************************************************/

void Read::transferMINFTagsToReadInfo()
{
  FUNCSTART("void Read::transferMINFTagsToReadInfo()");

  vector<multitag_t>::iterator tI=REA_tags.begin();
  while(tI!=REA_tags.end()){
    if(tI->identifier==REA_tagentry_idMINF){
      // extract values from tag comment
      // they're not GenBank comments, but stored the same way
      //  let's misuse the function then :-)
      string value;
      tI->extractGenBankKeyValueFromComment("ST",value);
      if(!value.empty()){
	REA_seqtype=stringToSeqType(value);
	if(REA_seqtype==getNumSequencingTypes()){
	  MIRANOTIFY(Notify::FATAL, "Error in " << getName() << " with unknown sequencing type " << value << " in ST part of MINF tag.\n");
	}
      }else{
	//REMOVEME: in version 3
	// keep compatibility to MIRA versions that used RT=...
	tI->extractGenBankKeyValueFromComment("RT",value);
	if(!value.empty()){
	  cerr << "Warning in " << getName() << ": using 'RT=' in MINF tag is deprecated, use 'ST='\n";
	  cout << "Warning in " << getName() << ": using 'RT=' in MINF tag is deprecated, use 'ST='\n";
	  REA_seqtype=stringToSeqType(value);
	  if(REA_seqtype==getNumSequencingTypes()){
	    if(value=="454GS20"){
	      REA_seqtype=SEQTYPE_454GS20;
	    }else{
	      MIRANOTIFY(Notify::FATAL, "Error in " << getName() << " with unknown sequencing type " << value << " in deprecated RT part of MINF tag.\n");
	    }
	  }
	}
      }

      tI->extractGenBankKeyValueFromComment("SN",value);
      REA_strain=REA_sc_strain.addEntry(value);
      //cout << "tI: Read " << getName() << "\tSN: " << REA_strain << '\n';

      tI->extractGenBankKeyValueFromComment("MT",value);
      REA_machine_type=REA_sc_machine_type.addEntry(value);

      tI->extractGenBankKeyValueFromComment("BB",value);
      if(!value.empty() && value!="0"){
	setBackbone(true);
      }

      tI->extractGenBankKeyValueFromComment("CER",value);
      //cout << tI-> comment << "\nmsrextract: " << value << endl;
      if(!value.empty() && value!="0"){
	setCoverageEquivalentRead(true);
      }

      // Ok, extracted everything. delete this tag
      tI=REA_tags.erase(tI);
    }else{
      tI++;
    }
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 * Returns code for sequencing type given in value string
 * or SEQTYPE_END if string was not recognised
 *
 *************************************************************************/

uint8 Read::stringToSeqType(const string & value)
{
  if(!value.empty()){
    for(uint8 i=0; i<REA_namesofseqtypes.size(); i++){
      if(value==REA_namesofseqtypes[i]) return i;
    }

    // *sigh* retain compatibility with mira < 3rc3
    if(value=="454GS") return SEQTYPE_454GS20;
  }

  return getNumSequencingTypes();
}


/*************************************************************************
 *
 *
 *************************************************************************/

void Read::setSequencingType(uint8 st)
{
  FUNCSTART("void Read::setSequencingType(uint8 st)");
  BUGIFTHROW(st>=getNumSequencingTypes(),"Trying to set illegal sequencing type " << st << " ?");
  REA_seqtype=st;
  FUNCEND();
}


///*************************************************************************
// *
// *
// *
// *
// *************************************************************************/
//
//uint32 Read::deleteTag(const string & identifier)
//{
//  FUNCSTART("void Read::deleteTag(const string & identifier)");
//
//  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));
//
//  uint32 deleted=0;
//  vector<multitag_t>::iterator tI=REA_tags.begin();
//  while(tI!=REA_tags.end()){
//    if(identifier==tI->getIdentifierStr()){
//      tI=REA_tags.erase(tI);
//      deleted++;
//    }else{
//      tI++;
//    }
//  }
//
//  FUNCEND();
//  return deleted;
//}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

uint32 Read::deleteTag(const multitag_t::mte_id_t identifier)
{
  FUNCSTART("uint32 Read::deleteTag(const multitag_t::mte_id_t identifier)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  uint32 deleted=0;
  vector<multitag_t>::iterator tI=REA_tags.begin();
  while(tI!=REA_tags.end()){
    if(identifier==tI->identifier){
      tI=REA_tags.erase(tI);
      deleted++;
    }else{
      tI++;
    }
  }

  FUNCEND();
  return deleted;
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

uint32 Read::deleteTag(const uint32 from, const uint32 to, const string & identifier)
{
  FUNCSTART("void Read::deleteTag(uint32 from, uint32 to, const string & identifier)");

  paranoiaBUGIF(checkRead()!=NULL, throw Notify(Notify::FATAL, THISFUNC, checkRead()));

  uint32 deleted=0;
  vector<multitag_t>::iterator tI=REA_tags.begin();
  while(tI!=REA_tags.end()){
    if(tI->from == from 
       && tI->to != to
       && (identifier.empty() == false 
	   || identifier==tI->getIdentifierStr())){
      tI=REA_tags.erase(tI);
      deleted++;
    }else{
      tI++;
    }
  }

  FUNCEND();
  return deleted;
}



/*************************************************************************
 *
 * Exchanges Ns in a read with gaps, takling care of adjustments and
 *  qualities
 *
 * Convenience function for Contig object
 *
 *************************************************************************/

void Read::exchangeNsWithGaps()
{
  FUNCSTART("void Read::exchangeNsWithGaps()");
  if(!REA_has_valid_data) {
    FUNCEND();
    return;
  }

  // TODO: perhaps also work with reverse? but not needed now
  refreshPaddedSequence();

  vector<char>::iterator cI=REA_padded_sequence.begin();
  vector<int32>::iterator aI=REA_adjustments.begin();
  vector<base_quality_t>::iterator qI=REA_qualities.begin();
  for(; cI!=REA_padded_sequence.end(); cI++, qI++){
    if(*cI=='N'){
      REA_pcs_dirty=true;
      *cI='*';
      if(REA_uses_adjustments) *aI=-1;
      switch(REA_seqtype){
      case SEQTYPE_SANGER :
      case SEQTYPE_SOLEXA :
      case SEQTYPE_ABISOLID : {
	base_quality_t newqual=0;
	base_quality_t poslooked=0;
	if(cI!=REA_padded_sequence.begin()){
	  newqual=*(qI-1);
	  poslooked++;
	}
	if((cI+1)!=REA_padded_sequence.end()){
	  // gcc 4.3.2 warns as it think the values could grow
	  //  higher than 255 ... they can't, base quals go to max
	  //  of 100.
	  //newqual+=*(qI+1);
	  // writing instead the version with cast to get gcc quiet
	  newqual=static_cast<base_quality_t>(newqual+(*(qI+1)));
	  poslooked++;
	}
	if(poslooked) {
	  *qI=static_cast<base_quality_t>(newqual/poslooked);
	}else{
	  // should not happen
	  *qI=0;
	}
	break;
      }
      case SEQTYPE_454GS20 : {
	*qI=1;
      }
      default: {
	*qI=0;
      }
      }
    }
    // manually increasing aI. not in for loop as adjustments may be empty
    if(REA_uses_adjustments) aI++;
  }

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *
 *************************************************************************/

const string & Read::getNameOfSequencingType(uint32 st)
{
  FUNCSTART("const string & Read::getNameOfSequencingType(uint32 st)");
  BUGIFTHROW(st>=REA_namesofseqtypes.size(),"Asking for name of unknown sequencing type " << st << " ?");
  FUNCEND();
  return REA_namesofseqtypes[st];
}

const string & Read::getShortNameOfSequencingType(uint32 st)
{
  FUNCSTART("const string & Read::getShortNameOfSequencingType(uint32 st)");
  BUGIFTHROW(st>=REA_namesofseqtypes.size(),"Asking for name of unknown sequencing type " << st << " ?");
  FUNCEND();
  return REA_shortnamesofseqtypes[st];
}




/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Read::blindSeqData(char base)
{
  FUNCSTART("void Read::blindSeqData(char base)")
  if(!REA_has_valid_data) {
    FUNCEND();
    return;
  }

  // TODO: perhaps also work with reverse? but not needed now

  if(!REA_ps_dirty){
    vector<char>::iterator cI=REA_padded_sequence.begin();
    for(; cI!=REA_padded_sequence.end(); cI++){
      if(*cI != '*' && *cI != 'n' && *cI != 'N') *cI=base;
    }
  }

  if(!REA_pcs_dirty){
    base=dptools::getComplementIUPACBase(base);
    vector<char>::iterator cI=REA_padded_complementsequence.begin();
    for(; cI!=REA_padded_complementsequence.end(); cI++){
      if(*cI != '*' && *cI != 'n' && *cI != 'N') *cI=base;
    }
  }

  FUNCEND();
}
