/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. If not, write to the 
 *Free Software Foundation, Inc., 59 
 *Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/unistd.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <libxml/uri.h>
#include <libxml/nanohttp.h>
#include <libxml/nanoftp.h>
#include "mlview-file-descriptor.h"

/**
 *@file
 *The class definition of #MlViewFileDescriptor.
 */

/**
 *The private members of #MlViewFileDescriptor class.
 */
struct _MlViewFileDescriptorPrivate {
        /**the string representation of the uri*/
        gchar *uri_str;

        /**
	 *The URI of this file descriptor.
	 *Note that the xmlURI type comes from
	 *libxml2. To get the libxml2 soft, go
	 *to http://xmlsoft.org
	 */
        xmlURI *uri;

        /**
	 *The last modification date of the current file.
	 *is valid only in the case of a local file
	 */
        time_t last_modif_date;

        /**
	 *wether the current file is local
	 *or not.
	 */
        gboolean is_local;
};

enum MLVIEW_FILE_DESCRIPTOR_STATUS {
        MLVIEW_FILE_DESCRIPTOR_BAD_PARAM = -2,
        MLVIEW_FILE_DESCRIPTOR_FETCH_FAILED = -1,
        MLVIEW_FILE_DESCRIPTOR_OK = 0,
        MLVIEW_FILE_DESCRIPTOR_IS_LOCAL = 1
};

#define PRIVATE(file_desc) (file_desc->priv)

/**
 *The constructor of #MlViewFileDescriptor class.
 *
 *@param a_file_uri The uri to associate to the file descriptor.
 *This null terminated char is duplicated. Then caller still owns
 *the pointer.
 *Note that this argument _must_ be a uri.
 *
 *@return the newly created file descriptor or NULL if could not
 *Have been created.
 */
MlViewFileDescriptor *
mlview_file_descriptor_new (const gchar * a_file_uri)
{
        MlViewFileDescriptor *fd = NULL;
        struct stat file_stats;
        int status;

        g_return_val_if_fail (a_file_uri != NULL, NULL);
        fd = g_malloc0 (sizeof (MlViewFileDescriptor));
        PRIVATE (fd) =
                g_malloc0 (sizeof (MlViewFileDescriptorPrivate));
        PRIVATE (fd)->uri = xmlParseURI (a_file_uri) ;
        PRIVATE (fd)->uri_str = g_strdup (a_file_uri);

        if (!PRIVATE (fd)->uri 
            || (!PRIVATE (fd)->uri->scheme)
            || (PRIVATE (fd)->uri->scheme
                && !strcmp (PRIVATE (fd)->uri->scheme,
                            "file"))) {
                PRIVATE (fd)->is_local = TRUE;
        } else {
                PRIVATE (fd)->is_local = FALSE;
        }
        if (PRIVATE (fd)->is_local == TRUE
            && PRIVATE (fd)->uri && PRIVATE (fd)->uri->path) {
                if ((status =
                     stat (PRIVATE (fd)->uri->path, &file_stats))
                    == 0) {
                        PRIVATE (fd)->last_modif_date =
                                file_stats.st_mtime;
                } else {
                        PRIVATE (fd)->last_modif_date = 0;
                }
        }
        return fd;
}


/**
 *The destructor of the #MlViewFileDescriptor class.
 *@param a_this the instance of #MlViewFileDescriptor to destroy.
 */
void
mlview_file_descriptor_destroy (MlViewFileDescriptor *a_this)
{

        g_return_if_fail (a_this != NULL);

        if (!PRIVATE (a_this))
                return;

        if (PRIVATE (a_this)->uri &&
            PRIVATE (a_this)->uri->path) {
                xmlFreeURI (PRIVATE (a_this)->uri);
                PRIVATE (a_this)->uri = NULL;
        }

        if (PRIVATE (a_this)->uri_str) {
                g_free (PRIVATE (a_this)->uri_str);
                PRIVATE (a_this)->uri_str = NULL;
        }

        g_free (PRIVATE (a_this));

        PRIVATE (a_this) = NULL;

        g_free (a_this);

        a_this = NULL;
}


/**
 *Sets the last modified time of the current instance 
 *of file descriptor to the current time. 
 *
 *@param a_this the current instance of MlViewFileDescriptor.
 */
void
mlview_file_descriptor_update_modified_time (MlViewFileDescriptor *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->is_local)
                PRIVATE (a_this)->last_modif_date =
                        time (NULL);
}


/**
 *Check wether the in memory file has been modified.
 *
 *@param a_this the current instance of #MlViewFileDescriptor. 
 *(in memory file)
 *@param a_is_modified out parameter. Is set to TRUE if the lastp modif date of
 *the in memory file is newer than the last modif date of the on disk file. 
 *This parameter is valid if and only if the return value of this method is 0.
 *
 *@return 0 upon successfull completion,  -1 otherwise.
 */
gint
mlview_file_descriptor_is_modified (const MlViewFileDescriptor *a_this,
                                    gboolean * a_is_modified)
{
        struct stat file_stats;
        guchar *file_path = NULL ;
        int status;

        *a_is_modified = FALSE;
        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->uri) {
                file_path = PRIVATE (a_this)->uri_str ;
        } else {
                if (PRIVATE (a_this)->uri->path == NULL
                    || !PRIVATE (a_this)->is_local) {
                        return -1;
                }
                file_path = PRIVATE (a_this)->uri->path ;
        }
        if (!file_path)
                return -1 ;
        status = stat (file_path, &file_stats) ;
        if (status != 0) {
                return -1;
        }
        /*
         *if the last modif date  of the on memory file is
         *newer than the last modif date of the on disk file,
         *return TRUE
         */
        *a_is_modified = (file_stats.st_mtime
                          <
                          PRIVATE (a_this)->
                          last_modif_date) ? TRUE : FALSE;
        return 0;
}


/**
 *Tells if the current file has read access for the current
 *user of the current process.
 *
 *@param a_this the this pointer.
 *@param a_is_readable out parameter. Is set to TRUE if the current file
 *has read access for the current user of the current process, FALSE otherwise.
 *@return 0 upon successfull completion, -1 otherwise.
 */
gint
mlview_file_descriptor_is_readable (const MlViewFileDescriptor *a_this,
                                    gboolean * a_is_readable)
{
        struct stat file_stats;
        guchar *file_path = NULL ;
        int status;
        mode_t mode;

        *a_is_readable = FALSE;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->uri)
                file_path = PRIVATE (a_this)->uri_str ;
        else
                file_path = PRIVATE (a_this)->uri->path ;
        if (!file_path)
                return -1;

        if (PRIVATE (a_this)->is_local) {
                status = stat (file_path,
                               &file_stats);
                if (status != 0) {
                        return -1;
                }
                mode = file_stats.st_mode;
                /*
                 *If this file offers the read access to 
                 *the current user, return TRUE
                 */
                *a_is_readable = (((file_stats.st_mode & S_IRUSR)
                                   == S_IRUSR)) ? TRUE : FALSE;
        }
        return 0;
}


/**
 *Tests if the file has write access for the current user of the 
 *current process.
 *
 *@param a_this the current instance of MlViewFileDescriptor
 *@param a_is_writeable out parameter. Is set to TRUE if the file has
 *write permission for the current user of the current process, FALSE
 *otherwise.
 *@return 0 upon successfull completion, -1 otherwise.
 */
gint
mlview_file_descriptor_is_writeable (const MlViewFileDescriptor *a_this,
                                     gboolean * a_is_writeable)
{
        struct stat file_stats;
        guchar *file_path = NULL ;
        int status;
        mode_t mode;

        *a_is_writeable = FALSE;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->uri)
                file_path = PRIVATE (a_this)->uri_str ;
        else
                file_path = PRIVATE (a_this)->uri->path ;

        if (!file_path)
                return -1 ;

        if (PRIVATE (a_this)->is_local) {
                status = stat (file_path, &file_stats);
                if (status != 0)
                        return -1;
                mode = file_stats.st_mode;
                /*
                 *If this file offers the write access 
                 *to the current user, return TRUE
                 */
                *a_is_writeable =
                        (((file_stats.st_mode & S_IWUSR)
                          == S_IWUSR)) ? TRUE : FALSE;
        }
        return 0;

}


/**
 *Gets the file path of the current file.
 *
 *@param a_this the this pointer.
 *@return the file path. Note that if no file path is associated,
 *this method returns NULL. Note also that the caller should not free the
 *returned string.
 */
gchar *
mlview_file_descriptor_get_file_path (const MlViewFileDescriptor*a_this)
{
        guchar *file_path = NULL ;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        if (!PRIVATE (a_this)->uri)
                file_path = PRIVATE (a_this)->uri_str ;
        else
                file_path = PRIVATE (a_this)->uri->path ;

        return file_path ;
}

/**
 *Sets the file path associated to the current instance of 
 *#MlViewFileDescriptor.
 *@param a_this the "this pointer".
 *@param a_file_path the file path to set.
 */
void
mlview_file_descriptor_set_file_path (MlViewFileDescriptor *a_this,
                                      gchar * a_file_path)
{
        struct stat file_stats = {0};
        int status;
        xmlChar * path = NULL ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->uri) {
                if (PRIVATE (a_this)->uri->path) {
                        path = xmlStrdup (a_file_path) ;
                        /*
                         *the following comments causes a leak.
                         *I must revisit the way the file_path are
                         *stored in hash tables in MlViewEditor
                         *before taking the risk of freeing this.
                         *Otherwise the hash table code will
                         *access freed memory.
                         *TODO: FIX THIS !!!
                         */
                        /*xmlFree (PRIVATE (a_this)->uri->path) ;*/
                        PRIVATE (a_this)->uri->path = NULL ;
                }
                PRIVATE (a_this)->uri->path = path ;
        } else {                
                if (PRIVATE (a_this)->uri_str) {
                        path = g_strdup (a_file_path) ;
                        g_free (PRIVATE (a_this)->uri_str) ;
                }
                PRIVATE (a_this)->uri_str = path ;
        }
        if (PRIVATE (a_this)->is_local) {
                status = stat (a_file_path, &file_stats);
                if (status == 0) {
                        PRIVATE (a_this)->last_modif_date =
                                file_stats.st_mtime;
                }
                PRIVATE (a_this)->last_modif_date = 0;
        }
}



/**
 *Gets the last modification date of the current file.
 *
 *@param a_this the current instance of MlViewFileDescriptor
 *@return the last modif date of the file. The last modif date is expressed
 *in seconds elapsed since epoch.
 */
guint
mlview_file_descriptor_get_lmd (const MlViewFileDescriptor * a_this)
{
        g_return_val_if_fail (a_this != NULL, 0);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, 0);

        return PRIVATE (a_this)->last_modif_date;
}


/**
 *Sets the last modified date to a_last_modif time 
 *
 *@param a_this the current instance of MlViewFileDescriptor.
 *@param a_last_modif_time the new modif time.
 */
void
mlview_file_descriptor_set_lmd (MlViewFileDescriptor *a_this,
                                guint a_last_modif_time)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->last_modif_date =
                a_last_modif_time;
}


/**
 *Tests if the current file is a directory.
 *
 *points to a directory and false if not.
 *@param a_this the "this pointer".
 *@param a_is_dir response. Is set to TRUE if the current file is
 *a directory, is set to FALSE otherwise.
 *@return 0 upon successfull completion, -1 otherwise.
 */
gint
mlview_file_descriptor_is_dir (const MlViewFileDescriptor *
                               a_this, gboolean * a_is_dir)
{
        struct stat file_stats;
        guchar *file_path = NULL ;
        gint status;

        *a_is_dir = FALSE;
        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->uri) {
                file_path = PRIVATE (a_this)->uri_str ;
        } else {
                if (PRIVATE (a_this)->uri->path == NULL
                    || !PRIVATE (a_this)->is_local) {
                        return FALSE;
                }
                file_path = PRIVATE (a_this)->uri->path ;
        }
        
        if (!file_path)
                return FALSE ;

        status = stat (file_path, &file_stats) ;
        if (status != 0)
                return -1;

        *a_is_dir =
                (S_ISDIR (file_stats.st_mode)) ? TRUE : FALSE;

        return 0;
}

/**
 *Tests if the file is a regular file.
 *
 *@param a_this the "this pointer".
 *@param a_is_reg out parameter. Is set to TRUE if the current file is a regular
 *file, is set to FALSE otherwise.
 *@return 0 upon successfull completion, -1 otherwise.
 */
gint
mlview_file_descriptor_is_regular_file (const MlViewFileDescriptor *a_this,
                                        gboolean * a_is_reg)
{
        gint status;
        guchar *file_path = NULL ;
        struct stat file_stats;

        *a_is_reg = FALSE;
        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->uri) {
                file_path = PRIVATE (a_this)->uri_str ;
        } else {                
                if (PRIVATE (a_this)->uri->path == NULL
                    || !PRIVATE (a_this)->is_local) {
                        return -1;
                }
                file_path = PRIVATE (a_this)->uri->path ;
        }

        status = stat (file_path, &file_stats);

        if (status != 0)
                return -1;

        *a_is_reg =
                (S_ISREG (file_stats.st_mode)) ? TRUE : FALSE;
        return 0;
}


/**
 *Creates a directory at the path associated to 
 *the current instance of
 *#MlViewFileDescriptor.
 *
 *@param a_this the "this pointer".
 *@param a_mode the mode of the newly created directory.
 *@return 0 upon successfull completion, an error code otherwise.
 */
gint
mlview_file_descriptor_create_directory (MlViewFileDescriptor *a_this,
                                         mode_t a_mode)
{
        guchar *file_path = NULL ;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this)->is_local ==
                              TRUE, -1);

        if (!PRIVATE (a_this)->uri) {
                file_path = PRIVATE (a_this)->uri_str ;
        } else {
                file_path = PRIVATE (a_this)->uri->path ;
        }

        if (!file_path)
                return -1 ;
        return mkdir (PRIVATE (a_this)->uri->path, a_mode);
}

/**
 *Creates a file at the path associated to the current instance of
 *#MlViewFileDescriptor.
 *
 *@param a_this creates a file descriptor.
 *@param a_mode the mode of the file to ceate.
 *@return 0 upon successfull completion, an error code otherwise.
 */
gint
mlview_file_descriptor_create_file (MlViewFileDescriptor *a_this, 
                                    mode_t a_mode)
{
        guchar *file_path = NULL ;
        gint fd = 0;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this)->is_local ==
                              TRUE, -1);
   
        if (!PRIVATE (a_this)->uri) {
                file_path = PRIVATE (a_this)->uri_str ;
        } else {
                file_path = PRIVATE (a_this)->uri->path ;
        }
        if (!file_path)
                return -1 ;
        fd = open (PRIVATE (a_this)->uri->path, O_CREAT,
                   a_mode);

        if (fd != -1) {
                /*open succeed */
                close (fd);
                return 0;
        }
        return -1;

        /*return open (PRIVATE (a_this)->uri->path, O_CREAT , a_mode) ; */

}


/**
 *Tests if the current file is on the local file system.
 *
 *@param a_this the "this pointer".
 *@param a_is_local out parameter. Is set to TRUE if the file is local,
 *is set to FALSE otherwise.
 *@return 0 upon successfull completion, an error code otherwise.
 */
gint
mlview_file_descriptor_is_local (MlViewFileDescriptor *a_this,
                                 gboolean * a_is_local)
{
        *a_is_local = FALSE;
        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        *a_is_local = PRIVATE (a_this)->is_local;
        return 0;
}


/**
 *If the file is remote, and the url is valid, 
 *fetches the file using the
 *the protocol suggested by the url 
 *scheme (http or ftp) and stores the result
 *on the local disk at the file path designated by file a_file_path. 
 *
 *@param a_file_path the local file path where to store the fetched file.
 *@param a_this the current file descriptor.
 *@return the status. 
 */
enum MLVIEW_FILE_DESCRIPTOR_STATUS
mlview_file_descriptor_fetch_file (MlViewFileDescriptor *a_this,
                                   const gchar * a_file_path)
{
        gboolean is_local = TRUE;
        gint status = 0;

        g_return_val_if_fail (a_this != NULL,
                              MLVIEW_FILE_DESCRIPTOR_BAD_PARAM);

        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              MLVIEW_FILE_DESCRIPTOR_BAD_PARAM);

        if (mlview_file_descriptor_is_local
            (a_this, &is_local))
                return MLVIEW_FILE_DESCRIPTOR_BAD_PARAM;

        if (is_local == TRUE)
                return MLVIEW_FILE_DESCRIPTOR_IS_LOCAL;

        if (!PRIVATE (a_this)->uri_str)
                return MLVIEW_FILE_DESCRIPTOR_BAD_PARAM;

        status = xmlNanoHTTPFetch (PRIVATE (a_this)->
                                   uri_str, a_file_path, NULL);

        if (status == -1) {
                return MLVIEW_FILE_DESCRIPTOR_FETCH_FAILED;
        }

        return MLVIEW_FILE_DESCRIPTOR_OK;
}
