# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Count mass difference for all peaks in given peaklist.

# load libs
import wx
import wx.grid
import re

# load modules
from nucleus import mwx
from count import mDiffCount
from modules.mformula.main import mFormula


class mDiff(wx.Panel):
    """Count mass difference for given peaklist"""

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document
        self.selected = -1

        # init counting modules
        self.mDiffCount = mDiffCount(self.config)
        self.mFormula = mFormula(self.config.elem)

        # set default param
        self.ctrlData = {}
        self.ctrlData['checkaa'] = self.config.cfg['mdiff']['checkaa']
        self.ctrlData['checkdip'] = self.config.cfg['mdiff']['checkdip']
        self.ctrlData['checkmod'] = self.config.cfg['mdiff']['checkmod']
        self.ctrlData['checkdiff'] = self.config.cfg['mdiff']['checkdiff']
        self.ctrlData['usemaxdiff'] = self.config.cfg['mdiff']['usemaxdiff']
        self.ctrlData['maxdiff'] = self.config.cfg['mdiff']['maxdiff']
        self.ctrlData['modgroups'] = self.config.cfg['mdiff']['modgroups']

        # set table colors
        self.colours = {}
        self.colours['zero'] = self.config.cfg['colours']['mdiffzero']
        self.colours['aa'] = self.config.cfg['colours']['mdiffaa']
        self.colours['dip'] = self.config.cfg['colours']['mdiffdip']
        self.colours['mod'] = self.config.cfg['colours']['mdiffmod']
        self.colours['val'] = self.config.cfg['colours']['mdiffval']

        # make items
        self.makeDifferenceTable()
        self.makeMatchList()
        hightlightBox = self.makeHighlightBox()
        massLimitsBox = self.makeMassLimitsBox()
        buttons = self.makeButtons()

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(hightlightBox, 0, wx.EXPAND|wx.ALL, 10)
        controls.Add(massLimitsBox, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(buttons, 0, wx.ALIGN_CENTRE|wx.ALL, 5)
        controls.Add(self.match_list, 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.TOP, 10)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.differenceTable, 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeDifferenceTable(self):
        """ Make main table of differences. """

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.SIMPLE_BORDER
        else:
            style=wx.SUNKEN_BORDER

        # make table
        self.differenceTable = wx.grid.Grid(self, -1, style=style)
        self.differenceTable.CreateGrid(0, 0)
        self.differenceTable.DisableDragColSize()
        self.differenceTable.DisableDragRowSize()
        self.differenceTable.SetDefaultRowSize(19)
        self.differenceTable.SetDefaultCellAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
        self.differenceTable.SetLabelFont(wx.Font(8, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
        self.differenceTable.SetColLabelSize(19)
        self.differenceTable.SetDefaultCellBackgroundColour(wx.WHITE)

        if wx.Platform == '__WXMAC__':
            self.differenceTable.SetDefaultCellFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
            self.differenceTable.SetLabelFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))

        # set event
        self.differenceTable.Bind(wx.grid.EVT_GRID_CELL_LEFT_CLICK, self.onLMC)
        self.differenceTable.Bind(wx.grid.EVT_GRID_CELL_LEFT_DCLICK, self.onSwap)
        self.differenceTable.Bind(wx.grid.EVT_GRID_SELECT_CELL, self.onSelect)
    # ----


    # ----
    def makeHighlightBox(self):
        """ Make box for check & highlight setup. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Check and Highlight"), wx.VERTICAL)

        self.highlightAA_check = wx.CheckBox(self, -1, "Amino acids")
        self.highlightDip_check = wx.CheckBox(self, -1, "Dipeptides")
        self.highlightMod_check = wx.CheckBox(self, -1, "Modifications")
        self.highlightMod_check.SetToolTip(wx.ToolTip("Modification groups can be set in preferences."))
        self.highlightDiff_check = wx.CheckBox(self, -1, "Difference")

        self.highlightDiff_value = wx.TextCtrl(self, -1, '', size=(100, -1))
        self.highlightDiff_value.SetToolTip(wx.ToolTip("Type mass or formula (e.g.Na-H)."))

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.highlightAA_check, 0, 0)
            mainBox.Add(self.highlightDip_check, 0, wx.TOP, 5)
            mainBox.Add(self.highlightMod_check, 0, wx.TOP, 5)
            mainBox.Add(self.highlightDiff_check, 0, wx.TOP, 5)
            mainBox.Add(self.highlightDiff_value, 0, wx.TOP, 5)
        else:
            mainBox.Add(self.highlightAA_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.highlightDip_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.highlightMod_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.highlightDiff_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.highlightDiff_value, 0, wx.ALL, 5)

        # set events
        self.highlightDiff_check.Bind(wx.EVT_CHECKBOX, self.onUseDifferenceSelected)
        self.highlightDiff_value.Bind(wx.EVT_TEXT_ENTER, self.onGenerate)

        # set defaults
        self.highlightAA_check.SetValue(self.ctrlData['checkaa'])
        self.highlightDip_check.SetValue(self.ctrlData['checkdip'])
        self.highlightMod_check.SetValue(self.ctrlData['checkmod'])
        self.highlightDiff_check.SetValue(self.ctrlData['checkdiff'])
        if not self.ctrlData['checkdiff']:
            self.highlightDiff_value.Enable(False)

        return mainBox
    # ----


    # ----
    def makeMassLimitsBox(self):
        """ Make box for mass limits setup. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Mass Limits"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        massLimitsMaxDiff_label = wx.StaticText(self, -1, "Max diff.: ")
        massLimitsMaxDiff_units = wx.StaticText(self, -1, " Da")
        self.massLimitsMaxDiff_value = wx.TextCtrl(self, -1, str(self.ctrlData['maxdiff']), size=(60, -1), validator=mwx.txtValidator('digits'))
        self.massLimitsMaxDiff_value.SetToolTip(wx.ToolTip("Max difference counted."))

        self.massLimitsUse_check = wx.CheckBox(self, -1, "Use max difference")

        # pack items
        grid.Add(massLimitsMaxDiff_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.massLimitsMaxDiff_value, (0, 1))
        grid.Add(massLimitsMaxDiff_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, 0)
            mainBox.Add(self.massLimitsUse_check, 0, wx.TOP, 5)
        else:
            mainBox.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.massLimitsUse_check, 0, wx.ALL, 5)

        # set events
        self.massLimitsMaxDiff_value.Bind(wx.EVT_TEXT_ENTER, self.onGenerate)
        self.massLimitsUse_check.Bind(wx.EVT_CHECKBOX, self.onUseLimitsSelected)

        # set defaults
        self.massLimitsUse_check.SetValue(self.ctrlData['usemaxdiff'])
        if not self.ctrlData['usemaxdiff']:
            self.massLimitsMaxDiff_value.Enable(False)

        return mainBox
    # ----


    # ----
    def makeButtons(self):
        """ Make buttons. """

        # make items
        generate_button = wx.Button(self, -1, "Generate")

        # set events
        generate_button.Bind(wx.EVT_BUTTON, self.onGenerate)

        return generate_button
    # ----


    # ----
    def makeMatchList(self):
        """ Make list of matched values. """

        # make items
        self.match_list = mwx.ListCtrl(self, -1, size=(155, -1))
        self.match_list.InsertColumn(0, "Matched", wx.LIST_FORMAT_LEFT)
        self.match_list.InsertColumn(1, "Error", wx.LIST_FORMAT_RIGHT)

        # set columns width
        self.match_list.SetColumnWidth(0, wx.LIST_AUTOSIZE_USEHEADER)
        self.match_list.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)
    # ----


    # ----
    def updateDifferenceTable(self, matched=None):
        """ Update data in the main table of peak differences. """

        # clear selection
        self.selected = -1

        # get peaklist data from dokument
        peaksNo = self.docData.getPeaklistLength()
        peaklist = self.docData.getPeaks()

        # delete old data
        if self.differenceTable.GetNumberRows() != 0:
            self.differenceTable.DeleteCols(0, self.differenceTable.GetNumberCols())
            self.differenceTable.DeleteRows(0, self.differenceTable.GetNumberRows())

        # create new cells
        self.differenceTable.AppendCols(peaksNo)
        self.differenceTable.AppendRows(peaksNo)

        # set columns atributes
        for x in range(peaksNo):
            cellAttr = wx.grid.GridCellAttr()
            cellAttr.SetReadOnly(True)
            self.differenceTable.SetColAttr(x, cellAttr)

        # set format for digits
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'

        # create labels
        for x in range(peaksNo):
            mass = digitsFormat % peaklist[x][0]
            self.differenceTable.SetColLabelValue(x, str(mass))
            self.differenceTable.SetRowLabelValue(x, str(mass))
            self.differenceTable.SetColFormatFloat(x, 0, self.config.cfg['common']['digits'])

        # paste data
        for x in range(peaksNo):
            for y in range(peaksNo):
                diff = abs(peaklist[x][0] - peaklist[y][0])

                # don't care for above maxdiff
                if self.ctrlData['usemaxdiff'] and diff > self.ctrlData['maxdiff']:
                    self.differenceTable.SetCellValue(x, y, '')
                    continue

                # same peaks - diagonal
                elif x == y:
                    self.differenceTable.SetCellValue(x, y, '---')
                else:
                    self.differenceTable.SetCellValue(x, y, str(diff))

                # set color according to matched type
                try:
                    if x == y:
                        self.differenceTable.SetCellBackgroundColour(x, y, self.colours['zero'])
                    elif 'val' in matched[diff]:
                        self.differenceTable.SetCellBackgroundColour(x, y, self.colours['val'])
                    elif 'aa' in matched[diff]:
                        self.differenceTable.SetCellBackgroundColour(x, y, self.colours['aa'])
                    elif 'dip' in matched[diff]:
                        self.differenceTable.SetCellBackgroundColour(x, y, self.colours['dip'])
                    elif 'mod' in matched[diff]:
                        self.differenceTable.SetCellBackgroundColour(x, y, self.colours['mod'])
                except KeyError:
                    pass

        # scrollbar hack
        h,w = self.differenceTable.GetSize()
        self.differenceTable.SetSize((h+1, w))
        self.differenceTable.SetSize((h, w))
        self.differenceTable.ForceRefresh()
        self.differenceTable.AutoSizeColumns()
    # ----


    # ----
    def updateMatchList(self, data=[]):
        """ Update data in the list of matched items. """

        # clear list
        self.match_list.DeleteAllItems()

        # paste new data
        errorType = self.docData.getMassParam('errortype')
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for x in range(len(data)):
            name = data[x][0]
            error = data[x][1]
            if data[x][2] != 'label' and errorType == 'Da':
                error = digitsFormat % error
            if data[x][2] != 'label':
                error = str(error) + ' ' + errorType

            # paste data
            self.match_list.InsertStringItem(x, str(name))
            self.match_list.SetStringItem(x, 1, str(error))

            # set color according to matched type
            if data[x][2] == 'aa':
                self.match_list.SetItemBackgroundColour(x, self.colours['aa'])
            elif data[x][2] == 'dip':
                self.match_list.SetItemBackgroundColour(x, self.colours['dip'])
            elif data[x][2] == 'mod':
                self.match_list.SetItemBackgroundColour(x, self.colours['mod'])
            elif  data[x][2] == 'val':
                self.match_list.SetItemBackgroundColour(x, self.colours['val'])

        # set columns width
        if self.match_list.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        self.match_list.SetColumnWidth(0, autosize)
        self.match_list.SetColumnWidth(1, autosize)
        self.match_list.updateLastCol()
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.differenceTable.SetFocus()
    # ----


    # ----
    def onLMC(self, evt=None, col=None, row=None):
        """ Match selected difference when left-click on table cell. """

        # get selected cell
        if evt:
            col = evt.GetCol()
            row = evt.GetRow()
        self.selected = [col, row]
        self.differenceTable.SelectBlock(row, col, row, col)

        # match diff
        diff = self.differenceTable.GetCellValue(row, col)
        matched = []
        if diff != '---' and diff != '' and diff != 0:
            digitsFormat = '%0.' + `self.ctrlData['digits']` + 'f'

            # get selected masses
            mass1 = self.differenceTable.GetColLabelValue(col)
            mass2 = self.differenceTable.GetRowLabelValue(row)

            # match diff and update table
            diff = float(diff)
            diffForm = digitsFormat % diff
            matched.append(['peak1:', mass1, 'label'])
            matched.append(['peak2:', mass2, 'label'])
            matched.append(['difference:', diffForm,'label'])
            matched += self.mDiffCount.checkSelectedDiff(diff, self.docData.getMassParam('errortype'))
            self.updateMatchList(matched)
    # ----


    # ----
    def onUseDifferenceSelected(self, evt):
        """ Enable/disable difference value textbox in the control panel. """

        # check if 'difference' checked
        if evt.IsChecked():
            self.highlightDiff_value.Enable(True)
        else:
            self.highlightDiff_value.Enable(False)
    # ----


    # ----
    def onUseLimitsSelected(self, evt):
        """ Enable/disable maxdiff textbox in the control panel. """

        # check if 'use limits' checked
        if evt.IsChecked():
            self.massLimitsMaxDiff_value.Enable(True)
        else:
            self.massLimitsMaxDiff_value.Enable(False)
    # ----


    # ----
    def onGenerate(self, evt):
        """ Get all the controls and generate diff table. """

        # check peaklist
        if not self.docData.getDataStatus('mPeak'):
            dlg = wx.MessageDialog(self, "No data in the peaklist! Please label any peaks first.", "No Data", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # get and parse controls
        if not self.getControls():
            return

        # set application working
        self.docMonitor('setAppStatus', "Checking differences...")

        # send data to count
        self.mDiffCount.ctrlData = self.ctrlData
        self.mDiffCount.peaklist = self.docData.getPeaks()

        # count differences
        matched = self.mDiffCount.preAnalysePeaklist()

        # update difference table
        self.updateDifferenceTable(matched)

        # clear matchlist
        self.match_list.DeleteAllItems()

        # set application ready
        self.docMonitor('setAppStatus', 0)

        # show 'no match' alert
        if self.config.cfg['common']['nomatchalert'] and matched == {} \
            and ( \
                self.ctrlData['checkaa'] \
                or self.ctrlData['checkdip'] \
                or self.ctrlData['checkmod'] \
                or self.ctrlData['checkdiff'] \
            ):
            message = "No match found!\nCheck error type in statusbar or increase mass tolerance."
            dlg = wx.MessageDialog(self, message, "No Match Alert", wx.OK|wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onSwap(self, evt):
        """ Go to the same position on the other side of diagonal line. """

        # swap position
        if self.selected != -1:
            self.onLMC(col=self.selected[1], row=self.selected[0])
    # ----


    # ----
    def onSelect(self, evt):
        """ Block selection. """
        return
    # ----


    # ----
    def getControls(self):
        """ Get and check controls. """

        errorMessage = ''

        # get checked
        self.ctrlData['diffvalue'] = False
        self.ctrlData['checkaa'] = self.highlightAA_check.IsChecked()
        self.ctrlData['checkdip'] = self.highlightDip_check.IsChecked()
        self.ctrlData['checkmod'] = self.highlightMod_check.IsChecked()
        self.ctrlData['checkdiff'] = self.highlightDiff_check.IsChecked()

        # get masstype and tolerance
        self.ctrlData['digits'] = self.config.cfg['common']['digits']
        self.ctrlData['masstype'] = self.docData.getMassParam('masstype')
        self.ctrlData['errortype'] = self.docData.getMassParam('errortype')
        self.ctrlData['tolerance'] = self.docData.getMassParam('tolerance')
        self.ctrlData['usemaxdiff'] = self.massLimitsUse_check.IsChecked()

        # get modification groups
        self.ctrlData['modgroups'] = self.config.cfg['mdiff']['modgroups']

        # get difference and limit values
        highlightValue = self.highlightDiff_value.GetValue()
        highlightValue = highlightValue.replace(',', '.')
        maxDiff = self.massLimitsMaxDiff_value.GetValue()
        maxDiff = maxDiff.replace(',', '.')

        # validate difference to search for
        if self.highlightDiff_check.IsChecked():
            try:
                self.ctrlData['diffvalue'] = float(highlightValue)
            except ValueError:
                self.ctrlData['diffvalue'] = self.mFormula.getMass(highlightValue, self.ctrlData['masstype'])

            if not self.ctrlData['diffvalue']:
                errorMessage = "Specified difference must be a number or formula!"

        # validate max difference value
        if self.ctrlData['usemaxdiff'] and not errorMessage:
            try:
                self.ctrlData['maxdiff'] = int(maxDiff)
            except ValueError:
                errorMessage = "Max difference must be a number!"

        # if controls not OK
        if errorMessage != '':
            errorDlg = wx.MessageDialog(self, errorMessage, "Value Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
            return False
        else:
            return True
    # ----
