# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Export spectrum data to selected format.

# load libs
import wx
import os.path

# load modules
import spectrum_ascii_xy


class mSpectrumExporter:
    """ Save spectrum data to selected format. """

    # ----
    def __init__(self, parent):
        self.parent = parent
    # ----


    # ----
    def exportData(self, document, lastDir=None):
        """ Show export dialog and save document data to file. """

        # get path from document
        path = document.getPath()

        # get file name
        if path:
            fileName = os.path.basename(path)
            fileName = os.path.splitext(fileName)
            fileName = fileName[0]
            if document.getScanID():
                fileName += '_scan' + document.getScanID()
            fileName += '.txt'
        else:
            fileName = 'spectrum.txt'

        # get dir name
        if path:
            path = os.path.dirname(path)
        else:
            path = lastDir

        # available formats
        wildcard = "Spectrum XY ASCII - tab delimiter (*.txt)|*.txt|" \
                    "Spectrum XY ASCII - space delimiter (*.txt)|*.txt|" \
                    "Spectrum XY ASCII - comma delimiter (*.txt)|*.txt|" \
                    "Spectrum XY ASCII - semicolon delimiter (*.txt)|*.txt"

        # raise export dialog
        dlg = wx.FileDialog(self.parent, "Export Spectrum", path, fileName, wildcard, wx.SAVE|wx.OVERWRITE_PROMPT)
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            choice = dlg.GetFilterIndex()
            dlg.Destroy()
        else:
            dlg.Destroy()
            return False

        # get and format data
        spectrum = document.getSpectrum()
        if spectrum:
            formats = ['%f\t%f\n', '%f %f\n', '%f,%f\n', '%f;%f\n']
            data = spectrum_ascii_xy.formatData(spectrum, formats[choice])
        else:
            data = False

        # check data
        if not data:
            dlg = wx.MessageDialog(self.parent, "Current document has no data to save in the selected format!", "No Data", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False


        # save file
        try:
            saveFile = file(path, 'w')
            saveFile.write(data.encode("utf-8"))
            saveFile.close()
            return True

        except IOError:
            dlg = wx.MessageDialog(self.parent, "File cannot be saved! Check permissions please.", "Save Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False
    # ----
