# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Select sequence from the list of sequences.

# load libs
import wx
import string

# load modules
from nucleus import mwx


class dlgSelectSequence(wx.Dialog):
    """Select one sequence from multisequence format"""

    # ----
    def __init__(self, parent, sequences):
        wx.Dialog.__init__(self, parent, -1, "Select Sequence", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.sequences = sequences
        self.selectedSequence = 0

        # make gui items
        sequenceList = self.makeSequenceList()
        buttons = self.makeButtonBox()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(sequenceList, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        else:
            mainSizer.Add(sequenceList, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 3)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeSequenceList(self):
        """ Make list of imported sequences. """

        # set style
        if wx.Platform == '__WXMAC__':
            style = wx.LC_REPORT|wx.LC_SINGLE_SEL|wx.LC_VRULES|wx.LC_HRULES|wx.SIMPLE_BORDER
        else:
            style = wx.LC_REPORT|wx.LC_SINGLE_SEL|wx.LC_VRULES|wx.LC_HRULES

        # make list
        sequences_list = mwx.ListCtrl(self, -1, size=(450, 220), style=style)
        sequences_list.SetToolTip(wx.ToolTip("Select sequence to import"))

        # make columns
        sequences_list.InsertColumn(0, "#")
        sequences_list.InsertColumn(1, "Length", wx.LIST_FORMAT_RIGHT)
        sequences_list.InsertColumn(2, "Title", wx.LIST_FORMAT_LEFT)

        # paste data
        for x in range(len(self.sequences)):
            sequences_list.InsertStringItem(x, str(x+1))
            sequences_list.SetStringItem(x, 1, str(len(self.sequences[x][1])))
            sequences_list.SetStringItem(x, 2, self.sequences[x][0])

        # set columns width
        for col in range(3):
            sequences_list.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)
        sequences_list.updateLastCol()

        # set events
        sequences_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
        sequences_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onLMDC)

        return sequences_list
    # ----


    # ----
    def makeButtonBox(self):
        """ Make main dialog buttons. """

        # make items
        self.Import_button = wx.Button(self, wx.ID_OK, "Import")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(self.Import_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set defaults
        self.Import_button.Enable(False)

        return buttonBox
    # ----


    # ----
    def onItemSelected(self, evt):
        """ Enable 'Import' button if valid sequence selected. """

        # get selected item index
        seqID = evt.m_itemIndex

        # check validity
        if seqID != -1:
            self.selectedSequence = seqID
            self.Import_button.Enable(True)
        else:
            self.Import_button.Enable(False)
    # ----


    # ----
    def onLMDC(self, evt):
        """ Import sequence on double-click on valid sequence. """

        # get selected item index
        seqID = evt.m_itemIndex

        # check validity
        if seqID != -1:
            self.selectedSequence = seqID
            self.EndModal(wx.ID_OK)
        else:
            self.Import_button.Enable(False)
    # ----
