# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Import sequence from FASTA format.

# load libs
import wx
import os

# load modules
import document_mmass
import sequence_fasta


class mSequenceImporter:
    """Import sequence from FASTA format"""

    # ----
    def __init__(self, parent):
        self.parent = parent
    # ----


    # ----
    def showDialog(self, lastDir='.', multiple=False):

        # get last dir
        if lastDir == '':
            lastDir = '.'

        # available formats
        wildcard = "All files|*.*|" \
                    "All supported documents|*.msd;*.fasta;*.fas;*.txt|" \
                    "mMass document (*.msd)|*.msd|" \
                    "FASTA format (*.fasta; *.fas)|*.fasta;*.fas"

        # set style
        if multiple:
            style = wx.OPEN|wx.MULTIPLE|wx.FILE_MUST_EXIST
        else:
            style = wx.OPEN|wx.FILE_MUST_EXIST

        # raise import dialog
        dlg = wx.FileDialog(self.parent, "Import Sequence", lastDir, '', wildcard, wx.OPEN|wx.FILE_MUST_EXIST)
        if dlg.ShowModal() == wx.ID_OK:
            if multiple:
                paths = dlg.GetPaths()
                paths.sort()
            else:
                paths = dlg.GetPath()
            dlg.Destroy()
            return paths
        else:
            dlg.Destroy()
            return False
    # ----


    # ----
    def importData(self, path):
        """ Show import dialog and import data. """

        # init data
        data = {}
        data['errors'] = ''
        data['title'] = ''
        data['sequence'] = []
        data['modifications'] = []

        docData = None

        # check if file exists
        if not os.path.exists(path):
            message = "File does not exists!"
            dlg = wx.MessageDialog(self.parent, message, "Import Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        # open file
        else:

            # busy cursor start
            wx.BeginBusyCursor()

            # FASTA
            fastaDoc = sequence_fasta.fastaDoc(self.parent)
            docData = fastaDoc.getSequence(path)
            if docData:
                data['title'] = docData['title']
                data['sequence'] = docData['sequence']
            elif docData == None:
                wx.EndBusyCursor()
                return None

            # mMass Document
            if not docData:
                mMassDoc = document_mmass.mMassDoc()
                docData = mMassDoc.getElement('sequences', path)
                if docData:
                    data['title'] = docData['seqtitle']
                    data['sequence'] = docData['sequence']
                    data['modifications'] = docData['modifications']

            # busy cursor end
            wx.EndBusyCursor()

        # check sequence
        if not docData:
            message = "Cannot import sequence data!\nFile is not accessible or incorrect data format."
            dlg = wx.MessageDialog(self.parent, message, "Format Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        return data
    # ----
