# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog for peak editing.

# load libs
import wx
import string

# load modules
from nucleus import mwx


class dlgEditPeak(wx.Dialog):
    """ Dialog for adding and editing values in peaklist. """

    def __init__(self, parent, title, data):
        wx.Dialog.__init__(self, parent, -1, title, style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.data = data

        # pack main frame
        mainBox = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.makeFieldsBox(), 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 10)
        else:
            mainBox.Add(self.makeFieldsBox(), 0, wx.ALL, 5)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 5)

        # fit layout
        mainBox.Fit(self)
        self.SetSizer(mainBox)
        self.SetMinSize(self.GetSize())
        self.Centre()


    # ----
    def makeFieldsBox(self):
        """ Make box with values. """

        # make items
        grid = mwx.GridBagSizer()

        mass_label = wx.StaticText(self, -1, "Mass/z: ")
        self.mass_value = wx.TextCtrl(self, -1, '', size=(250, -1), validator=mwx.txtValidator('float'))

        intensity_label = wx.StaticText(self, -1, "Intensity: ")
        self.intensity_value = wx.TextCtrl(self, -1, '', size=(250, -1), validator=mwx.txtValidator('float'))

        annotations_label = wx.StaticText(self, -1, "Annotations: ")
        self.annotations_value = wx.TextCtrl(self, -1, '', size=(250, 80), style=wx.TE_MULTILINE)

        # pack items
        grid.Add(mass_label, (0, 0), flag=wx.ALIGN_TOP|wx.ALIGN_RIGHT)
        grid.Add(self.mass_value, (0, 1))
        grid.Add(intensity_label, (1, 0), flag=wx.ALIGN_TOP|wx.ALIGN_RIGHT)
        grid.Add(self.intensity_value, (1, 1))
        grid.Add(annotations_label, (2, 0), flag=wx.ALIGN_TOP|wx.ALIGN_RIGHT)
        grid.Add(self.annotations_value, (2, 1), flag=wx.EXPAND)
        grid.AddGrowableRow(2)
        grid.AddGrowableCol(1)

        # set defaults
        self.mass_value.SetValue(str(self.data[0]))
        self.intensity_value.SetValue(str(self.data[1]))
        self.annotations_value.SetValue(self.data[2])

        return grid
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box with buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(OK_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events and defaults
        wx.EVT_BUTTON(self, wx.ID_OK, self.onDlgOK)
        OK_button.SetDefault()

        return mainBox
    # ----


    # ----
    def onDlgOK(self, evt):
        """ Get and parse new values. """

        # get values
        mass = self.mass_value.GetValue()
        mass = mass.replace(',', '.')

        intens = self.intensity_value.GetValue()
        intens = intens.replace(',', '.')
        if intens == '':
            intens = '0'

        annots = self.annotations_value.GetValue()
        annots = string.strip(annots)
        annots = annots.replace('\n\r', '; ')
        annots = annots.replace('\n', '; ')

        # parse values
        errorMessage = ''
        try:
            mass = float(mass)
            intens = float(intens)
        except ValueError:
            errorMessage = "Mass and intensity must be a number!"

        # if no error
        if errorMessage == '':
            self.data = [mass, intens, annots, 0]
            self.EndModal(wx.ID_OK)
        else:
            errorDlg = wx.MessageDialog(self, errorMessage, "Value Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
    # ----
