# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Spectrum printer.

# load libs
import wx
import os.path


class mSpectrumPrinter:
    """ Define application printer. """

    # ----
    def __init__(self, parent):
        """ Set initial printing params. """
        self.parent = parent

        # set defaults
        printData = wx.PrintData()
        printData.SetOrientation(wx.LANDSCAPE)
        printData.SetQuality(wx.PRINT_QUALITY_MEDIUM)
        self.pageSetup = wx.PageSetupDialogData()
        self.pageSetup.SetMarginBottomRight((20, 20))
        self.pageSetup.SetMarginTopLeft((20, 20))
        self.pageSetup.SetPrintData(printData)
    # ----


    # ----
    def onPrint(self, printout):
        """ Print page. """

        # set printer
        dlgPrintData = wx.PrintDialogData(self.pageSetup.GetPrintData())
        printer = wx.Printer(dlgPrintData)

        # print page
        status = printer.Print(self.parent, printout)

        # remember new printing params
        if status:
            printData = wx.PrintData(printer.GetPrintDialogData().GetPrintData())
            self.pageSetup.SetPrintData(printData)
        elif printer.GetLastError() == wx.PRINTER_ERROR:
            message = "Print Preview failed!\nCheck that default printer is configured."
            dlg = wx.MessageDialog(self.parent, message, "Print Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False
    # ----


    # ----
    def onPreview(self, printout1, printout2):
        """ Preview page before printing. """

        # make preview
        preview = wx.PrintPreview(printout1, printout2, self.pageSetup.GetPrintData())
        if not preview.Ok():
            message = "Print Preview failed!\nCheck that default printer is configured."
            dlg = wx.MessageDialog(self, message, "Print Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
        preview.SetZoom(50)

        # raise preview dialog
        frame = wx.PreviewFrame(preview, self.parent, "Preview")
        frame.Initialize()
        frame.SetPosition(self.parent.GetPosition())
        frame.SetSize(self.parent.GetSize())
        frame.Centre(wx.BOTH)
        frame.Show(True)
    # ----


    # ----
    def onPageSetup(self):
        """Brings up the page setup dialog"""

        # get current data
        data = self.pageSetup
        dlg = wx.PageSetupDialog(self.parent, data)
        try:
            if dlg.ShowModal() == wx.ID_OK:
                data = dlg.GetPageSetupData()
                self.pageSetup.SetMarginBottomRight(data.GetMarginBottomRight())
                self.pageSetup.SetMarginTopLeft(data.GetMarginTopLeft())
                self.pageSetup.SetPrintData(data.GetPrintData())
        finally:
            dlg.Destroy()
    # ----


    # ----
    def getPageSetup(self):
        """ Get current page setup. """
        return self.pageSetup
    # ----
