# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main document frame.

# load libs
import sys
import os.path
import wx
import webbrowser
import urllib
import re

# load modules
import images
from nucleus.ids import *
from nucleus.controls import controls
from nucleus.menubar import menuBar
from nucleus.toolbar import toolBar
from nucleus.statusbar import statusBar
from nucleus.docwindow import docWindow
from nucleus.errors import errorHandler

from modules.mpref.app_pref import mAppPref
from modules.mpref.mod_pref import mModPref
from nucleus.dlg_about import dlgAbout

from modules.mimport.document import mDocumentImporter
from modules.mimport.peaklist import mPeaklistImporter
from modules.mimport.sequence import mSequenceImporter

class docFrame(wx.Frame):
    """ Main document frame. """

    # ----
    def __init__(self, parent):

        self.config = parent.config
        self.appMonitor = parent.appMonitor
        self.document = None

        # initialize document frame
        wx.Frame.__init__(self, None, -1, "mMass", style=wx.DEFAULT_FRAME_STYLE|wx.NO_FULL_REPAINT_ON_RESIZE)
        self.DragAcceptFiles(True)

        # initialize main application error handler
        errHandler = errorHandler(self, self.config.cfg['common']['errtoconsole'], self.config.cfg['common']['debugmode'])

        # set document icon
        icons = wx.IconBundle()
        icons.AddIcon(images.getIcon16Icon())
        icons.AddIcon(images.getIcon22Icon())
        icons.AddIcon(images.getIcon24Icon())
        icons.AddIcon(images.getIcon32Icon())
        icons.AddIcon(images.getIcon48Icon())
        self.SetIcons(icons)

        # initialize application controls monitor
        self.controls = controls()

        # make menubar
        self.menuBar = menuBar()
        self.SetMenuBar(self.menuBar)
        self.setMenuBarEvents()
        self.controls.menuBar = self.menuBar

        # make toolbar
        self.toolBar = toolBar(self, self.config)
        self.SetToolBar(self.toolBar)
        self.toolBar.Realize()
        self.controls.toolBar = self.toolBar

        # make statusbar
        self.statusBar = statusBar(self)
        self.SetStatusBar(self.statusBar)
        self.controls.statusBar = self.statusBar

        # update controls
        self.docMonitor('onInit')

        # set window events
        wx.EVT_SIZE(self, self.onSize)
        wx.EVT_CLOSE(self, self.onFileClose)
        wx.EVT_DROP_FILES(self, self.onDragAndDropFiles)

        # set frame size
        size = (self.config.cfg['common']['appwidth'], self.config.cfg['common']['appheight'])
        self.SetSize(size)

        # show document frame
        if self.config.cfg['common']['maximize']:
            self.Show(True)
            self.Maximize()
        else:
            if wx.Platform == '__WXMAC__':
                self.Centre(wx.BOTH)
            self.Show(True)
    # ----


    # ----
    def setMenuBarEvents(self):
        """ Set events for main menu items. """

        # file menu events
        self.Bind(wx.EVT_MENU, self.onFileNew, id=ID_fileNew)
        self.Bind(wx.EVT_MENU, self.onFileOpen, id=ID_fileOpen)
        self.Bind(wx.EVT_MENU, self.onFileSave, id=ID_fileSave)
        self.Bind(wx.EVT_MENU, self.onFileSaveAs, id=ID_fileSaveAs)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles0)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles1)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles2)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles3)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles4)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles5)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles6)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles7)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles8)
        self.Bind(wx.EVT_MENU, self.onFileOpenRecent, id=ID_fileRecentFiles9)
        self.Bind(wx.EVT_MENU, self.onFilePrintSpectrum, id=ID_filePrintSpectrum)
        self.Bind(wx.EVT_MENU, self.onFilePrintReport, id=ID_filePrintReport)
        self.Bind(wx.EVT_MENU, self.onFilePageSetup, id=ID_filePageSetup)
        self.Bind(wx.EVT_MENU, self.onFileImportPeaklist, id=ID_fileImportPeaklist)
        self.Bind(wx.EVT_MENU, self.onFileImportSequence, id=ID_fileImportSequence)
        self.Bind(wx.EVT_MENU, self.onFileExportSpectrumData, id=ID_fileExportSpectrumData)
        self.Bind(wx.EVT_MENU, self.onFileExportSpectrumImage, id=ID_fileExportSpectrumImage)
        self.Bind(wx.EVT_MENU, self.onFileExportPeaklist, id=ID_fileExportPeaklist)
        self.Bind(wx.EVT_MENU, self.onFileInfo, id=ID_fileInfo)
        self.Bind(wx.EVT_MENU, self.onFileClose, id=ID_fileClose)
        self.Bind(wx.EVT_MENU, self.onExit, id=ID_fileExit)

        # peaklist menu events
        self.Bind(wx.EVT_MENU, self.onPeaklistUndo, id=ID_peaklistUndo)
        self.Bind(wx.EVT_MENU, self.onPeaklistRedo, id=ID_peaklistRedo)
        self.Bind(wx.EVT_MENU, self.onPeaklistAddPeak, id=ID_peaklistAddPeak)
        self.Bind(wx.EVT_MENU, self.onPeaklistClearAllAnnots, id=ID_peaklistClearAllAnnots)
        self.Bind(wx.EVT_MENU, self.onPeaklistDeleteAllPeaks, id=ID_peaklistDeleteAllPeaks)
        self.Bind(wx.EVT_MENU, self.onPeaklistSelectAllPeaks, id=ID_peaklistSelectAllPeaks)
        self.Bind(wx.EVT_MENU, self.onPeaklistSelectAnnotatedPeaks, id=ID_peaklistSelectAnnotatedPeaks)
        self.Bind(wx.EVT_MENU, self.onPeaklistSelectByTreshold, id=ID_peaklistSelectByTreshold)
        self.Bind(wx.EVT_MENU, self.onPeaklistInvertSelection, id=ID_peaklistInvertSelection)
        self.Bind(wx.EVT_MENU, self.onPeaklistCopyToClipboard, id=ID_peaklistCopyToClipboard)
        self.Bind(wx.EVT_MENU, self.onPeaklistRefresh, id=ID_peaklistRefresh)

        # spectrum menu events
        self.Bind(wx.EVT_MENU, self.onSpectrumSetMouseFce, id=ID_spectrumLabelPoint)
        self.Bind(wx.EVT_MENU, self.onSpectrumSetMouseFce, id=ID_spectrumLabelPeak)
        self.Bind(wx.EVT_MENU, self.onSpectrumSetMouseFce, id=ID_spectrumDeleteLabel)
        self.Bind(wx.EVT_MENU, self.onSpectrumSetMouseFce, id=ID_spectrumMeasureDistance)
        self.Bind(wx.EVT_MENU, self.onSpectrumAddToCompare, id=ID_spectrumAddToCompare)
        self.Bind(wx.EVT_MENU, self.onSpectrumCopyToClipboard, id=ID_spectrumCopyToClipboard)
        self.Bind(wx.EVT_MENU, self.onSpectrumSetRange, id=ID_spectrumSetRange)
        self.Bind(wx.EVT_MENU, self.onSpectrumViewOptions, id=ID_spectrumViewOptions)
        self.Bind(wx.EVT_MENU, self.onSpectrumRefresh, id=ID_spectrumRefresh)

        # processing menu events
        self.Bind(wx.EVT_MENU, self.onProcessingUndo, id=ID_processingUndo)
        self.Bind(wx.EVT_MENU, self.onProcessingCalibration, id=ID_processingInternalCalibration)
        self.Bind(wx.EVT_MENU, self.onProcessingCalibration, id=ID_processingStatisticalCalibration)
        self.Bind(wx.EVT_MENU, self.onProcessingBaseline, id=ID_processingBaseline)
        self.Bind(wx.EVT_MENU, self.onProcessingSmooth, id=ID_processingSmooth)
        self.Bind(wx.EVT_MENU, self.onProcessingCrop, id=ID_processingCrop)

        # matching menu events
        self.Bind(wx.EVT_MENU, self.onMatchingMatch, id=ID_matchingMatchData)
        self.Bind(wx.EVT_MENU, self.onMatchingInfo, id=ID_matchingMatchInfo)
        self.Bind(wx.EVT_MENU, self.onMatchingAnnotatePeaklist, id=ID_matchingAnnotate)

        # sequence menu events
        self.Bind(wx.EVT_MENU, self.onSequenceEditMod, id=ID_sequenceEditMod)
        self.Bind(wx.EVT_MENU, self.onSequenceSearchMass, id=ID_sequenceSearchMass)
        self.Bind(wx.EVT_MENU, self.onSequenceSearchSequence, id=ID_sequenceSearchSeq)

        # tools menu events
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMSpec)
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMSeq)
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMCut)
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMFrag)
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMDiff)
        self.Bind(wx.EVT_MENU, self.onToolSwitch, id=ID_toolsMComp)
        self.Bind(wx.EVT_MENU, self.onToolMascotSearch, id=ID_toolsMMascotPMF)
        self.Bind(wx.EVT_MENU, self.onToolMascotSearch, id=ID_toolsMMascotSeq)
        self.Bind(wx.EVT_MENU, self.onToolMascotSearch, id=ID_toolsMMascotMSMS)
        self.Bind(wx.EVT_MENU, self.onToolSeqConverter, id=ID_toolsMConv)

        # link menu events
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksMatrixScience)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksProspector)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksExpasy)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksUniMod)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksDeltaMass)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksUniProt)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksEMBLEBI)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksPIR)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksNCBI)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksBLAST)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksClustalW)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksFASTA)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksMUSCLE)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksPDB)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_linksSearchXLinks)

        # config menu events
        self.Bind(wx.EVT_MENU, self.onConfigMassType, id=ID_configMonoisotopic)
        self.Bind(wx.EVT_MENU, self.onConfigMassType, id=ID_configAverage)
        self.Bind(wx.EVT_MENU, self.onConfigTolerance, id=ID_configTolerance)
        self.Bind(wx.EVT_MENU, self.onConfigShowPeaklist, id=ID_configShowPeaklist)
        self.Bind(wx.EVT_MENU, self.onConfigSplitting, id=ID_configPeaklistBottom)
        self.Bind(wx.EVT_MENU, self.onConfigModifications, id=ID_configModifications)
        self.Bind(wx.EVT_MENU, self.onConfigPreferences, id=ID_configPreferences)

        # help menu events
        self.Bind(wx.EVT_MENU, self.onHelpUpdate, id=ID_helpUpdate)
        self.Bind(wx.EVT_MENU, self.onLinksOpen, id=ID_helpHomepage)
        self.Bind(wx.EVT_MENU, self.onHelpAbout, id=ID_helpAbout)
    # ----


    # ----
    def setTitle(self, title=None):
        """ Set frame title. """

        # get document path
        if not title and self.document:
            title = self.document.docData.getPath()

        # get spectrum scan ID
        if self.document:
            scanID = self.document.docData.getScanID()
            if scanID:
                title = '%s (scan #%s)' % (title, scanID)

        # set title
        if not title:
            self.SetTitle('mMass')
        else:
            title = 'mMass - [%s]' % title
            self.SetTitle(title)
    # ----


    # ----
    def setDocumentData(self, data, param=None):
        """ Set data to document. """
        self.document.setDocumentData(data, param)
    # ----


    # ----
    def makeDocument(self, data=None):
        """ Initialize document structure and window. """

        self.document = docWindow(self)
        self.document.setDocumentData(data)
        self.docMonitor('onInit')
    # ----


    # ----
    def destroyDocument(self):
        """ Destroy document structure and window. """

        self.document.Destroy()
        self.document = None
        self.docMonitor('onInit')
    # ----


    # ----
    def isEmpty(self):
        """ Check if no document initialized. """

        if not self.document:
            return True
        else:
            return False
    # ----


    # ----
    def checkSameDocument(self, path):
        """ Check document path and scan ID. """

        if self.document:

            # get current data
            currentPath = self.document.docData.getPath()
            scanID = self.document.docData.getScanID()

            # check same path and not multiscan document
            if currentPath == path and not scanID:
                return True
            else:
                return False

        # no document in current frame
        else:
            return False
    # ----


    # ----
    def onSize(self, evt):
        """ Save frame size if resized. """

        # remember frame size
        self.config.cfg['common']['maximize'] = int(self.IsMaximized())
        if not self.IsMaximized():
            self.config.cfg['common']['appwidth'] = self.GetSizeTuple()[0]
            self.config.cfg['common']['appheight'] = self.GetSizeTuple()[1]

        evt.Skip()
    # ----


    # ----
    def onDragAndDropFiles(self, evt):
        """ Open files dragged to application window. """

        files = evt.GetFiles()
        for path in files:
            self.onFileOpen(path=path)
    # ----


    # ----
    def onFileNew(self, evt):
        """ Create blank document for new peaklist. """
        self.appMonitor('onFileNew')
    # ----


    # ----
    def onFileOpen(self, evt=None, path=None):
        """ Rise open dialog and import document data. """

        # set application working
        self.docMonitor('setAppStatus', "Loading document...")

        # init importer
        importer = mDocumentImporter(self)

        # get document paths
        if not path:
            paths = importer.showDialog(self.config.cfg['common']['lastspecdir'], multiple=True)
            if not paths:
                self.docMonitor('setAppStatus', 0)
                return
        else:
            paths = [path]

        # load documents
        for path in paths:

            # check if not already open
            if self.appMonitor('checkOpenPath', path):
                continue

            # import document data
            data = importer.importData(path)
            if not data:
                continue

            # check document description
            if data['docType'] != 'mSD' and self.config.cfg['common']['updatedescr']:
                data = importer.checkDescription(data,
                                        self.config.cfg['common']['operator'],
                                        self.config.cfg['common']['contact'],
                                        self.config.cfg['common']['institution'],
                                        self.config.cfg['common']['instrument'])

            # raise new document
            self.appMonitor('onFileOpen', data)

            # remember document directory
            self.config.cfg['common']['lastspecdir'] = os.path.split(path)[0]

        # update application
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onFileOpenRecent(self, evt):
        """ Open file from history. """

        # get path
        IDs = [
                ID_fileRecentFiles0,
                ID_fileRecentFiles1,
                ID_fileRecentFiles2,
                ID_fileRecentFiles3,
                ID_fileRecentFiles4,
                ID_fileRecentFiles5,
                ID_fileRecentFiles6,
                ID_fileRecentFiles7,
                ID_fileRecentFiles8,
                ID_fileRecentFiles9
                ]
        index = IDs.index(evt.GetId())
        path = self.config.cfg['history']['f' + str(index)]

        # open path
        self.onFileOpen(path=path)
    # ----


    # ----
    def onFilePrintSpectrum(self, evt):
        """ Print spectrum. """
        self.document.onFilePrintSpectrum()
    # ----


    # ----
    def onFilePrintReport(self, evt):
        """ Print sample report. """
        self.document.onFilePrintReport()
    # ----


    # ----
    def onFilePageSetup(self, evt):
        """ Setup page before printing. """
        self.document.onFilePageSetup()
    # ----


    # ----
    def onFileSave(self, evt=None):
        """ Save document data to file. """

        # save document
        path = self.document.onFileSave()

        # update file history
        if path:
            self.appMonitor('onFileSave', path)
            return True
        else:
            return False
    # ----


    # ----
    def onFileSaveAs(self, evt):
        """ Save document data to file. """
        return self.document.onFileSave(saveAs=True)
    # ----


    # ----
    def onFileInfo(self, evt):
        """ Show information about current document. """
        self.document.onFileInfo()
    # ----


    # ----
    def onFileImportPeaklist(self, evt):
        """ Rise import dialog and import peaklist data from file. """

        # set application working
        self.docMonitor('setAppStatus', "Importing peaklist...")

        # init importer
        importer = mPeaklistImporter(self)

        # get document path
        path = importer.showDialog(self.config.cfg['common']['lastspecdir'])
        if not path:
            self.docMonitor('setAppStatus', 0)
            return

        # import document data
        data = importer.importData(path)
        if not data:
            self.docMonitor('setAppStatus', 0)
            return

        # ask to rewrite current data if any
        if self.document:
            dlg = wx.MessageDialog(self, "Import peaklist to current document?\n(Press \"No\" to make new document.)", "Import To Current Document", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)

            # import data to current document
            if dlg.ShowModal() == wx.ID_YES:
                dlg.Destroy()
                self.document.onFileImportPeaklist(data['peaklist'])
                self.docMonitor('setAppStatus', 0)
                return

            else:
                dlg.Destroy()

        # check if not already open
        if self.appMonitor('checkOpenPath', path):
            self.docMonitor('setAppStatus', 0)
            return

        # update application
        self.docMonitor('setAppStatus', 0)

        # raise new document
        self.appMonitor('onFileOpen', data)
    # ----


    # ----
    def onFileImportSequence(self, evt):
        """ Import sequence from ASCII file. """

        # set application working
        self.docMonitor('setAppStatus', "Importing sequence...")

        # init importer
        importer = mSequenceImporter(self)

        # get document path
        path = importer.showDialog(self.config.cfg['common']['lastseqdir'])
        if not path:
            self.docMonitor('setAppStatus', 0)
            return

        # import document data
        data = importer.importData(path)
        if not data:
            self.docMonitor('setAppStatus', 0)
            return

        # import data to current document
        self.document.onFileImportSequence(data)

        # remember document directory
        self.config.cfg['common']['lastseqdir'] = os.path.split(path)[0]
    # ----


    # ----
    def onFileExportSpectrumData(self, evt):
        """ Export spectrum to ASCII file. """
        self.document.onFileExportSpectrumData()
    # ----


    # ----
    def onFileExportSpectrumImage(self, evt):
        """ Export spectrum to image file. """
        self.document.onFileExportSpectrumImage()
    # ----


    # ----
    def onFileExportPeaklist(self, evt):
        """ Export peaklist to ASCII file. """
        self.document.onFileExportPeaklist()
    # ----


    # ----
    def onFileClose(self, evt=None):
        """ Ask to save unsaved data and close current document. """

        # show current document frame
        self.Show(True)

        # destroy if no document
        if not self.document:
            self.appMonitor('onFileClose')
            self.Destroy()

        # check 'changed status' and ask to close current document
        elif self.document.onFileClose():
            self.destroyDocument()

            # destroy document frame if not last
            if self.appMonitor('onFileClose', self) \
                or not evt \
                or evt.GetId() != ID_fileClose:
                self.Destroy()

        # stop closing
        else:
            return False
    # ----


    # ----
    def onPeaklistUndo(self, evt):
        """ Step back in peaklist history. """
        self.document.onPeaklistUndo()
    # ----


    # ----
    def onPeaklistRedo(self, evt):
        """ Step forward in peaklist history. """
        self.document.onPeaklistRedo()
    # ----


    # ----
    def onPeaklistAddPeak(self, evt):
        """ Add new peak to peaklist. """
        self.document.onPeaklistAddPeak()
    # ----


    # ----
    def onPeaklistClearAllAnnots(self, evt):
        """ Clear all annotations in peaklist. """
        self.document.onPeaklistClearAllAnnots()
    # ----


    # ----
    def onPeaklistDeleteAllPeaks(self, evt):
        """ Delete all peaks in peaklist. """
        self.document.onPeaklistDeleteAllPeaks()
    # ----


    # ----
    def onPeaklistSelectAllPeaks(self, evt):
        """ Select all peaks in peaklist. """
        self.document.onPeaklistSelectAllPeaks()
    # ----


    # ----
    def onPeaklistSelectAnnotatedPeaks(self, evt):
        """ Select all annotated peaks in peaklist. """
        self.document.onPeaklistSelectAnnotatedPeaks()
    # ----


    # ----
    def onPeaklistSelectByTreshold(self, evt):
        """ Select peaks by intensity treshold. """
        self.document.onPeaklistSelectByTreshold()
    # ----


    # ----
    def onPeaklistInvertSelection(self, evt):
        """ Invert current peaklist selection. """
        self.document.onPeaklistInvertSelection()
    # ----


    # ----
    def onPeaklistCopyToClipboard(self, evt):
        """ Copy peaklist data to clipboard. """
        self.document.onPeaklistCopyToClipboard()
    # ----


    # ----
    def onPeaklistRefresh(self, evt):
        """ Redraw peaklist. """
        self.document.onPeaklistRefresh()
    # ----


    # ----
    def onSpectrumSetMouseFce(self, evt):
        """ Set mouse fce in spectrum module. """

        # get fce
        itemID = evt.GetId()
        fces = {ID_spectrumLabelPoint:'point', \
                ID_spectrumLabelPeak:'peak', \
                ID_spectrumDeleteLabel:'delete', \
                ID_spectrumMeasureDistance:'distance'}
        fce = fces[itemID]

        # set fce to document
        self.document.onSpectrumSetMouseFce(fce)
    # ----


    # ----
    def onSpectrumAddToCompare(self, evt):
        """ Add spectrum to compare with current. """
        self.document.onSpectrumAddToCompare()
    # ----


    # ----
    def onSpectrumCopyToClipboard(self, evt):
        """ Copy current spectrum view to clipboard. """
        self.document.onSpectrumCopyToClipboard()
    # ----


    # ----
    def onSpectrumSetRange(self, evt):
        """ Set spectrum range. """
        self.document.onSpectrumSetRange()
    # ----


    # ----
    def onSpectrumViewOptions(self, evt):
        """ Show spectrum view options. """
        self.document.onSpectrumViewOptions()
    # ----


    # ----
    def onSpectrumRefresh(self, evt):
        """ Set view params for spectrum and redraw spectrum. """
        self.document.onSpectrumRefresh()
    # ----


    # ----
    def onProcessingUndo(self, evt):
        """ Revert spectrum and peaklist to init data. """
        self.document.onProcessingUndo()
    # ----


    # ----
    def onProcessingCalibration(self, evt):
        """ Data calibration. """

        # get calibration type
        if evt.GetId() == ID_processingInternalCalibration:
            self.document.onProcessingCalibration('internal')
        elif evt.GetId() == ID_processingStatisticalCalibration:
            self.document.onProcessingCalibration('statistic')
    # ----


    # ----
    def onProcessingBaseline(self, evt):
        """ Do spectrum baseline correction. """
        self.document.onProcessingBaseline()
    # ----


    # ----
    def onProcessingSmooth(self, evt):
        """ Smooth spectrum by Savitzky-Golay filter. """
        self.document.onProcessingSmooth()
    # ----


    # ----
    def onProcessingCrop(self, evt):
        """ Crop spectrum and peaklist data to selected range. """
        self.document.onProcessingCrop()
    # ----


    # ----
    def onMatchingMatch(self, evt):
        """ Match module data to peaklist. """
        self.document.onMatchingMatch()
    # ----


    # ----
    def onMatchingInfo(self, evt):
        """ Show info about last match. """
        self.document.onMatchingInfo()
    # ----


    # ----
    def onMatchingAnnotatePeaklist(self, evt):
        """ Annotate matched peaks in peaklist. """
        self.document.onMatchingAnnotatePeaklist()
    # ----


    # ----
    def onSequenceEditMod(self, evt):
        """ Rise dialog for sequence modifications. """
        self.document.onSequenceEditMod()
    # ----


    # ----
    def onSequenceSearchMass(self, evt):
        """ Search for specified mass in sequence. """
        self.document.onSequenceSearchMass()
    # ----


    # ----
    def onSequenceSearchSequence(self, evt):
        """ Search for specified sub-sequence in sequence. """
        self.document.onSequenceSearchSequence()
    # ----


    # ----
    def onToolSwitch(self, evt):
        """ Switch current module. """

        # get selected module
        evtID = evt.GetId()
        modules = {ID_toolsMSpec: 'mSpec', \
                    ID_toolsMSeq: 'mSeq', \
                    ID_toolsMCut: 'mCut', \
                    ID_toolsMFrag: 'mFrag', \
                    ID_toolsMDiff: 'mDiff', \
                    ID_toolsMComp: 'mComp'}

        # switch module
        self.document.onToolSwitch(modules[evtID])
    # ----


    # ----
    def onToolMascotSearch(self, evt):
        """ Send peaklist data to Mascot server. """

        # PMF search
        if evt.GetId() == ID_toolsMMascotPMF:
            self.document.onToolMascotSearch('pmf')

        # sequence query
        elif evt.GetId() == ID_toolsMMascotSeq:
            self.document.onToolMascotSearch('seq')

        # MS/MS
        elif evt.GetId() == ID_toolsMMascotMSMS:
            self.document.onToolMascotSearch('msms')
    # ----


    # ----
    def onToolSeqConverter(self, evt):
        """ Show sequence converter tool. """
        self.document.onToolSeqConverter()
    # ----


    # ----
    def onLinksOpen(self, evt=None, link=None):
        """ Find and open selected link in default webbrowser. """

        # get link
        if not link and evt:
            linkID = evt.GetId()
            links = {ID_helpHomepage:'mmass',
                    ID_linksMatrixScience:'matrixscience',
                    ID_linksProspector:'prospector',
                    ID_linksExpasy:'expasy',
                    ID_linksUniMod:'unimod',
                    ID_linksDeltaMass:'deltamass',
                    ID_linksUniProt:'uniprot',
                    ID_linksEMBLEBI:'emblebi',
                    ID_linksPIR:'pir',
                    ID_linksNCBI:'ncbi',
                    ID_linksBLAST:'blast',
                    ID_linksClustalW:'clustalw',
                    ID_linksFASTA:'fasta',
                    ID_linksMUSCLE:'muscle',
                    ID_linksPDB:'pdb',
                    ID_linksSearchXLinks:'searchxlinks'}
            link = self.config.cfg['links'][links[linkID]]

        # open link in webbrowser
        try:
            webbrowser.open(link)
        except:
            dlg = wx.MessageDialog(self, "Cannot open the link.", "Browser Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onConfigMassType(self, evt):
        """ Set actual type of mass (average/monoisotopic) to document. """

        if evt.GetId() == ID_configAverage:
            self.document.onConfigMassType('amass')
        else:
            self.document.onConfigMassType('mmass')
    # ----


    # ----
    def onConfigTolerance(self, evt):
        """ Raise dialog to set tolerance value. """
        self.document.onConfigTolerance()
    # ----


    # ----
    def onConfigShowPeaklist(self, evt):
        """ Show/hide peaklist panel. """
        self.document.onConfigShowPeaklist()
    # ----


    # ----
    def onConfigSplitting(self, evt):
        """ Set current splitting - peaklist position. """
        self.document.onConfigSplitting()
    # ----


    # ----
    def onConfigModifications(self, evt):
        """ Rise modifications editor. """

        # raise edit dialog
        mModPref(self, self.config)
        
        # check applied modifications in all documents
        if not self.appMonitor('checkModifications'):
            message = "Some of the used modifications are no more available and were removed from the sequence!\nCheck sequence editor in all opened documents please."
            dlg = wx.MessageDialog(self, message, "Unknown Modification Found", wx.OK|wx.ICON_EXCLAMATION)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onConfigPreferences(self, evt):
        """ Rise dialog for application preferences. """

        dlg = mAppPref(self, self.config)
        if dlg.ShowModal() == wx.ID_OK:
            dlg.Destroy()
        else:
            dlg.Destroy()
    # ----


    # ----
    def onHelpAbout(self, evt):
        """ About application. """

        dlg = dlgAbout(self)
        if dlg.ShowModal() == wx.ID_OK:
            dlg.Destroy()
            self.onLinksOpen(link=self.config.cfg['links']['mmass'])
        else:
            dlg.Destroy()
    # ----


    # ----
    def onHelpUpdate(self, evt):
        """ Check for newer version of mMass. """

        self.docMonitor('setAppStatus', 'Checking for updates...')

        currentVersion = '020400'
        newerAvailable = self.checkForUpdate(currentVersion)

        # bad connecion
        if newerAvailable == None:
            message = "Unable to connect mMass webpage."
            dlg = wx.MessageDialog(self, message, "Connection Failed", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()

        # no newer version available
        elif newerAvailable == False:
            message = "You are using the newest version currently available."
            dlg = wx.MessageDialog(self, message, "You are up to date!", wx.OK|wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()

        # update available
        elif newerAvailable == True:
            message = "Newer version of mMass is available. Do you want to check the mMass homepage?"
            dlg = wx.MessageDialog(self, message, "New version available!", wx.YES_NO|wx.ICON_INFORMATION|wx.YES_DEFAULT)
 
            if dlg.ShowModal() == wx.ID_YES:
                dlg.Destroy()

                # go to mMass homepage
                try:
                    webbrowser.open(self.config.cfg['links']['mmass'])
                except:
                    dlg = wx.MessageDialog(self, "Cannot open the link.", "Browser Error", wx.OK|wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()

            # do not open the link
            else:
                dlg.Destroy()

        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onExit(self, evt):
        """ Close all documents and exit application. """

        self.onFileClose()
        self.appMonitor('onFileExit')
    # ----


    # ----
    def checkAppliedModifications(self):
        """ Check applied sequence modifications. """

        if self.document:
            return self.document.checkAppliedModifications()
        else:
            return True
    # ----


    # ----
    def checkForUpdate(self, currentVersion):
        """ Check for newer version of mMass. """

        # connect to webpage
        try:
            versionFile = urllib.urlopen(self.config.cfg['links']['mmassupdate'])
            latestVersion = versionFile.read()
            latestVersion = latestVersion.strip()
        except IOError:
            return None

        # parse latest version
        if re.match('^[0-9]{6}$', latestVersion):
            if int(latestVersion) > int(currentVersion):
                return True
            else:
                return False

        # bad value from webpage
        else:
            return None
    # ----


    # ----
    def docMonitor(self, event, value=None):
        """ Monitor of changes in document modules. """

        # show application status on statusbar
        if event == 'setAppStatus':
            self.controls.setAppStatus(value)

        # enable tools and update status on document init
        elif event == 'onInit':

            # enable all items
            if not self.document:
                self.controls.enableAllItems(False)
                return
            else:
                self.controls.enableAllItems(True)

            # get document status
            changed = self.document.docData.getChangeStatus()
            processed = self.document.docData.getProcessingStatus()
            peakLength = self.document.docData.getPeaklistLength()
            specLength = self.document.docData.getSpectrumLength()
            specRange = self.document.docData.getSpectrumXRange()
            seqLength = self.document.docData.getSequenceLength()
            mods = self.document.docData.getModificationsCount()
            moduleData = self.document.docData.getDataStatus('mSpec')
            matchStatus = self.document.docData.getMatchStatus('mSpec')

            massType = self.document.docData.getMassParam('masstype')
            errorType = self.document.docData.getMassParam('errortype')
            tolerance = self.document.docData.getMassParam('tolerance')
            tolerance = '%s %s' % (tolerance, errorType)

            # update gui
            self.controls.setStatusBar('tolerance', tolerance)
            self.controls.setStatusBar('peaklist', peakLength)
            self.controls.setStatusBar('sequence', (seqLength, mods))
            self.controls.setStatusBar('spectrum', (specLength, specRange))
            self.controls.toggleSplitWindow(self.config.cfg['mpeak']['showpeaklist'])
            self.controls.toggleSplittingType(self.config.cfg['mpeak']['splitting'])
            self.controls.toggleMassType(massType)
            self.controls.toggleModule('mSpec')
            self.controls.enableFileSave(changed)
            self.controls.enablePrintingTools(bool(specLength), bool(peakLength))
            self.controls.toggleSpectrumFce(self.config.cfg['mspec']['mousefce'])
            self.controls.enableSpectrumTools(bool(specLength), bool(peakLength))
            self.controls.enableMatchingTools('mSpec', moduleData, matchStatus)
            self.controls.enablePeaklistTools(bool(peakLength))
            self.controls.enablePeaklistUndoRedo(False, False)
            self.controls.enableProcessingUndo(processed)
            self.controls.enableProcessingTools(bool(specLength), bool(peakLength))
            self.controls.enableSequenceTools(bool(seqLength))
            self.document.onSpectrumSetMouseFce(self.config.cfg['mspec']['mousefce'])

        # update buttons if module has changed
        elif event == 'onModuleChanged':
            moduleData = self.document.docData.getDataStatus(value)
            matchStatus = self.document.docData.getMatchStatus(value)
            self.controls.toggleModule(value)
            self.controls.enableMatchingTools(value, moduleData, matchStatus)

        # update spectrum fce buttons
        elif event == 'onSpectrumFceChanged':
            self.controls.toggleSpectrumFce(value)

        # update masstype buttons and status bar
        elif event == 'onMassTypeChanged':
            self.controls.toggleMassType(value)

        # update errortype buttons and status bar
        elif event == 'onToleranceChanged':
            self.controls.setStatusBar('tolerance', value)

        # update splitting button
        elif event == 'onPeaklistView':
            self.controls.toggleSplitWindow(bool(value))
            self.controls.toggleSplittingType(value)

        # enable/disable save fce and change title
        elif event == 'onDocumentSaved':
            changeStatus = self.document.docData.getChangeStatus()
            path = self.document.docData.getPath()
            self.controls.enableFileSave(changeStatus)
            self.setTitle(path)

        # enable/disable save fce
        elif event == 'onDocumentChanged':
            changeStatus = self.document.docData.getChangeStatus()
            self.controls.enableFileSave(changeStatus)

        # spectrum has changed
        elif event == 'onSpectrumChanged':
            specLength = self.document.docData.getSpectrumLength()
            specRange = self.document.docData.getSpectrumXRange()
            peakLength = self.document.docData.getPeaklistLength()
            processing = self.document.docData.getProcessingStatus()

            self.controls.setStatusBar('spectrum', (specLength, specRange))
            self.controls.enableFileSave(True)
            self.controls.enablePrintingTools(peaklist=bool(peakLength), spectrum=bool(specLength))
            self.controls.enableSpectrumTools(bool(specLength), bool(peakLength))
            self.controls.enableProcessingUndo(processing)
            self.controls.enableProcessingTools(spectrum=bool(specLength), peaklist=bool(peakLength))

            self.document.mSpec.updatePeaklistData()
            self.document.mSpec.updateSpectrumData()
            self.document.mSpec.refreshCanvas()

        # peaklist has changed
        elif event == 'onPeaklistChanged':
            module = self.document.getCurrentModule()
            specLength = self.document.docData.getSpectrumLength()
            peakLength = self.document.docData.getPeaklistLength()
            moduleData = self.document.docData.getDataStatus(module)
            matchStatus = self.document.docData.getMatchStatus(module)
            processing = self.document.docData.getProcessingStatus()
            undoStatus = self.document.docData.getPeaklistUndoStatus()

            self.controls.setStatusBar('peaklist', peakLength)
            self.controls.enableFileSave(True)
            self.controls.enablePrintingTools(spectrum=bool(specLength), peaklist=bool(peakLength))
            self.controls.enableSpectrumTools(bool(specLength), bool(peakLength))
            self.controls.enablePeaklistTools(bool(peakLength))
            self.controls.enableProcessingUndo(processing)
            self.controls.enableProcessingTools(spectrum=bool(specLength), peaklist=bool(peakLength))
            self.controls.enableMatchingTools(module, bool(peakLength)*moduleData, matchStatus)
            self.controls.enablePeaklistUndoRedo(undoStatus[0], undoStatus[1])

            if value in ('processing', 'specAdd', 'specDelete'):
                self.document.mPeak.updatePeakList()
            elif value == 'import':
              self.document.mPeak.updatePeakList()
              self.document.mSpec.updatePeaklistData()
              self.document.mSpec.refreshCanvas(rescale=True)
            else:
              self.document.mPeak.updatePeakList()
              self.document.mSpec.updatePeaklistData()
              self.document.mSpec.refreshCanvas()

        # module data matched to peaklist
        elif event == 'onPeaklistMatched':
            module = self.document.getCurrentModule()
            matchStatus = self.document.docData.getMatchStatus(module)
            self.controls.enableMatchingTools(module, True, matchStatus)

        # sequence has changed
        elif event == 'onSequenceChanged':
            seqLength = self.document.docData.getSequenceLength()
            mods = self.document.docData.getModificationsCount()
            self.controls.setStatusBar('sequence', (seqLength, mods))
            self.controls.enableFileSave(True)
            self.controls.enableSequenceTools(bool(seqLength))

        # new data in module
        elif event == 'onModuleReset':
            peaklistData = self.document.docData.getDataStatus('mPeak')
            matchStatus = self.document.docData.getDataStatus(value)
            self.controls.enableMatchingTools(value, peaklistData*matchStatus, False)

        # update file history in main menu
        elif event == 'updateFileHistory':
            self.controls.updateFileHistory(value)

        # send peaklist data to mComp
        elif event == 'sendPeaklist':
            self.document.mComp.addPeaklist(value[0], value[1])
    # ----
