# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main application error handler.

# load libs
import sys
import traceback
import wx
import string


class errorHandler:
    """ Main application error handler. """

    # ----
    def __init__(self, parent, showError, debugmode):

        sys.excepthook = self.onException
        self.parent = parent
        self.showError = showError
        self.debugMode = debugmode
    # ----


    # ----
    def onException(self, type, value, tb):
        """ Catch any exception in the application. """

        # show exception in console
        if self.showError or self.debugMode:
            sys.__excepthook__(type, value, tb)

        # stop if debugmode
        if self.debugMode:
            return

        # get exception
        exceptionText = traceback.format_exception(type, value, tb)
        exceptionText = string.join(exceptionText)

        # write exception to log file
        self.writeExceptionToLog(exceptionText)

        # raise dialog
        dlg = wx.MessageDialog(self.parent, \
            "An application error has occurred and future operations may not be safe anymore!\n\n"
            "To help to improve the application please send the generated 'errors.log' file\n"
            "with a short error description to my e-mail address: mmass@biographics.cz.\n"
            "Thank you.\n\n"
            "Would you like to terminate the application?", \
            "Application Error", wx.YES_NO|wx.ICON_ERROR|wx.NO_DEFAULT)
        response = dlg.ShowModal()
        dlg.Destroy()

        # destroy application if YES button pressed
        if response == wx.ID_YES:
            try:
                self.parent.Destroy()
            except:
                sys.exit()
    # ----


    # ----
    def writeExceptionToLog(self, exceptionText):
        """ Write exception to log file. """

        try:
            message = exceptionText+'------\n'
            logFile = file('errors.log', 'a')
            logFile.write(message.encode("utf-8"))
            logFile.close()
        except IOError:
            pass
    # ----
