# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import wx

# load modules
from ids import *
import config
import mwx
import images
import mspy
import doc

from dlg_notation import dlgNotation


# PEAKLIST PANEL
# --------------

class panelPeaklist(wx.Panel):
    """Make peaklist panel."""
    
    def __init__(self, parent):
        wx.Panel.__init__(self, parent, -1, size=(150, -1), style=wx.NO_FULL_REPAINT_ON_RESIZE)
        
        self.parent = parent
        
        self.currentDocument = None
        self.peakListMap = None
        self.selectedPeak = None
        
        # make GUI
        self.makeGUI()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        # make peaklist list
        self.makePeakList()
        
        # init editing panel
        editor = self.makeEditor()
        
        # init lower toolbar
        toolbar = self.makeToolbar()
        
        # pack gui elements
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(self.peakList, 1, wx.EXPAND|wx.ALL, mwx.LISTCTRL_NO_SPACE)
        self.mainSizer.Add(editor, 0, wx.EXPAND)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND)
        
        # hide peak editor
        self.mainSizer.Hide(1)
        
        # fit layout
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
    # ----
    
    
    def makeToolbar(self):
        """Make bottom toolbar."""
        
        # init toolbar panel
        panel = mwx.bgrPanel(self, -1, images.lib['bgrBottombar'], size=(-1, mwx.BOTTOMBAR_HEIGHT))
        
        self.addPeak_butt = wx.BitmapButton(panel, -1, images.lib['peaklistAdd'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.addPeak_butt.SetToolTip(wx.ToolTip("Add peak manually..."))
        self.addPeak_butt.Bind(wx.EVT_BUTTON, self.onAdd)
        
        self.deletePeak_butt = wx.BitmapButton(panel, -1, images.lib['peaklistDelete'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.deletePeak_butt.SetToolTip(wx.ToolTip("Remove peaks..."))
        self.deletePeak_butt.Bind(wx.EVT_BUTTON, self.onDelete)
        
        self.annotatePeak_butt = wx.BitmapButton(panel, -1, images.lib['peaklistAnnotate'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.annotatePeak_butt.SetToolTip(wx.ToolTip("Annotate peak..."))
        self.annotatePeak_butt.Bind(wx.EVT_BUTTON, self.onAnnotate)
        
        self.editPeak_butt = wx.BitmapButton(panel, -1, images.lib['peaklistEditorOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.editPeak_butt.SetToolTip(wx.ToolTip("Show / hide peak editor"))
        self.editPeak_butt.Bind(wx.EVT_BUTTON, self.onEdit)
        
        self.peaksCount = wx.StaticText(panel, -1, "")
        self.peaksCount.SetFont(wx.SMALL_FONT)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.BOTTOMBAR_LSPACE)
        sizer.Add(self.addPeak_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(self.deletePeak_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.annotatePeak_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.editPeak_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.AddSpacer(10)
        sizer.Add(self.peaksCount, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        sizer.AddSpacer(mwx.BOTTOMBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makePeakList(self):
        """Make peaklist list."""
        
        # init peaklist
        self.peakList = mwx.sortListCtrl(self, -1, size=(191, -1), style=mwx.LISTCTRL_STYLE_MULTI)
        self.peakList.SetFont(wx.SMALL_FONT)
        self.peakList.setSecondarySortColumn(0)
        self.peakList.setAltColour(mwx.LISTCTRL_ALTCOLOUR)
        
        # set events
        self.peakList.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
        self.peakList.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onItemActivated)
        self.peakList.Bind(wx.EVT_KEY_DOWN, self.onListKey)
        
        # make columns
        self.peakList.InsertColumn(0, "m/z", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(1, "int.", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(2, "r. int.", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(3, "z", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(4, "s/n", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(5, "base", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(6, "fwhm", wx.LIST_FORMAT_RIGHT)
        self.peakList.InsertColumn(7, "resol.", wx.LIST_FORMAT_RIGHT)
        
        # set column widths
        for col, width in enumerate((90,80,55,30,50,60,60,60)):
            self.peakList.SetColumnWidth(col, width)
        
        # set DnD
        dropTarget = fileDropTarget(self.parent.onDocumentDropping)
        self.peakList.SetDropTarget(dropTarget)
    # ----
    
    
    def makeEditor(self):
        """Make panel for peak editing."""
        
        # init panel
        panel = mwx.bgrPanel(self, -1, images.lib['bgrPeakEditor'])
        
        # make elements
        peakMZ_label = wx.StaticText(panel, -1, "m/z:")
        self.peakMZ_value = wx.TextCtrl(panel, -1, '', size=(80, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('floatPos'))
        peakMZ_label.SetFont(wx.SMALL_FONT)
        self.peakMZ_value.SetFont(wx.SMALL_FONT)
        
        peakIntensity_label = wx.StaticText(panel, -1, "a.i.:")
        self.peakIntensity_value = wx.TextCtrl(panel, -1, '', size=(80, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('floatPos'))
        peakIntensity_label.SetFont(wx.SMALL_FONT)
        self.peakIntensity_value.SetFont(wx.SMALL_FONT)
        
        peakCharge_label = wx.StaticText(panel, -1, "charge:")
        self.peakCharge_value = wx.TextCtrl(panel, -1, '', size=(80, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('int'))
        peakCharge_label.SetFont(wx.SMALL_FONT)
        self.peakCharge_value.SetFont(wx.SMALL_FONT)
        
        peakSN_label = wx.StaticText(panel, -1, "s/n:")
        self.peakSN_value = wx.TextCtrl(panel, -1, '', size=(80, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('floatPos'))
        peakSN_label.SetFont(wx.SMALL_FONT)
        self.peakSN_value.SetFont(wx.SMALL_FONT)
        
        peakBaseline_label = wx.StaticText(panel, -1, "base:")
        self.peakBaseline_value = wx.TextCtrl(panel, -1, '', size=(80, mwx.SMALL_TEXTCTRL_HEIGHT), style=wx.TE_PROCESS_ENTER, validator=mwx.validator('floatPos'))
        peakBaseline_label.SetFont(wx.SMALL_FONT)
        self.peakBaseline_value.SetFont(wx.SMALL_FONT)
        
        self.peakMonoisotopic_check = wx.CheckBox(panel, -1, "monoisotopic")
        self.peakMonoisotopic_check.SetValue(True)
        self.peakMonoisotopic_check.SetFont(wx.SMALL_FONT)
        
        self.peakAdd_butt = wx.Button(panel, -1, "Add", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.peakAdd_butt.SetFont(wx.SMALL_FONT)
        self.peakAdd_butt.Bind(wx.EVT_BUTTON, self.onAddPeak)
        
        self.peakReplace_butt = wx.Button(panel, -1, "Replace", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.peakReplace_butt.SetFont(wx.SMALL_FONT)
        self.peakReplace_butt.Bind(wx.EVT_BUTTON, self.onReplacePeak)
        self.peakReplace_butt.Enable(False)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(peakMZ_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakMZ_value, (0,1), flag=wx.EXPAND)
        grid.Add(peakIntensity_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakIntensity_value, (1,1), flag=wx.EXPAND)
        grid.Add(peakCharge_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakCharge_value, (2,1), flag=wx.EXPAND)
        grid.Add(peakSN_label, (3,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakSN_value, (3,1), flag=wx.EXPAND)
        grid.Add(peakBaseline_label, (4,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakBaseline_value, (4,1), flag=wx.EXPAND)
        grid.Add(self.peakMonoisotopic_check, (5,1), flag=wx.ALIGN_LEFT|wx.ALIGN_CENTER_VERTICAL)
        grid.AddGrowableCol(1)
        
        buttons = wx.BoxSizer(wx.HORIZONTAL)
        buttons.Add(self.peakAdd_butt, 0, wx.RIGHT, 10)
        buttons.Add(self.peakReplace_butt, 0)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.EXPAND|wx.ALIGN_CENTER|wx.ALL, 10)
        mainSizer.Add(buttons, 0, wx.ALIGN_CENTER|wx.RIGHT|wx.LEFT|wx.BOTTOM, 10)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def onItemSelected(self, evt):
        """Highlight selected peak in the spectrum and refresh peak editor."""
        
        evt.Skip()
        
        # get selected peak
        self.selectedPeak = evt.GetData()
        peak = self.currentDocument.spectrum.peaklist[self.selectedPeak]
        
        # highlight point in the spectrum
        selected = self.peakList.getSelected()
        if len(selected) == 1:
            self.parent.updateMassPoints([peak.mz])
        
        # show current peak in editing panel
        self.updatePeakEditor(peak)
    # ----
    
    
    def onItemActivated(self, evt):
        """Create annotation for activated peak."""
        self.onAnnotate()
    # ----
    
    
    def onListKey(self, evt):
        """Key pressed."""
        
        # get key
        key = evt.GetKeyCode()
        
        # select all
        if key == 65 and evt.CmdDown():
            for x in range(self.peakList.GetItemCount()):
                self.peakList.SetItemState(x, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
        
        # copy
        elif key == 67 and evt.CmdDown():
            self.copyToClipboard()
        
        # delete
        elif key==wx.WXK_DELETE or (key==wx.WXK_BACK and evt.CmdDown()):
            self.onDeleteSelected()
            
        # other keys
        else:
            evt.Skip()
    # ----
    
    
    def onAdd(self, evt):
        """Plus button pressed."""
        
        # check document
        if self.currentDocument == None:
            wx.Bell()
            return
            
        # empty data in peak editor
        self.updatePeakEditor(None)
        
        # show editing panel if not shown
        if not self.mainSizer.IsShown(1):
            self.editPeak_butt.SetBitmapLabel(images.lib['peaklistEditorOn'])
            self.mainSizer.Show(1)
            self.Layout()
    # ----
    
    
    def onDelete(self, evt):
        """Minus button pressed."""
        
        # popup menu
        menuDeleteSelectedID = wx.NewId()
        menuDeleteByThresholdID = wx.NewId()
        menuDeleteAllID = wx.NewId()
        
        menu = wx.Menu()
        menu.Append(menuDeleteSelectedID, "Delete Selected")
        menu.Append(menuDeleteByThresholdID, "Delete by Threshold...")
        menu.AppendSeparator()
        menu.Append(menuDeleteAllID, "Delete All")
        
        self.Bind(wx.EVT_MENU, self.onDeleteSelected, id=menuDeleteSelectedID)
        self.Bind(wx.EVT_MENU, self.onDeleteByThreshold, id=menuDeleteByThresholdID)
        self.Bind(wx.EVT_MENU, self.onDeleteAll, id=menuDeleteAllID)
        
        self.PopupMenu(menu)
        menu.Destroy()
        self.SetFocus()
    # ----
    
    
    def onAnnotate(self, evt=None):
        """Annotate selected peak."""
        
        # check document and selected peak
        if self.currentDocument == None or self.selectedPeak == None:
            wx.Bell()
            return
        
        # make annotation
        peak = self.currentDocument.spectrum.peaklist[self.selectedPeak]
        annot = doc.annotation(label='', mz=peak.mz, intensity=peak.intensity, baseline=peak.baseline, charge=peak.charge)
        
        # get annotation label
        dlg = dlgNotation(self.parent, annot, button='Add')
        if dlg.ShowModal() == wx.ID_OK:
            dlg.Destroy()
        else:
            dlg.Destroy()
            return
        
        # add annotation
        self.currentDocument.annotations.append(annot)
        self.currentDocument.sortAnnotations()
        self.parent.onDocumentChanged(('annotations'))
    # ----
    
    
    def onEdit(self, evt):
        """Show / hide peak editing panel."""
        
        # hide peak editing panel
        if self.mainSizer.IsShown(1):
            self.editPeak_butt.SetBitmapLabel(images.lib['peaklistEditorOff'])
            self.mainSizer.Hide(1)
            self.Layout()
        
        # show peak editind panel
        else:
            self.editPeak_butt.SetBitmapLabel(images.lib['peaklistEditorOn'])
            self.mainSizer.Show(1)
            self.Layout()
    # ----
    
    
    def onDeleteSelected(self, evt=None):
        """Delete selected peaks."""
        
        # check document
        if self.currentDocument == None:
            wx.Bell()
            return
        
        # get selected peaks
        indexes = []
        for item in self.peakList.getSelected():
            indexes.append(self.peakList.GetItemData(item))
        
        # update data
        self.currentDocument.backup(('spectrum'))
        self.currentDocument.spectrum.peaklist.delete(indexes)
        self.parent.onDocumentChanged(('spectrum'))
    # ---
    
    
    def onDeleteByThreshold(self, evt=None):
        """Delete peaks by selected threshold."""
        
        # check document
        if self.currentDocument == None:
            wx.Bell()
            return
        
        # show threshold dialog
        dlg = dlgThreshold(self.parent)
        if dlg.ShowModal() == wx.ID_OK:
            threshold, thresholdType = dlg.getData()
            dlg.Destroy()
            
            indexes = []
            
            # use absolute intensity
            if thresholdType == 'Absolute int.':
                for i, peak in enumerate(self.currentDocument.spectrum.peaklist):
                    if peak.intensity - peak.baseline < threshold:
                        indexes.append(i)
            
            # use relative intensity
            elif thresholdType == 'Relative int.':
                threshold /= 100
                for i, peak in enumerate(self.currentDocument.spectrum.peaklist):
                    if peak.relIntensity < threshold:
                        indexes.append(i)
            
            # use s/n
            elif thresholdType == 's/n':
                for i, peak in enumerate(self.currentDocument.spectrum.peaklist):
                    if peak.sn != None and peak.sn < threshold:
                        indexes.append(i)
            
            # use m/z
            elif thresholdType == 'm/z':
                for i, peak in enumerate(self.currentDocument.spectrum.peaklist):
                    if peak.mz < threshold:
                        indexes.append(i)
            
            # delete peaks
            if indexes:
                self.currentDocument.backup(('spectrum'))
                self.currentDocument.spectrum.peaklist.delete(indexes)
                self.parent.onDocumentChanged(('spectrum'))
        
        else:
            dlg.Destroy()
            return
    # ----
    
    
    def onDeleteAll(self, evt=None):
        """Delete all peaks."""
        
        # check document
        if self.currentDocument == None:
            wx.Bell()
            return
        
        # empty peaklist
        self.currentDocument.backup(('spectrum'))
        self.currentDocument.spectrum.peaklist.empty()
        self.parent.onDocumentChanged(('spectrum'))
    # ----
    
    
    def onAddPeak(self, evt):
        """Get peak data and add peak."""
        
        # check document
        if self.currentDocument == None:
            wx.Bell()
            return
        
        # add new peak
        peak = self.getPeakEditorData()
        if peak:
            self.currentDocument.backup(('spectrum'))
            self.currentDocument.spectrum.peaklist.append(peak)
            self.parent.onDocumentChanged(('spectrum'))
    # ----
    
    
    def onReplacePeak(self, evt):
        """Get peak data and refresh current peak."""
        
        # check selection
        if self.selectedPeak == None:
            wx.Bell()
            return
        
        # set new data to peak
        peak = self.getPeakEditorData()
        if peak:
            self.currentDocument.backup(('spectrum'))
            self.currentDocument.spectrum.peaklist[self.selectedPeak] = peak
            self.currentDocument.spectrum.peaklist.refresh()
            self.parent.onDocumentChanged(('spectrum'))
    # ----
    
    
    def setData(self, document):
        """Set new document."""
        
        # set new data
        self.currentDocument = document
        
        # update peaklist
        self.updatePeakList()
    # ----
    
    
    def updatePeakList(self):
        """Refresh peaklist."""
        
        # clear previous data
        self.peakList.DeleteAllItems()
        self.selectedPeak = None
        self.updatePeakEditor(None)
        
        # check data
        if not self.currentDocument:
            self.peaksCount.SetLabel('')
            self.peakListMap = None
            self.peakList.setDataMap(None)
            return
        
        # update peaks count
        count = '%d' % len(self.currentDocument.spectrum.peaklist)
        if count != '0':
            self.peaksCount.SetLabel(count)
        else:
            self.peaksCount.SetLabel('')
        
        # set new data map
        self.peakListMap = []
        for peak in self.currentDocument.spectrum.peaklist:
            self.peakListMap.append((
                peak.mz,
                peak.intensity-peak.baseline,
                peak.relIntensity*100,
                peak.charge,
                peak.sn,
                peak.baseline,
                peak.fwhm,
                peak.resolution()
            ))
        self.peakList.setDataMap(self.peakListMap)
        
        # add new data
        for row, item in enumerate(self.peakListMap):
            self.updatePeakListItem(row, item, insert=True)
            self.peakList.SetItemData(row, row)
        
        # sort data
        self.peakList.sort()
        
        # scroll top
        if len(self.currentDocument.spectrum.peaklist)>0:
            self.peakList.EnsureVisible(0)
    # ----
    
    
    def updatePeakListItem(self, row, item, insert=False):
        """Refresh item data in the list."""
        
        # set formats
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        intFormat = '%0.' + `config.main['intDigits']` + 'f'
        fwhmFormat = '%0.' + `max(config.main['mzDigits'],3)` + 'f'
        
        # format data
        mz = mzFormat % (item[0])
        intensity = intFormat % (item[1])
        
        relint = ""
        if item[2]:
            relint = '%0.2f' % (item[2])
        
        charge = ""
        if item[3] != None:
            charge = str(item[3])
        
        sn = ""
        if item[4] != None:
            sn = '%0.1f' % (item[4])
            
        base = intFormat % (item[5])
        
        fwhm = ""
        if item[6]:
            fwhm = fwhmFormat % (item[6])
        
        resolution = ""
        if item[7]:
            resolution = '%0.0f' % (item[7])
        
        # add data
        if insert:
            self.peakList.InsertStringItem(row, mz)
        else:
            self.peakList.SetStringItem(row, 0, mz)
        
        self.peakList.SetStringItem(row, 1, intensity)
        self.peakList.SetStringItem(row, 2, relint)
        self.peakList.SetStringItem(row, 3, charge)
        self.peakList.SetStringItem(row, 4, sn)
        self.peakList.SetStringItem(row, 5, base)
        self.peakList.SetStringItem(row, 6, fwhm)
        self.peakList.SetStringItem(row, 7, resolution)
    # ----
    
    
    def updatePeakEditor(self, peak=None):
        """Refresh peak editing panel."""
        
        # empty data
        self.peakMZ_value.SetValue('')
        self.peakIntensity_value.SetValue('')
        self.peakCharge_value.SetValue('')
        self.peakSN_value.SetValue('')
        self.peakBaseline_value.SetValue('')
        self.peakMonoisotopic_check.SetValue(True)
        self.peakReplace_butt.Enable(False)
        
        # set peak data
        if peak:
            self.peakMZ_value.SetValue(str(peak.mz))
            self.peakIntensity_value.SetValue(str(peak.intensity))
            if peak.charge:
                self.peakCharge_value.SetValue(str(peak.charge))
            if peak.sn:
                self.peakSN_value.SetValue(str(peak.sn))
            self.peakBaseline_value.SetValue(str(peak.baseline))
            if peak.isotope == 0:
                self.peakMonoisotopic_check.SetValue(True)
            else:
                self.peakMonoisotopic_check.SetValue(False)
            self.peakReplace_butt.Enable(True)
    # ----
    
    
    def getPeakEditorData(self):
        """Get data of edited peak."""
        
        try:
            # get data
            mz = self.peakMZ_value.GetValue()
            intensity = self.peakIntensity_value.GetValue()
            baseline = self.peakBaseline_value.GetValue()
            sn = self.peakSN_value.GetValue()
            charge = self.peakCharge_value.GetValue()
            monoisotope = self.peakMonoisotopic_check.GetValue()
            
            # format data
            mz = float(mz)
            intensity = float(intensity)
            
            if baseline:
                baseline = float(baseline)
            else:
                baseline = 0.
            
            if sn:
                sn = float(sn)
            else:
                sn = None
            
            if charge:
                charge = int(charge)
            else:
                charge = None
            
            if monoisotope:
                isotope = 0
            else:
                isotope = None
            
            # make peak
            peak = mspy.peak(mz=mz, intensity=intensity, baseline=baseline, sn=sn, charge=charge, isotope=isotope)
            return peak
        
        except:
            wx.Bell()
            return False
    # ----
    
    
    def getSelectedPeaks(self):
        """Get selected peaks."""
        
        # get peaklist
        peaklist = []
        for item in self.peakList.getSelected():
            peak = self.currentDocument.spectrum.peaklist[self.peakList.GetItemData(item)]
            peaklist.append(peak)
        
        return peaklist
    # ----
    
    
    def copyToClipboard(self):
        """Copy current selection to clipboard."""
        
        # get selected
        selection = self.peakList.getSelected()
        if not selection:
            return
        
        # show copy dialog
        dlg = dlgCopy(self.parent)
        if dlg.ShowModal() == wx.ID_OK:
            columns = dlg.getData()
            dlg.Destroy()
            
            # get data
            buff = ''
            for row in selection:
                line = ''
                for col in columns:
                    item = self.peakList.GetItem(row, col)
                    line += item.GetText() + '\t'
                buff += '%s\n' % (line.rstrip())
            
            # make text object for data
            obj = wx.TextDataObject()
            obj.SetText(buff.rstrip())
            
            # paste to clipboard
            if wx.TheClipboard.Open():
                wx.TheClipboard.SetData(obj)
                wx.TheClipboard.Close()
        
        else:
            dlg.Destroy()
    # ----
    


class dlgThreshold(wx.Dialog):
    """Set threshold."""
    
    def __init__(self, parent):
        wx.Dialog.__init__(self, parent, -1, "Threshold", style=wx.DEFAULT_DIALOG_STYLE|wx.STAY_ON_TOP)
        
        self.parent = parent
        self.threshold = None
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        staticSizer = wx.StaticBoxSizer(wx.StaticBox(self, -1, ""), wx.VERTICAL)
        
        # make elements
        threshold_label = wx.StaticText(self, -1, "Minimal value:")
        self.threshold_value = wx.TextCtrl(self, -1, "", size=(120, -1), validator=mwx.validator('floatPos'))
        self.threshold_value.Bind(wx.EVT_TEXT, self.onChange)
        
        thresholdType_label = wx.StaticText(self, -1, "Threshold type:")
        choices=['Absolute int.', 'Relative int.', 's/n', 'm/z']
        self.thresholdType_combo = wx.ComboBox(self, -1, choices=choices, size=(120, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.thresholdType_combo.Select(0)
        
        cancel_butt = wx.Button(self, wx.ID_CANCEL, "Cancel")
        self.delete_butt = wx.Button(self, wx.ID_OK, "Delete")
        self.delete_butt.Bind(wx.EVT_BUTTON, self.onDelete)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(threshold_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.threshold_value, (0,1))
        grid.Add(thresholdType_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.thresholdType_combo, (1,1))
        
        staticSizer.Add(grid, 0, wx.ALL, 5)
        
        buttSizer = wx.BoxSizer(wx.HORIZONTAL)
        buttSizer.Add(cancel_butt, 0, wx.RIGHT, 15)
        buttSizer.Add(self.delete_butt, 0)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(staticSizer, 0, wx.CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        mainSizer.Add(buttSizer, 0, wx.CENTER|wx.LEFT|wx.RIGHT|wx.BOTTOM, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def onChange(self, evt):
        """Check data."""
        
        # get data
        try:
            self.threshold = float(self.threshold_value.GetValue())
        except:
            self.threshold = None
    # ----
    
    
    def onDelete(self, evt):
        """Delete."""
        
        # check value and end
        if self.threshold != None:
            self.EndModal(wx.ID_OK)
        else:
            wx.Bell()
    # ----
    
    
    def getData(self):
        """Return values."""
        return self.threshold, self.thresholdType_combo.GetValue()
    # ----
    


class dlgCopy(wx.Dialog):
    """Set coumns to copy."""
    
    def __init__(self, parent):
        wx.Dialog.__init__(self, parent, -1, "Select Columns", style=wx.DEFAULT_DIALOG_STYLE|wx.STAY_ON_TOP)
        
        self.parent = parent
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        staticSizer = wx.StaticBoxSizer(wx.StaticBox(self, -1, ""), wx.VERTICAL)
        
        # make elements
        self.colMZ_check = wx.CheckBox(self, -1, "M/Z")
        self.colIntensity_check = wx.CheckBox(self, -1, "Intensity")
        self.colRelIntensity_check = wx.CheckBox(self, -1, "Relative intensity")
        self.colCharge_check = wx.CheckBox(self, -1, "Charge")
        self.colSN_check = wx.CheckBox(self, -1, "S/N ratio")
        self.colBaseline_check = wx.CheckBox(self, -1, "Baseline")
        self.colFWHM_check = wx.CheckBox(self, -1, "FWHM")
        self.colResolution_check = wx.CheckBox(self, -1, "Resolution")
        
        self.colMZ_check.SetValue(True)
        self.colIntensity_check.SetValue(True)
        
        self.copy_butt = wx.Button(self, wx.ID_OK, "Copy")
        
        # pack elements
        itemSizer = wx.BoxSizer(wx.VERTICAL)
        itemSizer.Add(self.colMZ_check, 0, wx.ALIGN_LEFT, 0)
        itemSizer.Add(self.colIntensity_check, 0, wx.ALIGN_LEFT|wx.TOP|wx.BOTTOM, 5)
        itemSizer.Add(self.colRelIntensity_check, 0, wx.ALIGN_LEFT|wx.BOTTOM, 5)
        itemSizer.Add(self.colCharge_check, 0, wx.ALIGN_LEFT|wx.BOTTOM, 5)
        itemSizer.Add(self.colSN_check, 0, wx.ALIGN_LEFT|wx.BOTTOM, 5)
        itemSizer.Add(self.colBaseline_check, 0, wx.ALIGN_LEFT|wx.BOTTOM, 5)
        itemSizer.Add(self.colFWHM_check, 0, wx.ALIGN_LEFT|wx.BOTTOM, 5)
        itemSizer.Add(self.colResolution_check, 0, wx.ALIGN_LEFT, 0)
        
        staticSizer.Add(itemSizer, 0, wx.ALL, 5)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(staticSizer, 0, wx.CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        mainSizer.Add(self.copy_butt, 0, wx.CENTER|wx.LEFT|wx.RIGHT|wx.BOTTOM, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def getData(self):
        """Delete peaks below threshold"""
        
        # get data
        columns = []
        if self.colMZ_check.IsChecked():
            columns.append(0)
        if self.colIntensity_check.IsChecked():
            columns.append(1)
        if self.colRelIntensity_check.IsChecked():
            columns.append(2)
        if self.colCharge_check.IsChecked():
            columns.append(3)
        if self.colSN_check.IsChecked():
            columns.append(4)
        if self.colBaseline_check.IsChecked():
            columns.append(5)
        if self.colFWHM_check.IsChecked():
            columns.append(6)
        if self.colResolution_check.IsChecked():
            columns.append(7)
        
        return columns
    # ----
    


class fileDropTarget(wx.FileDropTarget):
    """Generic drop target for files."""
    
    def __init__(self, fce):
        wx.FileDropTarget.__init__(self)
        self.fce = fce
    # ----
    
    
    def OnDropFiles(self, x, y, paths):
        """Open dropped files."""
        self.fce(paths=paths)
    # ----
    

    