# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import wx
import numpy

# load modules
from ids import *
import config
import mwx
import images
import mspy
import mspy.plot


# SPECTRUM PANEL WITH CANVAS AND TOOLBAR
# --------------------------------------

class panelSpectrum(wx.Panel):
    """Make spectrum panel."""
    
    def __init__(self, parent, documents):
        wx.Panel.__init__(self, parent, -1, size=(100, 100), style=wx.NO_FULL_REPAINT_ON_RESIZE)
        
        self.parent = parent
        
        self.documents = documents
        self.currentDocument = None
        self.currentTmpSpectrum = None
        self.currentNotationMarks = None
        self.currentTool = 'measure'
        
        # init container
        self.container = mspy.plot.container()
        obj = mspy.plot.points([], showInGel=False)
        self.container.append(obj)
        obj = mspy.plot.points([], showInGel=False)
        self.container.append(obj)
        
        # init peak picking module
        self.peakPicking = mspy.peakPicking()
        
        # make GUI
        self.makeGUI()
        
        # select default tool
        self.setCurrentTool(self.currentTool)
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        # init spectrum canvas
        self.makeSpectrumCanvas()
        
        # init toolbar
        toolbar = self.makeToolbar()
        
        # pack gui elements
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(self.spectrumCanvas, 1, wx.EXPAND)
        sizer.Add(toolbar, 0, wx.EXPAND)
        
        # fit layout
        sizer.Fit(self)
        self.SetSizer(sizer)
    # ----
    
    
    def makeSpectrumCanvas(self):
        """Make plot canvas and set defalt parameters."""
        
        # init canvas
        self.spectrumCanvas = mspy.plot.canvas(self, style=mwx.PLOTCANVAS_STYLE_PANEL)
        self.spectrumCanvas.draw(self.container)
        
        # set default params
        self.spectrumCanvas.setProperties(xLabel=config.spectrum['xLabel'])
        self.spectrumCanvas.setProperties(yLabel=config.spectrum['yLabel'])
        self.spectrumCanvas.setProperties(showZero=True)
        self.spectrumCanvas.setProperties(showGrid=config.spectrum['showGrid'])
        self.spectrumCanvas.setProperties(showLegend=config.spectrum['showLegend'])
        self.spectrumCanvas.setProperties(showPosBar=config.spectrum['showPosBar'])
        self.spectrumCanvas.setProperties(posBarHeight=config.spectrum['barHeight'])
        self.spectrumCanvas.setProperties(showIntBar=config.spectrum['showIntBar'])
        self.spectrumCanvas.setProperties(intBarHeight=config.spectrum['barHeight'])
        self.spectrumCanvas.setProperties(showGel=config.spectrum['showGel'])
        self.spectrumCanvas.setProperties(gelHeight=config.spectrum['gelHeight'])
        self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
        self.spectrumCanvas.setProperties(showCurDistance=config.spectrum['showTracker'])
        self.spectrumCanvas.setProperties(showCurXPos=True)
        self.spectrumCanvas.setProperties(showCurYPos=True)
        self.spectrumCanvas.setProperties(showCurImage=config.spectrum['showCursorImage'])
        self.spectrumCanvas.setProperties(checkLimits=config.spectrum['checkLimits'])
        self.spectrumCanvas.setProperties(autoScaleY=config.spectrum['autoscale'])
        self.spectrumCanvas.setProperties(overlapLabels=config.spectrum['overlapLabels'])
        self.spectrumCanvas.setProperties(xPosDigits=config.main['mzDigits'])
        self.spectrumCanvas.setProperties(yPosDigits=config.main['intDigits'])
        self.spectrumCanvas.setProperties(distanceDigits=config.main['mzDigits'])
        self.spectrumCanvas.setProperties(reverseDrawing=True)
        
        axisFont = wx.Font(config.spectrum['axisFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        self.spectrumCanvas.setProperties(axisFont=axisFont)
        
        # set events
        self.spectrumCanvas.canvas.Bind(wx.EVT_MOTION, self.onCanvasMMotion)
        self.spectrumCanvas.canvas.Bind(wx.EVT_LEFT_UP, self.onCanvasLMU)
        
        # set DnD
        dropTarget = fileDropTarget(self.parent.onDocumentDropping)
        self.spectrumCanvas.canvas.SetDropTarget(dropTarget)
    # ----
    
    
    def makeToolbar(self):
        """Make bottom toolbar."""
        
        # init toolbar panel
        panel = mwx.bgrPanel(self, -1, images.lib['bgrBottombar'], size=(-1, mwx.BOTTOMBAR_HEIGHT))
        
        # make canvas toolset
        self.canvasParameters_butt = wx.BitmapButton(panel, ID_viewCanvasProperties, images.lib['spectrumParams'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.canvasParameters_butt.SetToolTip(wx.ToolTip("Canvas properties..."))
        self.canvasParameters_butt.Bind(wx.EVT_BUTTON, self.onCanvasProperties)
        
        image = (images.lib['spectrumLabelsOff'], images.lib['spectrumLabelsOn'])[config.spectrum['showLabels']]
        self.showLabels_butt = wx.BitmapButton(panel, ID_viewLabels, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.showLabels_butt.SetToolTip(wx.ToolTip("Show / hide labels"))
        self.showLabels_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumTicksOff'], images.lib['spectrumTicksOn'])[config.spectrum['showTicks']]
        self.showTicks_butt = wx.BitmapButton(panel, ID_viewTicks, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.showTicks_butt.SetToolTip(wx.ToolTip("Show / hide ticks"))
        self.showTicks_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumNotationMarksOff'], images.lib['spectrumNotationMarksOn'])[config.spectrum['showNotationMarks']]
        self.showNotationMarks_butt = wx.BitmapButton(panel, ID_viewNotationMarks, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.showNotationMarks_butt.SetToolTip(wx.ToolTip("Show / hide annotation marks"))
        self.showNotationMarks_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumLabelAngleOff'], images.lib['spectrumLabelAngleOn'])[bool(config.spectrum['labelAngle'])]
        self.labelAngle_butt = wx.BitmapButton(panel, ID_viewLabelAngle, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.labelAngle_butt.SetToolTip(wx.ToolTip("Labels orientation"))
        self.labelAngle_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumGelOff'], images.lib['spectrumGelOn'])[config.spectrum['showGel']]
        self.showGel_butt = wx.BitmapButton(panel, ID_viewGel, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.showGel_butt.SetToolTip(wx.ToolTip("Show / hide gel"))
        self.showGel_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumTrackerOff'], images.lib['spectrumTrackerOn'])[config.spectrum['showTracker']]
        self.showTracker_butt = wx.BitmapButton(panel, ID_viewTracker, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.showTracker_butt.SetToolTip(wx.ToolTip("Show / hide cursor tracker"))
        self.showTracker_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumAutoscaleOff'], images.lib['spectrumAutoscaleOn'])[config.spectrum['autoscale']]
        self.autoscale_butt = wx.BitmapButton(panel, ID_viewAutoscale, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.autoscale_butt.SetToolTip(wx.ToolTip("Autoscale intensity"))
        self.autoscale_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        image = (images.lib['spectrumNormalizeOff'], images.lib['spectrumNormalizeOn'])[config.spectrum['normalize']]
        self.normalize_butt = wx.BitmapButton(panel, ID_viewNormalize, image, size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.normalize_butt.SetToolTip(wx.ToolTip("Normalize intensity"))
        self.normalize_butt.Bind(wx.EVT_BUTTON, self.parent.onView)
        
        # make processing toolset
        self.toolsParameters_butt = wx.BitmapButton(panel, -1, images.lib['spectrumParams'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsParameters_butt.SetToolTip(wx.ToolTip("Tools properties..."))
        self.toolsParameters_butt.Bind(wx.EVT_BUTTON, self.onToolsProperties)
        
        self.toolsLabelPeak_butt = wx.BitmapButton(panel, ID_spectrumLabelPeak, images.lib['spectrumLabelPeakOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsLabelPeak_butt.SetToolTip(wx.ToolTip("Label peak"))
        self.toolsLabelPeak_butt.Bind(wx.EVT_BUTTON, self.parent.onToolsSpectrum)
        
        self.toolsLabelPoint_butt = wx.BitmapButton(panel, ID_spectrumLabelPoint, images.lib['spectrumLabelPointOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsLabelPoint_butt.SetToolTip(wx.ToolTip("Label point"))
        self.toolsLabelPoint_butt.Bind(wx.EVT_BUTTON, self.parent.onToolsSpectrum)
        
        self.toolsDeleteLabel_butt = wx.BitmapButton(panel, ID_spectrumDeleteLabel, images.lib['spectrumDeleteLabelOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsDeleteLabel_butt.SetToolTip(wx.ToolTip("Delete label"))
        self.toolsDeleteLabel_butt.Bind(wx.EVT_BUTTON, self.parent.onToolsSpectrum)
        
        self.toolsMeasure_butt = wx.BitmapButton(panel, ID_spectrumMeasure, images.lib['spectrumMeasureOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsMeasure_butt.SetToolTip(wx.ToolTip("Measure distances"))
        self.toolsMeasure_butt.Bind(wx.EVT_BUTTON, self.parent.onToolsSpectrum)
        
        self.toolsOffset_butt = wx.BitmapButton(panel, ID_spectrumOffset, images.lib['spectrumOffsetOff'], size=(mwx.BOTTOMBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.toolsOffset_butt.SetToolTip(wx.ToolTip("Offset spectrum"))
        self.toolsOffset_butt.Bind(wx.EVT_BUTTON, self.parent.onToolsSpectrum)
        
        # make cursor info
        self.cursorInfo = wx.StaticText(panel, -1, "")
        self.cursorInfo.SetFont(wx.SMALL_FONT)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.BOTTOMBAR_LSPACE)
        sizer.Add(self.canvasParameters_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(self.showLabels_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.showTicks_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.showNotationMarks_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.labelAngle_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.showGel_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.showTracker_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.autoscale_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.normalize_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.AddSpacer(20)
        sizer.Add(self.toolsParameters_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(self.toolsLabelPeak_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.toolsLabelPoint_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.toolsDeleteLabel_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.toolsMeasure_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.Add(self.toolsOffset_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        sizer.AddSpacer(20)
        sizer.Add(self.cursorInfo, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        sizer.AddSpacer(mwx.BOTTOMBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def onCanvasMMotion(self, evt):
        """Update cursor info on mouse motion."""
        
        # update cursor
        self.spectrumCanvas.onMMotion(evt)
        self.updateCursorInfo()
    # ----
    
    
    def onCanvasLMU(self, evt):
        """Process selected mouse fce."""
        
        # check document
        if self.currentDocument == None:
            evt.Skip()
            if self.currentTool != 'measure':
                wx.Bell()
            return
        
        # get cursor positions
        selection = self.spectrumCanvas.getSelectionBox()
        position = self.spectrumCanvas.getCursorPosition()
        distance = self.spectrumCanvas.getDistance()
        self.spectrumCanvas.onLMU(evt)
        
        # convert selection for flipped documents
        if selection and self.documents[self.currentDocument].flipped:
            y1 = -1 * selection[1]
            y2 = -1 * selection[3]
            selection = (selection[0], y2, selection[2], y1)
        
        # convert normalized selection to real values
        if selection and config.spectrum['normalize']:
            normalization = self.documents[self.currentDocument].spectrum.normalization()
            y1 = (selection[1] * normalization[0]) + normalization[1]
            y2 = (selection[3] * normalization[0]) + normalization[1]
            selection = (selection[0], y1, selection[2], y2)
        
        # convert selection for offset documents
        if selection and not config.spectrum['normalize']:
            x1 = selection[0] - self.documents[self.currentDocument].offset[0]
            x2 = selection[2] - self.documents[self.currentDocument].offset[0]
            y1 = selection[1] - self.documents[self.currentDocument].offset[1]
            y2 = selection[3] - self.documents[self.currentDocument].offset[1]
            selection = (x1, y1, x2, y2)
        
        # label peak
        if self.currentTool == 'labelpeak' and selection \
            and len(self.documents[self.currentDocument].spectrum.points):
            
            # get peak
            self.peakPicking.pickingHeight = config.processing['peakpicking']['pickingHeight']
            self.peakPicking.adaptiveNoise = config.processing['peakpicking']['adaptiveNoise']
            points = self.documents[self.currentDocument].spectrum.points
            peak = self.peakPicking.labelPeak(points, selection[0], selection[2])
            if peak:
                
                # set as monoisotopic
                if config.processing['peakpicking']['monoisotopic']:
                    peak.isotope = 0
                
                # update document
                self.documents[self.currentDocument].backup(('spectrum'))
                self.documents[self.currentDocument].spectrum.peaklist.append(peak)
                self.parent.onDocumentChanged(('spectrum'))
        
        # label point
        elif self.currentTool == 'labelpoint' and position \
            and len(self.documents[self.currentDocument].spectrum.points):
            
            # get peak
            self.peakPicking.adaptiveNoise = config.processing['peakpicking']['adaptiveNoise']
            points = self.documents[self.currentDocument].spectrum.points
            peak = self.peakPicking.labelPoint(points, position[0])
            if peak:
                
                # set as monoisotopic
                if config.processing['peakpicking']['monoisotopic']:
                    peak.isotope = 0
                
                # update document
                self.documents[self.currentDocument].backup(('spectrum'))
                self.documents[self.currentDocument].spectrum.peaklist.append(peak)
                self.parent.onDocumentChanged(('spectrum'))
        
        # delete peaks
        elif self.currentTool == 'deletelabel' and selection:
            
            # remove peaks
            indexes = []
            for x, peak in enumerate(self.documents[self.currentDocument].spectrum.peaklist):
                if (selection[0] < peak.mz < selection[2]) and (selection[1] < peak.intensity < selection[3]):
                    indexes.append(x)
            
            # update document
            if indexes:
                self.documents[self.currentDocument].backup(('spectrum'))
                self.documents[self.currentDocument].spectrum.peaklist.delete(indexes)
                self.parent.onDocumentChanged(('spectrum'))
        
        # offset spectrum
        elif self.currentTool == 'offset' and distance and distance != [0,0]:
            if not config.spectrum['normalize']:
                if self.documents[self.currentDocument].flipped:
                    self.documents[self.currentDocument].offset[1] -= distance[1]
                else:
                    self.documents[self.currentDocument].offset[1] += distance[1]
                self.updateSpectrumProperties(self.currentDocument)
            else:
                wx.Bell()
        
        else:
            evt.Skip()
    # ----
    
    
    def onCanvasProperties(self, evt=None):
        """Show canvas properties dialog."""
        
        # raise dialog
        dlg = dlgCanvasProperties(self.parent, self.updateCanvasProperties)
        dlg.ShowModal()
        dlg.Destroy()
    # ----
    
    
    def onToolsProperties(self, evt=None):
        """Show processing properties dialog."""
        
        # raise dialog
        dlg = dlgToolsProperties(self.parent)
        dlg.ShowModal()
        dlg.Destroy()
    # ----
    
    
    def setCurrentTool(self, tool):
        """Slect spectrum tool."""
        
        # set current tool
        self.currentTool = tool
        
        # set icons off
        self.toolsLabelPeak_butt.SetBitmapLabel(images.lib['spectrumLabelPeakOff'])
        self.toolsLabelPoint_butt.SetBitmapLabel(images.lib['spectrumLabelPointOff'])
        self.toolsDeleteLabel_butt.SetBitmapLabel(images.lib['spectrumDeleteLabelOff'])
        self.toolsMeasure_butt.SetBitmapLabel(images.lib['spectrumMeasureOff'])
        self.toolsOffset_butt.SetBitmapLabel(images.lib['spectrumOffsetOff'])
        
        # set tool
        if tool == 'labelpeak':
            self.toolsLabelPeak_butt.SetBitmapLabel(images.lib['spectrumLabelPeakOn'])
            self.spectrumCanvas.setLMBFunction('range')
            self.spectrumCanvas.setProperties(showCurTracker=False)
            self.spectrumCanvas.setCursorImage(images.lib['cursorArrowPeak'])
            
        elif tool == 'labelpoint':
            self.toolsLabelPoint_butt.SetBitmapLabel(images.lib['spectrumLabelPointOn'])
            self.spectrumCanvas.setLMBFunction('point')
            self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurXPos=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurYPos=config.spectrum['showTracker'])
            if config.spectrum['showTracker']:
                self.spectrumCanvas.setCursorImage(images.lib['cursorCrossPoint'])
            else:
                self.spectrumCanvas.setCursorImage(images.lib['cursorArrowPoint'])
        
        elif tool == 'deletelabel':
            self.toolsDeleteLabel_butt.SetBitmapLabel(images.lib['spectrumDeleteLabelOn'])
            self.spectrumCanvas.setLMBFunction('rectangle')
            self.spectrumCanvas.setProperties(showCurTracker=False)
            self.spectrumCanvas.setCursorImage(images.lib['cursorArrowDelete'])
        
        elif tool == 'measure':
            self.toolsMeasure_butt.SetBitmapLabel(images.lib['spectrumMeasureOn'])
            self.spectrumCanvas.setLMBFunction('xDistance')
            self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurDistance=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurXPos=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurYPos=config.spectrum['showTracker'])
            if config.spectrum['showTracker']:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_CROSS))
            else:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_ARROW))
        
        elif tool == 'offset':
            self.toolsOffset_butt.SetBitmapLabel(images.lib['spectrumOffsetOn'])
            self.spectrumCanvas.setLMBFunction('yDistance')
            self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurDistance=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurXPos=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurYPos=config.spectrum['showTracker'])
            if config.spectrum['showTracker']:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_CROSS))
            else:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_ARROW))
    # ----
    
    
    def setSpectrumProperties(self, docIndex):
        """Set spectrum properties."""
        
        # check document
        if docIndex == None:
            return
        
        # get document
        docData = self.documents[docIndex]
        spectrum = self.container[docIndex+2]
        
        spectrum.setProperties(legend=docData.title)
        spectrum.setProperties(visible=docData.visible)
        spectrum.setProperties(flipped=docData.flipped)
        spectrum.setProperties(xOffset=docData.offset[0])
        spectrum.setProperties(yOffset=docData.offset[1])
        spectrum.setProperties(normalized=config.spectrum['normalize'])
        spectrum.setProperties(xOffsetDigits=config.main['mzDigits'])
        spectrum.setProperties(yOffsetDigits=config.main['intDigits'])
        
        spectrum.setProperties(showInGel=True)
        spectrum.setProperties(showSpectrum=True)
        spectrum.setProperties(showTicks=config.spectrum['showTicks'])
        spectrum.setProperties(showPoints=config.spectrum['showDataPoints'])
        spectrum.setProperties(showGelLegend=config.spectrum['showGelLegend'])
        spectrum.setProperties(spectrumColour=docData.colour)
        
        spectrum.setProperties(labelAngle=config.spectrum['labelAngle'])
        spectrum.setProperties(labelCharge=config.spectrum['labelCharge'])
        spectrum.setProperties(labelDigits=config.main['mzDigits'])
        spectrum.setProperties(labelBgr=config.spectrum['labelBgr'])
        
        labelFont = wx.Font(config.spectrum['labelFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        spectrum.setProperties(labelFont=labelFont)
        
        if docIndex == self.currentDocument:
            spectrum.setProperties(showLabels=config.spectrum['showLabels'])
            spectrum.setProperties(tickColour=config.spectrum['tickColour'])
        else:
            spectrum.setProperties(showLabels=(config.spectrum['showLabels'] and config.spectrum['showAllLabels']))
            spectrum.setProperties(tickColour=docData.colour)
    # ----
    
    
    def updateCanvasProperties(self, ID=None, refresh=True):
        """Update canvas properties."""
        
        # update button image
        if ID != None:
            if ID == ID_viewLabels:
                image = (images.lib['spectrumLabelsOff'], images.lib['spectrumLabelsOn'])[bool(config.spectrum['showLabels'])]
                self.showLabels_butt.SetBitmapLabel(image)
            elif ID == ID_viewTicks:
                image = (images.lib['spectrumTicksOff'], images.lib['spectrumTicksOn'])[bool(config.spectrum['showTicks'])]
                self.showTicks_butt.SetBitmapLabel(image)
            elif ID == ID_viewNotationMarks:
                image = (images.lib['spectrumNotationMarksOff'], images.lib['spectrumNotationMarksOn'])[bool(config.spectrum['showNotationMarks'])]
                self.showNotationMarks_butt.SetBitmapLabel(image)
            elif ID == ID_viewLabelAngle:
                image = (images.lib['spectrumLabelAngleOff'], images.lib['spectrumLabelAngleOn'])[bool(config.spectrum['labelAngle'])]
                self.labelAngle_butt.SetBitmapLabel(image)
            elif ID == ID_viewGel:
                image = (images.lib['spectrumGelOff'], images.lib['spectrumGelOn'])[bool(config.spectrum['showGel'])]
                self.showGel_butt.SetBitmapLabel(image)
            elif ID == ID_viewTracker:
                image = (images.lib['spectrumTrackerOff'], images.lib['spectrumTrackerOn'])[bool(config.spectrum['showTracker'])]
                self.showTracker_butt.SetBitmapLabel(image)
            elif ID == ID_viewAutoscale:
                image = (images.lib['spectrumAutoscaleOff'], images.lib['spectrumAutoscaleOn'])[bool(config.spectrum['autoscale'])]
                self.autoscale_butt.SetBitmapLabel(image)
            elif ID == ID_viewNormalize:
                image = (images.lib['spectrumNormalizeOff'], images.lib['spectrumNormalizeOn'])[bool(config.spectrum['normalize'])]
                self.normalize_butt.SetBitmapLabel(image)
        
        # set canvas properties
        self.spectrumCanvas.setProperties(gelHeight=config.spectrum['gelHeight'])
        self.spectrumCanvas.setProperties(xPosDigits=config.main['mzDigits'])
        self.spectrumCanvas.setProperties(yPosDigits=config.main['intDigits'])
        self.spectrumCanvas.setProperties(distanceDigits=config.main['mzDigits'])
        self.spectrumCanvas.setProperties(overlapLabels=config.spectrum['overlapLabels'])
        self.spectrumCanvas.setProperties(autoScaleY=config.spectrum['autoscale'])
        self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
        self.spectrumCanvas.setProperties(showCurDistance=config.spectrum['showTracker'])
        self.spectrumCanvas.setProperties(showGel=config.spectrum['showGel'])
        self.spectrumCanvas.setProperties(showPosBar=config.spectrum['showPosBar'])
        self.spectrumCanvas.setProperties(posBarHeight=config.spectrum['barHeight'])
        self.spectrumCanvas.setProperties(showIntBar=config.spectrum['showIntBar'])
        self.spectrumCanvas.setProperties(intBarHeight=config.spectrum['barHeight'])
        self.spectrumCanvas.setProperties(showLegend=config.spectrum['showLegend'])
        self.spectrumCanvas.setProperties(showGrid=config.spectrum['showGrid'])
        
        axisFont = wx.Font(config.spectrum['axisFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        self.spectrumCanvas.setProperties(axisFont=axisFont)
        
        # set cursor tracker according to current tool
        if self.currentTool == 'labelpeak':
            self.spectrumCanvas.setLMBFunction('rectangle')
            self.spectrumCanvas.setProperties(showCurTracker=False)
        
        elif self.currentTool == 'labelpoint':
            self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurXPos=config.spectrum['showTracker'])
            if config.spectrum['showTracker']:
                self.spectrumCanvas.setCursorImage(images.lib['cursorCrossPoint'])
            else:
                self.spectrumCanvas.setCursorImage(images.lib['cursorArrowPoint'])
        
        elif self.currentTool == 'deletelabel':
            self.spectrumCanvas.setProperties(showCurTracker=False)
        
        elif self.currentTool in ('measure', 'offset'):
            self.spectrumCanvas.setProperties(showCurTracker=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurDistance=config.spectrum['showTracker'])
            self.spectrumCanvas.setProperties(showCurXPos=config.spectrum['showTracker'])
            if config.spectrum['showTracker']:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_CROSS))
            else:
                self.spectrumCanvas.setCursorImage(wx.StockCursor(wx.CURSOR_ARROW))
        
        # set properties for documents
        labelFont = wx.Font(config.spectrum['labelFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        for docIndex in range(len(self.documents)):
            self.setSpectrumProperties(docIndex)
        
        # update tmp spectra
        self.updateNotationMarks(self.currentNotationMarks, refresh=False)
        self.updateTmpSpectrum(self.currentTmpSpectrum, refresh=False)
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def updateCursorInfo(self):
        """Update cursor info on MS scan canvas"""
        
        label = ''
        
        # get values
        position = self.spectrumCanvas.getCursorPosition()
        distance = self.spectrumCanvas.getDistance()
        
        # format numbers
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        intFormat = '%0.' + `config.main['intDigits']` + 'f'
        ppmFormat = '%0.' + `config.main['ppmDigits']` + 'f'
        
        # distance measurement
        if distance and position and self.currentTool == 'measure':
            format = 'm/z: %s    dist: %s    ppm: %s' % (mzFormat, mzFormat, ppmFormat)
            label += format % (position[0], distance[0], 1e6*distance[0]/position[0])
        
        # offset dragging
        elif distance and position and self.currentTool == 'offset':
            format = 'a.i.: %s    dist: %s' % (intFormat, intFormat)
            label += format % (position[1], distance[1])
        
        elif position:
            format = 'm/z: %s    a.i.: %s' % (mzFormat, intFormat)
            label = format % (position[0], position[1])
        
        # show info
        self.cursorInfo.SetLabel(label)
    # ----
    
    
    def updateTmpSpectrum(self, points, refresh=True):
        """Set new data to tmp spectrum."""
        
        self.currentTmpSpectrum = points
        
        # check spectrum
        if points == None:
            points = []
        
        # snap to current spectrum
        flipped = False
        xOffset = 0
        yOffset = 0
        if self.currentDocument != None and len(points):
            
            # normalize points
            if config.spectrum['normalize']:
                normalization = self.documents[self.currentDocument].spectrum.normalization()
                points = (numpy.array(points) - numpy.array((0, normalization[1]))) / numpy.array((1, normalization[0]))
            
            # offset points
            else:
                xOffset = self.documents[self.currentDocument].offset[0]
                yOffset = self.documents[self.currentDocument].offset[1]
            
            # flip points
            flipped = self.documents[self.currentDocument].flipped
        
        # add tmp spectrum to container
        obj = mspy.plot.points(points, \
            flipped=flipped, \
            xOffset=xOffset, \
            yOffset=yOffset, \
            showInGel=False, \
            showLines=True, \
            showPoints=False, \
            exactFit=True, \
            lineColour=config.spectrum['tmpSpectrumColour']\
        )
        self.container[0] = obj
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def updateNotationMarks(self, points, refresh=True):
        """Set new data to notation marks."""
        
        self.currentNotationMarks = points
        
        # check spectrum and view option
        if points == None or not config.spectrum['showNotationMarks']:
            points = []
        
        # snap data to current spectrum
        flipped = False
        xOffset = 0
        yOffset = 0
        if self.currentDocument != None and len(points):
            
            # normalize points
            if config.spectrum['normalize']:
                normalization = self.documents[self.currentDocument].spectrum.normalization()
                points = (numpy.array(points) - numpy.array((0, normalization[1]))) / numpy.array((1, normalization[0]))
            
            # offset points
            else:
                xOffset = self.documents[self.currentDocument].offset[0]
                yOffset = self.documents[self.currentDocument].offset[1]
            
            # flip points
            flipped = self.documents[self.currentDocument].flipped
        
        # add points to container
        obj = mspy.plot.points(points, \
            flipped=flipped, \
            xOffset=xOffset, \
            yOffset=yOffset, \
            showInGel=False, \
            showPoints=True, \
            showLines=False, \
            exactFit=True, \
            pointColour=config.spectrum['notationMarksColour']\
        )
        self.container[1] = obj
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def updateSpectrum(self, docIndex, refresh=True):
        """Reload spectrum data."""
        
        # make spectrum
        docData = self.documents[docIndex]
        spectrum = mspy.plot.spectrum(docData.spectrum)
        
        # update container
        self.container[docIndex+2] = spectrum
        
        # set spectrum properties
        self.setSpectrumProperties(docIndex)
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def updateSpectrumProperties(self, docIndex, refresh=True):
        """Update all spectrum properties."""
        
        # update spectrum properties
        self.setSpectrumProperties(docIndex)
        
        # update tmp spectra
        self.updateNotationMarks(self.currentNotationMarks, refresh=False)
        self.updateTmpSpectrum(self.currentTmpSpectrum, refresh=False)
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def selectSpectrum(self, docIndex, refresh=True):
        """Set document as active."""
        
        # hide labels on last active document
        if self.currentDocument != None:
            self.container[self.currentDocument+2].setProperties(showLabels=(config.spectrum['showLabels'] and config.spectrum['showAllLabels']))
            self.container[self.currentDocument+2].setProperties(tickColour=self.documents[self.currentDocument].colour)
        
        # set current document
        self.currentDocument = docIndex
        if self.currentDocument != None:
            self.spectrumCanvas.setCurrentObject(self.currentDocument+2)
            self.container[self.currentDocument+2].setProperties(showLabels=config.spectrum['showLabels'])
            self.container[self.currentDocument+2].setProperties(tickColour=config.spectrum['tickColour'])
        else:
            self.spectrumCanvas.setCurrentObject(None)
        
        # update tmp spectra
        self.updateTmpSpectrum(None, refresh=False)
        self.updateNotationMarks(None, refresh=False)
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def appendLastSpectrum(self, refresh=True):
        """Append new spectrum to container."""
        
        # append spectrum
        docData = self.documents[-1]
        spectrum = mspy.plot.spectrum(docData.spectrum)
        self.container.append(spectrum)
        
        # set spectrum properties
        docIndex = len(self.documents) - 1
        self.setSpectrumProperties(docIndex)
        
        # redraw plot
        if refresh:
            self.refresh(fullsize=True)
    # ----
    
    
    def deleteSpectrum(self, docIndex, refresh=True):
        """Remove selected spectrum from container."""
        
        # remove spectrum
        del self.container[docIndex+2]
        
        # set current document
        if docIndex == self.currentDocument:
            self.currentDocument = None
            self.spectrumCanvas.setCurrentObject(None)
            self.updateTmpSpectrum(None, refresh=False)
            self.updateNotationMarks(None, refresh=False)
        
        # redraw plot
        if refresh:
            self.refresh()
    # ----
    
    
    def highlightPoints(self, points):
        """Highlight specified points in the spectrum."""
        self.spectrumCanvas.highlightXPoints(points)
    # ----
    
    
    def refresh(self, fullsize=False):
        """Redraw spectrum."""
        
        # check for flipped documents and update canvas symmetry
        self.spectrumCanvas.setProperties(ySymmetry=False)
        for docData in self.documents:
            if docData.visible and docData.flipped:
                self.spectrumCanvas.setProperties(ySymmetry=True)
                break
        
        # redraw canvas
        self.spectrumCanvas.refresh(fullsize=fullsize)
    # ----
    
    
    def getBitmap(self, width, height, printerScale):
        """Get spectrum image."""
        return self.spectrumCanvas.getBitmap(width, height, printerScale)
    # ----
    
    
    def getPrintout(self, filterSize, title):
        """Get spectrum printout."""
        return self.spectrumCanvas.getPrintout(filterSize, title)
    # ----
    
    
    def getCurrentRange(self):
        """Get current X range."""
        return self.spectrumCanvas.getXCurrentRange()
    # ----
    
    


class dlgCanvasProperties(wx.Dialog):
    """Set canvas properties."""
    
    def __init__(self, parent, onChangeFce):
        
        # initialize document frame
        wx.Dialog.__init__(self, parent, -1, "Canvas Properties", style=wx.DEFAULT_DIALOG_STYLE|wx.STAY_ON_TOP)
        self.onChangeFce = onChangeFce
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        self.Layout()
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.CentreOnParent()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        # make canvas params
        mzDigits_label = wx.StaticText(self, -1, "m/z precision:")
        self.mzDigits_slider = wx.Slider(self, -1, config.main['mzDigits'], 0, 6, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.mzDigits_slider.SetTickFreq(1,1)
        self.mzDigits_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        intDigits_label = wx.StaticText(self, -1, "Intensity precision:")
        self.intDigits_slider = wx.Slider(self, -1, config.main['intDigits'], 0, 6, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.intDigits_slider.SetTickFreq(1,1)
        self.intDigits_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        barHeight_label = wx.StaticText(self, -1, "Bars height:")
        self.barHeight_slider = wx.Slider(self, -1, config.spectrum['barHeight'], 4, 20, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.barHeight_slider.SetTickFreq(5,1)
        self.barHeight_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        gelHeight_label = wx.StaticText(self, -1, "Gel height:")
        self.gelHeight_slider = wx.Slider(self, -1, config.spectrum['gelHeight'], 10, 50, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.gelHeight_slider.SetTickFreq(5,1)
        self.gelHeight_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        axisFontSize_label = wx.StaticText(self, -1, "Canvas font size:")
        self.axisFontSize_slider = wx.Slider(self, -1, config.spectrum['axisFontSize'], 5, 15, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.axisFontSize_slider.SetTickFreq(2,1)
        self.axisFontSize_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        labelFontSize_label = wx.StaticText(self, -1, "Label font size:")
        self.labelFontSize_slider = wx.Slider(self, -1, config.spectrum['labelFontSize'], 5, 15, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.labelFontSize_slider.SetTickFreq(2,1)
        self.labelFontSize_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(mzDigits_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.mzDigits_slider, (0,1))
        grid.Add(intDigits_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.intDigits_slider, (1,1))
        grid.Add(barHeight_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.barHeight_slider, (2,1))
        grid.Add(gelHeight_label, (3,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.gelHeight_slider, (3,1))
        grid.Add(axisFontSize_label, (4,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.axisFontSize_slider, (4,1))
        grid.Add(labelFontSize_label, (5,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.labelFontSize_slider, (5,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def onChange(self, evt):
        """Set parameter and update canvas while scrolling."""
        
        # get canvas params
        config.main['mzDigits'] = self.mzDigits_slider.GetValue()
        config.main['intDigits'] = self.intDigits_slider.GetValue()
        config.spectrum['barHeight'] = self.barHeight_slider.GetValue()
        config.spectrum['gelHeight'] = self.gelHeight_slider.GetValue()
        config.spectrum['axisFontSize'] = self.axisFontSize_slider.GetValue()
        config.spectrum['labelFontSize'] = self.labelFontSize_slider.GetValue()
        
        # set params to canvas and update
        self.onChangeFce()
    # ----
    
    


class dlgToolsProperties(wx.Dialog):
    """Set tools properties."""
    
    def __init__(self, parent):
        
        # initialize document frame
        wx.Dialog.__init__(self, parent, -1, "Tools Properties", style=wx.DEFAULT_DIALOG_STYLE|wx.STAY_ON_TOP)
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        self.Layout()
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        # make peak picking params
        peakpickingHeight_label = wx.StaticText(self, -1, "Picking height:")
        self.peakpickingHeight_slider = wx.Slider(self, -1, config.processing['peakpicking']['pickingHeight']*100, 1, 100, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.peakpickingHeight_slider.SetTickFreq(10,1)
        self.peakpickingHeight_slider.Bind(wx.EVT_SCROLL, self.onChange)
        
        peakpickingAdaptiveNoise_label = wx.StaticText(self, -1, "Adaptive threshold:")
        self.peakpickingAdaptiveNoise_check = wx.CheckBox(self, -1, "")
        self.peakpickingAdaptiveNoise_check.SetValue(config.processing['peakpicking']['adaptiveNoise'])
        self.peakpickingAdaptiveNoise_check.Bind(wx.EVT_CHECKBOX, self.onChange)
        
        peakpickingMonoisotopic_label = wx.StaticText(self, -1, "Set as monoisotopic:")
        self.peakpickingMonoisotopic_check = wx.CheckBox(self, -1, "")
        self.peakpickingMonoisotopic_check.SetValue(config.processing['peakpicking']['monoisotopic'])
        self.peakpickingMonoisotopic_check.Bind(wx.EVT_CHECKBOX, self.onChange)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(peakpickingHeight_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingHeight_slider, (0,1), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(peakpickingAdaptiveNoise_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingAdaptiveNoise_check, (1,1), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(peakpickingMonoisotopic_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingMonoisotopic_check, (2,1), flag=wx.ALIGN_CENTER_VERTICAL)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def onChange(self, evt):
        """Update values."""
        
        # get params
        config.processing['peakpicking']['pickingHeight'] = self.peakpickingHeight_slider.GetValue()/100.
        config.processing['peakpicking']['adaptiveNoise'] = self.peakpickingAdaptiveNoise_check.GetValue()
        config.processing['peakpicking']['monoisotopic'] = self.peakpickingMonoisotopic_check.GetValue()
    # ----
    
    


class dlgSpectrumOffset(wx.Dialog):
    """Set spectrum offset."""
    
    def __init__(self, parent, offset):
        wx.Dialog.__init__(self, parent, -1, "Spectrum offset", style=wx.DEFAULT_DIALOG_STYLE|wx.STAY_ON_TOP)
        
        self.parent = parent
        self.offset = offset
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        staticSizer = wx.StaticBoxSizer(wx.StaticBox(self, -1, ""), wx.VERTICAL)
        
        # make elements
        offset_label = wx.StaticText(self, -1, "Intensity offset:")
        self.offset_value = wx.TextCtrl(self, -1, str(self.offset[1]), size=(120, -1), style=wx.TE_PROCESS_ENTER, validator=mwx.validator('float'))
        self.offset_value.Bind(wx.EVT_TEXT, self.onChange)
        self.offset_value.Bind(wx.EVT_TEXT_ENTER, self.onOffset)
        
        cancel_butt = wx.Button(self, wx.ID_CANCEL, "Cancel")
        self.offset_butt = wx.Button(self, wx.ID_OK, "Offset")
        self.offset_butt.Bind(wx.EVT_BUTTON, self.onOffset)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(offset_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.offset_value, (0,1))
        
        staticSizer.Add(grid, 0, wx.ALL, 5)
        
        buttSizer = wx.BoxSizer(wx.HORIZONTAL)
        buttSizer.Add(cancel_butt, 0, wx.RIGHT, 15)
        buttSizer.Add(self.offset_butt, 0)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(staticSizer, 0, wx.CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        mainSizer.Add(buttSizer, 0, wx.CENTER|wx.LEFT|wx.RIGHT|wx.BOTTOM, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def onChange(self, evt):
        """Check data."""
        
        # get data
        try:
            self.offset = [0, float(self.offset_value.GetValue())]
        except:
            self.offset = None
    # ----
    
    
    def onOffset(self, evt):
        """Offset."""
        
        # check value and end
        if self.offset != None:
            self.EndModal(wx.ID_OK)
        else:
            wx.Bell()
    # ----
    
    
    def getData(self):
        """Return values."""
        return self.offset
    # ----
    
    


class fileDropTarget(wx.FileDropTarget):
    """Generic drop target for files."""
    
    def __init__(self, fce):
        wx.FileDropTarget.__init__(self)
        self.fce = fce
    # ----
    
    
    def OnDropFiles(self, x, y, paths):
        """Open dropped files."""
        self.fce(paths=paths)
    # ----
    

