//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  coorfile.cpp
//
//  Example:
//     Conversion between different coordinate file formats:
//     PDB, mmCIF and BINARY.
//
//  08 Mar 2001
//
//  SGI make:  f77 -o coorfile coorfile.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif


void  PrintInstructions()  {
  printf ( 
    "\n"
    "Coordinate file conversions PDB<->CIF<->BIN<->PDB\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "coorfile -k input_file output_file\n"
    "\n"
    "where key 'k' specifies the format of output_file:\n"
    "\n"
    "   key   Format\n"
    " ------------------------------\n"
    "   PDB   PDB file\n"
    "   CIF   CIF file\n"
    "   BIN   MMDB binary file\n"
    " ------------------------------\n"
    "\n"
    "Format of input_file is detected automatically.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager;
int            kin,kout,RC,lcount;
char           S[500];

  if (argc<4)  {
    printf ( " ***** WRONG COMMAND LINE.\n" );
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  //  2.  Determine the format of output file
  kout = -10;
  if (!strcasecmp(argv[1],"-PDB"))  kout = MMDB_FILE_PDB;
  if (!strcasecmp(argv[1],"-CIF"))  kout = MMDB_FILE_CIF;
  if (!strcasecmp(argv[1],"-BIN"))  kout = MMDB_FILE_Binary;
  if (kout<0)  {
    printf ( " ***** WRONG OPERATION KEY.\n" );
    PrintInstructions();
    return 2;
  }

  //  3.  Determine the format of input file. Although MMDB
  //      provides a function for reading with automatical recgnition
  //      of input file format (see e.g. example sel_exmp1.cpp), here
  //      we demonstrate a conservative approach, which may be useful
  //      for enforcing the input of incomplete files (doubtful,
  //      honestly) or just checking the file format without actually
  //      reading it.
  kin = isMMDBBIN ( argv[2] );
  if (kin<0)  {
    printf ( " ***** INPUT FILE\n\n %s\n\nDOES NOT EXIST.",argv[2] );
    return 3;
  }
  if (kin==0)  kin = MMDB_FILE_Binary;
  else if (isPDB(argv[2])==0)  kin = MMDB_FILE_PDB;
  else if (isCIF(argv[2])==0)  kin = MMDB_FILE_CIF;
  else  {
    printf ( " ***** UNRECOGNIZABLE FORMAT OF INPUT FILE\n"
             "       %s\n",
             argv[2] );
    return 4;
  }

  //  4.  Create an instance of MMDB
  MMDBManager = new CMMDBManager();
  //  4.  Read coordinate file.
  //    4.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager->SetFlag ( MMDBF_PrintCIFWarnings | MMDBF_FixSpaceGroup |
                         MMDBF_IgnoreDuplSeqNum | MMDBF_IgnoreHash );

  //    4.2 MMDB provides separate functions for reading the
  //        coordinate files in each format (note that there is
  //        a common function for all formats, see e.g. example
  //        sel_exmp1.cpp):
  switch (kin)  {
    default               :
    case MMDB_FILE_PDB    : 
        printf ( " ...reading PDB file %s\n",argv[2] );
        RC = MMDBManager->ReadPDBASCII ( argv[2] );
      break;
    case MMDB_FILE_CIF    :
        printf ( " ...reading CIF file %s\n",argv[2] );
        RC = MMDBManager->ReadCIFASCII ( argv[2] );
      break;
    case MMDB_FILE_Binary :
        printf ( " ...reading MMDB binary file %s\n",argv[2] );
        RC = MMDBManager->ReadMMDBF ( argv[2] );
  }

  //    4.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager->GetInputBuffer ( S, lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager;
    return 5;
  } else
    printf ( "   --- Ok.\n" );

  printf ( " Total %i atoms\n"
           " Space group     %s\n"
           " Space group fix %s\n",
           MMDBManager->GetNumberOfAtoms(),
           MMDBManager->GetSpaceGroup   (),
           MMDBManager->GetSpaceGroupFix() );

  //  5.  Now output the whole MMDB in the required format:
  switch (kout)  {
    default               :
    case MMDB_FILE_PDB    : 
        printf ( " ...writing PDB file %s\n",argv[3] );
        RC = MMDBManager->WritePDBASCII ( argv[3] );
      break;
    case MMDB_FILE_CIF    :
        printf ( " ...writing CIF file %s\n",argv[3] );
        RC = MMDBManager->WriteCIFASCII ( argv[3] );
      break;
    case MMDB_FILE_Binary :
        printf ( " ...writing MMDB binary file %s\n",argv[3] );
        RC = MMDBManager->WriteMMDBF ( argv[3] );
  }

  //    6. Checking for write-file errors is obsolete,
  //       but nevertheless ...
  if (RC) {
    printf ( " ***** ERROR #%i WRITE:\n\n %s\n\n",
             RC,GetErrorDescription(RC) );
    delete MMDBManager;
    return 6;
  } else
    printf ( "   --- Ok.\n" );

  delete MMDBManager;

  return 0;

}
