//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  spseatoms.cpp
//
//  Example:
//     Superposing two atom sets.
//
//  17 Sep 2002
//
//  SGI make:  f77 -o spseatoms spseatoms.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MATH_H
#include <math.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions ( pstr argv0 )  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Superposing two atom sets.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "%s coor_file\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format.\n"
    "\n",argv0
         );
}



// ===========================================================


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager;
int            RC,lcount;
char           S[500];
mat44          T;
realtype       rmsd,dx,dy,dz;
int            i,j,nA;
PPCAtom        A1,A2;

  if (argc<2)  {
    PrintInstructions ( argv[0] );
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  //  2.  Create an instance of MMDB
  MMDBManager = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }

  //  4.  Get atoms to play with
  MMDBManager->GetAtomTable ( A2,nA );
  //      and copy them into a new set of atoms:
  A1 = new PCAtom[nA];
  for (i=0;i<nA;i++)  {
    A1[i] = new CAtom();
    A1[i]->Copy ( A2[i] );
  }
  printf ( " ---  %i atoms were selected for rotation.\n",nA );

  
  //  5.  Rotate the newly created set of atoms

  EulerRotation ( A1, nA,
                  Pi/2.0,  // Euler's alpha
                  0.0,     // Euler's beta
                  0.0,     // Euler's gamma
                  0.0,     // x-center of rotation
                  0.0,     // y-center of rotation
                  0.0      // z-center of rotation
                );

  //  6.  Calculate the rotational-translational matrix that
  //      superposes A1 over A2

  i = SuperposeAtoms ( T,A1,nA,A2,NULL );
  printf ( " --- superposition done with return code %i.\n\n"
           "  Rotational-translational matrix:\n\n",i );
  for (i=0;i<4;i++)  {
    for (j=0;j<4;j++)
      printf ( " %11.4g",T[i][j] );
    printf ( "\n" );
  }

  //  7.  Apply the transormation matrix to A1 and calculate the
  //      total RMS difference
  rmsd = 0.0;
  j    = 0;
  for (i=0;i<nA;i++)
    if (!A1[i]->Ter)  {
      A1[i]->Transform ( T );
      dx = A1[i]->x - A2[i]->x;
      dy = A1[i]->y - A2[i]->y;
      dz = A1[i]->z - A2[i]->z;
      rmsd += dx*dx + dy*dy + dz*dz;
      j++;
    }
  if (j>0)  rmsd = sqrt(rmsd/j);
  printf ( "\n --- achieved rmsd %11.4g\n",rmsd );


  //  8. Since array A1 was allocated by the application, the
  //     application is responsible for its disposal. Array A2
  //     will be disposed by MMDB manager.
  for (i=0;i<nA;i++)
    if (A1[i])  delete A1[i];
  if (A1)  delete A1;

  //  9. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection index SelAtom.
  //     It will be disposed by its manager automatically,
  //     therefore an attempt to dispose it from the application
  //     will cause a crash.
  if (MMDBManager)  delete MMDBManager;

  return 0;

}



