/*
     This code is distributed under the terms and conditions of the
     CCP4 licence agreement as `Part ii)' software.  See the conditions
     in the CCP4 manual for a copyright statement.
*/

/*   ccplib.c
     Peter Briggs CCP4 May 2001

     General library functions and utilities

     ccperror(level,message)
     Error reporting and program termination
     FIXME needs to call a QPRINT clone which checks against the
     verbosity level before printing messages

     ccp4printf(level,format,printargs)
     A wrapper for vprintf - acts like printf with an additional
     argument which is checked against the reference verbosity
     level.

     ccp4fyp(argc,argv)
     Initialise environment for CCP4 programs and parse the
     command line arguments
     FIXME doesn't obey the CCP4FYP search logic for environ.def
     and default.def

     p_ccp4malloc(size,message)
     Allocate memory or exit program with message on failure
     FIXME need to decide if this is actually useful in its
     current form

     ccp4setenv(logical_name,value,envname,envtype,envext,ienv,
     no_overwrt)
     Set up file names and associate with logical_name in
     environment
     FIXME doesn't properly handle unrecognised input

     ccp4modulo(x,y)
     Return x modulo y

     ccpexists(filename)
     Check if file exists and can be opened for reading

     ccpputenv(logical_name,file_name)
     Set environment variable logical_name to have value file_name
*/

/*------------------------------------------------------------------*/

/* CCP4 library clones */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <stdarg.h>

/* Library header file */
#include "ccplib.h"
#include "cparser.h"

/*------------------------------------------------------------------*/

/* ccperror

   Error reporting and program termination

   ccperror is called with a message level ierr and a message string
   The exact behaviour is determined by the value of ierr:

   0 : normal termination and stop
   1 : fatal error and stop
   2 : report severe warning
   3 : report information
   4 : report from library

   ierr=-1 also reports last system error and terminates.

   The text in message is sent to standard output, and to standard
   error for ierr=1 (or -1).

*/
int ccperror(int ierr, const char* message)
{
  char *prog_name=NULL;

  /* Get the program name */
  prog_name = ccp4ProgramName(NULL);

  if (ierr==0) {
    /* Level 0 : normal termination */
    /* Need to get the amount of time elapsed since start of
       program
       This would be initialised by ccp4fyp */
    ccp4printf(0," %s:  %s\n",prog_name,message);
    exit(0);

  } else if (ierr==1 || ierr==-1) {
    /* Level 1 (-1) : fatal error */
    /* If ierr=-1 then print last system error
       N.B. Use of perror in this context is untested by me */
    if (ierr < 0) perror("Last system error message");
    /* Need to get the amount of time elapsed since start of
       program
       This would be initialised by ccp4fyp */
    /* Send the message to the standard error */
    fprintf(stderr," %s:  %s\n",prog_name,message);
    /* Also to the standard out */
    ccp4printf(0," %s:  %s\n",prog_name,message);
    exit(1);

  } else if (ierr==2) {
    /* Level 2 : severe warning */
    ccp4printf(0," \n $TEXT:Warning: $$ comment $$ \n WARNING: %s\n $$\n",
	   message);

  } else if (ierr>2)  {
    /* Levels higher than 2 : report information */
    ccp4printf(0,"%s\n",message);
  }
  return 0;
}

/*------------------------------------------------------------------*/

/* ccp4printf

   This is a wrapper for vprintf

   If the supplied message is less than or equal to the reference
   verbosity level then the format string and remaining arguments
   are passed to vprintf to be printed on stdout.
   Otherwise nothing is printed.

   The format string has the same format as the format strings
   passed to printf, with the remaining arguments being the values
   (if any) to be inserted into placeholders in the format string.

   Returns the number of bytes printed, or zero if no printing was
   performed, or a negative number for an error from vprintf.
*/
int ccp4printf(int level, char *format, ...)
{
  int     nbytes=0;
  va_list args;

  /* Check the level against the refence verbosity level */
  if (level <= ccp4VerbosityLevel(-1)) { 
    /* Use the vprint function to print */
    va_start(args,format);
    nbytes = vprintf(format,args);
    va_end(args);
  }
  /* Return to calling function */
  return nbytes;
}

/*------------------------------------------------------------------*/

/* ccp4fyp

   Initialise environment for CCP4 programs and parse the
   command line arguments
*/
int ccp4fyp(int argc, char *argv[])
{
  int  iarg=1,arg_end=0,ienv=0,diag=0;
  int  ihelp,def_init=1,env_init=1;
  char *testarg=NULL,line[MAXLINE];
  char *envname[MAXNAMES],*envtype[MAXNAMES],*envext[MAXNAMES];
  char *def_file=NULL,*env_file=NULL,*cinclude=NULL;
  FILE *envfp,*deffp;
  char *logical_name=NULL,*file_name=NULL,*file_type=NULL,*file_ext=NULL;
  int  lfilename;
  char *envfilename,*deffilename;
  PARSERARRAY *parser;

  /* Begin */
  if (diag) printf("CCP4FYP: starting\n");

  /* Initialise timing information */
  /* NOT IMPLEMENTED */

  /* Process command line option switches first
     Ignore iarg=0 because this is the executable name */

  while (iarg < argc && !arg_end) {
    if (diag) printf("CCP4FYP: command line argument %d = \"%s\"\n",iarg,argv[iarg]);
    if (argv[iarg][0] == '-') {
      if (diag) printf("--> This is an option switch\n");
      testarg = (char*) malloc((strlen(argv[iarg])+1)*sizeof(char));
      strtoupper(testarg,&(argv[iarg][1]));
      if (diag) printf("--> Truncated and uppercased it is now \"%s\"\n",testarg);

      /* Test for each possible option:
	 -v(erbose) 0-9  verbose output level
	 -h(elp) 0-9     (alias for -v)
	 -n              don't read environ.def and default.def
	 -d <filename>   use <filename> instead of default.def
	 -e <filename>   use <filename> instead of environ.def
	 -i              print CCP4 library version, program name and
	                 program version to standard output, and exit. 
	 -nohtml         suppress printing of html tags
	 -nosummary      suppress printing of summary tags
      */

      /* Verbose option
	 -v <0-9>
	 -h <0-9>
      */
      /* The message levels need to be store somewhere globally
	 accessible
	 Level 0 switches off all output (silent mode)
	 Level 9 prints everything
	 Level 1 is normal I guess
	 It is not clear from documentation precisely what is/is not
	 output for the levels between 0 and 9
      */
      if (testarg[0] == 'V' || testarg[0] == 'H') {
	iarg++;
	if (diag) printf("--> Identified -v option:");
	if (iarg < argc) {
	  if (strlen(argv[iarg]) == 1 && isdigit(argv[iarg][0])) {
	    ihelp = atoi(argv[iarg]);
	  } else {
	    ihelp = 1;
	  }
	  if (diag) printf(" level %d\n",ihelp);
	} else {
	  ihelp = 1;
	  if (diag) printf(" no help level - defaulting to 1\n");
	}

	/* Don't read environ.def and default.def
	   -n
	*/
      } else if (testarg[0] == 'N' && strlen(testarg) == 1) {
	if (diag) printf("--> Identified -n option\n");
	def_init = 0;
	env_init = 0;

	/* Use non-default default.def file
	   -d <filename>
	*/
      } else if (testarg[0] == 'D') {
	iarg++;
	if (diag) printf("--> Identified -d option:");
	if (iarg < argc) {
	  def_file = (char *) malloc(sizeof(char)*(strlen(argv[iarg])+1));
	  if (!def_file) ccperror(1,"Can't fetch filename for -d option");
	  strcpy(def_file,argv[iarg]);
	  def_init = 1;
	  if (diag) printf(" default.def file is \"%s\"\n",def_file);
	} else {
	  if (diag) printf(" no filename found\n");
	  ccperror(1,"Use: -d filename");
	}

	/* Use non-default environ.def file
	   -e <filename>
	*/
      } else if (testarg[0] == 'E') {
	iarg++;
	if (diag) printf("--> Identified -e option:");
	if (iarg < argc) {
	  env_file = (char *) malloc(sizeof(char)*(strlen(argv[iarg])+1));
	  if (!env_file) ccperror(1,"Can't fetch filename for -d option");
	  strcpy(env_file,argv[iarg]);
	  def_init = 1;
	  if (diag) printf(" environ.def file is \"%s\"\n",env_file);
	} else {
	  if (diag) printf(" no filename found\n");
	  ccperror(1,"Use: -e filename");
        }

	/* Info option
	   -i(nfo)
	*/
      } else if (testarg[0] == 'I') {
	if (diag) printf("--> Identified -i(nfo) option\n");
	/* Print program information and stop */
	/* The program and version information should be
	   stored somewhere on startup
	   For the time only print the program name */
	printf("Program: %s\n",ccp4ProgramName(NULL));
	exit(0);

	/* Unrecognised switch */
      } else {
	ccp4printf(1,"Ignoring unrecognised switch \"%s\"\n",argv[iarg]);
      }

      /* Next argument */
      iarg++;

    } else {
      /* Found an argument which doesn't start with a "-"
	 This is the end of the command line switches */
      if (diag) printf("CCP4FYP: end of command line switches\n");
      arg_end = 1;
    }
  }

  /* Initialise debug (verbosity) level */
  ccp4VerbosityLevel(ihelp);

  /* Initialise html, summary tags */
  /* NOT IMPLEMENTED */

  /* Read in environ.def */
  if (env_init) {
    if (diag) printf("CCP4FYP: reading environ.def file\n");
    /* Non-standard file? */
    if (env_file) {
      if (diag) printf("--> environ.def file was supplied on the command line\n");
    } else {
      /* Use the standard environ.def in $CINCL */
      if (diag) printf("--> use standard environ.def file\n");
      /* Get value of CINCL variable */
      cinclude = (char *) getenv("CINCL");
      if (!cinclude) {
	if (diag) printf("--> CINCL env var has no value assigned\n");
      } else {
	if (diag) printf("--> CINCL is \"%s\"\n",cinclude);
	/* Set the full path for the environ.def file
	   NB the following assumes UNIX system plus lots of
	   other assumptions which must be weeded out before
	   this is ready */
	lfilename = strlen(cinclude) + strlen("environ.def") + 2;
	envfilename = (char *) malloc(sizeof(char)*lfilename);
	if (!envfilename)
	  ccperror(-1,"CCP4FYP: cannot set environ.def pathname");
	strcpy(envfilename,cinclude);
	strcat(envfilename,"/");
	strcat(envfilename,"environ.def");
	if (diag) printf("--> Full path for environ.def is \"%s\"\n",envfilename);
	/* Open the environ.def file as read-only*/
	ccp4printf(2,"Opening file \"%s\"\n",envfilename);
	envfp = fopen(envfilename,"r");
	if (!envfp)
	  ccperror(1,"CCP4FYP: failed to open environ.def");

        /* Set a cparser array to deal with input from
	   environ.def */
	parser = (PARSERARRAY *) cparse_start(MAXTOKS);

	/* Set the delimiters to whitespace, = and .
	   This should split lines into the three components */
	cparse_delimiters(parser," \t=.",NULL);

	/* Read from the file until EOF*/
	while (fgets(line,MAXLINE,envfp)) {

	  /* Remove the trailing newline from fgets */
	  line[strlen(line)-1] = '\0';

	  /* Use cparse to get the tokens on each line */
	  cparse_reset(parser);
	  if (cparse(line,parser) == 3) {
	    logical_name = parser->token[0].fullstring;
	    file_type    = parser->token[1].fullstring;
	    file_ext     = parser->token[2].fullstring;
	    if (!logical_name || !file_type || !file_ext)
	      ccperror(-1,"CCP4FYP: couldn't parse line from environ.def");

	    /* environ.def contains lines of the form
	       LOGICALNAME=type.ext # comments
	       where type is "in", "out" or "inout"
	       and ext is the default extension, e.g. "mtz" or "scr"
	    */

	    /* Store in arrays for use when decoding default.def
	       and logical names on command line */
	    if (ienv+1 == MAXNAMES)
	      ccperror(1,"CCP4FYP: too many logical names in environ.def file");

	    /* Logical name in envname */
	    envname[ienv] = (char *)
	      p_ccp4malloc(sizeof(char)*(strlen(logical_name)+1),NULL);
	    strcpy(envname[ienv],logical_name);

	    /* File type in envtype */
	    envtype[ienv] = (char *)
	      p_ccp4malloc(sizeof(char)*(strlen(file_type)+1),NULL);
	    strcpy(envtype[ienv],file_type);

	    /* File extension in envext */
	    envext[ienv] = (char *)
	      p_ccp4malloc(sizeof(char)*(strlen(file_ext)+1),NULL);
	    strcpy(envext[ienv],file_ext);
	    
	    if (diag) printf("Decoded line: %s = %s.%s\n",envname[ienv],
		   envtype[ienv],envext[ienv]);

	    /* Increment ienv counter for number of name-pairs
	       read in */
	    ienv++;

	    /* Reset number of tokens before reading next line */
	    cparse_reset(parser);

	  } 
	}
	/* Close the environ.def file */
	fclose(envfp);

	/* Finished with the parser array */
	cparse_end(parser);
      }
    }
  }

  /* Read in default.def */
  if (def_init) {
    if (diag) printf("CCP4FYP: reading default.def file\n");
    /* Non-standard file? */
    if (def_file) {
      if (diag) printf("--> default.def file was supplied on the command line\n");
    } else {
      /* Use the standard environ.def in $CINCL */
      if (diag) printf("--> use standard default.def file\n");
      /* Get value of CINCL variable */
      cinclude = (char *) getenv("CINCL");
      if (!cinclude) {
	if (diag) printf("--> CINCL env var has no value assigned\n");
      } else {
	if (diag) printf("--> CINCL is \"%s\"\n",cinclude);
	/* Set the full path for the default.def file
	   NB the following assumes UNIX system plus lots of
	   other assumptions which must be weeded out before
	   this is ready */
	lfilename = strlen(cinclude) + strlen("default.def") + 2;
	deffilename = (char *) malloc(sizeof(char)*lfilename);
	if (!deffilename)
	  ccperror(-1,"CCP4FYP: cannot set default.def pathname");
	strcpy(deffilename,cinclude);
	strcat(deffilename,"/");
	strcat(deffilename,"default.def");
	if (diag) printf("--> Full path for default.def is \"%s\"\n",deffilename);
	/* Open the default.def file as read-only*/
	ccp4printf(2,"Opening file \"%s\"\n",deffilename);
	deffp = fopen(deffilename,"r");
	if (!deffp)
	  ccperror(1,"CCP4FYP: failed to open default.def");

        /* Set a cparser array to deal with input from
	   default.def */
	parser = (PARSERARRAY *) cparse_start(MAXTOKS);

	/* Set the delimiters to whitespace and =
	   This should split lines into the two components */
	cparse_delimiters(parser," \t=",NULL);

	/* Read from the file until EOF*/
	while (fgets(line,MAXLINE,deffp)) {

	  /* Remove the trailing newline from fgets */
	  line[strlen(line)-1] = '\0';

	  /* Use cparse to get the tokens on each line */
	  parser->ntokens = 0;
	  if (cparse(line,parser) == 2) {
	    logical_name = parser->token[0].fullstring;
	    file_name    = parser->token[1].fullstring;
	    if (!logical_name || !file_name)
	      ccperror(-1,"CCP4FYP: couldn't parse line from default.def");

	    /* environ.def contains lines of the form
	       LOGICALNAME=FILENAME # comments
	    */
	    if (diag) printf("Decoded line: %s = %s\n",logical_name,file_name);

	    /* Set up the environment for this pair
	       Don't overwrite any existing logical name */
	    ccp4setenv(logical_name,file_name,envname,envtype,envext,ienv,1);
	  }  
	  /* Reset number of tokens before reading next line */
	  parser->ntokens = 0;
	}
	/* Close the environ.def file */
	fclose(deffp);

	/* Finished with the parser array */
	cparse_end(parser);
      }
    }
  }

  /* Read in the rest of the command line arguments
     These should consist of pairs of arguments i.e.
     <logical name> <file name>
  */

  ccp4printf(2,"Processing Command Line Arguments\n");
  while (iarg < argc) {
    /* Get logical name and uppercase it */
    logical_name = (char*) malloc((strlen(argv[iarg])+1)*sizeof(char));
    strtoupper(logical_name,argv[iarg]);
    if (diag) printf("--> Logical name: \"%s\"",logical_name);
    iarg++;
    /* Get associated filename */
    if (iarg < argc) {
      file_name = (char*) malloc((strlen(argv[iarg])+1)*sizeof(char));
      strcpy(file_name,argv[iarg]);
      if (diag) printf("   file name: \"%s\"\n",file_name);
      /* Set up the environment for this pair
	 Do overwrite any existing logical name */
      ccp4setenv(logical_name,file_name,envname,envtype,envext,ienv,0);
      iarg++;
    } else {
      /* No associated filename - exit with error */
      if (diag) printf("   no associated logical name\n");
      ccperror(1,"Use: <logical name> <filename> ...");
    }
    /* Next pair of arguments */
  }
  ccp4printf(2,"End of pre-processing stage\n");

  /* End of CCP4FYP */
  return 0;
}

/*------------------------------------------------------------------*/

/* p_ccp4malloc

   This is a wrapper for the standard malloc command which
   traps for a NULL result and exits

   On a fatal error, the string "message" is passed to
   ccperror.
*/
void *p_ccp4malloc(size_t size, char *message)
{
  void *ptr;

  ptr = (void *) malloc(size);
  if (!ptr && size) {
    if (message) {
      ccperror(1,message);
    } else {
      ccperror(1,"P_CCP4MALLOC: failed to allocate memory");
    }
  }
  return ptr;
}

/*------------------------------------------------------------------*/

/* ccp4setenv

   Set environment variables

   Associates logical_name with value. It is passed arrays of name,
   type and extension for ienv number of name lines read from
   environ.def, which may be used to modify the value before
   assignment.

   If no_overwrt is true then any existing logical_name are not
   redefined.
*/
int ccp4setenv(char *logical_name, char* value, char **envname,
	      char **envtype, char **envext, int ienv, int no_overwrt)
{
  int  diag=0;
  int  icount,ichar,lext=0,lroot=0,lpath=0,lname,procid;
  char tmp_name[]="programX",*clibd,*cscr;
  char *file_ext=NULL,*file_root=NULL,*file_path=NULL,*file_name=NULL;
  char *prog_name,*tmpstr1;

  /* Begin */
  if (diag) printf("CCP4SETENV: started, ienv = %d\n",ienv);

  /* Exit if the logical name already exists and we are in
     no-overwrite mode */
  if (getenv(logical_name) && no_overwrt) return 0;
  
  /* Get the program (executable) name */
  /* This is actually the first argument from the argument list
     argv[0] which will need to be passed to ccp4setenv, or accessed
     from some global variable */
  prog_name = tmp_name;
  if (diag) printf("CCP4SETENV: program name = \"%s\"\n",prog_name);

  /* Look for a match between logical_name and the names from
     environ.def */
  if (diag) printf("CCP4SETENV: looking for a match to logical name \"%s\"\n",
		   logical_name);
  icount = 0;
  while (icount<ienv && !strmatch(logical_name,envname[icount])) {
    /* printf("%d: Logical name is \"%s\", current possibility is \"%s\"\n",
       icount,logical_name,envname[icount]);*/
    icount++;
  }
  if (icount == ienv) {
    /* Not in the list - non-standard logical name */
    if (diag) printf("%s is non-standard logical name\n",logical_name);
    /* Add it to the list */
    /* FIXME: This means altering ienv etc ... not doing it now */
  } else {
    if (diag) printf("CCP4SETENV: Matched logical name for number %d\n",icount);
  }

  /* Initialise a parser array to examine the supplied filename */
  /*parser = (PARSERARRAY *) cparse_start(20);
  if (!parser)
  ccperror(1,"CCP4SETENV: couldn't initialise parser array\n");*/

  /* Split the supplied value up into components to extract get the
     file extension, the file root (i.e. name less extension) and
     the file path
     It is possible there is no path and/or no extension */

  /* Get file path
     FIXME This is unix-specific */
  if (diag) printf("CCP4SETENV: length of value = %d\n",strlen(value));
  ichar = strlen(value) - 1;
  while (ichar >= 0 && value[ichar] != '/') ichar--;
  if (ichar < 0) {
    /* There is no path */
    if (diag) printf("CCP4SETENV: no file path found\n");
    lpath = -1;
  } else {
    /* There is a path in this filename */
    if (diag) printf("CCP4SETENV: there is a file path");
    lpath = ichar;
    if (lpath) {
      /* Path contains at least one directory
	 Store file path */
      file_path = (char *) p_ccp4malloc(sizeof(char)*(lpath+1),NULL);
      strncpy(file_path,value,lpath);
      /* Add trailing null since strncpy can't get it from
	 value (we are cutting out the middle of a string) */
      file_path[lpath] = '\0';
      if (diag) printf(" = \"%s\"\n",file_path);
    } else {
      /* Path contains a single character so it must be
	 at the root of the filesystem */
      if (diag) printf(" = root\n");
    }
  }

  /* Get file extension */
  ichar = strlen(value) - 1;
  while (ichar >= lpath && value[ichar] != '.') ichar--;
  if (ichar < lpath) {
    /* There is no extension */
    if (diag) printf("CCP4SETENV: no file extension found\n");
    lext = -1;
  } else {
    if (diag) printf("CCP4SETENV: there is a file extension");
    lext = strlen(value) - ichar;
    if (lext) {
      /* Store the extension */
      file_ext = (char *) p_ccp4malloc(sizeof(char)*(lext+1),NULL);
      strncpy(file_ext,&(value[ichar+1]),lext);
      if (diag) printf(" = \"%s\"\n",file_ext);
    } else {
      /* There is a dot but no extension
	 Not sure how to deal with this! */
      if (diag) printf(" = empty (dot with no extension)\n");
    }
  }   

  /* Get file root */
  ichar = lpath + 1;
  lroot = strlen(value) - (ichar + lext);
  if (lroot) {
    /* Store the file root */
    file_root = (char *) p_ccp4malloc(sizeof(char)*(lroot+1),NULL);
    strncpy(file_root,&(value[ichar]),lroot);
    /* Add the trailing null because we are cutting out the middle of
       a string with strncpy, which doesn't supply it itself */
    file_root[lroot] = '\0';
    if (diag) printf("CCP4SETENV: file root is \"%s\"\n",file_root);
  } else {
    /* There is no file root */
    if (diag) printf("CCP4SETENV: no file root!\n");
  }

  /* Add the appropriate file extension if none was supplied
     The exception is for /dev/null or NL: */
  if (!strmatch(value,"/dev/null") && !strmatch(value,"NL:")) {
    if (lext < 0) {
      /* Add extension */
      if (icount < ienv) {
	lext = strlen(envext[icount]);
	file_ext = (char *) p_ccp4malloc(sizeof(char)*(lext+1),NULL);
	strncpy(file_ext,envext[icount],(lext+1));
	if (diag) printf("CCP4SETENV: added extension \"%s\"\n",file_ext);
      } else {
	if (diag) printf("CCP4SETENV: not adding extension for non-standard file name\n");
      }
    }
    /* If no path supplied then get appropriate path depending
       on the extension
       .dic, .lib, .bes, .prt = $CLIBD
       .scr = $CCP4_SCR
    */
    if (lpath < 0) {
      /* Fetch the appropriate path name from the environment */
      
      if (strmatch(file_ext,"lib") || strmatch(file_ext,"dic")
	  || strmatch(file_ext,"bes") || strmatch(file_ext,"prt")) {
	/* Fetch CLIBD */
	clibd = (char *) getenv("CLIBD");
	if (clibd) {
	  if (diag) printf("CCP4SETENV: CLIBD = \"%s\"\n",clibd);
	  /* Store in file_path */
	  lpath = strlen(clibd);
	  file_path = (char *) p_ccp4malloc(sizeof(char)*(lpath+1),NULL);
	  strncpy(file_path,clibd,(lpath+1));
	  if (diag) printf("CCP4SETENV: set file path to CLIBD = \"%s\"\n",file_path);
	} else {
	  /* Couldn't get CLIBD */
	  ccperror(1,"CCP4SETENV: couldn't get CLIBD from environment - check setup");
	}

      } else if (strmatch(file_ext,"scr")) {
	/* Fetch CCP4_SCR */
	cscr = (char *) getenv("CCP4_SCR");
	if (cscr) {
	  if (diag) printf("CCP4SETENV: CCP4_SCR = \"%s\"\n",cscr);
	  /* Store in file_path */
	  lpath = strlen(cscr);
	  file_path = (char *) p_ccp4malloc(sizeof(char)*(lpath+1),NULL);
	  strncpy(file_path,cscr,(lpath+1));
	  if (diag) printf("CCP4SETENV: set file path to CCP4_SCR = \"%s\"\n",file_path);
	} else {
	  /* Couldn't get CCP4_SCR */
	  ccperror(1,"CCP4SETENV: couldn't get CCP4_SCR from environment - check setup");
	}
	/* Replace scr extension with the process id
	 In fact to guarantee that it is always 5 characters,
	 take the id number modulo 100,000
	 FIXME there maybe rounding problems doing it this
	 way - should be fixed up */
	procid = (int) getpid();
	if (diag) printf("CCP4SETENV: initial procid = %d\n",procid);
	procid = (int) ccp4modulo((double) procid,MODULO);
	if (diag) printf("CCP4SETENV: procid = %d",procid);
	if (file_ext) free(file_ext);
	file_ext = (char*) p_ccp4malloc(sizeof(char)*6,NULL);
	sprintf(file_ext,"%05d",procid);
	if (diag) printf(" giving file extension \"%s\"\n",file_ext);
      }
      /* No special path for this particular extension */
    }
  } else {
    if (diag) printf("CCP4SETENV: detected dev-null\n");
  }

  /* Build the filename */
  lname = lpath + 1;
  file_name = ccp4realloc(file_name,sizeof(char)*(lname + 1));
  if (lpath < 0) {
    file_name[0] = '\0';
  } else if (lpath == 0) {
    strncpy(file_name,"/",2);
  } else {
    strncpy(file_name,file_path,lname);
    strcat(file_name,"/");
  }
  if (diag) printf("CCP4SETENV: building filename = \"%s\"\n",file_name);
  lname = lname + lroot;
  file_name = ccp4realloc(file_name,sizeof(char)*(lname + 1));
  if (lroot) {
    strcat(file_name,file_root);
  }
  if (diag) printf("CCP4SETENV: building filename = \"%s\"\n",file_name);
  if (lext > -1) {
    lname = lname + lext + 1;
    file_name = ccp4realloc(file_name,sizeof(char)*(lname + 1));
    strcat(file_name,".");
    if (lext) {
      strcat(file_name,file_ext);
    }
  }
  if (diag) printf("CCP4SETENV: building filename = \"%s\"\n",file_name);

  /* Test that (non-default) input files exist */
  if (icount < ienv) {
    if (strmatch(envtype[icount],"in") && !no_overwrt) {
      /* Does file exist? */
      if (diag) printf("CCP4SETENV: checking for existence of input file\n");
      if (ccpexists(file_name)) {
	if (diag) printf("CCP4SETENV: \"%s\" can be opened for reading\n",file_name);
      } else {
	if (diag) printf("CCP4SETENV: \"%s\" cannot be opened for reading\n",file_name);
	ccperror(-1,"Cannot find file");
      }
    }
  } else {
    if (diag) printf("CCP4SETENV: cannot determine file type (input or output)\n");
  }

  /* Set the environment variable */
  if (ccpputenv(logical_name,file_name)) {
    if (diag) printf("CCP4SETENV: ccpputenv returned okay\n");
  } else {
    ccperror(-1,"Cannot create environment variable");
  }

  if (diag) {
    tmpstr1 = getenv(logical_name);
    if (tmpstr1) {
      printf("CCP4SETENV: logical name %s has value \"%s\"\n",logical_name,tmpstr1);
    } else {
      printf("CCP4SETENV: failed to fetch value for logical_name %s\n",logical_name);
    }
  }

  /* Free dynamically allocated memory before returning */
  if (file_ext) free(file_ext);
  if (file_root) free(file_root);
  if (file_path) free(file_path);
  if (file_name) free(file_name);

  return 0;
}

/*------------------------------------------------------------------*/

/* ccp4modulo

   Given x (float) and y (integer), returns x modulo y.
*/
float ccp4modulo(float x, int y)
{
  float xf,yf,n;
  if (y > 0) {
    xf = x/((double) y);
    yf = modff(xf,&n);
    return yf*y;
  }
  return 0;
}
/*------------------------------------------------------------------*/

/* ccpexists

   Check if named file exists
   The check is performed by attempting to fopen the file for
   read only, then immediately closing the file. If this method
   proves to be unsatisfactory then it may be necessary to investigate
   using access or stat instead.

   Returns 1 if the file can be opened for read (=exists), 0 otherwise.
*/
int ccpexists(char *filename)
{
  FILE *fp;

  if (filename) {
    fp = fopen(filename,"r");
    if (fp) {
      fclose(fp);
      return 1;
    }
  }
  return 0;
}

/*------------------------------------------------------------------*/

/* ccpputenv

   This is a wrapper for the C putenv command. It must be supplied
   with a logical name (the name of a variable which will set in the
   environment) and a file name (which will be assigned to that
   variable).
   FIXME For some systems might also need to use setenv

   Returns 1 if successful and 0 otherwise.
*/
int ccpputenv(char *logical_name, char *file_name)
{
  int ltmpstr,diag=0;
  char *tmpstr;

  if (logical_name && file_name) {
    /* Allocate memory for temporary string */
    ltmpstr = strlen(logical_name) + strlen(file_name) + 1;
    tmpstr = (char *) malloc(sizeof(char)*(ltmpstr+1));
    /* putenv requires a string of the form "logical_name=file_name" */
    if (tmpstr) {
      strcpy(tmpstr,logical_name);
      strcat(tmpstr,"=");
      strcat(tmpstr,file_name);
      if (diag) printf("CCPPUTENV: string going into putenv is \"%s\"\n",tmpstr);
      /* putenv returns 0 on success */
      if (putenv(tmpstr) == 0) return 1;
    }
    /* Don't free tmpstr as this causes an error - see man pages for
       putenv */
  }
  return 0;
}
/*
     This code is distributed under the terms and conditions of the
     CCP4 licence agreement as `Part ii)' software.  See the conditions
     in the CCP4 manual for a copyright statement.
*/

/*   ccp4program.c
     Peter Briggs CCP4 May 2001

     Utilies to set and fetch program information
*/

#include <stdio.h>

/*------------------------------------------------------------------*/

/* ccp4programname

   Set or return program name

   Always returns a pointer to the program name
   If progname is not NULL then set the program name to
   progname.
*/
char *ccp4ProgramName(char *progname)
{
  static char *programname=NULL;
  int         lname;

  if (progname) {
    lname = strlen(progname)+1;
    programname = (char *) realloc(programname,lname);
    strncpy(programname,progname,lname);
  }
  return programname;
}

/*------------------------------------------------------------------*/

/* ccp4VerbosityLevel

   Set or return the reference verbosity level

   Always return the verbosity level - if verboselevel is
   between 0 and 9 then reset the verbosity level to
   verboselevel
*/
int ccp4VerbosityLevel(int level)
{
  /* The default level is 1 */
  static int verbositylevel=1;

  if (level > -1 && level < 10)
    verbositylevel = level;
  return verbositylevel;
}

void ccp4rcs(char *progname) {

  printf(" \n");
  printf(" ###############################################################\n");
  printf(" ###############################################################\n");
  printf(" ###############################################################\n");
  printf(" ### CCP PROGRAM SUITE: %s  ##########\n", progname);
  printf(" ###############################################################\n");
  printf(" User:   Run date:   Run time:\n\n\n");
  printf(" Please reference: Collaborative Computational Project, Number 4. 1994.\n");
  printf(" The CCP4 Suite: Programs for Protein Crystallography. Acta Cryst. D50, 760-763.\n");
  printf(" as well as any specific reference in the program write-up.\n\n");

}

/* CCP4REALLOC
   This is a wrapper for the realloc function, which adds some
   error trapping */

void *ccp4realloc(void *ptr, size_t size)
{ void *val; 

  val = realloc (ptr, size);
  if (!val && size)
    {
      perror ("Failure in ccp4realloc");
      abort ();
    }
  return val;}
