/*
  pngloader.c - code based on xbubble by Ivan Djelic <ivan@savannah.gnu.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef USE_PNG

#include "pngloader.h"

#define MASK_THRESHOLD ( 127 )

static void create_pixmap_from_image( Display *display,
				      XImage *xim, Pixmap *pixmap );
static unsigned char * load_png_file( const char *file, int *width, 
				      int *height, int *has_alpha );
static int 
lsb( unsigned long mask ) {
  int k = 0;  
  if ( !mask )
    return 0;
  while ( !( mask & 0x01 )) {
    k++;
    mask >>= 1;
  }
  return k;
}

static int 
msb( unsigned long mask ) {
  int k = -1;
  if ( !mask )
    return 0;
  while ( mask ) {
    k++;
    mask >>= 1;
  }
  return k;
}


RgbaImage 
create_rgba_image_from_png_file( const char *filename ) {
  unsigned char * data;
  int width, height, has_alpha;
  RgbaImage ri;
  data = load_png_file( filename, &width, &height, &has_alpha);
  if ( data == NULL )
    return NULL;
  ri = (RgbaImage) malloc( sizeof(struct _RgbaImage) );
  ri->data = data;
  ri->width = width;
  ri->height = height;
  ri->has_alpha = has_alpha;
  return ri;
}

static 
void create_pixmap_from_image( Display *display,
			       XImage *xim, Pixmap *pixmap ) {
  GC gc;
  XGCValues gcv;
  gcv.foreground = 1;
  gcv.background = 0;
  *pixmap = XCreatePixmap(display, RootWindow(display, DefaultScreen(display)),
			  xim->width, xim->height, xim->depth);
  gc = XCreateGC( display, *pixmap, GCForeground | GCBackground, &gcv);
  XPutImage( display, *pixmap, gc, xim, 0,0,0,0, xim->width, xim->height);
  XFreeGC( display, gc);
}


int create_pixmaps_from_rgba_image( Display *display,
				    RgbaImage ri, Pixmap *rgb, Pixmap *mask) {
  unsigned char *src;
  unsigned long r, g, b, pixel; /* 32 bits */
  int x, y, bitmap_pad; 
  int lr_shift, lg_shift, lb_shift, rr_shift, rg_shift, rb_shift;
  XImage *rgb_image, *mask_image;
  
  int depth = DefaultDepth(display, DefaultScreen(display)); 
  Visual *visual = DefaultVisual(display, DefaultScreen(display));
  
  if ( depth < 8 )
    return 0;

  bitmap_pad = ( depth > 16 )? 32 : (( depth > 8 )? 16 : 8 );
  /* create XImages */
  rgb_image = XCreateImage( display, visual, depth, ZPixmap, 0, NULL,
  			    ri->width, ri->height, bitmap_pad, 0);
  mask_image = XCreateImage( display, visual, 1, ZPixmap, 0, NULL,
			     ri->width, ri->height, 8, 0);
  if (( rgb_image == NULL )||( mask_image == NULL )) {
    XDestroyImage(rgb_image);
    XDestroyImage(mask_image);
    return 0;
  }
  /* allocate data using bytes_per_line field */
  rgb_image->data = (char *) malloc( rgb_image->bytes_per_line*ri->height );
  mask_image->data = (char *) malloc( mask_image->bytes_per_line*ri->height );
  if (( rgb_image->data == NULL )||( mask_image->data == NULL )) {
    XDestroyImage(rgb_image);
    XDestroyImage(mask_image);
    return 0;
  }
  if ( depth > 8 ) {
    /* compute shifts for every channel */
    lr_shift = lsb( visual->red_mask );
    lg_shift = lsb( visual->green_mask );
    lb_shift = lsb( visual->blue_mask );
    rr_shift = 7 - msb( visual->red_mask ) + lr_shift;
    rg_shift = 7 - msb( visual->green_mask ) + lg_shift;
    rb_shift = 7 - msb( visual->blue_mask ) + lb_shift;
  }
  else { /* assume we have a RRRGGGBB 8-bits palette */
    lr_shift = 5;
    lg_shift = 2;
    lb_shift = 0;
    rr_shift = 5;
    rg_shift = 5;
    rb_shift = 6;
  }

  /* put pixels in images */
  src = ri->data;

  for ( y = 0; y < ri->height; y++ )
    for ( x = 0; x < ri->width; x++ ) {
      r = ( *src++ ) >> rr_shift;
      g = ( *src++ ) >> rg_shift;
      b = ( *src++ ) >> rb_shift;
      /* read alpha channel */
      if ( *src++ > MASK_THRESHOLD ) {
	pixel = ( r << lr_shift )|( g << lg_shift )|( b << lb_shift );
	XPutPixel( rgb_image, x, y, pixel);
	XPutPixel( mask_image, x, y, 1);
      }
      else
	XPutPixel( mask_image, x, y, 0);
    }
  create_pixmap_from_image(display, rgb_image, rgb);
  if ( mask != NULL )
    create_pixmap_from_image(display, mask_image, mask);
  /* cleanup */
  XDestroyImage(rgb_image);
  XDestroyImage(mask_image);
  return 1;
}

/*
 * load_png_file() allocates a byte array and fills it with
 * a contiguous sequence of 4-byte pixels in RGBA format:
 * byte 0: red channel
 * byte 1: green channel
 * byte 2: blue channel
 * byte 3: alpha channel / filler byte if no alpha layer
 *
 * load_png_file returns NULL upon failure.
 */


static unsigned char * 
load_png_file( const char *file, 
	       int *width, int *height, int *has_alpha ) {
  FILE *fd;
  unsigned char *data;
  unsigned char header[8];
  int  bit_depth, color_type;
  png_uint_32  png_width, png_height, i, rowbytes;
  png_structp png_ptr;
  png_infop info_ptr;
  png_bytep *row_pointers;

  fd = fopen( file, "rb" );
  if ( fd == NULL ) {
    perror(file);
    return NULL;
  }
  /* ensure that we opened a PNG file */
  fread( header, 1, 8, fd );
  if ( ! png_check_sig( header, 8 ) ) {
    fclose(fd);
    fprintf(stderr,"File %s does not have a valid PNG signature.\n", file);
    return NULL;
  }
  /* create PNG structs */
  png_ptr = png_create_read_struct( PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if ( ! png_ptr ) {
    fclose(fd);
    return NULL;
  }
  info_ptr = png_create_info_struct(png_ptr);
  if ( ! info_ptr ) {
    png_destroy_read_struct( &png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    fclose(fd);
    return NULL;
  }
  /* libpng does a longjmp here when it encounters an error */
  if ( setjmp( png_ptr->jmpbuf ) ) {
    png_destroy_read_struct( &png_ptr, &info_ptr, NULL);
    fclose(fd);
    return NULL;
  }
  png_init_io( png_ptr, fd );
  png_set_sig_bytes( png_ptr, 8);
  png_read_info( png_ptr, info_ptr);
  png_get_IHDR( png_ptr, info_ptr, &png_width, &png_height, &bit_depth, 
		&color_type, NULL, NULL, NULL);
  *width = (int) png_width;
  *height = (int) png_height;

  /* convert image format to RGB(A) */
  if (( color_type == PNG_COLOR_TYPE_PALETTE )||
      ( png_get_valid( png_ptr, info_ptr, PNG_INFO_tRNS )))
    png_set_expand(png_ptr);
  if (( color_type == PNG_COLOR_TYPE_GRAY )||
      ( color_type == PNG_COLOR_TYPE_GRAY_ALPHA ))
    png_set_gray_to_rgb(png_ptr);
 
  if ( info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA )
    *has_alpha = 1;
  else
    *has_alpha = 0;

  /* we only handle 8 bits per channel */
  if ( bit_depth == 16 )
    png_set_strip_16(png_ptr);
  /* insert filling bytes if necessary */
  png_set_filler(png_ptr, 0xff, PNG_FILLER_AFTER);

  /* update info data */
  png_read_update_info( png_ptr, info_ptr);

  /* allocate space for data and row pointers */
  rowbytes = png_get_rowbytes( png_ptr, info_ptr);
  data = (unsigned char *) malloc( rowbytes*(*height) );
  row_pointers = (png_bytep *) malloc( (*height)*sizeof(png_bytep));
  if (( data == NULL )||( row_pointers == NULL )) {
    png_destroy_read_struct( &png_ptr, &info_ptr, NULL);
    free(data);
    free(row_pointers);
    return NULL;
  }
  /* initialize row pointers */
  for ( i = 0;  i < *height; i++ )
    row_pointers[i] = data + i*rowbytes;
  /* read all image data */
  png_read_image( png_ptr, row_pointers );
  png_read_end( png_ptr, NULL);
  /* cleanup */
  free(row_pointers);
  png_destroy_read_struct( &png_ptr, &info_ptr, NULL);
  fclose(fd);
  /* return an array of RGB(A) values */
  return data;
}

#endif 
